#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import json
import uuid

from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.netaddr import valid_nmap_range

try:
    from shinken.synchronizer.component.component_manager import component_manager
    from shinken.synchronizer.dao.def_items import METADATA
except ImportError:
    from synchronizer.component.component_manager import component_manager
    from synchronizer.dao.def_items import METADATA

if TYPE_CHECKING:
    from synchronizer.synchronizerdaemon import Synchronizer
    from synchronizer.business.source.source import Source
    from shinken.misc.type_hint import Number, Optional, Dict, Union

OK = 0
BAD_PORT = 1
BAD_PORT_ORDER = 2

app = None  # type: Optional[Synchronizer]


def set_app(_app):
    global app
    app = _app


def key_sort(s1, s2):
    if s1.startswith(u'_') and s2.startswith(u'_'):
        return cmp(s1.lower(), s2.lower())
    if s1.startswith(u'_') and not s2.startswith(u'_'):
        return 1
    if not s1.startswith(u'_') and s2.startswith(u'_'):
        return -1
    return cmp(s1, s2)


def _get_source_data(source_name):
    # type: (unicode) -> Optional[Source]
    mongo_component = component_manager.get_mongo_component()
    sources = [s for s in app.sources]
    source = next((s for s in sources if s.get_name() == source_name.strip()), None)  # type: Source
    if not source:
        return None
    source.update_from_last_synchronization(mongo_component.col_last_synchronization)
    return source


def get_discovery_conf(source_name, range_uuid):
    # type: (unicode, unicode) -> Optional[Dict]
    user = app.get_user_auth()
    mongo_component = component_manager.get_mongo_component()
    if range_uuid:
        source = _get_source_data(source_name)
        conf = mongo_component.col_discovery_confs.find_one({u'_id': range_uuid})
        if source is None:
            return app.abort(404, app.t(u'source.no_source_with_this_name') % source_name)
        if conf is None:
            return app.abort(404, app.t(u'discovery.no_conf_with_this_id'))
        METADATA.update_metadata(conf, METADATA.IN_CREATION, False)
        METADATA.update_metadata(conf, METADATA.ITEM_TYPE, u'discovery')
        
        return {
            u'source'     : source,
            u'app'        : app,
            u'user'       : user,
            u'item'       : conf,
            u'source_name': source_name
        }


def new_discovery_conf(source_name):
    # type: (unicode) -> Optional[Dict]
    user = app.get_user_auth()
    conf = {
        u'_id'           : uuid.uuid1().hex,
        u'discovery_name': u'',
        u'enabled'       : u'1'
    }
    source = _get_source_data(source_name)
    if source is None:
        return app.abort(404, app.t(u'source.no_source_with_this_name') % source_name)
    METADATA.update_metadata(conf, METADATA.IN_CREATION, True)
    METADATA.update_metadata(conf, METADATA.ITEM_TYPE, u'discovery')
    
    return {
        u'source'     : source,
        u'app'        : app,
        u'user'       : user,
        u'item'       : conf,
        u'source_name': source_name
    }


def post_discovery_conf(source_name):
    # type: (unicode) -> unicode
    forms_item = json.loads(app.request.forms[u'item'])
    conf_id = forms_item.get(u'_id', u'')
    discovery_name = forms_item.get(u'discovery_name', u'').strip()
    iprange = forms_item.get(u'iprange', u'').strip()
    port_range = forms_item.get(u'port_range', u'').strip()
    extra_option = forms_item.get(u'extra_option', u'').strip()
    scan_interval = int(forms_item.get(u'scan_interval', u'60').strip())
    notes = forms_item.get(u'notes', u'').strip()
    enabled = (forms_item.get(u'enabled', u'1') == u'1')
    
    bad_fields = []
    
    if not discovery_name:
        bad_fields.append(u'\'%s\'' % app.t(u'discovery.discovery_name'))
    if not iprange:
        bad_fields.append(u'\'%s\'' % app.t(u'discovery.iprange'))
    else:
        if not _validate_iprange_field(iprange):
            bad_fields.append(u'\'%s\'' % app.t(u'discovery.iprange'))
    
    if port_range:
        if not _validate_port_range_field(port_range):
            bad_fields.append(u'\'%s\'' % app.t(u'discovery.port_range'))
    
    nbr_bad_fields = len(bad_fields)
    if nbr_bad_fields > 0:
        app.response.status = 400
        if nbr_bad_fields == 1:
            fields = bad_fields[nbr_bad_fields - 1]
            return app.t(u'validator.disco_bad_field') % fields
        elif nbr_bad_fields == 2:
            fields = (u' %s ' % app.t(u'element.and')).join(bad_fields)
        else:
            fields = u', '.join(bad_fields[:-1])
            fields += u' %s ' % app.t(u'element.and')
            fields += bad_fields[nbr_bad_fields - 1]
        return app.t(u'validator.disco_bad_fields') % fields
    
    new_conf = {
        u'discovery_name': discovery_name,
        u'iprange'       : iprange,
        u'scan_interval' : scan_interval,
        u'notes'         : notes,
        u'enabled'       : enabled,
        u'port_range'    : port_range,
        u'extra_option'  : extra_option
    }
    
    _to_return = app.save_source_conf_to_backend(source_name, conf_id, new_conf)
    if _to_return == u'name_already_exist':
        app.response.status = 400
        return app.t(u'validator.disco_already_exist') % discovery_name


def _validate_iprange_field(iprange):
    # type: (unicode) -> bool
    for _ip_range in iprange.split(u' '):
        try:
            _valid_ip_range(_ip_range)
        except ValueError:
            return False
    return True


def _valid_ip_range(ip_range):
    # type: (unicode) -> None
    if not valid_nmap_range(ip_range):
        raise ValueError
    
    if u'/' in ip_range and int(ip_range.split(u'/')[1]) < 16:
        raise ValueError


def _validate_port_range_field(port_range):
    # type: (unicode) -> bool
    try:
        p_ranges = port_range.split(u',')
        for p_range in p_ranges:
            ports = p_range.split(u'-')
            size = len(ports)
            if size == 1:
                if not _validate_port_value(ports[0]):
                    return False
            elif size == 2:
                status = _validate_port_range(ports[0], ports[1])
                if status == BAD_PORT:
                    app.response.status = 400
                    return False
                elif status == BAD_PORT_ORDER:
                    app.response.status = 400
                    return False
            else:
                app.response.status = 400
                return False
    except ValueError:
        app.response.status = 400
        return False
    return True


def _validate_port_value(port):
    # type: (unicode) -> bool
    if not 0 < int(port) < 65536:
        return False
    return True


def _validate_port_range(port1, port2):
    # type: (unicode, unicode) -> Number
    if not (_validate_port_value(port1) and _validate_port_value(port2)):
        return BAD_PORT
    if int(port1) > int(port2):
        return BAD_PORT_ORDER
    return OK


def delete_discovery_conf(source_name):
    # type: (unicode) -> None
    conf_id = app.request.forms.get(u'_id', u'')
    if not conf_id:
        return app.abort(400, u'Missing property _id')
    
    app.delete_source_conf_to_backend(source_name, conf_id)
    
    return
