#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018
# This file is part of Shinken Enterprise, all rights reserved.

import re
import traceback

from shinken.log import logger
from shinken.objects.item import Item, Items
from shinken.property import IntegerProp, StringProp, EditableListProp

NOT_ALLOWED_CHARACTERS = re.compile(r'''["'<>]''')


class Tagger(Item):
    id = 1  # zero is always special in database, so we do not take risk here
    my_type = 'tagger'
    
    properties = Item.properties.copy()
    properties.update({
        'tagger_name': StringProp(fill_brok=['full_status']),
        'modules'    : EditableListProp(default='', fill_brok=['full_status']),
        'order'      : IntegerProp(default='1', fill_brok=['full_status']),
        'description': StringProp(default='', fill_brok=['full_status']),
    })
    
    running_properties = Item.running_properties.copy()
    running_properties.update({
        'output'  : StringProp(default=''),
        'errors'  : StringProp(default=[]),
        'warnings': StringProp(default=[]),
        'state'   : StringProp(default='PENDING'),
    })
    macros = {}
    
    
    # For debugging purpose only (nice name)
    def get_name(self):
        return self.tagger_name
    
    
    def is_correct(self):
        r = super(Tagger, self).is_correct()
        if NOT_ALLOWED_CHARACTERS.search(self.get_name()):
            err = '''Forbidden characters ", ', < or > found in the name of taggers "%s"''' % (self.get_name())
            logger.error(err)
            r = False
        return r
    
    
    def hook_post_merge(self, hosts):
        for module in self.modules:
            f = getattr(module, 'hook_post_merge', None)
            if f is not None and callable(f):
                try:
                    f(hosts)
                    self.output = 'Tagging OK'
                    self.state = 'OK'
                    self.errors = []
                    self.warnings = []
                except Exception, exp:
                    logger.error('Module instance %s failed on hook_post_merge call: %s' % (module.get_name(), str(traceback.print_exc())))
                    self.output = 'Error: %s' % str(exp)
                    self.state = 'CRITICAL'
                    self.errors = ['Error: %s' % str(exp)]
                    self.warnings = []


class Taggers(Items):
    name_property = "tagger_name"
    inner_class = Tagger
    
    
    def linkify(self, modules):
        self.linkify_s_by_plug(modules)
    
    
    def linkify_modules_instances(self, mod_manager):
        for s in self:
            new_modules = []
            for plug in s.modules:
                plug_name = plug.get_name()
                # don't tread void names
                if plug_name == '':
                    continue
                
                mod = None
                for m in mod_manager.get_all_instances():
                    if m.get_name() == plug_name:
                        mod = m
                if mod is not None:
                    new_modules.append(mod)
                else:
                    err = "Error: the module %s is unknown for %s" % (plug_name, s.get_name())
                    s.configuration_errors.append(err)
            s.modules = new_modules
