#!/usr/bin/env bash

# Functions implementing operations needed for protected fields management

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

SYNCHRONIZER_CFG=/etc/shinken/synchronizer.cfg
KEYFILE_OWNER=shinken
KEYFILE_GROUP=shinken

function _spf_trim {
    local var="$*"
    # remove leading whitespace characters
    var="${var#"${var%%[![:space:]]*}"}"
    # remove trailing whitespace characters
    var="${var%"${var##*[![:space:]]}"}"
    echo -n "$var"
}

# Checks that the parameters given to a function are correct
# Returns with code 1 from the calling function
# Creates local variables for each parameter
# - First param : list of parameter names separated by spaces
# - Next params : "$@" : list of parameters given to the calling
#                 function ; "$@" preserves spaces in parameters
#                 correctly
# Usage :
#    eval $(_spf_check_parameters "param1 param2 param3" "$@")
#
#    In case of success, local vars "param1", "param2" and "param3" are created with
#  the expected values
#    In case of error the calling function will return with code 1

function _spf_check_parameters {
    local expected_parameters=( $1 )
    shift
    local parameters_expected_count=${#expected_parameters[@]}

    if [ ${parameters_expected_count} -ne $# ] ;then
        printf -- "${FUNCNAME[1]}: Expecting ${parameters_expected_count} parameter(s) (${expected_parameters[@]}) ; got [$*]\n" >&2
        printf "Traceback (most recent last) :\n" >&2
        for i in $(seq $((${#FUNCNAME[@]} - 1)) -1 1); do
            printf "File : [%s] Function : [%s]\n" "${BASH_SOURCE[$i]}" "${FUNCNAME[$i]}" >&2
        done

        printf "return 1\n"
        return
    fi

    for parameter in ${expected_parameters[@]} ; do
        escaped_param=${1//\'/\\\'}
        escaped_param=${escaped_param//\"/\\\"}
        escaped_param=${escaped_param//\$/\\\$}
        escaped_param=${escaped_param//\\/\\\\}
        printf "local %s=\"%s\"\n" "$parameter" "$escaped_param"
        shift
    done
}

function spf_check_mongo_connexion() {
    eval $(_spf_check_parameters "mongo_db" "$@")

    if mongo --host "${MONGO_HOST}" --port "${MONGO_PORT}" "$mongo_db" /dev/null 2>&1 | grep -qi failed ; then
        spf_show_error "Failed to connect to MongoDB server"
        exit 1
    fi
}

function spf_show_error {
    eval $(_spf_check_parameters "error_message" "$@")

    printf "\n\033[31mFAILED: \033[0m ${error_message}\n" >&2
}

function _show {
    eval $(_spf_check_parameters "color_code data" "$@")

    printf "\033[%dm%s\033[0m" "$color_code" "$data"
}
# Yellow
function show_action {
    _show 33 "$@"
}
# Yellow
function show_question {
    _show 33 "$@"
}
# White
function show_info {
    _show 97 "$@"
}
# Yellow
function show_important_info {
    _show 33 "$@"
}
# Red
function show_critical_info {
    _show 31 "$@"
}
# Green
function show_success {
    _show 32 "$@"
}
# Purple
function show_warning {
    _show 35 "$@"
}
# Purple
function show_data {
    _show 35 "$@"
}
# Red
function show_failure {
    _show 31 "$@"
}
# Purple / Cyan
function show_command {
    eval $(_spf_check_parameters "command params" "$@")
    printf "\033[35m%s \033[36m%s\033[0m\n" "$command" "$params"
}
# Cyan
function show_command_parameter {
    _show 36 "$@"
}

function spf_validate_cfg_file_with_file {
    eval $(_spf_check_parameters "cfg_file" "$@")
    local nb_definitions
    local parameter_to_check
    local has_errors=0
    local errors="$(show_critical_info "\n###### Synchronizer Configuration file corrupted ######")\n"
    errors="$errors\n$(show_critical_info "Please fix this file") $(show_data $cfg_file) $(show_critical_info "before using protected fields")\n"

   for parameter_to_check in protect_fields__activate_encryption protect_fields__encryption_keyfile protect_fields__substrings_matching_fields; do
       nb_definitions=$(grep -c "${parameter_to_check}" ${cfg_file})
       if [ $nb_definitions -gt 1 ]; then
            errors="$errors\n* $(show_important_info "The parameter") $(show_data ${parameter_to_check}) $(show_important_info "is defined multiple times in") $(show_data "$cfg_file")"
            has_errors=1
       fi
   done
    if [ "$has_errors" == "1" ]; then
        printf "$errors\n"
        exit 1
    fi
}

function spf_validate_synchronizer_cfg_file {
    spf_validate_cfg_file_with_file "$SYNCHRONIZER_CFG"
}

function spf_create_key {
    eval $(_spf_check_parameters "key_name" "$@")

    echo "$key_name" | grep -q '\|'
    if [ $? -ne 0 ] ; then
        spf_show_error "The '|' character is not allowed in key names"
        return 1
    fi
    key=$(python -m shinken.synchronizer.dao.crypto)
    if [ $? -ne 0 ]; then
        spf_show_error "Unable to generate the key"
        return 1
    fi
    echo -n "${key_name}|${key}"
}

function spf_get_activated_from_synchronizer_cfg_with_file {
   eval $(_spf_check_parameters "cfg_file" "$@")

   spf_validate_cfg_file_with_file "$cfg_file"

   local activated="$(grep protect_fields__activate_encryption ${cfg_file} | cut -d= -f2)"

   if [ -z "$activated" ] ; then
       echo "false"
       # spf_show_error "The parameter \033[35mprotect_fields__activate_database_encryption\033[0m is missing from ${cfg_file}"
       # return 1
       return 0
   fi


   if [ $activated -eq 1 ] ; then
        echo "true"
   else
        echo "false"
   fi
}

function spf_get_keyfile_name_from_synchronizer_cfg_with_file {
    eval $(_spf_check_parameters "cfg_file" "$@")

    spf_validate_cfg_file_with_file "$cfg_file"

    local keyfile="$(grep protect_fields__encryption_keyfile ${cfg_file} | cut -d= -f2)"

    if [ -z "$keyfile" ] ; then
#     spf_show_error "The parameter \033[35mprotect_fields__encryption_keyfile\033[0m is missing from ${cfg_file}"
     return 1
    fi

   echo "$keyfile"
}

function spf_set_keyfile_name_in_synchronizer_cfg_with_file {
    eval $(_spf_check_parameters "cfg_file key_file" "$@")

    spf_validate_cfg_file_with_file "$cfg_file"

    local current_keyfile=$(spf_get_keyfile_name_from_synchronizer_cfg_with_file "$cfg_file")
    if [ -z "$current_keyfile" ] ; then
        echo "protect_fields__encryption_keyfile=$key_file" >> "$cfg_file"
    else
        sed -i -e "s/protect_fields__encryption_keyfile=$current_keyfile/protect_fields__encryption_keyfile=$key_file/" "$cfg_file"
    fi
}

function spf_set_activated_in_synchronizer_cfg_with_file {
    eval $(_spf_check_parameters "cfg_file activated" "$@")

    spf_validate_cfg_file_with_file "$cfg_file"

    local current_activated=$(spf_get_keyfile_name_from_synchronizer_cfg_with_file "$cfg_file")

    if [ "$activated" = "true" ] ; then
        activated=1
    elif [ "$activated" = "false" ] ; then
        activated=0
    fi

    if [ -z "$current_activated" ] ; then
        echo "protect_fields__activate_encryption=$activated" >> "$cfg_file"
    else
        sed -i -e "s/protect_fields__activate_encryption=$current_activated/protect_fields__activate_encryption=$activated/" "$cfg_file"
        sed -i -e "s/protect_fields__activate_database_encryption=$current_activated/protect_fields__activate_database_encryption=$activated/" "$cfg_file"
        sed -i -e "s/protect_fields__activate_interface_encryption=$current_activated/protect_fields__activate_interface_encryption=$activated/" "$cfg_file"
    fi
}

function spf_get_keyfile_name_from_synchronizer_cfg {
    spf_get_keyfile_name_from_synchronizer_cfg_with_file "$SYNCHRONIZER_CFG"
}

function spf_set_keyfile_name_in_synchronizer_cfg {
    eval $(_spf_check_parameters "key_file" "$@")
    spf_set_keyfile_name_in_synchronizer_cfg_with_file "$SYNCHRONIZER_CFG" "$key_file"
}

function spf_get_activated_from_synchronizer_cfg {
    spf_get_activated_from_synchronizer_cfg_with_file "$SYNCHRONIZER_CFG"
}

function spf_set_activated_file_name_in_synchronizer_cfg {
    eval $(_spf_check_parameters "activated" "$@")
    spf_set_activated_in_synchronizer_cfg_with_file "$SYNCHRONIZER_CFG" "$activated"
}

function _spf_write_keyfile {
    eval $(_spf_check_parameters "complete_key key_file" "$@")

    local key_directory=$(dirname "$key_file")

    if [ ! -d "$key_directory" ]; then
        mkdir -p "$key_directory"
        if [ $? -ne 0 ]; then
            spf_show_error "Unable to create directory \033[35m$key_directory\033[0m"
            return 1
        fi
    fi

    if [ -f "$key_file" ] ; then
        key_name=$(spf_get_key_name_from_key_file "$key_file")
        printf "\n"
        spf_show_error "A key with the name $(show_data "$key_name") is already present.\n"
        printf "I will not overwrite it ; please remove it or move it if you want to generate a new key.\n\n" >&2
        printf "You can also specify another location as a parameter : $(show_data "$(basename $0) -f /etc/shinken/secrets/other_location__key")\n" >&2
        return 1
    fi
    echo "${complete_key}" > "$key_file"
    chown ${KEYFILE_OWNER}:${KEYFILE_GROUP} "$key_directory"
    chown ${KEYFILE_OWNER}:${KEYFILE_GROUP} "$key_file"
    chmod 600 "$key_directory"
    chmod 600 "$key_file"
}

function spf_create_keyfile {
    eval $(_spf_check_parameters "key_name key_file" "$@")

    local complete_key="$(spf_create_key "$key_name")"
    _spf_write_keyfile "$complete_key" "$key_file"

}

function spf_get_key_name_from_key_file {
    eval $(_spf_check_parameters "key_file" "$@")

    if [ ! -r "$key_file" ] ; then
        spf_show_error "Key file \033[35m${key_file}\033[0m is not readable"
        return 1
    fi

    local complete_key="$(cat $key_file)"
    echo "${complete_key%%|*}"
}

function spf_get_key_hash_from_key_file {
    eval $(_spf_check_parameters "key_file" "$@")
    local cfg_keyvalue
    cfg_keyvalue=$(spf_get_key_value_from_key_file "$key_file")
    if [ $? -ne 0 ] ; then
        spf_show_error "Unable to read key value from keyfile $keyfile"
        return 1
    fi
    local cfg_keyhash=$(echo $(_spf_trim "$cfg_keyvalue") | python -c 'import hashlib,sys;print hashlib.sha256(sys.stdin.readlines()[0].strip()).hexdigest()')

    echo $cfg_keyhash
}

function spf_get_key_value_from_key_file {
    eval $(_spf_check_parameters "key_file" "$@")

    if [ ! -r "$key_file" ] ; then
        spf_show_error "Key file \033[35m${key_file}\033[0m is not readable"
        return 1
    fi

    local complete_key
    complete_key="$(cat "$key_file")"
    echo  "${complete_key##*|}"
}

function _spf_decode_key {
    eval $(_spf_check_parameters "key_backup" "$@")

    local decoded_key
    decoded_key="$(echo ${key_backup} | base64 -d 2>/dev/null)"
    if [ -z "$decoded_key" ]; then
        spf_show_error "Unable to decode the key ; please check the key you have provided."
        return 1
    else
        echo $decoded_key
    fi
}

function _spf_encode_key {
    eval $(_spf_check_parameters "complete_key" "$*")

    if [ -z "$complete_key" ] ; then
        spf_show_error "The key is empty"
        return 1
    fi

    local encoded_key
    encoded_key="$(echo ${complete_key} | base64 -w 0)"
    if [ $? -ne 0 ] ; then
        return 1
    else
        echo $encoded_key
    fi

}

function spf_get_key_name_from_backup {
    eval $(_spf_check_parameters "key_backup" "$@")

    local complete_key key_name

    complete_key="$(_spf_decode_key "$key_backup")"
    if [[ $?  = 1 ]] ; then
        return 1
    fi

    key_name=${complete_key%%|*}
    if [ "$key_name" = "$complete_key" ]; then
        spf_show_error "The key export you provided is not valid. Please check you provided the correct one."
        return 1
    fi
    echo $key_name
    return 0
}

function spf_get_key_value_from_backup {
    eval $(_spf_check_parameters "key_backup" "$@")

    local complete_key key_value decoded_key

    complete_key="$(_spf_decode_key "$key_backup")"
    if [[ $?  = 1 ]] ; then
        return 1
    fi
    key_value=${complete_key##*|}

    # Basic checks to test the validity of the key

    if [ "$key_value" == "$complete_key" ]; then
        spf_show_error "The key export you provided does not have a valid format. Please check you provided the correct one."
        return 1
    fi

    decoded_key=$(_spf_decode_key "$key_value")
#    if [ ${#decoded_key} -ne 32 ]; then
#        spf_show_error "The key export you provided does not have a valid format. Please check you provided the correct one."
#        return 1
#    fi

    echo $key_value
}

function spf_get_key_hash_from_backup {
    eval $(_spf_check_parameters "key_backup" "$@")

    local key_value
    key_value="$(spf_get_key_value_from_backup "$key_backup")"
    if [[ $? = 1 ]]; then
        return 1
    fi

    local key_hash
    key_hash=$(echo $(_spf_trim "$key_value") | python -c 'import hashlib,sys; print hashlib.sha256(sys.stdin.readlines()[0].strip()).hexdigest()')
    if [[ $? != 0 ]]; then
        spf_show_error "The key export you provided does not have a valid format. Please check you provided the correct one."
        return 1
    fi

    echo "$key_hash"
}

function spf_restore_key_file_from_backup {
    eval $(_spf_check_parameters "key_backup key_file" "$@")

    local complete_key="$(_spf_decode_key "$key_backup")"

    _spf_write_keyfile "$complete_key" "$key_file"
}

function spf_backup_key_from_key_file {
    eval $(_spf_check_parameters "key_file" "$@")

    local complete_key=$(cat "$key_file")
    complete_key=$(_spf_trim "$complete_key")
    local encoded_key
    encoded_key="$(_spf_encode_key "$complete_key")"
    if [ $? -ne 0 ]; then
        return 1
    fi
    echo "$encoded_key"
}

function _spf_get_field_from_db_with_database {
    eval $(_spf_check_parameters "field_name db_name" "$@")

    local field_value
    field_value=$(mongo --host "${MONGO_HOST}" --port "${MONGO_PORT}" "$db_name" --quiet --eval "db.getCollection('synchronizer-info').findOne({'_id':'protected_fields_info'}).${field_name}")
    if [ $? -ne 0 ] ; then
        echo ""
        return 1
    else
        echo "$field_value"
    fi
}

function _spf_get_field_from_db {
    eval $(_spf_check_parameters "field_name" "$@")

    local mongo_uri=$(grep mongodb_uri ${SYNCHRONIZER_CFG} | cut -d= -f2)
    local mongo_database=$(grep mongodb_database ${SYNCHRONIZER_CFG} | cut -d= -f2)

    if [ -z "$mongo_uri" ] || [ -z "$mongo_database" ]; then
        spf_show_error "Mongo database not defined in ${SYNCHRONIZER_CFG}"
        return 1
    fi

    local field_value
    field_value=$(_spf_get_field_from_db_with_database "${field_name}" "${mongo_database}")
    if [ $? -ne 0 ] ; then
        echo ""
        return 1
    else
        echo "$field_value"
    fi
}

function spf_get_key_name_from_db {
    local db_keyname
    db_keyname=$(_spf_get_field_from_db protect_fields__encryption_key_name)
    local get_field_result=$?
    echo $db_keyname
    return $get_field_result
}

function spf_get_key_hash_from_db {
    local db_keyhash
    db_keyhash=$(_spf_get_field_from_db protect_fields__encryption_keyfile_hash)
    local get_field_result=$?
    echo $db_keyhash
    return $get_field_result
}

function spf_get_key_extracted_from_db {
    local db_extracted
    db_extracted=$(_spf_get_field_from_db extracted_key)
    local get_field_result=$?
    echo $db_extracted
    return $get_field_result
}

function spf_get_key_activated_from_db {
    local activated
    activated=$(_spf_get_field_from_db protect_fields__activate_database_encryption)
    if [ $? -ne 0 ] ; then
        activated="false"
    fi
    local get_field_result=$?
    echo $activated
    return $get_field_result
}

function spf_get_key_name_from_db_with_database {
    eval $(_spf_check_parameters "db_name" "$@")
    local db_keyname
    db_keyname=$(_spf_get_field_from_db_with_database protect_fields__encryption_key_name "$db_name")
    local get_field_result=$?
    echo $db_keyname
    return $get_field_result
}

function spf_get_key_hash_from_db_with_database {
    eval $(_spf_check_parameters "db_name" "$@")
    local db_keyhash
    db_keyhash=$(_spf_get_field_from_db_with_database protect_fields__encryption_keyfile_hash "$db_name")
    local get_field_result=$?
    echo $db_keyhash
    return $get_field_result
}

function spf_get_key_extracted_from_db_with_database {
    eval $(_spf_check_parameters "db_name" "$@")
    local db_extracted
    db_extracted=$(_spf_get_field_from_db_with_database extracted_key "$db_name")
    local get_field_result=$?
    echo $db_extracted
    return $get_field_result
}

function spf_get_key_activated_from_db_with_database {
    eval $(_spf_check_parameters "db_name" "$@")
    local activated
    activated=$(_spf_get_field_from_db_with_database protect_fields__activate_database_encryption "$db_name")
    local get_field_result=$?
    echo $activated
    return $get_field_result
}

function spf_check_consistency_config_db {
    local db_keyname db_keyhash db_activated keyfile_name cfg_keyhash cfg_keyname cfg_activated

    db_activated=$(spf_get_key_activated_from_db 2>/dev/null)
    if [ $? -ne 0 ]; then
        db_activated="false"
    fi

    db_keyname=$(spf_get_key_name_from_db 2>/dev/null)
    db_keyhash=$(spf_get_key_hash_from_db 2>/dev/null)
    keyfile_name=$(spf_get_keyfile_name_from_synchronizer_cfg 2>/dev/null)
    cfg_keyhash=$(spf_get_key_hash_from_key_file "$keyfile_name" 2>/dev/null)
    cfg_keyname=$(spf_get_key_name_from_key_file "$keyfile_name" 2>/dev/null)
    cfg_activated=$(spf_get_activated_from_synchronizer_cfg 2> /dev/null)
    local has_error=0

    if [ "$db_activated" = "false"  -a "$cfg_activated" = "false" ] ; then
        return 0
    fi

    if [ "$db_activated" = "false" -a "$cfg_activated" = "true" ] ; then
        printf "\n$(show_warning "WARNING:") Encryption is disabled in Synchronizer DB but enabled in config file\n"
        printf "         There is no risk of data loss but next time the Synchronizer is restarted, encryption will be enabled.\n"
        return 0
    fi

    if [ -n "$db_keyhash" ] && [ "$cfg_keyhash" != "$db_keyhash" ]; then
        spf_show_error "Key values are different between Synchronizer DB and config file"
        printf "Key Hash computed from configuration files : $cfg_keyhash\n" >&2
        printf "Key Hash stored in database                : $db_keyhash\n" >&2
        has_error=1
    fi

    if [ -n "$db_keyname" ] && [ "$cfg_keyname" != "$db_keyname" ]; then
        spf_show_error "Key names are different between Synchronizer DB and config file"
        printf "Key name defined in configuration files : $cfg_keyname\n" >&2
        printf "Key name stored in database             : $db_keyname\n" >&2
        has_error=1
    fi

    return $has_error
}

function spf_check_consistency_backup_db_with_database {
    eval $(_spf_check_parameters "key_backup db_name" "$@")
    local has_error=0
    local backup_keyhash  backup_keyname
    local db_keyname db_keyhash

    db_keyname=$(spf_get_key_name_from_db_with_database "$db_name")
    db_keyhash=$(spf_get_key_hash_from_db_with_database "$db_name")
    backup_keyhash=$(spf_get_key_hash_from_backup "$key_backup")
    if [[ $? != 0 ]]; then
        return 1
    fi
    backup_keyname=$(spf_get_key_name_from_backup "$key_backup")
    if [[ $? != 0 ]]; then
        return 1
    fi

    if [ -n "$db_keyhash" ] && [ "$backup_keyhash" != "$db_keyhash" ]; then
        spf_show_error "Key values are different between backup DB and provided key export"
        printf "Key hash from your export   : $backup_keyhash\n" >&2
        printf "Key hash stored in database : $db_keyhash\n" >&2
        has_error=1
    fi

    if [ -n "$db_keyname" ] && [ "$backup_keyname" != "$db_keyname" ]; then
        spf_show_error "Key names are different between backup DB and provided key export"
        printf "Key name from the export    : $backup_keyname\n" >&2
        printf "Key name stored in database : $db_keyname\n" >&2
        has_error=1
    fi

    return $has_error
}

function synchronizer_is_started()
{
    if service shinken-synchronizer status >/dev/null 2>/dev/null; then
            return 0
    else
            return 1
    fi
}

spf_validate_synchronizer_cfg_file
