# !/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.
import cPickle
import os

from shinken.log import LoggerFactory
from shinken.misc.os_utils import locked_safe_write_file
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.data_hub.data_hub import DataHubConfig
from shinkensolutions.data_hub.data_hub_backup.data_hub_backup import DataHubsBackup, DataHubBackup
from shinkensolutions.data_hub.data_hub_factory.data_hub_factory import DataHubFactory
from shinkensolutions.data_hub.data_hub_utils.data_hub_unpickler import DataHubUnpickler
from shinkensolutions.lib_checks.common import RaiseOnExitOptionParser, ParseOptionError, Result, EXIT_STATUS, BREAK_LINE

if TYPE_CHECKING:
    from shinken.misc.type_hint import List

VERSION = u'0.1'

logger = LoggerFactory.get_logger(name=u'DATA HUB BACKUP AND RESTORE')


class DataHubBackupRestoreConst(object):
    DATA_HUB_BACKUP_RESTORE__BACKUP = u'backup'


def _get_data_hub_confs_from_wanted_category(wanted_category):
    # type: (unicode) -> List
    data_hub_confs = DataHubFactory.list_by_category_data_hub_configuration(logger, wanted_category)
    return data_hub_confs


result = Result()
parser = RaiseOnExitOptionParser(u'%prog [options] [--help]', version=u'%prog ' + VERSION)
parser.add_option(u'-c', u'--category', dest=u'wanted_category', default=u'', help=u'It will retrieve only data hub with this category')
parser.add_option(u'-d', u'--directory', dest=u'backup_directory', default=u'', help=u'Backup directory')
parser.add_option(u'-f', u'--file_name', dest=u'backup_file_name', default=u'', help=u'Backup file name')
parser.add_option(u'-m', u'--mode', dest=u'mode', default=u'', help=u'backup or restore')


def _parse_args():
    opts = None
    try:
        opts, args = parser.parse_args()
        if args:
            parser.error(u'Does not accept arguments.')
        if not opts.mode:
            parser.error(u'Missing parameter mode (-m/--mode)')
        if not opts.backup_directory:
            parser.error(u'Missing parameter backup_directory (-d/--directory)')
        if opts.mode.lower() not in (u'backup', u'restore'):
            parser.error(u'The mode (-m/--mode) can only be "backup" or "restore"')
    except ParseOptionError as e:
        if e.msg:
            result.hard_exit(EXIT_STATUS.CRITICAL, u'Fail to parse command argument: %s %s' % (BREAK_LINE, BREAK_LINE.join(e.msg.split(u'\n'))))
        exit(0)
    return opts


def backup_data_hub(wanted_category, backup_directory, backup_file_name):
    # type: (unicode, unicode, unicode) -> None
    data_hub_confs_list = _get_data_hub_confs_from_wanted_category(wanted_category)
    
    backup = DataHubsBackup()
    
    for data_hub_conf in data_hub_confs_list:  # type: DataHubConfig
        data_hub = DataHubFactory.build_and_init_data_hub(logger, data_hub_conf)
        backup.add_backup(data_hub_conf.data_hub_id, DataHubBackup(data_hub_conf, data_hub.backup()))
    
    data_to_backup = cPickle.dumps(backup)
    locked_safe_write_file(os.path.join(backup_directory, u'%s.pickle' % backup_file_name), data_to_backup)


def restore_data_hub(wanted_category, backup_directory, backup_file_name):
    # type: (unicode, unicode, unicode) -> None
    real_file_path = os.path.join(backup_directory, u'%s.pickle' % backup_file_name)
    if not os.path.isfile(real_file_path):
        result.hard_exit(EXIT_STATUS.CRITICAL, u'The backup file %s does not exist' % real_file_path)
    with open(real_file_path, u'r') as shared_file:
        data_hubs_backup = DataHubUnpickler.loads(shared_file.read())  # type: DataHubsBackup
    
    data_hubs_backup = data_hubs_backup.filter_by_category(wanted_category)
    
    for data_hub_id, data_hub_backup in data_hubs_backup:
        data_hub = DataHubFactory.build_and_init_data_hub(logger, data_hub_backup.data_hub_config)
        data_hub.restore(data_hub_backup.data_backup)


def main():
    options = _parse_args()
    
    wanted_category = options.wanted_category
    mode = options.mode
    backup_directory = options.backup_directory
    backup_file_name = options.backup_file_name or u'backup__data_hubs__%s' % wanted_category
    
    if mode == "backup":
        backup_data_hub(wanted_category, backup_directory, backup_file_name)
    elif mode == "restore":
        restore_data_hub(wanted_category, backup_directory, backup_file_name)


if __name__ == '__main__':
    main()
