#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.
import re

import datetime
import time

from shinken.log import logger
from .sanatize_utils import get_synchronizer_db, check_sanatize_database_info_flag, update_database_info_collection

try:
    from shinken.synchronizer.dao.validators.validator import ILLEGAL_CHARS
    from shinken.synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
    from shinken.synchronizer.dao.def_items import ITEM_STATE, ITEM_TYPE
    from shinken.synchronizer.dao.datamanagerV2 import DataManagerV2
    from shinken.synchronizer.dao.transactions.transactions import DBTransaction
except ImportError:
    from synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
    from synchronizer.dao.def_items import ITEM_STATE, ITEM_TYPE
    from synchronizer.dao.datamanagerV2 import DataManagerV2
    from synchronizer.dao.transactions.transactions import DBTransaction
    from synchronizer.dao.validators.validator import ILLEGAL_CHARS

HOST_COLLECTION_NAMES_BY_STATE = {
    ITEM_STATE.STAGGING    : u'configuration-stagging-host',
    ITEM_STATE.PRODUCTION  : u'configuration-production-host',
    ITEM_STATE.WORKING_AREA: u'configuration-working-area-host'
}
SHINKEN_RENAME_PREFIX = '__SHINKEN_AUTO_RENAME'

from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Dict
    from typing import Collection

HOST_TPL_FORBIDDEN_CHAR_SANATIZE_SUMMARY_FILE = u'/var/log/shinken/remove_templates_with_forbidden_characters.log'


def remove_empty_service_excludes_by_id_links():
    # type: () -> bool
    g_did_run = False
    
    prop_to_remove = u'service_excludes_by_id'
    sanatize_database_flag = u'service_excludes_by_id_links_already_cleaned'
    dataprovider_mongo, db = _get_dataprovider_mongo()
    
    if check_sanatize_database_info_flag(db, sanatize_database_flag, 1):
        logger.info(u'This fix has already been executed on this database.')
        return g_did_run
    
    datamanager = DataManagerV2(dataprovider_mongo)
    
    with DBTransaction(datamanager):
        for _, collection_name in HOST_COLLECTION_NAMES_BY_STATE.iteritems():
            collection = db.get_collection(collection_name)
            
            # This query will get all elements with the 'prop_to_remove' and wit an empty 'links' in it
            query = {
                u'$and': [
                    {prop_to_remove: {u'$exists': 1}},  # Does the 'prop_to_remove' exists
                    {u'%s.links' % prop_to_remove: {u'$eq': []}}  # Does the 'prop_to_remove' has a property 'links' equals to an empty list
                ]
            }
            result = collection.update_many(query, update={u'$unset': {prop_to_remove: u''}})
            
            if result and result.modified_count:
                g_did_run = True
    
    if g_did_run:
        update_database_info_collection(db, sanatize_database_flag, 1)
    
    return g_did_run


def replace_forbidden_chars_from_used_templates():
    # type: () -> bool
    g_did_run = False
    dataprovider_mongo, db = _get_dataprovider_mongo()
    datamanager = DataManagerV2(dataprovider_mongo)
    str_regex = u'[%s]' % ILLEGAL_CHARS.get(u'value')
    
    hosttpl_name_key = u'use.links.name'
    changes_to_log = []
    with DBTransaction(datamanager):
        for item_state, collection_name in HOST_COLLECTION_NAMES_BY_STATE.iteritems():
            collection = db.get_collection(collection_name)
            # We will only match host tpls because the value "name" is only used by hosttpls
            query = {hosttpl_name_key: {u'$regex': str_regex}}
            
            mongo_result = collection.find(query)
            
            if not mongo_result:
                continue
            
            g_did_run = _remove_forbidden_chars_from_use(mongo_result, str_regex, collection, item_state, changes_to_log)
    _write_change_on_file(changes_to_log)
    return g_did_run


def _write_change_on_file(changes):
    # type: (List) -> None
    with open(HOST_TPL_FORBIDDEN_CHAR_SANATIZE_SUMMARY_FILE, u'a+') as summary:
        for _line in changes:
            summary.write('[%s] : %s' % (datetime.datetime.fromtimestamp(int(time.time())).strftime(u'%Y-%m-%d %H:%M:%S'), _line.encode(u'utf-8')))


def _remove_forbidden_chars_from_use(element_list, str_regex, collection, item_state, changes_to_log):
    # type: (List[Dict], unicode, Collection, unicode, List[unicode]) -> bool
    dataprovider_mongo, db = _get_dataprovider_mongo()
    g_did_run = False
    
    changes_to_log.append('The following elements used templated with at least one non authorized characters : %s\n' % str_regex)
    
    regex = ILLEGAL_CHARS.get(u'regex')
    for element in element_list:
        g_did_run = True
        
        name_key = u'name' if u'name' in element else u'host_name'
        item_type = ITEM_TYPE.HOSTTPLS if u'name' in element else ITEM_TYPE.HOSTS
        item_name = element.get(name_key, u'UNKNOWN')
        
        host_name_line = '\tElement "%s" (type: %s, state: %s):\n' % (item_name, item_type, item_state)
        changes_to_log.append(host_name_line)
        
        _links = element.get(u'use', {}).get(u'links')
        for _idx in xrange(len(_links)):
            use = _links[_idx]
            if u'name' not in use:
                continue
            
            old_name = use[u'name']  # type: unicode
            
            matchs = regex.findall(old_name)
            if not matchs:
                continue
            
            new_name = u'%s' % (re.sub(str_regex, u'', old_name))
            tmp_new_name = new_name
            while dataprovider_mongo.find_item_by_name(re.escape(tmp_new_name), item_type=ITEM_TYPE.HOSTTPLS, item_state=ITEM_STATE.STAGGING):
                tmp_new_name = u'_%s' % tmp_new_name
            
            if tmp_new_name != new_name:
                summary_line = u'\t\tRenamed [ %s ] into [ %s ] because [ %s ] is a real host template\n' % (old_name, tmp_new_name, new_name)
                new_name = tmp_new_name
            else:
                summary_line = u'\t\tRenamed [ %s ] into [ %s ]\n' % (old_name, new_name)
            use[u'name'] = new_name
            changes_to_log.append(summary_line)
        
        if _links:
            seen = set()
            new_l = []
            for _link in _links:
                _link_tuple = tuple(_link.items())
                if _link_tuple not in seen:
                    seen.add(_link_tuple)
                    new_l.append(_link)
            element[u'use'][u'links'] = new_l
            changes_to_log.append(u'\t\tDuplicates removed\n')
        
        collection.update(filter={u'_id': element[u'_id']}, update=element)
    
    return g_did_run


def _get_dataprovider_mongo():
    db = get_synchronizer_db()
    dataprovider_mongo = DataProviderMongo(db)
    return dataprovider_mongo, db
