#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.


import base64
import codecs
import glob
import imp
import logging
import logging.handlers
import math
import optparse
import os
import random
import re
import shutil
import subprocess
import sys
import time
import traceback
import uuid
from datetime import date
from datetime import datetime

import pymongo
from pymongo.errors import BulkWriteError

from lib_sanatize import sanatize_decorator
from lib_sanatize.config_file_sanatize import convert_broker_module_livedata_configuration_files_to_new_format, convert_livedata_module_sla_provider_configuration_files_to_new_format
from lib_sanatize.config_file_sanatize import set_default_values_in_cfg_files, __load_full_configuration
from lib_sanatize.sanatize_common import rename_graphite_checks_metrics_files
from lib_sanatize.sanatize_decorator import ALL_FIX, add_fix, add_doc, add_context_daemons, add_version, auto_launch, add_data_type, cyan
from lib_sanatize.sanatize_decorator import need_shinken_stop
from lib_sanatize.sanatize_log import DEFAULT_LOG_FILENAME, orig_stdout, orig_stderr, log
from lib_sanatize.sanatize_log import sanitize_log
from lib_sanatize.sanatize_utils import check_sanatize_database_info_flag, \
    update_database_info_collection, connect_to_mongo
from lib_sanatize.sanatize_utils import get_datamanager_v2_from_data_provider_metadata, get_synchronizer_db, get_broker_db, add_history_info, VERSION, natural_version, is_shinken_running, do_match_daemons, get_output_and_color

try:
    import pwd
    import grp
except ImportError:
    # don't expect to have this on windows :)
    pwd = grp = None
from lib_sanatize.sanatize_mongo import remove_empty_service_excludes_by_id_links, replace_forbidden_chars_from_used_templates, HOST_TPL_FORBIDDEN_CHAR_SANATIZE_SUMMARY_FILE

try:
    import shinken
except ImportError:
    print 'Cannot import shinken lib, please install it before launching this tool'
    sys.exit(2)

from delete_invalid_service_overrides import print_or_delete_invalid_service_overrides
from shinkensolutions.localinstall import do_import_data_history_sanatize
from shinkensolutions.system_tools import run_command_with_return_code, create_symlink_from_os_version
from shinken.log import logger
from shinken.objects.config import Config
from shinken.objects.module import Module as ShinkenModuleDefinition
from shinkensolutions.crypto import AESCipher
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.date_helper import get_diff_date, Date, date_now

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional
    from synchronizer.dao.datamanagerV2 import DataManagerV2
    from synchronizer.dao.items import BaseItem

try:
    from shinken.synchronizer.dao.helpers import get_name_from_type, safe_add_to_dict
    from shinken.synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
    from shinken.synchronizer.dao.def_items import DEF_ITEMS, ITEM_STATE, ITEM_TYPE, SERVICE_OVERRIDE, METADATA
    from shinken.synchronizer.dao.items import get_item_instance
    from shinken.synchronizer.synchronizerdaemon import Synchronizer
    from shinken.synchronizer.dao.validators.validator import Validator
except ImportError:
    from synchronizer.dao.helpers import get_name_from_type, safe_add_to_dict
    from synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
    from synchronizer.dao.def_items import DEF_ITEMS, ITEM_STATE, ITEM_TYPE, SERVICE_OVERRIDE, METADATA
    from synchronizer.dao.items import get_item_instance
    from synchronizer.synchronizerdaemon import Synchronizer
    from synchronizer.dao.validators.validator import Validator

if not os.getuid() == 0:
    print "ERROR: this script must be run as root"
    sys.exit(2)

SANATIZE_SUMMARY = {}


########################### FIX Functions

# Will look at contact template without 'name' and remove it from database
@add_fix
@add_doc('Fix contact templates that are invalid, like no name or even duplicate generic-contact object')
@add_context_daemons(['synchronizer'])
@add_version('02.03.03')
@auto_launch(False)
@need_shinken_stop(True)
def fix_invalid_contact_templates():
    g_did_run = False
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    col = db.get_collection('configuration-stagging-contact')
    contacts = col.find({})
    to_clean = []
    for c in contacts:
        register = c.get('register', '1')
        if register != '0':
            continue
        _id = c['_id']
        name = c.get('name', None)
        if name is None:
            to_clean.append(_id)
            g_did_run = True
    if len(to_clean) == 0:
        logger.debug("    No contact template to clean")
        return g_did_run
    logger.info("    %d contact template to clean" % len(to_clean))
    for _id in to_clean:
        col.remove({'_id': _id})
    return g_did_run


# Will look at contact template without 'name' and remove it from database
@add_doc('Fix contact template generic-template in duplicate')
@add_context_daemons(['synchronizer'])
@add_version('02.03.02')
@add_fix
@auto_launch(False)
@need_shinken_stop(True)
def fix_duplicate_generic_contact():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    col = db.get_collection('configuration-stagging-contact')
    contacts = col.find({})
    generic_contact = []
    for c in contacts:
        if c.get('register', '1') == '0' and c.get('name', None) == 'generic-contact':
            generic_contact.append(c)
    if len(generic_contact) in [0, 1]:
        logger.debug("    No generic-contact duplication.")
        return False
    # oups we have a duplication here
    # keep the first one
    logger.info("    Removing %d duplicate element" % (len(generic_contact) - 1))
    for c in generic_contact[1:]:
        col.remove({'_id': c['_id']})
    return True


# Will look at contact template without 'name' and remove it from database
@add_doc('Fix invalid objects from the first 2.x versions, like "" values in configuration database')
@add_context_daemons(['synchronizer'])
@add_version('02.03.03')
@add_fix
@auto_launch(False)
@need_shinken_stop(True)
def fix_empty_value_from_synchronizer_data():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    data_provider_mongo = DataProviderMongo(db)
    for item_state in [ITEM_STATE.NEW, ITEM_STATE.MERGE_SOURCES, ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION]:
        for item_type in DEF_ITEMS.iterkeys():
            nb_cleaned = 0
            items = data_provider_mongo.find_items(item_type, item_state)
            nb_total = len(items)
            for item in items:
                _id = item['_id']
                to_del = []
                for (k, v) in item.iteritems():
                    if v == '':
                        to_del.append(k)
                if len(to_del) != 0:
                    nb_cleaned += 1
                    for k in to_del:
                        del item[k]
                    data_provider_mongo.save_item(item, item_type, item_state)
            if nb_total != 0 and nb_cleaned != 0:
                if nb_cleaned != nb_total:
                    g_did_run = True
                    logger.info("   Did clean [%s-%s] objects [%d/%d] are now cleaned" % (item_state, item_type, nb_cleaned, nb_total))
    return g_did_run


@add_fix
@add_doc('Fix duplicate objects in the database.')
@add_context_daemons(['synchronizer'])
@add_version('02.03.03')
@auto_launch(False)
@need_shinken_stop(True)
def fix_duplicate_elements():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    DATA_KEYS = {'host'                    : 'host_name',
                 'command'                 : 'command_name',
                 'timeperiod'              : 'timeperiod_name',
                 'contact'                 : 'contact_name',
                 'contactgroup'            : 'contactgroup_name',
                 'hostgroup'               : 'hostgroup_name',
                 'businessimpactmodulation': 'business_impact_modulation_name',
                 'notificationway'         : 'notificationway_name',
                 'escalation'              : 'escalation_name',
                 'macromodulation'         : 'macromodulation_name',
                 'service'                 : ''
                 }
    
    # First loop over classic objects with real names
    for (t, k_name) in DATA_KEYS.iteritems():
        did_run = False
        for col in [db.get_collection('configuration-stagging-%s' % t), db.get_collection('configuration-production-%s' % t)]:
            objs = col.find({})
            # which objects do we will need to clean?
            classic_objs = {}
            template_objs = {}
            to_clean = []
            for o in objs:
                if k_name:
                    register = o.get('register', '1')
                    if register == '0':
                        name = o.get('name', None)
                        if name is None:  # unamed template? come one, what a crazy configuration
                            continue
                        if name not in template_objs:
                            template_objs[name] = []
                        template_objs[name].append(o)
                    else:  # classic objects
                        oname = o.get(k_name, None)
                        if oname is None:  # object without a name? ... get lost
                            continue
                        if oname not in classic_objs:
                            classic_objs[oname] = []
                        classic_objs[oname].append(o)
                else:  # service...
                    # they are a bit more triky. We can clean only the element that have the host_name&description set (template or not)
                    sdesc = o.get('service_description', None)
                    hname = o.get('host_name', None)
                    name = o.get('name', None)
                    if name is not None:  # classic templates
                        if sdesc is not None:  # mix? cannot clean this!
                            continue
                        if name not in template_objs:
                            template_objs[name] = []
                        template_objs[name].append(o)
                    else:
                        if sdesc is None or hname is None:  # cannot clean a unclear service, bailout
                            continue
                        skey = '%s--%s' % (sdesc, hname)
                        if skey not in classic_objs:
                            classic_objs[skey] = []
                        classic_objs[skey].append(o)
            
            # Now go clean them
            all_elements = []
            # Mix al, but without mix names
            for (oname, lst) in classic_objs.iteritems():
                all_elements.append((oname, lst))
            for (oname, lst) in template_objs.iteritems():
                all_elements.append((oname, lst))
            # Go clean
            for (oname, lst) in all_elements:
                # no problem here
                if len(lst) <= 1:
                    continue
                
                
                def sortkey(o):
                    try:
                        return uuid.UUID(hex=o['_id']).time
                    except:
                        return -1
                
                
                # sort elements based on their uuids, hope it can works
                lst = sorted(lst, key=sortkey)
                
                new_lst = []
                nb_valid = 0
                # Maybe there are more than 1, and there is a syncui one
                for o in lst:
                    if 'syncui' in o.get('sources', ''):  # fuck a manually changed element, don't modify it
                        nb_valid += 1
                        continue
                    new_lst.append(o)
                
                # Maybe there is no much element to clean, so bailout this name, we can't clean it, too risky
                if len(new_lst) == 0:
                    continue
                
                # ok we will have to fix something here...
                did_run = True
                g_did_run = True
                # maybe there was no valid element saved, so skip one
                if nb_valid == 0:
                    to_clean.extend(new_lst[1:])
                else:  # there was at least one element save, so clean all the others :)
                    to_clean.extend(new_lst)
            if did_run:
                total = col.find({}, only_count=True)
                nb_clean = len(to_clean)
                logger.info("   Did clean %6d duplicate %15ss [%d-%d=%d]" % (nb_clean, t, total, nb_clean, total - nb_clean))
                for o in to_clean:
                    # print o.get('sources', ''), o.get('pack', '')
                    logger.debug("  CLEAN: %s => %s" % (t, o.get('name', o.get(k_name, o.get('service_description', '')))))
                    col.remove({'_id': o['_id']})
    
    return g_did_run


# Fix share collection
@add_fix
@add_doc('Fix share collection.')
@add_context_daemons(['broker'])
@add_data_type('user')
@add_version('02.03.04')
@auto_launch(False)
@need_shinken_stop(True)
def fix_share_collection():
    g_did_run = False  # did we fix something?
    db = get_broker_db()
    if not db.list_name_collections():
        return g_did_run
    
    col_share = db.get_collection('share')
    col_hive = db.get_collection('hive')
    col_list = db.get_collection('list')
    
    shares = col_share.find({})
    
    for share in shares:
        if 'screen' in share:
            screen_type = share['screen'].get('type', None)
            screen_uuid = share['screen']['uuid']
            screen_name = share['name']
            
            if not screen_type:
                hive_uuid = col_hive.find_one({'_id': screen_uuid}, {'_id': 1})
                list_uuid = col_list.find_one({'_id': screen_uuid}, {'_id': 1})
                
                if hive_uuid:
                    share['screen']['type'] = 'hive'
                elif list_uuid:
                    share['screen']['type'] = 'list'
                
                col_share.save(share)
                
                g_did_run = True
                print "Fix screen [ %s ] type found [ %s ]" % (screen_name, share['screen']['type'])
    
    return g_did_run


# Will clean if there are double SYNC_KEYS in stagging
@add_fix
@add_doc('Will clean if there are double SYNC_KEYS in stagging.')
@add_context_daemons(['synchronizer'])
@add_version('02.03.03-U1')
@auto_launch(True)
@need_shinken_stop(True)
def fix_double_sync_keys():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    col = db.get_collection('synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'})
    
    if database_info and database_info.get('fix_double_sync', 0) == 3:
        logger.info(u'This fix has already been executed on this database.')
        return g_did_run
    
    try:
        from shinken.synchronizer.dao.transactions.transactions import DBTransaction
    except ImportError:
        return 'You cannot run this fix on your shinken version please upgrade it.'
    
    data_provider_mongo = DataProviderMongo(db)
    
    for item_type in ITEM_TYPE.ALL_TYPES:
        logger.info('Searching double sync-key for type : %s have started.' % item_type)
        to_update_states = (ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION, ITEM_STATE.WORKING_AREA) if DEF_ITEMS[item_type]['has_work_area'] else (ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION)
        for item_state in to_update_states:
            to_regenerates = {}
            keys = {}
            for item in data_provider_mongo.find_items(item_type, item_state):
                
                sync_keys = set([sync_key.lower() for sync_key in item.get('_SYNC_KEYS', [])])
                item_name = get_name_from_type(item_type, item)
                item_id = item['_id']
                minimal_sync_keys = compute_minimal_sync_keys(item, item_type)
                
                item['__MINIMAL_KEYS'] = minimal_sync_keys
                item['__NAME'] = item_name
                
                if not sync_keys:
                    logger.info('[%12s] %s [ %s ]-[ %s ] item don\'t have _SYNC_KEYS. We set minimal sync keys:[ %s ] to it.' % (item_state, item_type, item_id, item_name, ','.join(minimal_sync_keys)))
                    to_regenerates[item['_id']] = item
                    continue
                
                for minimal_sync_key in minimal_sync_keys:
                    if minimal_sync_key not in sync_keys:
                        logger.info('[%12s] %s [ %s ]-[ %s ] item missing minimal sync key:[ %s ]. We add it to the item. Current sync_keys: [ %s ]' % (item_state, item_type, item_id, item_name, minimal_sync_key, ','.join(sync_keys)))
                        to_regenerates[item['_id']] = item
                        sync_keys.add(minimal_sync_key)
                        item['_SYNC_KEYS'] = list(sync_keys)
                
                for sync_key in sync_keys:
                    if sync_key in keys:
                        keys[sync_key].append(item)
                    else:
                        keys[sync_key] = [item]
            
            for share_sync_key, items in keys.iteritems():
                if len(items) > 1:
                    for item in items:
                        item_name = get_name_from_type(item_type, item)
                        item_id = item['_id']
                        logger.info('[%12s] %s [ %s ]-[ %s ] item share the sync-key [ %s ] with an other item. We set minimal sync keys:[ %s ] to it.' % (item_state, item_type, item_id, item_name, share_sync_key, ','.join(item['__MINIMAL_KEYS'])))
                        to_regenerates[item['_id']] = item
                        item['_SYNC_KEYS'] = item['__MINIMAL_KEYS']
            
            if to_regenerates:
                with DBTransaction(data_provider_mongo):
                    for item in to_regenerates.itervalues():
                        g_did_run = True
                        
                        del item['__MINIMAL_KEYS']
                        del item['__NAME']
                        
                        data_provider_mongo.save_item(item, item_type, item_state)
    
    update_database_info_collection(db, u'fix_double_sync', 3)
    
    if g_did_run:
        update_database_info_collection(db, u'must_remake_import', 1)
    col.save(database_info)
    
    return g_did_run


def compute_minimal_sync_keys(item, item_type):
    is_template = ITEM_TYPE.is_template(item_type)
    new_sync_keys = []
    
    if item_type in ITEM_TYPE.ALL_DEDICATED_SERVICES:
        item_sync_keys_uuid = item.get('_SE_UUID', 'core-%s-%s' % (item_type, item['_id']))
        new_sync_keys.append(item_sync_keys_uuid.lower())
    else:
        item_name = get_name_from_type(item_type, item)
        item_sync_keys_name = ('%s-tpl' % item_name if is_template else item_name)
        new_sync_keys.append(item_sync_keys_name.lower())
        
        item_sync_keys_uuid = item.get('_SE_UUID', None)
        if item_sync_keys_uuid:
            new_sync_keys.append(item_sync_keys_uuid.lower())
    
    return new_sync_keys


@add_fix
@add_doc('Regenerate invalid or missing sync-keys for a item_type.')
@add_context_daemons(['synchronizer'])
@add_version('02.07.05')
@auto_launch(False)
def regenerate_invalid_or_missing_sync_keys():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    try:
        from shinken.synchronizer.dao.transactions.transactions import DBTransaction
    except ImportError:
        return u'You cannot run this fix on your shinken version please upgrade it.'
    
    item_type = opts.item_type
    
    if not item_type:
        logger.warning(u'Please choose a item type with --item-type option.')
        return False
    if item_type not in ITEM_TYPE.ALL_TYPES:
        logger.error(u'Unknown item_type : [%s] must be one of following : \n* %s' % (item_type, u'\n* '.join(ITEM_TYPE.ALL_TYPES)))
        return False
    
    db = get_synchronizer_db()
    data_provider_mongo = DataProviderMongo(db)
    
    to_save_items = []
    
    logger.info('Sanitize sync-key for type : %s have started.' % item_type)
    to_update_states = (ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION, ITEM_STATE.WORKING_AREA) if DEF_ITEMS[item_type]['has_work_area'] else (ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION)
    for item_state in to_update_states:
        to_save_items = []
        for item in data_provider_mongo.find_items(item_type, item_state):
            old_sync_keys = set(item.get('_SYNC_KEYS', []))
            minimal_sync_keys = compute_minimal_sync_keys(item, item_type)
            item_name = get_name_from_type(item_type, item)
            item_id = item['_id']
            
            if not old_sync_keys:
                item['_SYNC_KEYS'] = list(minimal_sync_keys)
                to_save_items.append(item)
                logger.info('[%12s] %s [ %s ]-[ %s ] item don\'t have _SYNC_KEYS. We set minimal sync keys:[ %s ] to it.' % (item_state, item_type, item_id, item_name, ','.join(minimal_sync_keys)))
                continue
            
            valide_sync_keys = set((item_value.lower() for item_value in item.itervalues() if hasattr(item_value, 'lower') and item_value.lower() in old_sync_keys))
            
            for minimal_sync_key in minimal_sync_keys:
                if minimal_sync_key not in valide_sync_keys:
                    valide_sync_keys.add(minimal_sync_key)
            
            if old_sync_keys == valide_sync_keys:
                logger.info('[%12s] %s [ %s ]-[ %s ] item have valide _SYNC_KEYS.' % (item_state, item_type, item_id, item_name))
                continue
            
            logger.info('[%12s] %s [ %s ]-[ %s ] item have some invalide or missing _SYNC_KEYS. We set new sync keys to it. [ %s ]->[ %s ]' % (item_state, item_type, item_id, item_name, ','.join(old_sync_keys), ','.join(valide_sync_keys)))
            item['_SYNC_KEYS'] = list(valide_sync_keys)
            to_save_items.append(item)
        
        if to_save_items:
            g_did_run = bool(to_save_items)
            with DBTransaction(data_provider_mongo):
                for item in to_save_items:
                    data_provider_mongo.save_item(item, item_type, item_state)
    
    if g_did_run:
        col = db.get_collection('synchronizer-info')
        database_info = col.find_one({'_id': 'database_info'})
        if not database_info:
            database_info = {'_id': 'database_info'}
        database_info['must_remake_import'] = 1
        col.save(database_info)
    
    return g_did_run


def _update_reversed_links(item, new_id, datamanagerV2):
    # type: (BaseItem, unicode, DataManagerV2) -> None
    
    reverse_links = item.get_reverse_links().copy()
    for reversed_link in reverse_links:
        linked_item = datamanagerV2.find_item_by_id(item_id=reversed_link.item_id, item_type=reversed_link.item_type, item_state=reversed_link.item_state)
        raw_linked_item = linked_item.get_raw_item(flatten_links=False, keep_metadata=True)
        mock_baseitem = get_item_instance(reversed_link.item_type, raw_linked_item)
        for link in mock_baseitem.get_links(reversed_link.key):
            if '_id' in link and link['_id'] == item['_id']:
                link['_id'] = new_id
        raw_linked_item.pop('@metadata', None)
        datamanagerV2.save_item(raw_linked_item, item_type=linked_item.get_type(), item_state=linked_item.get_state())


@add_fix
@add_doc('Will remove shinken-core contact from configuration.')
@add_context_daemons(['synchronizer'])
@add_version('02.03.03-U1')
@auto_launch(True)
@need_shinken_stop(True)
def fix_remove_shinken_core():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    data_tables = ['host',
                   'command',
                   'timeperiod',
                   'contact',
                   'contactgroup',
                   'hostgroup',
                   'businessimpactmodulation',
                   'notificationway',
                   'escalation',
                   'macromodulation',
                   'service'
                   ]
    
    for col in [db.get_collection('configuration-stagging-contact'), db.get_collection('configuration-production-contact')]:
        item = col.find_one({'contact_name': 'shinken-core'})
        if item:
            col.remove({'_id': item['_id']})
            g_did_run = True
            
            # Remove all references as well
            for col in [db.get_collection('configuration-stagging-host'), db.get_collection('configuration-production-host'),
                        db.get_collection('configuration-stagging-service'), db.get_collection('configuration-production-service'),
                        db.get_collection('configuration-stagging-escalation'), db.get_collection('configuration-production-escalation')]:
                items = col.find({'contacts': re.compile('shinken-core')})
                for item in items:
                    new_item = item.copy()
                    new_item['contacts'] = ','.join([c for c in item['contacts'].split(',') if c and c.strip() != 'shinken-core'])
                    add_history_info(new_item, item)
                    col.update({'_id': new_item['_id']}, set_update=new_item)
            for col in [db.get_collection('configuration-stagging-contactgroup'), db.get_collection('configuration-production-contactgroup')]:
                items = col.find({'members': re.compile('shinken-core')})
                for item in items:
                    new_item = item.copy()
                    new_item['members'] = ','.join([c for c in item['members'].split(',') if c and c.strip() != 'shinken-core'])
                    add_history_info(new_item, item)
                    col.update({'_id': item['_id']}, set_update=new_item)
            
            for table in data_tables:
                col = db.get_collection('configuration-stagging-%s' % table)
                items = col.find({'last_modification.contact': '16f36b30185f11e69025f8bc126497d6'})
                for item in items:
                    item['last_modification']['contact'] = '-1'
                    item['last_modification']['contact_name'] = 'shinken-core'
                    col.update({'_id': item['_id']}, set_update=item)
    
    return g_did_run


# Will clean deprecated check after poller update see SEF-1053 / SEF-1143
@add_fix
@add_doc('Will clean deprecated check after poller update.')
@add_context_daemons(['synchronizer'])
@add_version('02.03.03-U2')
@auto_launch(True)
@need_shinken_stop(True)
def fix_remove_deprecated_check():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    col_stagging = db.get_collection('configuration-stagging-service')
    col_production = db.get_collection('configuration-production-service')
    
    service_to_del = ['Scheduler - Performance API Connection',
                      'Scheduler - Performance Late Checks',
                      'Scheduler - Performance Latency',
                      'Poller - Performance API Connection',
                      'Poller - Performance CPU Load',
                      'Reactionner - Performance CPU Load',
                      'Reactionner - Performance API Connection']
    
    to_del = list(col_stagging.find({'service_description': {'$in': service_to_del}}, {'_id': 1, 'service_description': 1}))
    if to_del:
        for todel_item in to_del:
            col_stagging.remove({'_id': todel_item['_id']})
        g_did_run = True
    
    to_del = list(col_production.find({'service_description': {'$in': service_to_del}}, {'_id': 1, 'service_description': 1}))
    if to_del:
        logger.debug("We will deleted deprecated check.")
        for todel_item in to_del:
            logger.debug("Deleting check [ %s ]." % todel_item['service_description'])
            col_production.remove({'_id': todel_item['_id']})
        g_did_run = True
    
    return g_did_run


@add_fix
@add_doc('Will clean undefined templates used by aix templates.')
@add_context_daemons(['synchronizer'])
@add_version('02.03.03-U2')
@auto_launch(True)
@need_shinken_stop(True)
def fix_remove_undefined_aix_templates():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    try:
        from shinken.synchronizer.dao.datamanagerV2 import DataManagerV2
        from shinken.synchronizer.dao.transactions.transactions import DBTransaction
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return g_did_run
    
    user = {
        'contact_name': 'shinken-core',
        'is_admin'    : '1',
        '_id'         : '-1',
    }
    
    col = db.get_collection('synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'})
    if not database_info or database_info.get('new_link_format', 0) != 1:
        logger.error("You cannot run this fix on your shinken version please upgrade it or run the sanatize 'migrate_links_name_into_id'.")
        return g_did_run
    
    to_remove_use = ("10min_long", "20min_long", "12hours_short", "20min_medium")
    
    dataprovider = DataProviderMongo(db)
    datamanagerV2 = DataManagerV2(dataprovider, compute_double_links=False, use_default_callbacks=False)
    
    nb_items = 0
    with DBTransaction(datamanagerV2=datamanagerV2):
        for item_state in [ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION]:
            for item_type in ITEM_TYPE.ALL_SERVICES:
                items = list(datamanagerV2.find_items(item_type, item_state=item_state, where={'pack': 'aix'}))
                for item in items:
                    to_del = []
                    links = item['use']['links']
                    for link in links:
                        if link['exists']:
                            continue
                        elif link['name'] in to_remove_use:
                            to_del.append(link)
                    for link_to_del in to_del:
                        links.remove(link_to_del)
                    
                    if to_del:
                        g_did_run = True
                        nb_items += 1
                        datamanagerV2.save_item(item, user=user, item_state=item_state, item_type=item_type)
    
    if g_did_run:
        logger.info('[fix_remove_undefined_aix_templates] Removed AIX templates for [%d] items.' % (nb_items))
    else:
        return g_did_run
    
    col = db.get_collection('synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'})
    if not database_info:
        database_info = {'_id': 'database_info'}
    
    database_info['remove_undefined_aix_templates'] = 1
    database_info['must_remake_import'] = 1
    col.save(database_info)
    
    return g_did_run


# Will clean old oracle checks which bloat the pack
@add_fix
@add_doc("""Will remove following oracle checks:
Oracle-$KEY$-enqueue-contention
Oracle-$KEY$-enqueue-waiting
Oracle-$KEY$-event-waits
Oracle-$KEY$-event-waiting
Oracle-$KEY$-latch-contention
Oracle-$KEY$-latch-waiting
Oracle-$KEY$-seg-top10-buffer-busy-waits
Oracle-$KEY$-seg-top10-logical-reads
Oracle-$KEY$-seg-top10-physical-reads
Oracle-$KEY$-seg-top10-row-lock-waits
Oracle-$KEY$-sysstat""")
@add_context_daemons(['synchronizer'])
@add_version('02.03.03-U2')
@auto_launch(False)
@need_shinken_stop(True)
def fix_remove_old_oracle_checks():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    col_stagging = db.get_collection('configuration-stagging-service')
    col_production = db.get_collection('configuration-production-service')
    
    service_to_del = [
        'Oracle-$KEY$-enqueue-contention',
        'Oracle-$KEY$-enqueue-waiting',
        'Oracle-$KEY$-event-waits',
        'Oracle-$KEY$-event-waiting',
        'Oracle-$KEY$-latch-contention',
        'Oracle-$KEY$-latch-waiting',
        'Oracle-$KEY$-seg-top10-buffer-busy-waits',
        'Oracle-$KEY$-seg-top10-logical-reads',
        'Oracle-$KEY$-seg-top10-physical-reads',
        'Oracle-$KEY$-seg-top10-row-lock-waits',
        'Oracle-$KEY$-sysstats'
    ]
    
    to_del = list(col_stagging.find({'service_description': {'$in': service_to_del}}, {'_id': 1, 'service_description': 1}))
    if to_del:
        for todel_item in to_del:
            col_stagging.remove({'_id': todel_item['_id']})
        g_did_run = True
    
    to_del = list(col_production.find({'service_description': {'$in': service_to_del}}, {'_id': 1, 'service_description': 1}))
    if to_del:
        logger.debug("We will delete deprecated check.")
        for todel_item in to_del:
            logger.debug("Deleting check [ %s ]." % todel_item['service_description'])
            col_production.remove({'_id': todel_item['_id']})
        g_did_run = True
    
    return g_did_run


# Will transform non-uppercase custom data keys and duplicate_foreach field
@add_fix
@add_doc("""Will force custom data keys and duplicate foreach to be uppercase""")
@add_context_daemons(['synchronizer'])
@add_version('02.03.03-U2')
@auto_launch(False)
@need_shinken_stop(True)
def fix_custom_data_uppercase():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    to_fix = [db.get_collection('configuration-stagging-service'), db.get_collection('configuration-production-service')]
    
    for col in to_fix:
        for service in col.find({'duplicate_foreach': {'$exists': True}}):
            if service['duplicate_foreach'] != service['duplicate_foreach'].upper():
                g_did_run = True
                service['duplicate_foreach'] = service['duplicate_foreach'].upper()
                logger.debug('Fixed %s' % service.get('name', service.get('service_description', '-- Missing name -- %s' % service)))
                col.update({'_id': service['_id']}, set_update=service)
    
    to_fix = [db.get_collection('configuration-stagging-service'), db.get_collection('configuration-production-service'),
              db.get_collection('configuration-stagging-host'), db.get_collection('configuration-production-host'),
              db.get_collection('configuration-stagging-contact'), db.get_collection('configuration-production-contact')]
    
    for col in to_fix:
        # Checks all items
        for item in col.find({}):
            keys_to_del = []
            for k in item.iterkeys():
                if k[0] == '_' and k != '_id' and k != k.upper():
                    keys_to_del.append(k)
                    g_did_run = True
            for k in keys_to_del:
                item[k.upper()] = item[k]
                del item[k]
            if keys_to_del:
                col.update({'_id': item['_id']}, set_update=item)
                logger.debug('Fixed %s' % item.get('name', item.get('contact_name', item.get('host_name', item.get('service_description', '-- Missing name -- %s' % item)))))
    
    return g_did_run


@add_fix
@add_doc("""Will fix checks and hosts with incoherent flapping thresholds""")
@add_context_daemons(['synchronizer'])
@add_version('02.03.03-U2')
@auto_launch(True)
@need_shinken_stop(True)
def fix_flapping_thresholds():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    to_fix = [db.get_collection('configuration-stagging-service'), db.get_collection('configuration-production-service'),
              db.get_collection('configuration-stagging-host'), db.get_collection('configuration-production-host')]
    
    for col in to_fix:
        for elem in col.find({'low_flap_threshold': {"$exists": 1}, 'high_flap_threshold': {"$exists": 1}}):
            if elem['low_flap_threshold'] > elem['high_flap_threshold']:
                g_did_run = True
                elem['low_flap_threshold'] = elem['high_flap_threshold']
                logger.debug('Fixed %s' % elem.get('name', elem.get('service_description', elem.get('host_name', '-- Missing name -- %s' % elem))))
                col.update({'_id': elem['_id']}, set_update=elem)
    
    return g_did_run


@add_fix
@add_doc("""Will remove host_name field inheritance from check templates""")
@add_context_daemons(['synchronizer'])
@add_version('02.05.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_host_name_inheritance():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    to_fix = [db.get_collection('configuration-stagging-service'), db.get_collection('configuration-production-service')]
    
    for col in to_fix:
        found_to_fix = col.find({'name': {'$exists': True}, 'host_name': {'$exists': True}})
        
        for service in found_to_fix:
            g_did_run = True
            for son in col.find({'use': re.compile('(^|[&|,])\s*%s\s*([&|,]|$)' % service['name']), 'host_name': {'$exists': False}}):
                logger.debug('Applying host_name inheritance %s -> %s' % (service['name'], son.get('service_description', son.get('name', '-- Missing name --'))))
                son['host_name'] = service['host_name']
                col.update({'_id': son['_id']}, set_update=son)
                if son.get('name', ''):
                    found_to_fix.append(son)  # Python supports extending a list that is currently iterated. Saves some recursion tricks
            del service['host_name']
            logger.debug('Removing host_name from %s' % (service['name']))
            col.update({'_id': service['_id']}, set_update=service)
    
    return g_did_run


@add_fix
@add_doc("""Will move bp_rule arguments to a new configuration key called bp_rule""")
@add_context_daemons(['synchronizer'])
@add_version('02.05.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_bp_rule_args():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    to_fix = ['stagging', 'production']
    
    for db_name in to_fix:
        col = db.get_collection('configuration-%s-host' % db_name)
        found_to_fix = list(col.find({'check_command': re.compile(r'^bp_rule.*')}))
        for cluster in found_to_fix:
            g_did_run = True
            cluster['is_cluster'] = '1'
            if '!' in cluster['check_command']:
                cluster['bp_rule'] = '!'.join(cluster['check_command'].split('!')[1:])
            del cluster['check_command']
            col.update({'_id': cluster['_id']}, set_update=cluster)
            logger.debug('Moving cluster %s rules to bp_rule key (in %s)' % (cluster.get('host_name', cluster.get('name', '-- Missing name --')), db_name))
    
    return g_did_run


# It will archive sla older than last week.
@add_fix
@add_doc('Will archive sla older than last week.')
@add_context_daemons(['broker'])
@add_data_type('sla')
@add_version('02.03.03-U1')
@auto_launch(False)
@need_shinken_stop(True)
def fix_no_archive_sla():
    g_did_run = False
    
    mod_conf = {
        'module_name'             : 'sla',
        'module_type'             : 'sla',
        'uri'                     : 'mongodb://localhost',
        'database'                : 'shinken',
        'broker_module_nb_workers': 1,
        'keep_raw_sla_day'        : '7'
    }
    
    try:
        from shinkensolutions.date_helper import Date
        import sys
        sys.path.append('/var/lib/shinken/modules/sla')
        _mod_definition = ShinkenModuleDefinition(params=mod_conf)
        _mod_definition.properties['type'] = 'sla'
        sla_source = imp.load_source('sla', '/var/lib/shinken/modules/sla/module.py')
        sla_module = sla_source.get_instance(_mod_definition, 'broker')
        sla_module.init()
        sla_module.archivator.on_init()
    except Exception as e:
        logger.error(u'The sla module load fail [%s]' % e)
        logger.error(traceback.format_exc())
        return u'The sla module load fail [ %s ]' % e
    
    now = time.time()
    exclude_collection = []
    for day_count in range(0, 1):
        tm = time.localtime(now - day_count * 86400)
        exclude_collection.append("%s_%s" % (tm.tm_yday, tm.tm_year))
    to_archives = []
    logger.debug("Exclude collection [ %s ]" % exclude_collection)
    try:
        for collection_name in sla_module.sla_database.list_name_collections():
            if collection_name in exclude_collection:
                continue
            
            try:
                yday = int(collection_name.split('_')[0])
                year = int(collection_name.split('_')[1])
            except Exception:
                continue
            
            if year > 2000:
                to_archives.append(Date(yday, year))
                logger.debug("to_archive [%s-%s]" % (yday, year))
        
        if not to_archives:
            return g_did_run
        
        for to_archive in to_archives:
            _g_did_run = sla_module.archivator._archive_collection(to_archive)
            todel = sla_module.sla_database.get_collection('has_been_archive_%s_%s' % (to_archive.year, to_archive.yday), None)
            if _g_did_run and todel:
                todel.drop()
            g_did_run = _g_did_run or g_did_run
    except Exception as e:
        raise e
    finally:
        sla_module.stop_all()
    
    return g_did_run


@add_fix
@add_doc('Will remove unknow key from configuration')
@add_context_daemons(['synchronizer'])
@add_version('02.04.00')
@auto_launch(False)
@need_shinken_stop(True)
def fix_remove_unknown_keys():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    try:
        from shinken.synchronizer.dao.def_items import NOT_TO_LOOK
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return g_did_run
    
    data_provider_mongo = DataProviderMongo(db)
    
    for item_type in DEF_ITEMS:
        if item_type == ITEM_TYPE.TIMEPERIODS:
            continue
        _class = DEF_ITEMS[item_type]['class']
        prop_dict = getattr(_class, 'properties', {})
        prop_passthrough_dict = getattr(_class, 'passthrough', {})
        prop_not_inherited_passthrough = getattr(_class, 'not_inherited_passthrough', {})
        # blacklist prop are not allowed
        # prop_blacklist_dict = getattr(_class, 'blacklist', {})
        
        allowed_properties = list(NOT_TO_LOOK) + prop_dict.keys() + prop_passthrough_dict.keys() + prop_not_inherited_passthrough.keys()
        
        for item_state in [ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION, ITEM_STATE.WORKING_AREA]:
            all_items = data_provider_mongo.find_items(item_type, item_state)
            for item in all_items:
                to_del_properties = [property for property in item if (property not in allowed_properties and not property.startswith('_'))]
                
                for property in to_del_properties:
                    print 'remove property [ %s ] in item [%s-%s]' % (property, item_type, get_name_from_type(item_type, item))
                    del item[property]
                    g_did_run = True
                
                if to_del_properties:
                    data_provider_mongo.save_item(item, item_type, item_state)
    
    if not g_did_run:
        return g_did_run
    
    col = db.get_collection('synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'})
    if not database_info:
        database_info = {'_id': 'database_info'}
    
    database_info['remove_unknown_key'] = 1
    database_info['must_remake_import'] = 1
    col.save(database_info)
    
    return g_did_run


@add_fix
@add_doc('Will delete non 0-5 business impact from configuration')
@add_context_daemons(['synchronizer'])
@add_version('02.04.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_business_impact():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    to_fix = [db.get_collection('configuration-stagging-service'), db.get_collection('configuration-production-service'),
              db.get_collection('configuration-stagging-host'), db.get_collection('configuration-production-host'),
              db.get_collection('configuration-stagging-businessimpactmodulation'), db.get_collection('configuration-production-businessimpactmodulation')]
    for col in to_fix:
        for item in col.find({'business_impact': {'$exists': True}}):
            if item['business_impact'] not in ('0', '1', '2', '3', '4', '5'):
                g_did_run = True
                del item['business_impact']
                col.update({'_id': item['_id']}, set_update=item)
    to_fix = [db.get_collection('configuration-stagging-contact'), db.get_collection('configuration-production-contact'),
              db.get_collection('configuration-stagging-notificationway'), db.get_collection('configuration-production-notificationway')]
    for col in to_fix:
        for item in col.find({'min_business_impact': {'$exists': True}}):
            if item['min_business_impact'] not in ('0', '1', '2', '3', '4', '5'):
                g_did_run = True
                del item['min_business_impact']
                col.update({'_id': item['_id']}, set_update=item)
    return g_did_run


@add_fix
@add_doc('Fix wrong uuid in sla info')
@add_context_daemons(['broker'])
@add_data_type('sla')
@add_version('02.05.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_uuid_in_sla_info():
    g_did_run = False
    db = get_broker_db()
    if not db.list_name_collections():
        return g_did_run
    
    col_sla_info = db.get_collection('sla_info')
    to_fixs = list(col_sla_info.find({'_id': re.compile(r'.*-.*-.*')}))
    
    if not to_fixs:
        return g_did_run
    
    col_sla_info.delete_many({'_id': re.compile(r'.*-.*-.*')})
    
    g_did_run = True
    for to_fix in to_fixs:
        to_fix['_id'] = to_fix['_id'][0:to_fix['_id'].rfind('-')]
    
    try:
        col_sla_info.insert_many(to_fixs)
    except BulkWriteError as bwe:
        have_only_dup_key_error = True
        write_errors = ""
        for errmsg in bwe.details['writeErrors']:
            err_msg = errmsg.get("errmsg", "errmsg not found")
            if 'dup key' not in err_msg:
                write_errors += err_msg
                have_only_dup_key_error = False
        
        if not have_only_dup_key_error:
            sanitize_log.write("[SLA][bulk-%s] update_sla bulk fail : %s" % ('sla_info', write_errors), logging.ERROR)
    
    return g_did_run


@add_fix
@add_doc('Will rename keys from configuration')
@add_context_daemons(['synchronizer'])
@add_version('02.04.00')
@auto_launch(False)
@need_shinken_stop(True)
def fix_rename_key(_old, _new):
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    if not _old:
        sanitize_log.write("Please specify a property to rename with --old option", logging.ERROR)
        return False
    if not _new:
        sanitize_log.write("Please specify a property to rename with --new option", logging.ERROR)
        return False
    
    try:
        from shinken.synchronizer.dao.def_items import NOT_TO_LOOK
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return False
    
    data_provider_mongo = DataProviderMongo(db)
    
    asking = False
    for item_type in DEF_ITEMS:
        _class = DEF_ITEMS[item_type]['class']
        prop_dict = getattr(_class, 'properties', {})
        prop_passthrough_dict = getattr(_class, 'passthrough', {})
        prop_not_inherited_passthrough = getattr(_class, 'not_inherited_passthrough', {})
        # blacklist prop are not allowed
        # prop_blacklist_dict = getattr(_class, 'blacklist', {})
        
        allowed_properties = NOT_TO_LOOK + prop_dict.keys() + prop_passthrough_dict.keys() + prop_not_inherited_passthrough.keys()
        
        if _old in allowed_properties:
            asking = True
            break
    
    if asking:
        logger.warn('Warning, the key you attempt to rename is a key used by Shinken, are you sure about renaming this key ? Enter YES to continue.')
        input_value = raw_input()
        if input_value != 'YES':
            logger.warn('Abort')
            return False
    
    for item_type in DEF_ITEMS:
        for item_state in [ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION, ITEM_STATE.WORKING_AREA]:
            all_items = data_provider_mongo.find_items(item_type, item_state)
            for item in all_items:
                if _old in item:
                    item[_new] = item[_old]
                    del item[_old]
                    print 'rename property [ %s ]->[ %s ] in item [%s-%s]' % (_old, _new, item_type, get_name_from_type(item_type, item))
                    data_provider_mongo.save_item(item, item_type, item_state)
                    g_did_run = True
    
    return g_did_run


@add_fix
@add_doc('Add missing add-ons to synchronizer configuration and make sure they are disabled when their status states "disabled"')
@add_context_daemons(['synchronizer'])
@add_version('02.05.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_synchronizer_install_missing_addons():
    g_run = False
    SHINKEN_ADDONS = ["nagvis", "nagvis-shinken-architecture"]
    
    for addon in SHINKEN_ADDONS:
        try:
            subprocess.check_call(['shinken-addons-has', addon], stdout=subprocess.PIPE)
        except subprocess.CalledProcessError as e:
            try:
                p = subprocess.Popen(["shinken-addons-disable", "--force", addon], stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
                (output, _) = p.communicate()
                if "ALREADY DISABLED" not in output:
                    g_run = True
            except subprocess.CalledProcessError as e:
                print "ERROR: Unable to disable Shinken Nagvis addons\n"
    
    return g_run


@add_fix
@add_doc('Add missing configuration files')
@add_context_daemons(['synchronizer'])
@add_version('02.05.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_missing_skeletons():
    SHINKEN_CFG_DIR = "/etc/shinken"
    SYNCHRONIZER_CFG_FILE = "/etc/shinken/synchronizer.cfg"
    g_run = False
    SKELETONS_DIR = "/etc/shinken-skeletons"
    SKELETONS_TO_COPY = ["listeners", "analyzers"]
    
    with open(SYNCHRONIZER_CFG_FILE) as fp:
        synchronizer_cfg = fp.readlines()
    
    cfg_dirs = []
    for cfg_line in synchronizer_cfg:
        cfg_line = cfg_line.strip()
        if cfg_line.startswith("#"):
            continue
        
        if cfg_line.startswith('cfg_dir'):
            cfg_dirs.append(cfg_line.split('=')[1].strip())
    
    lines_to_add = []
    for skel in SKELETONS_TO_COPY:
        skel_dir = os.path.join(SKELETONS_DIR, skel)
        cfg_dir = os.path.join(SHINKEN_CFG_DIR, skel)
        if not os.path.exists(cfg_dir):
            shutil.copytree(skel_dir, cfg_dir)
            g_run = True
            if skel not in cfg_dirs:
                lines_to_add.append('cfg_dir=%s\n' % skel)
    
    if lines_to_add:
        synchronizer_cfg.extend(lines_to_add)
        with open(SYNCHRONIZER_CFG_FILE, 'w') as fp:
            fp.writelines(synchronizer_cfg)
    
    return g_run


@add_fix
@add_doc('Add missing sources for Synchronizer')
@add_context_daemons(['synchronizer'])
@add_version('02.05.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_missing_sources():
    return set_default_values_in_cfg_files('synchronizer', 'source', '/etc/shinken/synchronizers', set(('listener-rest', 'server-analyzer')))


@add_fix
@add_doc('Make sure protected fields parameters are all present and up to date')
@add_context_daemons(['synchronizer'])
@add_version('02.05.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_update_protected_fields_parameters():
    SYNCHRONIZER_CFG_FILE = "/etc/shinken/synchronizer.cfg"
    
    g_run = False
    
    with open(SYNCHRONIZER_CFG_FILE) as fp:
        lines = fp.readlines()
    
    params_to_update = {
        'protect_fields__activate_database_encryption' : {
            'found'     : False,
            'updated'   : False,
            'prev_value': None,
            'next_name' : 'protect_fields__activate_encryption',
            'next_value': None,
            'default'   : '0'},
        'protect_fields__activate_encryption'          : {
            'found'     : False,
            'updated'   : False,
            'prev_value': None,
            'next_name' : 'protect_fields__activate_encryption',
            'next_value': None,
            'default'   : '0'},
        'protect_fields__activate_interface_encryption': {
            'found'     : False,
            'updated'   : False,
            'prev_value': None,
            'next_name' : 'protect_fields__activate_encryption',
            'next_value': None,
            'default'   : '0'},
        'protect_fields__encryption_keyfile'           : {
            'found'     : False,
            'updated'   : False,
            'prev_value': None,
            'next_name' : 'protect_fields__encryption_keyfile',
            'next_value': None,
            'default'   : '/etc/shinken/secrets/protected_fields_key'},
        'protect_fields__substrings_matching_fields'   : {
            'found'     : False,
            'updated'   : False,
            'prev_value': None,
            'next_name' : 'protect_fields__substrings_matching_fields',
            'next_value': None,
            'default'   : 'PASSWORD,PASSPHRASE,PASSE,DOMAINUSER,MSSQLUSER,MYSQLUSER,ORACLE_USER,SSH_USER,LOGIN'},
    }
    
    param_count = {}
    index_of_protected_fields_section = None
    
    # Find all versions of each of the parameters to migrate, and keep the last one
    for (index, line) in enumerate(lines):
        line = line.strip()
        if line.startswith("#"):
            continue
        if not line.startswith("protect_fields__"):
            continue
        
        (param, value) = line.split("=")
        
        if param in params_to_update:
            param_count[param] = param_count.get(param, 0) + 1
            
            if not index_of_protected_fields_section:
                index_of_protected_fields_section = index
            params_to_update[param]['prev_value'] = value
            params_to_update[param]['next_value'] = value
            params_to_update[param]['found'] = True
            params_to_update[params_to_update[param]['next_name']]['prev_value'] = value
            params_to_update[params_to_update[param]['next_name']]['next_value'] = value
            params_to_update[params_to_update[param]['next_name']]['found'] = True
    
    # If the parameters don't appear in the file, their default values will be inserted at top
    if not index_of_protected_fields_section:
        index_of_protected_fields_section = 1
    
    # Undefined parameters get their default values
    for key, param in params_to_update.iteritems():
        if not param['found']:
            param['next_value'] = param['default']
    
    # Prepare new config file to be rewritten with the new version of each parameter
    # Keep the list of lines to delete (those containing duplicate versions of the parameters)
    index_to_delete = []
    for (index, line) in enumerate(lines):
        line = line.strip()
        previous_line = lines[index].strip()
        if line.startswith("#"):
            continue
        if not line.startswith("protect_fields__"):
            continue
        
        (param, value) = line.split("=")
        
        if param in params_to_update:
            current_param = params_to_update[param]
            if current_param['updated'] == False and current_param['next_value'] and not params_to_update[current_param['next_name']]['updated']:
                lines[index] = "%s=%s\n" % (current_param['next_name'], current_param['next_value'])
                current_param['updated'] = True
                params_to_update[current_param['next_name']]['updated'] = True
                if lines[index].strip() != previous_line:
                    g_run = True
            else:
                current_param['updated'] = True
                index_to_delete.append(index)
    
    error_messages = []
    for param, count in param_count.iteritems():
        if count > 1:
            error_messages.append("Param [ %s ] defined multiple times in synchronizer.cfg " % param)
    
    if error_messages:
        error_messages.insert(0, "SANATIZE ABORTED TO PREVENT DATA CORRUPTION")
        return "\n".join(error_messages)
    
    # Remove multiple params definitions
    for index in sorted(index_to_delete, reverse=True):
        g_run = True
        del lines[index]
    
    # Add parameters missing from file
    for param in [params_to_update[current_param] for current_param in params_to_update if not params_to_update[current_param]['updated']]:
        if not params_to_update[param['next_name']]['updated']:
            lines.insert(index_of_protected_fields_section, "%s=%s\n" % (param['next_name'], param['next_value']))
            g_run = True
    
    with open(SYNCHRONIZER_CFG_FILE, "w") as fp:
        fp.writelines(lines)
    
    return g_run


@add_fix
@add_doc('Make sure overload configuration files and main configuration files respect the new configuration files tree')
@add_context_daemons(['synchronizer', 'arbiter'])
@add_version('02.05.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_all_move_overload_files():
    g_run = False
    ROOT_DIR = "/etc/shinken-user/configuration"
    DAEMONS = {
        'arbiters'     : '/etc/shinken/shinken.cfg',
        'synchronizers': '/etc/shinken/synchronizer.cfg',
        # In 2.4, OVERLOAD file for broker is not moved
    }
    shinken_uid, shinken_gid = pwd.getpwnam("shinken")[2:4]
    
    for daemon, daemon_cfg_file in DAEMONS.iteritems():
        
        overload_file = "%s_cfg_overload.cfg" % daemon[:-1]
        daemon_dir = os.path.join(ROOT_DIR, 'daemons', daemon)
        if not os.path.isdir(daemon_dir):
            os.mkdir(daemon_dir)
        
        src_overload_file = os.path.join(ROOT_DIR, overload_file)
        dst_overload_file = os.path.join(ROOT_DIR, 'daemons', daemon, overload_file)
        if os.path.exists(src_overload_file) and not os.path.exists(dst_overload_file):
            shutil.move(src_overload_file, dst_overload_file)
            g_run = True
        
        if not os.path.exists(dst_overload_file):
            # Create empty overload file
            with open(dst_overload_file, "w"):
                pass
            os.chown(dst_overload_file, shinken_uid, shinken_gid)
            g_run = True
        
        with open(daemon_cfg_file) as main_cfg_file_fp:
            daemon_config = main_cfg_file_fp.readlines()
        
        overload_path_found = False
        for lineno, config in enumerate(daemon_config):
            config = config.strip()
            if config.startswith('cfg_file=') and overload_file in config:
                daemon_config[lineno] = "cfg_file=%s\n" % dst_overload_file
                overload_path_found = True
                g_run = g_run or config.split('=')[1].strip() != dst_overload_file
        
        # If not entry for the overload file was found, we add it
        if not overload_path_found:
            daemon_config.append("cfg_file=%s\n" % dst_overload_file)
            g_run = True
        
        with open(daemon_cfg_file, "w") as fp:
            fp.writelines(daemon_config)
    
    return g_run


@add_fix
@add_doc('Remove undesired hostgroups in working area')
@add_context_daemons(['synchronizer'])
@add_version('02.05.00')
@auto_launch(True)
@need_shinken_stop(True)
def remove_hostgroups_in_working_area():
    g_did_run = False
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    col = db.get_collection('configuration-working-area-hostgroup')
    if col.count() > 0:
        col.remove({})
        g_did_run = True
    return g_did_run


@add_fix
@add_doc("Make index for sla module")
@add_context_daemons(['broker'])
@add_version('02.06.00')
@auto_launch(True)
@need_shinken_stop(True)
def make_index_for_sla_module():
    g_did_run = False
    shinken_database = get_broker_db()
    collection_names = shinken_database.list_name_collections()
    if not collection_names:
        return g_did_run
    
    if 'sla_archive' not in collection_names or 'sla_info' not in collection_names:
        return g_did_run
    
    col_archive = shinken_database.get_collection('sla_archive')
    col_sla_info = shinken_database.get_collection('sla_info')
    
    col_sla_info_index_information = col_sla_info.index_information()
    col_archive_index_information = col_archive.index_information()
    
    # Add index for the collections
    if 'names' not in col_sla_info_index_information:
        col_sla_info.ensure_index([('host_name', pymongo.ASCENDING), ('service_description', pymongo.ASCENDING)], name='names')
        g_did_run = True
    if 'monitoring_start_time' not in col_sla_info_index_information:
        col_sla_info.ensure_index([('monitoring_start_time', pymongo.ASCENDING)], name='monitoring_start_time')
        g_did_run = True
    
    if 'hname_type_year_yday' not in col_archive_index_information:
        col_archive.ensure_index([('hname', pymongo.ASCENDING), ('type', pymongo.ASCENDING), ('year', pymongo.ASCENDING), ('yday', pymongo.ASCENDING), ], name='hname_type_year_yday')
        g_did_run = True
    if 'hname_sdesc_type_year_yday' not in col_archive_index_information:
        col_archive.ensure_index([('hname', pymongo.ASCENDING), ('type', pymongo.ASCENDING), ('sdesc', pymongo.ASCENDING), ('year', pymongo.ASCENDING), ('yday', pymongo.ASCENDING), ], name='hname_sdesc_type_year_yday')
        g_did_run = True
    if 'uuid_year_yday_idx' not in col_archive_index_information:
        col_archive.ensure_index([('uuid', pymongo.ASCENDING), ('year', pymongo.ASCENDING), ('yday', pymongo.ASCENDING), ], name='uuid_year_yday_idx')
        g_did_run = True
    if 'uuid_idx' not in col_archive_index_information:
        col_archive.ensure_index([('uuid', pymongo.ASCENDING)], name='uuid_idx')
        g_did_run = True
    if 'version_idx' not in col_archive_index_information:
        col_archive.ensure_index([('version', pymongo.ASCENDING)], name='version_idx')
        g_did_run = True
    
    return g_did_run


@add_fix
@add_doc("Migrate synchronizer's link from name to ids")
@add_context_daemons(['synchronizer'])
@add_version('02.06.00')
@auto_launch(True)
@need_shinken_stop(True)
def migrate_links_name_into_id():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    try:
        from shinken.synchronizer.dao.def_items import ITEM_TYPE
        from shinken.synchronizer.dao.datamanagerV2 import DataManagerV2
        from shinken.synchronizer.dao.dataprovider.dataprovider_memory import DataProviderMemory
        from shinken.synchronizer.dao.item_saving_formatter import build_link
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return g_did_run
    
    col = db.get_collection('synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'}) or {}
    if database_info.get('new_link_format', 0) == 1:
        return g_did_run
    
    data_provider_mongo = DataProviderMongo(db)
    
    managed_states = (ITEM_STATE.STAGGING, ITEM_STATE.NEW, ITEM_STATE.WORKING_AREA, ITEM_STATE.PRODUCTION)
    all_items = {}
    to_save_items = {}
    for item_state in managed_states:
        all_items[item_state] = {}
        to_save_items[item_state] = {}
        for item_type in DEF_ITEMS.iterkeys():
            list_item = data_provider_mongo.find_items(item_type, item_state)
            all_items[item_state][item_type] = list_item
            to_save_items[item_state][item_type] = []
    
    data_provider_memory = DataProviderMemory(all_items)
    datamanagerV2 = DataManagerV2(data_provider_memory, compute_double_links=False, use_default_callbacks=False)
    
    for item_state in managed_states:
        # We need to build links for services first because their links (esp. "host_name") are needed to build links
        # for service_overrides
        all_types_except_services = set(ITEM_TYPE.ALL_TYPES).difference(ITEM_TYPE.ALL_SERVICES)
        for item_type in list(ITEM_TYPE.ALL_SERVICES) + list(all_types_except_services):
            for item in all_items[item_state][item_type]:
                edited = build_link(item, item_type, item_state, datamanagerV2)
                if edited:
                    to_save_items[item_state][item_type].append(item)
    
    for item_state, item_types in to_save_items.iteritems():
        for item_type, to_save_items in item_types.iteritems():
            for to_save_item in to_save_items:
                data_provider_mongo.save_item(to_save_item, item_type=item_type, item_state=item_state)
                g_did_run = True
    
    # The format of merge items change and we don't want to keep any of this items in this collections.
    # They will be rebuild with the must_remake_import = 1 option.
    for item_type in DEF_ITEMS.iterkeys():
        data_provider_mongo.get_collection(item_type, ITEM_STATE.MERGE_SOURCES).drop()
    
    if not g_did_run:
        return g_did_run
    
    col = db.get_collection('synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'})
    if not database_info:
        database_info = {'_id': 'database_info'}
    database_info['new_link_format'] = 1
    database_info['must_remake_import'] = 1
    col.save(database_info)
    return g_did_run


def _cipher_override(data_provider_mongo, db_cipher, item, item_state, item_type, source=''):
    effective = False
    if 'service_overrides' in item:
        item['service_overrides'], effective = db_cipher._read_service_override(item['service_overrides'], db_cipher.cipher_value, item_state, item, item_type, None)
        data_provider_mongo.save_item(item, item_type=item_type, item_state=item_state, item_source=source)
    return effective


# Will make double link for host, host template, hostgroup, contact, contact template, contactgroup, in stagging and production
@add_fix
@add_doc('Will make double link for host, host template, hostgroup, contact, contact template, contactgroup, in stagging and production. (update for 02.06.00)')
@add_context_daemons(['synchronizer'])
@add_version('02.06.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_double_link():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    try:
        from shinken.synchronizer.dao.datamanagerV2 import FALLBACK_USER
        from shinken.synchronizer.dao.def_items import DEF_ITEMS, ITEM_STATE
        from shinken.synchronizer.dao.transactions.transactions import DBTransaction
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return g_did_run
    
    col = db.get_collection('synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'})
    
    if database_info and database_info.get('all_double_link', 0) == 1:
        logger.info(u'This fix has already been executed on this database.')
        return g_did_run
    
    if not database_info or database_info.get('new_link_format', 0) != 1:
        logger.error("You cannot run this fix on your shinken version please upgrade it or run the sanatize 'migrate_links_name_into_id'.")
        return g_did_run
    
    datamanagerV2 = get_datamanager_v2_from_data_provider_metadata(db)
    
    nb_items = 0
    with DBTransaction(datamanagerV2=datamanagerV2):
        for item_state in [ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION]:
            for item_type in DEF_ITEMS:
                items = list(datamanagerV2.find_items(item_type, item_state=item_state))
                for item in items:
                    if datamanagerV2._make_double_link(item, item_type, FALLBACK_USER, item)[2]:
                        g_did_run = True
                        nb_items += 1
    
    if not g_did_run:
        return g_did_run
    
    logger.info('[fix_double_link] make double link for [%d] items.' % nb_items)
    
    col = db.get_collection('synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'})
    if not database_info:
        database_info = {'_id': 'database_info'}
    
    database_info['all_double_link'] = 1
    database_info['must_remake_import'] = 1
    col.save(database_info)
    
    return g_did_run


@add_fix
@add_doc("Encrypt service overrides properties which are protected fields")
@add_context_daemons(['synchronizer'])
@add_version('02.06.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_service_overrides_encryption():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    try:
        from hashlib import sha256
        from shinken.synchronizer.dao.crypto import DatabaseCipher
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return g_did_run
    
    col = db.get_collection('synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'}) or {}
    if database_info.get('new_link_format', 0) == 0:
        logger.warning("The database must have the new link format for this sanatize to run.")
        return g_did_run
    if database_info.get("overrides_encrypted", 0) == 1:
        logger.info("Overrides already encrypted")
        return g_did_run
    
    synchronizer_info = col.find_one({'_id': 'protected_fields_info'}) or {}
    enabled = synchronizer_info.get('protect_fields__activate_database_encryption')
    
    if not enabled:
        logger.info("Encryption not enabled ; this sanatize is not necessary")
        return g_did_run
    
    db_key_name = synchronizer_info.get('protect_fields__encryption_key_name')
    db_key_hash = synchronizer_info.get('protect_fields__encryption_keyfile_hash')
    properties = ",".join(synchronizer_info.get('protect_fields__substrings_matching_fields'))
    keyfile_name = None
    for line in open('/etc/shinken/synchronizer.cfg'):
        if line.startswith('protect_fields__encryption_keyfile'):
            keyfile_name = line.split('=')[1].strip()
    
    if enabled and not keyfile_name:
        logger.critical("Encryption is enabled but the keyfile is not defined in your configuration.")
        return g_did_run
    
    try:
        complete_key = open(keyfile_name).read().strip()
        key_value = complete_key[complete_key.index("|") + 1:]
        key_name = complete_key[:complete_key.index("|")].strip().decode('utf8', 'ignore')
    except (OSError, IOError) as e:
        logger.critical("[protected_fields]  Make sure the file exists and that the user 'shinken' is allowed to read it")
        logger.critical("[protected_fields]  The synchronizer will not start because it will not be able to process encrypted data.")
        logger.critical("[protected_fields]  Cannot read the protected fields secret file '%s' : %s " % (keyfile_name, str(e)))
        return g_did_run
    except ValueError:
        logger.critical("[protected_fields]  The keyfile seems to be corrupted. If you have an export available, you can use it to restore the correct key")
        logger.critical("[protected_fields]  using the command shinken-protected-fields-keyfile-restore and restart the synchronizer.")
        logger.critical("[protected_fields]  The synchronizer will not start in order not to corrupt data.")
        logger.critical("[protected_fields]  The key contained in the keyfile does not have the right structure.")
        return g_did_run
    
    if db_key_hash != sha256(key_value).hexdigest() or db_key_name != key_name:
        logger.critical("protected_fields] The keyfile defined in the synchronizer configuration doesn't match the key present in your database")
        return g_did_run
    
    db_cipher = DatabaseCipher(enabled, properties, keyfile_name)
    
    data_provider_mongo = DataProviderMongo(db)
    
    # Source name is the collection name with the prefix "data-" and the suffix -<type> removed
    data_sources = set([name.replace('data-', '').rsplit('-', 1)[0] for name in db.list_name_collections() if name.startswith('data-')])
    managed_states = [ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA, ITEM_STATE.PRODUCTION, ITEM_STATE.MERGE_SOURCES, ITEM_STATE.RAW_SOURCES]
    
    for item_state in managed_states:
        for item_type in ITEM_TYPE.ALL_HOST_CLASS:
            if item_state == ITEM_STATE.RAW_SOURCES:
                for source in data_sources:
                    for item in data_provider_mongo.find_items(item_type, item_state, item_source=source):
                        g_did_run |= _cipher_override(data_provider_mongo, db_cipher, item, item_state, item_type, source=source)
            else:
                for item in data_provider_mongo.find_items(item_type, item_state):
                    g_did_run |= _cipher_override(data_provider_mongo, db_cipher, item, item_state, item_type)
    
    col = db.get_collection('synchronizer-info')
    database_info = col.find_one({'_id': 'database_info'})
    if not database_info:
        database_info = {'_id': 'database_info'}
    database_info['must_remake_import'] = 1
    database_info['overrides_encrypted'] = 1
    col.save(database_info)
    return g_did_run


@add_fix
@add_doc("Will enable synchronizer-module-database-backup for version < 02.06.00")
@add_context_daemons(['synchronizer'])
@add_version('02.06.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_enable_synchronizer_database_module():
    daemon_type = 'synchronizer'
    daemon_cfg_dir = '/etc/shinken/synchronizers'
    modules_dir = '/etc/shinken/modules/'
    skeleton_module_dir = '/etc/shinken-skeletons/modules/'
    module_file = 'synchronizer-module-database-backup.cfg'
    
    g_run = set_default_values_in_cfg_files(daemon_type, 'modules', daemon_cfg_dir, set(('synchronizer-module-database-backup',)))
    
    if not os.path.exists(os.path.join(modules_dir, module_file)):
        shutil.copy(os.path.join(skeleton_module_dir, module_file), os.path.join(modules_dir, module_file))
        try:
            subprocess.check_call(['chown', 'shinken:shinken', os.path.join(modules_dir, module_file)], stdout=subprocess.PIPE)
        except subprocess.CalledProcessError as e:
            print 'ERROR: Unable to set the rights on file "%s"\n' % os.path.join(modules_dir, module_file)
        g_run = True
    
    return g_run


@add_fix
@add_doc("Will clean DATA in items for escape XSS char")
@add_context_daemons(['synchronizer'])
@add_version('02.06.00')
@auto_launch(True)
@need_shinken_stop(True)
def fix_safety_replacement_character_in_synchronizer_properties():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    try:
        from shinkensolutions.toolbox.box_tools_string import ToolsBoxString
        from shinken.synchronizer.dao.crypto import DatabaseCipher
    except ImportError:
        logger.error(u'You cannot run this fix on your shinken version please upgrade it.')
        return g_did_run
    
    col = db.get_collection(u'synchronizer-info')
    database_info = col.find_one({u'_id': u'database_info'})
    
    # If we find the key data_already_safety_replaced in the database, data has already been modified by this or another sanatize, we don't want to remondify it
    if database_info and database_info.get(u'data_already_safety_replaced', 0) == 1:
        logger.info(u'This fix has already been executed on this database.')
        return g_did_run
    
    synchronizer_info = db.get_collection(u'synchronizer-info').find_one({u'_id': u'protected_fields_info'}) or {}
    encryption_enabled = synchronizer_info.get(u'protect_fields__activate_database_encryption', False)
    encryption_properties = u','.join(synchronizer_info.get(u'protect_fields__substrings_matching_fields', ''))
    encryption_key_file_name = next((line.split(u'=')[1].strip() for line in open(u'/etc/shinken/synchronizer.cfg') if line.startswith(u'protect_fields__encryption_keyfile')), '')
    
    if encryption_enabled and not encryption_key_file_name:
        logger.critical(u'Encryption is enabled but the keyfile is not defined in your configuration.')
        return g_did_run
    
    if encryption_enabled:
        db_cipher = DatabaseCipher(encryption_enabled, encryption_properties, encryption_key_file_name)
    else:
        db_cipher = None
    data_provider_mongo = DataProviderMongo(db, db_cipher)
    
    for item_state in (ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA, ITEM_STATE.PRODUCTION):
        for item_type in ITEM_TYPE.ALL_TYPES:
            for item in data_provider_mongo.find_items(item_type, item_state):
                save_item = False
                for property_name, property_value in item.iteritems():
                    if ((item_type in (ITEM_TYPE.CONTACTS, ITEM_TYPE.HOSTS) and property_name == u'display_name') or property_name.startswith(u'_')) and \
                            property_value and isinstance(property_value, basestring) and \
                            re.search(r'[<>"\'/&]', property_value):
                        property_value_initial = property_value
                        property_value = ToolsBoxString.escape_XSS(property_value)
                        # We unescape & as it not escape anymore
                        property_value = property_value.replace(u'&amp;', u'&')
                        
                        item[property_name] = property_value
                        save_item = save_item | (property_value_initial != property_value)
                if save_item:
                    data_provider_mongo.save_item(item, item_type, item_state)
                    g_did_run = True
    if g_did_run:
        if not database_info:
            database_info = {u'_id': u'database_info'}
        database_info[u'must_remake_import'] = 1
        col.save(database_info)
    
    return g_did_run


@add_fix
@add_doc("Will add internal options to retention mongo cfg")
@add_context_daemons(['scheduler'])
@add_version('02.06.03')
@auto_launch(True)
@need_shinken_stop(True)
def add_internal_option_to_retention_mongo_cfg():
    g_did_run = False
    cfg_file = "/etc/shinken/modules/retention-mongodb.cfg"
    
    updated_file = []
    f = open(cfg_file, 'r')
    line_index = 1
    last_line = 0
    for line in f.readlines():
        updated_file.append(line)
        if "INTERNAL : DO NOT EDIT FOLLOWING PARAMETER WITHOUT YOUR DEDICATED SUPPORT" in line:
            return g_did_run
        if '}' in line:
            last_line = line_index
        line_index += 1
    f.close()
    if last_line == 0:
        return g_did_run
    to_add = "\n" \
             "    #======== INTERNAL options =========\n" \
             "    #INTERNAL : DO NOT EDIT FOLLOWING PARAMETER WITHOUT YOUR DEDICATED SUPPORT\n" \
             "    # == Number of day we conserve retention data, after this time, we will delete data. By default it is 7 days ==\n" \
             "    #nb_of_max_retention_day	    7\n" \
             "    # == maximum number of elements load in one bulk pass ==\n" \
             "    #size_chunk_to_load	        1000\n" \
             "    # == maximum number of elements delete in one bulk pass ==\n" \
             "    #size_chunk_to_load	        1000\n"
    updated_file[last_line - 1:last_line - 1] = to_add
    f = open(cfg_file, 'w')
    f.write("".join(updated_file))
    f.close()
    g_did_run = True
    return g_did_run


@add_fix
@add_doc("Will add ssl parameters to the architecture export module")
@add_context_daemons(['arbiter'])
@add_version('02.07.00')
@auto_launch(True)
@need_shinken_stop(True)
def add_ssl_architecture_export():
    g_did_run = False
    cfg_file_path = "/etc/shinken/modules/architecture-export.cfg"
    
    if not os.path.exists(cfg_file_path):
        logger.info('The file %s doesn\'t exists, skipped' % cfg_file_path)
        return g_did_run
    
    updated_file = []
    with open(cfg_file_path, 'r') as cfg_file:
        line_index = 1
        last_line = 0
        for line in cfg_file.readlines():
            updated_file.append(line)
            if "listener_use_ssl" in line:
                return g_did_run
            if '}' in line:
                last_line = line_index
            line_index += 1
    
    if last_line == 0:
        return g_did_run
    
    to_add = ["\n    # Connection to the shinken-listener\n", "    #   if the connexion parameter to the shinken-listener has been changed, set the new value here.\n", "    #listener_use_ssl                     0\n",
              "    #listener_login                       Shinken\n", "    #listener_password                    OFU2SE4zOU1FMDdaQlJENFtljgwTcWn2hQ5ocksBWS0=\n"]
    updated_file[last_line - 1:last_line - 1] = to_add
    
    with open(cfg_file_path, 'w') as cfg_file:
        cfg_file.write("".join(updated_file))
    
    g_did_run = True
    return g_did_run


@add_fix
@add_doc("Will add ssl parameters to the shinken listener")
@add_context_daemons(['synchronizer'])
@add_version('02.07.00')
@auto_launch(True)
@need_shinken_stop(True)
def add_ssl_shinken_listener():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    synchronizer_info = db.get_collection('synchronizer-info').find_one({'_id': 'protected_fields_info'}) or {}
    encryption_enabled = True if synchronizer_info else False
    encryption_enabled = False if not encryption_enabled or not synchronizer_info['protect_fields__activate_database_encryption'] else True
    encryption_key_file_name = next((line.split('=')[1].strip() for line in open('/etc/shinken/synchronizer.cfg') if line.startswith('protect_fields__encryption_keyfile')), '')
    
    if encryption_enabled:
        if not encryption_key_file_name:
            logger.critical("Encryption is enabled but the keyfile is not defined in your configuration.")
            return g_did_run
        try:
            with open(encryption_key_file_name) as fd:
                data = fd.read()
            key = base64.decodestring(data[data.index("|") + 1:])
            cipher = AESCipher(key)
            password = cipher.encrypt('OFU2SE4zOU1FMDdaQlJENFtljgwTcWn2hQ5ocksBWS0=')
        except:
            logger.critical("Encryption is enabled but we can't take the key from your keyfile %s." % encryption_key_file_name)
            return g_did_run
    else:
        password = 'OFU2SE4zOU1FMDdaQlJENFtljgwTcWn2hQ5ocksBWS0='
    
    source_conf = db.get_collection('sources-configuration')
    listener_shinken_conf = source_conf.find_one({'_id': "listener-shinken"})
    conf_to_apply = {'_id': "listener-shinken", 'configuration': {'authentication': True, 'login': 'Shinken', 'password': password, 'use_ssl': False, 'ssl_key': '', 'ssl_cert': ''}}
    
    if not listener_shinken_conf:
        source_conf.save(conf_to_apply)
        g_did_run = True
    else:
        _keys = listener_shinken_conf.get('configuration', {}).keys()
        if "authentication" in _keys and "ssl_key" in _keys and "ssl_cert" in _keys and listener_shinken_conf.get('configuration', {}).get('authentication') is True:
            return g_did_run
        source_conf.update({'_id': "listener-shinken"}, update={"$set": conf_to_apply})
        g_did_run = True
    return g_did_run


@add_fix
@add_doc('Will normalize _SE_UUIDs so that they all use types instead of classes')
@add_context_daemons(['synchronizer'])
@add_version('02.07.00')
@auto_launch(False)
@need_shinken_stop(True)
def normalize_se_uuids():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    data_provider_mongo = DataProviderMongo(db)
    
    last_synchronization_col = db.get_collection('last_synchronization')
    if last_synchronization_col:
        sources = [source['_id'].rsplit('-', 1)[0] for source in last_synchronization_col.find({}, {'_id': 1})]
    else:
        sources = []
    
    for item_type in DEF_ITEMS:
        _class = DEF_ITEMS[item_type]['class']
        
        for source in sources:
            all_items = data_provider_mongo.find_items(item_type, item_state=ITEM_STATE.RAW_SOURCES, item_source=source)
            g_did_run |= _update_items_se_uuids_and_sync_keys(all_items, data_provider_mongo, ITEM_STATE.RAW_SOURCES, item_type, item_source=source)
        
        for item_state in [ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION, ITEM_STATE.WORKING_AREA, ITEM_STATE.MERGE_SOURCES, ITEM_STATE.NEW]:
            all_items = data_provider_mongo.find_items(item_type, item_state)
            g_did_run |= _update_items_se_uuids_and_sync_keys(all_items, data_provider_mongo, item_state, item_type)
    
    return g_did_run


def _update_items_se_uuids_and_sync_keys(all_items, data_provider_mongo, item_state, item_type, item_source=''):
    import hashlib
    g_did_run = False
    for item in all_items:
        item_se_uuid = item.get('_SE_UUID')
        if not item_se_uuid:
            continue
        
        (core, se_uuid_class_or_type, se_uuid_id) = item_se_uuid.split('-', 2)
        if se_uuid_class_or_type != item_type:
            new_se_uuid = "%s-%s-%s" % (core, item_type, se_uuid_id)
            se_uuid_hash = hashlib.md5(new_se_uuid).hexdigest()
            item['_SE_UUID'] = new_se_uuid
            item['_SE_UUID_HASH'] = se_uuid_hash
            
            if item_se_uuid in item.get('_SYNC_KEYS'):
                item['_SYNC_KEYS'].remove(item_se_uuid)
                item['_SYNC_KEYS'].append(new_se_uuid)
            g_did_run = True
            data_provider_mongo.save_item(item, item_type, item_state, item_source=item_source)
    return g_did_run


@add_fix
@add_doc("Will add rules_path to the discovery module config")
@add_context_daemons(['synchronizer'])
@add_version('02.07.00')
@auto_launch(True)
@need_shinken_stop(True)
def add_rules_path_discovery():
    g_did_run = False
    cfg_file = "/etc/shinken/sources/discovery.cfg"
    
    updated_file = []
    f = open(cfg_file, 'r')
    line_index = 1
    last_line = 0
    for line in f.readlines():
        updated_file.append(line)
        if "rules_path" in line:
            return g_did_run
        if '}' in line:
            last_line = line_index
        line_index += 1
    f.close()
    if last_line == 0:
        return g_did_run
    to_add = [
        "    rules_path              /etc/shinken-user/configuration/daemons/synchronizers/sources/discovery/discovery_rules.json\n"]
    updated_file[last_line - 1:last_line - 1] = to_add
    f = open(cfg_file, 'w')
    f.write("".join(updated_file))
    f.close()
    g_did_run = True
    return g_did_run


@add_fix
@add_doc("Will add nmap_mac_prefixes_path to the discovery module config")
@add_context_daemons(['synchronizer'])
@add_version('02.07.00')
@auto_launch(True)
@need_shinken_stop(True)
def add_nmap_mac_prefixes_path_discovery():
    g_did_run = False
    cfg_file = "/etc/shinken/sources/discovery.cfg"
    
    updated_file = []
    f = open(cfg_file, 'r')
    line_index = 1
    last_line = 0
    for line in f.readlines():
        updated_file.append(line)
        if "nmap_mac_prefixes_path" in line:
            return g_did_run
        if '}' in line:
            last_line = line_index
        line_index += 1
    f.close()
    if last_line == 0:
        return g_did_run
    to_add = [
        "    nmap_mac_prefixes_path  /etc/shinken-user/configuration/daemons/synchronizers/sources/discovery/nmap/nmap-mac-prefixes\n"]
    updated_file[last_line - 1:last_line - 1] = to_add
    f = open(cfg_file, 'w')
    f.write("".join(updated_file))
    f.close()
    g_did_run = True
    return g_did_run


@add_fix
@add_doc('Will update notes_multi_url to the new format')
@add_context_daemons(['synchronizer'])
@add_version('02.07.00')
@auto_launch(True)
@need_shinken_stop(True)
def update_notes_multi_url_syntax():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    # No need to run on source datas because this will be done by the merge validator
    to_fix = [db.get_collection('configuration-stagging-service'), db.get_collection('configuration-production-service'),
              db.get_collection('newelements-service'), db.get_collection('newelements-host'),
              db.get_collection('merge_from_source-service'), db.get_collection('merge_from_source-host'),
              db.get_collection('configuration-stagging-host'), db.get_collection('configuration-production-host'), db.get_collection('configuration-working-area-host')]
    
    for col in to_fix:
        for item in col.find({'notes_multi_url': {'$exists': True}}):
            updated_urls = _compute_updated_notes_multi_url(item['notes_multi_url'])
            if updated_urls:
                item['notes_multi_url'] = updated_urls
                col.update({'_id': item['_id']}, set_update=item)
                g_did_run = True
        
        for item in col.find({'service_overrides': {'$exists': True}}):
            override_updated = False
            for override in item['service_overrides']['links']:
                if override['key'] != 'notes_multi_url':
                    continue
                updated_overrides = _compute_updated_notes_multi_url(override['value'])
                if updated_overrides:
                    override['value'] = updated_overrides
                    override_updated = True
            
            if override_updated:
                col.update({'_id': item['_id']}, set_update=item)
                g_did_run = True
    
    return g_did_run


@add_fix
@add_doc('Will enable cipher/uncipher transformation on your synchronizer collector linker sources')
@add_context_daemons(['synchronizer'])
@add_version('02.07.02')
@auto_launch(True)
@need_shinken_stop(True)
def update_synchronizer_collector_linker_collection_for_cipher_transformation():
    g_did_run = False
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    info = db.get_collection('synchronizer-info')
    protected_fields_info = info.find_one({'_id': 'protected_fields_info'}) or {}
    encryption = protected_fields_info.get('protect_fields__activate_database_encryption', '0') in ('1', True)
    for collection_name in db.list_name_collections(include_system_collections=False):
        if not collection_name.endswith('_confs'):
            continue
        collection = db.get_collection(collection_name)
        items = collection.find()
        for item in items:
            if not item.get('sync_name', None):
                break
            if not '@metadata' in item or ('crypted' not in item['@metadata'] and 'uncrypted' not in item['@metadata']):
                item['@metadata'] = {'uncrypted': True} if not encryption else {'crypted': True}
                g_did_run = True
                collection.save(item)
    return g_did_run


def _compute_updated_notes_multi_url(property_value):
    list_of_urls = property_value.split('~#~')
    updated_urls = []
    for url_spec in list_of_urls:
        url_components = url_spec.split('~=')
        if len(url_components) == 4:
            continue
        elif len(url_components) == 1:  # Simple URL
            url_name = ''
            url_value = url_spec
            url_tag = "tag"
            url_display = "NO"
        elif len(url_components) == 2:  # name~=URL
            (url_name, url_value) = url_components
            url_tag = "tag"
            url_display = "NO"
        else:
            continue
        
        updated_urls.append("%s~=%s~=%s~=%s" % (url_name, url_value, url_tag, url_display))
    if updated_urls:
        return "~#~".join(updated_urls)
    return None


@add_fix
@add_doc("Removes comment on the pidfile lines in mongodb configuration files (mongod, mongos and mongo-configsrv)")
@add_version('02.07.03')
@add_context_daemons(['synchronizer', 'broker', 'scheduler'])
@auto_launch(True)
@need_shinken_stop(False)
def remove_pidfile_comment_in_mongodb_config_files():
    g_did_run = False
    cfg_files = [
        "/etc/mongod.conf",
        "/etc/mongos.conf",
        "/etc/mongo-configsrv.conf",
    ]
    
    for cfg_file in cfg_files:
        if not os.path.exists(cfg_file):
            continue
        
        tmp_cfg_file = "%s.shinken_sanitize" % cfg_file
        
        with open(cfg_file, 'r') as f:
            with open(tmp_cfg_file, "w") as tmp_file:
                for line in f.readlines():
                    if "pidfilepath" in line.lower():
                        if "#" in line:
                            tmp_file.write("%s\n" % line.split("#")[0].rstrip())
                            g_did_run = True
                            continue
                    tmp_file.write(line)
        
        if g_did_run:
            shutil.move(cfg_file, "%s.shinken_sanitize_backup" % cfg_file)
            shutil.move(tmp_cfg_file, cfg_file)
        else:
            os.remove(tmp_cfg_file)
    
    return g_did_run


def _do_add_spare_daemon_to_broker_definition(broker, note_lines, spare_daemon_name=''):
    # Maybe the broker already have this property, skip it
    if broker.have_spare_daemon_property:
        return
    
    broker_file, broker_define_line = broker.imported_from.split(':', 1)
    broker_define_line = int(broker_define_line)
    # print "ADDING TO %s => %s, line %s" % (broker.get_name(), broker_file, broker_define_line)
    
    comments_lines = [
        '    # spare_daemon: name of the daemon that will take this daemon job if it dies',
        '    # IMPORTANT:',
        '    #   * a spare_daemon can only be the spare of 1 (and only one) master daemon',
        '    #   * a spare_daemon cannot have a spare_daemon',
        '    #   * the spare must have modules with the same module_type as the master',
        '    # spare_daemon              broker-spare',
    ]
    comments_lines.extend(['    # %s' % line for line in note_lines])
    
    err = Synchronizer.change_property_into_cfg(broker_file, broker_define_line, 'spare_daemon', spare_daemon_name, comments_lines)
    if err:
        logger.warning('Cannot add the "spare_daemon" in the broker file "%s" : %s' % (broker_file, err))
    logger.debug("FILE EDITED: %s" % broker_file)
    return True


@add_fix
@add_doc("Will add the spare_daemon option to the brokers cfg")
@add_context_daemons(['arbiter'])
@add_version('02.07.06')
@auto_launch(True)
@need_shinken_stop(True)
def fix_new_spare_daemon_broker_option():
    # In this sanatize, we want to have a WARNING level for logging, because maybe we won't find some
    # broker match, and NEED to display them to the admin
    old_level = logger.level
    old_console_level = sanitize_log.get_console_level()
    try:
        if logging.INFO < old_level:
            logger.setLevel(logging.INFO)
        if logging.DEBUG < old_console_level:
            sanitize_log.set_console_level(logging.DEBUG)
        return _fix_new_spare_daemon_broker_option()
    finally:  # Important: reset old levels
        logger.setLevel(old_level)
        sanitize_log.set_console_level(old_console_level)


def __load_full_configuration():
    # Launch the configuration, but without logging
    old_level = logger.level
    logger.setLevel(logging.FATAL)
    cfg_path = '/etc/shinken/shinken.cfg'
    conf = Config()
    buf = conf.read_config([cfg_path])
    if not conf.conf_is_correct:
        logger.setLevel(old_level)
        err = 'The sanatize can\'t load default configuration at [ %s ].' % cfg_path
        raise Exception(err)
    
    logger.setLevel(old_level)
    
    raw_objects = conf.read_config_buf(buf, [])
    conf.load_packs()
    return raw_objects


def __get_and_relink_broker_from_configuration(raw_objects):
    from shinken.brokerlink import BrokerLink
    from shinken.objects.module import Module
    
    ## Relink Modules and Brokers
    module_definitions = raw_objects.get('module', [])
    module_types_by_module_name = {}
    for module_definition in module_definitions:
        module = Module(module_definition)
        module.pythonize()
        module_name = module_definition.get('module_name', '')
        module_type = module_definition.get('module_type', '')
        if module_name and module_type:
            module_types_by_module_name[module_name] = module_type
    
    all_brokers_have_spare_daemon_property = True  # if ALL daemons have the property, we are done
    
    # We are looking only for the broker currently
    brokers = raw_objects.get('broker', [])
    brokers_by_realm = {}
    for broker_define in brokers:
        broker_have_spare_daemon_property = 'spare_daemon' in broker_define
        if not broker_have_spare_daemon_property:
            all_brokers_have_spare_daemon_property = False
        broker = BrokerLink(broker_define)
        broker.pythonize()
        realm = getattr(broker, 'realm', 'All')
        if realm not in brokers_by_realm:
            brokers_by_realm[realm] = []
        brokers_by_realm[realm].append(broker)
        # we need to have modules types (not name), but number of each types
        # is important, so cannot have a set() here
        modules_types = []
        for module_name in broker.modules:
            module_type = module_types_by_module_name.get(module_name, '')
            if module_type:
                modules_types.append(module_type)
        modules_types.sort()  # for comparing, we need to have it sorted
        broker.modules_types = modules_types
        broker.have_spare_daemon_property = broker_have_spare_daemon_property
        logger.debug("%s:: BROKER (%s) => %s (already have the spare_daemon property: %s)" % (realm, broker.get_name(), broker.modules_types, broker.have_spare_daemon_property))
    
    return brokers_by_realm, all_brokers_have_spare_daemon_property


def __pretty_daemon_list_str(daemons):
    return ', '.join([daemon.get_name() for daemon in daemons])


_WARNING = '\033[33m'
_RESET = '\033[0m'


def __fix_new_spare_daemon_broker_one_realm(realm, brokers):
    # WARNING: in all the code below, the broker can be with MISSING PROPERTIES!
    today_str = date.today().strftime('%Y %B %d')
    spare_brokers = [broker for broker in brokers if getattr(broker, 'spare', False)]
    master_brokers = [broker for broker in brokers if not getattr(broker, 'spare', False)]
    logger.debug('%s:: Starting to fix realm broker => Master=%s   Spare=%s ' % (realm, __pretty_daemon_list_str(master_brokers), __pretty_daemon_list_str(spare_brokers)))
    # Maybe there are no spare in this realm => do nothing
    if len(spare_brokers) == 0:
        logger.debug('Realm=%s => there are no spare brokers, skipping this realm' % realm)
        for broker in master_brokers:
            _do_add_spare_daemon_to_broker_definition(broker, ['NOTE: lines added by sanatize (launched at %s on version V02.07.06-Patched04) because realm %s have no spare daemon' % (today_str, realm)])
        return
    
    # Do not touch the ones that already have a valid parameter
    # and also skip/fix disabed daemons
    _already_in_new_conf_daemons = set()
    
    # First manage disabled daemons, we set the property if not set, and forget them for the rest of the sanatize
    both_lists = [master_brokers, spare_brokers]
    for lst in both_lists:
        lst_copy = lst[:]  # we will call .remove in the original one
        for broker in lst_copy:
            # Skip enabled ones
            if getattr(broker, 'enabled', True):
                continue
            spare_daemon_name = getattr(broker, 'spare_daemon', None)
            # Disabled broker need the comment, but only if not already have
            if spare_daemon_name is None:  # not set
                logger.debug("Realm=%s  %s is disabled" % (realm, broker.get_name()))
                notes = ['NOTE: lines added by sanatize (launched at %s) : was enabled=0 during the sanatize, so we skip this daemon.' % today_str]
                _do_add_spare_daemon_to_broker_definition(broker, notes, '')
            # in all case, now this disabled daemon is out of scope
            _already_in_new_conf_daemons.add(broker)
            # this broker is no more need in master or spare list
            lst.remove(broker)
            continue
    
    # For master brokers
    for broker in master_brokers:
        spare_daemon_name = getattr(broker, 'spare_daemon', None)
        
        if spare_daemon_name is None:
            continue
        
        logger.debug("Realm=%s  %s spare => (spare_daemon=%s)" % (realm, broker.get_name(), spare_daemon_name))
        _already_in_new_conf_daemons.add(broker)
        for other_broker in spare_brokers:
            if other_broker.get_name() == spare_daemon_name:
                _already_in_new_conf_daemons.add(other_broker)
                break
    
    # Keep trace of daemons that are NOT already managed
    not_migrated_spare_brokers = [broker for broker in spare_brokers if broker not in _already_in_new_conf_daemons]
    not_migrated_master_brokers = [broker for broker in master_brokers if broker not in _already_in_new_conf_daemons]
    
    logger.debug("Realm=%s Resting brokers after look at spare_daemon property: %s %s" % (realm, __pretty_daemon_list_str(not_migrated_spare_brokers), __pretty_daemon_list_str(not_migrated_master_brokers)))
    
    # Maybe we are rid of all brokers now, exit
    if not not_migrated_master_brokers and not not_migrated_spare_brokers:
        return
    
    # Ok, there are still some brokers that we must find an exact match
    master_by_types = {}
    spare_by_types = {}
    for broker in not_migrated_master_brokers:
        types_str = ', '.join(broker.modules_types)  # we need to have a static object to use as dict key
        if types_str not in master_by_types:
            master_by_types[types_str] = []
        master_by_types[types_str].append(broker)
    
    for broker in not_migrated_spare_brokers:
        types_str = ', '.join(broker.modules_types)  # we need to have a static object to use as dict key
        if types_str not in spare_by_types:
            spare_by_types[types_str] = []
        spare_by_types[types_str].append(broker)
    
    # We want to have exact couples, it means 1 master, and 1 spare with the exact same module types
    # other brokers will raise WARNING about unmatch brokers
    exact_couples = []
    # Now are we able to match brokers with the good types, with 2 and ONLY 2 brokers?
    for (module_types_str, master_brokers) in master_by_types.items():
        spare_brokers = spare_by_types.get(module_types_str, [])
        logger.debug("%s::  %s => %s / %s" % (realm, module_types_str, __pretty_daemon_list_str(master_brokers), __pretty_daemon_list_str(spare_brokers)))
        if len(master_brokers) != 1 or len(spare_brokers) != 1:
            continue
        exact_couples.append((master_brokers[0], spare_brokers[0]))  # [0] => we did check that exists
    
    # Add the spare_daemon property in the daemons that have a valid spare
    for (master_broker, spare_broker) in exact_couples:
        # First, make them as migrated
        not_migrated_master_brokers.remove(master_broker)
        not_migrated_spare_brokers.remove(spare_broker)
        notes = ['NOTE: lines added by sanatize (launched at %s) : founded matched master "%s" <=> spare "%s" in realm %s' % (today_str, master_broker.get_name(), spare_broker.get_name(), realm)]
        _do_add_spare_daemon_to_broker_definition(master_broker, notes, spare_broker.get_name())
        _do_add_spare_daemon_to_broker_definition(spare_broker, notes, '')
    
    # Flag as cannot find all the others brokers
    for broker in not_migrated_master_brokers:
        logger.warning('%s[realm=%-10s] Cannot find valid spare broker for "%s" (one and ONLY one daemon with the same module types)%s' % (_WARNING, realm, broker.get_name(), _RESET))
        _do_add_spare_daemon_to_broker_definition(broker, ['NOTE: lines added by sanatize (launched at %s) because cannot find spare daemon for this daemon (one and ONLY one spare daemon with the same module types)' % (today_str)])
    for broker in not_migrated_spare_brokers:
        logger.warning('%s[realm=%-10s] Cannot find valid master broker for "%s" (one and ONLY one daemon with the same module types)%s' % (_WARNING, realm, broker.get_name(), _RESET))
        _do_add_spare_daemon_to_broker_definition(broker, ['NOTE: lines added by sanatize (launched at %s) because cannot find master daemon for this daemon (one and ONLY one spare daemon with the same module types)' % (today_str)])


def _fix_new_spare_daemon_broker_option():
    g_did_run = False
    
    try:
        raw_objects = __load_full_configuration()
    except Exception as exp:
        logger.error(str(exp))
        return g_did_run
    
    brokers_by_realm, all_brokers_have_spare_daemon_property = __get_and_relink_broker_from_configuration(raw_objects)
    
    # If all daemons already have the property, then quit this sanatize
    if all_brokers_have_spare_daemon_property:
        logger.debug('All the brokers already have the spare_daemon property, so sanatize is not necessary')
        return False
    
    g_did_run = True
    
    for realm, brokers in brokers_by_realm.items():
        __fix_new_spare_daemon_broker_one_realm(realm, brokers)
    
    return g_did_run


def _add_new_options_to_config_file(config_file, bypass_if_line_containing, add_before_line_containing, lines_to_add):
    g_did_run = False
    
    updated_file = []
    with open(config_file, 'r') as f:
        line_index = 1
        line_to_add_before = 0
        for line in f.readlines():
            updated_file.append(line)
            if bypass_if_line_containing in line:
                return g_did_run
            if add_before_line_containing in line:
                line_to_add_before = line_index
            line_index += 1
    
    if line_to_add_before == 0:
        return g_did_run
    updated_file[line_to_add_before - 1:line_to_add_before - 1] = lines_to_add
    with open(config_file, 'w') as f:
        f.write("".join(updated_file))
    g_did_run = True
    return g_did_run


@add_fix
@add_doc("Will add the sla output storing and worker sections to it's config")
@add_context_daemons(['broker'])
@add_version('02.08.00')
@auto_launch(True)
@need_shinken_stop(True)
def add_sla_output_store_and_worker_options():
    g_did_run = False
    config_file = "/etc/shinken/modules/sla.cfg"
    bypass_if_line_containing = 'store_output'
    add_before_line_containing = 'INTERNAL options'
    to_add = [
        '    # ======== SLA stored output ========\n',
        '    # This option enables or disables storing sla outputs.\n',
        '    # If 1, the output will be stored (Default value)\n',
        '    # If 0, the output and long output will not be stored (downtime and acknowledge will still be stored)\n',
        '    # store_output      1\n',
        '\n',
        '    # This option enables or disables storing sla long outputs.\n',
        '    # If 1, the long output will be stored (Default value)\n',
        '    # If 0, the long output will not be stored (output, downtime and acknowledge will still be stored)\n',
        '    # store_long_output     1\n',
        '\n',
        '    # This option will be used to filter which outputs and long outputs to store depending on the status of the sla.\n',
        '    # Possible values are : OK, WARNING, CRITICAL, UNKNOWN\n',
        '    # Seperator is : ,\n',
        '    # Default value : empty (all output states are stored)\n',
        '    # Example to only store OK and UNKNOWN outputs :  list_of_stored_output_status    OK,UNKNOWN\n',
        '    # list_of_stored_output_status\n',
        '    #======== Workers in the broker ========\n'
        '    # This module will use workers in the broker, each worker will manage a shard of all hosts/checks.\n'
        '    # This parameter is used by the broker to set the number of workers. Each worker will use one CPU, which will balance the sla processing load among CPUs.\n'
        '    # default: 1\n'
        '    broker_module_nb_workers     1\n'
        '\n',
    ]
    return _add_new_options_to_config_file(config_file, bypass_if_line_containing, add_before_line_containing, to_add)


@add_fix
@add_doc("Will add the retry options to the synchronize-import config file")
@add_context_daemons(['arbiter'])
@add_version('02.08.00')
@auto_launch(True)
@need_shinken_stop(True)
def add_retry_options_to_synchronizer_import():
    config_file = "/etc/shinken/modules/synchronizer-import.cfg"
    bypass_if_line_containing = 'max_try'
    add_before_line_containing = '}'
    to_add = [
        '\n',
        '    # Maximum number of tries for loading the synchronizer configuration before reporting a failure\n',
        '    #max_try     90\n',
        '\n',
        '    # Sleep time (in seconds) between two successive tries of loading the synchronizer configuration\n',
        '    #sleep_time   2\n',
        '\n'
    ]
    return _add_new_options_to_config_file(config_file, bypass_if_line_containing, add_before_line_containing, to_add)


@add_fix
@add_doc("Will add the map_realm_layout option to the architecture-export config file")
@add_context_daemons(['arbiter'])
@add_version('02.08.00')
@auto_launch(True)
@need_shinken_stop(True)
def add_map_realm_layout_to_architecture_export_module():
    config_file = "/etc/shinken/modules/architecture-export.cfg"
    bypass_if_line_containing = 'map_realm_layout'
    add_before_line_containing = '}'
    to_add = [
        '\n',
        '    # Sort order for realms in the NagVis maps\n',
        '    # - sort_by_name\n',
        '    # - sort_by_size (default)\n',
        '    #map_realm_layout sort_by_size\n',
        '\n'
    ]
    return _add_new_options_to_config_file(config_file, bypass_if_line_containing, add_before_line_containing, to_add)


@add_fix
@add_doc("Cleans up the monitoring packs by overwriting modified files with files from the Shinken installation")
@add_version('02.07.04')
@add_context_daemons(['synchronizer'])
@auto_launch(True)
@need_shinken_stop(True)
def cleanup_monitoring_pack():
    import os
    
    g_did_run = False
    root_path = "/etc/shinken/packs"
    
    for dir, _, files in os.walk(root_path):
        for file in files:
            if file.endswith('.rpmnew'):
                g_did_run = True
                source_file = os.path.join(dir, file)
                dest_file = source_file.replace('.rpmnew', '')
                
                os.unlink(dest_file)
                shutil.move(source_file, dest_file)
    return g_did_run


@add_fix
@add_doc('Make sure carbon-cache daemon (graphite writer) options are up to date')
@add_context_daemons(['broker'])
@add_version('02.07.04')
@auto_launch(False)
@need_shinken_stop(True)
def fix_carbon_cache_parameters():
    CARBON_CACHE_CONF = '/opt/graphite/conf/carbon.conf'
    
    did_run = False
    
    with open(CARBON_CACHE_CONF) as fp:
        lines = fp.readlines()
    forced_parameters = {'USER': 'apache', 'MAX_UPDATES_PER_SECOND': 'inf', 'MAX_CREATES_PER_MINUTE': 'inf'}
    
    new_lines = []
    
    for line in lines:
        line = line.strip()
        line = line + '\n'  # be sure to be in unix format
        if line.startswith("#"):
            new_lines.append(line)
            continue
        if '=' not in line:
            new_lines.append(line)
            continue
        (param, value) = line.split("=", 1)
        param = param.strip()
        value = value.strip()
        if param not in forced_parameters.keys():
            new_lines.append(line)
            continue
        forced_value = forced_parameters[param]
        if value != forced_value:
            logger.info('Updating the carbon-cache parameter %s:%s=>%s' % (param, value, forced_value))
            did_run = True
            new_lines.append('%s = %s\n' % (param, forced_value))
        else:
            logger.debug('The carbon-cache parameter %s is already at %s' % (param, value))
            new_lines.append(line)
    
    if did_run:
        logger.info('Writing the carbon cache configuration file %s with updated parameters' % CARBON_CACHE_CONF)
        with open(CARBON_CACHE_CONF, "w") as fp:
            fp.writelines(new_lines)
        try:
            subprocess.check_call(['/etc/init.d/carbon-cache', 'restart'], stdout=subprocess.PIPE)
        except subprocess.CalledProcessError as exp:
            print 'ERROR: Unable to restart carbon-cache"\n' % exp
    
    return did_run


@add_fix
@add_doc('Clean old Nagvis-shinken .tmp file')
@add_context_daemons(['arbiter'])
@add_version('02.08.01')
@auto_launch(True)
@need_shinken_stop(True)
def clean_old_nagvis_shinken_tmp_files():
    g_did_run = False
    
    nagvis_path = '/etc/shinken/external/nagvis/'
    paths_to_check = ['etc/maps', 'etc/conf.d/', 'share/userfiles/images/maps/']
    
    for tmp_path in paths_to_check:
        full_path = os.path.join(nagvis_path, tmp_path, '*.tmp')
        file_list = glob.glob(full_path)
        
        if file_list:
            g_did_run = True
            for file_path in file_list:
                logger.debug('File to delete %s' % file_path)
                try:
                    os.unlink(file_path)
                except OSError:
                    logger.error('The file %s can not be deleted' % file_path)
                    return None
    
    return g_did_run


@add_fix
@add_doc('Remove presence_protection from elements that are not shinken-core')
@add_context_daemons(['synchronizer'])
@add_version('02.07.05')
@auto_launch(True)
@need_shinken_stop(True)
def remove_presence_protection():
    g_did_run = False
    
    try:
        from shinken.synchronizer.dao.def_items import NAGIOS_TABLE_KEYS, ITEM_STATE
        from shinken.synchronizer.business.core_item import CORE_ITEMS
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return g_did_run
    
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    ids_to_skip = []
    for item_type, items in CORE_ITEMS.iteritems():
        ids_to_skip.extend([e['_id'] for e in items])
    
    for item_state in (ITEM_STATE.MERGE_SOURCES, ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION):
        for (item_class, key_name) in NAGIOS_TABLE_KEYS.iteritems():
            col = db.get_collection('configuration-%s-%s' % (item_state, item_class))
            items = col.find({'presence_protection': '1'})
            
            for item in items:
                if item.get('_id', '') in ids_to_skip:
                    continue
                
                did_run = item.pop('presence_protection', None)
                
                if did_run:
                    g_did_run = True
                    logger.info('Remove presence_protection from item %s:%s in state %s ' % (item_class, item.get('name', item.get(key_name, item.get('service_description'))), item_state))
                    col.save(item)
    
    return g_did_run


@add_fix
@add_doc('Change the format of work_area_info so that multiple users can own the changes.')
@add_context_daemons(['synchronizer'])
@add_version('02.07.06')
@auto_launch(True)
@need_shinken_stop(True)
def change_work_area_info_for_multiple_users():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    try:
        from shinken.synchronizer.business.item_controller.work_area_helper import WORK_AREA_INFO_KEY
    except ImportError:
        logger.error("You cannot run this fix on your shinken version please upgrade it.")
        return g_did_run
    
    data_provider_mongo = DataProviderMongo(db)
    for item_state in [ITEM_STATE.NEW, ITEM_STATE.MERGE_SOURCES, ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION]:
        for item_type in DEF_ITEMS.iterkeys():
            items = data_provider_mongo.find_items(item_type, item_state)
            for item in items:
                _item_modified = False
                _work_area_info = item.get(WORK_AREA_INFO_KEY, {})
                if isinstance(_work_area_info.get('get_by_user', []), basestring):
                    _work_area_info['get_by_user'] = [_work_area_info['get_by_user']]
                    _item_modified = True
                if isinstance(_work_area_info.get('get_by_user_name', []), basestring):
                    _work_area_info['get_by_user_name'] = [_work_area_info['get_by_user_name']]
                    _item_modified = True
                if _item_modified:
                    data_provider_mongo.save_item(item, item_type, item_state)
                    g_did_run = True
    return g_did_run


@add_fix
@add_doc('This fix delete duplicates service overrides.')
@add_context_daemons(['synchronizer'])
@add_version('02.08.01')
@auto_launch(True)
def deleted_service_override_useless():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    try:
        from shinken.synchronizer.dao.def_items import SERVICE_OVERRIDE
    except ImportError:
        logger.error(u'You cannot run this fix on your shinken version please upgrade it.')
        return g_did_run
    
    data_provider_mongo = DataProviderMongo(db)
    all_checks = data_provider_mongo.find_items(ITEM_TYPE.ALL_SERVICES, ITEM_STATE.STAGGING)
    all_checks = dict([(item[u'_id'], item) for item in all_checks])
    for item_state in [ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING]:
        all_hosts = data_provider_mongo.find_items(ITEM_TYPE.ALL_HOST_CLASS, item_state)
        for host in all_hosts:
            service_override = host.get(SERVICE_OVERRIDE, {})
            if not service_override:
                continue
            
            to_remove = []
            for link in service_override[u'links']:
                if not link.get(u'has_already_been_linked', False):
                    continue
                check_link = link[u'check_link']
                if check_link[u'exists']:
                    check_id = check_link[u'_id']
                    check = all_checks.get(check_id, {})
                    if check and check.get(u'duplicate_foreach', None) and not link.get(u'dfe_key', None):
                        to_remove.append(link)
                elif u'$KEY$' in check_link[u'name']:
                    to_remove.append(link)
            
            if to_remove:
                service_override[u'links'] = [lk for lk in service_override[u'links'] if lk not in to_remove]
                if not service_override[u'links']:
                    del host[SERVICE_OVERRIDE]
                item_type = METADATA.get_metadata(host, METADATA.ITEM_TYPE)
                data_provider_mongo.save_item(host, item_type, item_state)
                g_did_run = True
    
    return g_did_run


@add_fix
@add_doc('This sanatize rename the graphite perfdata metric files.')
@add_context_daemons([])  # To make the function run in all daemons
@add_version('02.07.06')
@auto_launch(True)
@need_shinken_stop(True)
def rename_graphite_scheduler_checks_metrics_files():
    folders_to_rename = {
        u'nb_poller.wsp'                 : u'nb_pollers.wsp',
        u'nb_poller_in_overload.wsp'     : u'nb_pollers_in_overload.wsp',
        u'nb_reactionner.wsp'            : u'nb_reactionners.wsp',
        u'nb_reactionner_in_overload.wsp': u'nb_reactionners_in_overload.wsp',
        u'notifications_done_by_sec.wsp' : u'notifications_and_event_handlers_done_by_sec.wsp',
        u'nb_late_event.wsp'             : u'nb_late_event_handlers.wsp'
        
    }
    return rename_graphite_checks_metrics_files(folders_to_rename)


@add_fix
@add_doc('This sanatize replace the duplicates uuid in each dashboard widgets by new ones.')
@add_context_daemons(['broker'])
@add_version('02.07.06')
@auto_launch(True)
@need_shinken_stop(True)
def replace_duplicate_widgets_uuids():
    g_did_run = False  # did we fix something?
    db = get_broker_db()
    collection_names = db.list_name_collections()
    if not collection_names:
        return g_did_run
    
    # Clean the "webui-info" collection
    if u'webui-info' in collection_names:
        logger.info(u'The sanatize were already launch one time but we need to re-launch it to change webui_info format')
        old_col_webui_info = db.get_collection(u'webui-info')
        old_col_webui_info.drop()
        g_did_run = True
    
    col_webui_info = db.get_collection('webui_info')
    database_info = col_webui_info.find_one({'_id': 'database_info'}) or {}
    if database_info.get('replace_duplicate_widgets_uuids', 0) == 1:
        return g_did_run
    
    col_dashboards = db.get_collection(u'dashboard')
    dashboards = col_dashboards.find({})
    
    for dashboard in dashboards:
        if u'widgets' not in dashboard:
            continue
        
        widgets = dashboard.get(u'widgets', list())
        for widget in widgets:
            widget_id = widget.get(u'uuid', u'')
            if not widget_id:
                widget_id = _generate_uuid()
                widget[u'uuid'] = widget_id
            nb_dupes = len([w for w in widgets if w.get(u'uuid', u'') == widget_id])
            if nb_dupes > 1:
                new_uuid = _generate_uuid()
                widget[u'uuid'] = new_uuid
                g_did_run = True
        col_dashboards.save(dashboard)
    
    col_webui_info.save({'_id': 'database_info', 'replace_duplicate_widgets_uuids': 1})
    return g_did_run


@add_fix
@add_doc('This sanatize replace the 10 000 value of daily_clean_batch_size into 10000.')
@add_context_daemons(['arbiter'])
@add_version('02.07.06')
@auto_launch(True)
@need_shinken_stop(True)
def replace_bad_formatted_sla_option_daily_clean_batch_size():
    code, _, _ = run_command_with_return_code(u'''grep 'daily_clean_batch_size.*10 000' /etc/shinken/*/*.cfg''')
    if code == 0:
        run_command_with_return_code(u'''sed  -i 's/daily_clean_batch_size.*10 000/daily_clean_batch_size        10000/g'  /etc/shinken/*/*.cfg''')
        return True
    else:
        return False


def _generate_uuid():
    # type: () -> unicode
    return u'%s%s-%s-%s-%s-%s%s%s' % (
        _get_random_hexadecimal_as_string(), _get_random_hexadecimal_as_string(), _get_random_hexadecimal_as_string(), _get_random_hexadecimal_as_string(), _get_random_hexadecimal_as_string(), _get_random_hexadecimal_as_string(),
        _get_random_hexadecimal_as_string(), _get_random_hexadecimal_as_string())


def _get_random_hexadecimal_as_string():
    # type: () -> unicode
    number = u'%x' % math.floor(random.randint(0x1, 0x10000))
    return number.ljust(4, u'0')


def _int_to_base_16_string(integer):
    # type: (int) -> unicode
    return u'%x' % integer


@add_fix
@add_doc('This sanatize rename the graphite perfdata metric files.')
@add_context_daemons([])  # To make the function run in all daemons
@add_version('02.07.06')
@auto_launch(True)
@need_shinken_stop(True)
def rename_graphite_vmware_cpu_stolen_checks_metrics_files():
    folders_to_rename = {
        u'cpu_stolen__vmware__percent_ready.wsp': u'cpu_stolen__vmware.wsp',
    }
    return rename_graphite_checks_metrics_files(folders_to_rename)


@need_shinken_stop(False)
def remove_invalid_service_overrides():
    # We use print to display information in the following function so we want the original stderr and stdout
    sys.stdout = orig_stdout
    sys.stderr = orig_stderr
    return print_or_delete_invalid_service_overrides(opts.only_summary)


@add_fix
@add_doc('This sanatize decode data encoded before save.')
@add_context_daemons([u'synchronizer'])
@add_version('02.07.06')
@auto_launch(True)
@need_shinken_stop(True)
def safety_replacement_encoded_character_in_data():
    g_did_run = False  # did we fix something?
    db = get_synchronizer_db()
    if not db.list_name_collections():
        return g_did_run
    
    try:
        from shinken.synchronizer.dao.dataprovider.dataprovider_mongo import DataProviderMongo
        from shinken.synchronizer.dao.def_items import ITEM_STATE, ITEM_TYPE, SERVICE_OVERRIDE, METADATA, NOT_TO_LOOK
        from shinkensolutions.toolbox.box_tools_string import ToolsBoxString
    except ImportError:
        logger.error(u'You cannot run this fix on your shinken version please upgrade it.')
        return g_did_run
    
    if check_sanatize_database_info_flag(db, u'data_already_safety_replaced', 1):
        logger.info(u'This fix has already been executed on this database.')
        return g_did_run
    
    data_provider_mongo = DataProviderMongo(db)
    
    types_with_data = ITEM_TYPE.ALL_HOST_CLASS | ITEM_TYPE.ALL_SERVICES | set(ITEM_TYPE.SONS[ITEM_TYPE.CONTACTTPLS])
    
    # Source name is the collection name with the prefix "data-" and the suffix -<type> removed
    data_sources = set([name.replace(u'data-', u'').rsplit(u'-', 1)[0] for name in db.list_name_collections() if name.startswith(u'data-')])
    managed_states = [ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA, ITEM_STATE.PRODUCTION, ITEM_STATE.RAW_SOURCES]
    
    for item_state in managed_states:
        if item_state == ITEM_STATE.RAW_SOURCES:
            for source in data_sources:
                for item in data_provider_mongo.find_items(types_with_data, item_state, item_source=source):
                    for _property, value in item.iteritems():
                        if _property in NOT_TO_LOOK:
                            continue
                        if _property.startswith(u'_'):
                            item[_property] = ToolsBoxString.unescape_XSS(item[_property])
                            item[_property] = item[_property].replace(u'&#x27;', u'\'')
                    item_type = METADATA.get_metadata(item, METADATA.ITEM_TYPE)
                    if item_type in ITEM_TYPE.ALL_HOST_CLASS:
                        service_overrides = item.get(SERVICE_OVERRIDE, u'')
                        if service_overrides:
                            item[SERVICE_OVERRIDE] = ToolsBoxString.unescape_XSS(service_overrides.replace(u'&#x27;', u'\''))
                    data_provider_mongo.save_item(item, item_type, item_state, item_source=source)
        else:
            for element in data_provider_mongo.find_items(types_with_data, item_state):
                for _property, value in element.iteritems():
                    if _property in NOT_TO_LOOK:
                        continue
                    if _property.startswith(u'_'):
                        element[_property] = ToolsBoxString.unescape_XSS(value)
                        element[_property] = element[_property].replace(u'&#x27;', u'\'')
                
                item_type = METADATA.get_metadata(element, METADATA.ITEM_TYPE)
                if item_type in ITEM_TYPE.ALL_HOST_CLASS:
                    service_overrides = element.get(SERVICE_OVERRIDE, {})
                    if service_overrides:
                        for service_override in service_overrides.get(u'links', []):
                            if service_override[u'key'].startswith(u'_'):
                                service_override[u'value'] = ToolsBoxString.unescape_XSS(service_override[u'value'])
                                service_override[u'value'] = service_override[u'value'].replace(u'&#x27;', u'\'')
                    if service_overrides.get(u'raw_value', u''):
                        service_overrides[u'raw_value'] = ToolsBoxString.unescape_XSS(service_overrides[u'raw_value'])
                        service_overrides[u'raw_value'] = service_overrides[u'raw_value'].replace(u'&#x27;', u'\'')
                
                data_provider_mongo.save_item(element, item_type, item_state)
        g_did_run = True
    
    update_database_info_collection(db, u'data_already_safety_replaced', 1)
    update_database_info_collection(db, u'must_remake_import', 1)
    
    return g_did_run


@add_fix
@add_doc('This sanatize clean invalid sla collection if needed.')
@add_context_daemons([u'broker'])
@add_version('02.08.01')
@auto_launch(True)
@need_shinken_stop(True)
def clean_invalid_sla_collection():
    g_did_run = False
    
    db = get_broker_db()
    collection_names = db.list_name_collections()
    if not collection_names:
        return g_did_run
    
    col_sla_info = db.get_collection('sla_info')
    sla_info = col_sla_info.find_one({'_id': 'SLA_INFO'})
    
    if not sla_info:
        return g_did_run
    
    if sla_info.get('clean_invalid_done_at_update', 0) >= 1:
        return g_did_run
    
    _date_now = date_now()
    for collection_name in collection_names:
        if collection_name.startswith('invalide_'):
            db.get_collection(collection_name).drop()
            g_did_run = True
        
        if re.match(r'[0-9]{1,3}_[0-9]{4}', collection_name):
            split = collection_name.split('_')
            collection_date = Date(int(split[0]), int(split[1]))
            diff_date = get_diff_date(_date_now, collection_date)
            
            # We also drop all raw sla collection too old or in future
            # raw sla collection of last 7 day are remove if there are already archive
            must_del = not (0 <= diff_date <= 7)
            if must_del:
                db.get_collection(collection_name).drop()
                g_did_run = True
            elif diff_date != 0:
                if 'has_been_archive_%s' % collection_name in collection_names:
                    db.get_collection(collection_name).drop()
    
    col_sla_info.update({'_id': 'SLA_INFO'}, update={'$set': {'_id': 'SLA_INFO', 'clean_invalid_done_at_update': 1}}, upsert=True)
    
    return g_did_run


@add_fix
@add_doc(u'This sanatize update configuration files format')
@add_context_daemons([u'arbiter'])
@add_version(u'02.08.01')
@auto_launch(True)
@need_shinken_stop(True)
def update_configuration_file_format():
    g_did_run = False
    
    try:
        raw_objects = __load_full_configuration()
    except Exception as err:
        logger.error(str(err))
        return g_did_run
    
    format_functions = {
        u'broker_module_livedata'      : convert_broker_module_livedata_configuration_files_to_new_format,
        u'livedata_module_sla_provider': convert_livedata_module_sla_provider_configuration_files_to_new_format
    }
    
    modules = raw_objects.get(u'module', [])
    _max_fix_name_length = 0
    result = {}
    _message_template_module = u'   \033[%%dm - Updating %%-%ds ' % (sanatize_decorator.NUMBER_INDENT_SPACES - len(u'- Updating '))
    _message_template_file_modified = u'         \033[%%dm %%-%ds \033[0m:  \033[%%dm %%s \033[0m' % (sanatize_decorator.NUMBER_INDENT_SPACES - 12)
    
    summary = []
    for module_type, func in format_functions.iteritems():
        updated_files = func(modules)
        result[module_type] = updated_files
    
    for module_type, _updated_files in result.iteritems():
        
        if True in _updated_files.values():
            
            summary.append(_message_template_module % (35, module_type))
            
            for file_path, g_did_run_file in _updated_files.iteritems():
                log_color, to_log = get_output_and_color(g_did_run_file)
                summary.append(_message_template_file_modified % (35, file_path, log_color, to_log))
            
            g_did_run = True
    
    SANATIZE_SUMMARY[u'update_configuration_file_format'] = summary
    
    return g_did_run


@add_fix
@add_doc('This sanatize decode data encoded before save.')
@add_context_daemons([])
@add_version('02.07.06')
@auto_launch(True)
@need_shinken_stop(False)
def replace_wmic_symlink_depending_on_os_version():
    link_path = u'/var/lib/shinken/libexec/wmic'
    python_minor = sys.version_info[1]
    os_version = u'centos6' if python_minor == 6 else u'centos7'
    
    g_did_run = create_symlink_from_os_version(link_path, os_version)
    os.chmod(link_path, 0o755)  # -rwxr-xr-x
    return g_did_run


# Open a file and search if a string is in it
def __file_have_string(file_path, s):
    # type: (unicode, unicode) -> Bool
    with codecs.open(file_path, 'r', 'utf8') as f:
        return s in f.read()


# We are adding the comment for the parameter broker__manage_brok__sub_process_broks_pusher_queue_batch_size
# on all broker.cfg files
@add_fix
@add_doc(u"Will add the broker__manage_brok__sub_process_broks_pusher_queue_batch_size option to the brokers cfg")
@add_context_daemons([u'arbiter'])
@add_version(u'02.07.06')
@auto_launch(True)
def fix_new_broker_queue_batch_size_parameter():
    try:
        from shinken.synchronizer.synchronizerdaemon import Synchronizer
    except ImportError:
        from synchronizer.synchronizerdaemon import Synchronizer
    
    g_did_run = False
    
    try:
        raw_objects = __load_full_configuration()
    except Exception as exp:
        logger.error(str(exp))
        return g_did_run
    
    brokers_by_realm, all_brokers_have_spare_daemon_property = __get_and_relink_broker_from_configuration(raw_objects)
    
    for realm, brokers in brokers_by_realm.items():
        
        for broker in brokers:
            broker_file, broker_define_line = broker.imported_from.split(':', 1)
            broker_define_line = int(broker_define_line)
            
            # If the broker is already updated, skip it
            if __file_have_string(broker_file, u'broker__manage_brok__sub_process_broks_pusher_queue_batch_size'):
                logger.debug(u'No need to update file: %s' % broker_file)
                continue
            logger.debug(u'NEED TO MODIFY FILE %s' % broker_file)
            
            # We did works, let the sanatize know it
            g_did_run = True
            
            comments_lines = [
                u'',  # so we are separated from other blocs
                u'    # broker__manage_brok__sub_process_broks_pusher_queue_batch_size:',
                u'    #   * defines the maximum number of broks the "queue brok pusher" ',
                u'    #     process will handle per send to external module ( like WebUI ) .',
                u'    #   * Remaining broks will be handled in next send.',
                u'    #   * IMPORTANT: increase this value can lead to error on the socket',
                u'    # Default: 100000 (broks/batch)',
                u'    # broker__manage_brok__sub_process_broks_pusher_queue_batch_size      100000',
            ]
            
            # NOTE: '#' -> so won't match any property, add # void, and set the comments
            err = Synchronizer.change_property_into_cfg(broker_file, broker_define_line, '#', '', comments_lines)
            if err:
                logger.warning(u'Cannot add the "spare_daemon" in the broker file "%s" : %s' % (broker_file, err))
            
            logger.debug(u"FILE EDITED: %s" % broker_file)
    
    return g_did_run


@add_fix
@add_doc(u"Will remove the broken property link 'service_excludes_by_id' on hosts, host templates and clusters")
@add_context_daemons([u'synchronizer'])
@add_version(u'02.07.06')
@auto_launch(True)
def fix_broken_links_to_service_excludes_by_id():
    return remove_empty_service_excludes_by_id_links()


@add_fix
@add_doc(u"Will comment host_check_timeout parameter in shinken.cfg and add comment above the parameter")
@add_context_daemons([u'arbiter'])
@add_version(u'02.07.06')
@auto_launch(True)
def comment_host_check_timeout_and_add_explanation():
    g_did_run = False
    comment_lines = [
        u'# /!\\\\ USELESS host_check_timeout PARAMETER',
        u'# The parameter "host_check_timeout" is no more used in Shinken. Use check_running_timeout instead.',
        u'#host_check_timeout'
    ]
    _code, _, __ = run_command_with_return_code(u'''grep 'host_check_timeout' /etc/shinken/shinken.cfg''')
    if _code != 0:
        return g_did_run
    
    for line in comment_lines[:2]:
        _code, _, __ = run_command_with_return_code(u'''grep '%s' /etc/shinken/shinken.cfg''' % line)
        if _code == 0:
            return g_did_run
    
    sed_command = u'''sed  -i 's?host_check_timeout?%s?g'  /etc/shinken/shinken.cfg''' % u'\\n'.join(comment_lines)
    _code, _stdout, _stderr = run_command_with_return_code(sed_command)
    if _code == 0:
        g_did_run = True
    else:
        g_did_run = False
        logger.error(_stderr)
    
    return g_did_run


@add_fix
@add_doc(u"Will comment host_check_timeout parameter in shinken.cfg and add comment above the parameter")
@add_context_daemons([u'arbiter'])
@add_version(u'02.07.06')
@auto_launch(True)
def remove_templates_with_forbidden_characters():
    g_did_run = replace_forbidden_chars_from_used_templates()
    if g_did_run:
        msg = u'\t\t%s: Renaming summary in file : %s' % (cyan(u'Info'), HOST_TPL_FORBIDDEN_CHAR_SANATIZE_SUMMARY_FILE)
        
        SANATIZE_SUMMARY[u'remove_templates_with_forbidden_characters'] = [msg]
    return g_did_run


if __name__ == '__main__':
    logger.handlers = []
    logger_handler = logging.StreamHandler()
    logger.addHandler(logger_handler)
    logger_handler.setFormatter(logging.Formatter('%(levelname)s: %(message)s'))
    logger.setLevel('INFO')
    
    parser = optparse.OptionParser("%prog ", version="%prog: " + VERSION, description='This tool  is used to check the state of your Shinken Enterprise installation and configuration')
    parser.add_option('-l', '--list', dest='list_only', action='store_true', help="List available fixes")
    parser.add_option('-r', '--run', dest='run_one', help="Run only one specific fix")
    parser.add_option('-a', '--auto', dest='run_auto', action='store_true', help="Run automatic fixes at once")
    parser.add_option('', '--all', dest='run_all', action='store_true', help="Run ALL fixes at once")
    parser.add_option('-v', '--verbose', dest='verbose', action='store_true', help="Show verbose output")
    parser.add_option('-q', '--quiet', dest='quiet', action='store_true', help="Do not output when no fix has been done")
    parser.add_option('-n', '--no-hint', dest='no_hint', action='store_true', help="Do not display hints for users")
    parser.add_option('-L', '--log_file', dest='log_file', default=DEFAULT_LOG_FILENAME, help="The log file path")
    parser.add_option('', '--old', dest='old', default='', help="Key to rename")
    parser.add_option('', '--new', dest='new', default='', help="New name of the key")
    parser.add_option('', '--item-type', dest='item_type', default='', help="use by fix 'fix_regenerate_sync_keys' to choose item type to regenerate")
    parser.add_option('', '--only-summary', dest='only_summary', action='store_true', default=False, help="use by fix 'delete_invalid_service_overrides' to only display the summary")
    parser.add_option('', '--mongo-host', dest='mongo_host', default='localhost', help="MongoDB server hostname")
    parser.add_option('', '--mongo-port', dest='mongo_port', default='27017', help="MongoDB server port")
    parser.add_option('', '--mongo-use-ssh', dest='mongo_use_ssh', default=False, action='store_true', help="use SSH to connect to MongoDB server")
    parser.add_option('', '--mongo-ssh-key', dest='mongo_ssh_key', default='/var/lib/shinken/.ssh/id_rsa', help="SSH private key used to connect to MongoDB server")
    parser.add_option('', '--mongo-ssh-user', dest='mongo_ssh_user', default='shinken', help="user on MongoDB server to connect to with SSH")
    
    opts, args = parser.parse_args()
    
    # Look if the user ask for local or global, and if not, guess
    list_only = opts.list_only
    run_all = opts.run_all
    run_one = opts.run_one
    run_auto = opts.run_auto
    
    sanitize_log.set_console_level(logging.INFO)
    if opts.verbose or list_only:
        logger.setLevel('DEBUG')
        sanitize_log.set_console_level(logging.DEBUG)
    sanitize_log.set_file_handler(os.path.expanduser(opts.log_file))
    
    log.debug("------------------------------------------------------------------------------")
    log.debug("------------------------------------------------------------------------------")
    log.debug("Running sanatize at %s" % datetime.now().strftime('%H:%M:%S %d-%m-%Y'))
    log.debug("------------------------------------------------------------------------------")
    log.debug("------------------------------------------------------------------------------")
    connect_to_mongo(mongo_host=opts.mongo_host, mongo_port=opts.mongo_port, use_ssh=opts.mongo_use_ssh, ssh_key=opts.mongo_ssh_key, ssh_user=opts.mongo_ssh_user)
    
    # First sort ALL_FIX based on their versions
    ALL_FIX = sorted(ALL_FIX, key=natural_version)
    
    if list_only:
        cur_version = ''
        for f in ALL_FIX:
            if not hasattr(f, 'auto_launch'):
                raise Exception('Please specify auto launch parameter on fix [ %s ].' % f.__name__)
            v = f.version
            if v != cur_version:
                cur_version = v
                log.info("\n################ Version \033[35m%s\033[0m ##########" % v)
            log.info("-------------------")
            log.info(" \033[32m%-40s\033[0m:\n\tDoc: %s" % (f.__name__, f.doc))
        sys.exit(0)
    
    if [run_all, run_one, run_auto].count(True) > 1:
        sys.exit("Error: You cannot ask for a run one and run all and run auto option. Please choose")
    if not run_all and not run_one and not run_auto:
        parser.print_help()
        sys.exit(0)
    
    _need_shinken_stop = True
    
    if run_one:
        selected_fix = next((f for f in ALL_FIX if f.__name__ == run_one), None)
        if not selected_fix:
            log.error(u'\n\033[31m ERROR: The fix "%s" does not exists\033[0m\n' % run_one)
            sys.exit(1)
        _need_shinken_stop = getattr(selected_fix, u'need_shinken_stop', True)
    
    if _need_shinken_stop or run_all or run_auto:
        if is_shinken_running():
            log.warning("\033[33m###################### Warning #######################\033[0m")
            log.warning("\033[33m# Shinken is curently running. We will stop it now.  #\033[0m")
            log.warning("\033[33m# (We cannot sanatize data if shinken is running)    #\033[0m")
            log.warning("\033[33m######################################################\033[0m\n")
            subprocess.call(["/etc/init.d/shinken stop > /dev/null 2>&1"], shell=True)
        # Ensure mongod is started before run sanatize
        subprocess.call(["/etc/init.d/mongod start > /dev/null 2>&1"], shell=True)
    
    cur_version = ''
    errors_count = 0
    message_template = "   \033[%%dm%%-%ds\033[0m:  \033[%%dm %%s \033[0m" % (sanatize_decorator.NUMBER_INDENT_SPACES + 2)
    
    for f in ALL_FIX:
        fname = f.__name__
        if run_all or fname == run_one or (run_auto and f.auto_launch):
            # If the function is not launchable on this type of server, skip it
            
            v = f.version
            if v != cur_version and not opts.quiet:
                cur_version = v
                log.debug('')
                log.debug("################ Version \033[35m%s\033[0m ##########" % v)
            
            log.debug('-----')
            log.debug('Executing: %s' % fname)
            
            error_message = None
            
            if do_match_daemons(f):
                try:
                    has_error = False
                    if fname == 'fix_rename_key':
                        r = f(opts.old, opts.new)
                    else:
                        r = f()
                        if not isinstance(r, bool):
                            error_message = r
                            has_error = True
                            errors_count += 1
                except Exception, exp:
                    has_error = True
                    errors_count += 1
                    print '\n\033[31m ERROR: %s => %s\033[0m\n' % (fname, traceback.format_exc())
                    r = None
                
                if has_error:
                    state = "executed [Failure]"
                    color = 31  # red
                elif r:
                    state = "executed [OK]"
                    color = 32  # green
                    data_type = getattr(f, "data_type", 'configuration')
                    # We did execute a sanatize, save it in the context
                    do_import_data_history_sanatize(fname, data_type=data_type)
                else:
                    state = "skip (unecessary)"
                    color = 36  # ligh blue
            else:
                state = u'skip (this sanatize is only needed for this daemons : %s)' % u', '.join(f.context_daemons)
                color = 36  # ligh blue
                
                r = True
            
            if r or not opts.quiet:
                sanitize_log.logger.info("   \033[%dm%-40s\033[0m:  \033[%dm %s \033[0m" % (35, fname, color, state))
                if fname in SANATIZE_SUMMARY:
                    for action in SANATIZE_SUMMARY[fname]:
                        sanitize_log.logger.info(action)
                if error_message:
                    print '\n\033[31m ERROR: %s => %s\033[0m\n' % (fname, error_message)
                    for message_line in error_message.split('\n'):
                        log.error("        \033[%dm %s" % (color, message_line))
    
    if errors_count:
        if not opts.no_hint:
            log_file = opts.log_file if opts.log_file else DEFAULT_LOG_FILENAME
            log.error("\n\n\033[31mSome errors occurred while running the fixes."
                      " Please check the log file for more information (\033[36m%s\033[31m)."
                      "Send this file to your Shinken support if needed.\033[0m" % log_file)
        sys.exit(1)
