#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2020:
# This file is part of Shinken Enterprise, all rights reserved.

import itertools
import re
from collections import OrderedDict

from shinken.basesubprocess import EventHandler
from shinken.log import PartLogger
from shinken.misc.type_hint import NoReturn, Tuple, List, Dict, Optional, Any
from shinkensolutions.api.synchronizer import Messages, MESSAGE
from shinkensolutions.api.synchronizer import component_manager, SourceTranslatePart, ITEM_TYPE
from shinkensolutions.api.synchronizer.http_lib_external.v01_00.html.distributor.option_distributor import CounterSelect
from shinkensolutions.api.synchronizer.http_lib_external.v01_00.html.distributor.option_distributor import DefineBy
from shinkensolutions.api.synchronizer.http_lib_external.v01_00.html.object.counter import Counter
from shinkensolutions.api.synchronizer.http_lib_external.v01_00.html.object.html.tag_template import TagTemplate
from shinkensolutions.api.synchronizer.source.api_item_properties import ApiItemProperties, CHECK_PROPERTY
from shinkensolutions.api.synchronizer.source.file_loader import FileLoader
from shinkensolutions.api.synchronizer.source.item.source_item import SourceItem
from shinkensolutions.api.synchronizer.source.source_configuration_value import SERVICE_MODE, ServiceMode
from shinkensolutions.api.synchronizer.source.source_exception import SourceException
from shinkensolutions.api.synchronizer.source.validation_state import ValidationState
from shinkensolutions.toolbox.box_tools_string import ToolsBoxString

TemplateName = str  # Type for template name. For type hint.
RULE_KNOWN_KEYS = ['name', 'template', r'condition[0-9]+', 'disable']
ILLEGAL_CHARS = re.compile("""[`~!$%^&*"|'<>?,()=/+]""")
SIMPLE_CONDITION_REGEX = r"^([^\n\t\r=]+)\s*=\s*(\^?[^\n\t\r]+\$?)$"


class ReloadSourceRulesManagerEvent(EventHandler):
    
    def __init__(self, source_rules_manager):
        # type: (SourceRulesManager) -> None
        super(ReloadSourceRulesManagerEvent, self).__init__('reload-source-rules-manager')
        self.source_rules_manager = source_rules_manager
    
    
    def callback(self):
        # type: () -> None
        self.source_rules_manager.reload_rule()


class SimpleCondition(object):
    def __init__(self, condition):
        self.condition = condition
        self.key = ''
        self.expression = ''
        self.error = ''
        self.reg_expression = ''
    
    
    @staticmethod
    def expression_to_regex(expression):
        reg_expression = expression
        startswith = ""
        endswith = ""
        if reg_expression.startswith('^'):
            reg_expression = reg_expression[1:]
            startswith = '^'
        if reg_expression.endswith('$'):
            reg_expression = reg_expression[:-1]
            endswith = '$'
        reg_expression = re.escape(reg_expression)
        reg_expression = '%s(%s)%s' % (startswith, reg_expression, endswith)
        return reg_expression
    
    
    def validate(self, api_item_properties, translate):
        # type: (ApiItemProperties, Any) -> str
        match = re.search(SIMPLE_CONDITION_REGEX, self.condition)
        if not match:
            self.error = translate('host_template_binding_rules.validate_condition_error.default')
            split = self.condition.split('=', 1)
            if len(split) == 2:
                if not split[0] and not split[1]:
                    self.error = translate('host_template_binding_rules.validate_condition_error.missing_everything')
                elif not split[0]:
                    self.error = translate('host_template_binding_rules.validate_condition_error.missing_key')
                elif not split[1]:
                    self.error = translate('host_template_binding_rules.validate_condition_error.missing_expression')
            elif len(split) == 1:
                self.error = translate('host_template_binding_rules.validate_condition_error.missing_equal')
        else:
            expression = match.group(2)
            key = match.group(1)
            check_property = api_item_properties.check_property(ITEM_TYPE.HOSTS, key)
            if expression == '^' or expression == '$' or expression == '^$':
                self.error = translate('host_template_binding_rules.validate_condition_error.only_regex_in_expression')
            elif check_property == CHECK_PROPERTY.IS_UNKNOWN:
                self.error = translate('host_template_binding_rules.validate_condition_error.unknown_source_key', api_item_properties.user_file.path)
            elif check_property == CHECK_PROPERTY.IS_DISABLE:
                self.error = translate('host_template_binding_rules.validate_condition_error.disable_source_key', api_item_properties.user_file.path)
        return self.error
    
    
    def build(self):
        match = re.search(SIMPLE_CONDITION_REGEX, self.condition)
        self.key = match.group(1)
        self.expression = match.group(2)
        self.reg_expression = self.expression_to_regex(self.expression)
    
    
    def match(self, source_item):
        # type: (SourceItem) -> bool
        key = self.key.replace('.', '__')
        result = re.search(self.reg_expression, source_item.get(key, ''), re.IGNORECASE)
        return result and result.re.groups != 0


class SourceCondition(object):
    def __init__(self, condition, api_item_properties, translate):
        self.expression = condition['expression']
        self.number = condition['number']
        self.conditions = []
        self.error_message = ''
        
        error_messages = []
        for cond in self.expression.split('AND'):
            simple_condition = SimpleCondition(cond.strip())
            self.conditions.append(simple_condition)
            error_message = simple_condition.validate(api_item_properties, translate)
            
            if error_message:
                error_messages.append(error_message)
            else:
                simple_condition.build()
        self.error_message = ' ,'.join(error_messages)
    
    
    def match(self, source_item):
        # type: (SourceItem) -> bool
        for condition in self.conditions:
            if not condition.match(source_item):
                return False
        return True
    
    
    # *****************************   HTML  ************************************************************#
    def get_html(self):
        # type: ()-> str
        _to_return = ['''<div class="shinken-label-condition">''']
        if self.error_message:
            _to_return.append('''<span class="shinken-format-warning"><span class="shinken-label-index shinken-between-parathensis">%s</span></span>''' % self.number)
        else:
            _to_return.append('''<span class="shinken-label-index shinken-between-parathensis">%s</span>''' % self.number)
        _to_return.append('''<span class="shinken-label">%s</span>''' % ToolsBoxString.escape_XSS(self.expression))
        _to_return.append('''</div>''')
        
        return ''.join(_to_return)


class SourceRules(object):
    def __init__(self, name, conditions, from_user, templates, validation_message, is_valid, is_enable):
        # type: (str, List[SourceCondition], bool, List[str], Messages, bool, bool)-> None
        self.name = name
        self.from_user = from_user
        self.templates = templates
        self.conditions = conditions
        self.is_enable = is_enable
        self.conditions_to_html = self.parse_condition_to_html()
        self.validation_message = validation_message
        self.is_valid = is_valid
        self.index = None
        self.errors_counter_html = ""
        self.error_counter_number = 0
        self.warnings_counter_html = ""
        self.warning_counter_number = 0
        self.counter_filter = CounterSelect.EMPTY
    
    
    def match(self, source_item):
        # type: (SourceItem) -> bool
        for condition in self.conditions:
            if condition.match(source_item):
                return True
        return False
    
    
    # *****************************   HTML  ************************************************************#
    def parse_condition_to_html(self):
        # type: () -> str
        if not self.conditions and self.is_enable:
            return '''<div class="shinken-format-error-lite"><span class="shinkon-warning-lite"></span> %s</div>''' % component_manager.get_translate_component().translator().translate('common.field_required')
        
        return ''.join([condition.get_html() for condition in self.conditions])
    
    
    def parse_name_to_html(self):
        # type: () -> str
        if not self.name:
            return '''<div class="shinken-format-error-lite"><span class="shinkon-warning-lite"></span> %s</div>''' % component_manager.get_translate_component().translator().translate('common.field_required')
        
        return '''<div class="shinken-label">%s</div>''' % ToolsBoxString.escape_XSS(self.name)
    
    
    def parse_templates_to_html(self):
        # type: ()-> str
        if not self.templates and self.is_enable:
            return '''<div class="shinken-format-error-lite"><span class="shinkon-warning-lite"></span> %s</div>''' % component_manager.get_translate_component().translator().translate('common.field_required')
        _to_return = []
        if not self.templates:
            return ''
        for template in self.templates:
            template_tag = TagTemplate(template)
            _to_return.append(template_tag.get_html())
        return ''.join(_to_return)
    
    
    def compute_validation_message_to_html(self):
        # type: ()-> NoReturn
        if self.is_valid:
            return
        
        self.error_counter_number = len(self.validation_message.messages_error)
        self.warning_counter_number = len(self.validation_message.messages_warning)
        
        if self.error_counter_number:
            self.counter_filter = CounterSelect.ERROR
            _error_counter = Counter('shinken-error-counter', number=str(self.error_counter_number))
            _error_counter.add_tooltip_messages(self.validation_message.extract_error_messages(), 'shinken-error-border')
            self.errors_counter_html = _error_counter.get_html()
        if self.warning_counter_number:
            self.counter_filter = '%s,%s' % (CounterSelect.ERROR, CounterSelect.WARNING) if self.error_counter_number else CounterSelect.WARNING
            _warning_counter = Counter('shinken-warning-counter', number=str(self.warning_counter_number))
            _warning_counter.add_tooltip_messages(self.validation_message.extract_warning_messages(), message_type='shinken-warning-border', placement="bottom-right")
            self.warnings_counter_html = _warning_counter.get_html()


class SourceRulesManager(object):
    def __init__(self, logger, file_loader, service_mode, translator, api_item_properties):
        # type: (PartLogger, FileLoader, ServiceMode, SourceTranslatePart, ApiItemProperties) -> None
        self.logger = logger
        self.file_loader = file_loader
        self.service_mode = service_mode
        self.rules = []  # type: List[SourceRules]
        self.user_rules_files = []  # type: List[Dict]
        self.reloading = True
        self._translator = translator
        self.validation_state = ValidationState(self.logger, self._translator)  # type: ValidationState
        self.api_item_properties = api_item_properties
        self.load()
    
    
    def reload_rule(self):
        self.api_item_properties.reload()
        self.load()
    
    
    def load(self):
        self.reloading = True
        validation_state = ValidationState(self.logger, self._translator)
        default_file = self.file_loader.host_template_binding_rule.default_file
        try:
            json_binding_rules = default_file.load()
        except Exception as e:
            validation_state.add_error(self._translator.translate('host_template_binding_rules.source_rules_manager.invalid_default_json', default_file.path, e))
            self.validation_state = validation_state
            return
        rules, validation_state = self._parse_json_rules(json_binding_rules, for_user=DefineBy.BY_DEFAULT)
        
        user_rules_files = []
        if self.service_mode == SERVICE_MODE.ON:
            user_file = self.file_loader.host_template_binding_rule.user_file
            if user_file.exist():
                status = 'ok'
                try:
                    json_binding_rules = user_file.load()
                    if json_binding_rules:
                        name_default_rules = [r.name for r in rules]
                        user_rules, _validation_state = self._parse_json_rules(json_binding_rules, for_user=DefineBy.BY_USER, default_rules=name_default_rules)
                        validation_state.update_from_validation_state(_validation_state)
                        
                        name_user_rules = [r.name for r in user_rules]
                        self.merge_rules(name_user_rules, rules, user_rules)
                        rules = user_rules
                except Exception as e:
                    validation_state.add_error(self._translator.translate('host_template_binding_rules.source_rules_manager.invalid_user_json', user_file.path, e))
                    status = 'error'
            
            else:
                status = 'not_existing'
                validation_state.add_warning(self._translator.translate('warning.missing_user_file', user_file.path))
            
            user_rules_files.append({'path': user_file.path, 'status': status})
        
        if validation_state.extra_counter_error_number:
            validation_state.add_error(self._translator.translate('host_template_binding_rules.source_rules_manager.config_error', validation_state.extra_counter_error_number))
        if validation_state.extra_counter_warning_number:
            validation_state.add_warning(self._translator.translate('host_template_binding_rules.source_rules_manager.config_warning', validation_state.extra_counter_warning_number))
        
        self.validation_state = validation_state
        self.reloading = False
        self.user_rules_files = user_rules_files
        rules.sort(key=lambda x: (x.from_user == DefineBy.BY_USER_AND_DISABLE, x.from_user == DefineBy.BY_DEFAULT, x.from_user == DefineBy.BY_USER))
        for index, rule in enumerate(rules):
            rule.index = index
        
        rules.sort(key=lambda x: (x.is_valid, x.index))
        
        self.rules = rules
        return rules
    
    
    @staticmethod
    def merge_rules(name_user_rules, rules, user_rules):
        for user_rule in user_rules:
            for default_rule in rules:
                user_rule.conditions = default_rule.conditions if default_rule.name == user_rule.name and not user_rule.conditions else user_rule.conditions
                user_rule.templates = default_rule.templates if default_rule.name == user_rule.name and user_rule.templates is None else user_rule.templates
        user_rules.extend((r for r in rules if r.name not in name_user_rules))
    
    
    @staticmethod
    def strip_rules_keys(json_binding_rules):
        json_binding_rules_stripped = []
        for raw_rule in json_binding_rules:
            json_binding_rules_stripped.append(dict((k.strip(), v) for (k, v) in raw_rule.iteritems()))
        return json_binding_rules_stripped
    
    
    @staticmethod
    def is_known_key(key):
        # type: (str) -> bool
        return next((True for rule_known_key in RULE_KNOWN_KEYS if re.match(rule_known_key, key)), False)
    
    
    def _parse_json_rules(self, json_binding_rules, for_user, default_rules=None):
        # type: (List, bool, Optional[List])-> Tuple[List, ValidationState]
        if default_rules is None:
            default_rules = []
        validation_state = ValidationState(self.logger, self._translator)
        rules = []
        if not json_binding_rules:
            return rules, validation_state
        json_binding_rules = self.strip_rules_keys(json_binding_rules)
        for raw_rule in json_binding_rules:
            raw_rule = OrderedDict(sorted(raw_rule.iteritems(), key=lambda x: x[0]))
            is_valid = True
            validation_message = Messages()
            
            rule_name = raw_rule.get('name', None)
            if not rule_name:
                is_valid = False
                validation_message.add_message_txt(MESSAGE.STATUS_ERROR, self._translator.translate('host_template_binding_rules.source_rules_manager.missing_name'))
            if len([raw_r.get('name', None) for raw_r in json_binding_rules if raw_r.get('name', None) == rule_name]) > 1:
                is_valid = False
                validation_message.add_message_txt(MESSAGE.STATUS_WARNING, self._translator.translate('host_template_binding_rules.source_rules_manager.same_name'))
            
            disable = raw_rule.get('disable', 'false').lower()
            is_enable = not disable == 'true'
            rule_template = raw_rule.get('template', None)
            if not is_enable:
                rule_template = rule_template.split(',') if rule_template is not None else None
                conditions = []
                for condition_name, condition_expression in raw_rule.iteritems():
                    if not condition_name.startswith('condition'):
                        continue
                    condition_number = int(condition_name.replace('condition', ''))
                    source_condition = SourceCondition({'expression': condition_expression, 'number': condition_number}, self.api_item_properties, self._translator.translate)
                    conditions.append(source_condition)
                conditions.sort(key=lambda x: x.number)
                rule = SourceRules(
                    name=rule_name,
                    conditions=conditions,
                    templates=rule_template,
                    from_user=DefineBy.BY_USER_AND_DISABLE,
                    validation_message=validation_message,
                    is_valid=is_valid,
                    is_enable=is_enable
                )
                rules.append(rule)
                continue
            
            rule_template = None if rule_template is None else [i.strip() for i in rule_template.split(',') if i.strip()]
            if rule_template:
                rule_template_with_illegal_char = [rt for rt in rule_template if SourceRulesManager._contains_illegal_chars(rt)]
                if rule_template_with_illegal_char:
                    is_valid = False
                    validation_message.add_message_txt(MESSAGE.STATUS_ERROR, self._translator.translate('host_template_binding_rules.source_rules_manager.invalid_char_template', ToolsBoxString.escape_XSS(','.join(rule_template_with_illegal_char))))
            elif rule_template is not None or rule_template is None and rule_name not in default_rules:
                is_valid = False
                validation_message.add_message_txt(MESSAGE.STATUS_ERROR, self._translator.translate('host_template_binding_rules.source_rules_manager.missing_template'))
            
            bad_keys = [k for k in raw_rule.iterkeys() if not self.is_known_key(k)]
            if bad_keys:
                is_valid = False
                validation_message.add_message_txt(MESSAGE.STATUS_ERROR, self._translator.translate('host_template_binding_rules.source_rules_manager.unknown_key', ToolsBoxString.escape_XSS(','.join(bad_keys))))
            conditions = []
            
            for condition_name, condition_expression in raw_rule.iteritems():
                if not condition_name.startswith('condition'):
                    continue
                
                try:
                    condition_number = int(condition_name.replace('condition', ''))
                except:
                    is_valid = False
                    validation_message.add_message_txt(MESSAGE.STATUS_ERROR, self._translator.translate('host_template_binding_rules.source_rules_manager.no_number_in_condition', rule_name))
                    continue
                
                source_condition = SourceCondition({'expression': condition_expression, 'number': condition_number}, self.api_item_properties, self._translator.translate)
                
                if source_condition.error_message:
                    is_valid = False
                    validation_message.add_message_txt(MESSAGE.STATUS_WARNING, self._translator.translate('host_template_binding_rules.source_rules_manager.invalid_condition',
                                                                                                          '''<span class='shinken-highlight-data-user shinken-between-parathensis'>%s</span>''' % source_condition.number,
                                                                                                          '''<span class='shinken-highlight-data-user'>%s</span>''' % condition_expression,
                                                                                                          '''<br><span class='shinken-note shinken-between-parathensis'>%s</span>''' % source_condition.error_message))
                
                conditions.append(source_condition)
            conditions.sort(key=lambda x: x.number)
            
            if not conditions and rule_name not in default_rules:
                is_valid = False
                validation_message.add_message_txt(MESSAGE.STATUS_ERROR, self._translator.translate('host_template_binding_rules.source_rules_manager.no_condition'))
            rule = SourceRules(
                name=rule_name,
                conditions=conditions,
                templates=rule_template,
                from_user=for_user,
                validation_message=validation_message,
                is_valid=is_valid,
                is_enable=is_enable
            )
            validation_state.extra_counter_error_number += len(validation_message.messages_error)
            validation_state.extra_counter_warning_number += len(validation_message.messages_warning)
            rules.append(rule)
        
        return rules, validation_state
    
    
    @staticmethod
    def _contains_illegal_chars(field):
        return re.search(ILLEGAL_CHARS, field)
    
    
    def match(self, source_item):
        # type: (SourceItem) -> List[TemplateName]
        matched_rules = []
        if self.validation_state.has_error():
            raise SourceException(self._translator.translate('error.rules_application_template_error', ','.join(self.validation_state.get_errors())))
        for rule in self.rules:
            if rule.is_enable and rule.match(source_item):
                matched_rules.append(rule)
        # One rule can have multiple template so we put all templates
        all_templates = list(itertools.chain.from_iterable([matched_rule.templates for matched_rule in matched_rules]))
        # remove duplicate template name and keep order
        seen = set()
        seen_add = seen.add
        return [x for x in all_templates if not (x in seen or seen_add(x))]
