#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.


import optparse
# This small lib is used by shinken checks to got helping functions
import os
import sys

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.toolbox.url_helper import ShinkenBaseUrlException, BaseUrl
from shinkensolutions.lib_checks.libs import paramiko
from shinkensolutions.lib_checks.libs.paramiko.util import log_to_file
from .common import EXIT_STATUS

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional
    from shinkensolutions.lib_checks.libs.paramiko.client import SSHClient

VERSION = 0.2


def printf(fmt, *args):
    sys.stdout.write(fmt % args)


def get_client(opts):
    hostname = opts.hostname
    port = opts.port
    ssh_key_file = opts.ssh_key_file
    user = opts.user
    passphrase = opts.passphrase
    
    # Ok now connect, and try to get values for memory
    client = connect(hostname, port, ssh_key_file, passphrase, user)
    return client


def connect(hostname, port, ssh_key_file, passphrase, user, quit_on_fail=True, timeout=None):
    # Maybe paramiko is missing, but now we really need ssh...
    if paramiko is None:
        _message = "<span style=\"color:#A9A9A9;font-weight: bold;\">[ERROR]</span> This plugin needs the python-paramiko module. Please install it"
        if quit_on_fail:
            print _message
            sys.exit(3)
        else:
            return EXIT_STATUS.CRITICAL, _message
    
    if not os.path.exists(os.path.expanduser(ssh_key_file)):
        _message = "<span style=\"color:#A9A9A9;font-weight: bold;\">[ERROR]</span> Missing ssh key file. please specify it with -i parameter"
        if quit_on_fail:
            print _message
            sys.exit(3)
        else:
            return EXIT_STATUS.CRITICAL, _message
    
    ssh_key_file = os.path.expanduser(ssh_key_file)
    client = paramiko.SSHClient()
    client.load_system_host_keys()
    client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
    
    # Trying to gt overload parameter from the ~/.ssh/config file
    ssh_config = paramiko.SSHConfig()
    user_config_file = os.path.expanduser('~/.ssh/config')
    if os.path.exists(user_config_file):
        with open(user_config_file) as f:
            try:
                ssh_config.parse(f)
            except Exception as exp:
                print('ERROR: cannot parse the ~/.ssh/config file: %s' % exp)
                sys.exit(2)
    
    parameter_sources = {}  # for display if there is an error with the connection
    proxy_command_used_str = ''
    
    cfg = {'hostname': hostname, 'port': port, 'username': user, 'key_filename': ssh_key_file, 'password': passphrase, 'look_for_keys': False}
    
    if timeout:
        cfg['timeout'] = timeout
    
    user_config = ssh_config.lookup(cfg['hostname'])
    for k in ('hostname', 'port', 'username', 'key_filename', 'password'):
        if k in user_config:
            cfg[k] = user_config[k]
            parameter_sources[k] = '<br/>* parameter %s read from ~/.ssh/config' % k
    
    if 'proxycommand' in user_config:
        cfg['sock'] = paramiko.ProxyCommand(user_config['proxycommand'])
        proxy_command_used_str = '<br/>* Using a proxycommand \'%s\' read from ~/.ssh/config' % user_config['proxycommand']
    
    log_to_file(u'/var/log/shinken/paramiko.log')  # deactivate the logging
    try:
        base_url = BaseUrl(hostname, cfg[u'port'] or 22)
        cfg[u'hostname'] = base_url.get_host()
        cfg[u'port'] = base_url.get_port()
        client.connect(**cfg)
    except ShinkenBaseUrlException:
        raise
    except Exception:
        _message = "<span style=\"color:#A9A9A9;font-weight: bold;\">[ERROR]</span> Connection failed to %s:%s 'Authentication failed with user %s' %s%s%s%s" % (hostname, cfg['port'], cfg['username'],
                                                                                                                                                                 parameter_sources.get('port', ''),
                                                                                                                                                                 parameter_sources.get('username', ''),
                                                                                                                                                                 parameter_sources.get('key_filename', ''),
                                                                                                                                                                 proxy_command_used_str,
                                                                                                                                                                 )
        if quit_on_fail:
            print _message
            sys.exit(3)
        else:
            return EXIT_STATUS.CRITICAL, _message
    
    if quit_on_fail:
        return client
    else:
        return EXIT_STATUS.OK, client


def close(client):
    try:
        client.close()
    except Exception:
        pass


# Try to parse and get int values from warning and critical parameters
def get_warn_crit(s_warn, s_crit):
    if s_warn.endswith(u'%'):
        s_warn = s_warn[:-1]
    if s_crit.endswith(u'%'):
        s_crit = s_crit[:-1]
    try:
        warn, crit = int(s_warn), int(s_crit)
    except ValueError:
        print u'<span style=\'color:#A9A9A9;font-weight: bold;\'>[ERROR]</span> Bad values for warning and/or critical : %s %s' % (s_warn, s_crit)
        sys.exit(3)
    
    if warn > crit:
        print u'<span style=\'color:#A9A9A9;font-weight: bold;\'>[ERROR]</span> Warning value %s can\'t be greater than critical one %s ' % (warn, crit)
        sys.exit(3)
    
    return warn, crit


def get_parser():
    parser = optparse.OptionParser(u'%prog [options]', version=u'%prog ' + str(VERSION))
    parser.add_option(u'-H', u'--hostname', default=u'', dest=u'hostname', help=u'Hostname to connect to')
    parser.add_option(u'-p', u'--port', dest=u'port', type=u'int', default=22, help=u'SSH port to connect to. Default : 22')
    parser.add_option(u'-i', u'--ssh-key', default=os.path.expanduser(u'~/.ssh/id_rsa'), dest=u'ssh_key_file', help=u'SSH key file to use. By default will take ~/.ssh/id_rsa.')
    parser.add_option(u'-u', u'--user', default=u'shinken', dest=u'user', help=u'remote user to use. By default shinken.')
    parser.add_option(u'-P', u'--passphrase', default=u'', dest=u'passphrase', help=u'SSH key passphrase. By default will use void')
    parser.add_option(u'-t', dest=u'modname', help=u'Check to load')
    parser.add_option(u'-l', action=u'store_true', dest=u'listmod', help=u'List all checks available')
    
    return parser


class GenCheck(object):
    
    def __init__(self):
        
        if TYPE_CHECKING:
            self.client = None  # type: Optional[SSHClient]
            self.opts = None  # type: Optional[Values]
            self.args = u''
        
        self.output = self.perfdata = ''
        self.exit_code = 3
        self.parser = get_parser()
    
    
    # By default, do nothing
    def fill_parser(self):
        return
    
    
    def parse_args(self):
        # Ok first job : parse args
        self.opts, self.args = self.parser.parse_args()
    
    
    def check_args(self):
        pass
    
    
    def get_client(self):
        # Ok now got an object that link to our destination
        self.client = get_client(self.opts)
    
    
    @staticmethod
    def do_check():
        print u'Unknown: no check selected'
        sys.exit(3)
    
    
    def set(self, output, exit_code, perfdata=''):
        self.output = output
        self.perfdata = perfdata
        self.exit_code = exit_code
    
    
    def exit(self):
        # first close the client
        self.client.close()
        
        if self.perfdata:
            print u'%s | %s' % (self.output, self.perfdata)
        else:
            print self.output
        sys.exit(self.exit_code)
