#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.
import time
from datetime import datetime, timedelta

import sla_common
from shinken.misc.type_hint import NoReturn, Optional, Dict, Tuple, List
from shinken.objects.module import Module as ShinkenModuleDefinition
from shinkensolutions.date_helper import date_now, get_previous_date, Date, compare_date, DATE_COMPARE, get_now
from sla_abstract_component import AbstractComponent
from sla_common import shared_data, RAW_SLA_KEY, FutureState
from sla_component_manager import ComponentManager
from sla_database_connection import SLADatabaseConnection
from sla_info import SLAInfo


class SLADatabase(AbstractComponent):
    
    def __init__(self, conf, component_manager, sla_database_connection, sla_info):
        # type: (ShinkenModuleDefinition, ComponentManager,  SLADatabaseConnection, Optional[SLAInfo]) -> NoReturn
        super(SLADatabase, self).__init__(conf, component_manager)
        self.sla_database_connection = sla_database_connection
        self.sla_info = sla_info
    
    
    def init(self):
        self.check_daily_format()
    
    
    def check_daily_format(self):
        if shared_data.get_migration_daily_done():
            return
        
        display_info = 'check raw sla collection format collections:[%s] all is migrate:[%s]'
        col_info = []
        _now = datetime.fromtimestamp(get_now())
        all_daily_is_migrate = True
        for day_count in range(0, 7):
            tm = (_now + timedelta(days=-day_count)).timetuple()
            date = Date(tm.tm_yday, tm.tm_year)
            
            if not self.collection_exist(date):
                continue
            
            daily_is_migrate = self.all_raw_sla_is_migrate(date)
            col_info.append('%s:%s' % (date, daily_is_migrate))
            all_daily_is_migrate = all_daily_is_migrate and daily_is_migrate
        if all_daily_is_migrate:
            shared_data.set_migration_daily_done(True)
        
        display_info = display_info % (','.join(col_info), shared_data.get_migration_daily_done())
        self.logger.info(display_info)
    
    
    def tick(self):
        pass
    
    
    def collection_exist(self, collection_name):
        return self.sla_database_connection.collection_exist(collection_name)
    
    
    def rename_collection(self, old_name, new_name):
        return self.sla_database_connection.rename_collection(old_name, new_name)
    
    
    def list_name_collections(self):
        return self.sla_database_connection.list_name_collections()
    
    
    def drop_collections(self, name_collection):
        return self.sla_database_connection.drop_collection(name_collection)
    
    
    def remove_archive(self, item_uuid, date):
        # type: (str, Date) -> NoReturn
        
        _where, _where_with_name = self._build_find_archive_where(item_uuid, date)
        self.sla_database_connection.col_archive.remove(_where)
        if _where_with_name:
            self.sla_database_connection.col_archive.remove(_where_with_name)
    
    
    def find_archive(self, item_uuid, date, where=None, lookup=None):
        # type: (str, Date, Dict, Dict) -> Optional[Dict]
        
        _where, _where_with_name = self._build_find_archive_where(item_uuid, date, where)
        archive = self.sla_database_connection.col_archive.find_one(_where, lookup)
        if not archive and _where_with_name:
            archive = self.sla_database_connection.col_archive.find_one(_where_with_name, lookup)
        return archive
    
    
    def find_archive_by_id(self, archive_id):
        # type: (str) -> Optional[Dict]
        return self.sla_database_connection.col_archive.find_one({'_id': archive_id})
    
    
    def find_archives(self, item_uuid, date_ranges, where=None, lookup=None):
        # type: (str, Tuple[Date,Date], Dict, Dict) -> List[Dict]
        _where, _where_with_name = self._build_find_archives_where(item_uuid, date_ranges, where)
        
        archives = list(self.sla_database_connection.col_archive.find(_where, lookup))
        if _where_with_name:
            archives.extend(self.sla_database_connection.col_archive.find(_where_with_name, lookup))
        
        return archives
    
    
    def count_archives_before_date(self, date):
        return self.sla_database_connection.col_archive.find(self._build_where_archives_before_date(date), only_count=True)
    
    
    def find_archives_before_date(self, date, limit):
        return self.sla_database_connection.col_archive.find(self._build_where_archives_before_date(date), {'_id': 1}, limit=limit)
    
    
    def remove_archives(self, where):
        return self.sla_database_connection.col_archive.remove(where)
    
    
    def count_archives_to_migrate(self, configuration_uuid):
        return self.sla_database_connection.col_archive.find({'version': {'$ne': sla_common.CURRENT_ARCHIVE_VERSION}, 'configuration_uuid': {'$ne': configuration_uuid}}, only_count=True)
    
    
    def find_archives_to_migrate(self, configuration_uuid, limit):
        return self.sla_database_connection.col_archive.find({'version': {'$ne': sla_common.CURRENT_ARCHIVE_VERSION}, 'configuration_uuid': {'$ne': configuration_uuid}}, limit=limit)
    
    
    def list_archives_uuid(self, date):
        # type: (Date) -> List[str]
        archives = self.sla_database_connection.col_archive.find({'year': date.year, 'yday': date.yday}, projection={'uuid': 1})
        return [i['uuid'] for i in archives]
    
    
    @staticmethod
    def _build_where_archives_before_date(date):
        return {'$or': [{'$and': [{'year': date.year}, {'yday': {'$lt': date.yday}}]}, {'year': {'$lt': date.year}}]}
    
    
    def _build_where_common(self, item_uuid, where=None):
        if where:
            _where = where.copy()
        else:
            _where = {}
        
        if item_uuid:
            _where['uuid'] = item_uuid
        
        _where_with_name = None
        if not shared_data.get_migration_archive_done():
            _where_with_name = _where.copy()
            del _where_with_name['uuid']
            host_name, service_description = self.sla_info.get_name(item_uuid)
            if host_name:
                _where_with_name['hname'] = host_name
                if service_description:
                    _where_with_name['sdesc'] = service_description
                    _where_with_name['type'] = 'service'
                else:
                    _where_with_name['type'] = 'host'
        
        return _where, _where_with_name
    
    
    def _build_find_archive_where(self, item_uuid, date, where=None):
        _where, _where_with_name = self._build_where_common(item_uuid, where=where)
        
        if date:
            _where['year'] = date.year
            _where['yday'] = date.yday
            if _where_with_name:
                _where_with_name['year'] = date.year
                _where_with_name['yday'] = date.yday
        
        return _where, _where_with_name
    
    
    def _build_find_archives_where(self, item_uuid, date_ranges, where=None):
        _where, _where_with_name = self._build_where_common(item_uuid, where=where)
        
        if date_ranges:
            start_date = date_ranges[0]
            end_date = date_ranges[1]
            
            if start_date and not end_date:
                _where['$or'] = [{'$and': [{'year': start_date.year}, {'yday': {'$gte': start_date.yday}}]}, {'year': {'$gt': start_date.year}}]
                if _where_with_name:
                    _where_with_name['$or'] = [{'$and': [{'year': start_date.year}, {'yday': {'$gte': start_date.yday}}]}, {'year': {'$gt': start_date.year}}]
            
            if not start_date and end_date:
                _where['$or'] = [{'$and': [{'year': end_date.year}, {'yday': {'$lte': end_date.yday}}]}, {'year': {'$lt': end_date.year}}]
                if _where_with_name:
                    _where_with_name['$or'] = [{'$and': [{'year': end_date.year}, {'yday': {'$lte': end_date.yday}}]}, {'year': {'$lt': end_date.year}}]
            
            if start_date and end_date:
                start_date_bound = {'$or': [{'$and': [{'year': start_date.year}, {'yday': {'$gte': start_date.yday}}]}, {'year': {'$gt': start_date.year}}]}
                end_date_bound = {'$or': [{'$and': [{'year': end_date.year}, {'yday': {'$lte': end_date.yday}}]}, {'year': {'$lt': end_date.year}}]}
                _where['$and'] = [start_date_bound, end_date_bound]
                if _where_with_name:
                    _where_with_name['$and'] = [start_date_bound, end_date_bound]
        
        return _where, _where_with_name
    
    
    def save_archive(self, archive):
        self.sla_database_connection.col_archive.save(archive)
    
    
    def save_archives(self, archives, update):
        if update:
            self.sla_database_connection.col_archive.replace_many(archives)
        else:
            self.sla_database_connection.col_archive.insert_many(archives)
        return True
    
    
    def save_raw_sla(self, date, raw_sla):
        current_collection = self.sla_database_connection.get_raw_sla_collection(date, True)
        current_collection.save(raw_sla)
    
    
    def find_raw_sla(self, date, item_uuid):
        current_collection = self.sla_database_connection.get_raw_sla_collection(date, False)
        raw_sla = current_collection.find({RAW_SLA_KEY.UUID: item_uuid})
        
        if not shared_data.get_migration_daily_done():
            host_name, service_description = self.sla_info.get_name(item_uuid)
            if host_name:
                where = {
                    RAW_SLA_KEY.HNAME: host_name,
                    RAW_SLA_KEY.TYPE : 'H',
                }
                if service_description:
                    where[RAW_SLA_KEY.TYPE] = 'S'
                    where[RAW_SLA_KEY.SDESC] = service_description
                raw_sla.extend(current_collection.find(where))
        return raw_sla
    
    
    def find_raw_sla_status(self, date):
        current_collection = self.sla_database_connection.get_raw_sla_collection(date, False)
        return current_collection.find_one({'_id': 'SLA_STATUS'})
    
    
    def build_raw_sla_bulk(self, date, on_execute_error):
        return self.sla_database_connection.bulk_sla_factory(date, on_execute_error)
    
    
    def all_raw_sla_is_migrate(self, date):
        current_collection = self.sla_database_connection.get_raw_sla_collection(date, False)
        return current_collection.find({RAW_SLA_KEY.TYPE: {'$exists': True}}, {'_id': 1}, limit=1, only_count=True) == 0
    
    
    def save_raw_sla_status(self, date, sla_status):
        current_collection = self.sla_database_connection.get_raw_sla_collection(date, True)
        current_collection.save(sla_status)
    
    
    def save_future_states(self, future_states):
        # type: (Dict[str, FutureState]) -> NoReturn
        t0 = time.time()
        # self.logger.debug('BMDEV [finding element] start ')
        in_base_future_states = self.sla_database_connection.col_sla_future_states.find(projection={'_id': 1, 'hash': '1'})
        # self.logger.debug('BMDEV [finding element] end in [%s] --- nb elements found in base [%s]' % (self.logger.format_chrono(t0), len(in_base_future_states)))
        
        # t1 = time.time()
        # self.logger.debug('BMDEV [building set] start ')
        in_base_item_uuid_future_states = set([i['_id'] for i in in_base_future_states])
        to_save_item_uuid_future_states = set(future_states.iterkeys())
        # self.logger.debug('BMDEV [building set] end in [%s] ---' % self.logger.format_chrono(t1))
        
        # t1 = time.time()
        # self.logger.debug('BMDEV [adding new] start ')
        to_save = []
        to_add_item_uuid = to_save_item_uuid_future_states - in_base_item_uuid_future_states
        if to_add_item_uuid:
            # t2 = time.time()
            to_save = [{'_id': item_id, 'future_state': future_states[item_id], 'hash': hash(future_states[item_id])} for item_id in to_add_item_uuid]
            # self.logger.debug('BMDEV [adding new] prepare data in [%s] --- nb elements to save [%s]' % (self.logger.format_chrono(t2), len(to_save)))
            
            t2 = time.time()
            self.sla_database_connection.col_sla_future_states.insert_many(to_save)
            # self.logger.debug('BMDEV [adding new] saving data in [%s] --- nb elements to save [%s]' % (self.logger.format_chrono(t2), len(to_save)))
        # self.logger.debug('BMDEV [adding new] end in [%s] --- nb elements to save [%s]' % (self.logger.format_chrono(t1), len(to_save)))
        
        # t1 = time.time()
        # self.logger.debug('BMDEV [update change] start ')
        
        item_uuid_in_base_and_to_save = to_save_item_uuid_future_states & in_base_item_uuid_future_states
        to_update = []
        if item_uuid_in_base_and_to_save:
            # t2 = time.time()
            uuid_to_update = []
            in_base_item_uuid_future_states = dict([(i['_id'], i.get('hash', 'no_hash_in_base')) for i in in_base_future_states if i['_id'] in item_uuid_in_base_and_to_save])
            for item_uuid in item_uuid_in_base_and_to_save:
                in_base_hash = in_base_item_uuid_future_states[item_uuid]
                to_save_hash = hash(future_states[item_uuid])
                if in_base_hash != to_save_hash:
                    to_update.append({'_id': item_uuid, 'future_state': future_states[item_uuid], 'hash': to_save_hash})
                    uuid_to_update.append(item_uuid)
            # self.logger.debug('BMDEV [update change] prepare data in [%s] --- nb elements to save [%s]' % (self.logger.format_chrono(t2), len(to_update)))
            
            if to_update:
                # t2 = time.time()
                self.sla_database_connection.col_sla_future_states.delete_many(uuid_to_update)
                # self.logger.debug('BMDEV [update change] removing data in [%s] --- nb elements to save [%s]' % (self.logger.format_chrono(t2), len(to_update)))
                # t2 = time.time()
                self.sla_database_connection.col_sla_future_states.insert_many(to_update)
                # self.logger.debug('BMDEV [update change] saving data in [%s] --- nb elements to save [%s]' % (self.logger.format_chrono(t2), len(to_update)))
        # self.logger.debug('BMDEV [update change] end in [%s] --- nb elements to save [%s]' % (self.logger.format_chrono(t1), len(to_update)))
        
        self.logger.info('Save future states in [%s]' % self.logger.format_chrono(t0))
    
    
    def load_future_states(self):
        future_states = {}
        entry_future_states = self.sla_database_connection.col_sla_future_states.find({})
        if entry_future_states:
            for entry_future_state in entry_future_states:
                if 'future_state' in entry_future_state:
                    raw_future_state = entry_future_state['future_state']
                    raw_future_state[0] = Date(raw_future_state[0][0], raw_future_state[0][1])
                    raw_future_state[6] = tuple(raw_future_state[6]) if raw_future_state[6] else tuple()
                    # surpatch_02_08_01_sla_future_state
                    if len(raw_future_state) > 7:
                        raw_future_state = raw_future_state[:7]
                    future_states[entry_future_state['_id']] = FutureState(*raw_future_state)
        
        return future_states
    
    
    def clean_future_states(self):
        t0 = time.time()
        t1 = time.time()
        date = date_now()
        to_del = []
        in_base_future_states = self.sla_database_connection.col_sla_future_states.find(projection={'_id': 1, 'future_state': {'$slice': 1}})
        for entry_future_state in in_base_future_states:
            if 'future_state' in entry_future_state:
                raw_future_state = entry_future_state['future_state']
                future_state_date = Date(raw_future_state[0][0], raw_future_state[0][1])
                
                if compare_date(date, future_state_date) == DATE_COMPARE.IS_BEFORE:
                    to_del.append(entry_future_state['_id'])
        search_time = time.time() - t1
        if to_del:
            t2 = time.time()
            self.sla_database_connection.col_sla_future_states.delete_many(to_del)
            self.logger.info('The clean of future states found [%s] entries to delete. There was deleted in [%s (%s for found entries, %s for delete entries)].' %
                             (len(to_del), self.logger.format_chrono(t0), self.logger.format_duration(search_time), self.logger.format_chrono(t2)))
        else:
            if search_time > 0.1:
                self.logger.info('The clean of future states not found entries to delete. The search took [%s].' % self.logger.format_duration(search_time))
            else:
                self.logger.debug('The clean of future states not found entries to delete. The search took [%s].' % self.logger.format_duration(search_time))
    
    
    def save_acknowledge(self, acknowledge):
        self.sla_database_connection.col_acknowledge.save(acknowledge)
    
    
    def update_acknowledges(self, acknowledges):
        self.sla_database_connection.col_acknowledge.replace_many(acknowledges)
    
    
    def find_acknowledge(self, acknowledge_id):
        return self.sla_database_connection.col_acknowledge.find_one({'_id': acknowledge_id})
    
    
    def find_acknowledge_not_migrate(self):
        return self.sla_database_connection.col_acknowledge.find({'type': {'$exists': True}})
    
    
    def save_downtime(self, downtime):
        self.sla_database_connection.col_downtime.save(downtime)
    
    
    def find_downtime(self, downtime_id):
        return self.sla_database_connection.col_downtime.find_one({'_id': downtime_id})
    
    
    def find_downtimes(self, downtime_ids):
        return self.sla_database_connection.col_downtime.find({'_id': {'$in': downtime_ids}})
    
    
    def compute_current_storage_size(self, keep_raw_sla_days):
        sla_archive_size = self.sla_database_connection.col_archive.get_size() if self.sla_database_connection.collection_exist('sla_archive') else 0
        sla_archived_before_migration_size = self.sla_database_connection.col_archive_before_migration.get_size() if self.sla_database_connection.collection_exist('sla_archive_version_0') else 0
        sla_future_states_size = self.sla_database_connection.col_sla_future_states.get_size() if self.sla_database_connection.collection_exist('sla_future_states') else 0
        sla_info_size = self.sla_database_connection.col_sla_info.get_size() if self.sla_database_connection.collection_exist('sla_info') else 0
        acknowledged_size = self.sla_database_connection.col_acknowledge.get_size() if self.sla_database_connection.collection_exist('acknowledge') else 0
        downtime_size = self.sla_database_connection.col_downtime.get_size() if self.sla_database_connection.collection_exist('downtime') else 0
        
        # Today's raw SLAs
        date = date_now()
        today_raw_sla_size = 0
        if self.sla_database_connection.collection_exist(date):
            today_raw_sla_size = self.sla_database_connection.get_raw_sla_collection(date, on_write=False).get_size()
        
        # Archived raw SLAs
        archived_raw_sla_size = 0
        for day in xrange(1, keep_raw_sla_days):
            date = get_previous_date(date)
            archived_collection_name = self.sla_database_connection.get_archived_sla_collection_name(date)
            if self.sla_database_connection.collection_exist(archived_collection_name):
                archived_raw_sla_size += self.sla_database_connection.get_collection(archived_collection_name).get_size()
        
        raw_sla_size = today_raw_sla_size + archived_raw_sla_size
        total_size = sla_archive_size + sla_archived_before_migration_size + sla_future_states_size + sla_info_size + acknowledged_size + downtime_size + raw_sla_size
        return total_size
    
    
    def compute_total_unique_elements_stored(self):
        col_archive_data = self.sla_database_connection.col_archive.aggregate([
            {'$group': {'_id': 0, 'uuids': {'$addToSet': '$uuid'}}},
            {'$project': {'uniqueUuidCount': {'$size': '$uuids'}}}
        ])
        if col_archive_data and len(col_archive_data['result']) > 0:
            return col_archive_data['result'][0]['uniqueUuidCount']
        return 0
