#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013:
#    Gabes Jean, naparuba@gmail.com
#
# This file is part of Shinken Enterprise, all rights reserved.

import httplib
import json
import os
import socket
import ssl
import threading
import time
import traceback
import urllib
import urlparse

try:
    import pwd
    import grp
except ImportError:
    # don't expect to have this on windows :)
    pwd = grp = None

import mapper
from nagvis_configuration_editor import NagVisConfigurationEditor
from shinken.basemodule import BaseModule, ModuleState
from shinken.brok import PersistantBrok
from shinken.log import LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_module.basemodule import BaseModule
from shinken.webui.bottlewebui import run, route, request, response
from shinken.webui.cherrypybackend import CherryPyServerHTTP
from shinkensolutions.lib_modules.configuration_reader import read_int_in_configuration, read_string_in_configuration, read_bool_in_configuration, read_list_in_configuration

if TYPE_CHECKING:
    from shinken.daemons.arbiterdaemon import Arbiter
    from shinken.log import PartLogger

_PATH_RECEIVED_ARCH_RETENTION = u'/var/lib/shinken/architecture_export_received.json'
properties = {
    u'daemons': [u'arbiter'],
    u'type'   : u'architecture_export',
    u'phases' : [u'configuration'],
}

logger = LoggerFactory.get_logger().get_sub_part(u'ARCHITECTURE-EXPORT')

MAP_GENERATION_TIMEOUT = 30


# called by the plugin manager to get a module
def get_instance(plugin):
    logger.info(u'Export Shinken architecture information to NagVis maps. %s' % plugin.get_name())
    
    # Catch errors
    path = plugin.path
    instance = ArchitectureExport(plugin, path)
    return instance


class ArchitectureExport(BaseModule):
    def __init__(self, mod_conf, path):
        BaseModule.__init__(self, mod_conf)
        self.path = path
        self.errors = None
        _conf_logger = logger.get_sub_part(u'CONFIGURATION')
        
        self.thread_counter = 0
        self.threads_status = {}
        self.threads_status_lock = threading.RLock()
        
        self._broks_to_send = {}
        # http server management
        self.srv = None
        self.srv_lock = threading.RLock()
        self.thread = None
        self.serveropts = {}
        self.arbiter = None
        try:
            _conf_logger.debug(u'Starting ...')
            self.architecture_name = read_string_in_configuration(mod_conf, u'architecture_name', u'Shinken', log_fct=_conf_logger)
            self.architecture_name_logger = logger.get_sub_part(self.architecture_name)
            self.send_my_architecture_to_recipients = read_list_in_configuration(mod_conf, u'send_my_architecture_to_recipients', [], log_fct=_conf_logger)
            
            self.map_base_url = read_string_in_configuration(mod_conf, u'map_base_url', u'', log_fct=_conf_logger)
            self.port = read_int_in_configuration(mod_conf, u'port', u'7780', log_fct=_conf_logger)
            self.host = read_string_in_configuration(mod_conf, u'host', u'0.0.0.0', log_fct=_conf_logger)
            self.use_ssl = read_bool_in_configuration(mod_conf, u'use_ssl', False, log_fct=_conf_logger)
            self.ssl_key = read_string_in_configuration(mod_conf, u'ssl_key', u'', log_fct=_conf_logger)
            self.ssl_cert = read_string_in_configuration(mod_conf, u'ssl_cert', u'', log_fct=_conf_logger)
            self.listener_use_ssl = read_int_in_configuration(mod_conf, u'listener_use_ssl', 0, log_fct=_conf_logger)
            self.listener_login = read_string_in_configuration(mod_conf, u'listener_login', u'Shinken', log_fct=_conf_logger)
            self.listener_password = read_string_in_configuration(mod_conf, u'listener_password', u'OFU2SE4zOU1FMDdaQlJENFtljgwTcWn2hQ5ocksBWS0=', log_fct=_conf_logger)
            self.broker_name = read_string_in_configuration(mod_conf, u'architecture_export__broker_connection__broker_name', u'broker-master', log_fct=_conf_logger)
            self.broker_livestatus = read_string_in_configuration(mod_conf, u'architecture_export__broker_connection__broker_livestatus', u'Livestatus', log_fct=_conf_logger)
            self.broker_webui_communication_type = read_string_in_configuration(mod_conf, u'architecture_export__broker_connection__broker_webui_communication_type', u'module', log_fct=_conf_logger)
            self.broker_webui_target = read_string_in_configuration(mod_conf, u'architecture_export__broker_connection__broker_webui_target', u'WebUI', log_fct=_conf_logger)
            
            self.ssh_port = int(getattr(mod_conf, u'ssh_port', u'22'))
            self.ssh_user = getattr(mod_conf, u'ssh_user', u'shinken')
            self.ssh_key_file = getattr(mod_conf, u'ssh_key_file', u'/var/lib/shinken/.ssh/id_rsa')
            self.ssh_timeout = int(getattr(mod_conf, u'ssh_timeout', u'5'))
            
            self.map_realm_layout = getattr(mod_conf, u'map_realm_layout', u'sort_by_size')
            if self.map_realm_layout not in mapper.MAP_REALM_LAYOUTS.keys():
                self.errors = u'The architecture-export Module could not initialize because the parameter \'map_realm_layout\' has an illegal value : %s' % self.map_realm_layout
                logger.error(u'%s' % self.errors)
            
            if self.ssl_key and not self.use_ssl:
                self.ssl_key = u''
            if self.ssl_cert and not self.use_ssl:
                self.ssl_cert = u''
            
            self.lang = u'en'
            _conf_logger.info(u'Configuration done, host: %s(%s)' % (self.host, self.port))
        except AttributeError:
            logger.error(u'The module is missing a property, check module declaration in architecture-export.cfg')
            raise
        except Exception as e:
            logger.error(u'Exception : %s' % str(e))
            raise

        if self.broker_webui_communication_type not in (u'module', u'url'):
            logger.error(u'architecture_export__broker_connection__broker_webui_communication_type parameter only can be "module" or "url"')
            raise Exception
    
    
    # used by the arbiter to get objects from modules
    def get_objects(self):
        return {}
    
    
    def get_state(self):
        if self.errors:
            return {"status": ModuleState.CRITICAL, "output": self.errors}
        return {"status": ModuleState.OK, "output": "OK"}
    
    
    # Used by the Arbiter to have the architecture name
    # TODO: the architecture need to be set elsewhere, maybe in a global way?
    def get_architecture_name(self):
        # type: () -> unicode
        return self.architecture_name
    
    
    # Called by Arbiter to say 'let's prepare yourself guy'
    def hook_daemon_daemonized(self, arb):
        # type: (Arbiter) -> None
        logger.info(u'Initialization of the Architecture Export module')
        self.arbiter = arb
        if self.arbiter.me.spare:
            return
        
        nagvis_configuration_editor = NagVisConfigurationEditor(logger, self.arbiter, self.broker_name, self.broker_livestatus, self.broker_webui_communication_type, self.broker_webui_target)
        nagvis_configuration_editor.read_and_set_nagvis_configuration()
        
        for mod in arb.conf.modules:
            if mod.module_type == u'webui':
                self.lang = read_string_in_configuration(mod, u'lang', u'en')
        self._start_http()
        # load previously presents architecture and send brok, the broker will already know that thoses maps exists
        try:
            self._broks_to_send = self._load_brok_arch()
            # Update broks info to account for arbiter address change
            self._broks_to_send = self._update_broks_arch_info(self._broks_to_send)
            self._add_arch_brok(self._broks_to_send)
        except Exception:
            pass
        
        # Send configuration after loading broks from retention to avoid conflicts when updating broks
        self.send_configuration(arb)
    
    
    def hook_tick(self, arb):
        to_del = []
        with self.threads_status_lock:
            for (thread_id, info) in self.threads_status.iteritems():
                status = info.get(u'status')
                thread = info.get(u'thread')
                server_uuid = info.get(u'server_uuid')
                _export_logger = logger.get_sub_part(info.get(u'architecture_name'))
                _map_generation_logger = _export_logger.get_sub_part(u'MAP GENERATOR')
                if status == u'DONE':
                    if thread:
                        thread.join()
                    
                    to_del.append(thread_id)
                    _export_logger.info(u'Export of architecture successful.')
                elif status == u'RUNNING':
                    if (info.get(u'timeout_logged') is None) and (int(time.time()) - info[u'creation_time'] > MAP_GENERATION_TIMEOUT):
                        _map_generation_logger.warning(u'Maps generation is taking more than %s seconds.' % MAP_GENERATION_TIMEOUT)
                        info[u'timeout_logged'] = True
                elif status == u'ERROR':
                    if thread:
                        thread.join()
                    
                    _map_generation_logger.error(u'Maps generation failed.')
                    to_del.append(thread_id)
                    self._broks_to_send.pop(server_uuid, None)
                    self._add_arch_brok(self._broks_to_send)
            
            for thread_id in to_del:
                del self.threads_status[thread_id]
    
    
    def quit(self):
        self.do_stop()
    
    
    def do_stop(self):
        self._save_brok_arch()
        self._stop_http()
    
    
    def _start_http(self):
        # We must protect against a user that spam the enable/disable button
        with self.srv_lock:
            # We already did start, skip it
            if self.thread is not None:
                if self.thread.is_alive():
                    return
                self.thread.join()
                self.thread = None
            self._init_http()
            self._start_thread()
    
    
    def _stop_http(self):
        # Already no more thread, we are great
        if self.thread is None:
            return
        # We must protect against a user that spam the enable/disable button
        with self.srv_lock:
            self.srv.stop()
            self.thread.join(1)
            self.thread = None
            _logger = logger.get_sub_part(u'SERVER')
            _logger.info(u'Http listener stopped')
    
    
    def _start_thread(self):
        self.thread = threading.Thread(None, target=self._http_start, name='architecture-export-http')
        self.thread.daemon = True
        self.thread.start()
    
    
    def _http_start(self):
        # Now block and run
        _logger = logger.get_sub_part(u'SERVER')
        _logger.info(u'Server starting')
        self.srv.start()
    
    
    # return a json containing the previous architecture broks
    def _load_brok_arch(self):
        try:
            file_path = _PATH_RECEIVED_ARCH_RETENTION
            if not os.path.exists(os.path.dirname(file_path)):
                os.makedirs(os.path.dirname(file_path))
            if os.path.isfile(file_path):
                self._broks_to_send = json.load(open(file_path, u'r'))
                if self._broks_to_send is None:
                    self._broks_to_send = {}
        except Exception as e:
            self._broks_to_send = {}
            logger.error(u'Previous arch broks retention cannot be loaded from file [%s] and will be empty. \n%s' % (_PATH_RECEIVED_ARCH_RETENTION, e))
        
        return self._broks_to_send
    
    
    def _save_brok_arch(self, _logger=None):
        # type: (PartLogger) -> None
        if _logger is None:
            _logger = logger
        try:
            file_path = _PATH_RECEIVED_ARCH_RETENTION
            if not os.path.exists(os.path.dirname(file_path)):
                os.makedirs(os.path.dirname(file_path))
            with open(file_path, u'w') as fd:
                json.dump(self._broks_to_send, fd)
            _logger.info(u'Save broks contents %s' % file_path)
        except:
            _logger.error(u'Save broks error in file [%s] fail : [%s]' % (_PATH_RECEIVED_ARCH_RETENTION, traceback.format_exc()))
    
    
    def _add_arch_brok(self, _broks_to_send):
        # Avoid sending None broks to Broker that can cause errors
        if _broks_to_send is None:
            return
        
        # this is optional, remove the old architecture_export_map avoid spamming the brokers
        architecture_export_brok_type = u'architecture_export_map'
        for key, brok in self.arbiter.persistant_broks.items():
            if brok.type == architecture_export_brok_type:
                self.arbiter.persistant_broks.pop(key, None)
        brok = PersistantBrok(architecture_export_brok_type, _broks_to_send)
        self.arbiter.add(brok)
    
    
    def _update_broks_arch_info(self, _broks_to_send):
        for server_uuid, arch_info in _broks_to_send.iteritems():
            _broks_to_send[server_uuid] = self._get_brok_content(server_uuid, arch_info[u'name'])
        return _broks_to_send
    
    
    def _get_thread_id(self):
        self.thread_counter += 1
        return self.thread_counter
    
    
    # We have the configuration loaded here,
    # this is the time to send the map to my recipient,
    # if disable is True it's time to tell my recipient that my arch have been disabled
    def send_configuration(self, arbiter, disabled=False):
        # type: (Arbiter, bool) -> None
        server_uuid = arbiter.server_uuid.replace(u'\n', u'')
        if len(arbiter.conf.synchronizers) >= 1:
            synchronizer = arbiter.conf.synchronizers[0]
        else:
            synchronizer = None
        
        broks_to_send = self._get_brok_content(server_uuid, self.architecture_name)
        mapper_instance = mapper.ArchitectureExportMapper(self.path,
                                                          self.architecture_name,
                                                          server_uuid,
                                                          synchronizer,
                                                          self.lang,
                                                          self.listener_use_ssl,
                                                          self.listener_login,
                                                          self.listener_password,
                                                          broks_to_send,
                                                          self.ssh_port,
                                                          self.ssh_user,
                                                          self.ssh_key_file,
                                                          self.ssh_timeout,
                                                          self.map_realm_layout)
        arch_realm = mapper_instance.get_architecture_realm(conf=arbiter.conf)
        
        self.architecture_name_logger.info(u'Sending architecture to %d recipient(s)' % (len(self.send_my_architecture_to_recipients)))
        for other_export_module in self.send_my_architecture_to_recipients:
            other_export_module = other_export_module.strip()
            original_address = other_export_module
            self.architecture_name_logger.info(u'Sending architecture to [%s]' % original_address)
            address_has_formatting_errors = False
            
            parsed_url = urlparse.urlparse(other_export_module)
            send_with_https = False

            if parsed_url.scheme == u'http':
                send_with_https = False
            elif parsed_url.scheme == u'https':
                send_with_https = True
            else:
                self.architecture_name_logger.error(u'External export module address "%s" has an incorrect format ( does not start with "http://" or "https://" )' % original_address)
                address_has_formatting_errors = True
            
            if not parsed_url.port:
                self.architecture_name_logger.error(u'No port specified in in external export module address "%s"' % original_address)
                address_has_formatting_errors = True
            
            
            if address_has_formatting_errors:
                self.architecture_name_logger.error(u'Due to address error, we will not send information to this address.')
                continue
            
            nb_try = 0
            max_try = 3
            while nb_try < max_try:  # do the try for max_try times
                nb_try += 1
                try:
                    if send_with_https:
                        args = {}
                        # If we are in SSL mode, do not look at certificate too much
                        # NOTE: ssl.SSLContext is only available on last python 2.7 versions
                        if hasattr(ssl, u'SSLContext'):
                            ssl_context = ssl.SSLContext(ssl.PROTOCOL_TLSv1)
                            ssl_context.check_hostname = False
                            ssl_context.verify_mode = ssl.CERT_NONE
                            conn = httplib.HTTPSConnection(parsed_url.netloc, timeout=3, context=ssl_context)
                        else:
                            conn = httplib.HTTPSConnection(parsed_url.netloc, timeout=3)
                    else:
                        conn = httplib.HTTPConnection(parsed_url.netloc, timeout=3)
                    headers = {u'Content-type': u'application/json', u'Accept': u'text/plain'}
                    path = u'/v1/architecture/%s/%s' % (server_uuid, self.architecture_name,)
                    path = urllib.pathname2url(path.encode(u'utf-8'))
                    # change the HTTP method if we need to notify about a disable
                    http_method = u'POST'
                    if disabled:
                        http_method = u'DELETE'
                    conn.request(http_method, path, json.dumps(arch_realm), headers)
                    response = conn.getresponse()
                    res = response.read()
                    if response.status not in (httplib.CREATED, httplib.OK):
                        self.architecture_name_logger.error(u'An error occurred when sending the architecture to [%s]: %s - %s' % (other_export_module, response.status, res))
                    break
                except (httplib.HTTPException, socket.error, socket.timeout) as exp:
                    message = u'Cannot send the architecture to [%s]; "%s" occurred, still %d tries' % (other_export_module, exp, max_try - nb_try)
                    if nb_try == 1:
                        self.architecture_name_logger.debug(message)
                    else:
                        self.architecture_name_logger.error(message)
                    time.sleep(nb_try)
    
    
    # generate a brok with the architecture_name and the url of the map
    def _get_brok_content(self, server_uuid, architecture_name):
        if not self.map_base_url:
            base_url = u'http://%s' % self.arbiter.me.address
        else:
            base_url = self.map_base_url
            if base_url.endswith(u'/'):
                base_url = base_url[:-1]
            if not (base_url.startswith(u'http://') or base_url.startswith(u'https://')):
                base_url = u'http://%s' % base_url
        
        return {
            u'name'    : architecture_name,
            u'tree_map': {
                u'name': mapper.translate(u'realms_tree_link_name', self.lang),
                u'url' : u'%s/shinken-core-map/frontend/nagvis-js/index.php?mod=Map&show=shinken_global-%s' % (base_url, server_uuid),
            },
            u'arch_map': {
                u'name': mapper.translate(u'shinken_architecture_link_name', self.lang),
                u'url' : u'%s/shinken-core-map/frontend/nagvis-js/index.php?mod=Map&show=shinken_architecture-%s' % (base_url, server_uuid),
            }
        }
    
    
    # We initialize the HTTP part. It's a simple wsgi backend
    # with a select hack so we can still exit if someone ask it
    def _init_http(self):
        _logger = logger.get_sub_part(u'SERVER')
        _logger.info(u'Init http socket for incoming request')
        
        
        # used when another Architecture Export module send us their configuration
        def receive_architecture(server_uuid, architecture_name):
            sender = request.environ.get('REMOTE_ADDR')
            _logger = logger.get_sub_part(architecture_name)
            _logger.info(u'[ %s ] ask me to map its architecture' % sender)
            _logger.info(u'The Architecture has been received, we will make its maps')
            response.content_type = u'application/json'
            
            if len(self.arbiter.conf.synchronizers) >= 1:
                synchronizer = self.arbiter.conf.synchronizers[0]
            else:
                synchronizer = None
            
            broks_to_send = self._get_brok_content(server_uuid, architecture_name)
            mapperInstance = mapper.ArchitectureExportMapper(self.path,
                                                             architecture_name,
                                                             server_uuid,
                                                             synchronizer,
                                                             self.lang,
                                                             self.listener_use_ssl,
                                                             self.listener_login,
                                                             self.listener_password,
                                                             broks_to_send,
                                                             self.ssh_port,
                                                             self.ssh_user,
                                                             self.ssh_key_file,
                                                             self.ssh_timeout,
                                                             self.map_realm_layout)
            try:
                thread_id = self._get_thread_id()
                self.threads_status[thread_id] = {
                    u'status'           : u'RUNNING',
                    u'creation_time'    : int(time.time()),
                    u'server_uuid'      : server_uuid,
                    u'architecture_name': architecture_name
                }
                # calling the make_map (threaded) failed on permission denied
                response.status = 201
                self._broks_to_send[server_uuid] = broks_to_send
                self._add_arch_brok(self._broks_to_send)
                self._save_brok_arch(_logger)
                mapperInstance.make_map(thread_id, self.threads_status, self.threads_status_lock, request.json)
            except Exception as e:
                logger.error(e)
                response.status = 500
                self._broks_to_send.pop(server_uuid, None)
                self._add_arch_brok(self._broks_to_send)
            return u'done'
        
        
        # used when another Architecture Export module send us their configuration and notify us that we need to disable it
        def receive_architecture_to_disable(server_uuid, architecture_name):
            sender = request.environ.get(u'REMOTE_ADDR')
            _logger = logger.get_sub_part(architecture_name)
            _logger.debug(u'%s ask me to disable the architecture' % sender)
            _logger.info(u'The Architecture needs to be disabled (request received)')
            response.content_type = u'application/json'
            if len(self.arbiter.conf.synchronizers) >= 1:
                synchronizer = self.arbiter.conf.synchronizers[0]
            else:
                synchronizer = None
            mapper_instance = mapper.ArchitectureExportMapper(self.path,
                                                              architecture_name,
                                                              server_uuid,
                                                              synchronizer,
                                                              self.lang,
                                                              self.listener_use_ssl,
                                                              self.listener_login,
                                                              self.listener_password,
                                                              self._broks_to_send,
                                                              self.ssh_port,
                                                              self.ssh_user,
                                                              self.ssh_key_file,
                                                              self.ssh_timeout,
                                                              self.map_realm_layout)
            try:
                self._broks_to_send.pop(server_uuid, None)
                self._add_arch_brok(self._broks_to_send)
                # calling the make_map (threaded) failed on permission denied
                mapper_instance.remove_map(request.json)
                response.status = 201
            except Exception as e:
                logger.error(e)
                response.status = 500
            return u'done'
        
        
        # used when the addon system disable the module
        def module_disabled():
            logger.info(u'Module disabled with the shinken-addons-disable command, remove my architecture to all my "recipient"')
            self.send_configuration(self.arbiter, disabled=True)
            response.status = 200
            return u'done'
        
        
        # Manually send architecture information
        def send_architecture_to_recipients():
            self.architecture_name_logger.info(u'Sending architecture information to all recipients listed in the architecture-export module configuration')
            self.send_configuration(self.arbiter)
            response.status = 200
            return u'done'
        
        
        try:
            # used when another Architecture Export module send us their configuration
            route(u'/v1/architecture/:server_uuid/:architecture_name', callback=receive_architecture, method=(u'POST', u'PUT', u'PATCH'))
            # used when another Architecture Export module send us their configuration and notify us that we need to disable it
            route(u'/v1/architecture/:server_uuid/:architecture_name', callback=receive_architecture_to_disable, method=u'DELETE')
            # used when the addon system disable the module
            route(u'/v1/architecture/disabled', callback=module_disabled, method=u'GET')
            # used when the addon system disable the module
            route(u'/v1/architecture/send', callback=send_architecture_to_recipients, method=u'POST')
            self.srv = run(host=self.host, port=self.port, server=CherryPyServerHTTP, use_ssl=self.use_ssl, ssl_key=self.ssl_key, ssl_cert=self.ssl_cert, **self.serveropts)
        except Exception as e:
            _logger.error(u'Exception : %s' % str(e))
            raise
        _logger.info(u'Server loaded')
