#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2021
# This file is part of Shinken Enterprise, all rights reserved.

import os
import shutil
import socket

try:
    import pwd
    import grp
except ImportError:
    # don't expect to have this on windows :)
    pwd = grp = None

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.util import to_bool

if TYPE_CHECKING:
    from shinken.brokerlink import BrokerLink
    from shinken.daemons.arbiterdaemon import Arbiter
    from shinken.misc.type_hint import Str, Optional, Dict
    from shinken.log import PartLogger
    from shinken.objects import Module as ModuleConfiguration


# This object will read module configuration and edit NagVis one
class NagVisConfigurationEditor(object):
    def __init__(self, logger, arbiter, broker_name, broker_livestatus, broker_webui_communication_type, broker_webui_target):
        # type: (PartLogger, Arbiter, unicode, unicode, unicode, unicode) -> NagVisConfigurationEditor
        self.logger = logger
        
        self.arbiter = arbiter
        
        self.broker_name = broker_name
        self.broker_webui_communication_type = broker_webui_communication_type
        self.broker_webui_target = broker_webui_target
        self.broker_livestatus = broker_livestatus
    
    
    @staticmethod
    def find_module_from_name(broker, module_name):
        # type: (BrokerLink, Str) -> Optional[ModuleConfiguration]
        for module in broker.modules:
            if module.get_name() == module_name:
                return module
        return None
    
    
    def find_broker_from_name(self, broker_name):
        # type: (Str) -> Optional[BrokerLink]
        for broker in self.arbiter.conf.brokers:
            if broker.get_name() == broker_name:
                return broker
        return None
    
    
    def _read_webui_direct_url_configuration(self, _logger):
        # type: (PartLogger) -> Optional[Dict]
        # If we have an url, we will parse it to get every informations we need
        _logger.info(u'The value of parameter [ architecture_export__broker_connection__broker_webui_communication_type ] is "url"')
        webui_plain_address = self.broker_webui_target
        webui_remote_user_variable = None
        if self.broker_webui_target.startswith(u'http://'):
            webui_protocol = u'http'
            webui_plain_address = webui_plain_address.replace(u'http://', u'')
        elif self.broker_webui_target.startswith(u'https://'):
            webui_protocol = u'https'
            webui_plain_address = webui_plain_address.replace(u'https://', u'')
        else:
            _logger.error(
                u'Url "%s" from parameter [ architecture_export__broker_connection__broker_webui_target ] has an incorrect format (does not start with "http://" or "https://"), we will not change the configuration of the communication between NagVis and the WebUI' % self.broker_webui_target)
            return None
        
        if len(webui_plain_address.split(u':')) != 2:
            if webui_protocol == u'http':
                _logger.info(u'No port specified in url "%s" from parameter [ architecture_export__broker_connection__broker_webui_target ], we will use default HTTP port [ 80 ]' % self.broker_webui_target)
                webui_port = u'80'
            else:
                _logger.info(u'No port specified in url "%s" from parameter [ architecture_export__broker_connection__broker_webui_target ], we will use default HTTPS port [ 443 ]' % self.broker_webui_target)
                webui_port = u'443'
        else:
            webui_port = webui_plain_address.split(u':')[1].strip(u'/')
            webui_plain_address = webui_plain_address.split(u':')[0]
        
        _logger.info(u'WebUI address will be %s' % self.broker_webui_target)
        webui_complete_address = self.broker_webui_target.strip('/')
        return {u'address': webui_plain_address, u'full_address': webui_complete_address, u'protocol': webui_protocol, u'port': webui_port, u'remote_user_variable': webui_remote_user_variable}
    
    
    def _read_webui_module_configuration(self, _logger, broker):
        # type: (PartLogger, BrokerLink) -> Optional[Dict]
        # If we directly have a WebUI module, we just need to read its configuration
        webui = self.find_module_from_name(broker, self.broker_webui_target)
        if not webui:
            _logger.error(
                u'The WebUI module [ %s ] set with parameter [ architecture_export__broker_connection__broker_webui_target ] do not exists on the Broker [ %s ], we will not change the configuration of the communication between NagVis and the WebUI' % (
                    self.broker_webui_target, self.broker_name))
            return None
        else:
            _logger.info(u'WebUI module [ %s ] found from parameter [ architecture_export__broker_connection__broker_webui_target ]' % self.broker_webui_target)
            webui_protocol = u'https' if to_bool(webui.use_ssl) else u'http'
            webui_port = webui.port
            webui_remote_user_variable = webui.remote_user_variable if to_bool(webui.remote_user_enable) else None
            if webui.host in (u'0.0.0.0', u'*'):
                webui_address = broker.address
                if broker.address in (u'localhost', u'127.0.0.1'):
                    webui_plain_address = socket.gethostbyname(socket.gethostname())
                    webui_complete_address = u'%s://%s:%s' % (webui_protocol, webui_plain_address, webui_port)
                    _logger.info(u'The WebUI address is [ %s ], we will use its public address [ %s ]' % (webui_address, webui_complete_address))
                else:
                    webui_plain_address = broker.address
                    webui_complete_address = u'%s://%s:%s' % (webui_protocol, webui_plain_address, webui_port)
                    _logger.info(u'The WebUI address is [ %s ]' % webui_complete_address)
            else:
                if webui.host in (u'localhost', u'127.0.0.1'):
                    _logger.warning(u'The WebUI only listens to [ %s ]. It will not be reachable outside of the hosting machine') % webui.host
                webui_plain_address = webui.host
                webui_complete_address = u'%s://%s:%s' % (webui_protocol, webui_plain_address, webui_port)
                _logger.info(u'The WebUI address is [ %s ]' % webui_complete_address)
        return {u'address': webui_plain_address, u'full_address': webui_complete_address, u'protocol': webui_protocol, u'port': webui_port, u'remote_user_variable': webui_remote_user_variable}
    
    
    def _read_webui_configuration(self, _logger, broker):
        # type: (PartLogger, BrokerLink) -> Optional[Dict]
        # Here we will find address, port and protocol for WebUI redirection
        if self.broker_webui_communication_type == u'url':
            return self._read_webui_direct_url_configuration(_logger)
        else:
            return self._read_webui_module_configuration(_logger, broker)
    
    
    def _read_livestatus_configuration(self, _logger, broker):
        # type: (PartLogger, BrokerLink) -> Optional[unicode]
        # Here we want to get the livestatus address from module name
        livestatus_address = u''
        if self.broker_livestatus:
            livestatus = self.find_module_from_name(broker, self.broker_livestatus)
            if not livestatus:
                _logger.error(
                    u'The Livestatus module [ %s ] set with parameter [ architecture_export__broker_connection__broker_livestatus ] do not exists on the Broker [ %s ], we will not change the configuration of the communication between NagVis and the Livestatus' % (
                        self.broker_livestatus, self.broker_name))
            else:
                _logger.info(u'Livestatus module [ %s ] found from parameter [ architecture_export__broker_connection__broker_livestatus ]' % self.broker_livestatus)
                livestatus_address_tmp = broker.address if livestatus.host in (u'0.0.0.0', u'*') else livestatus.host
                livestatus_address = u'%s:%s' % (livestatus_address_tmp, livestatus.port)
                _logger.info(u'The Livestatus address is [ %s ]' % livestatus_address)
                livestatus_address = u'socket="tcp:%s"' % livestatus_address
        return livestatus_address
    
    
    def read_and_set_nagvis_configuration(self):
        _logger = self.logger.get_sub_part(u'READING CONFIGURATION')
        broker = self.find_broker_from_name(self.broker_name)
        webui_info = None
        livestatus_address = None
        if not broker:
            _logger.error(u'The Broker [ %s ] set with parameter [ architecture_export__broker_connection__broker_name ] do not exists. Abort edition of the NagVis config file' % self.broker_name)
        else:
            _logger.info(u'Broker [ %s ] found from parameter [ architecture_export__broker_connection__broker_name ]' % broker.get_name())
            webui_info = self._read_webui_configuration(_logger, broker)
            livestatus_address = self._read_livestatus_configuration(_logger, broker)
        
        need_modification = True
        if not webui_info and not livestatus_address:
            need_modification = False
            _logger.info(u'=================================================================')
            _logger.info(u'DUE TO CONFIGURATION ERROR, The connection between NagVis and Shinken modules will NOT be updated')
            _logger.info(u'=================================================================')
        self.edit_nagvis_file(u'nagvis-shinken-architecture', u'/etc/shinken/external/nagvis/etc/nagvis.ini.php', webui_info, livestatus_address, need_modification)
        self.edit_nagvis_file(u'nagvis', u'/opt/nagvis/etc/nagvis.ini.php', webui_info, livestatus_address, need_modification)
        if not need_modification:
            _logger.info(u'=================================================================')
    
    
    def edit_nagvis_file(self, addon_name, configuration_path, webui_info, livestatus_address, need_modification=True):
        # type: (unicode, unicode, Optional[Dict], unicode, bool) -> None
        _logger = self.logger.get_sub_part(addon_name).get_sub_part(u'CONFIGURATION')
        if need_modification:
            _logger.info(u'UPDATING connection between NagVis and Shinken modules')
            _logger.info(u'Editing %s file' % configuration_path)
        else:
            _logger.info(u'The connection between NagVis and Shinken modules will not be changed')
            _logger.info(u'Reading %s file' % configuration_path)
        with open(configuration_path) as f:
            configuration_content = f.readlines()
        
        if configuration_content:
            with open(u'%s.tmp' % configuration_path, 'w+') as f:
                reading_backend = False
                if webui_info:
                    _logger.info(u'   - Setting WebUI connection to allow redirection on NagVis maps')
                else:
                    _logger.info(u'   - No WebUI redirection configuration found, we will keep previous NagVis configuration. Redirection to WebUI on NagVis maps may not work properly')
                
                for line in configuration_content:
                    if reading_backend and line.startswith(u'socket='):
                        if livestatus_address:
                            _logger.info(u'   - Setting Livestatus connection to get Shinken objects status')
                            _logger.info(u'       => "socket" parameter set to [ %s ]' % livestatus_address)
                            line = u'%s\n' % livestatus_address
                        else:
                            _logger.info(u'   - No Livestatus module found, we will keep previous NagVis configuration. Status of Shinken elements on NagVis maps may not work properly')
                            _logger.info(u'       => "socket" parameter not changed. It is still [ %s ]' % line.strip().split('=')[1])
                        reading_backend = False
                    
                    if line.startswith(u'[backend_shinken_livestatus]'):
                        reading_backend = True
                    
                    if webui_info:
                        if line.startswith(u';shinken_auth_protocol') or line.startswith(u'shinken_auth_protocol'):
                            _logger.info(u'       => "shinken_auth_protocol" parameter set to [ %s ]' % webui_info[u'protocol'])
                            line = u'shinken_auth_protocol="%s"\n' % webui_info[u'protocol']
                        elif line.startswith(u';shinken_auth_port') or line.startswith(u'shinken_auth_port'):
                            _logger.info(u'       => "shinken_auth_port" parameter set to [ %s ]' % webui_info[u'port'])
                            line = u'shinken_auth_port="%s"\n' % webui_info[u'port']
                        elif line.startswith(u';shinken_auth_address') or line.startswith(u'shinken_auth_address'):
                            _logger.info(u'       => "shinken_auth_address" parameter set to [ %s ]' % webui_info[u'address'])
                            line = u'shinken_auth_address="%s"\n' % webui_info[u'address']
                        elif line.startswith(u';shinken_auth_remote_user_variable') or line.startswith(u'shinken_auth_remote_user_variable'):
                            if webui_info[u'remote_user_variable']:
                                _logger.info(u'       => "shinken_auth_remote_user_variable" parameter set to [ %s ]' % webui_info[u'remote_user_variable'])
                                line = u'shinken_auth_remote_user_variable="%s"\n' % webui_info[u'remote_user_variable']
                            else:
                                line = u';shinken_auth_remote_user_variable=""\n'
                    else:
                        if line.startswith(u';shinken_auth_protocol') or line.startswith(u'shinken_auth_protocol'):
                            _logger.info(u'       => "shinken_auth_protocol" parameter not changed. It is still [ %s ]' % line.strip().split('=')[1])
                        elif line.startswith(u';shinken_auth_port') or line.startswith(u'shinken_auth_port'):
                            _logger.info(u'       => "shinken_auth_port" parameter not changed. It is still [ %s ]' % line.strip().split('=')[1])
                        elif line.startswith(u';shinken_auth_address') or line.startswith(u'shinken_auth_address'):
                            _logger.info(u'       => "shinken_auth_address" parameter not changed. It is still [ %s ]' % line.strip().split('=')[1])
                        elif line.startswith(u';shinken_auth_remote_user_variable') or line.startswith(u'shinken_auth_remote_user_variable'):
                            _logger.info(u'       => "shinken_auth_remote_user_variable" not changed. It is still [ %s ]' % line.strip().split('=')[1])
                    
                    if line.startswith(u'hosturl='):
                        if webui_info:
                            _logger.info(u'       => "hosturl" parameter set to [ %s ]' % (u'%s/detail-by-name/[host_name]' % webui_info[u'full_address']))
                            line = u'hosturl="%s/detail-by-name/[host_name]"\n' % webui_info[u'full_address']
                        else:
                            _logger.info(u'       => "hosturl" parameter not changed. It is still [ %s ]' % line.strip().split('=')[1])
                    elif line.startswith(u'serviceurl='):
                        if webui_info:
                            _logger.info(u'       => "serviceurl" parameter set to [ %s ]' % (u'%s/detail-by-name/[host_name]/checks/[service_description]' % webui_info[u'full_address']))
                            line = u'serviceurl="%s/detail-by-name/[host_name]/checks/[service_description]"\n' % webui_info[u'full_address']
                        else:
                            _logger.info(u'       => "serviceurl" parameter not changed. It is still [ %s ]' % line.strip().split('=')[1])
                    f.write(line)
            shutil.move(u'%s.tmp' % configuration_path, configuration_path)
            uid = pwd.getpwnam(u'shinken').pw_uid
            gid = grp.getgrnam(u'apache').gr_gid
            os.chmod(configuration_path, 0664)
            os.chown(configuration_path, uid, gid)
        if need_modification:
            _logger.info(u'File %s edited' % configuration_path)
