#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import json
import time

from shinken.log import logger
from shinken.misc.perfdata import PerfDatas
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional
    from ...module import broker_module_livedata

app = None  # type: Optional[broker_module_livedata]


def add_error(error, ip):
    actual_time = time.time()
    app.nb_errors_by_hour.append(actual_time)
    app.errors.append({'time': actual_time, 'error': error})
    last_time_warn = actual_time
    errors = []
    if ip in app.errors_per_ip.keys():
        last_time_warn = app.errors_per_ip[ip]['last_time_warn']
        errors = app.errors_per_ip[ip]['errors']
    for error_time in errors:
        if actual_time - error_time >= 3600:
            del error_time
        else:
            break
    errors.append(actual_time)
    app.errors_per_ip[ip] = {'last_time_warn': last_time_warn, 'nb_errors': len(errors), 'errors': errors}
    if app.errors_per_ip[ip]['nb_errors'] == 5 or actual_time - app.errors_per_ip[ip]['last_time_warn'] > 300 or app.errors_per_ip[ip]['nb_errors'] % 500 == 0:
        app.errors_per_ip[ip]['last_time_warn'] = actual_time
        return True
    return False


def add_response_time(actual_time, response_time):
    app.average_time_response.update_avg(response_time)
    if response_time > app.worst_response_time['response_time']:
        app.worst_response_time = {'time': actual_time, 'response_time': response_time}


def get_summary(elt):
    clsname = elt.__class__.my_type
    is_host = (clsname == 'host')
    is_service = (clsname == 'service')
    
    status = elt.state
    if status == 'UP':
        status = 'OK'
    if status == 'DOWN':
        status = 'CRITICAL'
    # Catch cluster degraded state
    if is_host and elt.got_business_rule:
        status = elt.bp_state
    
    # DISABLED is for hosts, service can't be like this because an host disabled got no service
    if is_host and 'disabled' in elt.tags:
        return {'status': status, 'summary': 'DISABLED'}
    
    if elt.downtimes and next((dt for dt in elt.downtimes if dt.ref == elt and dt.is_in_effect), None):
        return {'status': status, 'summary': 'DOWNTIME'}
    
    if elt.is_in_inherited_downtime():
        return {'status': status, 'summary': 'INHERITED-DOWNTIME'}
    
    if elt.in_partial_downtime:
        return {'status': status, 'summary': 'PARTIAL-DOWNTIME'}
    
    if elt.acknowledgement and not elt.acknowledgement.automatic:
        return {'status': status, 'summary': 'ACKNOWLEDGED'}
    
    if (elt.acknowledgement and elt.acknowledgement.automatic) or (is_service and elt.host.acknowledgement and not elt.host.acknowledgement.automatic):
        return {'status': status, 'summary': 'INHERITED-ACKNOWLEDGED'}
    
    if elt.is_partial_acknowledged:
        return {'status': status, 'summary': 'PARTIAL-ACKNOWLEDGED'}
    
    if elt.is_flapping:
        return {'status': status, 'summary': 'FLAPPING'}
    
    if elt.got_business_rule and elt.is_partial_flapping:
        return {'status': status, 'summary': 'PARTIAL-FLAPPING'}
    
    return {'status': status, 'summary': 'NOTHING'}


def get_all_monitored_elements():
    start_time = time.time()
    # First we need to check if the token is good
    remote_ip = app.request.remote_addr
    token = app.request.GET.get('token', '')
    if remote_ip in app.errors_per_ip.keys():
        nb_error_ip = app.errors_per_ip[remote_ip]['nb_errors']
    else:
        nb_error_ip = 0
    if token != app.token:
        error = '%s try to call the api all-monitored-elements with the \'%s\' token (wrong token), %d errors with this ip' % (remote_ip, token, nb_error_ip + 1)
        to_logger = add_error(error, remote_ip)
        if to_logger is True:
            logger.error(error)
        
        return app.abort(401, 'wrong token', True, to_logger)
    data = app.datamgr.get_hosts().items
    to_return = {'clusters': [], 'hosts': []}
    
    for item in data:
        item = data[item]
        is_cluster = False
        uuid_name = 'host_uuid'
        if item.got_business_rule:
            is_cluster = True
            uuid_name = 'cluster_uuid'
        _dict = {uuid_name: item.uuid, 'checks': []}
        for check in item.services:
            _dict['checks'].append('%s-%s' % (_dict[uuid_name], check.uuid))
        if is_cluster:
            to_return['clusters'].append(_dict)
        else:
            to_return['hosts'].append(_dict)
    
    end_time = time.time()
    add_response_time(start_time, end_time - start_time)
    return json.dumps(to_return)


def get_all_hosts():
    start_time = time.time()
    # First we need to check if the token is good
    remote_ip = app.request.remote_addr
    token = app.request.GET.get(u'token', u'')
    query_hostgroup_names = app.request.GET.get(u'hostgroup_name', '')
    query_hostgroup_names = set(query_hostgroup_names.split(u'^^') if query_hostgroup_names else [])
    
    if remote_ip in app.errors_per_ip.keys():
        nb_error_ip = app.errors_per_ip[remote_ip][u'nb_errors']
    else:
        nb_error_ip = 0
    if token != app.token:
        error = u'%s try to call the api host with the \'%s\' token (wrong token), %d errors with this ip' % (remote_ip, token, nb_error_ip + 1)
        to_logger = add_error(error, remote_ip)
        if to_logger is True:
            logger.error(error)
        
        return app.abort(401, u'wrong token', True, to_logger)
    data = app.datamgr.get_hosts().items
    to_return = {u'hosts': []}
    
    for item in data:
        item = data[item]
        if item.got_business_rule:
            continue
        if query_hostgroup_names:
            if not item.hostgroups:
                continue
            match_hostgroup_name = next((hostgroup.hostgroup_name for hostgroup in item.hostgroups if hostgroup.hostgroup_name in query_hostgroup_names), None)
            if not match_hostgroup_name:
                continue
        _dict = {u'host_uuid': item.uuid, u'host_name': item.host_name, u'checks': []}
        for check in item.services:
            _dict[u'checks'].append(u'%s-%s' % (_dict[u'host_uuid'], check.uuid))
        to_return[u'hosts'].append(_dict)
    
    end_time = time.time()
    add_response_time(start_time, end_time - start_time)
    return json.dumps(to_return)


def get_all_clusters():
    start_time = time.time()
    # First we need to check if the token is good
    remote_ip = app.request.remote_addr
    token = app.request.GET.get('token', '')
    if remote_ip in app.errors_per_ip.keys():
        nb_error_ip = app.errors_per_ip[remote_ip]['nb_errors']
    else:
        nb_error_ip = 0
    if token != app.token:
        error = '%s try to call the api cluster with the \'%s\' token (wrong token), %d errors with this ip' % (remote_ip, token, nb_error_ip + 1)
        to_logger = add_error(error, remote_ip)
        if to_logger is True:
            logger.error(error)
        
        return app.abort(401, 'wrong token', True, to_logger)
    data = app.datamgr.get_hosts().items
    to_return = {'clusters': []}
    
    for item in data:
        item = data[item]
        if not item.got_business_rule:
            continue
        _dict = {'cluster_uuid': item.uuid, 'checks': []}
        for check in item.services:
            _dict['checks'].append('%s-%s' % (_dict['cluster_uuid'], check.uuid))
        to_return['clusters'].append(_dict)
    
    end_time = time.time()
    add_response_time(start_time, end_time - start_time)
    return json.dumps(to_return)


def get_element_cluster(uuid):
    start_time = time.time()
    # First we need to check if the token is good
    remote_ip = app.request.remote_addr
    token = app.request.GET.get('token', '')
    if remote_ip in app.errors_per_ip.keys():
        nb_error_ip = app.errors_per_ip[remote_ip]['nb_errors']
    else:
        nb_error_ip = 0
    if token != app.token:
        error = '%s try to call the api cluster/uuid with the \'%s\' token (wrong token), %d errors with this ip' % (remote_ip, token, nb_error_ip + 1)
        to_logger = add_error(error, remote_ip)
        if to_logger is True:
            logger.error(error)
        
        return app.abort(401, 'wrong token', True, to_logger)
    elt = app.datamgr.get_host_by_uuid(uuid)
    
    # If the element is unknown, we need to tell it
    if elt is None:
        error = '%s try to call the api cluster/uuid but the uuid \'%s\' doesn\'t return anything, %d errors with this ip' % (remote_ip, uuid, nb_error_ip + 1)
        to_logger = add_error(error, remote_ip)
        if to_logger is True:
            logger.error(error)
        return app.abort(404, 'can\'t find cluster with this uuid', to_logger=to_logger)
    
    # No business rule = not a cluster
    if not elt.got_business_rule:
        error = '%s try to call the api cluster/uuid but the uuid \'%s\' is a host, %d errors with this ip' % (remote_ip, uuid, nb_error_ip + 1)
        to_logger = add_error(error, remote_ip)
        if to_logger is True:
            logger.error(error)
        return app.abort(404, 'can\'t find cluster with this uuid', to_logger=to_logger)
    
    logger.debug('Successfully found the cluster with uuid \'%s\'' % uuid)
    d = {}
    # All we need is here
    e = get_summary(elt)
    
    d['status'] = e['status']
    d['context'] = e['summary']
    d['cluster_uuid'] = elt.uuid
    d['cluster_display_name'] = elt.display_name
    d['cluster_name'] = elt.host_name
    d['output'] = elt.output
    d['long_output'] = elt.long_output
    d['business_impact'] = elt.business_impact
    d['status_since'] = elt.last_state_change
    d['raw_perf_data'] = elt.perf_data
    d['perf_data'] = PerfDatas(elt.perf_data).as_list()
    d['confirmed_state'] = elt.state_type
    d['notes_url'] = elt.notes_url
    d['notes_multi_url'] = elt.notes_multi_url
    d['status_confirmed_since'] = ''
    
    if elt.state_type == 'HARD':
        d['status_confirmed_since'] = elt.last_hard_state_change
    end_time = time.time()
    add_response_time(start_time, end_time - start_time)
    
    return json.dumps(d)


def get_element_host(uuid):
    start_time = time.time()
    # First we need to check if the token is good
    remote_ip = app.request.remote_addr
    token = app.request.GET.get('token', '')
    if remote_ip in app.errors_per_ip.keys():
        nb_error_ip = app.errors_per_ip[remote_ip]['nb_errors']
    else:
        nb_error_ip = 0
    if token != app.token:
        error = '%s try to call the api host/uuid with the \'%s\' token (wrong token), %d errors with this ip' % (remote_ip, token, nb_error_ip + 1)
        to_logger = add_error(error, remote_ip)
        if to_logger is True:
            logger.error(error)
        return app.abort(401, 'wrong token', True, to_logger)
    elt = app.datamgr.get_host_by_uuid(uuid)
    
    # If the element is unknown, we need to tell it
    if elt is None:
        error = '%s try to call the api host/uuid but the uuid \'%s\' doesn\'t return anything, %d errors with this ip' % (remote_ip, uuid, nb_error_ip + 1)
        to_logger = add_error(error, remote_ip)
        if to_logger is True:
            logger.error(error)
        return app.abort(404, 'can\'t find host with this uuid', to_logger=to_logger)
    
    # If business rule, it's a cluster
    if elt.got_business_rule:
        error = '%s try to call the api host/uuid but the uuid \'%s\' is a cluster, %d errors with this ip' % (remote_ip, uuid, nb_error_ip + 1)
        to_logger = add_error(error, remote_ip)
        if to_logger is True:
            logger.error(error)
        return app.abort(404, 'can\'t find host with this uuid', to_logger=to_logger)
    
    logger.debug('Successfully found the host with uuid \'%s\'' % uuid)
    
    d = {}
    # All we need is here
    e = get_summary(elt)
    
    d['status'] = e['status']
    d['context'] = e['summary']
    d['host_uuid'] = elt.uuid
    d['host_display_name'] = elt.display_name
    d['host_name'] = elt.host_name
    d['output'] = elt.output
    d['long_output'] = elt.long_output
    d['business_impact'] = elt.business_impact
    d['status_since'] = elt.last_state_change
    d['raw_perf_data'] = elt.perf_data
    d['perf_data'] = PerfDatas(elt.perf_data).as_list()
    d['confirmed_state'] = elt.state_type
    d['notes_url'] = elt.notes_url
    d['notes_multi_url'] = elt.notes_multi_url
    d['status_confirmed_since'] = ''
    
    if elt.state_type == 'HARD':
        d['status_confirmed_since'] = elt.last_hard_state_change
    end_time = time.time()
    add_response_time(start_time, end_time - start_time)
    return json.dumps(d)


def get_element_check_on_cluster(uuid):
    # First we need to check if the token is good
    start_time = time.time()
    remote_ip = app.request.remote_addr
    token = app.request.GET.get('token', '')
    if remote_ip in app.errors_per_ip.keys():
        nb_error_ip = app.errors_per_ip[remote_ip]['nb_errors']
    else:
        nb_error_ip = 0
    if token != app.token:
        error = '%s try to call the api check-on-cluster with the \'%s\'token (wrong token), %d errors with this ip' % (remote_ip, token, nb_error_ip + 1)
        to_logger = add_error(error, remote_ip)
        if to_logger is True:
            logger.error(error)
        return app.abort(401, 'wrong token', True, to_logger)
    
    if '-' not in uuid:
        error = '%s try to call the api check-on-cluster but the uuid \'%s\' doesn\'t return anything, %d errors with this ip' % (remote_ip, uuid, nb_error_ip + 1)
        to_logger = add_error(error, remote_ip)
        if to_logger is True:
            logger.error(error)
        return app.abort(404, 'can\'t find check on cluster with this uuid', to_logger=to_logger)
    
    # Second we need to check if the uuid is a right check
    if not app.is_check_uuid(uuid):
        error = '%s try to call the api check-on-cluster but the uuid \'%s\' is not a check-on-cluster, %d errors with this ip' % (remote_ip, uuid, nb_error_ip + 1)
        to_logger = add_error(error, remote_ip)
        if to_logger is True:
            logger.error(error)
        return app.abort(404, 'can\'t find check on cluster with this uuid', to_logger=to_logger)
    
    # If the element is unknown, we need to tell it
    elt = app.datamgr.get_service_by_uuid(uuid)
    if elt is None:
        error = '%s try to call the api check-on-cluster but the uuid \'%s\' doesn\'t return anything, %d errors with this ip' % (remote_ip, uuid, nb_error_ip + 1)
        to_logger = add_error(error, remote_ip)
        if to_logger is True:
            logger.error(error)
        return app.abort(404, 'can\'t find check on cluster with this uuid', to_logger=to_logger)
    
    # If no business rule, the check is not on a cluster
    if not elt.host.got_business_rule:
        error = '%s try to call the api check-on-cluster but the uuid \'%s\' is a check-on-host, %d errors with this ip' % (remote_ip, uuid, nb_error_ip + 1)
        to_logger = add_error(error, remote_ip)
        if to_logger is True:
            logger.error(error)
        return app.abort(404, 'can\'t find check on cluster with this uuid', to_logger=to_logger)
    
    logger.debug('Successfully found the check-on-cluster with uuid \'%s\'' % uuid)
    
    d = {}
    
    # All we need is here
    e = get_summary(elt)
    
    d['status'] = e['status']
    d['context'] = e['summary']
    d['status_since'] = elt.last_state_change
    d['output'] = elt.output
    d['long_output'] = elt.long_output
    d['check_name'] = elt.service_description
    d['check_display_name'] = elt.display_name
    d['business_impact'] = elt.business_impact
    d['check_uuid'] = elt.uuid
    d['confirmed_state'] = elt.state_type
    d['notes_url'] = elt.notes_url
    d['notes_multi_url'] = elt.notes_multi_url
    
    # Now we need information on the cluster
    host = elt.host
    d['cluster_name'] = host.host_name
    d['cluster_uuid'] = host.uuid
    d['cluster_display_name'] = host.display_name
    d['raw_perf_data'] = elt.perf_data
    d['perf_data'] = PerfDatas(elt.perf_data).as_list()
    d['status_confirmed_since'] = ''
    
    if elt.state_type == 'HARD':
        d['status_confirmed_since'] = elt.last_hard_state_change
    end_time = time.time()
    add_response_time(start_time, end_time - start_time)
    return json.dumps(d)


def get_element_check_on_host(uuid):
    start_time = time.time()
    remote_ip = app.request.remote_addr
    # First we need to check if the token is good
    token = app.request.GET.get('token', '')
    if remote_ip in app.errors_per_ip.keys():
        nb_error_ip = app.errors_per_ip[remote_ip]['nb_errors']
    else:
        nb_error_ip = 0
    if token != app.token:
        error = '%s try to call the api check-on-host with the \'%s\' token (wrong token), %d errors with this ip' % (remote_ip, token, nb_error_ip + 1)
        to_logger = add_error(error, remote_ip)
        if to_logger is True:
            logger.error(error)
        return app.abort(401, 'wrong token', True, to_logger)
    
    if '-' not in uuid:
        error = '%s try to call the api check-on-cluster but the uuid \'%s\' doesn\'t return anything, %d errors with this ip' % (remote_ip, uuid, nb_error_ip + 1)
        to_logger = add_error(error, remote_ip)
        if to_logger is True:
            logger.error(error)
        return app.abort(404, 'can\'t find check on host with this uuid', to_logger=to_logger)
    
    # Second we need to check if the uuid is a right check
    if not app.is_check_uuid(uuid):
        error = '%s try to call the api check-on-host but the uuid \'%s\' is not a check-on-host, %d errors with this ip' % (remote_ip, uuid, nb_error_ip + 1)
        to_logger = add_error(error, remote_ip)
        if to_logger is True:
            logger.error(error)
        return app.abort(404, 'can\'t find check on host with this uuid', to_logger=to_logger)
    
    # If the element is unknown, we need to tell it
    elt = app.datamgr.get_service_by_uuid(uuid)
    if elt is None:
        error = '%s try to call the api check-on-host but the uuid \'%s\' doesn\'t return anything, %d errors with this ip' % (remote_ip, uuid, nb_error_ip + 1)
        to_logger = add_error(error, remote_ip)
        if to_logger is True:
            logger.error(error)
        return app.abort(404, 'can\'t find check on host with this uuid', to_logger=to_logger)
    
    # If business_rule, the check is not on a host
    if elt.host.got_business_rule:
        error = '%s try to call the api check-on-host but the uuid \'%s\' is a check-on-cluster, %d errors with this ip' % (remote_ip, uuid, nb_error_ip + 1)
        to_logger = add_error(error, remote_ip)
        if to_logger is True:
            logger.error(error)
        return app.abort(404, 'can\'t find check on host with this uuid', to_logger=to_logger)
    
    logger.debug('Successfully found the check-on-host with uuid \'%s\'' % uuid)
    
    d = {}
    
    # All we need is here
    e = get_summary(elt)
    
    d['status'] = e['status']
    d['context'] = e['summary']
    d['status_since'] = elt.last_state_change
    d['output'] = elt.output
    d['long_output'] = elt.long_output
    d['check_name'] = elt.service_description
    d['check_display_name'] = elt.display_name
    d['business_impact'] = elt.business_impact
    d['check_uuid'] = elt.uuid
    d['confirmed_state'] = elt.state_type
    d['notes_url'] = elt.notes_url
    d['notes_multi_url'] = elt.notes_multi_url
    
    # Now we need information on the cluster
    host = elt.host
    d['host_name'] = host.host_name
    d['host_uuid'] = host.uuid
    d['host_display_name'] = host.display_name
    d['raw_perf_data'] = elt.perf_data
    d['perf_data'] = PerfDatas(elt.perf_data).as_list()
    d['status_confirmed_since'] = ''
    
    if elt.state_type == 'HARD':
        d['status_confirmed_since'] = elt.last_hard_state_change
    end_time = time.time()
    add_response_time(start_time, end_time - start_time)
    return json.dumps(d)


def ping():
    d = {'response': 'pong'}
    return json.dumps(d)


def missing_uuid():
    remote_ip = app.request.remote_addr
    if remote_ip in app.errors_per_ip.keys():
        nb_error_ip = app.errors_per_ip[remote_ip]['nb_errors']
    else:
        nb_error_ip = 0
    error = '%s try to call the api without referencing an UUID, %d errors with this ip' % (remote_ip, nb_error_ip + 1)
    to_logger = add_error(error, remote_ip)
    if to_logger is True:
        logger.error(error)
    return app.abort(404, 'uuid missing in the url', to_logger=to_logger)


pages = {
    ping                        : {'routes': ['/api/v1/ping'], 'view': None, 'wrappers': ['json']},
    get_all_monitored_elements  : {'routes': ['/api/v1/all-monitored-elements'], 'view': None, 'wrappers': ['json']},
    get_element_host            : {'routes': ['/api/v1/host/:uuid'], 'view': None, 'wrappers': ['json']},
    get_all_hosts               : {'routes': ['/api/v1/host'], 'view': None, 'wrappers': ['json']},
    get_element_check_on_host   : {'routes': ['/api/v1/check-on-host/:uuid'], 'view': None, 'wrappers': ['json']},
    get_element_cluster         : {'routes': ['/api/v1/cluster/:uuid'], 'view': None, 'wrappers': ['json']},
    get_all_clusters            : {'routes': ['/api/v1/cluster'], 'view': None, 'wrappers': ['json']},
    get_element_check_on_cluster: {'routes': ['/api/v1/check-on-cluster/:uuid'], 'view': None, 'wrappers': ['json']},
    missing_uuid                : {'routes': ['/api/v1/host/', '/api/v1/check-on-host/', '/api/v1/cluster/', '/api/v1/check-on-cluster/'], 'view': None, 'wrappers': ['json']},
}
