# !/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) shinken-solutions team:
#
# This module is a sample for third party developers to bootstrap
# their own broker module.
#
# This module is a worker based one. It will allow to have N workers that will
# automatically manage 1/N oh the hosts of your realm(s).


import json
import sys

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_module.brokermoduleworker import BrokerModuleWorker as BrokerExportWorker
from shinken.util import safe_add_to_dict
from shinken.webui import bottlewebui as bottle
from shinken.webui.bottlewebui import response
from shinken.webui.cherrypybackend import CherryPyServerHTTP
from shinkensolutions.date_helper import parse_string_to_date, date_now, from_to_date_generator, format_date_to_string, Date

sys.path.append('/var/lib/shinken/modules')
from sla.component.sla_component_manager import ComponentManager  # noqa : we need to update sys.path before import other module
from sla.component.sla_common import shared_data  # noqa : we need to update sys.path before import other module
from sla.component.sla_info import SLAInfo  # noqa : we need to update sys.path before import other module
from sla.component.sla_database_connection import SLADatabaseConnection  # noqa : we need to update sys.path before import other module
from sla.component.sla_compute_percent_sla import ComputePercentSla  # noqa : we need to update sys.path before import other module
from sla.component.sla_database import SLADatabase  # noqa : we need to update sys.path before import other module
from sla.component.sla_archive import SLAArchive  # noqa : we need to update sys.path before import other module

if TYPE_CHECKING:
    from shinken.objects.module import Module
    from shinken.misc.type_hint import Any, Dict, Optional
    from shinken.webui.bottlewebui import Bottle


class BrokerModuleSlaApiWorker(BrokerExportWorker):
    module_configuration = None
    port = None
    host = None
    use_ssl = None
    ssl_key = None
    ssl_cert = None
    srv = None
    logger_init = None
    component_manager = None
    sla_info = None
    sla_database_connection = None
    compute_percent_sla = None
    sla_database = None
    sla_archive = None
    
    
    def init_worker(self, module_configuration, **kwargs):
        # type: (Module, Optional[Dict[Any,Any]]) -> None
        self.module_configuration = module_configuration
        
        self.port = int(getattr(module_configuration, u'port', u'7790'))
        self.host = getattr(module_configuration, u'host', u'0.0.0.0')
        self.use_ssl = getattr(module_configuration, u'use_ssl', u'0') == u'1'
        self.ssl_key = getattr(module_configuration, u'ssl_key', u'')
        self.ssl_cert = getattr(module_configuration, u'ssl_cert', u'')
        shared_data.set_default_values(getattr(module_configuration, u'configuration_default_value', {}))
        
        error_handler = None
        self.component_manager = ComponentManager(self.logger)
        self.sla_database_connection = SLADatabaseConnection(self.module_configuration, self.component_manager, log_database_parameters=True)
        self.compute_percent_sla = ComputePercentSla(self.module_configuration, self.component_manager)
        self.sla_info = SLAInfo(self.module_configuration, self.component_manager, error_handler, self.sla_database_connection)
        self.sla_database = SLADatabase(self.module_configuration, self.component_manager, self.sla_database_connection, self.sla_info)
        self.sla_archive = SLAArchive(self.module_configuration, self.component_manager, self.sla_info, self.compute_percent_sla, self.sla_database, father_name=u'webui')
        
        self.srv = None
        self.logger_init = self.logger.get_sub_part(u'INITIALISATION')
        
        self.init()
    
    
    def worker_main(self, **kwargs):
        # type: (Optional[Dict[Any,Any]]) -> None
        self._http_start()
    
    
    def init(self):
        # type: () -> None
        self.logger_init.info(u'Initialization of the module')
        
        self.component_manager.init()
        self._init_http_server()
    
    
    def _init_http_server(self):
        # type: () -> None
        self.logger_init.info(u'Init http server')
        self.logger_init.info(u'   port       : [%s]' % self.port)
        self.logger_init.info(u'   host       : [%s]' % self.host)
        self.logger_init.info(u'   use_ssl    : [%s]' % self.use_ssl)
        self.logger_init.info(u'   ssl_key    : [%s]' % self.ssl_key)
        self.logger_init.info(u'   ssl_cert   : [%s]' % self.ssl_cert)
        try:
            # instantiate a new Bottle object, don't use the default one otherwise all module will share the same
            app = bottle.Bottle()
            app = self._init_routes(app)
            self.srv = app.run(host=self.host, port=self.port, server=CherryPyServerHTTP, use_ssl=self.use_ssl, ssl_key=self.ssl_key, ssl_cert=self.ssl_cert)
        except Exception, e:
            self.logger_init.error(u'Fail to start http server with Exception : %s' % str(e))
            raise
        self.logger_init.info(u'Server loaded')
    
    
    def do_stop(self):
        # type: () -> None
        self.stop_listener()
    
    
    def stop_listener(self):
        # type: () -> None
        self.srv.stop()
        self.logger.info(u'Stop http server')
    
    
    def _http_start(self):
        # type: () -> None
        # Now block and run
        self.logger.info(u'Starting http server')
        self.srv.start()
    
    
    @staticmethod
    def _format_sla_archive(sla_archive):
        # type: (Dict[unicode, Any]) -> Dict[unicode, Any]
        host_uuid = sla_archive[u'uuid']
        host_uuid = host_uuid.split(u'-')[0] if u'-' in host_uuid else host_uuid
        sla_date = Date(sla_archive[u'yday'], sla_archive[u'year'])
        if host_uuid is None:
            return {
                u'sla_total': u'N.A.',
                u'sla_ok'   : u'N.A.'
            }
        return {
            u'sla_total'     : sla_archive.get(u'sla_total', 0),
            u'sla_ok'        : sla_archive.get(u'sla_ok', 0),
            u'sla_crit'      : sla_archive.get(u'sla_crit', 0),
            u'sla_warn'      : sla_archive.get(u'sla_warn', 0),
            u'sla_unknown'   : sla_archive.get(u'sla_unknown', 0),
            u'sla_missing'   : sla_archive.get(u'sla_missing', 0),
            u'sla_inactive'  : sla_archive.get(u'sla_inactive', 0),
            u'sla_thresholds': sla_archive.get(u'thresholds', shared_data.get_default_sla_thresholds()),
            u'sla_date'      : format_date_to_string(sla_date)
        }
    
    
    @staticmethod
    def _parse_request():
        host_names = bottle.request.POST.get(u'father_name', u'')
        host_uuids = bottle.request.POST.get(u'father_uuid', u'')
        check_name = bottle.request.POST.get(u'check_name', u'')
        raw_start_date = bottle.request.POST.get(u'date', u'')
        raw_end_date = bottle.request.POST.get(u'until', u'')
        
        if not host_names and not host_uuids:
            bottle.abort(400, u'no father_name and no father_uuid')
        
        if raw_start_date:
            try:
                start_date = parse_string_to_date(raw_start_date, str_format=u'%d_%m_%Y')
            except ValueError:
                return bottle.abort(400, u'date wrong format')
        else:
            start_date = date_now()
        
        if raw_end_date:
            try:
                end_date = parse_string_to_date(raw_end_date, str_format=u'%d_%m_%Y')
            except ValueError:
                return bottle.abort(400, u'until wrong format')
        else:
            end_date = start_date
        
        host_names = host_names.split(u'^^') if host_names else []
        host_uuids = host_uuids.split(u'^^') if host_uuids else []
        return check_name, end_date, host_names, host_uuids, start_date
    
    
    def _search_items_uuid(self, check_name, host_names, host_uuids):
        elements_not_found = []
        host_uuids = set(host_uuids)
        host_names = set(host_names)
        expected_uuids = list(set(host_uuids))[:]
        
        for host_uuid in expected_uuids:
            host_name = self.sla_info.get_name(host_uuid)[0]
            if not host_name:
                elements_not_found.append({
                    u'father_uuid': host_uuid,
                })
                host_uuids.remove(host_uuid)
            elif host_name in host_names:
                host_names.remove(host_name)
        item_uuids = set(host_uuids)
        
        if check_name:
            for host_uuid in host_uuids.copy():
                host_name = self.sla_info.get_name(host_uuid)[0]
                check_uuid = self.sla_info.get_uuid(host_name, check_name)
                if check_uuid:
                    item_uuids.add(check_uuid)
                else:
                    element_not_found = {
                        u'father_uuid': host_uuid,
                        u'check_name' : check_name,
                    }
                    if host_name:
                        element_not_found[u'father_name'] = host_name
                    elements_not_found.append(element_not_found)
                item_uuids.remove(host_uuid)
            
            for host_name in host_names.copy():
                check_uuid = self.sla_info.get_uuid(host_name, check_name)
                host_uuid = self.sla_info.get_uuid(host_name, u'')
                if check_uuid:
                    item_uuids.add(check_uuid)
                else:
                    element_not_found = {
                        u'father_name': host_name,
                        u'check_name' : check_name,
                    }
                    if host_uuid:
                        element_not_found[u'father_uuid'] = host_uuid
                    elements_not_found.append(element_not_found)
                    item_uuids.remove(host_uuid)
        else:
            for host_name in host_names:
                host_uuid = self.sla_info.get_uuid(host_name, u'')
                if host_uuid:
                    item_uuids.add(host_uuid)
                else:
                    elements_not_found.append({
                        u'father_name': host_name,
                    })
        nb_elements_expected = len(host_names) + len(expected_uuids)
        nb_elements_found = nb_elements_expected - len(elements_not_found)
        return elements_not_found, item_uuids, nb_elements_expected, nb_elements_found
    
    
    def _search_sla(self, end_date, item_uuids, start_date):
        where = {u'uuid': {u'$in': list(item_uuids)}}
        archives = self.sla_database.find_archives('', (start_date, end_date), where=where)
        by_uuid_archives = {}
        for archive in archives:
            safe_add_to_dict(by_uuid_archives, archive[u'uuid'], archive)
        
        elements_found = []
        for item_uuid in item_uuids:
            item_archives = by_uuid_archives.get(item_uuid, [])
            by_date_archives = dict(((Date(e[u'yday'], e[u'year']), e) for e in item_archives))
            
            host_name, check_name = self.sla_info.get_name(item_uuid)
            if u'-' in item_uuid:
                return_value_by_item = {
                    u'father_name': host_name,
                    u'check_name' : check_name,
                    u'father_uuid': item_uuid.split(u'-')[0],
                    u'check_uuid' : item_uuid,
                    u'sla'        : [],
                }
            else:
                return_value_by_item = {
                    u'father_name': host_name,
                    u'father_uuid': item_uuid,
                    u'sla'        : [],
                }
            
            for date in from_to_date_generator(start_date, end_date):
                sla_archive = by_date_archives.get(date, None)
                if not sla_archive:
                    sla_archive = self.sla_archive.build_archive_for_missing_day(date, item_uuid, sla_thresholds=shared_data.get_default_sla_thresholds())
                return_value_by_item[u'sla'].append(self._format_sla_archive(sla_archive))
            elements_found.append(return_value_by_item)
        return elements_found
    
    
    def _init_routes(self, app):
        # type: (Bottle) -> Bottle
        @app.error(400)
        def custom400(error):
            return json.dumps(u'ERROR 400 : %s' % error.output)
        
        
        def sla_api():
            # type: () -> unicode
            response.content_type = u'application/json'
            
            check_name, end_date, host_names, host_uuids, start_date = self._parse_request()
            elements_not_found, item_uuids, nb_elements_expected, nb_elements_found = self._search_items_uuid(check_name, host_names, host_uuids)
            elements_found = self._search_sla(end_date, item_uuids, start_date)
            
            return_value = {
                u'nb_elements_found'   : nb_elements_found,
                u'nb_elements_expected': nb_elements_expected,
                u'elements_found'      : elements_found,
                u'elements_NOT_found'  : elements_not_found,
            }
            return json.dumps(return_value)
        
        
        app.route(u'/shinken/v1/sla_api', callback=sla_api, method=u'ANY')
        return app
