#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.

import threading
import time
from datetime import datetime

from ec_common import shared_data, STATE, ACK, DOWNTIME, STATE_TYPE, ITEM_TYPE
from ec_database_connection import ECDatabaseConnection, ECBulk
from ec_event import Event
from shinken.acknowledge import Acknowledge
from shinken.check import NO_END_VALIDITY
from shinken.downtime import Downtime
from shinken.log import PART_INITIALISATION
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_module.brokermoduleworker import BrokerModuleWorker
from shinken.objects.itemsummary import HostSummary, CheckSummary
from shinken.thread_helper import LockWithTimer
from shinkensolutions.date_helper import get_datetime_with_local_time_zone, get_now, timestamp_from_datetime
from shinkensolutions.lib_modules.configuration_reader import read_int_in_configuration, read_string_in_configuration
from shinkensolutions.shinken_time_helper import print_human_readable_period, print_human_readable_date_time

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Union, List, Optional
    from shinken.log import PartLogger

LOOP_SPEED = 1  # in sec
EVENT_BULK_SPEED = 1  # in sec
CHECK_MISSING_DATA_SPEED = 10  # in sec

SHINKEN_INACTIVE_OUTPUT = {
    u'en': u'''The status is set to UNKNOWN because Broker was not running for %s.''',
    u'fr': u'''Le statut est positionné à UNKNOWN car le Broker était éteint pendant %s.''',  # noqa : string in french
}
MISSING_DATA_OUTPUT = {
    u'en': u'''The status is set to UNKNOWN because last check by Shinken Enterprise on this element should have been done at %s ( waiting time has exceeded tolerance threshold of %s ).''',
    u'fr': u'''Le statut est positionné à UNKNOWN car la dernière vérification faite par Shinken Enterprise sur cet élément aurait dû avoir lieu à %s  ( l'attente a dépassé le seuil de tolérance de %s ).''',  # noqa : string in french
}

_cache_write_downtime = set()
_cache_write_acknowledge = set()


class StateCache(object):
    comprehensive_state = 0
    state_type = 0
    event_uuid = u''
    missing_data_activation_time = None  # type: datetime
    last_state_update_time = None  # type: datetime
    state_validity_duration = 0
    state_ending_time = None  # type: datetime
    
    
    def __str__(self):
        return u'comprehensive_state:[%s] state_type:[%s] event_uuid:[%s] missing_data_activation_time:[%s] last_state_update_time:[%s] state_validity_duration:[%s] state_ending_time:[%s]' % (
            self.comprehensive_state,
            self.state_type,
            self.event_uuid,
            self.missing_data_activation_time,
            self.last_state_update_time,
            self.state_validity_duration,
            self.state_ending_time
        )
    
    
    def __init__(self, comprehensive_state, state_type, missing_data_activation_time, event_uuid, last_state_update_time, state_validity_duration, state_ending_time):
        # type: (int, int, Optional[datetime], unicode, Optional[datetime], Optional[int], Optional[datetime]) -> None
        self.comprehensive_state = comprehensive_state
        self.state_type = state_type
        self.event_uuid = event_uuid
        self.missing_data_activation_time = missing_data_activation_time
        self.last_state_update_time = last_state_update_time
        self.state_validity_duration = state_validity_duration
        self.state_ending_time = state_ending_time
    
    
    def update_missing_data_info(self, missing_data_activation_time, last_state_update_time, state_validity_duration, state_ending_time):
        # type: (datetime, datetime, int, datetime) -> None
        self.missing_data_activation_time = missing_data_activation_time
        self.last_state_update_time = last_state_update_time
        self.state_validity_duration = state_validity_duration
        self.state_ending_time = state_ending_time
    
    
    def update_state_type(self, state_type):
        # type: (int) -> None
        self.state_type = state_type
    
    
    @classmethod
    def from_database_entry(cls, _to_load):
        # type: (Dict) -> StateCache
        return StateCache(
            comprehensive_state=_to_load[u'comprehensive_state'],
            state_type=_to_load[u'state_type'],
            event_uuid=_to_load[u'event_uuid'],
            missing_data_activation_time=None,
            last_state_update_time=None,
            state_validity_duration=None,
            state_ending_time=None
        )
    
    
    def to_database_entry(self):
        # type: () -> Dict
        return {
            u'comprehensive_state': self.comprehensive_state,
            u'state_type'         : self.state_type,
            u'event_uuid'         : self.event_uuid,
        }


class BrokHandlerModuleWorker(BrokerModuleWorker):
    database_connection = None  # type: ECDatabaseConnection
    _state_cache = {}  # type: Dict[str, StateCache]
    _event_bulk = None  # type: ECBulk
    _state_cache_update = set()
    _see_this_boot = set()
    _state_cache_lock = threading.RLock()
    _last_check_missing_data = None  # type: Optional[int]
    _next_check_missing_data = None  # type: Optional[float]
    _lock_check_missing_data = None
    _last_event_bulk_execute = -1
    _write_event_cumulative = (0.0, 0)
    _missing_data_output_template = u''
    _inactive_output_template = u''
    logger_compute_missing_data = None  # type: PartLogger
    _lang = u''
    minimal_time_before_an_element_become_missing_data_at_startup = NO_END_VALIDITY
    worker_start_time = None  # type: Optional[float]
    
    
    def init_worker(self, configuration):
        global CHECK_MISSING_DATA_SPEED
        if self.logger.is_debug():
            self._state_cache_lock = LockWithTimer(self._state_cache_lock, my_logger=self.logger, lock_name='state_cache_lock')
        
        # We do not use stats from our parent
        self.compute_stats = False
        self.worker_start_time = time.time()
        CHECK_MISSING_DATA_SPEED = read_int_in_configuration(configuration, u'check_missing_data_speed', CHECK_MISSING_DATA_SPEED)
        lang = read_string_in_configuration(configuration, u'lang', u'en')
        self._lang = lang
        self.minimal_time_before_an_element_become_missing_data_at_startup = read_int_in_configuration(configuration, u'minimal_time_before_an_element_become_missing_data_at_startup', NO_END_VALIDITY)
        self._missing_data_output_template = MISSING_DATA_OUTPUT.get(lang, MISSING_DATA_OUTPUT[u'en'])
        self._inactive_output_template = SHINKEN_INACTIVE_OUTPUT.get(lang, SHINKEN_INACTIVE_OUTPUT[u'en'])
        
        self.database_connection = ECDatabaseConnection(configuration, self.logger)
        self.database_connection.init(u'%s : WORKER %s' % (self.get_name(), self._worker_id))
        
        self.logger_compute_missing_data = self.logger.get_sub_part(u'STATE-FRESHNESS', len(u'STATE-FRESHNESS'))
        logger_init = self.logger.get_sub_part(PART_INITIALISATION, len(PART_INITIALISATION))
        
        if self._load_state_cache():
            self._next_check_missing_data = self.worker_start_time + self.minimal_time_before_an_element_become_missing_data_at_startup
            self.logger_compute_missing_data.debug(u'requesting missing data check at [%s]' % self.logger.format_time(self._next_check_missing_data))
        else:
            self._next_check_missing_data = None
        self._last_check_missing_data = None
        self._lock_check_missing_data = threading.RLock()
        self._event_bulk = self.database_connection.event_bulk_factory()
        self._write_event_cumulative = (0.0, 0)
        
        logger_init.debug(u'worker start at %s with lang=[%s]' % (self.logger.format_time(self.worker_start_time), lang))
        
        logger_init.info(u'Parameters loaded')
        logger_init.info(u'  - check_missing_data_speed -------------------------------------- : [%s]' % CHECK_MISSING_DATA_SPEED)
        logger_init.info(u'  - lang ---------------------------------------------------------- : [%s]' % self._lang)
        logger_init.info(u'  - minimal_time_before_an_element_become_missing_data_at_startup - : [%s]' % self.minimal_time_before_an_element_become_missing_data_at_startup)
    
    
    def on_init(self):
        pass
    
    
    def manage_service_check_result_brok(self, new_info):
        self._handle_brok(new_info)
    
    
    def manage_host_check_result_brok(self, new_info):
        self._handle_brok(new_info)
    
    
    def manage_update_service_status_brok(self, new_info):
        self._handle_brok(new_info)
    
    
    def manage_update_host_status_brok(self, new_info):
        self._handle_brok(new_info)
    
    
    def manage_initial_service_status_brok(self, new_info):
        self._handle_brok(new_info)
    
    
    def manage_initial_host_status_brok(self, new_info):
        self._handle_brok(new_info)
    
    
    def worker_main(self):
        while not self.interrupted:
            self._tick()
            self.interruptable_sleep(LOOP_SPEED)
    
    
    def get_raw_stats(self):
        data = super(BrokHandlerModuleWorker, self).get_raw_stats()
        data['write_event_cumulative'] = self._write_event_cumulative
        return data
    
    
    def _do_in_worker_manage_broks_thread_loop(self):
        super(BrokHandlerModuleWorker, self)._do_in_worker_manage_broks_thread_loop()
        now = get_now()
        self._lock_check_missing_data.acquire()
        if self._next_check_missing_data and self._next_check_missing_data <= now:
            self._next_check_missing_data = None
            # self.logger_compute_missing_data.debug(u'checking missing data')
            self._lock_check_missing_data.release()
            
            next_check = self._check_missing_data()
            self._last_check_missing_data = now
            sooner_next_loop_time = now + CHECK_MISSING_DATA_SPEED
            if next_check:
                next_check = timestamp_from_datetime(next_check)
                if sooner_next_loop_time > next_check:
                    next_check = sooner_next_loop_time
            
            self._lock_check_missing_data.acquire()
            if not self._next_check_missing_data or (next_check and next_check < self._next_check_missing_data):
                self._next_check_missing_data = next_check
            elif self._next_check_missing_data < sooner_next_loop_time:
                self._next_check_missing_data = sooner_next_loop_time
            # self.logger_compute_missing_data.log_perf(now, u'', u'check done', min_time=0, info_time=0.4)
            # self.logger_compute_missing_data.debug(u'next missing data check in %ss' % ((next_check - now) if next_check else None))
        self._lock_check_missing_data.release()
    
    
    @staticmethod
    def _is_state_expired(expiration_time):
        if not expiration_time or expiration_time == NO_END_VALIDITY:
            return False
        return expiration_time < get_datetime_with_local_time_zone()


    # TODO: MISSING DATA start time mismatch between regenerator and Event/SLA modules, cf. SEF-9992

    def _check_missing_data(self):
        # type: () -> Optional[datetime]
        next_missing_data = None
        now = get_datetime_with_local_time_zone(time.time())
        for inventory_item in self.get_all_inventory():
            cache_value = self._state_cache.get(inventory_item.get_uuid(), None)  # type: StateCache
            # self.logger_compute_missing_data.debug(u'[%s] check missing data of %s : last_state_update_time:[%s] expiration_time:[%s]' % (
            #     inventory_item.get_uuid(),
            #     inventory_item.get_instance_name(),
            #     self.logger.format_datetime(cache_value.last_state_update_time) if cache_value else u'no cache_value',
            #     self.logger.format_datetime(cache_value.expiration_time) if cache_value else u'no cache_value',
            #
            # ))
            if not cache_value:
                continue
            
            if cache_value.missing_data_activation_time and cache_value.missing_data_activation_time != NO_END_VALIDITY and (not next_missing_data or (next_missing_data > cache_value.missing_data_activation_time > now)):
                next_missing_data = cache_value.missing_data_activation_time
            
            if not self._is_state_expired(cache_value.missing_data_activation_time):
                continue
            
            if (cache_value.comprehensive_state % 10) == STATE.MISSING_DATA:
                continue
            # self.logger_compute_missing_data.debug(u'[%s] found new missing_data at [%s] from [%s] ' % (
            #     inventory_item.get_uuid(),
            #     self.logger.format_datetime(cache_value.missing_data_activation_time),
            #     self.logger.format_datetime(get_datetime_with_local_time_zone()),
            # ))
            
            missing_data_event = self._build_missing_data_event(inventory_item, cache_value)
            self._put_event(missing_data_event)
        return next_missing_data
    
    
    def _handle_brok(self, brok):
        brok_data = brok.data
        # self.logger.debug(u'MANAGE BROKS', u'[%s][%s-%s] new [%s] brok at [%s]' % (brok_data['instance_uuid'], brok_data['host_name'], brok_data.get('service_description', ''), brok.type, self.logger.format_time(brok_data['creation_date'])))
        event = Event.from_brok_data(brok)
        
        # self.logger.debug('_handle_brok type:[%s] creation:[%s] last_chk:[%s] end:[%s] missing:[%s] -> brok[%s]' % (
        #     brok.type,
        #     self.logger.format_time(brok_data['creation_date']),
        #     self.logger.format_time(brok_data['last_chk']),
        #     self.logger.format_time(brok_data['state_validity_end_time']),
        #     self.logger.format_time(brok_data['missing_data_activation_time']),
        #     brok_data))
        self._put_event(event)
        
        downtime = brok_data.get('in_scheduled_downtime', None)
        active_downtime_uuids = brok_data.get('active_downtime_uuids', None)
        if downtime:
            downtimes = brok_data.get('downtimes', None)
            self._create_downtime(active_downtime_uuids, downtimes)
        
        acknowledgement = brok_data.get('acknowledgement', None)
        # logger.debug('[%s][%s-%s] new [%s] brok [%s]/[%s]' % (
        #     brok_data['instance_uuid'],
        #     brok_data['host_name'],
        #     brok_data.get('service_description', ''),
        #     brok.type, brok_data.get('acknowledgement', None),
        #     brok_data.get('partial_acknowledge', None)))
        if acknowledgement:
            self._create_acknowledge(acknowledgement, event.item_uuid)
        
        partial_acknowledge = brok_data.get('partial_acknowledge', None)
        if partial_acknowledge:
            self._create_acknowledge(partial_acknowledge, event.item_uuid)
    
    
    def _create_downtime(self, uuid_downtimes, downtimes):
        # type: (List[basestring], List[Downtime]) -> None
        if not uuid_downtimes:
            return
        for uuid_downtime in uuid_downtimes[:]:
            if uuid_downtime in _cache_write_downtime:
                continue
            
            prev_downtime = self.database_connection.find_downtime(uuid_downtime)
            if prev_downtime:
                _cache_write_downtime.add(uuid_downtime)
                continue
            
            downtime = next((i for i in downtimes if i.uuid == uuid_downtime), None) if downtimes else None
            if not downtime:
                # Inherited downtime aren't in broks so we expect to have the downtime definition in the host brok
                continue
            
            downtime_entry = {
                '_id'    : uuid_downtime,
                'author' : downtime.author,
                'comment': downtime.comment,
            }
            for k in downtime.__class__.properties:
                downtime_entry[k] = getattr(downtime, k)
            _cache_write_downtime.add(uuid_downtime)
            self.database_connection.save_downtime(downtime_entry)
            # self.logger.debug('Saving new downtime: %s' % downtime_entry)
    
    
    def _create_acknowledge(self, acknowledge, item_uuid):
        # type: (Acknowledge, basestring) -> None
        acknowledge_id = acknowledge.id
        if acknowledge_id in _cache_write_acknowledge:
            return
        prev_acknowledge = self.database_connection.find_acknowledge(acknowledge_id)
        if prev_acknowledge is None:
            # ok there is not such acknowledge before, save it
            acknowledge_entry = {'_id': acknowledge_id, 'item_uuid': item_uuid}
            for k in acknowledge.__class__.properties:
                acknowledge_entry[k] = getattr(acknowledge, k)
            self.database_connection.save_acknowledge(acknowledge_entry)
            # self.logger.debug('Saving new acknowledge: %s' % acknowledge_entry)
        _cache_write_acknowledge.add(acknowledge.id)
    
    
    def _put_event(self, event):
        # type: (Event) -> None
        start_time = time.time()
        
        # Missing data thread fine tuning
        missing_data_activation_time = event.get_missing_data_activation_time()
        if event.from_brok and missing_data_activation_time and missing_data_activation_time != NO_END_VALIDITY:
            missing_data_activation_time = timestamp_from_datetime(missing_data_activation_time)
            change_missing_data_check_time = True
            if missing_data_activation_time < start_time:
                cache_value = self._state_cache.get(event.item_uuid, None)
                if cache_value and cache_value.comprehensive_state == STATE.MISSING_DATA:
                    change_missing_data_check_time = False
            if change_missing_data_check_time:
                with self._lock_check_missing_data:
                    sooner_next_missing_check_time = (self._last_check_missing_data + CHECK_MISSING_DATA_SPEED) if self._last_check_missing_data else start_time
                    missing_data_activation_time = missing_data_activation_time if missing_data_activation_time > sooner_next_missing_check_time else sooner_next_missing_check_time
                    if not self._next_check_missing_data or self._next_check_missing_data > missing_data_activation_time:
                        self._next_check_missing_data = missing_data_activation_time
                        # self.logger_compute_missing_data.debug(u'changing missing data check time to [%s]' % self.logger.format_time(missing_data_activation_time))
        
        if self._is_state_expired(event.get_missing_data_activation_time()):
            if event.from_brok:
                cache_value = self._state_cache.get(event.item_uuid, None)  # type: StateCache
                if cache_value and cache_value.comprehensive_state != STATE.MISSING_DATA and (cache_value.comprehensive_state == STATE.SHINKEN_INACTIVE or cache_value.last_state_update_time is None):
                    # At startup, after inactive event or with loaded cache from database, we do not have real expiration info.
                    # We update cache with first incoming brok
                    with self._state_cache_lock:
                        cache_value.missing_data_activation_time = event.get_missing_data_activation_time()
                        cache_value.state_validity_duration = event.get_state_validity_duration()
                        # self.logger.debug(u'[%s] put_event updated internal cache[%s] status with brok data missing_data_activation_time at [%s]' % (event.item_uuid, cache_value, self.logger.format_datetime(event.get_missing_data_activation_time())))
            # self.logger.debug(u'[%s] put_event reject event because the state[%s] has expired at [%s]' % (event.item_uuid, event.state_id, self.logger.format_datetime(event.get_missing_data_activation_time())))
            return
        
        cache_value = self._state_cache.get(event.item_uuid, None)  # type: StateCache
        event_comprehensive_state = event.get_comprehensive_state()
        have_state_in_cache = cache_value and cache_value.comprehensive_state == event_comprehensive_state
        if have_state_in_cache:
            # self.logger.debug(u'[%s] put_event event in cache comprehensive_state:[%s]' % (event.item_uuid, event_comprehensive_state))
            with self._state_cache_lock:
                if event.state_type != cache_value.state_type:
                    # self.logger.debug(u'[%s] must update state_type of event:[%s] to [%s]' % (event.item_uuid, cache_value.event_uuid, event.state_type))
                    self._event_bulk.update_event(cache_value.event_uuid, event.state_type, event.event_hard_since)
                    self._state_cache[event.item_uuid].update_state_type(event.state_type)
                    self._state_cache_update.add(event.item_uuid)
                    self._write_event_cumulative = (start_time, self._write_event_cumulative[1] + 1)
                
                cache_value.update_missing_data_info(event.get_missing_data_activation_time(), event.event_since, event.get_state_validity_duration(), event.get_state_ending_time())
            # self.logger.debug(u'[%s] put_event cache updated to :[%s]' % (event.item_uuid, cache_value))
        else:
            with self._state_cache_lock:
                if cache_value and cache_value.comprehensive_state == STATE.SHINKEN_INACTIVE:
                    # After restart, and inactive event, fake a new status from restart time
                    event_start = cache_value.state_ending_time if cache_value.state_ending_time and cache_value.state_ending_time != NO_END_VALIDITY else get_datetime_with_local_time_zone(self.worker_start_time)
                    event.update_event_since(event_start)
                # self.logger.debug(u'[%s] put_event event not in cache comprehensive_state:[%s] since:[%s] to missing_data_activation_time:[%s]' % (
                #     event.item_uuid,
                #     event_comprehensive_state,
                #     self.logger.format_datetime(event.event_since),
                #     self.logger.format_datetime(event.get_missing_data_activation_time())))
                self._event_bulk.insert_event(event)
                self._state_cache[event.item_uuid] = StateCache(
                    comprehensive_state=event_comprehensive_state,
                    state_type=event.state_type,
                    missing_data_activation_time=event.get_missing_data_activation_time(),
                    event_uuid=event.event_uuid,
                    last_state_update_time=event.event_since,
                    state_validity_duration=event.get_state_validity_duration(),
                    state_ending_time=event.get_state_ending_time()
                )
                self._state_cache_update.add(event.item_uuid)
                self._write_event_cumulative = (start_time, self._write_event_cumulative[1] + 1)
            # self.logger.debug(u'[%s] put_event created cache entry[%s] event:[%s]' % (event.item_uuid, self._state_cache[event.item_uuid], event.__dict__))
    
    
    def _tick(self):
        now = get_now()
        with self._state_cache_lock:
            if now - self._last_event_bulk_execute > EVENT_BULK_SPEED:
                self._event_bulk.bulks_execute()
                self._last_event_bulk_execute = now
            
            if self._state_cache_update:
                to_update = [{'_id': _id, 'state_cache': self._state_cache[_id].to_database_entry()} for _id in self._state_cache_update]
                self._state_cache_update = set()
                self.database_connection.update_state_cache(to_update)
    
    
    def _load_state_cache(self):
        # type: () -> bool
        have_missing_data = False
        worker_start_datetime = get_datetime_with_local_time_zone(self.worker_start_time)
        raw_state_caches = self.database_connection.load_state_cache()
        for raw_state_cache in raw_state_caches:
            self._state_cache[raw_state_cache['_id']] = StateCache.from_database_entry(raw_state_cache['state_cache'])
            if self._state_cache[raw_state_cache['_id']].comprehensive_state == STATE.SHINKEN_INACTIVE:
                self._state_cache[raw_state_cache['_id']].state_ending_time = worker_start_datetime
                if self.minimal_time_before_an_element_become_missing_data_at_startup > 0:
                    have_missing_data = True
                    self._state_cache[raw_state_cache['_id']].state_validity_duration = self.minimal_time_before_an_element_become_missing_data_at_startup
                    self._state_cache[raw_state_cache['_id']].missing_data_activation_time = get_datetime_with_local_time_zone(self.worker_start_time + self.minimal_time_before_an_element_become_missing_data_at_startup)
        return have_missing_data
    
    
    def add_shinken_inactive_event(self, item):
        item_uuid = item.get_uuid()
        if item_uuid in self._see_this_boot:
            return
        shinken_inactive_event = self._build_shinken_inactive_event(item)
        self._put_event(shinken_inactive_event)
        self._see_this_boot.add(item_uuid)
    
    
    def _build_shinken_inactive_event(self, item):
        # type: (Union[HostSummary, CheckSummary]) -> Event
        # self.logger.debug(u'_build_shinken_inactive_event host[%s,(%s)] cache:[%s]' % (str(item), dir(item), str(self._state_cache.get(item.get_uuid(), None))))
        
        now = self.worker_start_time
        event_since = get_datetime_with_local_time_zone(shared_data.get_last_start_time())
        inactive_duration = self.logger.format_duration(now - shared_data.get_last_start_time())
        shinken_inactive_event = BrokHandlerModuleWorker._build_artificial_event(item, event_since, STATE.SHINKEN_INACTIVE, self._inactive_output_template % inactive_duration)
        shinken_inactive_event.state_validity_end_time = now
        shinken_inactive_event.state_validity_end_datetime = get_datetime_with_local_time_zone(shinken_inactive_event.state_validity_end_time)
        
        if self.minimal_time_before_an_element_become_missing_data_at_startup > 0:
            # self.logger.debug(u'_build_shinken_inactive_event update expiration times to [%s]s' % self.minimal_time_before_an_element_become_missing_data_at_startup)
            shinken_inactive_event.missing_data_activation_time = now + self.minimal_time_before_an_element_become_missing_data_at_startup
            shinken_inactive_event.missing_data_activation_datetime = get_datetime_with_local_time_zone(shinken_inactive_event.missing_data_activation_time)
            shinken_inactive_event.state_validity_period = self.minimal_time_before_an_element_become_missing_data_at_startup
        return shinken_inactive_event
    
    
    def _build_missing_data_event(self, item, state_cache):
        # type: (Union[HostSummary, CheckSummary], StateCache) -> Event
        state_ending_time_string = print_human_readable_date_time(state_cache.state_ending_time, self._lang)
        threshold_expiration_state_string = print_human_readable_period(state_cache.state_validity_duration)
        missing_data_output = self._missing_data_output_template % (state_ending_time_string, threshold_expiration_state_string)
        missing_data_event = BrokHandlerModuleWorker._build_artificial_event(item, state_cache.state_ending_time if state_cache.state_ending_time else state_cache.missing_data_activation_time, STATE.MISSING_DATA, missing_data_output)
        return missing_data_event
    
    
    @staticmethod
    def _build_artificial_event(item, event_since, state_id, output):
        missing_data_event = Event(
            item_uuid=item.get_uuid(),
            event_since=event_since,
            event_hard_since=event_since,
            item_type=item.get_type(),
            state_id=state_id,
            state_type=STATE_TYPE.HARD,
            state_validity_period=NO_END_VALIDITY,
            flapping=False,
            acknowledged=ACK.NONE,
            downtime=DOWNTIME.NONE,
            partial_flapping=False,
            partial_ack=False,
            partial_dt=False,
            active_downtime_uuids=[],
            output=output,
            long_output='',
            realm=item.get_realm(),
            name=item.get_full_name(),
            host_name=item.get_host_name(),
            check_name=item.get_name() if item.get_type() == ITEM_TYPE.CHECK else '',
        )
        return missing_data_event
    
    
    def callback__a_new_host_added(self, host_uuid):
        # self.logger.debug(u'callback__a_new_host_added host:[%s] inactive_period:[%s]' % (host_uuid, shared_data.get_shinken_inactive_period()))
        if not shared_data.get_shinken_inactive_period():
            return
        
        # Unknown element don't have shinken inactive it must be there first time
        if not self.database_connection.is_in_state_cache(host_uuid):
            return
        
        host = self.get_host_from_uuid(host_uuid)
        self.add_shinken_inactive_event(host)
        for check in host.get_checks().itervalues():
            self.add_shinken_inactive_event(check)
