#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import itertools
import re
import socket

import simplejson as json
from shinken.basemodule import BaseModule, SOURCE_STATE
from shinken.synchronizer.dao.def_items import ITEM_TYPE

from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.util import make_unicode

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional
    from synchronizer.synchronizerdaemon import Synchronizer

import warnings

try:
    from collections import OrderedDict
except ImportError:
    from ordereddict import OrderedDict


# We do not want the warnings print to stderr to pop out in the CLI or the agent
# NOTe: the warnings.filterwarnings('ignore') is not working with some warnings
# like in debian9 SSL, so use another way
def _disable_warns(*args, **kwargs):
    pass


warnings.showwarning = _disable_warns

properties = {
    'daemons': ['synchronizer'],
    'type'   : 'ldap-import',
}


class _TYPE(object):
    HOST = 'host'
    HOSTGROUP = 'hostgroup'
    CONTACTGROUP = 'contactgroup'
    CONTACT = 'contact'


MANDATORY_CONFIGURATION_KEYS = ('url', 'ldap_protocol', 'base', 'username', 'password')
ALL_CONFIGURATION_KEYS = (
    'url',
    'ldap_protocol',
    'base',
    'username',
    'password',
    'hosts_base',
    'hosts_filter_with_group',
    'hostgroups_base',
    'contacts_base',
    'contacts_filter_with_group',
    'contactgroups_base',
    'hosts_filter',
    'hostgroups_filter',
    'contacts_filter',
    'contactgroups_filter'
)
DEFAULT_FILES = {
    'ad'      : {
        'mapping'   : '/etc/shinken/_default/sources/active-directory-source/active-directory-mapping.json',
        'rules'     : '/etc/shinken/_default/sources/active-directory-source/active-directory-rules.json',
        'connection': '/etc/shinken/_default/sources/active-directory-source/active-directory-connection.json'
    },
    'openldap': {
        'mapping'   : '/etc/shinken/_default/sources/openldap-source/openldap-mapping.json',
        'rules'     : '/etc/shinken/_default/sources/openldap-source/openldap-rules.json',
        'connection': '/etc/shinken/_default/sources/openldap-source/openldap-connection.json'
    }
}

DEFAULT_MODE = 'ad'

DEFAULT_POSSIBLE_KEYS = {
    ITEM_TYPE.HOSTS        : 'host_name, address',
    ITEM_TYPE.HOSTGROUPS   : 'hostgroup_name',
    ITEM_TYPE.CONTACTS     : 'contact_name, email',
    ITEM_TYPE.CONTACTGROUPS: 'contactgroup_name',
}

DEPRECATED_MAPPING_KEY_NAME = {
    'host.host_name'                : 'host.name',
    'host.display_name'             : 'host.distinguishedName',
    'host.address'                  : 'host.dNSHostName',
    'host._OS'                      : 'host.operatingSystem',
    'host._OS_SP'                   : 'host.operatingSystemServicePack',
    
    'hostgroup.hostgroup_name'      : 'hostgroup.name',
    
    'contactgroup.contactgroup_name': 'contactgroup.name',
    # Yes contact.member -> contactgroup.member it was this bad
    'contactgroup.members'          : 'contact.member',
    
    'contact.contact_name'          : 'contact.name',
    'contact.email'                 : 'contact.mail',
    'contact.display_name'          : 'contact.displayName',
    'contact._PHONE'                : 'contact.telephoneNumber',
    'contact._MOBILE'               : 'contact.mobile',
    'contact._COUNTRY'              : 'contact.co',
    'contact._CITY'                 : 'contact.l',
    'contact._COMPANY'              : 'contact.company',
}

DEPRECATED_MAPPING_KEY_NAME_REVERSED = dict([(v, k) for k, v in DEPRECATED_MAPPING_KEY_NAME.iteritems()])

# Lazy import as ldap can be super long to import
ldap = None


class BadMapping(Exception):
    pass


class ConfigurationError(Exception):
    def __init__(self, msg, output_msg="", warnings=None, errors=None, state=SOURCE_STATE.CRITICAL):
        if warnings is None:
            warnings = []
        if errors is None:
            errors = []
        self.log_msg = msg
        self.warnings = warnings
        self.errors = errors
        self.state = state
        self.output_msg = output_msg or msg


def add_unik(_list, _value):
    if _value not in _list:
        _list.append(_value)


def get_entry(e, prop):
    return e[1][prop][0].strip() if prop in e[1] else ''


def get_entries(e, prop):
    return [i.strip() for i in e[1][prop] if i.strip()] if prop in e[1] else []


def split_string(_str, _sep):
    return [s.strip() for s in _str.split(_sep) if s.strip()]


# called by the plugin manager to get a module
def get_instance(plugin):
    global ldap
    logger.info('[ldap-import] Get a LDAP import module for plugin %s' % plugin.get_name())
    
    # Import only if we need, as the ldap lib can be long to import
    import ldap as ldap
    ldap = ldap
    
    mode = getattr(plugin, 'mode', 'no_mode_found')
    if mode == 'no_mode_found':
        logger.warning('[ldap-import] The mode is not set in your source definition file. By default, I will use the "%s" mode.' % DEFAULT_MODE)
        mode = DEFAULT_MODE
    if mode != 'ad' and mode != 'openldap':
        logger.warning('[ldap-import] The mode used in your cfg file [%s] is not correct. It can be only "ad" or "openldap". I will use the "%s" mode.' % (mode, DEFAULT_MODE))
        mode = DEFAULT_MODE
    
    connection_configuration_file = getattr(plugin, 'connection_configuration_file', '')
    rules_configuration_file = getattr(plugin, 'rules_configuration_file', '')
    mapping_configuration_file = getattr(plugin, 'mapping_configuration_file', '')
    
    instance = LDAP_Import(plugin, mode, connection_configuration_file, rules_configuration_file, mapping_configuration_file)
    return instance


class LDAP_Import(BaseModule):
    def __init__(self, mod_conf, mode, connection_configuration_file, rules_configuration_file, mapping_configuration_file):
        BaseModule.__init__(self, mod_conf)
        self.MAPPING = {}
        self.warnings = []
        self.errors = []
        self.all_known_items = {
            _TYPE.HOST        : {},
            _TYPE.HOSTGROUP   : {},
            _TYPE.CONTACT     : {},
            _TYPE.CONTACTGROUP: {},
        }
        self.mode = mode
        self.connection_configuration_file = connection_configuration_file
        self.rules_configuration_file = rules_configuration_file
        self.mapping_configuration_file = mapping_configuration_file
        self.syncdaemon = None  # type: Optional[Synchronizer]
        self.hosts_base = ''
        self.hostgroups_base = ''
        self.contacts_base = ''
        self.contactgroups_base = ''
        self.filters = {
            'hosts'        : {'value': '', 'is_default': True},
            'hostgroups'   : {'value': '', 'is_default': True},
            'contacts'     : {'value': '', 'is_default': True},
            'contactgroups': {'value': '', 'is_default': True},
        }
        
        # BGL ALERT : the source_controller take self.possible_keys to inform the user if he has double sync keys items
        # todo : the source_controller must take into account the item_type in the possible_keys and the self._possible_keys mut replace self.possible_keys
        self._possible_keys = {}
        self.possible_keys = set()
        
        for item_type in (ITEM_TYPE.HOSTS, ITEM_TYPE.HOSTGROUPS, ITEM_TYPE.CONTACTS, ITEM_TYPE.CONTACTGROUPS):
            _possible_keys = getattr(mod_conf, 'properties_used_as_synckey_for_%s' % item_type, '')
            if not _possible_keys:
                _possible_keys = DEFAULT_POSSIBLE_KEYS.get(item_type)
            self._possible_keys[item_type] = set([key.strip() for key in _possible_keys.split(',')])
            self.possible_keys.update(['%s:%s' % (item_type, key.strip()) for key in _possible_keys.split(',')])
    
    
    def init(self):
        logger.info('[ldap-import] Initialization of the LDAP import module')
    
    
    def load(self, daemon):
        self.syncdaemon = daemon
        self.output_error_files = self.syncdaemon.t('import-ldap.output_cr_files_problems')
        self.output_error_import = self.syncdaemon.t('import-ldap.output_cr_import_error')
    
    
    def get_objects(self):
        self.MAPPING = {}
        self.warnings = []
        self.errors = []
        self.all_known_items = {
            _TYPE.HOST        : {},
            _TYPE.HOSTGROUP   : {},
            _TYPE.CONTACT     : {},
            _TYPE.CONTACTGROUP: {},
        }
        self.filters = {
            'hosts'        : {'value': '', 'is_default': True},
            'hostgroups'   : {'value': '', 'is_default': True},
            'contacts'     : {'value': '', 'is_default': True},
            'contactgroups': {'value': '', 'is_default': True},
        }
        
        # Load the connection parameters (server, user, password, etc)
        try:
            self._load_connection_file()
            self._load_mapping_file()
            self._load_rules_file()
        except ConfigurationError as e:
            logger.error('[ldap-import] %s' % e.log_msg)
            res = {'state': e.state, 'output': e.output_msg, 'objects': {}, 'errors': e.errors, 'warnings': e.warnings}
            return res
        
        if self.errors:
            s_err = self.syncdaemon.t('import-ldap.output_cr_import_error_log') % ', '.join(self.errors)
            logger.error('[ldap-import] %s' % s_err)
            res = {'state': SOURCE_STATE.CRITICAL, 'output': self.output_error_import, 'objects': {}, 'errors': [self.errors], 'warnings': self.warnings}
            return res
        
        # Strip everything after the / and/or : to check the host name is valid
        hname = self.url.split('://')[1].split('/')[0].split(':')[0]
        try:
            socket.gethostbyname(hname)  # dns query
        except:
            # Failed to resolve name
            s_err = self.syncdaemon.t('import-ldap.cr_unable_resolve_hostname') % hname
            logger.error('[ldap-import] %s' % s_err)
            return {'state': SOURCE_STATE.CRITICAL, 'output': self.output_error_import, 'objects': {}, 'errors': [s_err], 'warnings': self.warnings}
        
        hosts = []
        hostgroups = []
        contacts = []
        contactgroups = []
        try:
            if self.hostgroups_base:
                logger.debug('[ldap-import] load hostgroups in base:[%s]' % self.hostgroups_base)
                hostgroups = []
                for hg_b in self.hostgroups_base:
                    hostgroups.extend(self.read_in_ldap_base(hg_b, _TYPE.HOSTGROUP).values())
            if self.contactgroups_base:
                logger.debug('[ldap-import] load contactgroups in base:[%s]' % self.contactgroups_base)
                contactgroups = []
                for cg_b in self.contactgroups_base:
                    contactgroups.extend(self.read_in_ldap_base(cg_b, _TYPE.CONTACTGROUP).values())
            if self.hosts_base:
                logger.debug('[ldap-import] load hosts in base:[%s]' % self.hosts_base)
                hosts = []
                for h_b in self.hosts_base:
                    hosts.extend(self.read_in_ldap_base(h_b, _TYPE.HOST).values())
            if self.contacts_base:
                logger.debug('[ldap-import] load contacts in base:[%s]' % self.contacts_base)
                contacts = []
                for h_c in self.contacts_base:
                    contacts.extend(self.read_in_ldap_base(h_c, _TYPE.CONTACT).values())
        # Ldap fail in a way, bad!
        except ldap.LDAPError, exp:
            # ldap exception are objects with .message that are dict with 'desc' with the real error message
            s_exp = str(exp)
            if hasattr(exp, 'message') and 'desc' in exp.message:
                s_exp = exp.message['desc']
            s_err = self.syncdaemon.t('import-ldap.output_cr_import_error_log') % s_exp
            logger.error('[ldap-import] %s' % s_err)
            res = {'state': SOURCE_STATE.CRITICAL, 'output': self.output_error_import, 'objects': {}, 'errors': [s_exp], 'warnings': self.warnings}
            return res
        # If the mapping are missing an entry? exit!
        except BadMapping, exp:
            s_err = self.syncdaemon.t('import-ldap.output_cr_mapping_file') % (exp)
            logger.error('[ldap-import] %s' % s_err)
            res = {'state': SOURCE_STATE.CRITICAL, 'output': self.output_error_import, 'objects': {}, 'errors': [s_err], 'warnings': self.warnings}
            return res
        
        map(lambda x: self._clean_group(x, _TYPE.HOSTGROUP), hostgroups)
        map(lambda x: self._clean_group(x, _TYPE.CONTACT), contactgroups)
        map(self._clean_item, itertools.chain(hostgroups, contactgroups, hosts, contacts))
        
        objects = {'host': hosts, 'contact': contacts, 'hostgroup': hostgroups, 'contactgroup': contactgroups}
        if self.warnings:
            res = {'state': SOURCE_STATE.WARNING, 'output': self.syncdaemon.t('import-ldap.output_warn_load_successful'), 'objects': objects, 'errors': [], 'warnings': self.warnings}
        else:
            res = {'state': SOURCE_STATE.OK, 'output': self.syncdaemon.t('import-ldap.output_ok_load_successful'), 'objects': objects, 'errors': [], 'warnings': []}
        return res
    
    
    # Load the connection file values
    def _load_connection_file(self):
        
        if self.connection_configuration_file == '':
            raise ConfigurationError(self.syncdaemon.t('import-ldap.nc_default_connection_file'), output_msg=self.syncdaemon.t('import-ldap.output_nc_file'), state=SOURCE_STATE.NOT_CONFIGURED_BUT_CAN_LAUNCH_IMPORT)
        
        values = self._load_json_file(self.connection_configuration_file)
        missing_mandatory_keys = set(MANDATORY_CONFIGURATION_KEYS).difference(values)
        
        errors = []
        all_missing_mandatory_keys = [missing_mandatory_keys]
        if missing_mandatory_keys:
            for key in missing_mandatory_keys:
                errors.append(self.syncdaemon.t('import-ldap.missing_configuration_key') % (key, self.connection_configuration_file))
        
        missing_mandatory_keys = [mandatory_configuration_key for mandatory_configuration_key in MANDATORY_CONFIGURATION_KEYS if not values[mandatory_configuration_key]]
        all_missing_mandatory_keys.append(missing_mandatory_keys)
        if missing_mandatory_keys:
            for key in missing_mandatory_keys:
                errors.append(self.syncdaemon.t('import-ldap.missing_configuration_key') % (key, self.connection_configuration_file))
        
        if errors:
            raise ConfigurationError(self.syncdaemon.t('import-ldap.missing_configuration_key_log') % (','.join(all_missing_mandatory_keys), self.connection_configuration_file), output_msg=self.output_error_files, errors=errors)
        
        unknown_keys = set(values).difference(ALL_CONFIGURATION_KEYS)
        if unknown_keys:
            errors = []
            for key in unknown_keys:
                errors.append(self.syncdaemon.t('import-ldap.unknown_configuration_key') % (key, self.connection_configuration_file))
            raise ConfigurationError(self.syncdaemon.t('import-ldap.unknown_configuration_key_log') % (','.join(unknown_keys), self.connection_configuration_file), output_msg=self.output_error_files, errors=errors)
        
        self.url = values['url']
        self.username = values['username']
        self.password = values['password']
        self.base = values['base']
        self.port = '389'
        self.ldap_protocol = values['ldap_protocol']
        
        address_and_port = self.url.split('ldap://')[1] if 'ldap://' in self.url else self.url.split('ldaps://')[1]
        address_and_port = address_and_port.strip('/').split(':')
        self.address = address_and_port[0]
        if len(address_and_port) > 1:
            self.port = address_and_port[1]
        
        default_values = self._load_json_file(DEFAULT_FILES[self.mode]['connection'])
        
        self.hosts_base = split_string(self._get_value('hosts_base', values, default_values), '|')
        self.hosts_filter_with_group = split_string(self._get_value('hosts_filter_with_group', values, default_values), '|')
        
        self.hostgroups_base = split_string(self._get_value('hostgroups_base', values, default_values), '|')
        
        self.contacts_base = split_string(self._get_value('contacts_base', values, default_values), '|')
        self.contacts_filter_with_group = split_string(self._get_value('contacts_filter_with_group', values, default_values), '|')
        
        self.contactgroups_base = split_string(self._get_value('contactgroups_base', values, default_values), '|')
        
        self._set_filter('hosts', 'hosts_filter', values, default_values)
        self._set_filter('hostgroups', 'hostgroups_filter', values, default_values)
        self._set_filter('contacts', 'contacts_filter', values, default_values)
        self._set_filter('contactgroups', 'contactgroups_filter', values, default_values)
        
        # Not already configured, say it
        if self.url == r'ldap://YOUR-DC-FQDN/':
            raise ConfigurationError(self.syncdaemon.t('import-ldap.output_nc_in_file') % self.connection_configuration_file, state=SOURCE_STATE.NOT_CONFIGURED_BUT_CAN_LAUNCH_IMPORT)
        
        # Check the url, hinting user about possible problems
        if not self.url.startswith('ldap://') and not self.url.startswith('ldaps://'):
            error = self.syncdaemon.t('import-ldap.cr_malformed_url') % self.connection_configuration_file
            raise ConfigurationError(error, output_msg=self.output_error_files, errors=[error])
    
    
    def _load_rules_file(self):
        
        # In the 02.06.03, no rules are necessary (meaning rules file empty). But it load file in case of adding rule later ( mechanism ready )
        raw_rules = self._load_json_file(DEFAULT_FILES[self.mode]['rules'])
        
        if self.rules_configuration_file:
            raw_rules = self._load_json_file(self.rules_configuration_file)
        tag_values = {_TYPE.HOST: [], _TYPE.CONTACT: []}
        match_values = {_TYPE.HOST: [], _TYPE.CONTACT: []}
        
        self._check_rules(raw_rules)
        
        # We loop over the plugin properties to match
        # *hosts_tag_tpl OU => add the tpl if OU match
        # *hosts_match_KEY_TAG pattern => add the TAG if the host.key match pattern
        for item_type in (_TYPE.HOST, _TYPE.CONTACT):
            for (_key, _value) in raw_rules.iteritems():
                if _key.startswith(item_type + 's_tag_') or _key.startswith(item_type + 's_template_'):
                    tpl = _key.split('_', 2)[2]
                    ou = _value.lower()
                    if tpl:
                        e = {'tpl': tpl, 'ou': ou}
                        tag_values[item_type].append(e)
                elif _key.startswith(item_type + 's_tag') or _key.startswith(item_type + 's_template'):
                    tag_values[item_type].append({'tpl': _value, 'ou': '__ALL__'})
                elif '_to_' + item_type + '_matching_' in _key:
                    action = _key.split('_')[0]
                    key = re.search('\[(.*?)\]', _key).group(1)
                    tag = re.search('\((.*?)\)', _key).group(1)
                    if action not in ('AddFirst', 'AddLast', 'Force') or not key or not tag:
                        self.errors.append(self.syncdaemon.t('import-ldap.cr_invalid_matching_rules') % _key)
                    pattern = r'%s' % _value
                    # logger.debug(action, key, tag, pattern)
                    e = {'key': key, 'tag': tag, 'pattern': pattern, 'action': action}
                    match_values[item_type].append(e)
        
        self.tag_values = tag_values
        self.match_values = match_values
        # DEPRECATED key in rule : contacts_group_filter
        if not self.contacts_filter_with_group:
            self.contact_group_filter = [s.strip() for s in raw_rules.get('contacts_group_filter', '').split('|')]
    
    
    # We load a mapping file, an update our global MAPPING
    def _load_mapping_file(self):
        
        # load default values and override it with user value
        default_values = self._load_json_file(DEFAULT_FILES[self.mode]['mapping'])
        self.MAPPING.update(default_values)
        
        if self.mapping_configuration_file:
            lmap = self._load_json_file(self.mapping_configuration_file)
            keys_in_file = [k for k in lmap.iterkeys()]
            duplicate_keys = [(k, v) for k, v in DEPRECATED_MAPPING_KEY_NAME.iteritems() if k in keys_in_file and v in keys_in_file]
            if duplicate_keys:
                error = self.syncdaemon.t('import-ldap.warn_duplicate_keys') % (self.mapping_configuration_file, '</li><li>'.join([self.syncdaemon.t('import-ldap.warn_keys_replace') % (k, v) for k, v in duplicate_keys]))
                raise ConfigurationError(error, output_msg=self.output_error_files, errors=[error])
            
            for k, v in lmap.iteritems():
                if k in DEPRECATED_MAPPING_KEY_NAME_REVERSED:
                    k = DEPRECATED_MAPPING_KEY_NAME_REVERSED[k]
                self.MAPPING[k] = v
        
        self.contact_class_filter = self.MAPPING.pop('contact.classFilter', '')
        self.contact_category_filter = self.MAPPING.pop('contact.categoryFilter', '')
        
        if self.contact_class_filter:
            message = self.syncdaemon.t('import-ldap.warn_deprecated_in_mapping') % ('contacts_filter', self.connection_configuration_file, 'contact.classFilter', self.mapping_configuration_file)
            logger.warning('[ldap-import] %s' % message)
            add_unik(self.warnings, message)
        
        if self.contact_category_filter:
            message = self.syncdaemon.t('import-ldap.warn_deprecated_in_mapping') % ('contacts_filter', self.connection_configuration_file, 'contact.categoryFilter', self.mapping_configuration_file)
            logger.warning('[ldap-import] %s' % message)
            add_unik(self.warnings, message)
        
        # Here the followings properties must be set in connection file but it can also be defined in mapping (DEPRECATED).
        # So if they are already defined, we don't erase it
        self._set_filter('hosts', 'host.filter', self.MAPPING, {})
        self._set_filter('contacts', 'contact.filter', self.MAPPING, {})
        self._set_filter('hostgroups', 'hostgroup.filter', self.MAPPING, {})
        self._set_filter('contactgroups', 'contactgroup.filter', self.MAPPING, {})
        _host_filter = self.MAPPING.pop('host.filter', None)
        if _host_filter:
            message = self.syncdaemon.t('import-ldap.warn_deprecated_in_mapping') % ('hosts_filter', self.connection_configuration_file, 'host.filter', self.mapping_configuration_file)
            logger.warning('[ldap-import] %s' % message)
            add_unik(self.warnings, message)
        
        _contact_filter = self.MAPPING.pop('contact.filter', None)
        if _contact_filter:
            message = self.syncdaemon.t('import-ldap.warn_deprecated_in_mapping') % ('contacts_filter', self.connection_configuration_file, 'contact.filter', self.mapping_configuration_file)
            logger.warning('[ldap-import] %s' % message)
            add_unik(self.warnings, message)
        
        _hostgroup_filter = self.MAPPING.pop('hostgroup.filter', None)
        if _hostgroup_filter:
            message = self.syncdaemon.t('import-ldap.warn_deprecated_in_mapping') % ('hostgroups_filter', self.connection_configuration_file, 'hostgroup.filter', self.mapping_configuration_file)
            logger.warning('[ldap-import] %s' % message)
            add_unik(self.warnings, message)
        
        _contactgroup_filter = self.MAPPING.pop('contactgroup.filter', None)
        if _contactgroup_filter:
            message = self.syncdaemon.t('import-ldap.warn_deprecated_in_mapping') % ('contactgroups_filter', self.connection_configuration_file, 'contactgroup.filter', self.mapping_configuration_file)
            logger.warning('[ldap-import] %s' % message)
            add_unik(self.warnings, message)
        
        # Catch all incorrect mapping
        invalid_mapping = [mapping for mapping in self.MAPPING.iterkeys() if '.' not in mapping]
        if invalid_mapping:
            message = self.syncdaemon.t('import-ldap.warn_invalid_mapping') % (self.mapping_configuration_file, invalid_mapping)
            logger.warning('[ldap-import] %s' % message)
            add_unik(self.warnings, message)
            for key in invalid_mapping:
                self.MAPPING.pop(key, None)
    
    
    # Load a specific json file by removing the # at the start of the lines
    def _load_json_file(self, jfile):
        logger.debug('[ldap-import] loading configuration file [%s]' % jfile)
        file_without_comment = ''
        try:
            with open(jfile, 'r') as f:
                file_lines = f.readlines()
            # We need to clean comments on this file, because json is normally without comments
            lines = ['' if line.strip().startswith('#') else line.strip() for line in file_lines]
            file_without_comment = '\n'.join(lines)
            # json is read into in ordered dict
            # This is important for rules, so the rules are applied in the order they are written in the file
            j = json.loads(file_without_comment, object_hook=OrderedDict)
            return j
        except Exception, exp:
            if file_without_comment:
                logger.debug('[ldap-import] file_without_comment :')
                logger.debug(file_without_comment)
            error = self.syncdaemon.t('import-ldap.cr_file_format') % (jfile, exp)
            raise ConfigurationError(error, output_msg=self.output_error_files, errors=[error])
    
    
    def read_in_ldap_base(self, base='', nagios_class='', item_dn=''):
        elements = {}
        base_save = base
        item_type = '%ss' % nagios_class
        
        if item_dn:
            search_filter = item_dn.split(',')[0]
            base = ','.join(item_dn.split(',')[1:])
            base_save = base
        else:
            # Ask for all user objects
            if nagios_class == 'contact_group':
                # First element is CN
                search_filter = base.split(',')[0]
                # Remaining elements are base
                base = ','.join(base.split(',')[1:])
            elif item_type in self.filters:
                search_filter = self.filters[item_type]['value']
            else:
                raise Exception('unknown type to read :[%s]' % nagios_class)
        
        if not search_filter:
            search_filter = '(objectclass=*)'
            logger.warning('[ldap-import] For %s (%ss_filter), there is no filters set. I define it to %s ' % (nagios_class, nagios_class, search_filter))
        
        # Get entry one by one
        # WARNING : do not increase this, or you will miss entries
        page_size = 1
        
        # Initialize Ldap connection, with the shinken user
        ldap.set_option(ldap.OPT_REFERRALS, 0)
        ldap_connection = ldap.initialize(self.url)
        ldap_connection.protocol_version = self.ldap_protocol
        
        # Warning : user the bin_s for really SYNC connection
        try:
            ldap_connection.simple_bind_s(self.username, self.password)
        except ldap.LDAPError as e:
            logger.debug('[ldap-import] Fail to bind with ldap base.')
            if hasattr(e, 'message') and 'desc' in e.message:
                if e.message['desc'] == 'Invalid DN syntax':
                    e.message['desc'] = '%s: "%s"' % (e.message['desc'], self.username)
                elif e.message['desc'] == 'Invalid credentials' or e.message['desc'] == 'No such object':
                    e.message['desc'] = self.syncdaemon.t('import-ldap.cr_invalid_credential')
            raise
        
        ldapsearch_request = 'ldapsearch -h %s -p %s -D "%s" -b "%s" -W "%s"' % (self.address, self.port, self.username, base, search_filter)
        logger.debug('[ldap-import] SERVER search for type [%s] in base [%s] with filter [%s]' % (nagios_class, base, search_filter))
        # To help user to debug, we print him the request to play with ldapsearch
        logger.debug('[ldap-import] resquest to play : %s' % ldapsearch_request)
        try:
            lc = ldap.controls.libldap.SimplePagedResultsControl(criticality=False, size=page_size, cookie='')
            page_ctrl_oid = ldap.controls.SimplePagedResultsControl.controlType
            
            # User can set base or dn with accent, so encode it before send request
            if isinstance(base, unicode):
                base = base.encode('utf8', 'replace')
            
            if isinstance(search_filter, unicode):
                search_filter = search_filter.encode('utf8', 'replace')
            
            # Send search request
            msgid = ldap_connection.search_ext(base, ldap.SCOPE_SUBTREE, search_filter, serverctrls=[lc])
            # Ok loop until we got no more entry
            while True:
                # Get the result pointer
                try:
                    rtype, rdata, rmsgid, serverctrls = ldap_connection.result3(msgid, timeout=10)
                except ldap.TIMEOUT:
                    logger.warning('[ldap-import] timeout for the result')
                    break
                except ldap.NO_SUCH_OBJECT:
                    warn = self.syncdaemon.t('import-ldap.warn_cannot_find_base') % (make_unicode(base), self.rules_configuration_file if item_dn else self.connection_configuration_file)
                    add_unik(self.warnings, warn)
                    logger.warning('[ldap-import] %s' % warn)
                    return elements
                
                # WARNING: Maybe we already did have result in rdata (TODO: find why we have 2 data entry!!!)
                other_entries = self.analyse_data(rdata, nagios_class)
                elements.update(other_entries)
                
                pctrls = [c for c in serverctrls if c.controlType == page_ctrl_oid]
                if pctrls:
                    # LDAP server supports pagination
                    cookie = lc.cookie = pctrls[0].cookie
                    if cookie:
                        lc.controlValue = (page_size, cookie)
                        msgid = ldap_connection.search_ext(base, ldap.SCOPE_SUBTREE, search_filter, serverctrls=[lc])
                        result_type, result_data = ldap_connection.result(msgid, 0)
                        
                        if result_data:
                            if result_type == ldap.RES_SEARCH_ENTRY:
                                other_entries = self.analyse_data(result_data, nagios_class)
                                elements.update(other_entries)
                            else:
                                logger.debug('[ldap-import] skip result, not type ldap.RES_SEARCH_ENTRY.')
                        else:
                            logger.debug('[ldap-import] ::: void result')
                            continue
                    else:
                        break
                else:
                    logger.warning('[ldap-import] Warning: The ldap server doesn\'t support control page. (RFC 2696 control). A non root ldap-user can be limited for import data')
                    break
            
            return elements
        except ldap.LDAPError as e:
            if hasattr(e, 'message') and 'desc' in e.message:
                if e.message['desc'] == 'Invalid DN syntax':
                    e.message['desc'] = '%s: "%s"' % (e.message['desc'], base)
                elif e.message['desc'] == 'Referral':
                    e.message['desc'] = 'Incorrect base or group filter: %s' % base_save
                elif e.message['desc'] == "Administrative limit exceeded":
                    add_unik(self.warnings, self.syncdaemon.t('import-ldap.warn_limit_exceeded') % (nagios_class, e.message['desc']))
                    return elements
                elif e.message['desc'] == 'Bad search filter':
                    add_unik(self.warnings, self.syncdaemon.t('import-ldap.warn_bad_filter') % ('%ss_filter' % nagios_class, search_filter, e.message['desc']))
                    return elements
            raise
    
    
    def analyse_data(self, raw_datas, _type):
        raw_datas = make_unicode(raw_datas)
        elements = {}
        for raw_data in raw_datas:
            try:
                # avoid bad data
                if len(raw_data) != 2 or raw_data[0] is None:
                    continue
                
                if 'organizationalUnit' in raw_data[1]['objectClass']:
                    # In the case of the filter is set to empty, the request will search with filter (objectClass=*), so the organisationnalUnit are in the response. We don't manage it, skip it !
                    continue
                
                item = None
                item_name = None
                if _type == _TYPE.HOST:
                    if self.hosts_filter_with_group and True not in [self._item_is_in_group(group_dn, _TYPE.HOSTGROUP, raw_data[0], raw_data) for group_dn in self.hosts_filter_with_group]:
                        continue
                    item, item_name = self.analyse_host_entry(raw_data)
                elif _type == _TYPE.HOSTGROUP:
                    item, item_name = self.analyse_hostgroup_entry(raw_data)
                elif _type == _TYPE.CONTACTGROUP:
                    item, item_name = self.analyse_contactgroup_entry(raw_data)
                elif _type == _TYPE.CONTACT:
                    if self.contact_class_filter and self.contact_class_filter not in raw_data[1]['objectClass']:
                        continue
                    # In AD, a person's object category is ['CN=Person,CN=Schema,CN=Configuration,DC=...']
                    if self.contact_category_filter and self.contact_category_filter not in raw_data[1]['objectCategory'][0].split(',')[0]:
                        continue
                    if self.contacts_filter_with_group and True not in [self._item_is_in_group(group_dn, _TYPE.CONTACTGROUP, raw_data[0], raw_data) for group_dn in self.contacts_filter_with_group]:
                        continue
                    item, item_name = self.analyse_contact_entry(raw_data)
                
                if item:
                    elements[item_name] = item
            except KeyError, exp:
                # Some fields are known to be bad for mailbox or service accounts
                if exp.args[0] in ['givenName', 'userPrincipalName', 'sn', 'extensionAttribute2']:
                    logger.warning('[ldap-import] ERROR: UNKNOWN FIELD in Ldap entry %s' % str(exp))
                    pass
                else:
                    logger.warning('[ldap-import] ERROR: UNKNOWN FIELD in Ldap entry %s' % str(exp))
                    pass
        
        return elements
    
    
    def analyse_host_entry(self, raw_data):
        name = get_entry(raw_data, self.get_mapping_key('host.host_name', mandatory=True))
        _tpls = self._apply_rules(name, raw_data, _TYPE.HOST)
        item = self._create_item(raw_data, _TYPE.HOST, name, _tpls)
        return item, name
    
    
    def analyse_contact_entry(self, raw_data):
        name = get_entry(raw_data, self.get_mapping_key('contact.contact_name', mandatory=True))
        _tpls = self._apply_rules(name, raw_data, _TYPE.CONTACT)
        item = self._create_item(raw_data, _TYPE.CONTACT, name, _tpls)
        return item, name
    
    
    def analyse_hostgroup_entry(self, raw_data):
        name = get_entry(raw_data, self.get_mapping_key('hostgroup.hostgroup_name', mandatory=True))
        item = self._create_item(raw_data, _TYPE.HOSTGROUP, name)
        return item, name
    
    
    def analyse_contactgroup_entry(self, raw_data):
        name = get_entry(raw_data, self.get_mapping_key('contactgroup.contactgroup_name', mandatory=True))
        item = self._create_item(raw_data, _TYPE.CONTACTGROUP, name)
        return item, name
    
    
    def _check_rules(self, raw_rules):
        for rule in raw_rules.keys():
            if rule.startswith('contacts_tag') or rule.startswith('hosts_tag') or rule.startswith('contacts_template') or rule.startswith('hosts_template'):
                continue
            
            # This 'rules' is deprecated
            if rule == 'contacts_group_filter':
                message = self.syncdaemon.t('import-ldap.warn_deprecated_in_rule') % ('contactgroups_filter', self.connection_configuration_file, 'contacts_group_filter', self.rules_configuration_file)
                logger.warning('[ldap-import] %s' % message)
                add_unik(self.warnings, message)
                continue
            
            matched = False
            if re.match(r'^(AddLast|AddFirst|Force)_template_\(.*\)_to_(host|contact)_matching_\[.*\]$', rule):
                continue
            
            # Rule doesn't match any case ? ok it's not correct
            message = self.syncdaemon.t('import-ldap.warn_invalid_rule') % rule
            logger.warning('[ldap-import] %s' % message)
            add_unik(self.warnings, message)
        pass
    
    
    def _apply_rules(self, name, raw_data, item_type):
        # logger.debug('[ldap-import] apply rules to:[%s] : [%s]' % (name, raw_data))
        dn_name = raw_data[0]
        
        _tpls = ''
        for d in self.tag_values[item_type]:
            tpl = d['tpl']
            ou = d['ou']
            try:
                if ou == '__ALL__' or dn_name.lower().endswith(ou):
                    _tpls = ','.join((tpl, _tpls))
            except UnicodeDecodeError as e:
                message = self.syncdaemon.t('import-ldap.warn_invalid_character_in_item') % (dn_name, e)
                logger.warning('[ldap-import] %s' % message)
                add_unik(self.warnings, message)
        
        for d in self.match_values[item_type]:
            key = '%s.%s' % (item_type, d['key'])
            tag = d['tag']
            pattern = d['pattern']
            action = d['action']
            try:
                # logger.debug('[ldap-import] in _apply_rules rule:[%s]' % d)
                
                if d['key'] == 'memberOf':
                    group_dn = pattern
                    group_type = _TYPE.HOSTGROUP if item_type == _TYPE.HOST else _TYPE.CONTACTGROUP
                    match = self._item_is_in_group(group_dn, group_type, dn_name, raw_data)
                else:
                    key = self.get_mapping_key(key)
                    if not key:
                        # OK key is not DEPRECATED and is not mapped. so take the original key
                        key = d['key']
                    values = get_entries(raw_data, key.split('.')[1] if '.' in key else key)
                    # logger.debug('[ldap-import] in _apply_rules rule key:[%s] value :[%s] pattern:[%s]' % (key, v, pattern))
                    if not values:
                        continue
                    match = False
                    for value in values:
                        match = re.match(pattern, value, re.I)
                        match = match and (match.group(0) == value)
                        if match:
                            break
                
                if match:
                    # logger.debug('[ldap-import] in _apply_rules adding a template:[%s]' % tag)
                    if _tpls:
                        if action == 'AddFirst':
                            _tpls = tag + ',' + _tpls
                        elif action == 'Force':
                            _tpls = tag
                        elif action == 'AddLast':
                            _tpls = _tpls + ',' + tag
                    else:
                        _tpls = tag
            except KeyError:
                # logger.debug('[ldap-import] in _apply_rules failed to get key:[%s]' % key)
                continue
        return _tpls
    
    
    def _item_is_in_group(self, group_dn, group_type, dn_name, raw_data):
        if not group_dn:
            return False
        group = self.all_known_items[group_type].get(group_dn, None)
        if not group:
            self.read_in_ldap_base(nagios_class=group_type, item_dn=group_dn)
        group = self.all_known_items[group_type].get(group_dn, None)
        if not group:
            logger.warning('[ldap-import] For filter we did not find the group:[%s]' % group_dn)
            return
        members = group['__RAW_MEMBERS__']
        ref_member_key = group['__RAW_REF_MEMBER_KEY__']
        
        if ref_member_key == '__DN__':
            match = dn_name in members
        else:
            ref_value = get_entry(raw_data, ref_member_key)
            match = ref_value in members
        return match
    
    
    def _create_item(self, raw_data, item_type, name, templates=None):
        # logger.debug('[ldap-import] creating %s name:[%s]' % (item_type, name))
        item = {}
        _keys = set()
        _possible_keys = self._possible_keys.get('%ss' % item_type, set())
        
        item_dn = raw_data[0]
        for k in self.MAPPING.iterkeys():
            _type, _prop_name = k.split('.')
            if _prop_name == 'members':
                continue
            if item_type == _type:
                value = get_entry(raw_data, self.get_mapping_key(k))
                if value:
                    item[_prop_name] = value
                    
                    if _prop_name in _possible_keys:
                        _keys.add(value)
        
        if item_type in (_TYPE.HOSTGROUP, _TYPE.CONTACTGROUP):
            _members, ref_member_key = self._get_group_members(raw_data, item_type)
            item['__RAW_MEMBERS__'] = _members
            item['__RAW_REF_MEMBER_KEY__'] = ref_member_key
        
        if item_type == _TYPE.CONTACT:
            pager = item.get('_MOBILE', item.get('_PHONE', ''))
            if pager:
                item['pager'] = pager
            
            # A long time ago, the short_mail was used as a sync_key, hard-coded here. To be able to find this behavior, we allow short mail in properties usable as a synchronization key and do not stored it
            if 'short_mail' in _possible_keys:
                short_mail = item.get('email', '').split('@', 1)[0]
                if short_mail:
                    _keys.add(short_mail)
        
        if templates:
            item['use'] = templates
        
        item['_SYNC_KEYS'] = (','.join(_keys)).lower()
        
        item['__RAW_DATA__'] = raw_data
        self.all_known_items[item_type][item_dn] = item
        return item
    
    
    def _get_group_members(self, raw_group_data, _type):
        if not raw_group_data:
            return ''
        _members = []
        
        key_name = self.get_mapping_key('%s.members' % _type)
        ref_member_key = self.get_mapping_key('%s.ref_member_key' % _type, '__DN__')
        if key_name not in raw_group_data[1]:
            if 'groupOfUniqueNames' in raw_group_data[1]['objectClass']:
                key_name = 'uniqueMember'
            elif 'groupOfNames' in raw_group_data[1]['objectClass']:
                key_name = 'member'
            elif 'posixGroup' in raw_group_data[1]['objectClass']:
                key_name = 'memberUid'
                ref_member_key = 'uid'
            elif 'group' in raw_group_data[1]['objectClass']:
                key_name = 'member'
            else:
                logger.warning('[ldap-import] For the group [%s] we cannot get members' % raw_group_data[0])
        
        _members = raw_group_data[1].get(key_name, [])
        return _members, ref_member_key
    
    
    def get_mapping_key(self, prop, default=None, mandatory=False):
        value = self.MAPPING.get(prop, None)
        
        if value is None:
            if not mandatory:
                return default
            raise BadMapping('Key :[%s] is not mapped' % prop)
        return value
    
    
    def _clean_group(self, group, _group_type):
        _members = group.pop('__RAW_MEMBERS__')
        ref_member_key = group.pop('__RAW_REF_MEMBER_KEY__')
        _member_type = _TYPE.HOST if _group_type == _TYPE.HOSTGROUP else _TYPE.CONTACT
        _members_names = []
        for _member_link in _members:
            for item_dn_name, item in self.all_known_items[_member_type].iteritems():
                item_name = item.get('host_name' if _TYPE.HOST == _member_type else 'contact_name', '')
                if not item_name:
                    continue
                
                if ref_member_key == '__DN__':
                    match = item_dn_name == _member_link
                else:
                    ref_value = get_entry(item['__RAW_DATA__'], ref_member_key)
                    match = ref_value == _member_link
                if match:
                    _members_names.append(item_name)
        
        if _members_names:
            group['members'] = ','.join(_members_names)
    
    
    def _clean_item(self, item):
        item.pop('__RAW_DATA__')
    
    
    def _set_filter(self, filter_key, key_to_load, value, default):
        if not self.filters[filter_key]['is_default']:
            return
        
        if value.get(key_to_load, None) != None:
            self.filters[filter_key]['value'] = value[key_to_load]
            self.filters[filter_key]['is_default'] = False
            return
        
        if default.get(key_to_load, None):
            self.filters[filter_key]['value'] = default[key_to_load]
    
    
    def _get_value(self, key, conf, default):
        try:
            _data = conf.get(key, default[key])
        except KeyError:
            error = self.syncdaemon.t('import-ldap.cr_default_values_ko') % (key)
            raise ConfigurationError(error, output_msg=self.output_error_files, errors=[error])
        return _data
