#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018
# This file is part of Shinken Enterprise, all rights reserved.

import time
import urllib
from collections import OrderedDict

import shinkensolutions.shinkenjson as json
from shinken.log import logger
from shinken.webui.bottlewebui import response
from shinkensolutions.api.synchronizer.source.abstract_module.listener_rest_module import BaseRESTListener

properties = {
    'daemons': ['synchronizer'],
    'type'   : 'listener_shinken',
}


# called by the plugin manager to get a broker
def get_instance(plugin):
    # logger.info("[WS_Arbiter] get_instance ...")
    instance = ShinkenRESTListener(plugin)
    return instance


class ShinkenRESTListener(BaseRESTListener):
    url_base_path = '/shinken/listener-shinken'
    
    
    def get_configuration_fields(self):
        if self._configuration_fields is None:
            self._configuration_fields = OrderedDict([
                ('configuration', OrderedDict([
                    ('authentication', {
                        'display_name': self._('analyzer.conf_authentication'),
                        'default'     : True,
                        'protected'   : False,
                        'help'        : '',
                        'type'        : 'checkbox',
                        'display_bind': ('login', 'password')
                    }),
                    ('login', {
                        'display_name': self._('analyzer.conf_login'),
                        'default'     : 'Shinken',
                        'protected'   : False,
                        'help'        : '',
                        'type'        : 'text',
                    }),
                    ('password', {
                        'display_name': self._('analyzer.conf_password'),
                        'default'     : 'OFU2SE4zOU1FMDdaQlJENFtljgwTcWn2hQ5ocksBWS0=',
                        'protected'   : True,
                        'help'        : '',
                        'type'        : 'text',
                    }),
                    ('use_ssl', {
                        'display_name': self._('analyzer.conf_use_ssl'),
                        'default'     : False,
                        'protected'   : False,
                        'help'        : '',
                        'type'        : 'checkbox',
                        'display_bind': ('ssl_key', 'ssl_cert')
                    }),
                    ('ssl_key', {
                        'display_name': self._('analyzer.conf_ssl_key'),
                        'default'     : '',
                        'protected'   : False,
                        'help'        : '',
                        'type'        : 'text',
                    }),
                    ('ssl_cert', {
                        'display_name': self._('analyzer.conf_ssl_cert'),
                        'default'     : '',
                        'protected'   : False,
                        'help'        : '',
                        'type'        : 'text',
                    }),
                ])
                 ),
            ])
        return self._configuration_fields
    
    
    def _init_routes(self, app):
        
        def get_sync_uuid():
            response.content_type = 'application/json'
            self._query_check_auth()
            uuid = self.syncdaemon._suuid
            result = {'rc': 200, 'data': {"suuid": uuid}}
            return result
        
        
        def get_remote_hosts(states="new,working_area,stagging,changes", keep_overrides_on_disabled_checks="false"):
            response.content_type = 'application/json'
            self._query_check_auth()
            conn = self.syncdaemon.get_synchronizer_syncui_connection()
            result = {'rc': 200, 'data': {}}
            nb_try = 0
            max_try = 5
            while nb_try < max_try:  # do the try for max_try times
                nb_try += 1
                try:
                    params = urllib.urlencode({
                        'states'                           : states,
                        'keep_overrides_on_disabled_checks': keep_overrides_on_disabled_checks,
                        'types'                            : 'hosts',
                        'private_key'                      : self.syncdaemon.get_private_http_key(),
                    })
                    conn.request("GET", "/internal/get_whole_configuration?%s" % params)
                    respond = conn.getresponse()
                    if respond.status != 200:
                        hosts = {}
                    else:
                        buf = respond.read()
                        answer = json.loads(buf)
                        hosts = answer.get('host', [])
                    if hosts:
                        result['data'] = [d for d in hosts if d.get('host_name', '')]
                    else:
                        result['data'] = []
                    conn.close()
                    result['rc'] = 200
                    break
                except Exception as exp:
                    logger.error('[listener-shinken] cannot get the Synchronizer objects data: "%s" still %d tries' % (str(exp), max_try - nb_try))
                    time.sleep(nb_try * 2)
                    result['rc'] = 504
            return result
        
        
        app = super(ShinkenRESTListener, self)._init_routes(app)
        app.route('%s/v1/get_remote_hosts' % self.url_base_path, callback=get_remote_hosts, method='GET')
        app.route('%s/v1/get_sync_uuid' % self.url_base_path, callback=get_sync_uuid, method='GET')
        return app
