#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import uuid

import sla_common
from shinken.log import PART_INITIALISATION
from shinken.misc.type_hint import List, Dict, Optional
from shinken.objects.module import Module as ShinkenModuleDefinition
from shinkensolutions.date_helper import DATE_COMPARE, date_now, get_start_of_day, get_end_of_day, compare_date, get_now, Date
from shinkensolutions.lib_modules.configuration_reader_mixin import ConfigurationFormat, ConfigurationReaderMixin, TypeConfiguration
from sla_abstract_component import AbstractComponent
from sla_common import STATUS, LIST_STATUS, RAW_SLA_KEY, shared_data
from sla_component_manager import ComponentManager
from sla_compute_percent_sla import ComputePercentSla
from sla_database import SLADatabase
from sla_info import SLAInfo

MARGIN_SLA_INACTIVE = 30
MARGIN_MISSING_DATA = 60
MARGIN_MISSING_DATA_AT_STARTUP = 600


class SLAArchive(AbstractComponent, ConfigurationReaderMixin):
    
    def __init__(self, conf, component_manager, sla_info, compute_percent_sla, sla_database, father_name=u'broker', prefix_module_property=u''):
        # type: (ShinkenModuleDefinition, ComponentManager,  SLAInfo, ComputePercentSla, SLADatabase, unicode, unicode) -> None
        global MARGIN_SLA_INACTIVE, MARGIN_MISSING_DATA, MARGIN_MISSING_DATA_AT_STARTUP
        super(SLAArchive, self).__init__(conf, component_manager)
        
        self.sla_info = sla_info
        self.compute_percent_sla = compute_percent_sla
        self.sla_database = sla_database
        
        self.margin_inactive = None
        self.margin_missing = None
        self.margin_missing_at_startup = None
        
        configuration_format = [
            ConfigurationFormat([u'time_before_shinken_inactive', u'%stime_before_shinken_inactive' % prefix_module_property], MARGIN_SLA_INACTIVE, TypeConfiguration.INT, u'margin_inactive'),
            ConfigurationFormat([u'minimal_time_before_an_element_become_missing_data', u'%sminimal_time_before_an_element_become_missing_data' % prefix_module_property], MARGIN_MISSING_DATA, TypeConfiguration.INT, u'margin_missing'),
            ConfigurationFormat([u'minimal_time_before_an_element_become_missing_data_at_startup', u'%sminimal_time_before_an_element_become_missing_data_at_startup' % prefix_module_property], MARGIN_MISSING_DATA_AT_STARTUP, TypeConfiguration.INT,
                                u'margin_missing_at_startup'),
        
        ]
        logger_initialisation = self.logger.get_sub_part(PART_INITIALISATION)
        ConfigurationReaderMixin.__init__(self, configuration_format, conf, logger_initialisation)
        self.read_configuration()
        
        archive_name = u'writing'
        if father_name == u'webui':
            archive_name = u'building'
        logger_initialisation.info(u'Reading configuration for sla archive %s' % archive_name)
        self.log_configuration(log_properties=True, show_values_as_in_conf_file=True)
    
    
    def init(self):
        pass
    
    
    def tick(self):
        pass
    
    
    def build_archive_for_missing_day(self, date, item_uuid, start_of_range=None, end_of_day=None, sla_thresholds=None, only_existing_day=False):
        current_date = date_now()
        item_monitoring_start_time = self.sla_info.get_monitoring_start_time(item_uuid)
        
        start_of_day = get_start_of_day(date)
        if start_of_range is not None and start_of_range != -1 and start_of_range > start_of_day:
            start_of_day = int(start_of_range)
        
        end_of_day = get_end_of_day(date) if end_of_day is None else end_of_day
        sla_thresholds = (99, 97) if sla_thresholds is None else sla_thresholds
        
        in_future = compare_date(current_date, date) == DATE_COMPARE.IS_AFTER
        in_past = end_of_day < item_monitoring_start_time
        
        if only_existing_day and (in_future or in_past):
            return None
        
        if start_of_day < item_monitoring_start_time < end_of_day:
            start_of_day = item_monitoring_start_time
        
        sla_archive = {
            u'_id'             : uuid.uuid4().hex,
            u'build_at'        : get_now(),
            u'uuid'            : item_uuid,
            u'yday'            : date.yday,
            u'year'            : date.year,
            u'ranges'          : [{u'rc': STATUS.SHINKEN_INACTIVE, u'end': end_of_day, u'start': start_of_day, u'ack': False, u'dt': False, u'flg': False}],
            u'version'         : sla_common.CURRENT_ARCHIVE_VERSION,
            u'history_inactive': end_of_day - start_of_day,
            u'history_total'   : end_of_day - start_of_day,
            u'thresholds'      : sla_thresholds,
            u'missing'         : True,
        }
        
        self.compute_percent_sla.compute_sla(sla_archive)
        for prefix in LIST_STATUS:
            _sum = sla_archive.get(u'sla_%s' % prefix, 0)
            if _sum:
                sla_archive[u'archive_%s' % prefix] = _sum
        sla_archive[u'archive_total'] = sla_archive[u'sla_total']
        sla_archive[u'archive_format'] = sla_archive[u'sla_format']
        sla_archive[u'total_ranges'] = len(sla_archive[u'ranges'])
        
        if in_future:
            sla_archive[u'in_future'] = True
        elif in_past:
            sla_archive[u'in_past'] = True
        
        return sla_archive
    
    
    def build_archive(self, date, item_uuid, end_of_day=None, inactive_ranges=None):
        # type: (Date, str, int, List) -> Optional[Dict]
        # self.logger.debug('asking build_archive [%s,%s]  [%s]' % (date, end_of_day, item_uuid))
        
        if shared_data.get_already_archived() and self.sla_database.find_archive(item_uuid, date, lookup={'_id': 1}):
            self.sla_database.remove_archive(item_uuid, date)
        
        sla_archive = {'uuid': item_uuid, 'yday': date.yday, 'year': date.year}
        
        if end_of_day is None:
            end_of_day = get_end_of_day(date)
            # self.logger.debug('build_archive end_of_day [%s]' % print_time(end_of_day))
        
        start_of_range = self.sla_info.get_monitoring_start_time(item_uuid)
        
        # If item did not exist this day, no need to build archive (#SEF-7829)
        if start_of_range > end_of_day:
            return None
        
        sla_thresholds = self.sla_info.get_sla_thresholds(item_uuid)
        # self.logger.debug('get_monitoring_start_time [%s]' % print_time(start_of_range))
        sla_archive['ranges'] = self._build_range(date, start_of_range, end_of_day, item_uuid, inactive_ranges=inactive_ranges)
        # self.logger.debug('build_range [%s]' % len(info['ranges']))
        if not sla_archive['ranges']:
            return self.build_archive_for_missing_day(date, item_uuid, start_of_range=start_of_range, end_of_day=end_of_day, sla_thresholds=sla_thresholds)
        self.compute_percent_sla.compute_state_sum(sla_archive['ranges'], sla_archive)
        self.compute_percent_sla.compute_sla(sla_archive)
        for prefix in LIST_STATUS:
            _sum = sla_archive.get('sla_%s' % prefix, 0)
            if _sum:
                sla_archive['archive_%s' % prefix] = _sum
        
        sla_archive['archive_total'] = sla_archive['sla_total']
        sla_archive['archive_format'] = sla_archive['sla_format']
        sla_archive['thresholds'] = sla_thresholds
        sla_archive['total_ranges'] = len(sla_archive['ranges'])
        sla_archive['version'] = sla_common.CURRENT_ARCHIVE_VERSION
        sla_archive['build_at'] = get_now()
        sla_archive['_id'] = uuid.uuid4().hex
        return sla_archive
    
    
    def compute_inactive_ranges(self, date, start_of_day, end_of_range):
        values = []
        sla_status = self.sla_database.find_raw_sla_status(date)
        if not sla_status:
            return [{
                'active_range'                          : False,
                RAW_SLA_KEY.CONTEXT_AND_STATUS          : STATUS.SHINKEN_INACTIVE * 100,
                RAW_SLA_KEY.START                       : start_of_day,
                RAW_SLA_KEY.END                         : end_of_range,
                RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME: end_of_range,
            }]
        
        sla_status = sla_status['active_ranges']
        sla_status = sorted(sla_status, key=lambda _value: _value['start'])
        
        # self.logger.debug('range %s -- %s' % (self.logger.format_time(start_of_day), self.logger.format_time(end_of_range)))
        last_end = start_of_day
        for sla_stat in sla_status:
            inactive_range = {
                'active_range'                : False,
                RAW_SLA_KEY.CONTEXT_AND_STATUS: STATUS.SHINKEN_INACTIVE * 100
            }
            
            start = sla_stat['start']
            end = sla_stat['end']
            if start > end:
                self.logger.warning('An inconsistency in shinken activity was found. Check your ntp server.')
                self.logger.warning('Inconsistency : start:[%s] - end:[%s]' % (self.logger.format_time(start), self.logger.format_time(end)))
                continue
            # self.logger.debug('active_range [%s  -  %s] last_end[%s] start-last_end:[%s]' % (self.logger.format_time(start), self.logger.format_time(end), self.logger.format_time(last_end), (start - last_end)))
            
            if (start - last_end) > self.margin_inactive:
                if last_end < start_of_day:
                    last_end = start_of_day
                inactive_range[RAW_SLA_KEY.START] = last_end
                inactive_range[RAW_SLA_KEY.END] = start
                inactive_range[RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME] = start
                
                values.append(inactive_range)
                # self.logger.debug('inactive_range1 %s : [%s  -  %s] ' % (inactive_range[RAW_SLA_KEY.CONTEXT_AND_STATUS], self.logger.format_time(inactive_range[RAW_SLA_KEY.START]), self.logger.format_time(inactive_range[RAW_SLA_KEY.END])))
            if end > last_end:
                last_end = end
        
        if (end_of_range - last_end) > self.margin_inactive:
            if last_end < start_of_day:
                last_end = start_of_day
            inactive_range = {
                'active_range'                          : False,
                RAW_SLA_KEY.CONTEXT_AND_STATUS          : STATUS.SHINKEN_INACTIVE * 100,
                RAW_SLA_KEY.START                       : last_end,
                RAW_SLA_KEY.END                         : end_of_range,
                RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME: end_of_range
            }
            values.append(inactive_range)
            # self.logger.debug('inactive_range2 %s : [%s  -  %s] ' % (inactive_range[RAW_SLA_KEY.CONTEXT_AND_STATUS], self.logger.format_time(inactive_range[RAW_SLA_KEY.START]), self.logger.format_time(inactive_range[RAW_SLA_KEY.END])))
        return values
    
    
    def _build_range(self, date, start_of_range, end_of_range, item_uuid, inactive_ranges=None):
        # type: (Date, int, int, str, List) -> List
        # self.logger.debug('_build_range_sla [%s]:[%s]-[%s] [%s]' % (date, print_time(start_of_range), print_time(end_of_range), item_uuid))
        
        start_of_day = get_start_of_day(date)
        if start_of_range is not None and start_of_range != -1 and start_of_range > start_of_day:
            start_of_day = int(start_of_range)
        # self.logger.debug('start_of_day [%s]' % self.logger.format_time_as_sla(start_of_day))
        
        if not inactive_ranges:
            inactive_ranges = self.compute_inactive_ranges(date, start_of_day, end_of_range)
        
        raw_sla = self.sla_database.find_raw_sla(date, item_uuid)
        # self.logger.debug('nb raw sla found %s' % len(raw_sla))
        
        inactive_ranges = self._prepare_ranges(u'inactive_ranges', inactive_ranges, self.margin_missing_at_startup, start_of_day, end_of_range)
        raw_sla = self._prepare_ranges(u'raw_sla', raw_sla, self.margin_missing, start_of_day, end_of_range)
        
        # self.logger.debug('inactive_ranges')
        # for i in inactive_ranges:
        #     self.logger.debug('inactive_ranges [%s] [%s-%s]' % (i[RAW_SLA_KEY.CONTEXT_AND_STATUS], self.logger.format_time(i[RAW_SLA_KEY.START]), self.logger.format_time(i[RAW_SLA_KEY.END])))
        #
        # self.logger.debug('raw_sla')
        # for i in raw_sla:
        #     self.logger.debug('raw_sla [%s] [%s-%s-%s] output:[%s][%s]' % (
        #         i[RAW_SLA_KEY.CONTEXT_AND_STATUS], self.logger.format_time(i[RAW_SLA_KEY.START]), self.logger.format_time(i[RAW_SLA_KEY.END]), self.logger.format_time(i.get(RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME, None)),
        #         i.get(RAW_SLA_KEY.OUTPUT, None), i.get(RAW_SLA_KEY.LONG_OUTPUT, None)
        #     ))
        
        mixed_raw_ranges = self._mix_ranges(inactive_ranges, raw_sla)
        # for i in mixed_raw_ranges:
        #     self.logger.debug(        'mixed_raw_ranges [%s] [%s-%s] output:[%s][%s]' % (i[RAW_SLA_KEY.CONTEXT_AND_STATUS], print_time(i[RAW_SLA_KEY.START]), print_time(i[RAW_SLA_KEY.END]), i.get(RAW_SLA_KEY.OUTPUT, None), i.get(RAW_SLA_KEY.LONG_OUTPUT, None)))
        ranges = [self._raw_range_to_range(i) for i in mixed_raw_ranges]
        
        ranges.reverse()
        return ranges
    
    
    @staticmethod
    def _mix_ranges(ranges1, ranges2):
        # type: (List,List) -> List
        if not ranges1:
            return ranges2
        if not ranges2:
            return ranges1
        
        ranges1_name = ranges1[0]['name']
        ranges2_name = ranges2[0]['name']
        
        ranges = ranges1 + ranges2
        ranges = sorted(ranges, key=lambda _value: _value[RAW_SLA_KEY.START])
        
        cursor = None
        current_range1 = None
        current_range2 = None
        range_build = []
        for current_range in ranges:
            if current_range['name'] == ranges1_name:
                current_range1 = current_range
            if current_range['name'] == ranges2_name:
                current_range2 = current_range
            
            if cursor is None:
                cursor = current_range.copy()
                range_build.append(cursor)
            else:
                if SLAArchive._range_collide(current_range1, current_range2):
                    win_range = SLAArchive._mix_state(current_range1, current_range2)
                else:
                    win_range = current_range.copy()
                
                del win_range[RAW_SLA_KEY.START]
                del win_range[RAW_SLA_KEY.END]
                
                if cursor[RAW_SLA_KEY.CONTEXT_AND_STATUS] != win_range[RAW_SLA_KEY.CONTEXT_AND_STATUS]:
                    if cursor[RAW_SLA_KEY.START] == current_range[RAW_SLA_KEY.START]:
                        cursor.update(win_range)
                    else:
                        cursor[RAW_SLA_KEY.END] = current_range[RAW_SLA_KEY.START]
                        cursor = current_range.copy()
                        cursor.update(win_range)
                        range_build.append(cursor)
                
                if cursor[RAW_SLA_KEY.END] < current_range[RAW_SLA_KEY.END]:
                    cursor[RAW_SLA_KEY.END] = current_range[RAW_SLA_KEY.END]
        return range_build
    
    
    @staticmethod
    def _range_collide(current_range1, current_range2):
        end1 = max(current_range1.get(RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME, 0), current_range1[RAW_SLA_KEY.END]) if current_range1 else 0
        end2 = max(current_range2.get(RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME, 0), current_range2[RAW_SLA_KEY.END]) if current_range2 else 0
        
        return current_range1 and current_range2 and (
                (current_range1[RAW_SLA_KEY.START] <= current_range2[RAW_SLA_KEY.START] <= end1) or
                (current_range2[RAW_SLA_KEY.START] <= current_range1[RAW_SLA_KEY.START] <= end2))
    
    
    @staticmethod
    def _mix_state(current_range1, current_range2):
        state1 = current_range1[RAW_SLA_KEY.CONTEXT_AND_STATUS]
        state2 = current_range2[RAW_SLA_KEY.CONTEXT_AND_STATUS]
        # from low to high
        state_priorities = (STATUS.MISSING_DATA, STATUS.UNKNOWN, STATUS.OK, STATUS.WARN, STATUS.CRIT, STATUS.SHINKEN_INACTIVE)
        for state_priority in state_priorities:
            if state1 / 100 % 10 == state_priority:
                return current_range2.copy()
            if state2 / 100 % 10 == state_priority:
                return current_range1.copy()
        
        return current_range1.copy()
    
    
    @staticmethod
    def _prepare_ranges(name, values, margin_new_range, start_of_range, end_of_range):
        # type: (unicode, List, int, int, int) -> List
        values = sorted(values, key=lambda _value: _value[RAW_SLA_KEY.START])
        cursor = None
        
        range_build = []
        for value in values:
            if RAW_SLA_KEY.CONTEXT_AND_STATUS in value:
                cache_value = value[RAW_SLA_KEY.CONTEXT_AND_STATUS]
            else:
                cache_value = value[RAW_SLA_KEY.ACK] + 10 * value[RAW_SLA_KEY.DT] + 100 * value[RAW_SLA_KEY.STATUS] + 1000 * value.get(RAW_SLA_KEY.FLAPPING, 0)
            start = value[RAW_SLA_KEY.START]
            end = value[RAW_SLA_KEY.END]
            ack_uuid = value.get(RAW_SLA_KEY.ACK_UUID, u'')
            output = value.get(RAW_SLA_KEY.OUTPUT, None)
            long_output = value.get(RAW_SLA_KEY.LONG_OUTPUT, None)
            downtimes_uuid = value.get(RAW_SLA_KEY.DOWNTIMES_UUID, [])
            missing_data_activation_time = value.get(RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME, 0)
            
            # Discard off limit ranges
            if 0 < end < start_of_range or start > end_of_range:
                continue
            
            # Crop off limit ranges to start at beginning of expected period (take special care of status with no end of validity)
            if start < start_of_range <= end or end <= 0 < start < start_of_range:
                start = start_of_range

            # Crop off limit ranges to finish at end of expected period (with a special care of status with no end of validity)
            if start <= end_of_range < end or end <= 0 < start <= end_of_range:
                end = end_of_range
                missing_data_activation_time = end_of_range
            
            if start > end > 0:
                # This status is obsolete and should be missing data (no end time, no missing data activation time)
                # As this status is incorrect, we ignore it
                if missing_data_activation_time < start:
                    continue
                # We may have received a context change between end of status and missing data activation time,
                # we set a possible end time (>= start time)
                end = missing_data_activation_time
            
            if not missing_data_activation_time:
                missing_data_activation_time = margin_new_range + end
            
            if cursor and RAW_SLA_KEY.END in cursor and RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME in cursor and cursor[RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME] >= cursor[RAW_SLA_KEY.END] > 0:
                margin_range = cursor[RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME] - cursor[RAW_SLA_KEY.END]
            else:
                margin_range = margin_new_range
            
            next_start = start_of_range if cursor is None else cursor[RAW_SLA_KEY.END]
            if next_start <= 0:
                next_start = start
            
            if (start - next_start) > margin_range:
                cursor = {
                    'name'                        : name,
                    RAW_SLA_KEY.CONTEXT_AND_STATUS: 100 * STATUS.MISSING_DATA,
                    RAW_SLA_KEY.START             : next_start,
                    RAW_SLA_KEY.END               : start,
                }
                if cursor[RAW_SLA_KEY.START] > end_of_range:
                    break
                range_build.append(cursor)
            
            if cursor is None:
                cursor = {
                    u'name'                                 : name,
                    RAW_SLA_KEY.CONTEXT_AND_STATUS          : cache_value,
                    RAW_SLA_KEY.START                       : start_of_range,
                    RAW_SLA_KEY.END                         : end,
                    RAW_SLA_KEY.ACK_UUID                    : ack_uuid,
                    RAW_SLA_KEY.OUTPUT                      : output,
                    RAW_SLA_KEY.LONG_OUTPUT                 : long_output,
                    RAW_SLA_KEY.DOWNTIMES_UUID              : downtimes_uuid,
                    RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME: missing_data_activation_time,
                }
                if cursor[RAW_SLA_KEY.START] > end_of_range:
                    break
                range_build.append(cursor)
            else:
                if cursor[RAW_SLA_KEY.CONTEXT_AND_STATUS] != cache_value:
                    cursor[RAW_SLA_KEY.END] = start
                    cursor = {
                        u'name'                                 : name,
                        RAW_SLA_KEY.CONTEXT_AND_STATUS          : cache_value,
                        RAW_SLA_KEY.START                       : start,
                        RAW_SLA_KEY.END                         : end,
                        RAW_SLA_KEY.ACK_UUID                    : ack_uuid,
                        RAW_SLA_KEY.OUTPUT                      : output,
                        RAW_SLA_KEY.LONG_OUTPUT                 : long_output,
                        RAW_SLA_KEY.DOWNTIMES_UUID              : downtimes_uuid,
                        RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME: missing_data_activation_time,
                    }
                    if cursor[RAW_SLA_KEY.START] > end_of_range:
                        break
                    range_build.append(cursor)
                else:
                    if end >= cursor[RAW_SLA_KEY.END]:
                        cursor[RAW_SLA_KEY.END] = end
                        cursor[RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME] = missing_data_activation_time
        
        if cursor:
            if 0 < cursor[RAW_SLA_KEY.END] < end_of_range:
                if cursor.get(RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME, 0) >= cursor[RAW_SLA_KEY.END] > 0:
                    margin_range = cursor[RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME] - cursor[RAW_SLA_KEY.END]
                else:
                    margin_range = margin_new_range
                
                if cursor[RAW_SLA_KEY.END] > 0 and (end_of_range - cursor[RAW_SLA_KEY.END]) > margin_range:
                    cursor = {
                        u'name'                       : name,
                        RAW_SLA_KEY.CONTEXT_AND_STATUS: 100 * STATUS.MISSING_DATA,
                        RAW_SLA_KEY.START             : cursor[RAW_SLA_KEY.END],
                        RAW_SLA_KEY.END               : end_of_range,
                    }
                    range_build.append(cursor)
            
            cursor[RAW_SLA_KEY.END] = end_of_range
        else:
            cursor = {
                u'name'                       : name,
                RAW_SLA_KEY.CONTEXT_AND_STATUS: 100 * STATUS.MISSING_DATA,
                RAW_SLA_KEY.START             : start_of_range,
                RAW_SLA_KEY.END               : end_of_range,
            }
            range_build.append(cursor)
        
        return range_build
    
    
    @staticmethod
    def _raw_range_to_range(raw_range):
        cache_value = raw_range[RAW_SLA_KEY.CONTEXT_AND_STATUS]
        start = raw_range[RAW_SLA_KEY.START]
        end = raw_range[RAW_SLA_KEY.END]
        ack_uuid = raw_range.get(RAW_SLA_KEY.ACK_UUID, '')
        downtimes_uuid = raw_range.get(RAW_SLA_KEY.DOWNTIMES_UUID, [])
        output = raw_range.get(RAW_SLA_KEY.OUTPUT, None)
        long_output = raw_range.get(RAW_SLA_KEY.LONG_OUTPUT, None)
        
        _range = {
            'ack'     : cache_value % 10,
            'dt'      : cache_value / 10 % 10,
            'rc'      : cache_value / 100 % 10,
            'flg'     : bool(cache_value / 1000 % 10),
            'p_flg'   : bool(cache_value / 10000 % 10),
            'p_ack'   : bool(cache_value / 100000 % 10),
            'p_dt'    : bool(cache_value / 1000000 % 10),
            'ack_uuid': ack_uuid if cache_value % 10 else '',
            'start'   : start,
            'end'     : end
        }
        if downtimes_uuid:
            _range['downtimes_uuid'] = downtimes_uuid
        if output is not None:
            _range['output'] = output
        if long_output is not None:
            _range['long_output'] = long_output
        return _range
