#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.module import Module as ShinkenModuleDefinition
from shinkensolutions.lib_modules.configuration_reader_mixin import ConfigurationReaderMixin, ConfigurationFormat, TypeConfiguration
from sla_abstract_component import AbstractComponent
from sla_common import LIST_STATUS, STATUS, LOG_PART
from sla_component_manager import ComponentManager

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, Union, Dict, Str, Any
    from shinken.objects.module import Module as ShinkenModuleDefinition
    from shinken.objects import Host, Service

RECOMPUTE_OLD_SLA = 1
DO_NOT_RECOMPUTE_OLD_SLA = 0

WARNING_COUNTS_AS_OK = 1
WARNING_COUNTS_AS_WARNING = 0

DOWNTIME_PERIOD_INCLUDE = 0
DOWNTIME_PERIOD_EXCLUDE = 1
DOWNTIME_PERIOD_OK = 2
DOWNTIME_PERIOD_CRITICAL = 3

UNKNOWN_PERIOD_INCLUDE = 0
UNKNOWN_PERIOD_EXCLUDE = 1
UNKNOWN_PERIOD_OK = 2

NO_DATA_PERIOD_INCLUDE = 0
NO_DATA_PERIOD_EXCLUDE = 1
NO_DATA_PERIOD_OK = 2

DOWNTIME_PERIOD_MAP = {
    u'exclude' : DOWNTIME_PERIOD_EXCLUDE,
    u'include' : DOWNTIME_PERIOD_INCLUDE,
    u'ok'      : DOWNTIME_PERIOD_OK,
    u'critical': DOWNTIME_PERIOD_CRITICAL,
}

UNKNOWN_PERIOD_MAP = {
    u'exclude': UNKNOWN_PERIOD_EXCLUDE,
    u'include': UNKNOWN_PERIOD_INCLUDE,
    u'ok'     : UNKNOWN_PERIOD_OK,
}

NO_DATA_PERIOD_MAP = {
    u'exclude': NO_DATA_PERIOD_EXCLUDE,
    u'include': NO_DATA_PERIOD_INCLUDE,
    u'ok'     : NO_DATA_PERIOD_OK,
}


class SLA_TENDENCY(object):
    STAGNANT = u'stagnant'
    INCREASING = u'increasing'
    DECREASING = u'decreasing'


class ComputePercentSla(AbstractComponent, ConfigurationReaderMixin):
    
    def __init__(self, conf, component_manager, prefix_module_property=u''):
        # type: (ShinkenModuleDefinition, ComponentManager, unicode) -> None
        super(ComputePercentSla, self).__init__(conf, component_manager)
        logger_init = self.logger.get_sub_part(LOG_PART.INITIALISATION)
        self.recompute_old_sla = None  # type: Optional[int]
        self.warning_counts_as_ok = None  # type: Optional[int]
        self.downtime_period = None  # type: Optional[int]
        self.unknown_period = None  # type: Optional[int]
        self.no_data_period = None  # type: Optional[int]
        self.str_downtime_period = None  # type: Optional[basestring]
        self.str_unknown_period = None  # type: Optional[basestring]
        self.str_no_data_period = None  # type: Optional[basestring]
        
        configuration_format = [
            ConfigurationFormat([u'recompute_old_sla', u'%srecompute_old_sla' % prefix_module_property], DO_NOT_RECOMPUTE_OLD_SLA, TypeConfiguration.INT, u'recompute_old_sla'),
            ConfigurationFormat([u'warning_counts_as_ok', u'%swarning_counts_as_ok' % prefix_module_property], WARNING_COUNTS_AS_WARNING, TypeConfiguration.INT, u'warning_counts_as_ok'),
            ConfigurationFormat([u'downtime_period', u'%sdowntime_period' % prefix_module_property], u'include', TypeConfiguration.STRING, u'str_downtime_period'),
            ConfigurationFormat([u'unknown_period', u'%sunknown_period' % prefix_module_property], u'include', TypeConfiguration.STRING, u'str_unknown_period'),
            ConfigurationFormat([u'no_data_period', u'%sno_data_period' % prefix_module_property], u'include', TypeConfiguration.STRING, u'str_no_data_period'),
        
        ]
        ConfigurationReaderMixin.__init__(self, configuration_format, conf, logger_init)
        self.read_configuration()
        
        if self.str_downtime_period not in DOWNTIME_PERIOD_MAP.keys():
            self.str_downtime_period = self.handle_error(u'%sdowntime_period' % prefix_module_property, self.str_downtime_period, u'include')
        
        if self.str_unknown_period not in UNKNOWN_PERIOD_MAP.keys():
            self.str_unknown_period = self.handle_error(u'%sunknown_period' % prefix_module_property, self.str_unknown_period, u'include')
        
        if self.str_no_data_period not in NO_DATA_PERIOD_MAP.keys():
            self.str_no_data_period = self.handle_error(u'%sno_data_period' % prefix_module_property, self.str_no_data_period, u'include')
        
        self.downtime_period = DOWNTIME_PERIOD_MAP[self.str_downtime_period]
        self.unknown_period = UNKNOWN_PERIOD_MAP[self.str_unknown_period]
        self.no_data_period = NO_DATA_PERIOD_MAP[self.str_no_data_period]
        
        self.log_configuration(log_properties=True, show_values_as_in_conf_file=True)
    
    
    def init(self):
        pass
    
    
    def tick(self):
        pass
    
    
    @staticmethod
    def compute_state_sum(ranges, info):
        if not ranges or len(ranges) == 0:
            # self.logger.debug('Trying to archive void day')
            # then simulate a void day, with an unknown state, the whole day, so tag this entry that we will 'lie' into it
            info[u'missing'] = True
            info[u'history_inactive'] = 1
            info[u'history_total'] = 1
            return
        
        total_period = 0
        for _range in ranges:
            start = _range[u'start']
            end = _range[u'end']
            status = _range[u'rc']
            period = end - start
            total_period += period
            
            if status == STATUS.OK:
                prefix = u'ok'
            elif status == STATUS.WARN:
                prefix = u'warn'
            elif status == STATUS.CRIT:
                prefix = u'crit'
            elif status == STATUS.UNKNOWN:
                prefix = u'unknown'
            elif status == STATUS.MISSING_DATA:
                prefix = u'missing'
            else:
                prefix = u'inactive'
            
            info[u'history_%s' % prefix] = info.get(u'history_%s' % prefix, 0) + period
            if _range[u'dt']:
                info[u'history_dt_%s' % prefix] = info.get(u'history_dt_%s' % prefix, 0) + period
        info[u'history_total'] = total_period
    
    
    @staticmethod
    def _set_sla_value(archive_day, sla_name, sla_value):
        if sla_value:
            archive_day[sla_name] = sla_value
        elif sla_name in archive_day:
            del archive_day[sla_name]
    
    
    def compute_sla(self, archive_day):
        sla_format = '%s%s%s%s' % (self.warning_counts_as_ok, self.unknown_period, self.no_data_period, self.downtime_period)
        sla_ok = archive_day.get('history_ok', 0)
        # self.logger.debug('sla_ok : history_ok')
        if self.warning_counts_as_ok:
            sla_ok += archive_day.get('history_warn', 0)
            # self.logger.debug('sla_ok : history_warn')
            if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_CRITICAL):
                sla_ok -= archive_day.get('history_dt_warn', 0)
                # self.logger.debug('sla_ok : - history_dt_warn')
        if self.downtime_period == DOWNTIME_PERIOD_OK:
            # We add downtime time but without the downtime OK part which is already in ok_sum
            sla_ok += archive_day.get('history_dt_crit', 0)
            # self.logger.debug('sla_ok : history_dt_crit')
            if self.warning_counts_as_ok == WARNING_COUNTS_AS_WARNING:
                sla_ok += archive_day.get('history_dt_warn', 0)
                # self.logger.debug('sla_ok : history_dt_warn')
            if not self.unknown_period == UNKNOWN_PERIOD_OK:
                sla_ok += archive_day.get('history_dt_unknown', 0)
                # self.logger.debug('sla_ok : history_dt_unknown')
            if not self.no_data_period == NO_DATA_PERIOD_OK:
                sla_ok += archive_day.get('history_dt_missing', 0) + archive_day.get('history_dt_inactive', 0)
                # self.logger.debug('sla_ok : history_dt_missing + history_dt_inactive')
        if self.downtime_period == DOWNTIME_PERIOD_CRITICAL:
            # We remove downtime OK part form ok_sum
            sla_ok -= archive_day.get('history_dt_ok', 0)
            # self.logger.debug('sla_ok : - history_dt_ok')
        if self.downtime_period == DOWNTIME_PERIOD_EXCLUDE:
            # We remove downtime OK part form ok_sum
            sla_ok -= archive_day.get('history_dt_ok', 0)
            # self.logger.debug('sla_ok : - history_dt_ok')
        if self.unknown_period == UNKNOWN_PERIOD_OK:
            sla_ok += archive_day.get('history_unknown', 0)
            # self.logger.debug('sla_ok : history_unknown')
            if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_CRITICAL):
                sla_ok -= archive_day.get('history_dt_unknown', 0)
                # self.logger.debug('sla_ok : - history_dt_unknown')
        if self.no_data_period == NO_DATA_PERIOD_OK:
            sla_ok += archive_day.get('history_missing', 0) + archive_day.get('history_inactive', 0)
            # self.logger.debug('sla_ok : history_missing + history_inactive')
            if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_CRITICAL):
                sla_ok -= (archive_day.get('history_dt_missing', 0) + archive_day.get('history_dt_inactive', 0))
                # self.logger.debug('sla_ok : - history_dt_missing - history_dt_inactive')
        
        sla_crit = archive_day.get('history_crit', 0)
        # self.logger.debug('sla_crit : history_crit[%s]'%history_crit)
        if self.downtime_period == DOWNTIME_PERIOD_CRITICAL:
            sla_crit += \
                archive_day.get('history_dt_ok', 0) + \
                archive_day.get('history_dt_warn', 0) + \
                archive_day.get('history_dt_unknown', 0) + \
                archive_day.get('history_dt_missing', 0) + \
                archive_day.get('history_dt_inactive', 0)
        if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_OK):
            sla_crit -= archive_day.get('history_dt_crit', 0)
        
        sla_warn = 0
        if self.warning_counts_as_ok == WARNING_COUNTS_AS_WARNING:
            sla_warn = archive_day.get('history_warn', 0)
            if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_OK, DOWNTIME_PERIOD_CRITICAL):
                sla_warn -= archive_day.get('history_dt_warn', 0)
        
        sla_unknown = 0
        if self.unknown_period == UNKNOWN_PERIOD_INCLUDE:
            sla_unknown = archive_day.get('history_unknown', 0)
            if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_OK, DOWNTIME_PERIOD_CRITICAL):
                sla_unknown -= archive_day.get('history_dt_unknown', 0)
        elif self.unknown_period == UNKNOWN_PERIOD_EXCLUDE and self.downtime_period == DOWNTIME_PERIOD_INCLUDE:
            sla_unknown = archive_day.get('history_dt_unknown', 0)
        
        sla_missing = 0
        sla_inactive = 0
        if self.no_data_period == NO_DATA_PERIOD_INCLUDE:
            sla_missing = archive_day.get('history_missing', 0)
            sla_inactive = archive_day.get('history_inactive', 0)
            # self.logger.debug('sla_missing : history_missing')
            # self.logger.debug('sla_inactive : history_inactive')
            if self.downtime_period in (DOWNTIME_PERIOD_EXCLUDE, DOWNTIME_PERIOD_OK, DOWNTIME_PERIOD_CRITICAL):
                sla_missing -= archive_day.get('history_dt_missing', 0)
                sla_inactive -= archive_day.get('history_dt_inactive', 0)
                # self.logger.debug('sla_missing : - history_dt_missing')a
                # self.logger.debug('sla_inactive : - history_dt_inactive')
        elif self.no_data_period == NO_DATA_PERIOD_EXCLUDE and self.downtime_period == DOWNTIME_PERIOD_INCLUDE:
            sla_missing = archive_day.get('history_dt_missing', 0)
            sla_inactive = archive_day.get('history_dt_inactive', 0)
            # self.logger.debug('sla_missing : history_dt_missing')
            # self.logger.debug('sla_inactive : history_dt_inactive')
        
        sla_total = archive_day.get('history_ok', 0) + archive_day.get('history_warn', 0) + archive_day.get('history_crit', 0)
        # self.logger.debug('sla_total : history_ok + history_warn + history_crit')
        if self.downtime_period == DOWNTIME_PERIOD_EXCLUDE:
            sla_total -= archive_day.get('history_dt_ok', 0) + archive_day.get('history_dt_warn', 0) + archive_day.get('history_dt_crit', 0)
            # self.logger.debug('sla_total : - history_dt_ok - history_dt_warn - history_dt_crit')
        if self.unknown_period in (UNKNOWN_PERIOD_INCLUDE, UNKNOWN_PERIOD_OK):
            sla_total += archive_day.get('history_unknown', 0)
            # self.logger.debug('sla_total : + history_unknown')
            if self.downtime_period == DOWNTIME_PERIOD_EXCLUDE:
                sla_total -= archive_day.get('history_dt_unknown', 0)
                # self.logger.debug('sla_total : - history_dt_unknown')
        elif self.downtime_period in (DOWNTIME_PERIOD_INCLUDE, DOWNTIME_PERIOD_OK, DOWNTIME_PERIOD_CRITICAL):
            sla_total += archive_day.get('history_dt_unknown', 0)
            # self.logger.debug('sla_total : + history_dt_unknown')
        if self.no_data_period in (NO_DATA_PERIOD_INCLUDE, NO_DATA_PERIOD_OK):
            sla_total += (archive_day.get('history_missing', 0) + archive_day.get('history_inactive', 0))
            # self.logger.debug('sla_total : history_missing + history_inactive')
            if self.downtime_period == DOWNTIME_PERIOD_EXCLUDE:
                sla_total -= archive_day.get('history_dt_missing', 0) + archive_day.get('history_dt_inactive', 0)
                # self.logger.debug('sla_total : - history_dt_missing - history_dt_inactive')
        elif self.downtime_period in (DOWNTIME_PERIOD_INCLUDE, DOWNTIME_PERIOD_OK, DOWNTIME_PERIOD_CRITICAL):
            sla_total += archive_day.get('history_dt_missing', 0) + archive_day.get('history_dt_inactive', 0)
            # self.logger.debug('sla_total : + history_dt_missing + history_dt_inactive')
        # self.logger.debug('sla_ok:[%s] / sla_total:[%s] = [%d]' % (sla_ok, sla_total, sla_ok * 100 / (sla_total if sla_total else 1)))
        
        self._set_sla_value(archive_day, 'sla_ok', sla_ok)
        self._set_sla_value(archive_day, 'sla_crit', sla_crit)
        self._set_sla_value(archive_day, 'sla_warn', sla_warn)
        self._set_sla_value(archive_day, 'sla_unknown', sla_unknown)
        self._set_sla_value(archive_day, 'sla_missing', sla_missing)
        self._set_sla_value(archive_day, 'sla_inactive', sla_inactive)
        
        archive_day['sla_total'] = sla_total
        archive_day['sla_format'] = sla_format
    
    
    def must_update_archive(self, archive):
        sla_format = '%s%s%s%s' % (self.warning_counts_as_ok, self.unknown_period, self.no_data_period, self.downtime_period)
        return self.recompute_old_sla and archive['sla_format'] != sla_format
    
    
    def update_sla(self, archive_day):
        must_update_archive = self.must_update_archive(archive_day)
        
        if must_update_archive:
            self.compute_sla(archive_day)
        
        # for current day we don't have archive entry so we take sla info even if we don't recompute old sla
        if not self.recompute_old_sla and 'archive_format' in archive_day:
            for prefix in LIST_STATUS:
                _sum = archive_day.get('archive_%s' % prefix, 0)
                self._set_sla_value(archive_day, 'sla_%s' % prefix, _sum)
            
            archive_day['sla_total'] = archive_day['archive_total']
            archive_day['sla_format'] = archive_day['archive_format']
        
        return must_update_archive
    
    
    def get_tendency_for_item(self, item, sla_archive_day):
        # type: (Union[Host, Service], Dict[Str, Any]) -> unicode
        
        sla_percent = 0.0
        _total = sla_archive_day['sla_total']
        
        if _total:
            sla_percent = (float(sla_archive_day.get('sla_ok', 0)) / _total)
        
        ranges = sla_archive_day.get('ranges', [])
        first_range = ranges[0] if len(ranges) > 0 else {}
        status = first_range.get('rc', STATUS.MISSING_DATA)
        return self._get_tendency_from_status_and_percent(item, status, sla_percent)
    
    
    def _get_tendency_from_status_and_percent(self, item, status, sla_percent):
        max_sla_percent = 1.0
        min_sla_percent = 0.0
        if item.is_in_downtime():
            if self.downtime_period == DOWNTIME_PERIOD_INCLUDE:
                pass  # If we need to include it, we will check its status instead
            elif self.downtime_period == DOWNTIME_PERIOD_EXCLUDE:
                return SLA_TENDENCY.STAGNANT
            elif self.downtime_period == DOWNTIME_PERIOD_OK:
                return SLA_TENDENCY.INCREASING if sla_percent < max_sla_percent else SLA_TENDENCY.STAGNANT
            elif self.downtime_period == DOWNTIME_PERIOD_CRITICAL:
                return SLA_TENDENCY.DECREASING if sla_percent > min_sla_percent else SLA_TENDENCY.STAGNANT
        
        if status == STATUS.OK and not item.is_flapping:
            return SLA_TENDENCY.INCREASING if sla_percent < max_sla_percent else SLA_TENDENCY.STAGNANT
        
        if status == STATUS.WARN and self.warning_counts_as_ok:
            return SLA_TENDENCY.INCREASING if sla_percent < max_sla_percent else SLA_TENDENCY.STAGNANT
        
        if status == STATUS.UNKNOWN:
            if self.unknown_period == UNKNOWN_PERIOD_INCLUDE:
                return SLA_TENDENCY.DECREASING if sla_percent > min_sla_percent else SLA_TENDENCY.STAGNANT
            if self.unknown_period == UNKNOWN_PERIOD_EXCLUDE:
                return SLA_TENDENCY.STAGNANT
            if self.unknown_period == UNKNOWN_PERIOD_OK:
                return SLA_TENDENCY.INCREASING if sla_percent < max_sla_percent else SLA_TENDENCY.STAGNANT
        
        if status in (STATUS.MISSING_DATA, STATUS.SHINKEN_INACTIVE):
            if self.no_data_period == UNKNOWN_PERIOD_INCLUDE:
                return SLA_TENDENCY.DECREASING if sla_percent > min_sla_percent else SLA_TENDENCY.STAGNANT
            if self.no_data_period == UNKNOWN_PERIOD_EXCLUDE:
                return SLA_TENDENCY.STAGNANT
            if self.no_data_period == UNKNOWN_PERIOD_OK:
                return SLA_TENDENCY.INCREASING if sla_percent < max_sla_percent else SLA_TENDENCY.STAGNANT
        
        # We checked everything, if we're here, the item is down
        return SLA_TENDENCY.DECREASING if sla_percent > min_sla_percent else SLA_TENDENCY.STAGNANT
