#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import time
from datetime import timedelta

from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.date_helper import timestamp_from_datetime
from sla_abstract_component import ThreadComponent
from sla_info import DEFAULT_FIRST_MONITORING_DATE

if TYPE_CHECKING:
    from sla_error_handler import SlaErrorHandler
    from shinken.objects.module import Module as ShinkenModuleDefinition
    from sla_component_manager import ComponentManager
    from sla_reader import SLAReader
    from sla_info import SLAInfo

SAMPLES_TO_STORE = 360


class SLAReaderData:
    def __init__(self):
        self.database_status = True
        self.stored_sla_stats = []
        self.raw_sla_stats = []
        self.history_requests = []
        self.widget_requests = []
        self.report_requests = []
        self.all_requests = []
        self.oldest_sla_date = 0


class SLAReaderStats(ThreadComponent):
    def __init__(self, conf, component_manager, error_handler, sla_info, sla_reader):
        # type: (ShinkenModuleDefinition, ComponentManager, SlaErrorHandler, SLAInfo, SLAReader) -> None
        super(SLAReaderStats, self).__init__(conf, component_manager, error_handler, only_one_thread_by_class=True, stop_thread_on_error=False, loop_speed=10)
        self.sla_info = sla_info
        self.sla_reader = sla_reader
        
        self._data = SLAReaderData()
        self._history_requests = (0, 0)
        self._widget_requests = (0, 0)
        self._report_requests = (0, 0)
        self._requests = []
        
        self._old_date_cache_time = 0
    
    
    def init(self):
        self.start_thread()
    
    
    def get_thread_name(self):
        return u'sla-reader-stats'
    
    
    def loop_turn(self):
        update_time = time.time()
        self._update_oldest_sla_date(update_time)
        self._update_reader_stats(update_time)
        self._update_plugin_stats(update_time)
    
    
    def get_raw_stats(self):
        return self._data.__dict__
    
    
    def add_stat_to_request_list(self, request_data):
        if request_data[0] == u'History':
            self._history_requests = (self._history_requests[0] + 1, self._history_requests[1] + request_data[2])
        elif request_data[0] == u'Widget':
            self._widget_requests = (self._widget_requests[0] + 1, self._widget_requests[1] + request_data[2])
        elif request_data[0] == u'Report':
            self._report_requests = (self._report_requests[0] + 1, self._report_requests[1] + request_data[2])
        self._requests.append(request_data)
    
    
    def _update_oldest_sla_date(self, update_time):
        if update_time - self._old_date_cache_time > 3600:
            self._old_date_cache_time = update_time
            oldest_sla_date = self.sla_info.get_first_monitoring_start_time()
            if oldest_sla_date != DEFAULT_FIRST_MONITORING_DATE:
                self._data.oldest_sla_date = timestamp_from_datetime(oldest_sla_date + timedelta(days=1))
    
    
    def _update_reader_stats(self, update_time):
        self._data.stored_sla_stats.insert(0, (update_time, self.sla_reader.archive_sla_read[0], self.sla_reader.archive_sla_read[1]))
        self._data.raw_sla_stats.insert(0, (update_time, self.sla_reader.raw_sla_read[0], self.sla_reader.raw_sla_read[1]))
        
        # 1 hour of data, 10 seconds per data
        if len(self._data.stored_sla_stats) > SAMPLES_TO_STORE:
            self._data.stored_sla_stats.pop(len(self._data.stored_sla_stats) - 1)
            self._data.raw_sla_stats.pop(len(self._data.raw_sla_stats) - 1)
    
    
    def _update_plugin_stats(self, update_time):
        self._data.history_requests.insert(0, (update_time, self._history_requests[0], self._history_requests[1]))
        self._data.widget_requests.insert(0, (update_time, self._widget_requests[0], self._widget_requests[1]))
        self._data.report_requests.insert(0, (update_time, self._report_requests[0], self._report_requests[1]))
        self._data.all_requests.insert(0, (update_time, self._requests))
        self._requests = []
        
        # 1 hour of data, 10 seconds per data
        if len(self._data.history_requests) > SAMPLES_TO_STORE:
            self._data.history_requests.pop(len(self._data.history_requests) - 1)
            self._data.widget_requests.pop(len(self._data.widget_requests) - 1)
            self._data.report_requests.pop(len(self._data.report_requests) - 1)
            self._data.all_requests.pop(len(self._data.all_requests) - 1)
