#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2021
# This file is part of Shinken Enterprise, all rights reserved.
import math
import time

from component.sla_archive import SLAArchive
from component.sla_compute_percent_sla import ComputePercentSla
from component.sla_database import SLADatabase
from component.sla_info import SLAInfo
from shinken.exceptions.business import ShinkenExceptionValueError, ShinkenExceptionKeyError
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_sub_module.base_sub_module_livedata import BaseSubModuleLivedata
from shinken.util import safe_add_to_dict
from shinkensolutions.broker.monitoring_item_manager.filter import RequestManagerLivedataV2, _FilterPagination
from shinkensolutions.date_helper import parse_string_to_date, date_now, from_to_date_generator, format_date_to_string, Date, compare_date, DATE_COMPARE, date_from_datetime, get_previous_date
from shinkensolutions.lib_modules.configuration_reader_mixin import ConfigurationReaderMixin, SeparatorFormat, ConfigurationFormat, TypeConfiguration
from sla.component.sla_common import shared_data, LOG_PART
from sla.component.sla_component_manager import ComponentManager
from sla.component.sla_database_connection import SLADatabaseConnection

if TYPE_CHECKING:
    from shinken.objects.contact import Contact
    from shinken.misc.type_hint import Optional, Any, Dict, List, Tuple
    from shinkensolutions.date_helper import Date
    from shinken.objects.module import Module as ShinkenModuleDefinition


class OutputFormat(object):
    CHECKS_ATTACHED_TO_FATHER = u'checks_attached_to_father'
    ELEMENTS_ON_SAME_LEVEL = u'elements_on_same_level'
    LIST_OF_SLA = u'list_of_sla'


ENABLED_PARAMETERS = [u'filter', u'output_format', u'output_field', u'period', u'page_settings']

ELEMENTS_COUNT = [
    u'nb_elements_total', u'nb_hosts_total', u'nb_clusters_total', u'nb_checks_total',
    u'nb_elements_filtered', u'nb_hosts_filtered', u'nb_clusters_filtered', u'nb_checks_filtered',
    u'nb_elements_in_page', u'nb_hosts_in_page', u'nb_clusters_in_page', u'nb_checks_in_page',
    u'nb_elements_not_found', u'nb_fathers_not_found', u'nb_checks_not_found'
]

DEFAULT_OUTPUT_FIELDS = [u'check_uuid', u'check_name', u'father_uuid', u'father_name', u'pagination', u'request_statistics']
DEFAULT_OUTPUT_FIELDS_WITH_TYPE = [u'check_uuid', u'check_name', u'father_uuid', u'father_name', u'pagination', u'request_statistics', u'type']
USER_ALLOWED_FILTER_TAG = [u'type', u'father_name', u'father_name_contains', u'father_uuid', u'father_templates', u'check_name', u'check_name_contains', u'check_uuid', u'address', u'realm', u'host_groups', u'notification_contacts',
                           u'notification_contact_groups', u'business_impact']

EXPECTING_FORMAT_ERROR_MESSAGE = u'Expecting ISO format YYYY_MM_DD'
SLA_PROVIDER_DEFAULT_DATE_FORMAT = u'%Y_%m_%d'


def format_list_to_dict(list_to_transform):
    # type: (List[Any]) -> Dict[Any:Any]
    dict_final = {}
    for value in list_to_transform:
        dict_final[value] = value
    return dict_final


class Sla_Pagination(_FilterPagination):
    def __init__(self):
        super(Sla_Pagination, self).__init__()
        self.sla_date_begin = None  # type: Optional[Date]
    
    
    def __str__(self):
        return super(Sla_Pagination, self).__str__().replace(u')', u', sla_date_begin=[%s])' % (u' of '.join(str(date) for date in self.sla_date_begin) if self.sla_date_begin else self.sla_date_begin))


class LivedataModuleSlaProvider(BaseSubModuleLivedata, ConfigurationReaderMixin):
    def __init__(self, module_configuration):
        # type: (ShinkenModuleDefinition) -> None
        super(LivedataModuleSlaProvider, self).__init__(module_configuration)
        logger_init = self.logger.get_sub_part(LOG_PART.INITIALISATION)
        logger_init.info(u'=============     %35s     ==============' % u'Starting module initialisation')
        
        prefix_module_property = u'broker__module_livedata__module_sla_provider'
        
        shared_data.set_default_values(getattr(module_configuration, u'configuration_default_value', {}))
        error_handler = None
        
        configuration_format = [
            SeparatorFormat(u'Module Identity'),
            ConfigurationFormat(u'module_name', u'', TypeConfiguration.STRING, u'')
        ]
        ConfigurationReaderMixin.__init__(self, configuration_format, module_configuration, logger_init)
        self.read_configuration()
        self.log_configuration(log_properties=True, show_values_as_in_conf_file=True)
        
        self.component_manager = ComponentManager(self.logger)  # type: ComponentManager
        self.sla_database_connection = SLADatabaseConnection(module_configuration, self.component_manager, log_database_parameters=True, prefix_module_property=prefix_module_property)  # type: SLADatabaseConnection
        self.compute_percent_sla = ComputePercentSla(module_configuration, self.component_manager, prefix_module_property=prefix_module_property)  # type: ComputePercentSla
        self.sla_info = SLAInfo(module_configuration, self.component_manager, error_handler, self.sla_database_connection)  # type: SLAInfo
        self.sla_database = SLADatabase(module_configuration, self.component_manager, self.sla_database_connection, self.sla_info)  # type: SLADatabase
        self.sla_archive = SLAArchive(module_configuration, self.component_manager, self.sla_info, self.compute_percent_sla, self.sla_database, prefix_module_property=prefix_module_property)  # type: SLAArchive
    
    
    def init(self):
        # type: () -> None
        start_time = time.time()
        logger_init = self.logger.get_sub_part(LOG_PART.INITIALISATION)
        logger_init.info(u'Reading module configuration')
        self.component_manager.init()
        logger_init.info(u'=============     %35s     ==============' % (u'Module initialized in %s' % self.logger.format_chrono(start_time)))
    
    
    def get_routes(self):
        # type: () -> Dict
        # we can't send directly self.get_all_sla() because it's a class method and it's immutable
        def sla_api():
            # type: () -> unicode
            self.response.content_type = u'application/json'
            user_token = self.request.headers.get(u'x-api-token', u'')
            
            if not isinstance(user_token, unicode):
                user_token = user_token.decode(u'UTF-8')
            
            if not isinstance(self.token, unicode):
                self.token = self.token.decode(u'UTF-8')
            
            if self.token != user_token:
                return self.abort(400, u'Wrong token')
            
            # STEP 1 : GET user parameters
            user_filters, page_settings, output_format, output_field, start_date, end_date = self._parse_request()
            
            # User and User token will be implemented on Livedata V2
            user = None
            
            # STEP 2 : Create Livedata Filter Item
            request_manager_livedata_v2 = self._build_filter_item(user_filters, user, output_format, output_field, page_settings)
            
            sla_pagination = None  # type: Optional[Sla_Pagination]
            # If output_format == list_of_sla, we have to create our pagination because of the spread of sla data over several days
            if output_format == OutputFormat.LIST_OF_SLA and request_manager_livedata_v2.page_size and request_manager_livedata_v2.page_size >= 1:
                sla_pagination = self._get_sla_pagination(request_manager_livedata_v2, start_date, end_date)
            
            # STEP 3 : GET filtered elements
            filtered_elements = request_manager_livedata_v2.get_list_elements(jsonify_elements=False, specific_module_pagination=sla_pagination)
            
            # STEP 4 : Request SLA for elements requested
            self.lock.release()
            by_uuid_sla_archives = self._search_sla(filtered_elements[u'elements_found'], output_format, start_date, end_date)
            
            # STEP 5 : Modify the output by adding sla
            self._add_sla_to_output(request_manager_livedata_v2, filtered_elements, output_format, by_uuid_sla_archives, start_date, end_date, sla_pagination)
            
            return filtered_elements
        
        
        pages = {
            sla_api: {u'routes': [u'/api/v1/sla'], u'view': None, u'method': u'ANY', u'wrappers': [u'json']}
        }
        return pages
    
    
    @staticmethod
    def _get_sla_pagination(request_manager_livedata_v2, start_date, end_date):
        # type: (RequestManagerLivedataV2, Date, Date) -> Sla_Pagination
        
        sla_pagination = Sla_Pagination()
        sla_days = [date for date in from_to_date_generator(start_date, end_date)]
        nb_sla_days = len(sla_days)
        nb_sla_from_the_beginning = request_manager_livedata_v2.page_size * request_manager_livedata_v2.page
        
        # If an item is split between two pages, we want to know which date the page starts with
        sla_pagination.sla_date_begin = sla_days[nb_sla_from_the_beginning % nb_sla_days]
        
        end_page = int(math.ceil(float(nb_sla_from_the_beginning) / float(nb_sla_days)))
        
        # Two case : 1- Page size >= SLA period      2- Page size < SLA period
        if request_manager_livedata_v2.page_size >= nb_sla_days:
            # An element may be split at the end of the last page, we have to end it in actual page
            start_page = end_page if nb_sla_from_the_beginning % nb_sla_days == 0 else end_page - 1
            nb_sla_from_the_beginning += request_manager_livedata_v2.page_size
            end_page = int(math.ceil(float(nb_sla_from_the_beginning) / float(nb_sla_days)))
        else:
            # Because an element is split into a lot of pages, we compute the float element per page to know if we need another element in our page
            nb_element_per_page = float(request_manager_livedata_v2.page_size) / float(nb_sla_days)
            nb_element_from_the_beginning = nb_element_per_page * request_manager_livedata_v2.page
            
            start_page = int(nb_element_per_page * request_manager_livedata_v2.page)
            nb_element_from_the_beginning += nb_element_per_page
            end_page = end_page if nb_element_from_the_beginning <= end_page else end_page + 1
        
        sla_pagination.pages.append({u'start': start_page, u'end': end_page})
        return sla_pagination
    
    
    def _parse_request(self):
        # type: () -> (unicode, unicode, unicode, List, Date, Date)
        user_filters = None  # type: Optional[unicode]
        page_settings = None  # type: Optional[unicode]
        output_format = None  # type: Optional[unicode]
        output_field = None  # type: Optional[List]
        start_date = None  # type: Optional[Date]
        end_date = None  # type: Optional[Date]
        
        try:
            RequestManagerLivedataV2.validate_bottle_post_parameter(self.request.POST, set(ENABLED_PARAMETERS))
            user_filters = RequestManagerLivedataV2.build_filters_from_bottle_post(self.request.POST)
            output_format = self.request.POST.get(u'output_format', OutputFormat.ELEMENTS_ON_SAME_LEVEL)
            period = self.request.POST.get(u'period', u'')
            page_settings = self.request.POST.get(u'page_settings', u'')
            output_field = self.request.POST.get(u'output_field', u'')
            
            start_date, end_date = self._validate_sla_period(period)
        except (ShinkenExceptionKeyError, ShinkenExceptionValueError) as e:
            self.abort(e.code, u'ERROR %s: %s' % (e.code, e.text))
        
        return user_filters, page_settings, output_format, output_field, start_date, end_date
    
    
    def _validate_sla_period(self, period):
        # type: (unicode) -> Tuple(Date)
        raw_start_date = None
        raw_end_date = None
        yesterday_date = get_previous_date(date_now())
        
        if period:
            period = period.split(u'~')
            if len(period) > 2:
                raise ShinkenExceptionValueError(400, u'period: invalid format')
            raw_start_date = period[0].split(u':')
            if len(raw_start_date) < 2 or len(raw_start_date) > 2:
                raise ShinkenExceptionValueError(400, u'period: start period has an invalid format')
            
            raw_end_date = period[1].split(u':') if len(period) > 1 else u''
            if len(raw_start_date) < 2 or len(raw_end_date) > 2:
                raise ShinkenExceptionValueError(400, u'period: end period has an invalid format')
            
            raw_start_date = raw_start_date[1]
            raw_end_date = raw_end_date[1] if len(raw_end_date) > 1 else u''
        
        if raw_start_date:
            try:
                start_date = parse_string_to_date(raw_start_date, str_format=SLA_PROVIDER_DEFAULT_DATE_FORMAT)
            except ValueError:
                raise ShinkenExceptionValueError(400, u'period: start period wrong format. %s.' % EXPECTING_FORMAT_ERROR_MESSAGE)
            if start_date[1] < 1900:
                raise ShinkenExceptionValueError(400, u'period: start period must be more or equal than 1900')
        else:
            start_date = yesterday_date
        
        if raw_end_date:
            try:
                end_date = parse_string_to_date(raw_end_date, str_format=SLA_PROVIDER_DEFAULT_DATE_FORMAT)
            except ValueError:
                raise ShinkenExceptionValueError(400, u'period: end period wrong format. %s.' % EXPECTING_FORMAT_ERROR_MESSAGE)
            if end_date[1] < 1900:
                raise ShinkenExceptionValueError(400, u'period: end period must be more or equal than 1900')
        else:
            end_date = start_date
        
        oldest_sla_date = date_from_datetime(self.sla_info.get_first_monitoring_start_time())
        if compare_date(oldest_sla_date, start_date) == DATE_COMPARE.IS_BEFORE:
            raise ShinkenExceptionValueError(400, u'period: the start period is not valid, as there is no SLA data for this date. You can filter elements from %s.' % format_date_to_string(oldest_sla_date, str_format=SLA_PROVIDER_DEFAULT_DATE_FORMAT))
        
        if compare_date(yesterday_date, end_date) == DATE_COMPARE.IS_AFTER:
            raise ShinkenExceptionValueError(400, u'period: the end period is invalid, as the requested period is in the future. You can filter elements until %s.' % format_date_to_string(yesterday_date, str_format=SLA_PROVIDER_DEFAULT_DATE_FORMAT))
        
        if compare_date(end_date, start_date) == DATE_COMPARE.IS_AFTER:
            raise ShinkenExceptionValueError(400, u'period: the end period is invalid, as it\'s less than the start period.')
        
        return start_date, end_date
    
    
    def _build_filter_item(self, user_filters, user, output_format, output_field, page_settings):
        # type: (unicode, Optional[Contact], unicode, List[unicode], unicode) -> RequestManagerLivedataV2
        request_manager_livedata_v2 = None  # type: Optional[RequestManagerLivedataV2]
        # Element property "type" is optional for the "CHECKS_ATTACHED_TO_FATHER" output format as the elements are ordered by type, but mandatory for "ELEMENTS_ON_SAME_LEVEL" and "LIST_OF_SLA"
        default_output_fields = DEFAULT_OUTPUT_FIELDS_WITH_TYPE if output_format != OutputFormat.CHECKS_ATTACHED_TO_FATHER else DEFAULT_OUTPUT_FIELDS
        default_output_fields.extend(ELEMENTS_COUNT)
        
        # FilterItem cannot have a default value for output_format as the LivedataV2 routes do not have the same default value, so we build our own bottle_post with default_value if the value is not set.
        bottle_post = {u'output_format': output_format, u'output_field': output_field}
        
        try:
            request_manager_livedata_v2 = RequestManagerLivedataV2(monitoring_item_manager=self.monitoring_item_manager,
                                                                   user=user,
                                                                   mandatory_output_fields=default_output_fields,
                                                                   user_allowed_output_field=[u'type'],
                                                                   user_filters=user_filters,
                                                                   user_allowed_filters=USER_ALLOWED_FILTER_TAG,
                                                                   page_settings=page_settings,
                                                                   bottle_post=bottle_post,
                                                                   additional_format=OutputFormat.LIST_OF_SLA,
                                                                   compute_not_found=True)
        except (ShinkenExceptionKeyError, ShinkenExceptionValueError) as e:
            self.abort(e.code, u'ERROR %s: %s' % (e.code, e.text))
        
        return request_manager_livedata_v2
    
    
    @staticmethod
    def _format_sla_archive(sla_archive):
        # type: (Dict[unicode, Any]) -> Dict[unicode, Any]
        host_uuid = sla_archive[u'uuid']
        host_uuid = host_uuid.split(u'-')[0] if u'-' in host_uuid else host_uuid
        sla_date = Date(sla_archive[u'yday'], sla_archive[u'year'])
        if host_uuid is None:
            return {
                u'sla_total': u'N.A.',
                u'sla_ok'   : u'N.A.'
            }
        return {
            u'sla_total'     : sla_archive.get(u'sla_total', 0),
            u'sla_ok'        : sla_archive.get(u'sla_ok', 0),
            u'sla_crit'      : sla_archive.get(u'sla_crit', 0),
            u'sla_warn'      : sla_archive.get(u'sla_warn', 0),
            u'sla_unknown'   : sla_archive.get(u'sla_unknown', 0),
            u'sla_missing'   : sla_archive.get(u'sla_missing', 0),
            u'sla_inactive'  : sla_archive.get(u'sla_inactive', 0),
            # Python 2.6 doesn't handle rounding, instead of 99.1 it will display 99.1000000000000001
            u'sla_thresholds': [float(u'%.3f' % sla_threshold) for sla_threshold in sla_archive.get(u'thresholds', shared_data.get_default_sla_thresholds())],
            u'sla_date'      : format_date_to_string(sla_date, str_format=SLA_PROVIDER_DEFAULT_DATE_FORMAT)
        }
    
    
    def _search_sla(self, filtered_elements, output_format, start_date, end_date):
        # type: (Dict, unicode, Date, Date) -> Dict
        if output_format == OutputFormat.CHECKS_ATTACHED_TO_FATHER:
            elements_found = filtered_elements.get(u'hosts', [])[:]
            elements_found.extend(filtered_elements[u'clusters'])
        else:
            elements_found = filtered_elements
        item_uuids = [element[u'check_uuid'] if u'check_uuid' in element else element[u'father_uuid'] for element in elements_found]
        where = {u'uuid': {u'$in': item_uuids}}
        
        archives = self.sla_database.find_archives(date_ranges=(start_date, end_date), where=where)
        by_uuid_archives = {}
        for archive in archives:
            safe_add_to_dict(by_uuid_archives, archive[u'uuid'], archive)
        
        return by_uuid_archives
    
    
    def _add_sla_to_output(self, request_manager_livedata_v2, filtered_elements, output_format, by_uuid_archives, start_date, end_date, sla_pagination):
        # type: (RequestManagerLivedataV2, Dict, unicode, Dict, Date, Date, Sla_Pagination) -> None
        sla_entries = []
        pagination = filtered_elements.get(u'pagination', {})
        # The output format "CHECKS_ATTACHED_TO_FATHER" is not formatted like the others, the filtered items are classified into two categories hosts and clusters
        if output_format == OutputFormat.CHECKS_ATTACHED_TO_FATHER:
            updated_request_statistics = {u'nb_sla_in_page': 0}
            
            for element_type, elements in filtered_elements[u'elements_found'].iteritems():
                for element in elements:
                    # If the filter asks us to return only items of type check, we should not display the SLA data of the parent
                    display_father_sla = not bool(request_manager_livedata_v2.filters)
                    if not display_father_sla:
                        display_father_sla = next((False for user_filters in request_manager_livedata_v2.filters for _filter in user_filters if u'extended_type' in _filter), True)
                    for user_filter in request_manager_livedata_v2.filters:
                        for item_type in user_filter.get(u'extended_type', []):
                            display_father_sla = True if item_type == element_type[:-1] else display_father_sla
                    
                    element_checks = element.get(u'checks', [])
                    for check in element_checks:
                        check_uuid = check[u'check_uuid']
                        self._assign_sla_to_parents(request_manager_livedata_v2, check, check_uuid, by_uuid_archives, output_format, start_date, end_date, updated_request_statistics)
                    
                    if display_father_sla:
                        father_uuid = element[u'father_uuid']
                        self._assign_sla_to_parents(request_manager_livedata_v2, element, father_uuid, by_uuid_archives, output_format, start_date, end_date, updated_request_statistics)
        else:
            updated_request_statistics = {u'nb_elements_in_page': 0, u'nb_hosts_in_page': 0, u'nb_clusters_in_page': 0, u'nb_checks_in_page': 0, u'nb_sla_in_page': 0}
            for element in filtered_elements[u'elements_found']:
                item_uuid = element[u'check_uuid'] if u'check_uuid' in element else element[u'father_uuid']
                element_type = element[u'type']
                
                if element_type == u'host':
                    updated_request_statistics[u'nb_hosts_in_page'] += 1
                elif element_type == u'cluster':
                    updated_request_statistics[u'nb_clusters_in_page'] += 1
                elif element_type == u'check_host' or element_type == u'check_cluster':
                    updated_request_statistics[u'nb_checks_in_page'] += 1
                updated_request_statistics[u'nb_elements_in_page'] += 1
                
                sla_entries = self._assign_sla_to_parents(request_manager_livedata_v2, element, item_uuid, by_uuid_archives, output_format, start_date, end_date, updated_request_statistics, sla_entries, sla_pagination)
                if output_format == OutputFormat.LIST_OF_SLA and request_manager_livedata_v2.page_size and request_manager_livedata_v2.page_size == len(sla_entries):
                    break
            if output_format == OutputFormat.LIST_OF_SLA:
                nb_elements_total = filtered_elements[u'request_statistics'][u'nb_elements_filtered']
                nb_sla_days = len([date for date in from_to_date_generator(start_date, end_date)])
                if pagination:
                    pagination[u'nb_total_page'] = int(math.ceil((nb_elements_total * nb_sla_days) / float(request_manager_livedata_v2.page_size)))
                    pagination[u'has_next_page'] = u'true' if pagination[u'nb_total_page'] > (request_manager_livedata_v2.page + 1) else u'false'
                filtered_elements[u'elements_found'] = sla_entries
        self._update_request_statistics(filtered_elements[u'request_statistics'], updated_request_statistics)
    
    
    def _assign_sla_to_parents(self, request_manager_livedata_v2, element, item_uuid, by_uuid_archives, output_format, start_date, end_date, updated_request_statistics, sla_entries=None, sla_pagination=None):
        # type: (RequestManagerLivedataV2, Dict, unicode, Dict, unicode, Date, Date, Dict, Optional[List], Sla_Pagination) -> (int, Optional[List[Dict]])
        item_archives = by_uuid_archives.get(item_uuid, [])
        by_date_archives = dict(((Date(e[u'yday'], e[u'year']), e) for e in item_archives))
        if not sla_entries:
            sla_entries = []
        for date in from_to_date_generator(start_date, end_date):
            if sla_pagination:
                if sla_pagination.sla_date_begin and date != sla_pagination.sla_date_begin:
                    continue
                sla_pagination.sla_date_begin = None
            updated_request_statistics[u'nb_sla_in_page'] += 1
            sla_archive = by_date_archives.get(date, None)
            if not sla_archive:
                sla_archive = self.sla_archive.build_archive_for_missing_day(date, item_uuid, sla_thresholds=shared_data.get_default_sla_thresholds())
            if output_format == OutputFormat.LIST_OF_SLA:
                element.update(self._format_sla_archive(sla_archive))
                sla_entries.append(element.copy())
                if request_manager_livedata_v2.page_size and request_manager_livedata_v2.page_size == len(sla_entries):
                    break
            else:
                if u'sla' not in element:
                    element[u'sla'] = []
                element[u'sla'].append(self._format_sla_archive(sla_archive))
        
        return sla_entries
    
    
    @staticmethod
    def _update_request_statistics(request_statistics, updated_request_statistics):
        # type: (Dict, Dict) -> None
        for counter_name, nb_element in updated_request_statistics.iteritems():
            request_statistics[counter_name] = nb_element
