#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.
import os
import random
import time

from component.sla_archive import SLAArchive
from component.sla_common import LOG_PART
from component.sla_common import shared_data
from component.sla_component_manager import ComponentManager
from component.sla_compute_percent_sla import ComputePercentSla
from component.sla_database import SLADatabase
from component.sla_database_connection import SLADatabaseConnection
from component.sla_info import SLAInfoNoThread
from shinken.basesubprocess import BaseSubProcess
from shinken.log import LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.module import Module as ShinkenModuleDefinition
from shinken.subprocess_helper.error_handler import ERROR_LEVEL
from shinkensolutions.date_helper import Date, get_now, date_now, get_previous_date, get_diff_date, get_start_of_day, get_end_of_day, timestamp_from_date
from shinkensolutions.lib_modules.configuration_reader_mixin import ConfigurationReaderMixin, ConfigurationFormat, TypeConfiguration
from sla.component.sla_collections_stats import SLACollectionsStats

if TYPE_CHECKING:
    from component.sla_error_handler import SlaErrorHandler
    from shinken.misc.type_hint import Optional

DEFAULT_LOOP_SPEED = 60  # in sec
STARTUP_PAUSE_DELAY = 3600  # in sec
SIZE_CHUNK_TO_ARCHIVE = 10000
TIME_BETWEEN_TWO_CHUNKS = 0.1
KEEP_RAW_SLA_DAY = 7
PATH_FILE_MANUAL_ARCHIVE_TO_REBUILD = u'/var/lib/shinken/tmp/manual_archive_to_rebuild'


class Archivator(BaseSubProcess, ConfigurationReaderMixin):
    
    def __init__(self, daemon_name, module_name, conf, error_handler):
        # type: (unicode, unicode, ShinkenModuleDefinition, SlaErrorHandler) -> None
        super(Archivator, self).__init__(u'archive', father_name=daemon_name, loop_speed=10, only_one_process_by_class=True, error_handler=error_handler)
        self.conf = conf
        self.module_name = module_name
        self.logger = LoggerFactory.get_logger(conf.get_name())
        self._error_handler = error_handler
        
        self.component_manager = None  # type: Optional[ComponentManager]
        self.sla_database_connection = None  # type: Optional[SLADatabaseConnection]
        self.sla_info = None  # type: Optional[SLAInfoNoThread]
        self.sla_database = None  # type: Optional[SLADatabase]
        self.sla_archive = None  # type: Optional[SLAArchive]
        self.compute_percent_sla = None  # type: Optional[ComputePercentSla]
        self.sla_collections_stats = None  # type: Optional[SLACollectionsStats]
        
        self.size_chunk_to_archive = None  # type: Optional[int]
        self.time_between_two_chunks = None  # type: Optional[int]
        self.keep_raw_sla_day = None  # type: Optional[int]
        
        self.total_sla_archived = 0
        self.last_taking_lock = 0
        self.logger_init = self.logger.get_sub_part(LOG_PART.INITIALISATION)
        self.logger_archiving = self.logger.get_sub_part(u'ARCHIVING')
        self.logger_init.info(u'Creating new archive process named : %s' % self.get_process_name())
        self.start_time = None
    
    
    def on_init(self):
        # type: () -> None
        start_time = time.time()
        # re-instancing logger to have pid on archive chapter
        self.logger_init = self.logger.get_sub_part(LOG_PART.INITIALISATION)
        self.logger_archiving = self.logger.get_sub_part(u'ARCHIVING')
        self.logger_init.info(u'Starting initialization of archive process [%s]' % self.get_process_name())
        
        global SIZE_CHUNK_TO_ARCHIVE, TIME_BETWEEN_TWO_CHUNKS, KEEP_RAW_SLA_DAY
        self.component_manager = ComponentManager(self.logger)
        self.sla_database_connection = SLADatabaseConnection(self.conf, self.component_manager)
        self.sla_info = SLAInfoNoThread(self.conf, self.component_manager, self._error_handler, self.sla_database_connection)
        self.sla_database = SLADatabase(self.conf, self.component_manager, self.sla_database_connection, self.sla_info)
        self.compute_percent_sla = ComputePercentSla(self.conf, self.component_manager)
        self.sla_archive = SLAArchive(self.conf, self.component_manager, self.sla_info, self.compute_percent_sla, self.sla_database)
        self.sla_collections_stats = SLACollectionsStats(self.conf, self.component_manager, self.sla_database_connection)
        
        configuration_format = [
            ConfigurationFormat(u'size_chunk_to_archive', SIZE_CHUNK_TO_ARCHIVE, TypeConfiguration.INT, u'size_chunk_to_archive'),
            ConfigurationFormat(u'time_between_two_chunks', TIME_BETWEEN_TWO_CHUNKS, TypeConfiguration.FLOAT, u'time_between_two_chunks'),
            ConfigurationFormat(u'keep_raw_sla_day', KEEP_RAW_SLA_DAY, TypeConfiguration.INT, u'keep_raw_sla_day'),
        ]
        ConfigurationReaderMixin.__init__(self, configuration_format, self.conf, self.logger_init)
        self.read_configuration()
        
        self.logger_init.info(u'Parameter load for sla writing')
        self.log_configuration(log_properties=True, show_values_as_in_conf_file=True)
        
        self.component_manager.init()
        self.logger_init.info(u'Archive process %s initialized in %s' % (self.get_process_name(), self.logger.format_chrono(start_time)))
        self.start_time = start_time
    
    
    def get_process_name(self):
        return u'%s [ - Module: %s - %s ]' % (self.father_name, self.module_name, self.name)
    
    
    def get_logger_name(self):
        return [self.father_name.replace('shinken-', '').strip(), self.module_name, '%s (pid:%s)' % (self.name.upper(), os.getpid())]
    
    
    def get_lock_name(self):
        return ' '.join(self.get_logger_name())
    
    
    def _take_lock(self, date):
        lock_name = self.get_lock_name()
        sla_status = self.sla_database.find_raw_sla_status(date)
        
        my_lock = next(([ln, le] for ln, le in sla_status.get(u'archive_locks', []) if ln == lock_name), None)
        # We refresh the lock at most every 15sec
        _now = get_now()
        if not my_lock or _now - my_lock[1] > 15:
            # We remove lock older than 60sec
            sla_status[u'archive_locks'] = [[ln, le] for ln, le in sla_status.get(u'archive_locks', []) if ln != lock_name and _now - le < 60]
            sla_status[u'archive_locks'].append([lock_name, _now])
            self.sla_database.save_raw_sla_status(date, sla_status)
    
    
    def _check_lock(self, date, sla_status):
        lock_name = self.get_lock_name()
        other_archive_locks = [[ln, le] for ln, le in sla_status.get(u'archive_locks', []) if ln != lock_name and get_now() - le < 30]
        if other_archive_locks:
            error_message = u'Fail to archive collection %s (for date %s) because it was lock by [%s] and i am [%s] (an other process try to archive this day)' % (
                SLADatabaseConnection.get_sla_collection_name(date),
                self.logger.format_sla_date(date),
                ','.join((i for i, _ in other_archive_locks)),
                lock_name
            )
            self._error_handler.handle_exception(error_message, u'archive is lock', self.logger, level=ERROR_LEVEL.ERROR)
            return False
        else:
            return True
    
    
    def _archive_collection(self, date):
        start_time = time.time()
        done_archive = False
        if not self.sla_database.collection_exist(date):
            return
        
        self.logger_archiving.info(u'Archiving the day %s' % self.logger.format_sla_date(date))
        
        sla_status = self.sla_database.find_raw_sla_status(date)
        if not sla_status:
            invalid_col_name = u'invalide_%s_%s_%s' % (date.yday, date.year, get_now())
            self.sla_database.rename_collection(date, invalid_col_name)
            error_message = u'We fail to archive raw sla of the %s because it not have the aliveness module information (sla_status). ' \
                            u'We save your data in mongo collection %s it will be keep %s day. ' \
                            u'Please contact your dedicated support before this time with the log of the broker and a shinken-backup of the sla.' % (
                                self.logger.format_sla_date(date), invalid_col_name, self.keep_raw_sla_day)
            self._error_handler.handle_exception(error_message, u'missing sla_status', self.logger, level=ERROR_LEVEL.ERROR)
            return True
        
        if not self._check_lock(date, sla_status):
            return True
        self._take_lock(date)
        self.sla_info.refresh()
        
        to_archives = self.sla_info.get_all_uuids()
        already_archive = self.sla_database.list_archives_uuid(date)
        if len(already_archive):
            self.logger_archiving.info(u'Next %s element SLA archive for the day %s will be skip because we already found a archive for them.' % (len(already_archive), self.logger.format_sla_date(date)))
            to_archives = list(set(to_archives) - set(already_archive))
        nb_to_archives = len(to_archives)
        
        if nb_to_archives > 0:
            self._stats_archive_start(start_time, nb_to_archives, timestamp_from_date(date))
        nb_elements_archived = 0
        
        nb_chunk_to_archives = nb_to_archives / self.size_chunk_to_archive + 1
        start_of_day = get_start_of_day(date)
        end_of_day = get_end_of_day(date)
        inactive_ranges = self.sla_archive.compute_inactive_ranges(date, start_of_day, end_of_day)
        current_chunk_to_archives = 0
        
        todo_archives = []
        
        self.logger_archiving.info(u'%s items to archive in %s blocks ( max size block : %s )' % (nb_to_archives, nb_chunk_to_archives, self.size_chunk_to_archive))
        while current_chunk_to_archives < nb_chunk_to_archives:
            t0 = time.time()
            max_range = min(nb_to_archives, (current_chunk_to_archives + 1) * self.size_chunk_to_archive)
            min_range = min(nb_to_archives, current_chunk_to_archives * self.size_chunk_to_archive)
            chunk_to_archives = to_archives[min_range:max_range]
            self.logger_archiving.info(u'     -  block %s: archiving %s elements' % (current_chunk_to_archives, len(chunk_to_archives)))
            
            for item_uuid in chunk_to_archives:
                # self.logger.debug('     -  block %s: archiving element %s' % (current_chunk_to_archives, item_uuid))
                archive = self.sla_archive.build_archive(date, item_uuid, inactive_ranges=inactive_ranges)
                if archive is not None:
                    done_archive = True
                    nb_elements_archived += 1
                    shared_data.set_sla_archived_during_current_archive(nb_elements_archived)
                    todo_archives.append(archive)
                    self._take_lock(date)
            
            if len(todo_archives) != 0:
                t1 = time.time()
                self.sla_database.save_archives(todo_archives, update=False)
                self.logger_archiving.info(u'     -  block %s: insert %s archive in mongo done in %s' % (current_chunk_to_archives, len(todo_archives), self.logger.format_chrono(t1)))
            
            self._take_lock(date)
            self.logger_archiving.info(u'     -  block %s: archiving block elements done in %s' % (current_chunk_to_archives, self.logger.format_chrono(t0)))
            todo_archives = []
            current_chunk_to_archives += 1
            time.sleep(self.time_between_two_chunks)
        
        if not shared_data.get_already_archived():
            has_been_archive_col_name = u'has_been_archive_%s_%s' % (date.yday, date.year)
            self.sla_database.rename_collection(date, has_been_archive_col_name)
        
        self.logger_archiving.info(u' %s entries archive for the day %s in %ss nb block %s' % (nb_to_archives, self.logger.format_sla_date(date), self.logger.format_chrono(start_time), nb_chunk_to_archives))
        return done_archive
    
    
    def loop_turn(self):
        self.component_manager.tick()
        # wait +/- 2min (do not have strong computation for some modules at the same time for example)
        self.loop_speed = DEFAULT_LOOP_SPEED + random.randint(0, 60)
        t0 = time.time()
        now = get_now()
        _date_now = date_now()
        tm = time.localtime(now)
        
        # Avoid startup overload, and give time to module to get all initial broks
        if self.start_time and self.start_time <= t0 and (t0 - self.start_time) < STARTUP_PAUSE_DELAY:
            self.logger_archiving.debug(u'Archiving process wait for %s ' % self.logger.format_duration(STARTUP_PAUSE_DELAY - (t0 - self.start_time)))
            return
        
        # we wait 00h05 to make a archive so brok with data at 23h59 which come at 00h01 have time to be manage
        if tm.tm_hour == 0 and tm.tm_min < 5:
            self.logger_archiving.debug(u'Archiving %s wait for 00h05' % self.logger.format_time(now))
            self.loop_speed = 5 * 60
            return
        
        done_archive = self._manual_archive_collection()
        
        date_to_archive = _date_now
        for day_count in range(1, 7):
            date_to_archive = get_previous_date(date_to_archive)
            _done_archive = self._archive_collection(date_to_archive)
            done_archive = done_archive or _done_archive
        
        self._remove_old_raw_sla_collections()
        
        if done_archive:
            time_taken = time.time() - t0
            self._stats_archive_end(time_taken)
            
            # When changing the algorithm to get the number of elements to remove, the refresh of the counter is managed by the sla_collections_stats, so no need to keep ths line below
            self.sla_collections_stats.update_count_elements_in_archive()
        
        self.sla_info.clear()
    
    
    def _manual_archive_collection(self):
        # Sometime we need to rebuild manually SLA.
        # We can use script : sla_recompute_data_for_one_day.py
        # Or collection in file PATH_FILE_MANUAL_ARCHIVE_TO_REBUILD will be rebuild one time
        done_archive = False
        if not os.path.exists(PATH_FILE_MANUAL_ARCHIVE_TO_REBUILD):
            return done_archive
        
        with open(PATH_FILE_MANUAL_ARCHIVE_TO_REBUILD) as f:
            to_archive_lines = f.readlines()
        
        for to_archive_line in to_archive_lines:
            has_been_archive = False
            is_already_archive = u''
            if to_archive_line.startswith(u'has_been_archive_'):
                has_been_archive = True
                to_archive_line = to_archive_line.replace(u'has_been_archive_', u'')
                is_already_archive = u'(is already archived)'
            tmp_to_archive_line = to_archive_line.split(u'_')
            date_to_archive = Date(int(tmp_to_archive_line[0]), int(tmp_to_archive_line[1]))
            
            old_has_been_archive = shared_data.get_already_archived()
            shared_data.set_already_archived(has_been_archive)
            self.logger_archiving.info(u'Start manual archive for %s %s' % (self.logger.format_sla_date(date_to_archive), is_already_archive))
            _done_archive = self._archive_collection(date_to_archive)
            shared_data.set_already_archived(old_has_been_archive)
            done_archive = done_archive or _done_archive
        
        os.remove(PATH_FILE_MANUAL_ARCHIVE_TO_REBUILD)
        return done_archive
    
    
    def _remove_old_raw_sla_collections(self):
        _date_now = date_now()
        for collection_name in self.sla_database.list_name_collections()[:]:
            if collection_name.startswith(u'has_been_archive_') or collection_name.startswith(u'invalide_'):
                split = collection_name.replace(u'has_been_archive_', '').replace(u'invalide_', '')
                split = split.split('_')
                collection_date = Date(int(split[0]), int(split[1]))
                must_del_archive = not (0 < get_diff_date(_date_now, collection_date) <= self.keep_raw_sla_day)
                
                if must_del_archive:
                    self.sla_database.drop_collections(collection_name)
                    self.logger_archiving.info(u'Collection %s with raw data of the %s can be deleted because it was %s days old, value from keep_raw_sla_day.' % (collection_name, self.logger.format_sla_date(collection_date), self.keep_raw_sla_day))
    
    
    def _stats_archive_start(self, start_time, nb_to_archive, progression_date):
        if not shared_data.get_archive_in_progress():
            if shared_data.get_latest_archive_start_time() != 0:
                shared_data.set_previous_archive_start_time(shared_data.get_latest_archive_start_time())
            shared_data.set_latest_archive_start_time(start_time)
            shared_data.set_archive_in_progress(True)
        
        shared_data.set_total_sla_current_archive(nb_to_archive)
        self.total_sla_archived += nb_to_archive
        shared_data.set_sla_archived_during_current_archive(0)
        shared_data.set_archive_progression_date(progression_date)
    
    
    def _stats_archive_end(self, time_taken):
        if shared_data.get_latest_archive_execution_time() != 0:
            shared_data.set_previous_archive_execution_time(shared_data.get_latest_archive_execution_time())
            shared_data.set_previous_archive_sla_archived(shared_data.get_total_sla_archived())
        
        shared_data.set_total_sla_archived(self.total_sla_archived)
        self.total_sla_archived = 0
        
        shared_data.set_latest_archive_execution_time(time_taken)
        shared_data.set_archive_in_progress(False)
        shared_data.set_total_unique_elements_valid(False)
    
    
    def after_fork_cleanup(self):
        self.conf = None
        self.module_name = None
        self.logger = None
        self._error_handler = None
        self.logger_init = None
        self.logger_archiving = None
        for c4 in [self.sla_collections_stats, self.sla_archive, self.compute_percent_sla, self.sla_database, self.sla_info, self.sla_database_connection, self.component_manager]:
            if hasattr(c4, u'after_fork_cleanup'):
                c4.after_fork_cleanup()
        self.component_manager = None
        self.sla_database_connection = None
        self.sla_info = None
        self.sla_database = None
        self.sla_archive = None
        self.compute_percent_sla = None
        self.sla_collections_stats = None
        ConfigurationReaderMixin.after_fork_cleanup(self)
