#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.
import time
import uuid

from component.sla_common import STATUS, RAW_SLA_KEY, FutureState, LOG_PART
from component.sla_component_manager import ComponentManager
from component.sla_database import SLADatabase
from component.sla_database_connection import SLADatabaseConnection, BulkSla
from shinken.brok import BROK_VERSION_V020801P7, BROK_VERSION_V020801P15
from shinken.check import NO_END_VALIDITY
from shinken.log import PartLogger
from shinken.misc.fast_copy import fast_deepcopy, NO_COPY
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_module.brokermoduleworker import BrokerModuleWorker
from shinken.objects.module import Module as ShinkenModuleDefinition
from shinken.subprocess_helper.error_handler import ERROR_LEVEL
from shinken.toolbox.full_status_manipulator import FullStatusManipulator
from shinkensolutions.date_helper import date_from_timestamp, get_now, get_previous_date, get_next_date, date_now, compare_date, DATE_COMPARE, Date, get_start_of_day, get_end_of_day
from shinkensolutions.lib_modules.configuration_reader_mixin import ConfigurationReaderMixin, ConfigurationFormat, TypeConfiguration

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Optional
    from component.sla_error_handler import SlaErrorHandler
    from shinken.brok import Brok
    from shinken.withworkersandinventorymodule import FromModuleToWorkerContainer

_cache_write_downtime = set()

LOOP_SPEED = 1  # in sec
MARGIN_SLA_INACTIVE = 30  # in sec
INITIAL_FUTURE_STATES_NEXT_CLEAN = 60 * 60  # in sec
PERIOD_FUTURE_STATES_CLEAN = 86400  # in sec


class LOG_PART_RAW_DATA(object):
    INITIALISATION = u'INITIALISATION'
    FUTURE_STATE = u'REPORT STATE'


class KEY_CACHE(object):
    START = 0
    END = 1
    STATUS = 2
    ID = 3
    DATE = 4
    MISSING_DATA_ACTIVATION_TIME = 5


class BrokHandlerModuleWorker(BrokerModuleWorker, ConfigurationReaderMixin):
    component_manager = None  # type: ComponentManager
    sla_database_connection = None  # type: SLADatabaseConnection
    sla_database = None  # type: SLADatabase
    error_handler = None  # type: SlaErrorHandler
    minimal_time_before_an_element_become_missing_data = 0  # type: int
    
    _stats_sla_current_minute = (0, 0)
    
    store_output = True
    store_long_output = True
    list_of_stored_output_status = set()
    margin_sla_inactive = None
    
    bulks = {}  # type: Dict[Date, BulkSla]
    future_states = {}  # type: Dict[unicode, FutureState]
    future_states_must_be_save = False
    future_states_last_report = False
    update_sla_state_cache = {}
    future_states_next_clean = 0
    
    logger_init = None  # type: PartLogger
    
    _last_manage_broks = 0
    
    
    def set_from_module_to_worker_container(self, from_module_to_worker_container):
        # type: (FromModuleToWorkerContainer) -> None
        self.error_handler = getattr(from_module_to_worker_container, u'error_handler', None)
        if not self.error_handler:
            raise Exception(u'error_handler is missing in module_extra_info')
    
    
    def init_worker(self, conf):
        # type: (ShinkenModuleDefinition) -> None
        global MARGIN_SLA_INACTIVE
        start_time = get_now()
        logger_init = self.logger.get_sub_part(LOG_PART.INITIALISATION)
        logger_init.info(u'Staring new worker.')
        
        self.component_manager = ComponentManager(self.logger)
        self.sla_database_connection = SLADatabaseConnection(conf, self.component_manager)
        # We don't give a SLAInfo to SLADatabase because we don't need to query archive here and we don't want the thread of SLAInfo
        self.sla_database = SLADatabase(conf, self.component_manager, self.sla_database_connection, None)
        
        self._stats_sla_current_minute = (0, 0)
        
        # Stored output configuration part
        self.store_output = None  # type: Optional[bool]
        self.store_long_output = None  # type: Optional[bool]
        self.list_of_stored_output_status = None  # type: Optional[unicode]
        self.minimal_time_before_an_element_become_missing_data = None  # type: Optional[int]
        
        configuration_format = [
            ConfigurationFormat(u'store_output', True, TypeConfiguration.BOOL, u'store_output'),
            ConfigurationFormat(u'store_long_output', True, TypeConfiguration.BOOL, u'store_long_output'),
            ConfigurationFormat(u'list_of_stored_output_status', u'', TypeConfiguration.LIST, u'list_of_stored_output_status'),
            ConfigurationFormat(u'minimal_time_before_an_element_become_missing_data', 0, TypeConfiguration.INT, u'minimal_time_before_an_element_become_missing_data'),
            ConfigurationFormat(u'time_before_shinken_inactive', MARGIN_SLA_INACTIVE, TypeConfiguration.INT, u'margin_sla_inactive'),
        ]
        ConfigurationReaderMixin.__init__(self, configuration_format, conf, logger_init)
        self.read_configuration()
        self.list_of_stored_output_status = set([STATUS.STATUS_MAP[i] for i in self.list_of_stored_output_status if i in STATUS.STATUS_MAP])
        
        # Internal attribute part
        self.bulks = {}
        self.future_states = {}  # type: Dict[str, FutureState]
        self.future_states_must_be_save = False
        self.future_states_last_report = False
        self.update_sla_state_cache = {}
        self.future_states_next_clean = get_now() + INITIAL_FUTURE_STATES_NEXT_CLEAN
        
        self._last_manage_broks = 0
        
        self.component_manager.init()
        self._load_future_states()
        
        logger_init.info(u'Parameter load for build raw sla')
        self.log_configuration(log_properties=True, show_values_as_in_conf_file=True)
        
        logger_init.info(u'New worker start in %s.' % self.logger.format_chrono(start_time))
    
    
    def manage_service_check_result_brok(self, new_info):
        # type: (Brok) -> None
        # self.logger.debug( 'manage_service_check_result_brok')
        self._handle_brok(new_info)
    
    
    def manage_host_check_result_brok(self, new_info):
        # type: (Brok) -> None
        # self.logger.debug( 'manage_host_check_result_brok')
        self._handle_brok(new_info)
    
    
    def manage_update_service_status_brok(self, new_info):
        # type: (Brok) -> None
        # self.logger.debug( 'manage_update_service_status_brok')
        self._handle_brok(new_info)
    
    
    def manage_update_host_status_brok(self, new_info):
        # type: (Brok) -> None
        # self.logger.debug( 'manage_update_host_status_brok')
        self._handle_brok(new_info)
    
    
    def manage_initial_service_status_brok(self, new_info):
        # type: (Brok) -> None
        # self.logger.debug( 'manage_initial_service_status_brok')
        self._handle_brok(new_info, initial_brok=True)
    
    
    def manage_initial_host_status_brok(self, new_info):
        # type: (Brok) -> None
        # self.logger.debug( 'manage_initial_host_status_brok')
        self._handle_brok(new_info, initial_brok=True)
    
    
    def worker_main(self):
        while not self.interrupted:
            self._tick()
            self.interruptable_sleep(LOOP_SPEED)
    
    
    def _handle_brok(self, brok, initial_brok=False):
        # type: (Brok, Optional[bool]) -> None
        new_info_data = brok.data
        item_uuid = new_info_data[u'instance_uuid']
        brok_version = new_info_data.get(u'brok_version', 0)
        
        check_time = int(new_info_data[u'last_chk'])
        full_status = new_info_data.get(u'current_full_status', None)
        
        if brok_version >= BROK_VERSION_V020801P15:
            check_time_with_context = int(new_info_data.get(u'full_status_change_time', check_time))
        else:
            check_time_with_context = int(new_info_data.get(u'update_time_full_status', check_time))
        
        if check_time_with_context > check_time:
            check_time = check_time_with_context
        
        if full_status is not None:
            state_id = FullStatusManipulator.get_state_from_full_status(full_status=full_status, full_status_change_time=check_time)
        else:
            state_id = new_info_data[u'state_id']
        
        # TODO: MISSING DATA start time mismatch between regenerator and Event/SLA modules, cf. SEF-9992

        # Missing data in initial broks means obsolete data, do not rewrite history with unreliable data (#SEF-9532)
        if initial_brok and state_id == STATUS.MISSING_DATA:
            return
        
        # before first check we force STATUS_MISSING_DATA in state_id
        if new_info_data[u'state'] in (u'PENDING', u'UNREACHABLE'):
            state_id = STATUS.UNKNOWN
        
        if '-' not in item_uuid:
            # Cluster case
            if new_info_data.get(u'got_business_rule', False):
                state_id = new_info_data[u'bp_state']
            # Host case
            elif state_id == STATUS.WARN:
                state_id = STATUS.CRIT
        
        output = new_info_data[u'output']
        long_output = new_info_data[u'long_output']
        
        if not self.store_output or (len(self.list_of_stored_output_status) > 0 and state_id not in self.list_of_stored_output_status):
            output = None
            long_output = None
        if not self.store_long_output:
            long_output = None
        
        downtime = new_info_data.get(u'in_scheduled_downtime', None)
        active_downtime_uuids = new_info_data.get(u'active_downtime_uuids', None)
        inherited_downtime = new_info_data.get(u'in_inherited_downtime', None)
        
        # Invalid or old brok, skip this
        if downtime is None:
            return
        
        dt_value = 0
        if downtime:
            dt_value = 1
            downtimes = new_info_data.get(u'downtimes', None)
            self._create_downtime(active_downtime_uuids, downtimes, brok)
        if inherited_downtime:
            dt_value = 2
        
        ack = 0
        if new_info_data[u'problem_has_been_acknowledged'] and not new_info_data[u'in_inherited_acknowledged']:
            ack = 1
        elif new_info_data[u'in_inherited_acknowledged']:
            ack = 2
        ack_uuid = ''
        
        if new_info_data.get(u'got_business_rule', False):
            check_time = get_now()
        
        if new_info_data[u'acknowledge_id'] is not None:
            ack_uuid = new_info_data[u'acknowledge_id']
        if new_info_data[u'acknowledgement'] is not None:
            self._create_acknowledge(new_info_data[u'acknowledgement'], item_uuid)
        
        state_validity_period = new_info_data[u'state_validity_period']
        
        flapping = new_info_data[u'is_flapping']
        partial_flapping = new_info_data.get(u'is_partial_flapping', False)
        partial_ack = new_info_data.get(u'is_partial_acknowledged', False)
        partial_dt = new_info_data.get(u'in_partial_downtime', False)
        
        if state_validity_period == NO_END_VALIDITY:
            missing_data_activation_time = NO_END_VALIDITY
            state_validity_end_time = NO_END_VALIDITY
        else:
            missing_data_activation_time = new_info_data.get(u'missing_data_activation_time', 0)
            state_validity_end_time = new_info_data.get(u'state_validity_end_time', 0)
            
            if brok_version < BROK_VERSION_V020801P7:
                missing_data_activation_time = (check_time + max(self.minimal_time_before_an_element_become_missing_data, state_validity_period))
                state_validity_end_time = check_time + state_validity_period
            if brok_version >= BROK_VERSION_V020801P15 and full_status and state_id == STATUS.MISSING_DATA:
                # WARNING: when full_status from Scheduler returns MISSING DATA state,
                # the following data will not be reliable due to reschedule
                # we must correct these in Broker (here) (#SEF-9128)
                missing_data_activation_time = 0
                state_validity_end_time = 0
                # state_validity_period = NO_END_VALIDITY  # not used here
        
        # self.logger.debug(u'brok-[%s] [%s] - check_time[%s] next_check[%s] state[%s-%s]:until[%s/%s] brok[%s]' % (brok.type, item_uuid, self.logger.format_time(check_time), self.logger.format_time(new_info_data.get(u'next_chk', 0)), state_id, new_info_data['state'], self.logger.format_duration(state_validity_period), self.logger.format_time(missing_data_activation_time), new_info_data))
        self._update_sla(
            check_time,
            state_id,
            flapping,
            partial_flapping,
            dt_value,
            partial_dt,
            ack,
            partial_ack,
            ack_uuid,
            item_uuid,
            output,
            long_output,
            active_downtime_uuids,
            state_validity_end_time,
            missing_data_activation_time
        )
    
    
    def _do_in_worker_manage_broks_thread_loop(self):
        try:
            super(BrokHandlerModuleWorker, self)._do_in_worker_manage_broks_thread_loop()
            now = get_now()
            if now - self._last_manage_broks > LOOP_SPEED:
                self._check_report_raw_sla()
                self._execute_bulks()
                self._last_manage_broks = now
        except Exception as e:
            self.error_handler.handle_exception('Fatal error caused by : %s' % e, e, self.logger, ERROR_LEVEL.FATAL)
            raise
    
    
    def _execute_bulks(self):
        for date, bulk in self.bulks.iteritems():
            insert_cmp, update_cmp = bulk.bulks_execute()
            self.logger.info('%s raw data update in database for the day %s' % (insert_cmp + update_cmp, self.logger.format_sla_date(date)))
        
        # We keeps in bulks only the bulks of yesterday, today, tomorrow
        bulks_keys_before_filter = self.bulks.keys()
        date = date_now()
        to_keeps = [get_previous_date(date), date, get_next_date(date)]
        self.bulks = dict(filter(lambda (key, _bulk): key in to_keeps, self.bulks.iteritems()))
        bulks_keys_after_filter = self.bulks.keys()
        if len(bulks_keys_before_filter) != len(bulks_keys_after_filter):
            self.logger.info('Size of bulks cache have change [%s]/[%s]' % (bulks_keys_before_filter, bulks_keys_after_filter))
    
    
    def _tick(self):
        try:
            if self.future_states_must_be_save:
                self._save_future_states()
            
            if self.future_states_next_clean < get_now():
                self._clean_future_states()
        except Exception as e:
            self.error_handler.handle_exception('Fatal error caused by : %s' % e, e, self.logger, ERROR_LEVEL.FATAL)
            raise
    
    
    def _create_acknowledge(self, acknowledge, item_uuid):
        acknowledge_id = acknowledge.id
        prev_acknowledge = self.sla_database.find_acknowledge(acknowledge_id)
        if prev_acknowledge is None:
            # ok there is not such acknowledge before, save it
            acknowledge_entry = {'_id': acknowledge_id, 'item_uuid': item_uuid}
            for k in acknowledge.__class__.properties:
                acknowledge_entry[k] = getattr(acknowledge, k)
            self.sla_database.save_acknowledge(acknowledge_entry)
            self.logger.debug('Saving new acknowledge: %s' % acknowledge_entry)
    
    
    def _create_downtime(self, uuid_downtimes, downtimes, brok):
        if not uuid_downtimes:
            return
        for uuid_downtime in uuid_downtimes[:]:
            if uuid_downtime in _cache_write_downtime:
                continue
            
            prev_downtime = self.sla_database.find_downtime(uuid_downtime)
            if prev_downtime:
                _cache_write_downtime.add(uuid_downtime)
                continue
            
            downtime = next((i for i in downtimes if i.uuid == uuid_downtime), None) if downtimes else None
            if not downtime:
                uuid_downtimes.remove(uuid_downtime)
                self.logger.error('This brok referencing the downtime uuid [%s] but we never got a downtime with this uuid' % uuid_downtime)
                brok_data = brok.data
                in_scheduled_downtime = brok_data.get('in_scheduled_downtime', None)
                inherited_downtime = brok_data.get('in_inherited_downtime', None)
                item_name = '%s-%s' % (brok.data['host_name'], brok.data.get('service_description', ''))
                self.logger.error(
                    'Fail to save this downtime from the brok [%s] of [%s] ( brok data => in_scheduled_downtime:[%s] inherited_downtime:[%s] downtimes:[%s] )' % (brok.type, item_name, in_scheduled_downtime, inherited_downtime, downtimes))
                continue
            
            downtime_entry = {
                '_id'    : uuid_downtime,
                'author' : downtime.author,
                'comment': downtime.comment,
            }
            _cache_write_downtime.add(uuid_downtime)
            self.sla_database.save_downtime(downtime_entry)
            self.logger.debug('Saving new downtime: %s' % downtime_entry)
    
    
    def _reset_cache(self):
        self.update_sla_state_cache = {}
    
    
    def _get_bulk_raw_sla(self, date):
        bulk = self.bulks.get(date, None)
        if bulk is None:
            bulk = self.sla_database.build_raw_sla_bulk(date, self._reset_cache)
            self.bulks[date] = bulk
        return bulk
    
    
    def _report_future_state(self, bulk, item_uuid, future_state, start_of_day, end_of_day, missing_data_activation_time):
        to_report = future_state.to_report
        to_report_left = 0
        
        if to_report == NO_END_VALIDITY:
            end_time = NO_END_VALIDITY
            to_report_left = NO_END_VALIDITY
            missing_data_activation_time = missing_data_activation_time
        else:
            end_time = to_report + start_of_day
            if end_time > end_of_day:
                to_report_left = end_time - end_of_day
                end_time = end_of_day
        
        _id = uuid.uuid4().hex
        sla_to_save = {
            '_id'                                   : _id,
            RAW_SLA_KEY.UUID                        : item_uuid,
            RAW_SLA_KEY.CONTEXT_AND_STATUS          : future_state.state,
            RAW_SLA_KEY.START                       : start_of_day,
            RAW_SLA_KEY.END                         : end_time,
            RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME: future_state.missing_data_activation_time,
            RAW_SLA_KEY.OUTPUT                      : future_state.output,
            RAW_SLA_KEY.LONG_OUTPUT                 : future_state.long_output,
        }
        
        if future_state.ack_uuid:
            sla_to_save[RAW_SLA_KEY.ACK_UUID] = future_state.ack_uuid
        if future_state.active_downtime_uuids:
            sla_to_save[RAW_SLA_KEY.DOWNTIMES_UUID] = future_state.active_downtime_uuids
        
        bulk.insert(sla_to_save)
        self.logger.debug(u'report_future_state for item:[%s] with CONTEXT_AND_STATUS:[%s] start:[%s] end:[%s] ack_uuid:[%s] downtimes_uuid:[%s] expire:[%s]' %
                          (item_uuid, future_state.state, self.logger.format_time(start_of_day), self.logger.format_time(end_time), future_state.ack_uuid, future_state.active_downtime_uuids, self.logger.format_time(missing_data_activation_time)))
        return to_report_left
    
    
    def _have_state_in_cache(self, item_uuid, cache_status_value, check_time_date, check_time):
        sla_state_cache = self.update_sla_state_cache.get(item_uuid, None)
        
        have_state_in_cache = \
            sla_state_cache and \
            sla_state_cache[KEY_CACHE.STATUS] == cache_status_value and \
            sla_state_cache[KEY_CACHE.DATE] == check_time_date
        
        if have_state_in_cache:
            if sla_state_cache[KEY_CACHE.END] <= 0:
                return have_state_in_cache
            elif sla_state_cache[KEY_CACHE.MISSING_DATA_ACTIVATION_TIME] > sla_state_cache[KEY_CACHE.END] > 0:
                has_expired = check_time >= sla_state_cache[KEY_CACHE.MISSING_DATA_ACTIVATION_TIME]
            else:
                has_expired = check_time >= (sla_state_cache[KEY_CACHE.END] + self.minimal_time_before_an_element_become_missing_data)
            return not has_expired
        
        return have_state_in_cache
    
    
    def _update_sla(self, check_time, status, flapping, partial_flapping, dt, partial_dt, ack, partial_ack, ack_uuid, item_uuid, output, long_output, active_downtime_uuids, state_validity_end_time, missing_data_activation_time):
        stats_start_execution = get_now()
        check_time_date = date_from_timestamp(check_time)
        end_of_day = get_end_of_day(check_time_date)
        start_time = int(check_time)
        state_validity_end_time = int(state_validity_end_time)
        active_downtime_uuids = tuple(active_downtime_uuids) if active_downtime_uuids else tuple()
        cache_status_value = len(active_downtime_uuids) * 10000000 + partial_dt * 1000000 + partial_ack * 100000 + partial_flapping * 10000 + flapping * 1000 + status * 100 + dt * 10 + ack
        bulk = self._get_bulk_raw_sla(check_time_date)
        future_state = self.future_states.get(item_uuid, None)
        
        if future_state and future_state.date == check_time_date:
            start_of_day = get_start_of_day(check_time_date)
            self._report_future_state(bulk, item_uuid, future_state, start_of_day, end_of_day, missing_data_activation_time)
            del self.future_states[item_uuid]
            self.future_states_must_be_save = True
        
        if state_validity_end_time <= 0:
            self.future_states[item_uuid] = FutureState(get_next_date(check_time_date), NO_END_VALIDITY, cache_status_value, output, long_output, ack_uuid, active_downtime_uuids, missing_data_activation_time)
            self.future_states_must_be_save = True
        elif state_validity_end_time > end_of_day:
            to_report = state_validity_end_time - end_of_day
            state_validity_end_time = end_of_day
            self.future_states[item_uuid] = FutureState(get_next_date(check_time_date), to_report, cache_status_value, output, long_output, ack_uuid, active_downtime_uuids, missing_data_activation_time)
            self.future_states_must_be_save = True
        elif future_state and future_state.date != check_time_date:
            del self.future_states[item_uuid]
            self.future_states_must_be_save = True
        
        # self.logger.debug( 'update_sla[%s] -> [%s]:[%s]' % (item_uuid, print_time(new_value_timestamp), cache_status_value))
        
        if self._have_state_in_cache(item_uuid, cache_status_value, check_time_date, check_time):
            sla_state_cache = self.update_sla_state_cache[item_uuid]
            if sla_state_cache[KEY_CACHE.END] == state_validity_end_time and sla_state_cache[KEY_CACHE.MISSING_DATA_ACTIVATION_TIME] == missing_data_activation_time:
                # self.logger.debug('update_sla[%s] -> broks with same time and state sent in double' % item_uuid)
                return
            # self.logger.debug('update_sla[%s] -> item state [%s] in cache, end_state:[%s]->[%s], expire_state:[%s]->[%s]' % (
            #     item_uuid,
            #     cache_status_value,
            #     self.logger.format_time_as_sla(sla_state_cache[KEY_CACHE.END]),
            #     self.logger.format_time_as_sla(state_validity_end_time),
            #     self.logger.format_time_as_sla(sla_state_cache[KEY_CACHE.MISSING_DATA_ACTIVATION_TIME]),
            #     self.logger.format_time_as_sla(missing_data_activation_time)))
            
            _id = sla_state_cache[KEY_CACHE.ID]
            bulk.update({'_id': _id}, {'$set': {
                RAW_SLA_KEY.END                         : state_validity_end_time,
                RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME: missing_data_activation_time
            }})
            sla_state_cache[KEY_CACHE.END] = state_validity_end_time
            sla_state_cache[KEY_CACHE.MISSING_DATA_ACTIVATION_TIME] = missing_data_activation_time
        else:
            _id = uuid.uuid4().hex
            sla_to_save = {
                '_id'                                   : _id,
                RAW_SLA_KEY.UUID                        : item_uuid,
                RAW_SLA_KEY.CONTEXT_AND_STATUS          : cache_status_value,
                RAW_SLA_KEY.START                       : start_time,
                RAW_SLA_KEY.END                         : state_validity_end_time,
                RAW_SLA_KEY.OUTPUT                      : output,
                RAW_SLA_KEY.LONG_OUTPUT                 : long_output,
                RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME: missing_data_activation_time,
            }
            
            # self.logger.debug('update_sla[%s] -> item state [%s] NOT in cache [%s,%s,%s]' %
            #                   (item_uuid, cache_status_value, self.logger.format_time_as_sla(sla_to_save[RAW_SLA_KEY.START]), self.logger.format_time_as_sla(sla_to_save[RAW_SLA_KEY.END]),
            #                    self.logger.format_time_as_sla(sla_to_save.get(RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME, None))))
            
            if ack:
                sla_to_save[RAW_SLA_KEY.ACK_UUID] = ack_uuid
            if active_downtime_uuids:
                sla_to_save[RAW_SLA_KEY.DOWNTIMES_UUID] = active_downtime_uuids
            
            bulk.insert(sla_to_save)
            self.update_sla_state_cache[item_uuid] = [sla_to_save[RAW_SLA_KEY.START], sla_to_save[RAW_SLA_KEY.END], cache_status_value, _id, check_time_date, sla_to_save[RAW_SLA_KEY.MISSING_DATA_ACTIVATION_TIME]]
        self._stats_sla_current_minute = (self._stats_sla_current_minute[0] + 1, self._stats_sla_current_minute[1] + (get_now() - stats_start_execution))
    
    
    def _check_report_raw_sla(self):
        time_start = get_now()
        nb_items_report = 0
        tm = time.localtime(get_now())
        date = Date(tm.tm_yday, tm.tm_year)
        
        here_are_no_report_done = not self.future_states_last_report
        last_report_was_in_the_past = compare_date(date, self.future_states_last_report) == DATE_COMPARE.IS_BEFORE
        wait_after_00h07 = tm.tm_hour > 0 or tm.tm_min > 7
        if not ((here_are_no_report_done or last_report_was_in_the_past) and wait_after_00h07):
            return
        self.logger.debug(LOG_PART_RAW_DATA.FUTURE_STATE, u'start report_raw_sla for the date:[%s]' % str(date))
        
        start_of_day = get_start_of_day(date)
        end_of_day = get_end_of_day(date)
        bulk = self._get_bulk_raw_sla(date)
        new_future_states = {}  # type: Dict[unicode, FutureState]
        for item_uuid, future_state in self.future_states.iteritems():
            _compare = compare_date(date, future_state.date)
            if _compare == DATE_COMPARE.IS_AFTER:
                new_future_states[item_uuid] = future_state
                continue
            if _compare == DATE_COMPARE.IS_BEFORE:
                # report from the past are just drop
                continue
            
            to_report_left = self._report_future_state(bulk, item_uuid, future_state, start_of_day, end_of_day, future_state.missing_data_activation_time)
            nb_items_report += 1
            self.logger.debug(LOG_PART_RAW_DATA.FUTURE_STATE,
                              u'reporting state of [%s] to the date [%s] and it left to report:[%s] expiring at:[%s]' % (item_uuid, date, to_report_left, self.logger.format_time(future_state.missing_data_activation_time)))
            if to_report_left:
                new_future_states[item_uuid] = FutureState(get_next_date(date), to_report_left, future_state.state, future_state.output, future_state.long_output, future_state.ack_uuid, future_state.active_downtime_uuids,
                                                           future_state.missing_data_activation_time)
        
        self.future_states = new_future_states
        self.future_states_must_be_save = True
        self.future_states_last_report = date
        
        self.logger.info(LOG_PART_RAW_DATA.FUTURE_STATE, u'Reported state of %s elements for day %s in %s' % (nb_items_report, self.logger.format_sla_date(date), self.logger.format_chrono(time_start)))
    
    
    def _save_future_states(self):
        future_states_to_save = fast_deepcopy(self.future_states, additional_dispatcher={FutureState: NO_COPY})
        self.future_states_must_be_save = False
        self.sla_database.save_future_states(future_states_to_save)
        self.logger.debug(LOG_PART_RAW_DATA.FUTURE_STATE, 'future_states have been save')
    
    
    def _load_future_states(self):
        start_time = get_now()
        self.future_states = self.sla_database.load_future_states()
        self.logger.get_sub_part(LOG_PART.INITIALISATION).info('Load previous state of %s elements done in %s.' % (len(self.future_states), self.logger.format_chrono(start_time)))
    
    
    def _clean_future_states(self):
        # At 00:xx there are archive, the db is working hard so we report the clean to the next hour
        tm = time.localtime(get_now())
        if tm.tm_hour == 0:
            self.future_states_next_clean = get_now() + 60 * 60
            return
        
        self.future_states_next_clean = get_now() + PERIOD_FUTURE_STATES_CLEAN
        self.sla_database.clean_future_states()
    
    
    def ask_sla_count(self):
        return self._stats_sla_current_minute
