#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import os
import time
from datetime import datetime, timedelta

from component.sla_common import LOG_PART
from component.sla_common import shared_data
from component.sla_component_manager import ComponentManager
from component.sla_database import SLADatabase
from component.sla_database_connection import SLADatabaseConnection
from component.sla_info import SLAInfoNoThread
from component.sla_migration import SLAMigration, SLAMigrationConfUpdateNotifier, SLAMigrationConfChecker
from shinken.basesubprocess import BaseSubProcess
from shinken.log import LoggerFactory, PART_INITIALISATION
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.module import Module as ShinkenModuleDefinition
from shinken.subprocess_helper.error_handler import ERROR_LEVEL
from shinkensolutions.date_helper import Date
from shinkensolutions.lib_modules.configuration_reader_mixin import ConfigurationReaderMixin, ConfigurationFormat, TypeConfiguration
from sla.component.sla_collections_stats import SLACollectionsStats

if TYPE_CHECKING:
    from component.sla_error_handler import SlaErrorHandler
    from shinken.misc.type_hint import Optional

MIGRATING_BATCH_SIZE = 1000
MIGRATING_PAUSE_TIME = 0.1
DAILY_CLEAN_BATCH_SIZE = 10000
DAILY_CLEAN_PAUSE_TIME = 2
INFINITE_NB_STORED_DAYS = -1
LOOP_SPEED = 10
TIME_WHEN_DELETE_OLD_SLA = '03:02'


class Migrator(BaseSubProcess, ConfigurationReaderMixin):
    
    def __init__(self, daemon_name, module_name, conf, error_handler):
        # type: (unicode, unicode, ShinkenModuleDefinition, SlaErrorHandler) -> None
        super(Migrator, self).__init__(u'migration', father_name=daemon_name, loop_speed=LOOP_SPEED, only_one_process_by_class=True, error_handler=error_handler)
        self.conf = conf
        self.module_name = module_name
        self.logger = LoggerFactory.get_logger(conf.get_name())
        
        self.component_manager = None  # type: Optional[ComponentManager]
        self.sla_info = None  # type: Optional[SLAInfoNoThread]
        self.sla_database_connection = None  # type: Optional[SLADatabaseConnection]
        self.sla_database = None  # type: Optional[SLADatabase]
        self.sla_migration = None  # type: Optional[SLAMigration]
        self.sla_collections_stats = None  # type: Optional[SLACollectionsStats]
        self.sla_migration_conf_checker = None  # type: Optional[SLAMigrationConfChecker]
        
        self.nb_stored_days = 0
        self.configuration_uuid = None
        self.start_archive_migration_time = None
        self.nb_archive_migrate = 0
        self.time_when_delete_old_sla = None
        self._error_handler = error_handler
        
        self.daily_clean_batch_size = None  # type: Optional[int]
        self.daily_clean_pause_time = None  # type: Optional[int]
        self.migrating_batch_size = None  # type: Optional[int]
        self.migrating_pause_time = None  # type: Optional[int]
        
        self.sla_conf_update_notifier = SLAMigrationConfUpdateNotifier(self.conf.get_name())
        
        logger_initialisation = self.logger.get_sub_part(PART_INITIALISATION)
        logger_initialisation.info(u'Creating new migration process named : %s' % self.get_process_name())
    
    
    def get_process_name(self):
        return u'%s [ - Module: %s - %s ]' % (self.father_name, self.module_name, self.name)
    
    
    def get_logger_name(self):
        return [self.father_name.replace(u'shinken-', '').strip(), self.module_name, u'%s (pid:%s)' % (self.name.upper(), os.getpid())]
    
    
    def on_init(self, unittest_skip_init=False):
        # type: (Optional[bool]) -> None
        # re-instancing logger to have pid on migrator chapter
        logger_init = self.logger.get_sub_part(LOG_PART.INITIALISATION)
        logger_init.info(u'Migration process %s initialization ' % self.get_process_name())
        global DAILY_CLEAN_BATCH_SIZE, DAILY_CLEAN_PAUSE_TIME, MIGRATING_BATCH_SIZE, MIGRATING_PAUSE_TIME
        if not unittest_skip_init:
            self.component_manager = ComponentManager(self.logger)
            self.sla_database_connection = SLADatabaseConnection(self.conf, self.component_manager)
            self.sla_info = SLAInfoNoThread(self.conf, self.component_manager, self._error_handler, self.sla_database_connection)
            self.sla_database = SLADatabase(self.conf, self.component_manager, self.sla_database_connection, self.sla_info)
            self.sla_migration = SLAMigration(self.conf, self.component_manager, self.sla_info)
        self.sla_collections_stats = SLACollectionsStats(self.conf, self.component_manager, self.sla_database_connection)
        self.sla_migration_conf_checker = SLAMigrationConfChecker(self.conf, self.component_manager, self._error_handler, self, self.sla_conf_update_notifier)
        
        configuration_format = [
            ConfigurationFormat(u'daily_clean_batch_size', DAILY_CLEAN_BATCH_SIZE, TypeConfiguration.INT, u'daily_clean_batch_size'),
            ConfigurationFormat(u'daily_clean_pause_time', DAILY_CLEAN_PAUSE_TIME, TypeConfiguration.FLOAT, u'daily_clean_pause_time'),
            ConfigurationFormat(u'broker_module_sla_migration_batch_size', MIGRATING_BATCH_SIZE, TypeConfiguration.INT, u'migrating_batch_size'),
            ConfigurationFormat(u'broker_module_sla_migration_pause_time', MIGRATING_PAUSE_TIME, TypeConfiguration.FLOAT, u'migrating_pause_time'),
            ConfigurationFormat(u'nb_stored_days', INFINITE_NB_STORED_DAYS, TypeConfiguration.INT, u'nb_stored_days'),
            ConfigurationFormat(u'time_when_delete_old_SLA', TIME_WHEN_DELETE_OLD_SLA, TypeConfiguration.STRING, u'time_when_delete_old_sla'),
        ]
        ConfigurationReaderMixin.__init__(self, configuration_format, self.conf, logger_init)
        self.read_configuration()
        
        if self.nb_stored_days != INFINITE_NB_STORED_DAYS and self.nb_stored_days < 7:
            logger_init.warning(u'Parameter nb_stored_days set by user at [%s] is too low we set it a minimal value:7' % self.nb_stored_days)
            self.nb_stored_days = 7
        
        try:
            if u':' in self.time_when_delete_old_sla:
                cleanup_time = self.time_when_delete_old_sla.split(':')
            else:
                cleanup_time = [self.time_when_delete_old_sla, 0]
            cleanup_time = map(int, cleanup_time)
            cleanup_hour = cleanup_time[0]
            cleanup_minute = cleanup_time[1]
            if len(cleanup_time) == 2 and 0 <= cleanup_hour <= 23 and 0 <= cleanup_minute <= 59:
                self.time_when_delete_old_sla = [cleanup_hour, cleanup_minute]
            else:
                raise ValueError
        except:
            logger_init.warning(u'Parameter time_when_delete_old_SLA set by user at [%s] is incorrect, value changed to:[03:02]' % self.time_when_delete_old_sla)
            self.time_when_delete_old_sla = [3, 2]
        
        logger_init.info(u'Parameter load for migrating')
        self.log_configuration(log_properties=True, show_values_as_in_conf_file=True)
        
        self.configuration_uuid = None
        self.start_archive_migration_time = None
        self.nb_archive_migrate = 0
        
        self.component_manager.init()
        logger_init.info(u'Migration process %s initialized' % self.get_process_name())
    
    
    def on_close(self):
        pass
    
    
    def loop_turn(self):
        self.component_manager.tick()
        if self.nb_stored_days != INFINITE_NB_STORED_DAYS:
            self._do_daily_clean()
        
        self.sla_database.check_daily_format()
        self._migrate_acknowledge()
        self._migrate_archives()
        self.sla_info.clear()
    
    
    def handle_initial_broks_done(self):
        self.sla_conf_update_notifier.set_last_update()
    
    
    def _do_daily_clean(self):
        logger_daily_clean = self.logger.get_sub_part(LOG_PART.DAILY_CLEAN)
        try:
            t0 = time.time()
            last_stored_timetuple = (datetime.today() + timedelta(days=-self.nb_stored_days)).timetuple()
            last_stored_date = Date(last_stored_timetuple.tm_yday, last_stored_timetuple.tm_year)
            _where = {'$or': [{'$and': [{'year': last_stored_date.year}, {'yday': {'$lt': last_stored_date.yday}}]}, {'year': {'$lt': last_stored_date.year}}]}
            nb_items_to_del = 0
            cleanup_time = time.localtime()
            cleanup_time = time.mktime(
                (cleanup_time.tm_year, cleanup_time.tm_mon, cleanup_time.tm_mday, self.time_when_delete_old_sla[0], self.time_when_delete_old_sla[1], cleanup_time.tm_sec, cleanup_time.tm_wday, cleanup_time.tm_yday, cleanup_time.tm_isdst))
            previous_clean_date = self.sla_info.get_last_clean_time()
            if previous_clean_date:
                previous_clean_is_too_old = not (0 <= (t0 - previous_clean_date) <= (24 * 60 * 60))
                time_to_clean = (previous_clean_date < cleanup_time <= t0)
            else:
                previous_clean_is_too_old = True
                time_to_clean = True
            
            if previous_clean_is_too_old or time_to_clean:
                logger_daily_clean.info(u'Start searching for archives older than %s days' % self.nb_stored_days)
                nb_items_to_del = self.sla_database.count_archives_before_date(last_stored_date)
                self.sla_info.set_last_clean_time(t0)
            
            must_clean = bool(nb_items_to_del)
            if must_clean:
                self._stats_save_start_daily_clean(nb_items_to_del)
                logger_daily_clean.info(u'Archive anterior to %s has been found ( %s records )' % (self.logger.format_sla_date(last_stored_date), nb_items_to_del))
            
            batch_id = 0
            while nb_items_to_del > 0:
                group_ids = self.sla_database.find_archives_before_date(last_stored_date, self.daily_clean_batch_size)
                batch = [i['_id'] for i in group_ids if i is not None]
                self.sla_database.remove_archives({'_id': {'$in': batch}})
                batch_id += 1
                time.sleep(self.daily_clean_pause_time)
                nb_items_to_del -= len(group_ids)
                shared_data.set_nb_sla_left_to_remove(nb_items_to_del)
            
            if must_clean:
                time_taken = time.time() - t0
                self._stats_save_finished_daily_clean(time_taken)
                logger_daily_clean.info(u'Archive anterior to %s has been remove ( %s records ) in %s' % (self.logger.format_sla_date(last_stored_date), nb_items_to_del, self.logger.format_duration(time_taken)))
                
                # When changing the algorithm to get the number of elements to add, the refresh of the counter is managed by the sla_collections_stats, so no need to keep ths line below
                self.sla_collections_stats.update_count_elements_in_archive()
        except Exception as e:
            self._error_handler.handle_exception(u'Removing %s days old SLA archive fail with error : %s. This will be retry in %ssec.' % (self.nb_stored_days, LOOP_SPEED, e), e, self.logger, ERROR_LEVEL.WARNING)
    
    
    def _migrate_acknowledge(self):
        time_start = time.time()
        to_migrate_acknowledge = self.sla_database.find_acknowledge_not_migrate()
        
        new_data = []
        for to_migrate in to_migrate_acknowledge:
            item_uuid = self.sla_info.get_uuid(to_migrate['hname'], to_migrate.get('sdesc', ''))
            if item_uuid:
                to_migrate['item_uuid'] = item_uuid
                del to_migrate['type']
                del to_migrate['hname']
                to_migrate.pop('sdesc', None)
                new_data.append(to_migrate)
        
        if new_data:
            self.logger.info('Must migrate %s acknowledge entry' % len(new_data))
            self.sla_database.update_acknowledges(new_data)
            self.logger.info('Migrate %s acknowledge entry done in %s' % (len(new_data), self.logger.format_chrono(time_start)))
    
    
    def _migrate_archives(self):
        if self.configuration_uuid is None or shared_data.get_migration_archive_done():
            return False
        
        self.start_archive_migration_time = time.time()
        shared_data.set_migration_in_progress(True)
        self.nb_archive_migrate = 0
        
        nb_to_migrate_archive = self.sla_database.count_archives_to_migrate(self.configuration_uuid)
        to_migrate_archives = self.sla_database.find_archives_to_migrate(self.configuration_uuid, self.migrating_batch_size)
        
        if nb_to_migrate_archive != 0:
            self._stats_set_total_sla_to_migrate(nb_to_migrate_archive)
        
        self.logger.info('Need to migrate %s archives' % nb_to_migrate_archive)
        
        while to_migrate_archives:
            for archive in to_migrate_archives:
                self.nb_archive_migrate += 1
                shared_data.set_nb_sla_left_to_migrate(nb_to_migrate_archive - self.nb_archive_migrate)
                self.sla_migration.migrate_archive(archive)
                archive['configuration_uuid'] = self.configuration_uuid
            
            self.sla_database.save_archives(to_migrate_archives, update=True)
            self.logger.info('Migrating in progress %s/%s archives' % (self.nb_archive_migrate, nb_to_migrate_archive))
            to_migrate_archives = self.sla_database.find_archives_to_migrate(self.configuration_uuid, self.migrating_batch_size)
            self.sla_info.clear()  # Avoid useless extra memory consumption with internal cache
            self.interruptable_sleep(self.migrating_pause_time)
        
        time_taken = time.time() - self.start_archive_migration_time
        self._stats_save_finished_migration(time_taken)
        if self.nb_archive_migrate:
            self.logger.info('Migrate %s archive done in:%s.' % (self.nb_archive_migrate, self.logger.format_duration(time_taken)))
        else:
            self.logger.info('No archive need migration. Check done in:%s.' % self.logger.format_duration(time_taken))
        
        self.start_archive_migration_time = None
        self.nb_archive_migrate = 0
    
    
    def after_fork_cleanup(self):
        for c4 in [self.sla_collections_stats, self.sla_migration, self.sla_database, self.sla_info, self.sla_database_connection, self.component_manager, self.sla_migration_conf_checker]:
            if hasattr(c4, u'after_fork_cleanup'):
                c4.after_fork_cleanup()
        self.component_manager = None
        self.sla_database_connection = None
        self.sla_info = None
        self.sla_database = None
        self.sla_migration = None
        self.sla_collections_stats = None
        self.sla_migration_conf_checker = None
        self.sla_conf_update_notifier = None
        self._error_handler = None
        self.conf = None
        self.module_name = None
        self.logger = None
        self.sla_migration_conf_checker = None
        self.sla_conf_update_notifier = None
        ConfigurationReaderMixin.after_fork_cleanup(self)
    
    
    @staticmethod
    def _stats_save_start_daily_clean(nb_items_to_del):
        shared_data.set_daily_clean_in_progress(True)
        shared_data.set_total_sla_to_remove(nb_items_to_del)
        shared_data.set_nb_sla_left_to_remove(nb_items_to_del)
    
    
    @staticmethod
    def _stats_save_finished_daily_clean(time_taken):
        shared_data.set_daily_clean_in_progress(False)
        shared_data.set_execution_time_last_daily_clean(time_taken)
    
    
    @staticmethod
    def _stats_save_finished_migration(time_taken):
        shared_data.set_migration_archive_done(True)
        shared_data.set_migration_in_progress(False)
        shared_data.set_execution_time_last_migration(time_taken)
    
    
    @staticmethod
    def _stats_set_total_sla_to_migrate(value):
        shared_data.set_total_sla_to_migrate(value)
        shared_data.set_nb_sla_left_to_migrate(value)
