#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import logging
import time

from component.sla_common import shared_data, LOG_PART
from component.sla_component_manager import ComponentManager
from component.sla_database import SLADatabase
from component.sla_database_connection import SLADatabaseConnection
from component.sla_error_handler import SlaErrorHandler
from component.sla_info import SLAInfo
from component.sla_writer_stats import SLAWriterStats
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_module.basemodule import ModuleState
from shinken.modules.base_module.brokermodule import WorkerBasedBrokerModule
from shinken.objects.module import Module as ShinkenModuleDefinition
from shinken.subprocess_helper.after_fork_cleanup import after_fork_cleanup
from shinken.subprocess_helper.error_handler import ERROR_LEVEL
from shinkensolutions.date_helper import get_now, date_now
from shinkensolutions.lib_modules.configuration_reader_mixin import ConfigurationReaderMixin, TypeConfiguration, ConfigurationFormat
from shinkensolutions.ssh_mongodb.sshtunnelmongomgr import mongo_by_ssh_mgr
from sla.component.sla_collections_stats import SLACollectionsStats
from sla_archivator import Archivator
from sla_brok_handler import BrokHandlerModuleWorker
from sla_migrator import Migrator

if TYPE_CHECKING:
    from shinken.withworkersandinventorymodule import FromModuleToWorkerContainer
    from shinken.misc.type_hint import Optional

# use in TU
COMPONENT_CREATION = True
ACCEPTED_BROK_TYPES = (u'service_check_result', u'host_check_result', u'update_service_status', u'update_host_status', u'initial_service_status', u'initial_host_status', u'initial_broks_done')
MARGIN_SLA_INACTIVE = 30


class SLAModuleBroker(WorkerBasedBrokerModule, ConfigurationReaderMixin):
    MODULE_WORKER_CLASS = BrokHandlerModuleWorker
    
    
    def __init__(self, configuration, display_info=True):
        # type: (ShinkenModuleDefinition, bool) -> None
        global MARGIN_SLA_INACTIVE
        WorkerBasedBrokerModule.__init__(self, configuration)
        if not display_info:
            self.logger.set_level(logging.ERROR)
        
        self.logger.get_sub_part(LOG_PART.INITIALISATION).info(u'=============     %35s     ==============' % u'Starting module initialisation')
        
        self.error_handler = SlaErrorHandler(self.name)
        self.external_speed_counter = 0
        self.external_done_counter = 0
        self.uri = None  # type: Optional[unicode]
        self._stop_done = False
        
        configuration_format = [
            ConfigurationFormat(u'uri', u'mongodb://localhost/?w=1&fsync=false', TypeConfiguration.STRING, u'uri'),
        ]
        ConfigurationReaderMixin.__init__(self, configuration_format, configuration)
        self.read_configuration()
        
        shared_data.set_default_values(getattr(configuration, u'configuration_default_value', {}))
        
        if COMPONENT_CREATION:
            broker_name = u'broker'
            if hasattr(configuration, u'father_config'):
                broker_name = configuration.father_config.get(u'broker_name', broker_name)
            if not broker_name.startswith(u'shinken'):
                broker_name = u'shinken-%s' % broker_name
            self.migrator = Migrator(broker_name, self.name, configuration, self.error_handler)
            self.archivator = Archivator(broker_name, self.name, configuration, self.error_handler)
            
            # SEF-9050 will free memory, this is no more needed :
            # We start these processes here, to avoid loading the memory with the one consumed by SLA_INFO
            # time.sleep(0.001)
            # self.migrator.start()
            # time.sleep(0.001)  # Do not remove, to avoid PID collision in Process class ....
            # self.archivator.start()
            
            self.component_manager = ComponentManager(self.logger)
            self.sla_database_connection = SLADatabaseConnection(configuration, self.component_manager, log_database_parameters=True)
            self.sla_info = SLAInfo(configuration, self.component_manager, self.error_handler, self.sla_database_connection)
            self.sla_database = SLADatabase(configuration, self.component_manager, self.sla_database_connection, self.sla_info)
            
            sla_collections_stats = SLACollectionsStats(configuration, self.component_manager, self.sla_database_connection, should_maintain_share_item=True)
            self.sla_writer_stats = SLAWriterStats(configuration, self.component_manager, self.error_handler, self.sla_database, self.sla_info, self, sla_collections_stats)
    
    
    def init(self, daemon_display_name=u'broker'):
        # type: (str) -> None
        self._stop_done = False
        
        start_time = time.time()
        logger_init = self.logger.get_sub_part(LOG_PART.INITIALISATION)
        logger_init.info(u'Reading module configuration')
        logger_init.info(u'Creating %s workers' % self._nb_workers)
        
        WorkerBasedBrokerModule.init(self)
        self.component_manager.init()
        self._update_active_range()
        # Already started process won't be launched again at init
        time.sleep(0.001)
        self.migrator.start()
        time.sleep(0.001)
        self.archivator.start()
        logger_init.info(u'=============     %35s     ==============' % (u'Module initialized in %s' % self.logger.format_chrono(start_time)))
    
    
    def want_brok(self, brok):
        return brok.type in ACCEPTED_BROK_TYPES
    
    
    def get_internal_state(self):
        return self.error_handler.get_internal_state()
    
    
    def get_state(self):
        stats = super(SLAModuleBroker, self).get_state()
        errors = self.error_handler.get_errors()
        to_update_stats = {
            u'status'       : self.error_handler.get_internal_state(),
            u'output'       : [error.message for error in errors],
            u'mongodb_stats': mongo_by_ssh_mgr.check_connexion_mongodb(self.uri)
        }
        stats.update(to_update_stats)
        return stats
    
    
    # -------------------- Broker part
    
    def manage_initial_service_status_brok(self, new_info):
        self.sla_info.handle_brok(u'service', new_info)
    
    
    def manage_initial_host_status_brok(self, new_info):
        self.sla_info.handle_brok(u'host', new_info)
    
    
    def manage_initial_broks_done_brok(self, _new_info):
        self.migrator.handle_initial_broks_done()
    
    
    def stop_all(self, update_stop_logger=None):
        if self._stop_done:
            return
        if not update_stop_logger:
            update_stop_logger = self.logger
        
        update_stop_logger.info(u'Start stopping all process of SLA Module Broker')
        
        # As tick() is disabled, try to prevent an inactive range due to long stop of other broker modules
        try:
            self._update_active_range()
        except:
            update_stop_logger.error(u'Failed to save SLA status, please check your mongodb connection.')
        
        try:
            if hasattr(self, u'migrator'):
                self.migrator.stop()
        except:
            pass
        
        try:
            if hasattr(self, u'archivator'):
                self.archivator.stop()
        except:
            pass
        
        try:
            WorkerBasedBrokerModule.stop_all(self)
        except:
            self.logger.error(u'Fail to stop workers')
            self.logger.print_stack()
        
        if hasattr(self, u'sla_writer_stats'):
            self.sla_writer_stats.stop()
        self.sla_info.stop()
        self.error_handler.stop()
        update_stop_logger.info(u'Stopping all process of SLA Module Broker done')
        self._stop_done = True
    
    
    def quit(self):
        try:
            self._update_active_range()
            self.logger.debug(u'Updated active range before quit')
        except:
            self.logger.error(u'could not refresh active range while quiting, please check your mongodb connection.')
    
    def hook_tick(self, broker):
        try:
            WorkerBasedBrokerModule.hook_tick(self, broker)
            self.component_manager.tick()
            self._update_active_range()
        except Exception as e:
            if broker.daemon_info.daemon_is_requested_to_stop.value:
                self.logger.info('update active fail because Broker %s is requested to stop.' % broker.name)
            else:
                self.error_handler.handle_exception('Fatal error caused by : %s' % e, e, self.logger, level=ERROR_LEVEL.FATAL)
                raise
    
    
    def _update_active_range(self):
        tick_time = get_now()
        date = date_now()
        sla_range_active = self.sla_database.find_raw_sla_status(date)
        if sla_range_active:
            last_tick_time = sla_range_active['active_ranges'][-1].get('end', 0)
            if last_tick_time != 0 and (tick_time - last_tick_time) > MARGIN_SLA_INACTIVE:
                last_end = int(last_tick_time)
                start = int(tick_time)
                sla_range_active['active_ranges'][-1]['end'] = last_end
                sla_range_active['active_ranges'].append({'start': start})
                self.logger.info('Starting a new SLA module activity period because last tick was more than %ss. It will make shinken inactive range in your elements SLA which is %s-%s.' % (
                    MARGIN_SLA_INACTIVE, self.logger.format_time(last_end), self.logger.format_time(start)))
            sla_range_active['active_ranges'][-1]['end'] = get_now()
        else:
            sla_range_active = {'_id': 'SLA_STATUS', 'active_ranges': [{'start': int(tick_time), 'end': int(tick_time) + 1}]}
        
        self.sla_database.save_raw_sla_status(date, sla_range_active)
    
    
    def get_raw_stats(self, param=''):
        try:
            if self.get_internal_state() == ModuleState.FATAL:
                return {}
            
            raw_stats = WorkerBasedBrokerModule.get_raw_stats(self, param)
            self.sla_writer_stats.process_raw_worker_stats(raw_stats)
            
            # self.logger.debug('asking get_raw_stats')
            data = self.sla_writer_stats.get_raw_stats()
            # self.logger.debug('get_raw_stats : [%s]' % data)
            return data
        except Exception as e:
            self.error_handler.handle_exception('Fail to get module information cause by : %s. This will only impact your check shinken on your sup of sup.' % e, e, self.logger, level=ERROR_LEVEL.WARNING)
            raise
    
    
    def get_from_module_to_worker_container(self):
        # type: () -> FromModuleToWorkerContainer
        from_module_to_worker_container = super(SLAModuleBroker, self).get_from_module_to_worker_container()
        from_module_to_worker_container.error_handler = self.error_handler
        return from_module_to_worker_container
    
    
    def after_fork_cleanup(self):
        # #SEF-9050 + #SEF-9078
        # "After fork" can occur for Migrator or Archivator process,
        # in these cases, we must take care of keeping needed data :
        # (Migrator or Archivator) instance + error_handler instance
        keep_error_handler = False
        self.sla_writer_stats.after_fork_cleanup()
        self.sla_database.after_fork_cleanup()
        self.sla_info.after_fork_cleanup()
        self.sla_database_connection.after_fork_cleanup()
        if after_fork_cleanup.after_fork_can_cleanup(self.migrator):
            self.migrator.after_fork_cleanup()
        else:
            keep_error_handler = True
        if after_fork_cleanup.after_fork_can_cleanup(self.archivator):
            self.archivator.after_fork_cleanup()
        else:
            keep_error_handler = True
        self.component_manager.after_fork_cleanup()
        if not keep_error_handler:
            self.error_handler.after_fork_cleanup()
        
        self.sla_writer_stats = None
        self.sla_database = None
        self.sla_info = None
        self.sla_database_connection = None
        self.migrator = None
        self.archivator = None
        self.component_manager = None
        self.error_handler = None
        
        ConfigurationReaderMixin.after_fork_cleanup(self)
