#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013:
#    Gabes Jean, naparuba@gmail.com
#
# This file is part of Shinken Enterprise, all rights reserved.

import json
import os
import time

from shinken.http_client import HTTPClient, HTTPExceptions
from shinken.log import logger
from shinken.modules.base_module.basemodule import BaseModule
from shinkensolutions.crypto import get_clear_message_form_obfuscated

try:
    from shinken.synchronizer.dao.def_items import PROP_DEFAULT_VALUE
except ImportError:
    from synchronizer.dao.def_items import PROP_DEFAULT_VALUE

properties = {
    u'daemons': [u'arbiter'],
    u'type'   : u'synchronizer-import',
    u'phases' : [u'configuration'],
}


def _show_error(error):
    bloc_error_size = 80
    logger.error(u'*' * bloc_error_size)
    logger.error(u'')
    logger.error(error)
    logger.error(u'')
    logger.error(u'*' * bloc_error_size)


# called by the plugin manager to get a module
def get_instance(plugin):
    logger.info(u'[Synchronizer Import] Get a Synchronizer import module for plugin %s' % plugin.get_name())
    
    # Catch errors
    url = getattr(plugin, u'url', None)
    if url is None:
        _show_error(u'The "url" parameter for the synchronizer-import module (in file %s) is mandatory. It must link to the synchronizer daemon private address (with the 7765 port).' % plugin.imported_from)
        return None
    master_key = getattr(plugin, u'master_key', None)
    if master_key is None:
        _show_error(
            u'The "master_key" parameter for the synchronizer-import module (in file %s) is mandatory. It must be the same as the synchronizer one (in the synchronizer.cfg file) so only allowed arbiter can get the configuration.' % plugin.imported_from)
        return None
    
    instance = Synchronizer_Import(plugin, url, master_key)
    return instance


# Just print some stuff
class Synchronizer_Import(BaseModule):
    
    def __init__(self, mod_conf, url, master_key):
        BaseModule.__init__(self, mod_conf)
        self.url = url
        self.master_key = master_key
        self.client = None
        self.activated = True  # Not activated means we run on spare. If so we do nothing
        self.max_try = int(getattr(mod_conf, u'max_try', 90))
        self.sleep_time = int(getattr(mod_conf, u'sleep_time', 2))
    
    
    # Called by Arbiter to say 'let's prepare yourself guy'
    def init(self):
        logger.info(u'[Synchronizer Import] Initialization of the Synchronizer import module')
    
    
    def do_loop_turn(self):
        # Only for make linter and pycharm happy
        pass
    
    
    def get_objects(self):
        # Arbiter is spare so module is not activate, do nothing
        if not self.activated:
            return {}
        
        logger.info(u'[Synchronizer Import] Requesting configuration from the synchronizer daemon')
        if self.client is None:
            self.client = HTTPClient(uri=self.url, timeout=60)
        logger.debug(u'[Synchronizer Import] client created')
        
        start_time = time.time()
        
        nb_try = 0
        data = {}
        fetch_data_success = False
        max_try = self.max_try
        while nb_try < max_try:  # do the try for max_try times
            nb_try += 1
            # Look if we just check or if we will look at the prod part
            env = os.environ
            
            to_load = u'production'
            if env.get('CHECK_STAGGING', u'0') == u'1':
                to_load = u'stagging'
            elif env.get('CHECK_PROPOSE', u'0') == u'1':
                to_load = u'propose'
            elif env.get('CHECK_PREPROD', u'0') == u'1':
                to_load = u'preprod'
            
            logger.info(u'Getting synchronizer [%s] configuration' % to_load)
            try:
                args = {
                    u'states'  : to_load,
                    u'auth_key': self.master_key,
                    u'nb_try'  : nb_try,
                    u'max_try' : max_try,
                }
                response = json.loads(self.client.post(u'/get_whole_configuration', args))
                if response[u'rc'] == 504:
                    logger.debug(u'[Synchronizer Import] Synchronizer is UP but Shinken objects are not yet fully initialized')
                    time.sleep(self.sleep_time)
                    continue
                if response[u'rc'] == 403:
                    logger.error(u'[Synchronizer Import] Synchronizer is UP but the provided master _key was rejected')
                    print u'[Synchronizer Import] Synchronizer is UP but the provided master _key was rejected'
                    break
                if response[u'rc'] != 200:
                    logger.debug(u'[Synchronizer Import] Synchronizer is UP but there is a problem to get the conf')
                    time.sleep(self.sleep_time)
                    continue
                data = response[u'data']
                fetch_data_success = True
                break
            except HTTPExceptions, exp:
                # but sleep a bit to not hammer the sycnhronizer, maybe it's just closed so sleep a bit until it's ready
                time.sleep(self.sleep_time)
                logger.error(u'[Synchronizer Import] cannot get the Synchronizer %s data: "%s" still %d tries with %ss pause time between try' % (self.url, str(exp), max_try - nb_try, self.sleep_time))
        
        if fetch_data_success is False:
            raise Exception(u'[Synchronizer Import] cannot get the Synchronizer %s data' % self.url)
        
        if not data:
            return {}
        # Sync data are without 's' in the end, arbiter need it...
        new_data = {}
        
        for (item_type, items) in data.iteritems():
            new_lst = []
            # Loop over objects and remove NON string values
            for item in items:
                # map _id as it should, in upper case here
                if u'_id' in item:
                    item[u'_ID'] = item[u'_id']
                    item[u'uuid'] = item[u'_id']
                    
                    del item[u'_id']
                to_del = [u'id']
                if item_type == u'timeperiod':
                    to_del = [u'sources', u'overwrited_protected']
                for (_key, _value) in item.iteritems():
                    if _value == PROP_DEFAULT_VALUE:
                        item[_key] = u'null'
                    elif not isinstance(_value, basestring):
                        to_del.append(_key)
                    elif _value == u'':
                        to_del.append(_key)
                    elif _key == u'service_overrides':
                        item[_key] = _value.replace(PROP_DEFAULT_VALUE, u'null')
                
                for _key in to_del:
                    try:
                        del item[_key]
                    except KeyError:
                        pass
                
                # Skip bp_rule as it's an inner command
                if item_type == u'command':
                    cname = item.get(u'command_name', u'')
                    if cname == u'bp_rule':
                        continue
                
                enable = item.get(u'enabled', u'1')
                if enable in [u'1', u'']:
                    new_lst.append(item)
                
                # Force a shinken-host to all hosts, and at the end
                # and manage *_contacts/groups matching
                if item_type == u'host':
                    _to_del = [u'edition_contacts', u'edition_contact_groups']
                    for _key in _to_del:
                        if _key in item:
                            del item[_key]
                    # and map notification_contacts/groups into contacts
                    if u'notification_contacts' in item:
                        item[u'contacts'] = item[u'notification_contacts']
                        del item[u'notification_contacts']
                    if u'notification_contact_groups' in item:
                        item[u'contact_groups'] = item[u'notification_contact_groups']
                        del item[u'notification_contact_groups']
                    
                    # Cluster are now with 'is_cluster'==1 and 'bp_rule' instead of check_command
                    if item.get(u'is_cluster', u'0') == u'1':
                        item[u'check_command'] = u''.join([u'bp_rule!', item.get(u'bp_rule', u'')])
                        item[u'check_interval'] = u'1'
                    
                    # force check_interval, retry_interval and max_check_attempts on clusters
                    ck = item.get(u'check_command', '')
                    if ck.startswith(u'bp_rule!'):
                        item[u'check_interval'] = u'1'
                        item[u'retry_interval'] = u'1'
                        item[u'max_check_attempts'] = u'1'
                        item[u'active_checks_enabled'] = u'0'
                
                if u'contact_groups' in item:
                    item[u'contact_groups'] = u','.join((cg.strip() for cg in item[u'contact_groups'].split(u',') if cg.strip()))
                if u'contactgroups' in item:
                    item[u'contactgroups'] = u','.join((cg.strip() for cg in item[u'contactgroups'].split(u',') if cg.strip()))
            
            new_data[item_type + u's'] = new_lst
        
        logger.info(u'[performance] Synchronizer elements are get in %.3fs' % (time.time() - start_time))
        return new_data
    
    
    # We have the configuration ans now who we are
    # if the current arbiter is spare, this module will do nothing, so user will not have to edit his spare conf file
    def hook_read_configuration(self, arb):
        self.activated = not arb.me.spare
        if not self.activated:
            logger.info(u'[Synchronizer Import] The Arbiter is configured as Spare, Synchronizer Import is disabled because we don\'t have synchronizer on this host')
    
    
    # This part is here to check that there is a (valid) key to
    # do NOT limit the hosts numbers by settings silly things into it
    def hook_early_configuration(self, arb):
        from shinkensolutions.crypto import are_keys_valid
        d = are_keys_valid()
        are_valid = d[u'are_valid']
        nodes_limit = d[u'nodes_limit']
        
        if not are_valid:
            logger.error(get_clear_message_form_obfuscated(u'Op!wbmje!lfz!gpvoe-!xjmm!mjnju!uif!ovncfs!pg!iptu!up!&t') % nodes_limit)
        else:
            logger.info(get_clear_message_form_obfuscated(u'Gpvoe!Foufsqsjtf!wbmje!lfz'))
        
        hosts = [host for host in arb.conf.hosts if getattr(host, u'register', u'1') != u'0' and not getattr(host, u'check_command', u'').startswith(u'bp_rule!')]
        
        hosts_count = len(hosts)
        exceded_hosts = hosts_count - nodes_limit
        
        if hosts_count > nodes_limit:
            logger.warning(get_clear_message_form_obfuscated(u'PWFSMPBEFE!MJDFODF!)!&j0&j!*!;!Uif!ovncfs!pg!iptut!bmmpxfe!jt!fydffefe!cz!&t/!Ejtbcmjoh!uiftf!iptut!;') % (hosts_count, nodes_limit, exceded_hosts))
            for host in hosts[nodes_limit:]:
                logger.warning(u'- %s' % getattr(host, u'host_name', u''))
                host.use = u'disabled,shinken-host'
                host.address = get_clear_message_form_obfuscated(u'ejtbcmfe!\\!Opeft!mjdfodf!FYDFFEFE!^')
                host.check_command = u'_internal_host_up'
                host.hostgroups = u''
                host.customs = {}
                host.business_impact = u'0'
