#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import datetime
import errno
import os
import re
import shutil
import tarfile
import threading
import time
import traceback

from bson import BSON

from components.component_factory import ComponentFactory
from components.component_manager import ComponentManager
from shinken.daemon import Daemon
from shinken.log import logger, LoggerFactory
from shinken.message import Message
from shinken.modules.base_module.basemodule import BaseModule, ModuleState
from shinken.modulesctx import modulesctx
from shinken.modulesmanager import ModulesManager
from shinkensolutions.lib_modules.configuration_reader import read_string_in_configuration, read_int_in_configuration, read_bool_in_configuration
from shinkensolutions.ssh_mongodb.mongo_error import ShinkenMongoException
from shinkensolutions.system_tools import create_tree
from work_hours import WorkHours

try:
    from shinken.synchronizer.synchronizer_mongo_conf import SynchronizerMongoConf
except ImportError:
    from synchronizer.synchronizer_mongo_conf import SynchronizerMongoConf

properties = {
    u'daemons' : [u'synchronizer'],
    u'type'    : u'synchronizer_module_database_backup',
    u'phases'  : [u'running'],
    u'external': True,
}

LOOP_INTERVAL = 60

DEFAULT_BACKUP_RATE = 60
DEFAULT_RETENTION_DAYS = 21
DEFAULT_BACKUP_DIRECTORY = u'/var/shinken-user/backup/synchronizer-module-database-backup'

REGEX_BACKUP_NAME = r'^[0-9]{4}-[0-9]{2}-[0-9]{2}_[0-9]{2}-[0-9]{2}_.*\.tgz$'

raw_logger = LoggerFactory.get_logger()


# called by the plugin manager to get an instance
def get_instance(plugin):
    logger.debug(u'Get a SynchronizerModuleDatabaseBackup instance for plugin %s' % plugin.get_name())
    
    return SynchronizerModuleDatabaseBackup(plugin)


class SynchronizerModuleDatabaseBackup(BaseModule, Daemon):
    CURSOR_NOT_FOUND_LIMIT_TRY = 3
    
    
    def __init__(self, modconf):
        BaseModule.__init__(self, modconf)
        self.errors = []
        self.warnings = []
        self.app = None
        self._mongo_conf = None
        self.backup_logger = LoggerFactory.get_logger(u'BACKUP')
        self.clean_logger = LoggerFactory.get_logger(u'CLEAN RETENTION')
        
        # Backup options
        self.backup_name = read_string_in_configuration(self.myconf, u'backup_name', u'')
        self.backup_dir = read_string_in_configuration(self.myconf, u'backup_directory', DEFAULT_BACKUP_DIRECTORY)
        self.backup_rate = read_int_in_configuration(self.myconf, u'backup_rate', DEFAULT_BACKUP_RATE, log_fct=self.logger)
        self.last_backup = None
        self.work_hours = None  # Will be set later
        
        # Retention
        self.retention_days = read_int_in_configuration(self.myconf, u'retention_days', DEFAULT_RETENTION_DAYS, log_fct=self.logger)
        
        # External module :
        self.modules_dir = None
        self.modules_manager = None
    
    
    def init(self):
        
        # create the tree for backup_directory and check the rights.
        create_tree(self.backup_dir)
        self.clean_tmp_file()
        
        # Set the mongo and work_hours parameters
        self.init_work_hours_and_rate()
    
    
    def load(self, app):
        self._mongo_conf = SynchronizerMongoConf(app.conf, LoggerFactory.get_logger())  # We want to backup the Synchronizer, so we need its mongo configuration
    
    
    def init_work_hours_and_rate(self):
        use_work_hours = read_bool_in_configuration(self.myconf, u'enable_specific_backup_interval_during_working_hours', False, log_fct=self.logger)
        days_worked = read_string_in_configuration(self.myconf, u'days_worked', '')
        rate_workings_hours = read_int_in_configuration(self.myconf, u'backup_interval_during_working_hours', 15, log_fct=self.logger)
        start = read_string_in_configuration(self.myconf, u'work_hours_start', '')
        end = read_string_in_configuration(self.myconf, u'work_hours_end', '')
        
        try:
            self.work_hours = WorkHours(use_work_hours, start, end, days_worked, self.backup_rate, rate_workings_hours, self.logger)
            self.work_hours.is_correct()
        except Exception as e:
            self.logger.error(u'WorkHours configuration : %s' % e.message)
            self.logger.error(u'There is a problem in the work_hours configuration. Will disable the option use_work_hours.')
            self.warnings.append(e.message)
            if self.work_hours:
                self.work_hours.enable = False
    
    
    def main(self):
        logger.set_name(self.name)
        self.debug_output = []
        
        self.modules_dir = modulesctx.get_modulesdir()
        self.modules_manager = ModulesManager(u'synchronizer-module-database-backup', self.modules_dir, [])
        self.modules_manager.set_modules(self.modules)
        
        self.do_load_modules()
        for inst in self.modules_manager.get_all_alive_instances():
            f = getattr(inst, u'load', None)
            if f and callable(f):
                f(self)
        
        for s in self.debug_output:
            self.logger.debug(s)
        del self.debug_output
        
        try:
            self.do_main()
        except Exception, exp:
            msg = Message(id=0, type=u'ICrash', data={u'name': self.get_name(), u'exception': exp, u'trace': traceback.format_exc()})
            self.from_module_to_main_daemon_queue.put(msg)
            # wait 2 sec so we know that the synchronizer got our message, and die
            time.sleep(2)
            raise
    
    
    # Real main function
    def do_main(self):
        # I register my exit function
        # We just forked, it is time for us to build our components
        ComponentFactory.build(self.myconf, self._mongo_conf, self.logger)
        self.set_exit_handler()
        
        while not self.interrupted or not self.errors:
            if self.need_backup():
                thread_backup = threading.Thread(None, self.backup_thread, 'backupThread')
                # This thread must stay alive to terminate his job !
                thread_backup.daemon = False
                thread_backup.start()
                thread_backup.join()
            
            if self.need_clean_retention():
                thread_clean_retention = threading.Thread(None, self.clean_retention_thread, 'cleanRetentionThread')
                # This thread must stay alive to terminate his job !
                thread_clean_retention.daemon = False
                thread_clean_retention.start()
            
            time.sleep(LOOP_INTERVAL)
    
    
    def need_backup(self):
        now = time.time()
        if not self.last_backup:
            self.last_backup = now
            return True
        
        # The delta between 2 backup must be minutes
        delta = (now - self.last_backup) / 60
        
        # The self.work_hours can be empty if it encounter an error at init
        _rate = self.work_hours.get_rate() if self.work_hours else self.backup_rate
        
        if delta >= _rate:
            self.last_backup = now
            return True
        else:
            return False
    
    
    @staticmethod
    def need_clean_retention():
        return True
    
    
    def backup_thread(self):
        # To launch the backup, we need to get the time to create the directory
        start_time = time.time()
        self.backup_logger.info(u'Start thread')
        # Then we compute the backup directory name with date and name set by user
        
        backup_date = datetime.datetime.fromtimestamp(start_time).strftime(u'%Y-%m-%d_%H-%M')
        backup_name = u'%s_%s' % (backup_date, self.backup_name)
        full_backup_directory_name = os.path.join(self.backup_dir, backup_name)
        try:
            self._do_backup_thread(start_time, backup_name, full_backup_directory_name)
        
        except ShinkenMongoException:
            self.backup_logger.error(u'The backup was not completed due to a connection problem with MongoDb, Aborting.')
        except Exception:
            self.backup_logger.error(u'An error occurred during the backup, aborting :')
            self.backup_logger.print_stack()
        finally:
            self.clean_raw_dump(full_backup_directory_name)  # If we have an exception, we still need to erase the raw dump folder
    
    
    def _do_backup_thread(self, start_time, backup_name, full_backup_directory_name):
        
        archive_name = u'%s.tgz' % backup_name
        mongo_component = ComponentManager.get_mongo_component()
        # ready ? let's go to backup, compress, and clean dump
        
        self.backup_logger.info(u'Dumping mongo database [ %s ]' % mongo_component.get_database_name())
        self.dump_mongo_database(full_backup_directory_name)
        
        # Then we compress this dump
        self.backup_logger.info(u'Compressing dump directory [ %s ] into [ %s ]' % (full_backup_directory_name, archive_name))
        self.compress_directory(archive_name, backup_name)
        
        full_archive_name = os.path.join(self.backup_dir, archive_name)
        self.backup_logger.info(u'Archive created [ %s ] in %.3fs' % (full_archive_name, time.time() - start_time))
    
    
    def clean_retention_thread(self):
        # To know which archive to delete, we need to get the date
        start_time = time.time()
        
        # Then we compute the older timestamp to keep
        older_to_keep = datetime.datetime.fromtimestamp(start_time) - datetime.timedelta(days=self.retention_days)
        removed_files = []
        
        # Will keep only the file with the format YYYY-MM-DD_HH-MM_All_You-Want Here.tgz
        archives = [arch for arch in os.listdir(self.backup_dir) if re.match(REGEX_BACKUP_NAME, arch)]
        for archive in archives:
            archive_time = archive[:16]
            archive_date = datetime.datetime.strptime(archive_time, u'%Y-%m-%d_%H-%M')
            if archive_date < older_to_keep:
                removed_files.append(archive)
                os.remove(os.path.join(self.backup_dir, archive))
        
        if removed_files:
            self.clean_logger.info(u'Files removed : %s' % u', '.join(removed_files))
    
    
    def dump_mongo_database(self, backup_dir):
        mongo_component = ComponentManager.get_mongo_component()
        dump_directory = os.path.join(backup_dir, mongo_component.get_database_name())
        try:
            create_tree(dump_directory)
        except OSError as e:
            _error = u'%s: %s' % (e.strerror, e.filename)
            self.backup_logger.error(_error)
            if _error not in self.errors:
                self.errors.append(_error)
            raise e
        collections = mongo_component.list_name_collections()
        for i, collection_name in enumerate(collections):
            if collection_name.startswith(u'tmp-'):
                continue
            col = mongo_component.get_collection(collections[i])
            find_start = time.time()
            cols_result = list(col.find())
            file_name = os.path.join(dump_directory, u'%s.bson' % collection_name)
            self.backup_logger.info(u'[PERFS] Collection [ %s ] with [ %s ] documents retrieved in in %.3fs' % (collection_name, len(cols_result), time.time() - find_start))
            with open(file_name, u'wb+') as f:
                for doc in cols_result:
                    f.write(BSON.encode(doc))
    
    
    def compress_directory(self, backup_name, backup_dir):
        cwd = os.getcwd()
        os.chdir(self.backup_dir)
        _name = u'tmp-%s' % backup_name
        tar_gz = None
        # This try,except,finally implementation is nothing more than a homemade "with".
        # Mandatory for Python <=2.6 because tarfile added __exit__ method with Python 2.7
        try:
            tar_gz = tarfile.open(_name.encode(u'utf-8'), 'w:gz')
            tar_gz.add(backup_dir)
        except AttributeError:
            pass
        finally:
            if tar_gz:
                tar_gz.close()
        
        os.rename(_name, backup_name)
        os.chdir(cwd)
    
    
    def clean_raw_dump(self, backup_dir):
        self.backup_logger.info(u'Cleaning dump directory [ %s ]' % backup_dir)
        try:
            shutil.rmtree(backup_dir)
        except OSError as e:
            if e.errno == errno.ENOENT:  # ENOENT means "ERROR NO ENTry", which is a "No such file or directory" error
                self.backup_logger.debug(u'Can\'t delete the dir [ %s ] as it is already deleted')
            else:
                self.backup_logger.debug(u'Error while deleting temporary backup_dir : [ %s ] - %s.' % (e.filename, e.strerror))
    
    
    def get_module_info(self):
        module_info = {}
        tmp = BaseModule.get_submodule_states(self)
        if tmp:
            module_info.update(tmp)
        
        module_info[u'errors'] = self.errors
        module_info[u'warnings'] = self.warnings
        if self.errors:
            # Be careful : the module state in "CRITICAL" is not correctly shown in the healthcheck.
            module_info[u'status'] = ModuleState.CRITICAL
            module_info[u'output'] = u'<br>'.join(self.errors)
        elif self.warnings:
            module_info[u'status'] = ModuleState.WARNING
            module_info[u'output'] = u'<br>'.join(self.warnings)
        return module_info
    
    
    def clean_tmp_file(self):
        files = os.listdir(self.backup_dir)
        for file_name in files:
            if file_name.startswith(u'tmp-'):
                os.remove(os.path.join(self.backup_dir, file_name))
    
    
    def do_loop_turn(self):
        # Only for linter/Pycharm
        pass
    
    
    def loop_turn(self):
        # Only for linter/Pycharm
        pass
