#!/usr/bin/python

# -*- coding: utf-8 -*-

# Copyright (C) 2014:
#    Gabes Jean, j.gabes@shinken-solutions.com


import re

import pymongo
from pymongo import MongoClient
from pymongo import MongoReplicaSetClient

from shinken.log import logger
from shinken.modules.base_module.basemodule import BaseModule
from shinken.util import get_sec_from_morning, get_wday
from trender import Trender

properties = {
    'daemons' : ['broker', 'webui'],
    'type'    : 'trending',
    'external': False,
}


# Called by the plugin manager to get a broker
def get_instance(mod_conf):
    logger.info("[Trending] Get a trending data module for plugin %s" % mod_conf.get_name())
    instance = Trending_broker(mod_conf)
    return instance


# Class for the Trending Broker
# Get broks and send them to a Mongodb instance with smooth averages
class Trending_broker(BaseModule):
    def __init__(self, modconf):
        BaseModule.__init__(self, modconf)
        self.backend = getattr(modconf, 'backend', 'mongodb')
        self.uri = getattr(modconf, 'uri', 'localhost')
        self.database = getattr(modconf, 'database', 'shinken')
        self.replica_set = getattr(modconf, 'replica_set', '')
        
        # 15min chunks
        self.chunk_interval = int(getattr(modconf, 'chunk_interval', '900'))
        
        # Some used variable init
        self.con = None
        self.db = None
        self.col = None
        
        self.host_dict = {}
        self.svc_dict = {}
        
        # And our final trender object
        self.trender = Trender(self.chunk_interval)
    
    
    # Called by Broker so we can do init stuff
    # TODO: add conf param to get pass with init
    # Conf from arbiter!
    def init(self):
        logger.info("[Trending broker] I init the %s server connection to %s:%s (%s)" % (self.get_name(), self.backend, self.uri, self.replica_set))
        if self.replica_set:
            self.con = MongoReplicaSetClient(self.uri, replicaSet=self.replica_set, safe=True)
        else:
            # Old versions of pymongo do not known about fsync
            if MongoReplicaSetClient:
                self.con = MongoClient(self.uri, safe=False)
            else:
                self.con = MongoClient(self.uri, safe=True)
        
        # Open a connection
        self.db = getattr(self.con, self.database)
        self.col = self.db['trending']
        self.col.ensure_index([('hname', pymongo.DESCENDING), ('sdesc', pymongo.DESCENDING), ('metric', pymongo.DESCENDING), ('cycle', pymongo.DESCENDING)])
    
    
    # For a perf_data like /=30MB;4899;4568;1234;0  "/var"=50MB;4899;4568;1234;0 /toto=
    # return ('/', {'value' : '30', 'warning':4899, 'critical':4568}), ('/var', {'value' : '50', 'warning':'4899', 'critical':'1234'})
    def get_metric_and_value(self, perf_data):
        res = []
        s = perf_data.strip()
        # Get all metrics non void
        elts = s.split(' ')
        metrics = [e for e in elts if e != '']
        
        for e in metrics:
            # logger.debug("[Trending broker] Groking: %s" % str(e))
            elts = e.split('=', 1)
            if len(elts) != 2:
                continue
            # name = self.illegal_char.sub('_', elts[0])
            name = elts[0].replace('/', 'SLASH').replace('"', '')
            
            raw = elts[1]
            # get metric value and its thresholds values if they exist
            if ';' in raw and len(filter(None, raw.split(';'))) >= 3:
                elts = raw.split(';')
                name_value = {name: {'value': elts[0], 'warning': elts[1], 'critical': elts[2]}}
            # get the first value of ;
            else:
                name_value = {name: {'value': raw, 'warning': None, 'critical': None}}
            
            # bailout if need
            if name_value[name]['value'] == '':
                continue
            
            # Try to get the int/float in it :)
            for key, d in name_value.items():
                value = d['value']
                m = re.search("(-?\d*\.?\d*)(.*)", value)
                if m:
                    name_value[key]['value'] = m.groups(0)[0]
                else:  # Not an int/float? skip it
                    continue
            # logger.debug("[Trending broker] End of grok: %s, %s" % (name, str(value)))
            for key, value in name_value.items():
                res.append((key, value))
        return res
    
    
    # A service check result brok has just arrived, we UPDATE data info with this
    def manage_service_check_result_brok(self, b):
        data = b.data
        
        perf_data = data['perf_data']
        couples = self.get_metric_and_value(perf_data)
        
        # If no values, we can exit now
        if len(couples) == 0:
            return
        
        hname = data['host_name'].replace(' ', '_').replace('.', 'DOT')
        sdesc = data['service_description'].replace(' ', '_')
        check_time = int(data['last_chk'])
        
        logger.debug("[Trending broker] Hostname: %s, Desc: %s, check time: %d, perfdata: %s" % (hname, sdesc, check_time, str(perf_data)))
        
        # Ok now the real stuff is here
        for (metric, d) in couples:
            value = d['value']
            warning = d['warning']
            critical = d['critical']
            try:
                value = float(value)
            except ValueError:
                return
            if value is not None:
                sec_from_morning = get_sec_from_morning(check_time)
                wday = get_wday(check_time)
                
                chunk_nb = sec_from_morning / self.chunk_interval
                
                # Now update mongodb
                self.trender.update_avg(self.col, check_time, wday, chunk_nb, value, hname, sdesc, metric, self.chunk_interval, warning, critical)
    
    
    def loop_turn(self):
        pass
    
    
    def do_loop_turn(self):
        pass
