#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
#    Gabes Jean, j.gabes@shinken-solutions.com
#
# This file is part of Shinken enterprise, all rigths reserved

import StringIO
import json
import urllib

from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional
    from webui.module import WebuiBroker

app = None  # type: Optional[WebuiBroker]


def batch():
    logger.debug("[BATCH] call new batch")
    app.response.content_type = 'application/json'
    
    user = app.get_user_auth()
    if not user:
        return app.abort(401, 'batch user is not authenticated', True)
    
    if hasattr(app.request.body, 'getvalue'):
        data = json.loads(app.request.body.getvalue())
    else:
        data = json.loads(app.request.body.read())
    
    if not data:
        return app.abort(400, 'Please give data')
    
    my_response = app.bottle.response.copy()
    
    default_environ = app.request.environ.copy()
    for (k, v) in app.request.environ.iteritems():
        if k.startswith('bottle') or k.startswith('route'):
            del default_environ[k]
    # CONTENT_LENGTH will be recompute for next call
    del default_environ['CONTENT_LENGTH']
    
    ret_batch = {}
    for (batch_key, batch_request) in data.iteritems():
        url = batch_request.get('url')
        logger.debug("[BATCH] url call : %s" % url)
        if not url:
            return app.abort(400, 'Please give url')
        
        batch_request['url'] = urllib.unquote(url)
        ret_batch[batch_key] = {}
        _handle(batch_request, ret_batch[batch_key], default_environ)
    
    # We must rebind the response for the result of the batch.
    # We add to this new reponse the headers save when we enter in batch.
    app.bottle.response.bind()
    for header in my_response.iter_headers():
        app.bottle.response.set_header(header[0], header[1], False)
    app.bottle.response.content_type = 'application/json'
    return json.dumps(ret_batch)


def build_environ(batch_request, default_environ):
    # logger.debug("[BATCH] ---- build_environ --- ")
    
    environ = default_environ.copy()
    # The presence of url in batch_request was already check
    url = batch_request['url'].split('?')
    path_info = url[0]
    query_string = ""
    if len(url) >= 2:
        query_string = url[1]
    environ['REQUEST_METHOD'] = batch_request.get('method', 'GET')
    environ['PATH_INFO'] = path_info
    environ['QUERY_STRING'] = query_string
    environ['REQUEST_URI'] = batch_request['url']
    raw_payload = batch_request.get('data')
    if isinstance(raw_payload, dict):
        raw_payload = json.dumps(raw_payload)
    if raw_payload:
        environ['wsgi.input'] = StringIO.StringIO(raw_payload)
        environ['CONTENT_LENGTH'] = len(raw_payload)
    return environ


def _handle(batch_request, ret_batch, default_environ):
    def start_response(status, response_headers, exc_info=None):
        ret_batch['status_text'] = status
        ret_batch['status'] = int(status[0:3])
        ret_batch['headers'] = response_headers
    
    
    environ = build_environ(batch_request, default_environ)
    try:
        output = app.bottle.default_app().wsgi(environ, start_response)
        if output:
            ret_batch['output'] = output[0]
    except Exception as err:
        logger.info("[BATCH] call for %s fail" % batch_request['url'])


pages = {
    batch: {'routes': ['/api/batch'], 'method': 'POST', 'static': True},
}
