#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.
import itertools
import json
import os
import time
import traceback

from event_container.ec_database_connection import ECDatabaseError
from shinken.log import LoggerFactory
from shinken.misc.filter import only_related_to
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from event_container.module import EventContainerWebUIModule
    from shinken.misc.type_hint import Optional
    from webui.module import WebuiBroker
    from shinken.misc.monitoring_item_manager.regenerator import Regenerator
    from shinken.objects.contact import Contact

logger = LoggerFactory.get_logger('event_container')
logger_fast_search = logger.get_sub_part('FAST-SEARCH')
app = None  # type: Optional[WebuiBroker]

# we cannot request an infinity of uuids in mongo (max query size = 16Mo), so limit the number
# of uuids we are giving, if more, give a slow host_name/check_name filter instead :'(
_MAX_NUMBER_OF_REQUESTING_UUIDS = int(os.environ.get('SHINKEN_SUPPORT_ONLY_MAX_NUMBER_OF_REQUESTED_UUIDS_IN_EVENT_CONTAINER', '100000'))


def _send_stat_to_module(stat_name, stat_endpoint, filters, stat_time):
    _module = get_event_container_module()
    _module.send_stat_to_reader_stats(stat_name, stat_endpoint, filters, stat_time)


def get_event_container_module():
    # type: () -> EventContainerWebUIModule
    event_container_module = next((mod for mod in app.modules_manager.get_all_instances() if mod.properties['type'] == 'event_container'), None)
    if not event_container_module:
        logger.error("The event_container module wasn't found")
        return app.abort(500, "The event_container module wasn't found")
    return event_container_module


def _get_pagination_info():
    # type: () -> (str, int, int)
    first_ordering_uuid = ''
    page_size = 100
    page_index = 0
    
    get_range = 'range not get from request'
    try:
        get_range = app.request.GET.get('range', '').strip()
        if get_range:
            for param in get_range.split('~'):
                p_name, p_value = param.split(':')
                
                if p_name == 'first_ordering_uuid':
                    first_ordering_uuid = p_value
                elif p_name == 'page_size':
                    page_size = int(p_value)
                elif p_name == 'page':
                    page_index = int(p_value)
                else:
                    raise Exception('unknown range param : [%s]' % get_range)
    except Exception as e:
        logger.warning('fail to parse range param : [%s] with error : [%s]' % (get_range, e))
    
    return first_ordering_uuid, page_size, page_index


def _make_one_filter_response(item_uuids, final_filters, original_host_check_filters, user, one_filter):
    # NOTE: The event filter part do not manage "void" values, but here, we really want to match nothing, so put a
    #       uuid that will never exist
    if len(item_uuids) == 0:
        item_uuids.append('NOT-EXISTINGUUID')
    
    # Maybe we are over the number of uuids, so must get back to slow query :'(
    if len(item_uuids) > _MAX_NUMBER_OF_REQUESTING_UUIDS:
        final_filters.extend(original_host_check_filters)
        logger_fast_search.warning(
            u'[user=%s] [filter=%s] The filter match too much uuids to query mongodb (%s > %s) we must fallback to the slower regexp based query.' % (
                user.get_name(), one_filter, len(item_uuids), _MAX_NUMBER_OF_REQUESTING_UUIDS))
        return '~'.join(final_filters), False
    
    # size is OK, go to the fast mode
    fast_uuid_filter = 'item_uuids:%s' % ('^^'.join(item_uuids))
    final_filters.append(fast_uuid_filter)
    return '~'.join(final_filters), True


def _parse_one_filter(one_filter, user, rg):
    # type: (unicode, Contact, Regenerator) -> (unicode, bool)
    start = time.time()
    cur_filter = one_filter.strip()
    logger_fast_search.debug('[%s] Parsing one line filter: %s' % (user.get_name(), cur_filter))
    parts = cur_filter.split('~')
    
    host_name_filter = None  # if filter by host name
    check_name_filter = None  # if filter by check name
    query_types = []  # if filter by element type
    
    # If we have too much uuids, we will have to fail back to
    # original queries, with regexp
    original_host_check_filters = []
    
    final_filters = []
    for part in parts:
        # Void part, not interesting
        if len(part) == 0:
            continue
        filter_tag, filter_value = part.split(':', 1)
        # Host & checks: try to change
        if filter_tag == 'host_name':
            host_name_filter = filter_value.lower()  # match is lower case
            original_host_check_filters.append(part)  # save aside the original query, if need
        elif filter_tag == 'service_description':
            check_name_filter = filter_value.lower()  # match is lower case
            original_host_check_filters.append(part)  # save aside the original query, if need
        elif filter_tag == 'type':
            query_types = filter_value.split('^^')  # can have more than one type
            original_host_check_filters.append(part)
        else:
            final_filters.append(part)
    
    item_uuids = []
    
    # By default we allow all types
    allow_hosts = True
    allow_clusters = True
    allow_checks = True
    if len(query_types) != 0:  # but maybe not all is allowed ^^
        allow_hosts = u'host' in query_types
        allow_clusters = u'cluster' in query_types
        allow_checks = u'check' in query_types
    
    # Maybe the names are not filtered
    if host_name_filter is None and check_name_filter is None:
        # we don't have names, but maybe we can have some luck with the type
        if allow_checks and allow_clusters and allow_hosts and user.is_admin:  # ok, no hope of smart filter, give back the filter, and as we are admin, no filter :)
            logger_fast_search.debug(u'[%s] => Admin user did not filter by name or types, give all other filters' % (user.get_name()))
            return '~'.join(final_filters), True  # as an admin, rights is always OK
    
        all_my_clusters = None
        all_my_hosts = None
        # we have a type or a non admin user, get such elements ^^
        # NOTE: also filter by user rights of course
        if allow_hosts:
            all_my_hosts = only_related_to([host for host in app.datamgr.get_hosts() if not host.got_business_rule], user, app)
            logger_fast_search.debug(u'[%s] Get all %s hosts' % (user.get_name(), len(all_my_hosts)))
            item_uuids.extend([host.get_instance_uuid() for host in all_my_hosts])
    
        if allow_clusters:
            all_my_clusters = only_related_to([host for host in app.datamgr.get_hosts() if host.got_business_rule], user, app)
            logger_fast_search.debug(u'[%s] Get all %s clusters' % (user.get_name(), len(all_my_clusters)))
            item_uuids.extend([cluster.get_instance_uuid() for cluster in all_my_clusters])
    
        if allow_checks:
            if not all_my_clusters:
                all_my_clusters = only_related_to([host for host in app.datamgr.get_hosts() if host.got_business_rule], user, app)
            if not all_my_hosts:
                all_my_hosts = only_related_to([host for host in app.datamgr.get_hosts() if not host.got_business_rule], user, app)
        
            all_my_checks_uuids = []
            for item in itertools.chain(all_my_hosts, all_my_clusters):
                all_my_checks_uuids.extend([check.get_instance_uuid() for check in item.services])
            logger_fast_search.debug(u'[%s] Get all %s checks' % (user.get_name(), len(all_my_checks_uuids)))
            if all_my_checks_uuids:
                item_uuids.extend(all_my_checks_uuids)
    
        logger_fast_search.debug(u'[%s] => No name filter, filter by types only (%s) matching %s uuids' % (user.get_name(), query_types, len(item_uuids)))
        return _make_one_filter_response(item_uuids, final_filters, original_host_check_filters, user, one_filter)

    # Special filters by names:
    logger_fast_search.debug(u'[%s] - filtering by name:  host/cluster name=%s, check name=%s' % (user.get_name(), host_name_filter, check_name_filter))

    hosts = only_related_to([host for host in app.datamgr.get_hosts()], user, app)
    if host_name_filter:
        hosts = [host for host in hosts if host_name_filter in host.get_name().lower()]

    logger_fast_search.debug(u'[%s] - filtering by name:  host/cluster name=%s, check name=%s, currently have %s hosts/clusters  (allow_hosts=%s, allow_clusters=%s, allow_checks=%s)' % (
        user.get_name(), host_name_filter, check_name_filter, len(hosts), allow_hosts, allow_clusters, allow_checks))

    # Now:
    # * have checks filter: only theses checks
    # * not: all hosts & host checks
    if check_name_filter:
        if allow_checks:  # maybe we did not allow checks at all, will give nothing
            item_uuids.extend([check.get_instance_uuid() for host in hosts for check in host.services if check_name_filter in check.get_name().lower()])
    else:  # put hosts and ALL their checks
        for host in hosts:
            is_cluster = host.got_business_rule
            if is_cluster:
                if allow_clusters:
                    item_uuids.append(host.get_instance_uuid())
            elif allow_hosts:  # normal host, so is ok if allow hosts in query
                item_uuids.append(host.get_instance_uuid())
            if allow_checks:  # take checks only if they are allowed in the query
                # logger.debug('Add %s because it match %s + %s' % (host.get_instance_uuid(), host_name_filter, check_name_filter))
                for check in host.services:
                    # logger.debug('Add %s because it match %s + %s' % (check.get_instance_uuid(), host_name_filter, check_name_filter))
                    item_uuids.append(check.get_instance_uuid())
    
    logger_fast_search.debug(u'[%s] => [%.3f]s Generating name filter (host/cluster name=%s, check name=%s, (allow_hosts=%s, allow_clusters=%s, allow_checks=%s). Fast index uuids done, match %s uuids' % (
        user.get_name(), time.time() - start, host_name_filter, check_name_filter, allow_hosts, allow_clusters, allow_checks, len(item_uuids)))
    
    return _make_one_filter_response(item_uuids, final_filters, original_host_check_filters, user, one_filter)


def _get_filters(user):
    date_filter = ''
    get_filters = ''
    is_already_managed_rights = True
    filters_json = u'no filter from request'
    cnt = 0
    try:
        get_filters = []
        filters_json = app.request.body.getvalue().decode('utf-8')
        filters_dict = json.loads(filters_json)
        get_filter = filters_dict.get('filter%s' % cnt, '')
        while get_filter or cnt == 0:  # go there at least one, so we have a "ALL" filter
            cur_filter = get_filter.strip()
            # TODO: the $in limit should be managed globally, with a total/consumed managed at this level
            #      and if too much, raise a stop for the user so it need to filter more
            parsed_filter, is_filter_with_rights = _parse_one_filter(cur_filter, user, app.rg)
            is_already_managed_rights &= is_filter_with_rights  # we are managing rights is ALL are managing rights
            if parsed_filter:
                get_filters.append(parsed_filter)
            cnt += 1
            get_filter = filters_dict.get('filter%s' % cnt, '')
        
        date_filter = filters_dict.get('filter_days', '')
    except Exception:
        logger.warning('[%s] fail to parse filter number=%s : [%s] with error : [%s]' % (user.get_name(), cnt, filters_json, traceback.format_exc()))
        is_already_managed_rights = False  # we are not happy about filters, so do not risk a security issue here
    
    return date_filter, get_filters, filters_json, is_already_managed_rights


def _list_all_values():
    all_values = {
        u'realm': list(app.datamgr.get_realms()),
    }
    return all_values


def event_container():
    event_container_start = int(time.time())
    
    user = app.get_user_auth()
    configuration_id = app.datamgr.get_configuration_id()
    consumer_lock = getattr(app, u'consumer_lock', None)
    
    # Get pagination information
    first_ordering_uuid, page_size, page_index = _get_pagination_info()
    
    # Get all filtering
    date_filter, get_filters, filters_json, is_already_managed_rights = _get_filters(user)
    
    # Get all realm
    all_values = _list_all_values()
    
    event_container_module = get_event_container_module()
    
    # Items from Regenerator must not be used after this function call (as fair lock will be released)
    try:
        final_events, has_next = event_container_module.get_events(
            user,
            get_filters,
            date_filter,
            first_ordering_uuid=first_ordering_uuid,
            page_index=page_index,
            page_size=page_size,
            are_rights_already_managed=is_already_managed_rights,
            active_context_lock=consumer_lock,
            filters_json=filters_json
        )
    except ECDatabaseError as e:
        return app.abort(514, 'Can\'t get events from module [%s]. Connection to database [%s] is down.' % (event_container_module.get_name(), e._uri))
    except Exception:
        logger.error('[Event container] [%s] Cannot get events from the module for the user: %s' % (user.get_name(), traceback.format_exc()))
        return app.abort(500, 'Can\'t get events from module [%s]' % (event_container_module.get_name()))
    
    ret = {
        u'elements'                          : final_events,
        u'nb_element_filter'                 : len(final_events),
        u'has_next'                          : has_next,
        u'page_index'                        : page_index,
        u'configuration_id'                  : configuration_id,
        u'timestamp_backend_processing_start': event_container_start,
        u'timestamp_backend_processing_end'  : int(time.time()),
        u'all_values'                        : all_values,
    }
    _send_stat_to_module(u'Event manager', u'%s%s' % (app.request.path, remove_token_to_query_string(app.request.query_string)), filters_json, time.time() - event_container_start)
    return ret


def remove_token_to_query_string(request):
    request_splitted = request.split(u'&')
    return u'&'.join([param for param in request_splitted if not param.startswith(u'_token')])


def debug_event_container():
    user = app.get_user_auth()
    event_container_module = get_event_container_module()
    
    # Get pagination information
    first_ordering_uuid, page_size, page_index = _get_pagination_info()
    
    filters = []
    # if user.is_admin:
    #     filters = [u'host_name:nautilus Host-01', u'host_name:nautilus Host-02']
    
    date_filter = ''
    
    final_events, have_next = event_container_module.get_events(user, filters, date_filter, first_ordering_uuid=first_ordering_uuid, page_index=page_index, page_size=page_size)
    if not first_ordering_uuid:
        first_ordering_uuid = final_events[0]['ordering_uuid']
    return {'data': final_events, 'have_next': have_next, 'first_ordering_uuid': first_ordering_uuid, 'page_index': page_index, 'page_size': page_size}


pages = {
    event_container      : {'routes': ['/event-container'], 'wrappers': ['json', 'auth'], 'method': 'POST'},
    debug_event_container: {'routes': ['/event-container-debug'], 'view': 'event_container_debug', 'wrappers': ['auth']}
}
