#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2016-2022:
#    Shinken solutions

import json
import time
import uuid

from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.webui.bottlewebui import HTTPError

if TYPE_CHECKING:
    from webui.module import WebuiBroker
    from shinken.misc.type_hint import Optional

app = None  # type: Optional[WebuiBroker]


def list_hive():
    logger.debug('call list_hive')
    user = app.get_user_auth()
    
    # Second we look for bdd access.
    hive_collection = app.get_hive_collection()
    if not hive_collection:
        logger.info('save_hive No more backend for saving the data')
        return app.abort(500, 'No more backend for saving the data')
    
    # Our main key is the uuid
    owner_id = user.uuid
    hives = [e for e in hive_collection.find({'owner.uuid': owner_id}, {'tiles': 0, '_id': 0})]
    return json.dumps(hives)


def _get_raw_hive(hive_uuid):
    collection = app.get_hive_collection()
    if not collection:
        return 500, 'No more backend for saving the data'
    
    hive = collection.find_one({'_id': hive_uuid}, {'_id': 0})
    if not hive:
        return 404, 'Hive not found'
    
    return 200, hive


def get_hive(hive_uuid):
    logger.debug('call get_hive %s' % hive_uuid)
    
    _code, _data = _get_raw_hive(hive_uuid)
    
    if _code != 200:
        logger.info('[get_hive] %s' % _data)
        return app.abort(_code, _data)
    
    return json.dumps(_data)


def get_full_hive(hive_uuid):
    logger.debug('[get_full_hive]call get_full_hive %s' % hive_uuid)
    
    last_migration_version = app.request.GET.get('last_migration_version', '')
    if not last_migration_version:
        logger.info('[get_full_hive] missing last_migration_version parameter')
        return app.abort(400, 'Missing last_migration_version parameter, Please relaunch your navigator (F5)')
    
    _code, _data = _get_raw_hive(hive_uuid)
    if _code != 200:
        logger.info('[get_full_hive] %s' % _data)
        return app.abort(_code, _data)
    
    need_migration = last_migration_version >= _data['version']
    update_all_widgets = app.request.GET.get('update_all_widgets', False)
    for tile in _data.get('tiles', None):
        _dashboard_code, _dashboard_data = _get_dashboard_object_up_to_date(tile['screen']['uuid'], update_all_widgets=update_all_widgets)
        if _dashboard_code != 200:
            tile['screen'] = {'uuid': tile['screen']['uuid'], 'is_unknown': True}
        else:
            if need_migration:
                logger.debug('[get_full_hive] hive [%s] need migration' % hive_uuid)
            elif last_migration_version >= _dashboard_data['version']:
                logger.debug('[get_full_hive] dashboard [%s] need migration' % _dashboard_data['uuid'])
            elif update_all_widgets:
                logger.debug('[get_full_hive] dashboard [%s] need all widget [update_all_widgets=True was forced]' % _dashboard_data['uuid'])
            else:
                # remove the non favorite widget so save network and js computation
                _dashboard_data['widgets'] = [widget for widget in _dashboard_data.get('widgets', list()) if widget['favorite']]
            tile['screen'] = _dashboard_data
    return json.dumps(_data)


def save_hive_post():
    save_hive(create_object=True)


def save_hive(hive_uuid='', create_object=False):
    # WARNING uuid parameter is not used. It is ONLY for REST compliance.
    logger.debug('call save_hive %s' % hive_uuid)
    
    user_uuid = app.request.GET.get('user_uuid', None)
    user = app.get_user_auth()
    
    if user.uuid != user_uuid:
        return app.abort(403, 'User logout')
    
    if hasattr(app.request.body, 'getvalue'):
        data = json.loads(app.request.body.getvalue())
    else:
        data = json.loads(app.request.body.read())
    
    hive_uuid = data.get('uuid', '')
    if not hive_uuid:
        logger.info('[save_hive] Data to save need a uuid')
        return app.abort(422, 'Data to save need a uuid')
    
    data['_id'] = data['uuid']
    try:
        app.save_versionned_object('hive', 'uuid', hive_uuid, data, create_object=create_object)
    except ValueError as e:
        return app.abort(409, 'Save object failed upper version : %s' % getattr(e, 'message', 'No message'))
    except LookupError as e:
        return app.abort(404, 'Save object failed : %s' % getattr(e, 'message', 'No message'))
    except Exception as e:
        return app.abort(500, 'Save object failed : %s' % getattr(e, 'message', 'No message'))
    return {}


def del_hive(hive_uuid):
    logger.debug('call del_hive %s' % hive_uuid)
    
    # Second we look for bdd access.
    collection = app.get_hive_collection()
    if not collection:
        logger.info('del_hive No more backend for saving the data')
        return app.abort(500, 'No more backend for saving the data')
    
    if not hive_uuid:
        logger.info('del_hive Data to save need a uuid')
        return app.abort(422, 'Data to save need a uuid')
    
    write_result = collection.remove({'_id': hive_uuid})
    nb_del = write_result.get('n', 0)
    
    if nb_del == 0:
        logger.info('del_hive They are not hive with id : %s' % hive_uuid)
        return app.abort(404, 'They are not hive with id : %s' % hive_uuid)
    
    share_deleted = remove_link_share(hive_uuid)
    
    return json.dumps({'delete_hive': nb_del, 'share_deleted': share_deleted})


def clone_hive(hive_uuid):
    # WARNING parameter is not used ONLY for REST compliance.
    logger.debug('call clone_hive %s' % hive_uuid)
    user_uuid = app.request.GET.get('user_uuid', None)
    user = app.get_user_auth()
    if user.uuid != user_uuid:
        return app.abort(403, 'User logout')
    new_data = json.loads(app.request.body.read())
    _code, _data = _get_raw_hive(hive_uuid)
    if _code != 200:
        logger.info('[clone_hive] %s' % _data)
        return app.abort(_code, _data)
    
    # assign a new uuid to the hive
    new_hive_uuid = str(uuid.uuid4())
    _data['_id'] = new_hive_uuid
    _data['uuid'] = new_hive_uuid
    _data['name'] = new_data['name']
    _data['owner'] = {'uuid': user_uuid}
    _data.pop('saveVersion', None)
    
    collection = app.get_dashboard_collection()
    if not collection:
        return 500, 'No more backend for saving the data'
    
    for tile in _data.get('tiles', []):
        old_dashboard_uuid = tile['screen']['uuid']
        dashboard = collection.find_one({'uuid': old_dashboard_uuid}, {'_id': 0})
        
        # update the dashboard uuid
        new_dashboard_uuid = str(uuid.uuid4())
        tile['screen']['uuid'] = new_dashboard_uuid
        
        dashboard['_id'] = new_dashboard_uuid
        dashboard['uuid'] = new_dashboard_uuid
        dashboard['parentScreenUuid'] = new_hive_uuid
        dashboard.pop('saveVersion', None)
        for widget in dashboard.get('widgets', []):
            widget['uuid'] = str(uuid.uuid4())
        app.save_versionned_object('dashboard', 'uuid', new_dashboard_uuid, dashboard, create_object=True)
    try:
        app.save_versionned_object('hive', 'uuid', new_hive_uuid, _data, create_object=True)
    except ValueError as e:
        return app.abort(409, 'Save object failed upper version : %s' % getattr(e, 'message', 'No message'))
    except LookupError as e:
        return app.abort(404, 'Save object failed : %s' % getattr(e, 'message', 'No message'))
    except Exception as e:
        return app.abort(500, 'Save object failed : %s' % getattr(e, 'message', 'No message'))
    return json.dumps({'uuid': new_hive_uuid})


def get_bulk_dashboard(dashboard_uuid):
    logger.debug('call get_bulk_dashboard for hive %s' % dashboard_uuid)
    uuids = app.request.GET.get('uuids', None)
    
    dashboard_collection = app.get_dashboard_collection()
    if not dashboard_collection:
        logger.info('get_bulk_dashboard No more backend for saving the data')
        return app.abort(500, 'No more backend for saving the data')
    
    if not uuids:
        logger.info('get_bulk_dashboard We need the list of dashboard uuid')
        return app.abort(422, 'We need the list of dashboard uuid')
    
    uuids = uuids.replace('[', '').replace(']', '').split(',')
    return_value = [e for e in dashboard_collection.find({'uuid': {'$in': uuids}})]
    return json.dumps(return_value)


STATUS_PRIORITY = {
    0: {
        'name'    : 'ok',
        'priority': 0,
    },
    1: {
        'name'    : 'warning',
        'priority': 2,
    },
    2: {
        'name'    : 'critical',
        'priority': 3,
    },
    3: {
        'name'    : 'unknown',
        'priority': 1,
    },
    # only by sla, should never arrive here, only use for sla
    4: {
        'name'    : 'missing-data',
        'priority': 4,
    },
    5: {
        'name'    : 'shinken-inactive',
        'priority': 5,
    },
}

CONTEXT_INFO = {
    'NOTHING'               : {
        'name'    : 'nothing',
        'priority': 0,
    },
    'ACKNOWLEDGED'          : {
        'name'    : 'acknowledged',
        'priority': 15,
    },
    'PARTIAL-ACKNOWLEDGED'  : {
        'name'    : 'partial-acknowledged',
        'priority': 19,
    },
    'INHERITED-ACKNOWLEDGED': {
        'name'    : 'inherited-acknowledged',
        'priority': 10,
    },
    'DOWNTIME'              : {
        'name'    : 'downtime',
        'priority': 35,
    },
    'PARTIAL-DOWNTIME'      : {
        'name'    : 'partial-downtime',
        'priority': 39,
    },
    'INHERITED-DOWNTIME'    : {
        'name'    : 'inherited-downtime',
        'priority': 30,
    },
    'FLAPPING'              : {
        'name'    : 'flapping',
        'priority': 25,
    },
    'PARTIAL-FLAPPING'      : {
        'name'    : 'partial-flapping',
        'priority': 29,
    },
    'DISABLED'              : {
        'name'    : 'disabled',
        'priority': 404,
    },
}

PARTIAL_CONTEXTS = set(('PARTIAL-ACKNOWLEDGED', 'PARTIAL-DOWNTIME', 'PARTIAL-FLAPPING'))


def _parse_to_partial(context):
    if context in ('ACKNOWLEDGED', 'INHERITED-ACKNOWLEDGED', 'PARTIAL-ACKNOWLEDGED'):
        context = 'PARTIAL-ACKNOWLEDGED'
    elif context in ('DOWNTIME', 'INHERITED-DOWNTIME', 'PARTIAL-DOWNTIME'):
        context = 'PARTIAL-DOWNTIME'
    elif context in ('FLAPPING', 'INHERITED-FLAPPING', 'PARTIAL-FLAPPING'):
        context = 'PARTIAL-FLAPPING'
    return context


def _get_not_inherited_context(context):
    if context in ('INHERITED-DOWNTIME',):
        context = 'DOWNTIME'
    elif context in ('INHERITED-ACKNOWLEDGED',):
        context = 'ACKNOWLEDGED'
    return context


def _compute_status_and_context_from_element_ones(elt_to_update, current_elt_status, current_elt_context):
    nothing_priority = CONTEXT_INFO['NOTHING']['priority']
    elt_to_update_context = elt_to_update['context']
    elt_to_update_status = elt_to_update['status']
    
    # inherited and not inherited context count as same for partial
    current_not_inherited_context = _get_not_inherited_context(current_elt_context)
    to_update_not_inherited_context = _get_not_inherited_context(elt_to_update_context)
    
    # First context:
    # a python copy adaptation of https://github.com/shinken-solutions/enterprise/blob/V02.06.00-RC001/app/app/shared/element/shinken-element-parent.js computeContext
    elt_to_update_context_priority = CONTEXT_INFO[elt_to_update_context]['priority']
    current_elt_context_priority = CONTEXT_INFO[current_elt_context]['priority']
    if current_elt_context_priority > elt_to_update_context_priority:
        if elt_to_update_context_priority != nothing_priority and to_update_not_inherited_context != current_not_inherited_context:
            elt_to_update['context_is_partial'] = True
        elt_to_update['context'] = current_elt_context
    elif current_elt_context_priority == nothing_priority:
        if current_elt_status != 0:  # element status is not OK
            elt_to_update['context_is_partial'] = True
    elif current_elt_context_priority < elt_to_update_context_priority and to_update_not_inherited_context != current_not_inherited_context:
        elt_to_update['context_is_partial'] = True
    elif current_elt_context_priority == elt_to_update_context_priority:
        if current_elt_context in PARTIAL_CONTEXTS:
            elt_to_update['context_is_partial'] = True
    
    # Then Status:
    # Fast path: if elt_to_update was OK, element value always same or worse
    if elt_to_update_status == 0:
        elt_to_update['status'] = current_elt_status
    else:  # ok real case
        # status not fully manage,
        if current_elt_status > 4:
            logger.error('An element with the unmanaged status %s arrive in the status computation' % current_elt_status)
        elt_to_update_status_priority = STATUS_PRIORITY[elt_to_update_status]['priority']
        current_elt_status_priority = STATUS_PRIORITY[current_elt_status]['priority']
        if current_elt_status_priority > elt_to_update_status_priority:
            elt_to_update['status'] = current_elt_status


# We will take a dashboard entry from the database, and we will update all widgets elements
# status/context/is_unknown to current values
def _update_dashboard_with_up_to_date_widgets(dashboard, update_all_widgets=False):
    widgets = dashboard.get('widgets', [])
    
    dashboard['has_unknown_element'] = False
    dashboard['status'] = 0  # OK
    dashboard['context'] = 'NOTHING'
    dashboard['status_computed'] = False
    sla_plugin = next((m for m in app.modules_manager.imported_modules if getattr(m, 'MODE_SLA', '') == 'sla'), None)
    
    for widget in widgets:
        
        # BEWARE: if in the future we are changing the widget
        if 'inCalculation' not in widget or 'element' not in widget:
            if update_all_widgets and widget.get('metrics', None):
                for metric in widget['metrics']:
                    _data = app.widget_service.api_get_elt_widget_basic(metric['uuid'])
                    metric['is_unknown'] = _data['element']['is_unknown']
            continue
        
        # compute only the favorite widget (for now)
        favorite = widget['favorite']
        
        # manage the sla widget update
        if favorite and widget.get('name', '') == 'sla':
            element_uuid = widget['element']['uuid']
            if sla_plugin:
                try:
                    element_data = sla_plugin.internal_get_sla_widget_data(element_uuid)
                    widget['element'] = element_data['element']
                except HTTPError:
                    widget['element']['is_unknown'] = True
                continue
        
        # then update the the widget and dashboard status depending on inCalculation widget
        if not update_all_widgets and not widget['inCalculation'] and not favorite:
            # widget stats not in calculation and not in favorite need a reset of their context and status (mongo info are not fresh)
            if widget['name'] == 'detail':
                widget['context'] = 'NOTHING'
                widget['status'] = 3  # unknown
            continue
        
        if 'elements_for_compute' not in widget:
            continue
        
        if widget.get('output_type', 0) != 0 and widget.get('display_metrics', None):
            _data = app.widget_service.api_get_elt_widget_full(widget['element']['uuid'])
        elif widget.get('output_type', 0) != 0:
            _data = app.widget_service.api_get_elt_widget_output(widget['element']['uuid'])
        elif widget.get('display_metrics', None):
            _data = app.widget_service.api_get_elt_widget_checklist(widget['element']['uuid'])
        else:
            _data = app.widget_service.api_get_elt_widget_basic(widget['element']['uuid'])
        
        widget['element'].update(_data['element'])
        if _data.get('children_list', []):
            widget['children_list'] = _data['children_list']
        
        # Reset widget context & status
        widget['status'] = 0  # OK
        widget['context'] = 'NOTHING'
        widget['context_is_partial'] = False
        # clean the element here to be sure the frontend will not use it
        widget['element'].pop('status', None)
        widget['element'].pop('context', None)
        
        widget['has_unknown_element'] = False
        if widget['element']['is_unknown']:
            widget['status'] = 3
            widget['elements_for_compute'] = []
        # want_all_checks
        if widget['type_element'] == 'type_element_all':
            parent_element = widget['element']
            host_element = app.datamgr.get_host_by_uuid(parent_element['uuid'])
            if host_element:
                elements_for_compute = list(host_element.services)
                elements_for_compute.append(host_element)
                # reset the element for, all elements will be re-added when status will be ask
                widget['elements_for_compute'] = []
            else:
                elements_for_compute = []
                widget['status'] = 3
        else:
            elements_for_compute = widget['elements_for_compute']
        
        for element_for_compute in elements_for_compute:
            if isinstance(element_for_compute, dict):
                element_uuid = element_for_compute['uuid']
                if app.is_check_uuid(element_uuid):
                    backend_element = app.datamgr.get_service_by_uuid(element_uuid)
                else:
                    backend_element = app.datamgr.get_host_by_uuid(element_uuid)
                # If the element is missing, skip this
                if element_uuid is None or backend_element is None:
                    status = 3
                    context = 'NOTHING'
                    element_for_compute['status'] = status
                    element_for_compute['context'] = context
                    widget['has_unknown_element'] = True
                    if widget.get('inCalculation', False):
                        dashboard['has_unknown_element'] = True
                    element_for_compute['is_unknown'] = True
                    _compute_status_and_context_from_element_ones(widget, status, context)
                    continue
                element_for_compute['is_unknown'] = False
            else:
                backend_element = element_for_compute
            # NOTE: (at august 2018, 2.05.02) the 'not authorized' case on the widget still give  status+context, and with no is_unknown, so don't change the case here
            summary = app.helper.get_summary(backend_element)
            # So update the element so we can have an up to date Dashboard header
            status = summary['status']
            context = summary['summary']
            if isinstance(element_for_compute, dict):
                element_for_compute['status'] = status
                element_for_compute['context'] = context
                if widget.get('element_output', '') == backend_element.instance_uuid:
                    element_for_compute['long_output'] = backend_element.long_output
                    element_for_compute['output'] = backend_element.output
                    element_for_compute['display_name'] = backend_element.display_name
            else:
                # we probably have an host or a service object, create a new 'element_for_compute' to allow frontend to be ok with that service
                widget_element = {
                    'uuid'      : backend_element.instance_uuid,
                    'status'    : status,
                    'context'   : context,
                    'is_unknown': False,
                }
                if hasattr(backend_element, 'service_description'):
                    widget_element['service_description'] = backend_element.service_description
                widget['elements_for_compute'].append(widget_element)
                if widget.get('element_output', '') == backend_element.instance_uuid:
                    widget_element['long_output'] = backend_element.long_output
                    widget_element['output'] = backend_element.output
                    widget_element['display_name'] = backend_element.display_name
            # And update the widget status and context depend on the new element
            _compute_status_and_context_from_element_ones(widget, status, context)
        
        if widget['context_is_partial'] is True:
            widget['context'] = _parse_to_partial(widget['context'])
        widget.pop('context_is_partial', None)
        # update the dashboard status and context
        if widget['inCalculation']:
            _compute_status_and_context_from_element_ones(dashboard, widget['status'], widget['context'])
            dashboard['status_computed'] = True
    
    if not dashboard['status_computed']:
        dashboard['status'] = 3  # unknown
        dashboard['context'] = 'NOTHING'
    elif dashboard.get('context_is_partial', False):
        dashboard['context'] = _parse_to_partial(dashboard['context'])
    
    dashboard.pop('context_is_partial', None)
    dashboard.pop('status_computed', None)


def _get_dashboard_object_up_to_date(dashboard_uuid, update_all_widgets=False):
    collection = app.get_dashboard_collection()
    if not collection:
        return 500, 'No more backend for saving the data'
    dashboard = collection.find_one({'uuid': dashboard_uuid}, {'_id': 0})
    
    if not dashboard:
        return 404, 'Dashboard not found'
    
    # We are trying to update widgets values, but do it only if we are SURE
    # about which version we are talking
    # < 02.05.01: do nothing
    # * >= 2.05.01++:
    dashboard_version = dashboard.get('version', 'V02.03.03-U01')
    if dashboard_version >= 'V02.05.01':
        t0 = time.time()
        _update_dashboard_with_up_to_date_widgets(dashboard, update_all_widgets=update_all_widgets)
        logger.debug('Time to update the dashboard %s widgets: %.3f' % (dashboard_uuid, time.time() - t0))
    
    return 200, dashboard


def get_dashboard(dashboard_uuid):
    logger.debug('call get_dashboard %s' % dashboard_uuid)
    
    _code, _data = _get_dashboard_object_up_to_date(dashboard_uuid, update_all_widgets=True)
    
    if _code != 200:
        logger.info('[get_dashboard] %s' % _data)
        return app.abort(_code, _data)
    # Add list same groups and info on hive
    collection = app.get_hive_collection()
    if not collection:
        return 500, 'No more backend for get the data'
    
    parent_screen_uuid = _data.get('parentScreenUuid', None)
    
    hive = collection.find_one({'uuid': parent_screen_uuid}, {'owner': 1, 'name': 1})
    _data['parentScreenName'] = hive.get('name', None)
    _data['parentScreenOwner'] = hive.get('owner', None)
    _data['in_the_same_group'] = get_all_dashboards_with_same_group(parent_screen_uuid, _data.get('groupId'))
    
    return json.dumps(_data)


def get_all_dashboards_with_same_group(parent_screen_uuid, group_id):
    logger.debug('call get_all_dashboards_with_same_group Hive [%s] group [%s]' % (parent_screen_uuid, group_id))
    
    if group_id == '0':
        return json.dumps([])
    
    dashboard_collection = app.get_dashboard_collection()
    if not dashboard_collection:
        return 500, 'No more backend for get the data'
    if group_id is None:
        group_id = 0
    dashboard = [e for e in dashboard_collection.find({'parentScreenUuid': parent_screen_uuid, 'groupId': int(group_id)}, {'_id': 0, 'uuid': 1, 'name': 1})]
    return dashboard


def save_dashboard_post():
    save_dashboard(create_object=True)


def save_dashboard(dashboard_uuid='', create_object=False):
    logger.debug('call save_dashboard %s' % dashboard_uuid)
    
    if hasattr(app.request.body, 'getvalue'):
        data = json.loads(app.request.body.getvalue())
    else:
        data = json.loads(app.request.body.read())
    
    dashboard_uuid = data.get('uuid', '')
    if not dashboard_uuid:
        logger.info('save_dashboard Data to save need a uuid')
        return app.abort(422, 'Data to save need a uuid')
    
    if not data.get('parentScreenUuid', None):
        return app.abort(500, 'Save object failed : No parentScreenUuid given')
    
    data['_id'] = dashboard_uuid
    try:
        app.save_versionned_object('dashboard', 'uuid', dashboard_uuid, data, create_object=create_object)
    except ValueError as e:
        return app.abort(409, 'Save object failed upper version : %s' % getattr(e, 'message', 'No message'))
    except LookupError as e:
        return app.abort(404, 'Save object failed : %s' % getattr(e, 'message', 'No message'))
    except Exception as e:
        return app.abort(500, 'Save object failed : %s' % getattr(e, 'message', 'No message'))
    return {}


def del_dashboard(dashboard_uuid):
    logger.debug('call del_dashboard %s' % dashboard_uuid)
    
    # Second we look for bdd access.
    collection = app.get_dashboard_collection()
    if not collection:
        logger.info('del_dashboard No more backend for saving the data')
        return app.abort(500, 'No more backend for saving the data')
    
    write_result = collection.remove({'_id': dashboard_uuid})
    nb_del = write_result.get('n', 0)
    logger.info('del_dashboard dashboards deleted : %s' % nb_del)
    
    if nb_del == 0:
        logger.info('del_dashboard They are not dashboard with id : %s' % dashboard_uuid)
        return app.abort(404, 'They are not dashboard with id : %s' % dashboard_uuid)
    
    share_deleted = remove_link_share(dashboard_uuid)
    return json.dumps({'delete_dashboard': nb_del, 'share_deleted': share_deleted})


def list_list():
    logger.debug('call list_list')
    user = app.get_user_auth()
    
    # Second we look for bdd access.
    list_collection = app.get_list_collection()
    if not list_collection:
        logger.info('list_list No more backend for saving the data')
        return app.abort(500, 'No more backend for saving the data')
    
    # Our main key is the uuid
    owner_id = user.uuid
    lists = [e for e in list_collection.find({'owner.uuid': owner_id}, {'data': 0, '_id': 0})]
    return json.dumps(lists)


def get_list(list_uuid):
    logger.debug('call get_list %s' % list_uuid)
    
    # Second we look for bdd access.
    collection = app.get_list_collection()
    if not collection:
        logger.info('get_list No more backend for saving the data')
        return app.abort(500, 'No more backend for saving the data')
    
    # Our main key is the uuid
    dashboard = collection.find_one({'uuid': list_uuid}, {'_id': 0})
    
    # if no previous entries, None is good
    if dashboard:
        return json.dumps(dashboard)
    else:
        logger.info('get_list dashboard not found')
        return app.abort(404, 'Dashboard not found')


def save_list_post():
    save_list()


def save_list(list_uuid=''):
    logger.debug('call save_list %s' % list_uuid)
    
    # Second we look for bdd access.
    list_collection = app.get_list_collection()
    if not list_collection:
        logger.info('save_list No more backend for saving the data')
        return app.abort(500, 'No more backend for saving the data')
    
    if list_uuid and list_collection.find({'uuid': list_uuid}, {'uuid': 1}, only_count=True) == 0:
        logger.info('save_list They are not list with id : %s' % list_uuid)
        return app.abort(404, 'They are not list with id : %s' % list_uuid)
    
    # Third load the data to save.
    if hasattr(app.request.body, 'getvalue'):
        data = json.loads(app.request.body.getvalue())
    else:  # file?
        data = json.loads(app.request.body.read())
    
    # Forth check uuid availability.
    list_uuid = data.get('uuid', '')
    if not list_uuid:
        logger.info('save_list Data to save need a uuid')
        return app.abort(422, 'Data to save need a uuid')
    
    # Eventually we save the data.
    list_collection.update(filter={'_id': list_uuid}, update=data, upsert=True)
    return {}


def del_list(list_uuid):
    logger.debug('call del_list')
    
    # Second we look for bdd access.
    collection = app.get_list_collection()
    if not collection:
        logger.info('del_list No more backend for saving the data')
        return app.abort(500, 'No more backend for saving the data')
    
    write_result = collection.remove({'_id': list_uuid})
    nb_del = write_result.get('n', 0)
    logger.info('del_list lists deleted %s' % nb_del)
    
    if nb_del == 0:
        logger.info('del_hive They are not list with id : %s' % list_uuid)
        return app.abort(404, 'They are not list with id : %s' % list_uuid)
    
    share_deleted = remove_link_share(list_uuid)
    return json.dumps({'delete_list': nb_del, 'share_deleted': share_deleted})


def remove_link_share(share_uuid):
    share_collection = app.get_share_collection()
    share_deleted = [e.get('uuid', '') for e in share_collection.find({'screen.uuid': share_uuid}, {'uuid': 1})]
    share_collection.remove({'screen.uuid': share_uuid})
    return share_deleted


pages = {
    list_hive                         : {'routes': ['/screen/hive'], 'method': 'GET', 'wrappers': ['auth', 'json']},
    get_hive                          : {'routes': ['/screen/hive/:hive_uuid'], 'method': 'GET', 'wrappers': ['auth', 'json']},
    get_full_hive                     : {'routes': ['/screen/hive/:hive_uuid/full'], 'method': 'GET', 'wrappers': ['auth', 'json']},
    save_hive                         : {'routes': ['/screen/hive/:hive_uuid'], 'method': 'PUT', 'wrappers': ['auth', 'json']},
    save_hive_post                    : {'routes': ['/screen/hive'], 'method': 'POST', 'wrappers': ['auth', 'json']},
    del_hive                          : {'routes': ['/screen/hive/:hive_uuid'], 'method': 'DELETE', 'wrappers': ['auth', 'json']},
    clone_hive                        : {'routes': ['/screen/hive/:hive_uuid/clone'], 'method': 'POST', 'wrappers': ['auth', 'json']},
    
    get_bulk_dashboard                : {'routes': ['/screen/hive/:uuid/dashboard'], 'method': 'GET', 'wrappers': ['auth', 'json']},
    
    get_dashboard                     : {'routes': ['/screen/dashboard/:dashboard_uuid'], 'method': 'GET', 'wrappers': ['auth', 'json']},
    save_dashboard                    : {'routes': ['/screen/dashboard/:dashboard_uuid'], 'method': 'PUT', 'wrappers': ['auth', 'json']},
    save_dashboard_post               : {'routes': ['/screen/dashboard'], 'method': 'POST', 'wrappers': ['auth', 'json']},
    del_dashboard                     : {'routes': ['/screen/dashboard/:dashboard_uuid'], 'method': 'DELETE', 'wrappers': ['auth', 'json']},
    get_all_dashboards_with_same_group: {'routes': ['/screen/dashboard/groups/:parent_screen_uuid/:group_id'], 'method': 'GET', 'wrappers': ['auth', 'json']},
    
    list_list                         : {'routes': ['/screen/list'], 'method': 'GET', 'wrappers': ['auth', 'json']},
    get_list                          : {'routes': ['/screen/list/:list_uuid'], 'method': 'GET', 'wrappers': ['auth', 'json']},
    save_list                         : {'routes': ['/screen/list/:list_uuid'], 'method': 'PUT', 'wrappers': ['auth', 'json']},
    save_list_post                    : {'routes': ['/screen/list'], 'method': 'POST', 'wrappers': ['auth', 'json']},
    del_list                          : {'routes': ['/screen/list/:list_uuid'], 'method': 'DELETE', 'wrappers': ['auth', 'json']},
}
