#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019
# This file is part of Shinken Enterprise, all rights reserved.

import StringIO
import base64
import json
import string
import time
import urllib
from datetime import datetime, timedelta

import pyexcel

from shinken.log import logger
from shinken.misc.filter import is_authorized
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.config import ILLEGAL_NAME_CHARACTERS
from shinkensolutions.date_helper import get_now, Date
from shinkensolutions.shinken_time_helper import print_human_readable_period, DisplayFormat
from shinkensolutions.ssh_mongodb.mongo_error import ShinkenMongoException

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional
    from ....sla.module import SLAModuleWebUI
    from ....webui.module import WebuiBroker

FULL_RANGE_DISPLAY = 'full'
app = None  # type: Optional[WebuiBroker]

PRECISION = 2
_LABEL_PATERN = '%.' + str(PRECISION) + 'f%%'
_DEFAULT_PATERN = '%.' + str(PRECISION + 2) + 'f%%'
_THRESHOLD_PATERN = '%.' + str(PRECISION + 1) + 'f%%'


class ReportException(Exception):
    pass


class ConfigurationException(Exception):
    pass


MODE_HISTORY_SLA = 'history_sla'
MODE_HISTORY = 'history'
MODE_SLA = 'sla'

DATA_TYPES = [
    MODE_HISTORY_SLA,
    MODE_HISTORY,
    MODE_SLA,
]

MODE_TO_GET = {
    MODE_HISTORY_SLA: ['history', 'sla'],
    MODE_HISTORY    : ['history'],
    MODE_SLA        : ['sla'],
}

MODE_ACCES = {
    MODE_HISTORY_SLA: ['history', 'sla', 'history_sla'],
    MODE_HISTORY    : ['history'],
    MODE_SLA        : ['sla'],
}

ALL_STATES = ['ok', 'warn', 'crit', 'unknown', 'missing', 'inactive']

KEY_TIME = 'time_%s'


class FORM_FIELD:
    DATA_TYPE = 'data_type'
    DAILY_SLA_GRAPH = 'shinken_daily_sla_graph'
    SCALE = 'shinken_graph_scale'
    YMIN = 'shinken_graph_ymin'
    YMIN_VALUE = 'shinken_graph_ymin_value'
    YMAX = 'shinken_graph_ymax'
    YMAX_VALUE = 'shinken_graph_ymax_value'


Y_MAX_VALUE = 100
Y_MIN_VALUE = 0
NB_DATA_MAX_GRAPH = 31


class GRAPHIC_SCALE:
    AUTO = 'SCALE_AUTO'
    LINEARE = 'SCALE_LINEAR'
    ZOOM_TOP = 'SCALE_ZOOM_TOP'
    LOGARITHMIC = 'SCALE_LOGARITHMIC'
    
    ALL = [
        AUTO,
        LINEARE,
        ZOOM_TOP,
        LOGARITHMIC,
    ]


class GRAPHIC_Y_MAX_OPTION:
    AUTO = 'Y_MAX_AUTO'
    VALEUR = 'Y_MAX_VALEUR'
    MAXIMUM_SLA = 'Y_MAX_MAXIMUM_SLA'
    SEUIL_CRITIC = 'Y_MAX_SEUIL_CRITIC'
    SEUIL_WARNING = 'Y_MAX_SEUIL_WARNING'
    
    ALL = [
        AUTO,
        VALEUR,
        MAXIMUM_SLA,
        SEUIL_CRITIC,
        SEUIL_WARNING,
    ]


class GRAPHIC_Y_MIN_OPTION:
    AUTO = 'Y_MIN_AUTO'
    VALEUR = 'Y_MIN_VALEUR'
    MINIMUM_SLA = 'Y_MIN_MINIMUM_SLA'
    SEUIL_CRITIC = 'Y_MIN_SEUIL_CRITIC'
    SEUIL_CRITIC_CENTER = 'Y_MIN_SEUIL_CRITIC_CENTER'
    SEUIL_WARNING = 'Y_MIN_SEUIL_WARNING'
    SEUIL_WARNING_CENTER = 'Y_MIN_SEUIL_WARNING_CENTER'
    
    ALL = [
        AUTO,
        VALEUR,
        MINIMUM_SLA,
        SEUIL_CRITIC,
        SEUIL_CRITIC_CENTER,
        SEUIL_WARNING,
        SEUIL_WARNING_CENTER,
    ]


AUTO_SETTINGS = dict()
KEY_Y_MAX = 'y_max'
KEY_Y_MIN = 'y_min'
KEY_OPTION = 'option'
KEY_VALEUR = 'valeur'

AUTO_SETTINGS[GRAPHIC_SCALE.AUTO] = {
    KEY_Y_MAX: {
        KEY_OPTION: GRAPHIC_Y_MAX_OPTION.VALEUR,
        KEY_VALEUR: 100
    },
    KEY_Y_MIN: {
        KEY_OPTION: GRAPHIC_Y_MIN_OPTION.SEUIL_CRITIC_CENTER,
        KEY_VALEUR: 0
    }
}
AUTO_SETTINGS[GRAPHIC_SCALE.LINEARE] = {
    KEY_Y_MAX: {
        KEY_OPTION: GRAPHIC_Y_MAX_OPTION.VALEUR,
        KEY_VALEUR: 100
    },
    KEY_Y_MIN: {
        KEY_OPTION: GRAPHIC_Y_MIN_OPTION.MINIMUM_SLA,
        KEY_VALEUR: 0.1
    }
}
AUTO_SETTINGS[GRAPHIC_SCALE.ZOOM_TOP] = {
    KEY_Y_MAX: {
        KEY_OPTION: GRAPHIC_Y_MAX_OPTION.VALEUR,
        KEY_VALEUR: 100
    },
    KEY_Y_MIN: {
        KEY_OPTION: GRAPHIC_Y_MIN_OPTION.SEUIL_CRITIC_CENTER,
        KEY_VALEUR: 0
    }
}
AUTO_SETTINGS[GRAPHIC_SCALE.LOGARITHMIC] = {
    KEY_Y_MAX: {
        KEY_OPTION: GRAPHIC_Y_MAX_OPTION.VALEUR,
        KEY_VALEUR: 100
    },
    KEY_Y_MIN: {
        KEY_OPTION: GRAPHIC_Y_MIN_OPTION.VALEUR,
        KEY_VALEUR: 0
    }
}


class FILTER_TYPE:
    DEPENDENCY_CLUSTER = 'dependency/cluster/'
    DEPENDENCY_CLUSTER_CHECK = 'dependency/cluster/check/'
    NORMAL_CLUSTER = 'normal/cluster/'
    NORMAL_CLUSTER_CHECK = 'normal/cluster/check/'
    NORMAL_HOST = 'normal/host/'
    NORMAL_HOST_CHECK = 'normal/host/check/'
    NORMAL_CHECK = 'normal/check/'
    NORMAL_HOST_TEMPLATE = 'normal/host_template/'
    NORMAL_HOST_TEMPLATE_CHECK = 'normal/host_template/check/'
    NORMAL_HOSTGROUP = 'normal/hostgroup/'
    NORMAL_HOSTGROUP_CHECK = 'normal/hostgroup/check/'
    
    ALL = [
        DEPENDENCY_CLUSTER,
        DEPENDENCY_CLUSTER_CHECK,
        NORMAL_CLUSTER,
        NORMAL_CLUSTER_CHECK,
        NORMAL_HOST,
        NORMAL_HOST_CHECK,
        NORMAL_CHECK,
        NORMAL_HOST_TEMPLATE,
        NORMAL_HOST_TEMPLATE_CHECK,
        NORMAL_HOSTGROUP,
        NORMAL_HOSTGROUP_CHECK
    ]


def safe_cast(val, to_type, default=None):
    try:
        return to_type(val)
    except (ValueError, TypeError):
        return default


def _send_stat_to_sla_module(stat_name, stat_endpoint, stat_time):
    sla_module = get_sla_module()
    sla_module.send_stat_to_reader_stats((stat_name, stat_endpoint, stat_time))


def total_table_sla_history_value_percent(value):
    if 0 <= value < 0.0001:
        return "< 0.01 %"
    return "%.2f %%" % (value * 100)


def html_get_value_bar(state, mode, value, time):
    if value > 0:
        return app.helper.get_value_bar(state, mode, value)
    if mode == 'sla':
        state_text = app._('reporting.sla')
        if state == 'ok':
            state_text += '''<span class="sla-icon shinicon-arrow-up"></span>'''
        else:
            state_text += '''<span class="sla-icon shinicon-arrow-down"></span>'''
        
        tooltip = 'data-toggle="tooltip" title="%s"' % app._('reporting.%s' % state)
    else:
        state_text = app._('reporting.%s' % state)
        tooltip = ''
    
    value_bar_html = '''
    <tr class="status-bar">
        <td class="icon-cell" %(tooltip)s>
            <div>
                %(icon)s
            </div>
        </td>
        <td class="status-label-cell">
            <div class="status-label">%(state_text)s</div>
        </td>
        <td class="status-bar-message">
                %(time)s
        </td>
        <td class="percentage-cell">
            <div class="percentage">< 0.01%%</div>
        </td>
    </tr>
    ''' % {
        "time"      : print_human_readable_period(time, display_format=_get_readable_period_date_format()),
        "icon"      : app.helper.get_sla_state_icon(state),
        "state_text": state_text,
        "state"     : state,
        "value"     : value * 100,
        "tooltip"   : tooltip,
    }
    return value_bar_html


def get_sla_module():
    # type: () -> SLAModuleWebUI
    sla_module = next((mod for mod in app.modules_manager.get_all_instances() if mod.properties['type'] == 'sla'), None)
    if not sla_module:
        logger.error('The sla module was not found')
        return app.abort(500, 'The sla module was not found')
    return sla_module


def internal_get_sla_widget_data(uuid):
    user = app.get_user_auth()
    _data = {
        'element': {
            'uuid'      : uuid,
            'is_unknown': False,
            'unsight'   : False
        }
    }
    data_element = _data['element']
    
    # Get item for uuid
    if app.is_check_uuid(uuid):
        item = app.datamgr.get_service_by_uuid(uuid)
        if item:
            data_element['host_name'] = item.host.get_name()
            data_element['service_description'] = item.service_description
    else:
        item = app.datamgr.get_host_by_uuid(uuid)
        if item:
            data_element['host_name'] = item.get_name()
    
    if item is None:
        return app.abort(404, '[get_sla_widget_data] Element is not found', True)
    if not is_authorized(item, user, app):
        logger.info('[get_sla_widget_data] User is not authorized to view this element')
        return {'element': app.helper.get_unauthorized_item(item)}
    
    sla_module = get_sla_module()
    
    today_archive, week_info = sla_module.compute_sla_week_info(uuid)
    current_tendency = sla_module.get_tendency_for_item(item, today_archive)
    
    summary = app.helper.get_summary(item)
    
    data_element['status'] = summary['status']
    data_element['sla'] = {
        'week_info'       : week_info,
        'current_tendency': current_tendency,
    }
    
    return _data


def get_sla_widget_data(uuid):
    t0 = time.time()
    stat_name = 'Widget'
    endpoint = app.request.path + app.request.query_string
    
    _data = internal_get_sla_widget_data(uuid)
    
    _send_stat_to_sla_module(stat_name, endpoint, time.time() - t0)
    return json.dumps(_data)


def get_history_ranges(uuid):
    user = app.get_user_auth()
    
    item_is_a_service = app.is_check_uuid(uuid)
    if item_is_a_service:
        item = app.datamgr.get_service_by_uuid(uuid)
    else:
        item = app.datamgr.get_host_by_uuid(uuid)
    
    if item is None:
        return app.abort(404, '[get_history_ranges] Element is not found', True)
    
    if not is_authorized(item, user, app):
        logger.info('[get_history_ranges] User is not authorized to view this element')
        return json.dumps(app.helper.get_unauthorized_item(item))
    
    is_display_full_ranges = bool(app.request.GET.get('display_full_ranges', False))
    
    current_range_display = app.request.GET.get('current_range_display', '')
    if not current_range_display:
        return app.abort(400, '[get_history_ranges] missing parameter current_range_display', True)
    
    try:
        current_range_display = int(current_range_display)
    except ValueError:
        return app.abort(400, '[get_history_ranges] parameter current_range_display is malformed', True)
    
    max_range_display = app.history__nb_changes_displayed
    if not is_display_full_ranges:
        try:
            max_range_display = int(app.request.GET.get('max_range_display', app.history__nb_changes_displayed))
        except ValueError:
            max_range_display = app.history__nb_changes_displayed
    
    date = app.request.GET.get('date', '')
    if not date:
        return app.abort(400, '[get_history_ranges] missing parameter date', True)
    try:
        date = Date(int(date[4:]), int(date[:4]))
    except ValueError:
        return app.abort(400, '[get_history_ranges] parameter date is malformed', True)
    
    sla_module = get_sla_module()
    if is_display_full_ranges:
        entry = sla_module.read_archive(uuid, date=date)
        _prepare_entry(entry, current_range_display, uuid, sla_module, full_mode=True)
    else:
        entry = sla_module.read_archive(uuid, date=date, limit_ranges=(current_range_display, max_range_display))
        _prepare_entry(entry, current_range_display, uuid, sla_module)
    return entry


def get_history(uuid):
    t0 = time.time()
    stat_name = 'History'
    endpoint = app.request.path + app.request.query_string
    
    user = app.get_user_auth()
    
    # Look for an host or a service?
    item_is_a_service = app.is_check_uuid(uuid)
    if item_is_a_service:
        item = app.datamgr.get_service_by_uuid(uuid)
    else:
        item = app.datamgr.get_host_by_uuid(uuid)
    
    if item is None:
        return app.abort(404, '[get_history] Element is not found', True)
    
    if not is_authorized(item, user, app):
        logger.info('[get_history] User is not authorized to view this element')
        return json.dumps(app.helper.get_unauthorized_item(item))
    
    sla_module = get_sla_module()
    max_range_display = app.request.GET.get('max_range_display', app.history__nb_changes_displayed)
    if max_range_display != FULL_RANGE_DISPLAY:
        try:
            max_range_display = int(max_range_display)
        except ValueError:
            max_range_display = app.history__nb_changes_displayed
    
    get_start = app.request.GET.get('start', '')
    get_end = app.request.GET.get('end', '')
    
    current_time = get_now()
    tm = time.localtime(current_time)
    current_date = Date(tm.tm_yday, tm.tm_year)
    
    if get_end:
        end_date = Date(int(get_end[4:]), int(get_end[:4]))
    else:
        end_date = current_date
    
    if get_start:
        start_date = Date(int(get_start[4:]), int(get_start[:4]))
    else:
        item_monitoring_start_time = sla_module.get_monitoring_start_time(item_uuid=uuid)
        if item_monitoring_start_time == -1:
            item_monitoring_start_time = sla_module.get_first_monitoring_start_time().timetuple()
        else:
            item_monitoring_start_time = datetime.fromtimestamp(item_monitoring_start_time).timetuple()
        start_date = Date(item_monitoring_start_time.tm_yday, item_monitoring_start_time.tm_year)
    
    date_ranges = (start_date, end_date)
    logger.debug('[get_history] history for [%s-%s] %s' % (item.get_full_name(), uuid, date_ranges))
    
    limit_ranges = None if max_range_display == FULL_RANGE_DISPLAY else (0, max_range_display)
    entries = sla_module.read_archive(uuid, date_ranges=date_ranges, limit_ranges=limit_ranges, only_existing_day=True)
    
    sla_threshold = (item.sla_warning_threshold, item.sla_critical_threshold)
    for i, entry in enumerate(entries):
        _day = time.strptime('%d %d' % (entry['year'], entry['yday']), '%Y %j')
        start_of_day = int(time.mktime(_day))
        entry['timestamp'] = start_of_day
        _prepare_entry(entry, 0, uuid, sla_module)
        if i == 0:
            sla_threshold = entry['thresholds']
    
    data = {
        'sla_threshold': sla_threshold,
        'days'         : entries,
    }
    _send_stat_to_sla_module(stat_name, endpoint, time.time() - t0)
    return json.dumps(data)


def _prepare_entry(entry, current_range_display, uuid, sla_module, full_mode=False):
    entry.pop('_id', None)
    
    total_ranges = entry['total_ranges']
    current_ranges = current_range_display if full_mode else len(entry['ranges']) + current_range_display
    have_all_range = current_ranges >= total_ranges
    
    entry['@page_info'] = {
        'have_all_range'  : have_all_range,
        'total_ranges'    : total_ranges,
        'current_ranges'  : current_ranges,
        'ranges_page_size': app.history__nb_changes_displayed,
    }
    
    for r in entry['ranges']:
        _prepare_range(r, uuid, sla_module)


def _prepare_range(range, uuid, sla_module):
    context = 'NOTHING'
    status = range['rc']
    if range.pop('p_flg', False):
        context = 'PARTIAL-FLAPPING'
    if range.pop('flg', False):
        context = 'FLAPPING'
    if range.pop('p_ack', False):
        context = 'PARTIAL-ACKNOWLEDGED'
    if range['ack'] == 2:
        context = 'INHERITED-ACKNOWLEDGED'
    if range['ack'] == 1 or range['ack'] is True:
        context = 'ACKNOWLEDGED'
    if range.pop('p_dt', False):
        context = 'PARTIAL-DOWNTIME'
    if range['dt'] == 2:
        context = 'INHERITED-DOWNTIME'
    if range['dt'] == 1 or range['dt'] is True:
        context = 'DOWNTIME'
    range['status'] = status
    range['context'] = context
    ack_uuid = range.pop('ack_uuid', '')
    downtimes_uuid = range.pop('downtimes_uuid', [])
    inherited_downtime_uuids = []
    if range['dt'] and downtimes_uuid:
        if app.is_check_uuid(uuid):
            service = app.datamgr.get_service_by_uuid(uuid)
            inherited_downtime_uuids = [downtime.uuid for downtime in service.host.downtimes]
            for uuid in inherited_downtime_uuids:
                if uuid in downtimes_uuid:
                    downtimes_uuid.remove(uuid)
        dts = list(sla_module.find_downtimes(downtimes_uuid))
        idts = list(sla_module.find_downtimes(inherited_downtime_uuids))
        if dts:
            range['downtimes'] = [{'author': dt['author'], 'comment': dt['comment']} for dt in dts]
        if idts:
            range['inherited_downtimes'] = [{'author': idt['author'], 'comment': idt['comment']} for idt in idts]
    if not range['dt']:
        if ack_uuid:
            a = sla_module.find_acknowledge(ack_uuid)
            if a:
                ack = {
                    'author' : a['author'],
                    'comment': a['comment'],
                }
                if range['ack'] == 1:
                    range['acknowledged'] = ack
                elif range['ack'] == 2:
                    range['inherited_acknowledged'] = ack
    
    del range['rc']
    del range['ack']
    del range['dt']


# An history entry got a year and a yday.
def _sort_history(e1, e2):
    y1 = e1['year']
    y2 = e2['year']
    d1 = e1['yday']
    d2 = e2['yday']
    if y1 != y2:
        return -1 * cmp(y1, y2)
    return -1 * cmp(d1, d2)


def get_reporting_form():
    # First we look for the user sid
    # so we bail out if it's a false one
    user = app.get_user_auth()
    err = app.request.GET.get('err', '')
    
    previous_search = app.request.query
    
    # Protect all form parameters about bad characters we do not want
    # We take as invalide name characters (because it also match lot of JS)
    form_parameters = (
        ('dateBegin', app._('reporting.period_from')),
        ('dateEnd', app._('reporting.period_to')),
        ('type', app._('reporting.filtre_by')),
        ('filter', app._('reporting.filtre_value')),
        ('data_type', app._('reporting.data_to_display')),
        ('warning_threshold', app._('reporting.warning_threshold')),
        ('critical_threshold', app._('reporting.critical_threshold'))
    )
    
    for parameter, display_name in form_parameters:
        value = previous_search.get(parameter, '')
        for c in ILLEGAL_NAME_CHARACTERS:
            if c in value:
                err = app._('reporting.invalide_character') % (display_name, c)
                break
    
    sla_module = get_sla_module()
    return {
        'app'                   : app,
        'user'                  : user,
        'err'                   : err,
        'FILTER_TYPE'           : FILTER_TYPE(),
        'previous_search'       : previous_search,
        'DEFAULT_SLA_THRESHOLDS': sla_module.get_default_sla_thresholds()
    }


def _get_readable_period_date_format():
    if app.lang == u'fr':
        return DisplayFormat('%.2fms', '%.2fs', '%sm', '%sh', '%s jours')
    return DisplayFormat('%.2fms', '%.2fs', '%sm', '%sh', '%s days')


def reporting_csv():
    app.response.headers['Content-Type'] = 'text/csv'
    login = app.request.GET.get('login', '')
    password = app.request.GET.get('password', '')
    if login and password:
        try:
            password = base64.b64decode(password).decode('utf8', 'ignore')
            login = base64.b64decode(login).decode('utf8', 'ignore')
        except Exception, e:  # bad encoding? get out!
            return app.abort(401, 'reporting_csv authentification error %s' % e, True)
        user = app.authenticate_user_by_modules(login, password, u'REPORTING')
    else:
        user = app.get_user_auth()
    
    if not user:
        return app.abort(401, 'reporting_csv user is not authenticated', True)

    current_search = _parse_form_to_dict(app, user)
    
    format = app.request.GET.get('format', 'label')
    date_begin = current_search['dateBegin']
    date_end = current_search['dateEnd']
    search_type = current_search['type']
    search_filter = current_search['filter']
    data_type = current_search['data_type']
    from_web = current_search['from_web']
    shinken_add_daily_details = current_search['shinken_add_daily_details']
    
    if data_type not in MODE_ACCES[user.acl_in_tab_history]:
        return app.abort(401, 'reporting_csv user cannot see [%s]' % data_type, True)
    
    try:
        reporting_data = _build_reporting_data(current_search, user)
    except ConfigurationException as e:
        if from_web:
            return app.bottle.redirect('/reporting?err=%s%s' % (urllib.quote_plus(e.message.encode('utf-8')), _restore_form_state(current_search)))
        return app.abort(500, e.message, True)
    except ReportException as e:
        if from_web:
            return app.bottle.redirect('/reporting?err=%s%s' % (urllib.quote_plus(e.message.encode('utf-8')), _restore_form_state(current_search)))
        return app.abort(400, e.message, True)
    
    valid_chars = '-_.() %s%s' % (string.ascii_letters, string.digits)
    _search_filter_clean = ''.join(c for c in search_filter if c in valid_chars)
    filename = '[%s]-[%s] %s [%s]' % (date_begin, date_end, app._('reporting.file_name_%s' % (search_type.replace('/', '_'))), _search_filter_clean)
    logger.debug('filename : [%s]' % filename)
    app.response.headers['Content-Disposition'] = 'attachment; filename="%s.csv"' % (filename)
    
    logger.debug('[reporting_csv] sla_result [%s]' % reporting_data)
    total_lines = []
    for mode in MODE_TO_GET[data_type]:
        total_line = [app._('reporting.total') % app._('reporting.%s' % mode)]
        for state in ALL_STATES:
            _value = reporting_data['result'][mode].get(state, 0)
            if _value:
                total_line.append(app._('reporting.%s' % state))
                total_line.append(_percent_format(_value, format))
        total_lines.append(total_line)
    
    lines_elements = []
    # ['Date', 'Element name', 'Type', 'Status', 'Value', 'Context'],
    date_range = '%s - %s' % (current_search['dateBegin'], current_search['dateEnd'])
    date_readable_format = _get_readable_period_date_format()
    for mode in MODE_TO_GET[data_type]:
        for item in reporting_data['by_item']:
            item_type = item['type']
            name = '%s/%s' % (item['hname'], item['sdesc']) if item_type == 'service' else item['hname']
            thresholds = item['result']['sla']['thresholds']
            
            if current_search['display_element_threshold'] == 'on':
                lines_elements.append([date_range, name, app._('reporting.%s' % mode), app._('reporting.on_treshold'), app._('reporting.warn'), "%s" % _percent_format(thresholds[0], "threshold")])
                lines_elements.append([date_range, name, app._('reporting.%s' % mode), app._('reporting.on_treshold'), app._('reporting.crit'), "%s" % _percent_format(thresholds[1], "threshold")])
            
            for state in ALL_STATES:
                if state in item['result'][mode]:
                    _value = item['result'][mode][state]
                    _time = item['result'][mode][KEY_TIME % state]
                    _time_str = print_human_readable_period(_time, display_format=date_readable_format)
                    lines_elements.append([date_range, name, app._('reporting.%s' % mode), app._('reporting.on_range'), app._('reporting.%s' % state), "%s" % _percent_format(_value, format), "%s" % _time_str])
            
            if shinken_add_daily_details == 'on':
                for day in item['by_day']:
                    _date = datetime.fromtimestamp(day['day']).strftime('%Y-%m-%d')
                    thresholds = day['result']['sla']['thresholds']
                    if current_search['display_element_threshold'] == 'on':
                        lines_elements.append([_date, name, app._('reporting.%s' % mode), app._('reporting.on_treshold'), app._('reporting.warn'), "%s" % _percent_format(thresholds[0], "threshold")])
                        lines_elements.append([_date, name, app._('reporting.%s' % mode), app._('reporting.on_treshold'), app._('reporting.crit'), "%s" % _percent_format(thresholds[1], "threshold")])
                    
                    for state in ALL_STATES:
                        if state in day['result'][mode]:
                            _value = day['result'][mode][state]
                            _time = day['result'][mode][KEY_TIME % state]
                            _time_str = print_human_readable_period(_time, display_format=date_readable_format)
                            lines_elements.append([_date, name, app._('reporting.%s' % mode), app._('reporting.on_day'), app._('reporting.%s' % state), "%s" % _percent_format(_value, format), "%s" % _time_str])
    
    thresholds = reporting_data['result']['sla']['thresholds']
    csv_array = [
        ['###########################################################################################################################'],
        [app._('reporting.user_info1')],
        [app._('reporting.user_info2')],
        ['###########################################################################################################################'],
        [''],
        [reporting_data['reportname']],
        [''],
        [app._('reporting.report_type'), current_search['type']],
        [app._('reporting.value_label'), current_search['filter'].decode('utf-8')],
        [app._('reporting.daily_details_add'), app._('reporting.on') if shinken_add_daily_details == 'on' else app._('reporting.off')],
        [app._('reporting.period_from'), current_search['dateBegin']],
        [app._('reporting.period_to'), current_search['dateEnd']],
    ]
    
    if current_search['display_report_threshold'] == 'on' and data_type != MODE_HISTORY:
        csv_array.append([app._('reporting.critical_threshold'), "%s" % _percent_format(thresholds[0], "threshold")])
        csv_array.append([app._('reporting.warning_threshold'), "%s" % _percent_format(thresholds[1], "threshold")])
    
    csv_array.append([''])
    
    csv_array.extend(total_lines)
    csv_array.extend([
        [''],
        [app._('reporting.date'), app._('reporting.name'), app._('reporting.data_to_display'), app._('reporting.range'), app._('reporting.state'), app._('reporting.value'), app._('reporting.time')],
    ])
    csv_array.extend(lines_elements)
    
    sheet = pyexcel.Sheet(csv_array)
    output = StringIO.StringIO()
    sheet.save_to_memory(file_type='csv', stream=output, delimiter=';', encoding='utf-8-sig')
    return output.getvalue()


def reporting_web():
    t0 = time.time()
    stat_name = 'Report'
    endpoint = app.request.path + app.request.query_string
    
    user = app.get_user_auth()
    
    current_search = _parse_form_to_dict(app, user)
    
    data_type = current_search['data_type']
    if data_type not in MODE_ACCES[user.acl_in_tab_history]:
        message = app._('reporting.unauthorized_history_access') % data_type
        current_search.pop('data_type', None)
        return app.bottle.redirect('/reporting?err=%s%s' % (urllib.quote_plus(message.encode('utf-8')), _restore_form_state(current_search)))
    
    try:
        reporting_data = _build_reporting_data(current_search, user)
    except (ReportException, ConfigurationException) as e:
        return app.bottle.redirect('/reporting?err=%s%s' % (urllib.quote_plus(e.message.encode('utf-8')), _restore_form_state(current_search)))
    
    _send_stat_to_sla_module(stat_name, endpoint, time.time() - t0)
    
    items = reporting_data['by_item']
    for i in range(len(items)):
        gen_sla_day_graph = (False for day in items[i]['by_day'][-NB_DATA_MAX_GRAPH:] if _get_sla_of_day(day) is not None)
        items[i]['no_data_graph'] = next(gen_sla_day_graph, True)
    
    reporting_data_graph = _get_reporting_data_graph(current_search, items)
    return {
        'app'                 : app,
        'user'                : user,
        'reporting_data'      : reporting_data,
        'reporting_data_graph': reporting_data_graph,
        'current_search'      : current_search,
        'current_url'         : _restore_form_state(current_search)
    }


def _get_reporting_data_graph(current_search, items):
    reporting_data_graph = []
    if current_search['shinken_daily_sla_graph'] == 'on':
        reporting_data_graph = [None] * len(items)
        
        for i in range(len(items)):
            xAxes = [day['day'] for day in items[i]['by_day']]
            
            sla_data = [_get_sla_of_day(day) for day in items[i]['by_day']]
            tCritic = [day['result']['sla']['thresholds'][1] for day in items[i]['by_day']]
            tWarn = [day['result']['sla']['thresholds'][0] for day in items[i]['by_day']]
            
            y_min, y_max = _calcul_y_min_y_max_graph(sla_data, tCritic, tWarn, current_search)
            
            scale_type = current_search['shinken_graph_scale']
            if y_min == 0 and (scale_type == GRAPHIC_SCALE.AUTO or scale_type == GRAPHIC_SCALE.ZOOM_TOP):
                scale_type = GRAPHIC_SCALE.LINEARE
            
            reporting_data_graph[i] = {
                'no_data_graph'      : items[i]['no_data_graph'],
                'item_uuid'          : items[i]['item'].get_instance_uuid(),
                'hname'              : items[i]['hname'],
                'xAxes'              : xAxes,
                'scale_type'         : scale_type,
                'yMax'               : y_max,
                'yMin'               : y_min,
                'warning_thresholds' : {
                    'label': app._('reporting.warning_threshold'),
                    'yAxes': tWarn,
                },
                'critical_thresholds': {
                    'label': app._('reporting.critical_threshold'),
                    'yAxes': tCritic,
                },
                'sla'                : {
                    'label': app._('reporting.sla'),
                    'yAxes': sla_data
                },
            }
    return reporting_data_graph


def _get_sla_of_day(day):
    if 'missing' in day['result'] and day['result']['missing']:
        return None
    if 'ok' not in day['result']['sla']:
        return 0
    return day['result']['sla']['ok'] * 100


def _calcul_y_min_y_max_graph(sla, tCritic, tWarn, current_search):
    sla = sla[-NB_DATA_MAX_GRAPH:]
    tCritic = tCritic[-NB_DATA_MAX_GRAPH:]
    tWarn = tWarn[-NB_DATA_MAX_GRAPH:]
    
    y_max = _cal_ymax_graph(
        sla,
        tCritic,
        tWarn,
        current_search['shinken_graph_scale'],
        current_search['shinken_graph_ymax'],
        current_search['shinken_graph_ymax_value']
    )
    y_min = _cal_ymin_graph(
        sla,
        tCritic,
        tWarn,
        current_search['shinken_graph_scale'],
        current_search['shinken_graph_ymin'],
        current_search['shinken_graph_ymin_value'],
        y_max
    )
    if y_max < y_min:
        y_max = 100
        y_min = _cal_ymin_graph(
            sla,
            tCritic,
            tWarn,
            current_search['shinken_graph_scale'],
            current_search['shinken_graph_ymin'],
            current_search['shinken_graph_ymin_value'],
            y_max
        )
    
    if y_max == y_min:
        y_max += 0.01
        y_min -= 0.01
    y_max = _getYValueInterval(y_max)
    y_min = _getYValueInterval(y_min)
    
    return y_min, y_max


def _max_of_not_none(tab, default):
    _to_return = None
    for e in tab:
        if e is None:
            continue
        elif _to_return is None:
            _to_return = e
        elif e > _to_return:
            _to_return = e
    if _to_return is None:
        return default
    return _to_return


def _min_of_not_none(tab, default):
    _to_return = None
    for e in tab:
        if e is None:
            continue
        elif _to_return is None:
            _to_return = e
        elif e < _to_return:
            _to_return = e
    if _to_return is None:
        return default
    return _to_return


def _cal_ymax_graph(sla, tCritic, tWarn, scale, option, value):
    _to_return = 100
    if option == GRAPHIC_Y_MAX_OPTION.VALEUR:
        _to_return = value
    elif option == GRAPHIC_Y_MAX_OPTION.MAXIMUM_SLA:
        _to_return = _max_of_not_none(sla, Y_MAX_VALUE) + value
    elif option == GRAPHIC_Y_MAX_OPTION.SEUIL_CRITIC:
        _to_return = _max_of_not_none(tCritic, Y_MAX_VALUE) + value
    elif option == GRAPHIC_Y_MAX_OPTION.SEUIL_WARNING:
        _to_return = _max_of_not_none(tWarn, Y_MAX_VALUE) + value
    else:  # AUTO
        _auto = AUTO_SETTINGS[scale][KEY_Y_MAX]
        return _cal_ymax_graph(sla, tCritic, tWarn, scale, _auto[KEY_OPTION], _auto[KEY_VALEUR])
    return _getYValueInterval(_to_return)


def _cal_ymin_graph(sla, tCritic, tWarn, scale, option, value, ymax):
    _to_return = 0
    if option == GRAPHIC_Y_MIN_OPTION.VALEUR:
        _to_return = value
    elif option == GRAPHIC_Y_MIN_OPTION.MINIMUM_SLA:
        _to_return = _min_of_not_none(sla, Y_MIN_VALUE) - value
    elif option == GRAPHIC_Y_MIN_OPTION.SEUIL_CRITIC:
        _to_return = _min_of_not_none(tCritic, Y_MIN_VALUE) - value
    elif option == GRAPHIC_Y_MIN_OPTION.SEUIL_CRITIC_CENTER:
        _to_return = ymax - (ymax - _min_of_not_none(tCritic, Y_MIN_VALUE)) * 2
    elif option == GRAPHIC_Y_MIN_OPTION.SEUIL_WARNING:
        _to_return = _min_of_not_none(tWarn, Y_MIN_VALUE) - value
    elif option == GRAPHIC_Y_MIN_OPTION.SEUIL_WARNING_CENTER:
        _to_return = ymax - (ymax - _min_of_not_none(tWarn, Y_MIN_VALUE)) * 2
    else:  # AUTO
        _auto = AUTO_SETTINGS[scale][KEY_Y_MIN]
        return _cal_ymin_graph(sla, tCritic, tWarn, scale, _auto[KEY_OPTION], _auto[KEY_VALEUR], ymax)
    return _getYValueInterval(_to_return)


def _getYValueInterval(value):
    return max(min(value, Y_MAX_VALUE), Y_MIN_VALUE)


def _restore_form_state(current_search):
    restore_url_string = ''
    for key in current_search.keys():
        restore_url_string += '&%s=%s' % (key, current_search[key])
    return restore_url_string


def _build_reporting_data(current_search, user):
    search_type = current_search['type']
    search_filter = current_search['filter'].decode('utf-8')
    date_begin = current_search['dateBegin']
    date_end = current_search['dateEnd']
    warning_threshold = current_search['warning_threshold'].replace(',', '.')
    critical_threshold = current_search['critical_threshold'].replace(',', '.')
    
    sla_module = get_sla_module()
    
    # try to convert form string to datetime or redirect error
    try:
        period_begin = datetime.strptime(date_begin, '%Y-%m-%d')
        period_end = datetime.strptime(date_end, '%Y-%m-%d')
    except ValueError:
        raise ReportException(app._('reporting.invalid_period_format'))
    
    # if period_begin is greater than period_end, it is a bad range.
    max_date = datetime.today() + timedelta(days=0)
    min_date = sla_module.get_first_monitoring_start_time()
    if period_begin.date() < min_date.date():
        raise ReportException(app._('reporting.outrange_period_begin') % (app.helper.print_date_locate(min_date), app.helper.print_date_locate(max_date)))
    if period_end.date() > max_date.date():
        raise ReportException(app._('reporting.outrange_period') % (app.helper.print_date_locate(min_date), app.helper.print_date_locate(max_date)))
    
    if period_begin > period_end:
        raise ReportException(app._('reporting.invalid_period_end_before_start'))
    
    default_sla_thresholds = sla_module.get_default_sla_thresholds()
    if warning_threshold:
        if "%" in str(warning_threshold):
            raise ReportException(app._('reporting.invalide_character') % (app._('reporting.warning_threshold'), '%'))
        else:
            try:
                warning_threshold = float(warning_threshold)
            except ValueError:
                raise ReportException(app._('reporting.invalid_warning_threshold'))
            
            if warning_threshold > 100 or warning_threshold < 0:
                raise ReportException(app._('reporting.invalid_warning_threshold'))
    else:
        warning_threshold = default_sla_thresholds[0]
    
    if critical_threshold:
        if "%" in str(critical_threshold):
            raise ReportException(app._('reporting.invalide_character') % (app._('reporting.critical_threshold'), '%'))
        else:
            try:
                critical_threshold = float(critical_threshold)
            except ValueError:
                raise ReportException(app._('reporting.invalid_critical_threshold'))
            
            if critical_threshold > 100 or critical_threshold < 0:
                raise ReportException(app._('reporting.invalid_critical_threshold'))
    else:
        critical_threshold = default_sla_thresholds[1]
    
    if warning_threshold < critical_threshold:
        raise ReportException(app._('reporting.invalid_threshold_range'))
    
    # convert datetime to timestamp
    timestamp_begin = int(time.mktime(period_begin.timetuple()))
    timestamp_end = int(time.mktime(period_end.timetuple()))
    
    if search_type not in FILTER_TYPE.ALL:
        raise ReportException(app._('reporting.invalid_type'))
    
    if search_filter == '':
        raise ReportException(app._('reporting.missing_filter'))
    
    items = []
    _by_item_infos = []
    _rapport_result = {'sla': {'thresholds': (warning_threshold, critical_threshold), }, 'history': {}}
    _rapport_infos = {
        'reportname'     : 'Unknown',
        'timestamp_begin': timestamp_begin,
        'timestamp_end'  : timestamp_end,
        'type'           : search_type,
        'by_item'        : _by_item_infos,
        'result'         : _rapport_result,
    }
    
    # Look at the filter type and get some items from it
    if search_type == FILTER_TYPE.NORMAL_HOSTGROUP:
        selected_hostgroup = next((hostgroup for hostgroup in app.datamgr.get_hostgroups() if hostgroup.hostgroup_name == search_filter), None)
        if selected_hostgroup is None:
            err = app._('reporting.no_such_group') % search_filter
            raise ReportException(err)
        items = selected_hostgroup.members
        _rapport_infos['reportname'] = '%s %s' % (app._('reporting.by_hostgroup'), search_filter)
    
    if search_type == FILTER_TYPE.NORMAL_HOST_TEMPLATE:
        items = [host for host in app.datamgr.get_hosts() if search_filter in host.tags]
        if not items:
            err = app._('reporting.no_such_host_template') % search_filter
            raise ReportException(err)
        _rapport_infos['reportname'] = '%s %s' % (app._('reporting.by_hosttemplate'), search_filter)
    
    if search_type == FILTER_TYPE.NORMAL_HOST:
        search_filter = search_filter.lower()
        items = [h for h in app.datamgr.get_hosts() if search_filter in h.host_name.lower() and not h.got_business_rule]
        if not items:
            err = app._('reporting.no_such_host') % search_filter
            raise ReportException(err)
        _rapport_infos['reportname'] = '%s %s' % (app._('reporting.by_hostname'), search_filter)
    
    if search_type == FILTER_TYPE.NORMAL_CLUSTER:
        search_filter = search_filter.lower()
        items = [h for h in app.datamgr.get_hosts() if search_filter in h.host_name.lower() and h.got_business_rule]
        if not items:
            err = app._('reporting.no_such_cluster') % search_filter
            raise ReportException(err)
        _rapport_infos['reportname'] = '%s %s' % (app._('reporting.by_cluster'), search_filter)
    
    if search_type == FILTER_TYPE.NORMAL_HOST_CHECK:
        if '/' not in search_filter:
            error_string = 'Wrong service name [%s]' % search_filter
            raise ReportException(error_string)
        
        search_filter_splitted = search_filter.split('/', 1)
        service = app.datamgr.get_service(search_filter_splitted[0], search_filter_splitted[1])
        if service is None:
            err = app._('reporting.no_such_check') % search_filter
            err = err.encode('utf8', 'ignore')
            raise ReportException(err)
        items = [service]
        _rapport_infos['reportname'] = '%s %s' % (app._('reporting.by_check'), search_filter)
    
    if search_type == FILTER_TYPE.NORMAL_CHECK:
        all_services = app.datamgr.get_services()
        items = [s for s in all_services if search_filter.lower() in s.service_description.lower()]
        if not items:
            err = app._('reporting.no_such_check') % search_filter
            raise ReportException(err)
        _rapport_infos['reportname'] = '%s %s' % (app._('reporting.by_check'), search_filter)
    
    days = []
    days.append(timestamp_begin)
    while timestamp_begin < timestamp_end:
        end_date = datetime.fromtimestamp(timestamp_begin) + timedelta(days=1)
        timestamp_begin = time.mktime(end_date.timetuple())
        days.append(timestamp_begin)
    
    at_least_one_authorized_host = False
    for item in items:
        if not is_authorized(item, user, app):
            continue
        
        at_least_one_authorized_host = True
        item_uuid = item.get_instance_uuid()
        item_sla_thresholds = sla_module.get_sla_thresholds(item_uuid)
        if item.__class__.my_type == 'host':
            host_name = item.get_name()
            service_description = ''
            _by_item_info = {'hname': host_name, 'type': 'host', 'item': item}
        else:
            host_name = item.host.get_name()
            service_description = item.service_description
            _by_item_info = {'hname': host_name, 'sdesc': service_description, 'type': 'service', 'item': item}
        
        _item_result = {'sla': {'thresholds': item_sla_thresholds}, 'history': {}}
        _by_item_info['result'] = _item_result
        _by_item_infos.append(_by_item_info)
        
        _by_day_infos = []
        _by_item_info['by_day'] = _by_day_infos
        
        item_monitoring_start_time = sla_module.get_monitoring_start_time(item_uuid=item.get_instance_uuid())
        logger.debug('[reporting] initial item [%s-%s] monitoring start at [%s]' % (host_name, service_description, item_monitoring_start_time))
        # TODO use from_to_date_generator
        for day in days:
            _day_result = {'sla': {}, 'history': {}}
            _by_day_info = {'day': day, 'result': _day_result}
            _by_day_infos.append(_by_day_info)
    
            local_time = time.localtime(day)
            date = Date(local_time.tm_yday, local_time.tm_year)
            try:
                entry = sla_module.read_archive(item_uuid, date)
            except ShinkenMongoException:
                raise ReportException(app._('reporting.mongo_unreachable'))
    
            _day_result['sla']['thresholds'] = entry.get('thresholds', item_sla_thresholds)
            # For future and past entry we count them and skip compute SLA.
            if entry.get('missing', False) and (entry.get('in_future', False) or entry.get('in_past', False)):
        
                if entry.get('in_future', False):
                    _day_result['in_future'] = True
                    _day_result['missing'] = True
                    _item_result['in_future'] = _item_result.get('in_future', 0) + 1
                    _rapport_result['in_future'] = _rapport_result.get('in_future', 0) + 1
                
                if entry.get('in_past', False):
                    _day_result['in_past'] = True
                    _day_result['missing'] = True
                    _item_result['in_past'] = _item_result.get('in_past', 0) + 1
                    _rapport_result['in_past'] = _rapport_result.get('in_past', 0) + 1
            else:
                for entry_name, entry_value in entry.iteritems():
                    if entry_name.startswith('history_'):
                        state_name = entry_name[8:]
                        _day_result['history'][state_name] = entry_value
                        _day_result['history'][KEY_TIME % state_name] = entry_value
                        entry_total = float(entry.get('history_total', 0))
                        if entry_total:
                            _item_result['history']['nb_entry'] = _item_result['history'].get('nb_entry', 0) + 1
                            _rapport_result['history']['nb_entry'] = _rapport_result['history'].get('nb_entry', 0) + 1
                            if 'history_format' != entry_name:
                                _item_result['history'][state_name] = _item_result['history'].get(state_name, 0) + entry_value / entry_total
                                _item_result['history'][KEY_TIME % state_name] = _item_result['history'].get(KEY_TIME % state_name, 0) + entry_value
                                _rapport_result['history'][state_name] = _rapport_result['history'].get(state_name, 0) + entry_value / entry_total
                                _rapport_result['history'][KEY_TIME % state_name] = _rapport_result['history'].get(KEY_TIME % state_name, 0) + entry_value
                    
                    if entry_name.startswith('sla_'):
                        state_name = entry_name[4:]
                        _day_result['sla'][state_name] = entry_value
                        _day_result['sla'][KEY_TIME % state_name] = entry_value
                        entry_total = float(entry.get('sla_total', 0))
                        if entry_total:
                            _item_result['sla']['nb_entry'] = _item_result['sla'].get('nb_entry', 0) + 1
                            _rapport_result['sla']['nb_entry'] = _rapport_result['sla'].get('nb_entry', 0) + 1
                            if 'sla_format' != entry_name:
                                _item_result['sla'][state_name] = _item_result['sla'].get(state_name, 0) + entry_value / entry_total
                                _item_result['sla'][KEY_TIME % state_name] = _item_result['sla'].get(KEY_TIME % state_name, 0) + entry_value
                                _rapport_result['sla'][state_name] = _rapport_result['sla'].get(state_name, 0) + entry_value / entry_total
                                _rapport_result['sla'][KEY_TIME % state_name] = _rapport_result['sla'].get(KEY_TIME % state_name, 0) + entry_value
            logger.debug('[reporting] _day_result %s : [%s]' % (date, _day_result))
            _sla_to_percent(_day_result)
        logger.debug('[reporting] _item_result [%s]' % _item_result)
        _sla_to_percent(_item_result)
        logger.debug('[reporting] _item_result after _sla_to_percent [%s]' % _item_result)
    if not at_least_one_authorized_host:
        err = app._('unauthorized.403_text')
        raise ReportException(err)
    
    _sla_to_percent(_rapport_result)
    logger.debug('[reporting] search [%s]:[%s] from:[%s] to:[%s]' % (search_type, search_filter, date_begin, date_end))
    return _rapport_infos


def _sla_to_percent(dict_to_change):
    if dict_to_change.get('missing'):
        return
    
    dict_to_change.pop('in_future', None)
    dict_to_change.pop('in_past', None)
    for key in ('history', 'sla'):
        sub_dict_to_change = dict_to_change.get(key, {})
        _max = 'unset'
        _total = sub_dict_to_change.get('total', 0)
        if not _total:
            continue
        for _state in ALL_STATES:
            _value = sub_dict_to_change.get(_state, 0)
            if _value and _total:
                sub_dict_to_change[_state] = round(_value / float(_total), 4)
                # sub_dict_to_change[KEY_TIME % _state] = _value
                if sub_dict_to_change.get(_max, 0) < sub_dict_to_change[_state]:
                    _max = _state
        for value_to_del in ('total', 'format'):
            sub_dict_to_change.pop(value_to_del, None)
        if _max != 'unset':
            sub_dict_to_change[_max] = 1 - sum([sub_dict_to_change.get(st, 0) for st in ALL_STATES if st != _max])


def _percent_format(_value, _format):
    if _format == 'label':
        return (_LABEL_PATERN % (100 * _value))
    elif _format == 'threshold':
        return (_THRESHOLD_PATERN % _value)
    else:
        return (_DEFAULT_PATERN % _value)


def _parse_form_to_dict(app, user=None):
    if user is None:
        user = app.get_user_auth()
    
    shinken_graph_scale = app.request.GET.get(FORM_FIELD.SCALE, GRAPHIC_SCALE.AUTO)
    if shinken_graph_scale not in GRAPHIC_SCALE.ALL:
        shinken_graph_scale = GRAPHIC_SCALE.AUTO
    
    shinken_graph_ymax = app.request.GET.get(FORM_FIELD.YMAX, GRAPHIC_Y_MAX_OPTION.AUTO)
    if shinken_graph_ymax not in GRAPHIC_Y_MAX_OPTION.ALL:
        shinken_graph_ymax = GRAPHIC_Y_MAX_OPTION.AUTO
    
    shinken_graph_ymin = app.request.GET.get(FORM_FIELD.YMIN, GRAPHIC_Y_MIN_OPTION.AUTO)
    if shinken_graph_ymin not in GRAPHIC_Y_MIN_OPTION.ALL:
        shinken_graph_ymin = GRAPHIC_Y_MIN_OPTION.AUTO
    
    shinken_display_total_historic = app.request.GET.get('shinken_display_total_historic', 'off')
    shinken_display_total_sla = app.request.GET.get('shinken_display_total_sla', 'off')
    shinken_daily_sla_graph = app.request.GET.get('shinken_daily_sla_graph', 'off')
    if shinken_daily_sla_graph == 'on':
        shinken_display_total_sla = 'on'
    
    # rebuild 'data_type' for compatibility of old form
    # data_type = app.request.GET.get('data_type', MODE_HISTORY_SLA)
    data_type = user.acl_in_tab_history
    if data_type == MODE_HISTORY_SLA:
        data_type = MODE_SLA
    
    if shinken_display_total_historic == 'on' and (shinken_display_total_sla == 'on' or shinken_daily_sla_graph == 'on'):
        data_type = MODE_HISTORY_SLA
    elif shinken_display_total_historic == 'on' and not (shinken_display_total_sla == 'on' or shinken_daily_sla_graph == 'on'):
        data_type = MODE_HISTORY
    elif shinken_display_total_historic == 'off' and (shinken_display_total_sla == 'on' or shinken_daily_sla_graph == 'on'):
        data_type = MODE_SLA
    
    return {
        'dateBegin'                     : app.request.GET.get('dateBegin', ''),
        'dateEnd'                       : app.request.GET.get('dateEnd', ''),
        'type'                          : app.request.GET.get('type', ''),
        'filter'                        : app.request.GET.get('filter', '').strip(),
        'data_type'                     : data_type,
        'warning_threshold'             : app.request.GET.get('warning_threshold', ''),
        'critical_threshold'            : app.request.GET.get('critical_threshold', ''),
        'display_element_threshold'     : app.request.GET.get('display_element_threshold', 'off'),
        'display_report_threshold'      : app.request.GET.get('display_report_threshold', 'off'),
        'shinken_add_daily_details'     : app.request.GET.get('shinken_add_daily_details', 'off'),
        'shinken_display_total_historic': shinken_display_total_historic,
        'shinken_display_total_sla'     : shinken_display_total_sla,
        'shinken_daily_sla_graph'       : shinken_daily_sla_graph,
        'shinken_graph_scale'           : shinken_graph_scale,
        'shinken_graph_ymax'            : shinken_graph_ymax,
        'shinken_graph_ymax_value'      : safe_cast(
            app.request.GET.get(FORM_FIELD.YMAX_VALUE, str(Y_MAX_VALUE)),
            float,
            Y_MAX_VALUE
        ),
        'shinken_graph_ymin'            : shinken_graph_ymin,
        'shinken_graph_ymin_value'      : safe_cast(
            app.request.GET.get(FORM_FIELD.YMIN_VALUE, str(Y_MIN_VALUE)),
            float,
            Y_MIN_VALUE
        ),
        'from_web'                      : app.request.GET.get('from_web', False),
    }


def generate_reporting():
    current_search = _parse_form_to_dict(app)
    current_search['from_web'] = True
    
    get = _restore_form_state(current_search)
    get = get.replace('&', '?', 1)
    if 'web' in app.request.GET:
        return app.bottle.redirect('/reporting-web%s' % get)
    elif 'csv' in app.request.GET:
        return app.bottle.redirect('/reporting-csv%s' % get)
    else:
        logger.error('[generate_reporting] option unknown')


# dummy function used to be able to declare the sla widget
def get_sla_widget():
    pass


def get_sla_widget_full():
    pass


pages = {
    get_history        : {'routes': ['/inner/history/:uuid'], 'static': True, 'wrappers': ['auth', 'json']},
    get_history_ranges : {'routes': ['/inner/history/ranges/:uuid'], 'static': True, 'wrappers': ['auth', 'json']},
    get_sla_widget_data: {'routes': ['/api/widget/sla/:uuid'], 'static': True, 'wrappers': ['auth']},
    get_sla_widget     : {
        'routes'             : ['/api/widget/sla'],
        'static'             : True,
        'widget'             : ['dashboard'],
        'widget_desc'        : 'Show one element SLA',
        'widget_name'        : 'sla',
        'widget_size'        : {'width': 2, 'height': 3},
        'widget_options'     : [{'name': 'name', 'value': '', 'type': 'hst_OR_srv', 'label': 'Name', 'required': True}],
        'widget_display_name': 'Sla',
        'widget_favoritable' : True,
        'old_style'          : False
    },
    get_sla_widget_full: {
        'routes'             : ['/api/widget/sla_full'],
        'static'             : True,
        'widget'             : ['dashboard'],
        'widget_desc'        : 'Show one element SLA Full',
        'widget_name'        : 'sla_full',
        'widget_size'        : {'width': 5, 'height': 3},
        'widget_options'     : [{'name': 'name', 'value': '', 'type': 'hst_OR_srv', 'label': 'Name', 'required': True}],
        'widget_display_name': 'Sla',
        'widget_favoritable' : True,
        'old_style'          : False
    },
    get_reporting_form : {'routes': ['/reporting'], 'view': 'reporting-form', 'static': True, 'wrappers': ['auth']},
    generate_reporting : {'routes': ['/generate-reporting'], 'wrappers': ['auth']},
    reporting_web      : {'routes': ['/reporting-web'], 'view': 'reporting-web', 'wrappers': ['auth']},
    reporting_csv      : {'routes': ['/reporting-csv'], 'wrappers': []}
}
