#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2009-2022:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import json
import re
import time

from shinken.external_command import ExternalCommand, ExternalCommandManager
from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional
    from webui.module import WebuiBroker

app = None  # type: Optional[WebuiBroker]


# Function handling $NOW$ macro
def subsNOW():
    return str(int(time.time()))


def subsSLASH():
    return '/'


# This dictionary associate macros with expansion function
subs = {
    '$NOW$'  : subsNOW,
    '$SLASH$': subsSLASH,
    # Add new macros here
}


# Expand macro in a string. It returns the string with macros defined in subs dictionary expanded
def expand_macros(cmd=None):
    macros = re.findall(r'(\$\w+\$)', cmd)
    cmd_expanded = cmd
    for macro in macros:
        subfunc = subs.get(macro)
        if subfunc is None:
            logger.debug("Macro [%s] is unknown, do nothing" % macro)
            continue
        logger.debug("Expand macro [%s] in [%s]" % (macro, cmd_expanded))
        cmd_expanded = cmd_expanded.replace(macro, subfunc())
    
    return cmd_expanded


# Our page
def do_action():
    # First we look for the user sid
    # so we bail out if it's a false one
    user = app.get_user_auth()
    data = json.loads(app.request.body.getvalue())
    
    # Maybe the user is not known at all
    if not user:
        return app.abort(401, app._('do_action.user_no_auth'), True)
    
    if getattr(app.myconf, 'demo', '0') == '1':
        return app.abort(401, app._('do_action.no_action_demo'))
    
    # Or he is not allowed to launch commands?
    if app.manage_acl and not user.can_submit_commands:
        return app.abort(403, app._('do_action.user_cant_do_action'), True)
    
    configuration_id_in_client = data.get('configuration_id', None)
    configuration_id = app.datamgr.get_configuration_id()
    # if angular client doesn't give configuration_id, doesn't check the configuration_id, should be changed when the client will have the configuration_id anytimes
    if configuration_id_in_client is not None and configuration_id_in_client != configuration_id:
        return app.abort(410, app._('do_action.configuration_outdated'), True)
    
    now = int(time.time())
    cmd_name = data['type_action']
    cmd_name = cmd_name.decode('utf8', 'replace')
    element_uuid = data['element_id']
    
    if cmd_name in ['DISABLE_SVC_FLAP_DETECTION', 'ENABLE_SVC_FLAP_DETECTION', 'PROCESS_SVC_CHECK_RESULT']:
        cmd_name = cmd_name.replace('SVC', 'SERVICE')
    
    # Check if the command exist in the external command list
    if cmd_name not in ExternalCommandManager.commands:
        return app.abort(404, app._('do_action.unknown_cmd') % cmd_name)
    
    if not user.acl_make_acknowledge and cmd_name in [
        'ACKNOWLEDGE_SVC_PROBLEM',
        'ACKNOWLEDGE_HOST_PROBLEM',
        'ACKNOWLEDGE_SVC_PROBLEM_EXPIRE',
        'ACKNOWLEDGE_HOST_PROBLEM_EXPIRE',
        'REMOVE_HOST_ACKNOWLEDGEMENT',
        'REMOVE_SVC_ACKNOWLEDGEMENT'
    ]:
        return app.abort(403, app._('do_action.no_right_ack'), True)
    
    if not user.acl_make_downtime and cmd_name in [
        'SCHEDULE_AND_PROPAGATE_HOST_DOWNTIME',
        'SCHEDULE_AND_PROPAGATE_TRIGGERED_HOST_DOWNTIME',
        'SCHEDULE_CONTACT_DOWNTIME',
        'SCHEDULE_HOSTGROUP_HOST_DOWNTIME',
        'SCHEDULE_HOSTGROUP_SVC_DOWNTIME',
        'SCHEDULE_HOST_SVC_DOWNTIME',
        'SCHEDULE_SERVICEGROUP_HOST_DOWNTIME',
        'SCHEDULE_SERVICEGROUP_SVC_DOWNTIME',
        'SCHEDULE_HOST_DOWNTIME',
        'SCHEDULE_SVC_DOWNTIME'
    ]:
        return app.abort(403, app._('do_action.no_right_downtime'), True)
    
    if not user.acl_make_downtime and cmd_name in [
        'DEL_HOST_DOWNTIME',
        'DEL_SVC_DOWNTIME',
        'DEL_ALL_HOST_DOWNTIMES',
        'DEL_ALL_SVC_DOWNTIMES',
    ]:
        return app.abort(403, app._('do_action.no_right_remove_downtime'), True)
    
    if not user.acl_force_result_check and cmd_name in ['PROCESS_HOST_CHECK_RESULT', 'PROCESS_SERVICE_CHECK_RESULT']:
        return app.abort(403, app._('do_action.no_right_force_res'), True)
    
    if not user.acl_force_retry_check and cmd_name in ['SCHEDULE_FORCED_HOST_CHECK', 'SCHEDULE_FORCED_SVC_CHECK']:
        return app.abort(403, app._('do_action.no_right_retry_check'), True)
    
    extcmd_list = []
    extcmd_list.append(cmd_name)
    type_first_param = ExternalCommandManager.commands[cmd_name]['args'][0]
    if type_first_param == 'service':
        names = app.get_name_from_uuid(element_uuid)
        if not names:
            return app.abort(404, app._('do_action.unknown_check') % element_uuid)
        extcmd_list.append(names[0].replace(';', r'\;'))
        extcmd_list.append(names[1].replace(';', r'\;'))
    elif type_first_param == 'host':
        names = app.get_name_from_uuid(element_uuid)
        if not names:
            return app.abort(404, app._('do_action.unknown_host') % element_uuid)
        extcmd_list.append(names[0])
    elif type_first_param == 'to_int':
        extcmd_list.append(element_uuid)
    
    if data.get('args', None):
        extcmd_list.extend(data['args'])
    
    extcmd = '[%d] %s' % (now, ';'.join(extcmd_list))
    logger.debug("do_action Got the form %s" % extcmd)
    
    # Expand macros
    extcmd = expand_macros(extcmd)
    logger.debug("do_action Got after macro expansion : %s" % extcmd)
    
    # Ok, if good, we can launch the command
    e = ExternalCommand(extcmd)
    app.push_external_command(e)
    
    return ""


pages = {
    do_action: {'routes': ['/action'], 'method': 'POST', 'wrappers': ['auth', 'json']},
}
