#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import traceback

import socket
import threading
import time

from shinken.http_client import HTTPExceptions
from shinken.log import logger, LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.property import IntegerProp, StringProp
from shinken.satellitelink import SatelliteLink, SatelliteLinks
from shinkensolutions.localinstall import get_shinken_current_version_and_patch
from .configuration_incarnation import ConfigurationIncarnation
from .util import to_mb_size

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Any, Optional
    from shinken.objects.config import Config
    from configuration_incarnation import PartConfigurationIncarnation

""" TODO: Add some comment about this class for the doc"""

raw_logger = LoggerFactory.get_logger()
logger_configuration = raw_logger.get_sub_part('CONFIGURATION')
logger_lastest_monitoring_configuration = logger_configuration.get_sub_part('LATEST-MONITORING-CONFIGURATION')


class ArbiterLink(SatelliteLink):
    id = 0  # type: int
    my_type = u'arbiter'  # type: unicode
    properties = SatelliteLink.properties.copy()  # type: Dict[unicode, Any]
    properties.update({
        'arbiter_name'         : StringProp(),
        'host_name'            : StringProp(default=""),
        'port'                 : IntegerProp(default='7770'),
        'managed_configuration': StringProp(default={}),
    })
    
    
    def get_name(self):
        # type: () -> str
        return getattr(self, 'arbiter_name', 'unknown')
    
    
    def get_config(self):
        return self.con.get('get_config')
    
    
    def _get_my_realm_name(self):
        # type: () -> unicode
        return u'(arbiter is over realms)'
    
    
    # Check is required when prop are set:
    # contacts OR contactgroups is need
    def is_correct(self):
        # type: () -> bool
        state = True
        cls = self.__class__
        
        for prop, entry in cls.properties.iteritems():
            if not hasattr(self, prop) and entry.required:
                # This should raise an error afterwards?
                # Log the issue
                logger.warning("%s arbiterlink is missing %s property" % (self.get_name(), prop))
                self.debug("%s arbiterlink is missing %s property" % (self.get_name(), prop))
                state = False  # Bad boy...
        return state
    
    
    # Look for ourself as an arbiter. If we search for a specific arbiter name, go forit
    # If not look be our fqdn name, or if not, our hostname
    def is_me(self, lookup_name):
        # type: (unicode) -> bool
        if lookup_name:
            return lookup_name == self.get_name()
        else:
            logger.debug(u'An arbiter configuration with the hostname:[%s] was found from the arbiter with hostname:[%s]' % (self.host_name, socket.getfqdn()))
            return self.host_name == socket.getfqdn() or self.host_name == socket.gethostname()
    
    
    def give_satellite_cfg(self):
        # type: () -> Dict[unicode, Any]
        return {
            u'port'               : self.port,
            u'address'            : self.address,
            u'name'               : self.arbiter_name,
            u'use_ssl'            : self.use_ssl,
            u'hard_ssl_name_check': self.hard_ssl_name_check,
        }
    
    
    def put_conf_to_arbiter_spare(self, master_configuration, arbiter_trace):
        # type: (Config, Dict) -> None
        conf_to_send = {
            u'full_conf'                  : master_configuration.whole_conf_pack,
            u'arbiter_trace'              : arbiter_trace,
            u'configuration_incarnation'  : self.configuration_incarnation,
            u'arbiter_master_full_version': get_shinken_current_version_and_patch(),
        }
        logger_lastest_monitoring_configuration.info('Sending the NEW Monitoring Configuration to the Arbiter SPARE "%s"' % self.get_name())
        before = time.time()
        is_sent = self.put_conf(conf_to_send)
        # Let assume that the arbiter spare is ow up to date
        if is_sent:
            self.managed_configuration = self.configuration_incarnation.dump_as_json()
            logger_lastest_monitoring_configuration.info('The NEW Monitoring Configuration to the Arbiter SPARE "%s" ( size=%sMB ) was sent in %.3fs' % (self.get_name(), to_mb_size(len(master_configuration.whole_conf_pack)), time.time() - before))
        else:
            logger_lastest_monitoring_configuration.error('Fail to send the NEW Monitoring Configuration %s to the Arbiter SPARE "%s". We wil retry the next turn (next second).' % (self.configuration_incarnation, self.get_name()))
    
    
    def _update_managed_configuration(self):
        # type: () -> None
        with self.big_lock:
            if self.con is None:
                self.create_connection()
            
            # If the connection failed to initialize, bail out
            if self.con is None:
                return
            
            try:
                managed_configuration = self.con.get('get_currently_managed_configuration')
                logger.debug("[%s] managed [%s]" % (self.get_name(), managed_configuration))
                
                # We can update our list now
                self.managed_configuration = managed_configuration
            except HTTPExceptions as exp:
                logger.warning("[%s][%s] Call to know what the daemon is currently managing did fail: %s" % (self._get_my_realm_name(), self.get_name(), exp))
    
    
    def get_current_managed_configuration(self):
        # type: () -> Optional[PartConfigurationIncarnation]
        if not self.managed_configuration:
            return None
        current_configuration_incarnation = ConfigurationIncarnation.create_from_json(self.managed_configuration)
        return current_configuration_incarnation
    
    
    def is_managing_the_valid_configuration(self):
        # type: () -> bool
        currently_managed_configuration_incarnation = ConfigurationIncarnation.create_from_json(self.managed_configuration)
        is_valid = self.configuration_incarnation.is_equal(currently_managed_configuration_incarnation)
        return is_valid
    
    
    def do_not_run(self):
        # type: () -> None
        current_thread = getattr(self, '_do_not_run_thread', None)
        if current_thread is None or not current_thread.is_alive():
            self._do_not_run_thread = threading.Thread(None, target=self._do_not_run_loop, name='do-not-run-%s' % self.get_name())
            self._do_not_run_thread.daemon = True
            self._do_not_run_thread.start()
    
    
    def _do_not_run_loop(self):
        # type: () -> None
        while self.spare:
            try:
                self._do_not_run()
            except Exception:  # catch all in the thread
                logger.error('The arbiter spare %s to not run thread did fail with the error: %s. Restarting it.' % (self.get_name(), traceback.format_exc()))
                return
            time.sleep(1)
    
    
    def _do_not_run(self):
        # type: () -> None
        if self.con is None:
            self.create_connection()
        try:
            self.con.get('do_not_run')
            return
        except HTTPExceptions as exp:
            err = 'Cannot ask the arbiter spare "%s" to keep being sleeping : %s' % (self.get_name(), exp)
            if self.alive:
                logger.warning(err)
            else:
                logger.info(err)
            self.con = None
            return
    
    
    def get_satellite_list(self, daemon_type):
        if self.con is None:
            self.create_connection()
        try:
            r = self.con.get_satellite_list(daemon_type)
            return r
        except HTTPExceptions:
            self.con = None
            return []
    
    
    def get_satellite_status(self, daemon_type, name):
        if self.con is None:
            self.create_connection()
        try:
            r = self.con.get_satellite_status(daemon_type, name)
            return r
        except HTTPExceptions:
            self.con = None
            return {}
    
    
    def get_all_states(self):
        if self.con is None:
            self.create_connection()
        try:
            r = self.con.get('get_all_states')
            return r
        except HTTPExceptions:
            self.con = None
            return None
    
    
    def get_objects_properties(self, table, properties=[]):
        if self.con is None:
            self.create_connection()
        try:
            r = self.con.get('get_objects_properties', {'table': table, 'properties': properties})
            return r
        except HTTPExceptions:
            self.con = None
            return None


class ArbiterLinks(SatelliteLinks):
    name_property = "name"
    inner_class = ArbiterLink
    
    
    # We must have a realm property, so we find our realm
    def linkify(self, modules):
        self.linkify_s_by_plug(modules)
