#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

import os
import time
import traceback

from shinken.log import PART_INITIALISATION, LoggerFactory
from shinken.message import Message
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.modules.base_module.basemodule import BaseModule
from shinken.property import BOOLEAN_STATES
from shinken.util import to_bool
from shinkensolutions.lib_modules.configuration_reader_mixin import ConfigurationReaderMixin, SeparatorFormat, ConfigurationFormat, TypeConfiguration
from shinkensolutions.locking.fair_lock_ordonnancer import FairLockGroupOrdonnancer

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, List, Callable, Any
    from shinken.log import PartLogger

BROKS_MANAGEMENT_EARLY_LOCK = False
BROKS_MANAGEMENT_ENABLE_CATCHUP = True
BROKS_MANAGEMENT_ALLOWED_LATE_SETS = 10
BROKS_MANAGEMENT_MAX_BROKS_IN_CATCHUP = 200000
BROKS_MANAGEMENT_CATCHUP_LOOPS = True

BROKS_GETTER__INCLUDE_DESERIALISATION_AND_CATCHUP_IN_LOCK = u'%s__broks_getter__include_deserialisation_and_catchup_in_lock'
BROKS_GETTER__ACTIVATE_LATE_SET_CATCHUP = u'%s__broks_getter__activate_late_set_catchup'
BROKS_GETTER__NB_LATE_SET_ALLOWED_BEFORE_CATCHUP = u'%s__broks_getter__nb_late_set_allowed_before_catchup'
BROKS_GETTER__CATCHUP_BROKS_MANAGED_BY_MODULE_IN_A_CATCHUP_LOOP = u'%s__broks_getter__catchup_broks_managed_by_module_in_a_catchup_loop'
BROKS_GETTER__CATCHUP_RUN_ENDLESS_UNTIL_NB_LATE_SET_ALLOWED_REACHED = u'%s__broks_getter__catchup_run_endless_until_nb_late_set_allowed_reached'


def read_int_in_configuration(conf, key_name, default_value, conf_error_stats=None, logger=None):
    # type: (object, str, int, Optional[dict], PartLogger) -> int
    
    val = getattr(conf, key_name, default_value)
    try:
        val = int(val)
    except:
        if logger:
            logger.error(u'%s has incorrect value[%s], resetting to default value [%s]' % (key_name, val, default_value))
        if conf_error_stats is not None:
            conf_error_stats.update({key_name: {u'conf_value': val, u'default_value': default_value}})
        val = default_value
    
    return val


def read_bool_in_configuration(conf, key_name, default_value, conf_error_stats=None, logger=None):
    # type: (object, str, bool, Optional[dict], PartLogger) -> bool
    
    val = getattr(conf, key_name, default_value)
    if isinstance(val, basestring):
        if val in BOOLEAN_STATES.keys():
            val = to_bool(val)
        else:
            if logger:
                logger.error(u'%s has incorrect value [%s] possible values are : [%s], resetting to default value [%s]' % (key_name, val, u','.join(BOOLEAN_STATES.iterkeys()), default_value))
            if conf_error_stats is not None:
                conf_error_stats.update({key_name: {u'conf_value': val, u'default_value': default_value}})
            val = default_value
    
    return val


class ContextClass(object):
    def __init__(self, acquire, release):
        # type: (Callable[None, None], Callable[None, None]) -> None
        self.acquire = acquire
        self.release = release
    
    
    def __enter__(self):
        self.acquire()
    
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        self.release()


class BrokerBaseModule(BaseModule, ConfigurationReaderMixin):
    def __init__(self, module_configuration):
        BaseModule.__init__(self, module_configuration)
        
        self._fair_lock_ordonnancer = None  # type: Optional[FairLockGroupOrdonnancer]
        self.consumer_lock = None  # type: Optional[ContextClass]
        self.producer_lock = None  # type: Optional[ContextClass]
        self._manage_broks_functions = []  # type: List[Callable[[Any], None]]
        self._writer_early_lock = BROKS_MANAGEMENT_EARLY_LOCK
        self._writer_catchup_enabled = BROKS_MANAGEMENT_ENABLE_CATCHUP
        self._writer_late_sets_allowed = BROKS_MANAGEMENT_ALLOWED_LATE_SETS
        self._writer_late_broks_max = BROKS_MANAGEMENT_MAX_BROKS_IN_CATCHUP
        self._writer_catchup_loops = BROKS_MANAGEMENT_CATCHUP_LOOPS
        
        if hasattr(self, u'module_type'):
            module_type = self.module_type
        else:
            module_type = getattr(self.myconf, u'module_type', None)
        
        log_init = self.logger.get_sub_part(PART_INITIALISATION, len(PART_INITIALISATION))
        
        if not module_type:
            log_init.warning(u'unknown module type, broks management will run with default values')
            return
        
        if module_type == u'broker_module_livedata':
            module_type_cfg_format = u'broker__module_livedata'
        else:
            module_type_cfg_format = u'broker__module_%s' % module_type
        
        configuration_format = [
            SeparatorFormat(u'BROKS GETTER'),
            ConfigurationFormat(BROKS_GETTER__INCLUDE_DESERIALISATION_AND_CATCHUP_IN_LOCK % module_type_cfg_format, BROKS_MANAGEMENT_EARLY_LOCK, TypeConfiguration.BOOL, u'_writer_early_lock'),
            ConfigurationFormat(BROKS_GETTER__ACTIVATE_LATE_SET_CATCHUP % module_type_cfg_format, BROKS_MANAGEMENT_ENABLE_CATCHUP, TypeConfiguration.BOOL, u'_writer_catchup_enabled'),
            ConfigurationFormat(BROKS_GETTER__NB_LATE_SET_ALLOWED_BEFORE_CATCHUP % module_type_cfg_format, BROKS_MANAGEMENT_ALLOWED_LATE_SETS, TypeConfiguration.INT, u'_writer_late_sets_allowed'),
            ConfigurationFormat(BROKS_GETTER__CATCHUP_BROKS_MANAGED_BY_MODULE_IN_A_CATCHUP_LOOP % module_type_cfg_format, BROKS_MANAGEMENT_MAX_BROKS_IN_CATCHUP, TypeConfiguration.INT, u'_writer_late_broks_max'),
            ConfigurationFormat(BROKS_GETTER__CATCHUP_RUN_ENDLESS_UNTIL_NB_LATE_SET_ALLOWED_REACHED % module_type_cfg_format, BROKS_MANAGEMENT_CATCHUP_LOOPS, TypeConfiguration.BOOL, u'_writer_catchup_loops'),
        ]
        ConfigurationReaderMixin.__init__(self, configuration_format, self.myconf, log_init)
        self.read_configuration()
        self.log_configuration(log_properties=True, show_values_as_in_conf_file=True)
    
    
    def lock_init(
            self,
            broks_eater,
            fair_lock_ordonnancer_name=u'ITEMS ACCESS ORDONNANCER',
            consumer_name=u'HTTP requests',
            producer_name=u'Broks management',
            consumer_max_switch_time=1,
            producer_max_switch_time=1,
            error_log_time=30,
            logger=None
    ):
        # type: (List[Callable[[Any], None]], unicode, unicode, unicode, int, int, int, Optional[PartLogger]) -> None
        
        self._manage_broks_functions = broks_eater
        
        if not logger:
            logger = self.logger
        
        self._fair_lock_ordonnancer = FairLockGroupOrdonnancer(
            name=fair_lock_ordonnancer_name,
            consumer_name=consumer_name,
            consumer_max_wish_switch_time=consumer_max_switch_time,
            producer_name=producer_name,
            producer_max_wish_switch_time=producer_max_switch_time,
            logger=logger,
            error_log_time=error_log_time
        )
        
        self.consumer_lock = ContextClass(acquire=self._fair_lock_ordonnancer.consumer_acquire, release=self._fair_lock_ordonnancer.consumer_release)
        self.producer_lock = ContextClass(acquire=self._fair_lock_ordonnancer.producer_acquire, release=self._fair_lock_ordonnancer.producer_release)
    
    
    def do_loop_turn(self):
        raise NotImplementedError()
    
    
    def _die_with_strong_error(self, error, with_trace_back=True):
        crash_logger = LoggerFactory.get_logger('CRASH - INSIDE MODULE PROCESS')
        crash_logger.error(error)
        traceback_message = traceback.format_exc() if with_trace_back else error
        msg = Message(id=0, type='ICrash', data={'name': self.get_name(), 'exception': error, 'trace': traceback_message})
        self.from_module_to_main_daemon_queue.put(msg)
        # wait 2 sec so we know that the broker got our message, and die
        time.sleep(2)
        os._exit(2)  # noqa : forced exit
    
    
    # The manage brok thread will ocnsume broks, but if it die, broks will stack in the queue manager
    # so this is not acceptable, and such error means a strong die with log in healthcheck
    def manage_brok_thread(self):
        try:
            self._manage_brok_thread()
        except Exception as exp:  # noqa: catch all to die
            self._die_with_strong_error(str(exp))
    
    
    def _manage_brok_thread(self):
        have_lock = False
        logger_perf = self.logger.get_sub_part(u'MANAGE BROKS', len(u'MANAGE BROKS')).get_sub_part(u'PERF', len(u'PERF'))
        logger_late_broks_sets = self.logger.get_sub_part(u'MANAGE BROKS', len(u'MANAGE BROKS')).get_sub_part(u'LATE BROKS SETS', len(u'LATE BROKS SETS'))
        while True:
            start = time.time()
            broks = self.to_q.get()
            
            if self._writer_early_lock and not have_lock:
                self._fair_lock_ordonnancer.producer_acquire()
                have_lock = True
            
            get_late_broks_set_time = time.time()
            nb_late_broks_sets_taken = 0
            if self._writer_catchup_enabled and (self.to_q.qsize() > self._writer_late_sets_allowed >= 0):
                t1 = time.time()
                while self.to_q.qsize() > 0:
                    t0 = time.time()
                    _late_broks_set = self.to_q.get()
                    _current_nb_late_broks = len(_late_broks_set)
                    broks.extend(_late_broks_set)
                    broks_to_process = len(broks)
                    logger_late_broks_sets.info(u'Getting brok set with %s broks in %.3fs [time for read queue size=%.3fs]. Total broks to process= %s/max:%s. Broks sets in queue: %s.' %
                                                (_current_nb_late_broks, time.time() - t0, t0 - t1, broks_to_process, self._writer_late_broks_max, self.to_q.qsize()))
                    nb_late_broks_sets_taken += 1
                    t1 = time.time()
                    
                    if broks_to_process > self._writer_late_broks_max:
                        logger_late_broks_sets.info(u'Late brok taken => limit reach : %s / limit: %s.' % (broks_to_process, self._writer_late_broks_max))
                        break
            
            after_get = time.time()
            # First unpickle broks, but outside the lock time
            for brok in broks:
                brok.prepare()
            after_prepare = time.time()
            
            # For updating, we cannot do it while answering queries, so wait for no readers
            if not self._writer_early_lock and not have_lock:
                self._fair_lock_ordonnancer.producer_acquire()
                have_lock = True
            
            after_wait_write_lock = time.time()
            broks_type_counter = {}
            broks_type_times = {}
            for brok in broks:
                try:
                    t0 = time.time()
                    for manage_broks_function in self._manage_broks_functions:
                        manage_broks_function(brok)
                    broks_type_counter[brok.type] = broks_type_counter.get(brok.type, 0) + 1
                    broks_type_times[brok.type] = broks_type_times.get(brok.type, 0.0) + (time.time() - t0)
                except Exception as exp:
                    self._die_with_strong_error(str(exp))  # no hope, cannot survive this
            
            end = time.time()
            logger_perf.info(u'[ %4d broks ] [ wait and get first set on queue=%.3fs ] [ get %s late sets on=%.3fs ] [ deserialization=%.3fs ] [ wait write lock=%.3fs ] [ manage broks=%.3fs ] [ total=%.3fs ]' % (
                len(broks), get_late_broks_set_time - start, nb_late_broks_sets_taken, after_get - get_late_broks_set_time, after_prepare - after_get, after_wait_write_lock - after_prepare, end - after_wait_write_lock, end - start))
            
            message = []
            for b_type, count in broks_type_counter.iteritems():
                message.append(u'[%s=%s]' % (b_type, count))
            logger_perf.info(u'                 => handled broks -> count by types : %s' % ' '.join(message))
            
            message = []
            for b_type, count in broks_type_times.iteritems():
                message.append(u'[%s=%0.3f]' % (b_type, count))
            logger_perf.info(u'                 => handled broks -> time by types : %s' % ' '.join(message))
            
            try:
                for _detail_timer_name, _detail_timer in self.rg._detail_timer.iteritems():
                    logger_detail_timer = logger_perf.get_sub_part(_detail_timer_name, register=False)
                    for sub_part_name, sub_part_time in _detail_timer.iteritems():
                        logger_detail_timer.debug(u'sub_part : time for %s [ %0.3f ]' % (sub_part_name, sub_part_time))
            except:
                # the broker module cannot have any regenerator
                pass
            
            # if we are late we don't release the lock and we continue to process broks
            if self._writer_catchup_enabled and self._writer_catchup_loops:
                qsize = self.to_q.qsize()
                if qsize > self._writer_late_sets_allowed >= 0:
                    logger_perf.info(u'Broks sets in queue after manage broks is %s. We keep the lock and continue the brok managing.' % qsize)
                    continue
            
            if have_lock:
                # We can release the lock as a writer
                self._fair_lock_ordonnancer.producer_release()
                have_lock = False
