#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2009-2012:
#    Gabes Jean, naparuba@gmail.com
#    Gerhard Lausser, Gerhard.Lausser@consol.de
#    Gregory Starck, g.starck@gmail.com
#    Hartmut Goebel, h.goebel@goebel-consult.de
#
# This file is part of Shinken.
#
# Shinken is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Shinken is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Shinken.  If not, see <http://www.gnu.org/licenses/>.

""" This class is a base class for nearly all configuration
 elements like service, hosts or contacts.
"""

import urllib
from collections import deque
from copy import copy

from shinken.brok import Brok
from shinken.commandcall import CommandCall, UNSET_POLLER_REACTIONNER_TAG_VALUE
from shinken.complexexpression import ComplexExpressionFactory
from shinken.graph import Graph
from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.property import StringProp, ListProp, IntegerProp, BoolProp
from shinken.rawdata import RawData, TYPE_RAW_DATA_CHECK_RESULT, TYPE_RAW_DATA_FULL
from shinken.util import strip_and_uniq, get_context_id, to_cluster_members_uuid_list_and_do_cache_result, get_service_realm

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, Union, List

MAX_INHERITANCE_LEVEL = 32


def _is_valid_inherited_value(data):
    return data not in ('null', '', None)


def _format_inherited_plus_value(list_as_string):
    if not list_as_string:
        return list_as_string
    elements = list_as_string.split(',')
    value = strip_and_uniq(elements)
    return ','.join(filter(_is_valid_inherited_value, value))


class PERIOD_PROPERTIES(object):
    # TODO GREG SEF-6462
    DEFAULT_VALUE = '24x7'
    HOST_NOTIFICATION_PERIOD = 'host_notification_period'
    SERVICE_NOTIFICATION_PERIOD = 'service_notification_period'
    NOTIFICATION_PERIOD = 'notification_period'
    CHECK_PERIOD = 'check_period'
    MODULATION_PERIOD = 'modulation_period'
    ESCALATION_PERIOD = 'escalation_period'
    MAINTENANCE_PERIOD = 'maintenance_period'
    
    
    @staticmethod
    def is_period(property_name):
        return property_name in (PERIOD_PROPERTIES.HOST_NOTIFICATION_PERIOD,
                                 PERIOD_PROPERTIES.SERVICE_NOTIFICATION_PERIOD,
                                 PERIOD_PROPERTIES.NOTIFICATION_PERIOD,
                                 PERIOD_PROPERTIES.CHECK_PERIOD,
                                 PERIOD_PROPERTIES.MODULATION_PERIOD,
                                 PERIOD_PROPERTIES.ESCALATION_PERIOD,
                                 PERIOD_PROPERTIES.MAINTENANCE_PERIOD)


class Item(object):
    properties = {
        'imported_from'   : StringProp(default='unknown'),
        'definition_order': IntegerProp(default='100'),
        'uuid'            : StringProp(default='', fill_brok=['full_status', 'check_result', 'next_schedule'], fill_raw_data=[TYPE_RAW_DATA_CHECK_RESULT, TYPE_RAW_DATA_FULL]),
        # TODO: find why we can't uncomment this line below.
        # 'register':                 BoolProp(default='1'),
    }
    
    running_properties = {
        # All errors and warning raised during the configuration parsing
        # and that will raised real warning/errors during the is_correct
        'configuration_warnings': ListProp(default=[]),
        'configuration_errors'  : ListProp(default=[]),
        'hash'                  : StringProp(default=''),
        # We save all template we asked us to load from
        'tags'                  : ListProp(default=[], fill_brok=['full_status']),
    }
    
    macros = {
    }
    # Some macros can be static on configuration, so we will be able to have a cache about them for
    # command resolution
    static_macros = set()
    # All passthrough cannot be "unset"  by the element page
    passthrough = {
        'alias'              : StringProp(default=''),
        'presence_protection': StringProp(default=''),
        'sources'            : StringProp(default=''),
        'import_date'        : StringProp(default='')
    }
    
    not_inherited_passthrough = {
        'pack'    : StringProp(default=''),
        'register': BoolProp(default=''),
        'editable': BoolProp(default=''),
        'enabled' : BoolProp(default=''),
    }
    
    
    def __init__(self, params=None, skip_useless_in_configuration=False):
        if params is None:
            params = {}
        # We have our own id of My Class type :)
        # use set attr for going into the slots
        # instead of __dict__ :)
        cls = self.__class__
        self.id = cls.id
        cls.id += 1
        self._is_correct__is_error_collected = False
        self.customs = {}  # for custom variables
        
        if skip_useless_in_configuration:
            self.init_running_properties_but_not_useless_in_configuration()
        else:
            self.init_running_properties()
        
        # [0] = _  -> new custom entry in UPPER case
        for key in params:
            # delistify attributes if there is only one value
            params[key] = self._compact_unique_attr_value(params[key])
            
            # checks for attribute value special syntax ( _ )
            if key.startswith("_"):
                if isinstance(params[key], list):
                    err = "no support for _ syntax in multiple valued attributes"
                    self.configuration_errors.append(err)
                    continue
                custom_name = key.upper()
                self.customs[custom_name] = params[key]
            else:
                setattr(self, key, params[key])
    
    
    # When values to set on attributes are unique (single element list),
    # return the value directly rather than setting list element.
    def _compact_unique_attr_value(self, val):
        if not isinstance(val, list):
            return val
        # ok not a simple value
        if len(val) > 1:
            return val
        elif len(val) == 0:
            return ''
        else:
            return val[0]
    
    
    def init_running_properties_but_not_useless_in_configuration(self):
        cls = self.__class__
        not_useless_in_configuration_properties = getattr(cls, '_not_useless_in_configuration_properties', None)
        if not_useless_in_configuration_properties is None:
            not_useless_in_configuration_properties = {}
            for prop, entry in self.__class__.running_properties.iteritems():
                if entry.useless_in_configuration:
                    continue
                not_useless_in_configuration_properties[prop] = entry
            # print "Build cache of not useless: %s (%d/%d)" % (not_useless_in_configuration_properties.keys(), len(not_useless_in_configuration_properties), len(cls.running_properties))
            setattr(cls, '_not_useless_in_configuration_properties', not_useless_in_configuration_properties)
        for prop, entry in not_useless_in_configuration_properties.iteritems():
            # Copy is slow, so we check type
            # Type with __iter__ are list or dict, or tuple.
            # Item need it's own list, so we copy
            val = entry.default
            if hasattr(val, '__iter__'):
                setattr(self, prop, copy(val))
            else:
                setattr(self, prop, val)
                # each instance to have his own running prop!
    
    
    def init_running_properties(self):
        for prop, entry in self.__class__.running_properties.iteritems():
            # Copy is slow, so we check type
            # Type with __iter__ are list or dict, or tuple.
            # Item need it's own list, so we copy
            val = entry.default
            if hasattr(val, '__iter__'):
                setattr(self, prop, copy(val))
            else:
                setattr(self, prop, val)
                # each instance to have his own running prop!
    
    
    def refil_running_properties(self):
        for prop, entry in self.__class__.running_properties.iteritems():
            if hasattr(self, prop):
                continue
            # Copy is slow, so we check type
            # Type with __iter__ are list or dict, or tuple.
            # Item need it's own list, so we copy
            val = entry.default
            if hasattr(val, '__iter__'):
                setattr(self, prop, copy(val))
            else:
                setattr(self, prop, val)
                # each instance to have his own running prop!
    
    
    def copy(self, skip_useless_in_configuration=False):
        """ Return a copy of the item, but give him a new id """
        cls = self.__class__
        i = cls({}, skip_useless_in_configuration=skip_useless_in_configuration)  # Dummy item but with it's own running properties
        for prop in cls.properties:
            if hasattr(self, prop):
                val = getattr(self, prop)
                setattr(i, prop, val)
        # Also copy the customs tab
        i.customs = copy(self.customs)
        return i
    
    
    def clean(self):
        """ Clean useless things not requested once item has been fully initialized&configured.
Like temporary attributes such as "imported_from", etc.. """
        for name in ('imported_from', 'use', 'plus', 'templates',):
            if hasattr(self, name):
                delattr(self, name)
    
    
    # Compute a cache for properties that should be taken for a fill_brok so we don't have to loop over properties
    # each time we are creating a brok
    def __get_class_properties_for_brok_type(self, brok_type):
        cls = self.__class__
        if not hasattr(cls, 'cache_properties_for_brok_type'):
            setattr(cls, 'cache_properties_for_brok_type', {})
        
        # if in cache, give it
        if brok_type in cls.cache_properties_for_brok_type:
            return cls.cache_properties_for_brok_type[brok_type]
        # else compute it
        cls.cache_properties_for_brok_type[brok_type] = []
        for prop, entry in cls.properties.iteritems():
            # Is this property intended for broking?
            if brok_type in entry.fill_brok:
                cls.cache_properties_for_brok_type[brok_type].append(prop)
        return cls.cache_properties_for_brok_type[brok_type]
    
    
    def __get_class_running_properties_for_brok_type(self, brok_type):
        cls = self.__class__
        if not hasattr(cls, 'cache_running_properties_for_brok_type'):
            setattr(cls, 'cache_running_properties_for_brok_type', {})
        
        # if in cache, give it
        if brok_type in cls.cache_running_properties_for_brok_type:
            return cls.cache_running_properties_for_brok_type[brok_type]
        # else compute it
        cls.cache_running_properties_for_brok_type[brok_type] = []
        for prop, entry in cls.running_properties.iteritems():
            # Is this property intended for broking?
            if brok_type in entry.fill_brok:
                cls.cache_running_properties_for_brok_type[brok_type].append(prop)
        return cls.cache_running_properties_for_brok_type[brok_type]
    
    
    def __str__(self):
        return str(self.__dict__) + '\n'
    
    
    # If a prop is absent and is not required, put the default value
    # NOTE: we do not use the fill_default_one_property because the function call cost can be
    # too must for just a 3 lines copy/paste
    def fill_default(self):
        """ Fill missing properties if they are missing """
        cls = self.__class__
        
        for prop, entry in cls.properties.iteritems():
            if not hasattr(self, prop) and entry.has_default:
                setattr(self, prop, entry.default)
    
    
    # If a prop is absent and is not required, put the default value
    def fill_default_one_property(self, prop):
        """ Fill missing properties if they are missing """
        cls = self.__class__
        
        entry = cls.properties[prop]
        if not hasattr(self, prop) and entry.has_default:
            setattr(self, prop, entry.default)
    
    
    # We load every useful parameter so no need to access global conf later
    # Must be called after a change in a global conf parameter
    def load_global_conf(cls, conf):
        """ Used to put global values in the sub Class like
        hosts or services """
        # conf have properties, if 'enable_notifications':
        # { [...] 'class_inherit': [(Host, None), (Service, None),
        #  (Contact, None)]}
        # get the name and put the value if None, put the Name
        # (not None) if not (not clear?)
        for prop, entry in conf.properties.iteritems():
            # If we have a class_inherit, and the arbiter really send us it
            # if 'class_inherit' in entry and hasattr(conf, prop):
            if hasattr(conf, prop):
                for (cls_dest, change_name) in entry.class_inherit:
                    if cls_dest == cls:  # ok, we've got something to get
                        value = getattr(conf, prop)
                        if change_name is None:
                            setattr(cls, prop, value)
                        else:
                            setattr(cls, change_name, value)
    
    
    # Make this method a classmethod
    load_global_conf = classmethod(load_global_conf)
    
    
    # Use to make python properties
    def pythonize(self):
        cls = self.__class__
        for prop, tab in cls.properties.iteritems():
            if not hasattr(self, prop):  # Will be catch at the is_correct moment
                continue
            raw_value = getattr(self, prop)
            try:
                new_val = tab.pythonize(raw_value)
                setattr(self, prop, new_val)
            except AttributeError:
                pass  # Will be catch at the is_correct moment
            except KeyError as exp:
                # print "Missing prop value", exp
                err = "The property '%s' of '%s' do not have value" % (prop, self.get_name())
                self.configuration_errors.append(err)
            except ValueError as exp:
                err = "Incorrect value for property '%s' of '%s' : %s" % (prop, self.get_name(), raw_value)
                self.configuration_errors.append(err)
    
    
    # Check is required prop are set:
    # template are always correct
    def is_correct(self):
        state = True
        properties = self.__class__.properties
        
        # Raised all previously saw errors like unknown contacts and co
        if len(self.configuration_errors) != 0:
            state = False
            for err in self.configuration_errors:
                logger.error("[item::%s] %s" % (self.get_name(), err))
        
        for prop, entry in properties.iteritems():
            if not hasattr(self, prop) and entry.required:
                logger.error("[item::%s] %s property is missing" % (self.get_name(), prop))
                state = False
        
        self._is_correct__is_error_collected = True
        return state
    
    
    def is_correct__append_in_configuration_errors(self, to_append):
        if self._is_correct__is_error_collected:
            return
        self.configuration_errors.append(to_append)
    
    
    # This function is used by service and hosts
    # to transform Nagios2 parameters to Nagios3
    # ones, like normal_check_interval to
    # check_interval. There is a old_parameters tab
    # in Classes that give such modifications to do.
    def old_properties_names_to_new(self):
        old_properties = self.__class__.old_properties
        for old_name, new_name in old_properties.iteritems():
            # Ok, if we got old_name and NO new name, we switch the name
            if hasattr(self, old_name) and not hasattr(self, new_name):
                value = getattr(self, old_name)
                setattr(self, new_name, value)
    
    
    # The arbiter is asking us our raw value before all explode or linking
    def get_raw_import_values(self):
        r = {}
        properties = self.__class__.properties.keys()
        # Register is not by default in the properties
        if 'register' not in properties:
            properties.append('register')
        
        for prop in properties:
            if hasattr(self, prop):
                v = getattr(self, prop)
                r[prop] = v
        return r
    
    
    def add_downtime(self, downtime):
        self.downtimes.append(downtime)
    
    
    def del_downtime(self, downtime_id):
        d_to_del = None
        for dt in self.downtimes:
            if dt.id == downtime_id:
                d_to_del = dt
                dt.can_be_deleted = True
        if d_to_del is not None:
            self.downtimes.remove(d_to_del)
    
    
    # Will flatten some parameters tagged by the 'conf_send_preparation'
    # property because they are too "linked" to be send like that (like realms)
    def prepare_for_conf_sending(self):
        cls = self.__class__
        
        for prop, entry in cls.properties.iteritems():
            # Is this property need preparation for sending?
            if entry.conf_send_preparation is not None:
                f = entry.conf_send_preparation
                if f is not None:
                    val = f(getattr(self, prop))
                    setattr(self, prop, val)
        
        if hasattr(cls, 'running_properties'):
            for prop, entry in cls.running_properties.iteritems():
                # Is this property need preparation for sending?
                if entry.conf_send_preparation is not None:
                    f = entry.conf_send_preparation
                    if f is not None:
                        val = f(getattr(self, prop))
                        setattr(self, prop, val)
    
    
    # Get the property for an object, with good value
    # and brok_transformation if need
    def get_property_value_for_brok(self, prop, tab):
        entry = tab[prop]
        # Get the current value, or the default if need
        value = getattr(self, prop, entry.default)
        
        # Apply brok_transformation if need
        # Look if we must preprocess the value first
        pre_op = entry.brok_transformation
        if pre_op is not None:
            value = pre_op(self, value)
        
        return value
    
    
    def build_full_status(self, context_change_time=None):
        # type: (Optional[Union[int,float]]) -> None
        pass
    
    
    # Fill data with info of item by looking at brok_type
    # in props of properties or running_properties
    def fill_data_brok_from(self, data, brok_type):
        cls = self.__class__
        
        self.build_full_status()
        
        # Now config properties
        for prop in self.__get_class_properties_for_brok_type(brok_type):
            entry = cls.properties[prop]  # Get the current value, or the default if need
            value = getattr(self, prop, entry.default)
            
            # Apply brok_transformation if need
            # Look if we must preprocess the value first
            pre_op = entry.brok_transformation
            if pre_op is not None:
                value = pre_op(self, value)
            data[prop] = value
        # Maybe the class do not have running_properties
        if hasattr(cls, 'running_properties'):
            # We've got prop in running_properties too
            for prop in self.__get_class_running_properties_for_brok_type(brok_type):
                entry = cls.running_properties[prop]
                # Get the current value, or the default if need
                value = getattr(self, prop, entry.default)
                
                # Apply brok_transformation if need look if we must preprocess the value first
                pre_op = entry.brok_transformation
                if pre_op is not None:
                    value = pre_op(self, value)
                data[prop] = value
    
    
    # Get a brok with initial status
    def get_initial_status_brok(self):
        cls = self.__class__
        my_type = cls.my_type
        data = {'id': self.id}
        
        self.fill_data_brok_from(data, 'full_status')
        b = Brok('initial_' + my_type + '_status', data)
        return b
    
    
    # Get a brok with update item status
    def get_update_status_brok(self):
        cls = self.__class__
        my_type = cls.my_type
        
        data = {'id': self.id}
        self.fill_data_brok_from(data, 'full_status')
        b = Brok('update_' + my_type + '_status', data)
        return b
    
    
    # Get a brok with check_result
    def get_check_result_brok(self):
        cls = self.__class__
        my_type = cls.my_type
        
        self.build_full_status()
        
        data = {
            'uuid'                         : self.uuid,
            'instance_uuid'                : self.get_instance_uuid(),
            'host_uuid'                    : self.host_uuid,
            'host_name'                    : self.host_name,
            'check_interval'               : self.check_interval,
            'acknowledgement'              : self.acknowledgement,
            'acknowledge_id'               : self.get_acknowkledge_id(),
            'partial_acknowledge'          : self.partial_acknowledge,
            'percent_state_change'         : self.percent_state_change,
            'last_state_id'                : self.last_state_id,
            'state_type'                   : self.state_type,
            'is_flapping'                  : self.is_flapping,
            'active_downtime_uuids'        : self.get_active_downtime_uuids(),
            'in_partial_downtime'          : self.in_partial_downtime,
            'is_partial_flapping'          : self.is_partial_flapping,
            'is_partial_acknowledged'      : self.is_partial_acknowledged,
            'in_scheduled_downtime'        : self.is_in_downtime(),
            'in_inherited_downtime'        : self.is_in_inherited_downtime(),
            'state_type_id'                : self.state_type_id,
            'last_state_change'            : self.last_state_change,
            'last_state_as_string'         : self.last_state_as_string,
            'last_hard_state_change'       : self.last_hard_state_change,
            'long_output'                  : self.long_output,
            'output'                       : self.output,
            'perf_data'                    : self.perf_data,
            'check_type'                   : self.check_type,
            'problem_has_been_acknowledged': self.is_acknowledged(),
            'in_inherited_acknowledged'    : self.is_in_inherited_acknowledged(),
            'last_state_type'              : self.last_state_type,
            'state'                        : self.state,
            'last_chk'                     : self.last_chk,
            'next_chk'                     : self.next_chk,
            'state_id'                     : self.state_id,
            'realm'                        : self.realm,
            'attempt'                      : self.attempt,
            'got_business_rule'            : self.got_business_rule,
            'state_validity_period'        : self.state_validity_period,
            'state_validity_end_time'      : self.state_validity_end_time,
            'missing_data_activation_time' : self.missing_data_activation_time,
            'full_status_change_time'      : self.full_status_change_time,
            'current_full_status'          : self.current_full_status,
        }
        if my_type == 'service':
            data['service_description'] = self.service_description
            data['realm'] = get_service_realm(self, None)
        elif self.got_business_rule:  # a cluster
            data['bp_state'] = self.bp_state
        b = Brok(my_type + '_check_result', data)
        
        return b
    
    
    # Get brok about the new schedule (next_check)
    def get_next_schedule_brok(self):
        cls = self.__class__
        my_type = cls.my_type
        
        data = {}
        self.fill_data_brok_from(data, 'next_schedule')
        b = Brok(my_type + '_next_schedule', data)
        return b
    
    
    # Link one command property to a class (for globals like oc*p_command)
    def linkify_one_command_with_commands(self, commands, prop):
        if hasattr(self, prop):
            command = getattr(self, prop).strip()
            if command != '':
                if hasattr(self, 'poller_tag'):
                    cmd_call = CommandCall(commands, command, poller_tag=self.poller_tag)
                elif hasattr(self, 'reactionner_tag'):
                    cmd_call = CommandCall(commands, command, reactionner_tag=self.reactionner_tag)
                else:
                    cmd_call = CommandCall(commands, command)
                setattr(self, prop, cmd_call)
            else:
                setattr(self, prop, None)
    
    
    def fill_data_of_raw_data(self, data, type_raw_data):
        cls = self.__class__
        for prop, entry in cls.properties.iteritems():
            if type_raw_data in entry.fill_raw_data:
                data[prop] = self.get_property_value_for_raw_data(prop, cls.properties)
        
        if hasattr(cls, 'running_properties'):
            for prop, entry in cls.running_properties.iteritems():
                if type_raw_data in entry.fill_raw_data:
                    data[prop] = self.get_property_value_for_raw_data(prop, cls.running_properties)
    
    
    def get_property_value_for_raw_data(self, prop, tab):
        entry = tab[prop]
        # Get the current value, or the default if need
        value = getattr(self, prop, entry.default)
        
        # Apply raw_data_transformation if need
        # Look if we must preprocess the value first
        pre_op = entry.raw_data_transformation
        if pre_op is not None:
            value = pre_op(self, value)
        
        return value
    
    
    def get_initial_status_raw_data(self):
        cls = self.__class__
        my_type = cls.my_type
        data = {'id': self.id}
        
        self.fill_data_of_raw_data(data, TYPE_RAW_DATA_FULL)
        # logger.debug("[raw_data] data [%s]"%data)
        return RawData(my_type + '_full', data)
    
    
    def get_update_status_raw_data(self):
        cls = self.__class__
        my_type = cls.my_type
        
        data = {
            'id'               : self.id,
            'uuid'             : self.uuid,
            'host_uuid'        : getattr(self, 'host_uuid', ''),
            'instance_uuid'    : self.get_instance_uuid(),
            'last_chk'         : self.last_chk,
            'state_id'         : self.state_id,
            'state_type_id'    : self.state_type_id,
            'context_id'       : get_context_id(self),
            'perf_data'        : self.perf_data,
            
            'got_business_rule': getattr(self, 'got_business_rule', False),
            'cluster_members'  : to_cluster_members_uuid_list_and_do_cache_result(self, None),
        }
        
        # logger.debug("[raw_data] get_update_status_raw_data [%s]" % data['uuid'])
        return RawData(my_type + '_check_result', data)
    
    
    def get_name(self):
        # type: ()-> str
        raise NotImplementedError()
    
    
    def get_item_type(self):
        # type: ()-> str
        return self.__class__.my_type
    
    
    def get_item_type_for_link(self):
        # type: ()-> str
        return "%ss" % self.get_item_type()
    
    
    def get_item_tpl_type_for_link(self):
        # type: ()-> str
        return "%stpls" % self.get_item_type()
    
    
    def get_uuid(self):
        # type: ()-> str
        return self.uuid or None
    
    
    def get_link_shinken(self):
        # type: ()-> str
        # commented because no link for now
        # _to_return = self.get_uuid()
        # if not _to_return:
        #     return self.get_name()
        # return '<a target="_blank" class="shinken-link" href="/elements/%s/%s">%s</a>' % (self.get_item_type_for_link(), _to_return, self.get_name())
        return self.get_name()
    
    
    def get_link_tpl_shinken(self):
        # WE CANNOT HAVE LINK WITH CLUSTER TPL BECAUSE THE "GOT BUISINESS RULES" IS NOT UP... I RETURN ONLY NAME FOR THE MOMENT
        # # type: ()-> str
        # _to_return = self.get_uuid()
        # if not _to_return:
        #     return self.get_name()
        # return '<a target="_blank" class="shinken-link" href="/elements/%s/%s">%s</a>' % (self.get_item_tpl_type_for_link(), _to_return, self.get_name())
        # return '<span class="shinken-link" style="pointer-events:none">%s</span>' % self.get_name()
        return self.get_name()


class Items(object):
    inner_class = None
    
    
    def __init__(self, items):
        self.items = {}
        self.configuration_warnings = []
        self.configuration_errors = []
        for i in items:
            self.items[i.id] = i
        self.templates = {}
        # We should keep a graph of templates relations
        self.templates_graph = Graph()
    
    
    def __iter__(self):
        return self.items.itervalues()
    
    
    def __len__(self):
        return len(self.items)
    
    
    def __delitem__(self, key):
        # we don't want it, we do not have it. All is perfect
        if key in self.items:
            del self.items[key]
    
    
    def __setitem__(self, key, value):
        self.items[key] = value
    
    
    def __getitem__(self, key):
        return self.items[key]
    
    
    def __contains__(self, key):
        return key in self.items
    
    
    # return a list of all enabled items
    def enabled(self):
        return [i for i in self if i.enabled]
    
    
    def get_full_configuration_errors(self):
        _to_return = []
        _to_return.extend(self.configuration_errors)
        for _current in self:
            _to_return.extend(_current.configuration_errors)
        return _to_return
    
    
    # We create the reversed list so search will be faster
    # We also create a twins list with id of twins (not the original
    # just the others, higher twins)
    def create_reversed_list(self):
        self.reversed_list = {}
        self.twins = []
        name_property = self.__class__.name_property
        for id in self.items:
            if hasattr(self.items[id], name_property):
                name = getattr(self.items[id], name_property)
                if name not in self.reversed_list:
                    self.reversed_list[name] = id
                else:
                    self.twins.append(id)
        self.create_reversed_list_for_id()
    
    
    # We create the reversed list so search will be faster
    # We also create a twins list with id of twins (not the original just the others, higher twins)
    def create_reversed_list_for_id(self):
        self.reversed_list_id = {}
        for id in self.items:
            if hasattr(self.items[id], "uuid"):
                uuid = getattr(self.items[id], "uuid")
                self.reversed_list_id[uuid] = id
    
    
    def reindex(self, items):
        # type: (List[Item]) -> None
        
        for item in items:
            name_property = self.__class__.name_property
            _id = item.id
            name = getattr(item, name_property)
            uuid = item.uuid
            self.reversed_list[name] = _id
            self.reversed_list_id[uuid] = _id
    
    
    def _find_id_by_name(self, name):
        if hasattr(self, 'reversed_list'):
            if name in self.reversed_list:
                return self.reversed_list[name]
            else:
                return None
        else:  # ok, an early ask, with no reversed list from now...
            name_property = self.__class__.name_property
            for i in self:
                if hasattr(i, name_property):
                    i_name = getattr(i, name_property)
                    if i_name == name:
                        return i.id
            return None
    
    
    def _find_id_by_uuid(self, uuid):
        if hasattr(self, 'reversed_list_id'):
            if uuid in self.reversed_list_id:
                return self.reversed_list_id[uuid]
            else:
                return None
        else:  # ok, an early ask, with no reversed list from now...
            for i in self:
                if hasattr(i, "uuid"):
                    i_name = getattr(i, "uuid")
                    if i_name == uuid:
                        return i.id
            return None
    
    
    def find_by_uuid(self, uuid):
        id = self._find_id_by_uuid(uuid)
        if id is not None:
            return self.items[id]
        else:
            return None
    
    
    def find_by_name(self, name):
        id = self._find_id_by_name(name)
        if id is not None:
            return self.items[id]
        else:
            return None
    
    
    def remove_by_name(self, name):
        id = self._find_id_by_name(name)
        if id is None:
            return
        # We can remove from items
        del self.items[id]
        # And also from  reversed list
        if hasattr(self, 'reversed_list'):
            if name in self.reversed_list:
                del self.reversed_list[name]
    
    
    def remove_by_uuid(self, uuid):
        id = self._find_id_by_uuid(uuid)
        if id is None:
            return
        # We can remove from items
        del self.items[id]
        # And also from  reversed list
        if hasattr(self, 'reversed_list_id'):
            if uuid in self.reversed_list_id:
                del self.reversed_list_id[uuid]
    
    
    # prepare_for_conf_sending to flatten some properties
    def prepare_for_sending(self):
        for i in self:
            i.prepare_for_conf_sending()
    
    
    # It's used to change old Nagios2 names to
    # Nagios3 ones
    def old_properties_names_to_new(self):
        for i in self:
            i.old_properties_names_to_new()
    
    
    def pythonize(self):
        for id in self.items:
            self.items[id].pythonize()
    
    
    def _show_twin_error(self, item):
        item_href = "<a target='_blank' href='/elements/%ss?filter=%s:%s'>%s</a>" % (item.__class__.my_type, self.name_property, urllib.quote(item.get_name()), item.get_name())
        logger.error("[items] %s items named %s are duplicated" % (item.__class__.my_type, item_href))
    
    
    def is_correct(self):
        # we are ok at the beginning. Hope we still ok at the end...
        config_is_correct = True
        # Some class do not have twins, because they do not have names
        # like servicedependencies
        twins = getattr(self, 'twins', None)
        if twins is not None:
            # Ok, look at no twins (it's bad!)
            for id in twins:
                # The item may have been removed, so the twin is not relevant anymore
                if id in self.items:
                    i = self.items[id]
                    self._show_twin_error(i)
                    config_is_correct = False
        
        # Then look if we have some errors in the conf
        # Juts print warnings, but raise errors
        for err in self.configuration_warnings:
            logger.warning(err)
        
        if self.configuration_errors:
            config_is_correct = False
        
        # Then look for individual ok
        for i in self:
            # Alias and display_name hook hook
            prop_name = getattr(self.__class__, 'name_property', None)
            if prop_name and not hasattr(i, 'alias') and hasattr(i, prop_name):
                setattr(i, 'alias', getattr(i, prop_name))
            if prop_name and getattr(i, 'display_name', '') == '' and hasattr(i, prop_name):
                setattr(i, 'display_name', getattr(i, prop_name))
            
            # Now other checks
            if not i.is_correct():
                logger.error("Item %s is incorrect" % (i.get_name()))
                config_is_correct = False
        
        return config_is_correct
    
    
    def clean(self):
        """ Request to remove the unnecessary attributes/others from our items """
        for i in self:
            i.clean()
        Item.clean(self)
    
    
    # If a prop is absent and is not required, put the default value
    def fill_default(self):
        for i in self:
            i.fill_default()
    
    
    def refil_running_properties(self):
        for i in self:
            i.refil_running_properties()
    
    
    def __str__(self):
        s = ''
        cls = self.__class__
        for id in self.items:
            s = s + str(cls) + ':' + str(id) + str(self.items[id]) + '\n'
        return s
    
    
    # We remove twins
    # Remember: item id respect the order of conf. So if an item is defined multiple times,
    # we want to keep the first one.
    # Services are also managed here but they are specials:
    # We remove twins services with the same host_name/service_description
    # Remember: host service are take into account first before hostgroup service
    # Id of host service are lower than hostgroups one, so they are
    # in self.twins_services
    # and we can remove them.
    def remove_twins(self):
        for id in self.twins:
            if id in self.items:
                del self[id]
        self.twins = []
    
    
    # We've got a contacts property with , separated contacts names
    # and we want have a list of Contacts
    def linkify_with_contacts(self, contacts):
        # Loop over all possible contacts ways: contacts, notification_contacts (should be void), edition_contacts, view_contacts
        possible_fields = ['contacts', 'notification_contacts', 'edition_contacts', 'view_contacts']
        for prop in possible_fields:
            for i in self:
                if hasattr(i, prop):
                    contacts_tab = getattr(i, prop)  # note: already split & strip & unique
                    new_contacts = deque()
                    for c_name in contacts_tab:
                        if c_name != '':
                            c = contacts.find_by_name(c_name)
                            if c is not None:
                                new_contacts.append(c)
                            # DEAD CODE WARNING: we did used to show a warning, but it's just too much for logs/output
                            #       so do nothing, it's up to the configuration daemon to show it
                            # else:
                            #    err = "the contact '%s' defined for '%s' is disabled or does not exist" % (c_name, i.get_name())
                            #    i.configuration_warnings.append(err)
                    # We are already sure we have unique elements
                    setattr(i, prop, tuple(new_contacts))
    
    
    # Make link between an object and its escalations
    def linkify_with_escalations(self, escalations):
        for i in self:
            if hasattr(i, 'escalations'):
                escalations_tab = i.escalations  # already split & strip & unique
                new_escalations = deque()
                for es_name in [e for e in escalations_tab if e != '']:
                    es = escalations.find_by_name(es_name)
                    if es is not None:
                        new_escalations.append(es)
                    else:  # Escalation not find, not good!
                        err = "the escalation '%s' defined for '%s' is disabled or does not exist" % (es_name, i.get_name())
                        i.configuration_errors.append(err)
                i.escalations = tuple(new_escalations)
    
    
    # Make link between item and it's resultmodulations
    def linkify_with_resultmodulations(self, resultmodulations):
        for i in self:
            if hasattr(i, 'resultmodulations'):
                resultmodulations_tab = i.resultmodulations  # note: already split & strip & unique
                new_resultmodulations = deque()
                if len(resultmodulations_tab) > 4:
                    i.configuration_errors.append("Too many result modulations applied on check.")
                for rm_name in resultmodulations_tab:
                    rm = resultmodulations.find_by_name(rm_name)
                    if rm is not None:
                        new_resultmodulations.append(rm)
                    else:
                        err = "the result modulation '%s' defined on the %s '%s' is disabled or does not exist" % (rm_name, i.__class__.my_type, i.get_name())
                        i.configuration_errors.append(err)
                        continue
                i.resultmodulations = tuple(new_resultmodulations)
    
    
    # Make link between item and it's business_impact_modulations
    def linkify_with_business_impact_modulations(self, business_impact_modulations):
        for i in self:
            if hasattr(i, 'business_impact_modulations'):
                business_impact_modulations_tab = i.business_impact_modulations  # note: already split & strip & unique
                new_business_impact_modulations = deque()
                for rm_name in business_impact_modulations_tab:
                    rm = business_impact_modulations.find_by_name(rm_name)
                    if rm is not None:
                        new_business_impact_modulations.append(rm)
                    else:
                        err = "the business impact modulation '%s' defined on the %s '%s' is disabled or does not exist" % (rm_name, i.__class__.my_type, i.get_name())
                        i.configuration_errors.append(err)
                        continue
                i.business_impact_modulations = tuple(new_business_impact_modulations)
    
    
    # If we've got a contact_groups/*_contact_groups properties, we search for all
    # theses groups and ask them their *contacts, and then add them
    # all into our contacts property
    def explode_contact_groups_into_contacts(self, contactgroups):
        # suffices: '' => just 'contacts'
        prefixes = ['', 'view_', 'notification_', 'edition_']
        for prefix in prefixes:
            cg_prop = prefix + 'contact_groups'  # contact_group property we will explode into
            c_prop = prefix + 'contacts'  # simple contact property
            for i in self:
                # not such property: not a crime, skip it
                if not hasattr(i, cg_prop):
                    continue
                v = getattr(i, cg_prop)
                if isinstance(v, list):
                    cgnames = v
                else:
                    cgnames = v.split(',')
                cgnames = strip_and_uniq(cgnames)
                for cgname in cgnames:
                    cg = contactgroups.find_by_name(cgname)
                    if cg is None:
                        err = "The contact group '%s' defined on the %s '%s' property %s is disabled or does not exist" % (cgname, i.__class__.my_type, i.get_name(), cg_prop)
                        i.configuration_errors.append(err)
                        continue
                    cnames = contactgroups.get_members_by_name(cgname)
                    # We add contacts into our *contacts
                    if cnames != []:
                        if hasattr(i, c_prop):
                            # already exist, just append the value to the string
                            setattr(i, c_prop, getattr(i, c_prop) + ',' + cnames)
                        else:
                            setattr(i, c_prop, cnames)
    
    
    # Link a timeperiod property (prop)
    def linkify_with_timeperiods(self, timeperiods, prop):
        for i in self:
            tpname = getattr(i, prop, '')  # already strip()
            if tpname:
                # Ok, got a real name, search for it
                tp = timeperiods.find_by_name(tpname)
                # If not found, it's an error
                if tp is None:
                    err = "The %s of the %s '%s' named '%s' is disabled or does not exist!" % (prop, i.__class__.my_type, i.get_name(), tpname)
                    i.configuration_errors.append(err)
                    continue
                # Got a real one, just set it :)
                setattr(i, prop, tp)
            else:
                # None is also a valid timeperiod, means always
                setattr(i, prop, None)
    
    
    def change_default_values_into_finals(self):
        for item in self:
            item.change_default_values_into_finals()
    
    
    def create_commandcall(self, item, commands, command_with_arg):
        comandcall = dict(commands=commands, call=command_with_arg)
        
        if hasattr(item, 'poller_tag'):
            comandcall['poller_tag'] = item.poller_tag
        elif hasattr(item, 'reactionner_tag'):
            comandcall['reactionner_tag'] = item.reactionner_tag
        else:  # maybe this type of object do not have any tag (like notification ways), so we will force to take command one
            comandcall['poller_tag'] = UNSET_POLLER_REACTIONNER_TAG_VALUE
            comandcall['reactionner_tag'] = UNSET_POLLER_REACTIONNER_TAG_VALUE
        return CommandCall(**comandcall)
    
    
    # Link one command property
    def linkify_one_command_with_commands(self, commands, prop):
        for item in self:
            if hasattr(item, prop):
                command_with_arg = getattr(item, prop).strip()
                cmd_call = self.create_commandcall(item, commands, command_with_arg) if command_with_arg != '' else None
                # if cmd_call:
                #    r_tag = cmd_call.poller_tag if cmd_call else 'NOT CREATED'
                #    logger.info('COMMAND CALL: %s::%s (%s) ==> %s (on command=%s)' % (item.my_type, item.get_full_name(), item.poller_tag, r_tag, command_with_arg))
                setattr(item, prop, cmd_call)
    
    
    # Link a command list (commands with , between) in real CommandCalls
    def linkify_command_list_with_commands(self, commands, prop):
        for item in self:
            if hasattr(item, prop):
                coms = getattr(item, prop).split(',')
                coms = strip_and_uniq(coms)
                com_list = [self.create_commandcall(item, commands, command_with_arg) for command_with_arg in coms]
                setattr(item, prop, com_list)
    
    
    # We've got a notificationways property with , separated contacts names
    # and we want have a list of NotificationWay
    def linkify_with_checkmodulations(self, checkmodulations):
        for i in self:
            if not hasattr(i, 'checkmodulations'):
                continue
            new_checkmodulations = []
            for cw_name in i.checkmodulations:
                cw = checkmodulations.find_by_name(cw_name)
                if cw is not None:
                    new_checkmodulations.append(cw)
                else:
                    err = "The checkmodulations of the %s '%s' named '%s' is disabled or does not exist!" % (i.__class__.my_type, i.get_name(), cw_name)
                    i.configuration_errors.append(err)
            # Get the list, but first make elements uniq
            i.checkmodulations = tuple(new_checkmodulations)
    
    
    # We've got list of macro modulations as list of names, and
    # we want real objects
    def linkify_with_macromodulations(self, macromodulations):
        for i in self:
            if not hasattr(i, 'macromodulations'):
                continue
            new_macromodulations = []
            for cw_name in i.macromodulations:
                cw = macromodulations.find_by_name(cw_name)
                if cw is not None:
                    new_macromodulations.append(cw)
                else:
                    err = "The macromodulations of the %s '%s' named '%s' is disabled or does not exist!" % (i.__class__.my_type, i.get_name(), cw_name)
                    i.configuration_errors.append(err)
            # Get the list, but first make elements uniq
            i.macromodulations = tuple(new_macromodulations)
    
    
    # Linkify with modules
    def linkify_s_by_plug(self, modules):
        for s in self:
            new_modules = []
            for plug_name in s.modules:
                plug_name = plug_name.strip()
                # don't tread void names
                if plug_name == '':
                    continue
                
                plug = modules.find_by_name(plug_name)
                if plug is not None:
                    new_modules.append(plug)
                else:
                    err = "Error: the module %s is unknown for %s" % (plug_name, s.get_name())
                    s.configuration_errors.append(err)
            s.modules = new_modules  # NOTE: do NOT set as tuple as some daemons need to modify it
    
    
    def evaluate_hostgroup_expression(self, expr, hosts, hostgroups, look_in='hostgroups'):
        # Maybe exp is a list, like numerous hostgroups entries in a service, link them
        if isinstance(expr, list):
            expr = '|'.join(expr)
        if look_in == 'hostgroups':
            f = ComplexExpressionFactory(look_in, hostgroups, hosts)
        else:  # templates
            f = ComplexExpressionFactory(look_in, hosts, hosts)
        expr_tree = f.eval_cor_pattern(expr)
        set_res = expr_tree.resolve_elements()
        return tuple(set_res)
    
    
    # If we've got a hostgroup_name property, we search for all
    # theses groups and ask them their hosts, and then add them
    # all into our host_name property
    def explode_host_groups_into_hosts(self, hosts, hostgroups):
        for i in self:
            hnames_list = []
            if hasattr(i, 'hostgroup_name'):
                hnames_list.extend(self.evaluate_hostgroup_expression(i.hostgroup_name, hosts, hostgroups))
            
            # Maybe there is no host in the groups, and do not have any
            # host_name too, so tag is as template to do not look at
            if hnames_list == [] and not hasattr(i, 'host_name'):
                i.register = '0'
            
            if hasattr(i, 'host_name'):
                hst = i.host_name.split(',')
                for h in hst:
                    h = h.strip()
                    # If the host start with a !, it's to be removed from
                    # the hostgroup get list
                    if h.startswith('!'):
                        hst_to_remove = h[1:].strip()
                        try:
                            hnames_list.remove(hst_to_remove)
                        # was not in it
                        except ValueError:
                            pass
                    # Else it's an host to add, but maybe it's ALL
                    elif h == '*':
                        for newhost in set(h.host_name for h in hosts.items.values() if getattr(h, 'host_name', '') != '' and not h.is_tpl()):
                            hnames_list.append(newhost)
                            # print "DBG in item.explode_host_groups_into_hosts , added '%s' to group '%s'" % (newhost, i)
                    else:
                        hnames_list.append(h)
            i.host_name = ','.join(list(set(hnames_list)))
            
            # Ok, even with all of it, there is still no host, put it as a template
            if i.host_name == '':
                i.register = '0'


class InheritableItem(Item):
    properties = Item.properties.copy()
    properties.update({
        'use' : ListProp(default='', merging='ordered'),
        'name': StringProp(default='', inherit=False),
    })
    
    
    def __init__(self, params=None, skip_useless_in_configuration=False):
        if params is None:
            params = {}
        cls = self.__class__
        self.id = cls.id
        cls.id += 1
        
        self.customs = {}  # For custom variables
        self.plus = {}  # for value with a +
        
        if skip_useless_in_configuration:
            self.init_running_properties_but_not_useless_in_configuration()
        else:
            self.init_running_properties()
        
        # [0] = +  -> new key-plus
        # [0] = _  -> new custom entry in UPPER case
        for key in params:
            # delistify attributes if there is only one value
            params[key] = self._compact_unique_attr_value(params[key])
            
            # checks for attribute value special syntax (+ or _)=
            handle_additive_inheritance = getattr(cls.properties.get(key, object()), 'handle_additive_inheritance', False)
            if handle_additive_inheritance and not isinstance(params[key], list) and len(params[key]) >= 1 and params[key][0] == '+':
                # Special case: a _MACRO can be a plus.
                # so add to plus but upper the key for the macro name
                if key[0] == "_":
                    self.plus[key.upper()] = params[key][1:]  # we remove the +
                else:
                    self.plus[key] = params[key][1:]  # we remove the +
            elif key[0] == "_":
                if isinstance(params[key], list):
                    err = "no support for _ syntax in multiple valued attributes"
                    self.configuration_errors.append(err)
                    continue
                custom_name = key.upper()
                self.customs[custom_name] = params[key]
            else:
                setattr(self, key, params[key])
    
    
    def is_tpl(self):
        """ Return if the elements is a template """
        return getattr(self, 'register', '1') == '0'
    
    
    def get_templates(self):
        if hasattr(self, 'use') and self.use != '':
            if isinstance(self.use, list):
                return self.use
            else:
                return self.use.split(',')
        else:
            return []
    
    
    # We fillfull properties with template ones if need
    # WARNING: we must beware of the template loop, and not lost time in this step
    def get_property_by_inheritance(self, items, prop_name, level=0, already_seen_templates=None, in_template_only_loop=True, allow_higher_template_lookup=True):
        # Protect against too much recursive lookup
        if level >= MAX_INHERITANCE_LEVEL:
            return None
        
        if already_seen_templates is None:
            already_seen_templates = set()
        
        handle_additive_inheritance = getattr(self.properties.get(prop_name, object()), 'handle_additive_inheritance', False)
        
        # If I have the prop, I take mine but I check if I must add a plus property
        # For check_command: bp_rule! should be null, but others values are inherited
        if hasattr(self, prop_name) and (prop_name != 'check_command' or getattr(self, prop_name) != 'bp_rule!'):
            value = getattr(self, prop_name)
            return value
        
        # Ok, I do not have prop, Maybe my templates do?
        # Same story for plus
        if allow_higher_template_lookup:
            # If we are in a templates loop: allow all
            if in_template_only_loop:
                next_level_allow_higher_template_lookup = True
            else:  # standard elements: allow only if we are the level1 element, not a template so
                next_level_allow_higher_template_lookup = False
            for i in self.templates:
                # If we have a template loop, we should not lost time here
                if i in already_seen_templates:
                    continue
                already_seen_templates.add(i)
                value = i.get_property_by_inheritance(items, prop_name, level=level + 1, already_seen_templates=already_seen_templates, in_template_only_loop=in_template_only_loop, allow_higher_template_lookup=next_level_allow_higher_template_lookup)
                
                if value is not None:
                    # If our template give us a '+' value, we should continue to loop
                    still_loop = False
                    if handle_additive_inheritance and not isinstance(value, list) and value.startswith('+'):
                        # Templates should keep their + inherited from their parents
                        if not self.is_tpl():
                            value = value[1:]
                        still_loop = True
                    
                    # Maybe in the previous loop, we set a value, use it too
                    if hasattr(self, prop_name):
                        # If the current value is strong, it will simplify the problem
                        if handle_additive_inheritance and not isinstance(value, list) and value.startswith('+'):
                            # In this case we can remove the + from our current
                            # tpl because our value will be final
                            value = getattr(self, prop_name) + ',' + value[1:]
                            value = _format_inherited_plus_value(value)
                        # In case of bp_rule we must not append current value with inherited one
                        elif prop_name != 'check_command' or not value.startswith('bp_rule!'):  # If not, se should keep the + sign of need
                            value = getattr(self, prop_name) + ',' + value
                            value = _format_inherited_plus_value(value)
                    # Ok, we can set it
                    setattr(self, prop_name, value)
                    
                    # If we only got some '+' values, we must still loop
                    # for an end value without it
                    if not still_loop:
                        # And set my own value in the end if need
                        if self.has_plus(prop_name):
                            value = getattr(self, prop_name) + ',' + self.get_plus_and_delete(prop_name)
                            value = _format_inherited_plus_value(value)
                            # Template should keep their '+'
                            if handle_additive_inheritance and self.is_tpl() and not value.startswith('+'):
                                value = '+' + value
                            setattr(self, prop_name, value)
                        elif handle_additive_inheritance and value.startswith('+'):
                            value = _format_inherited_plus_value(value)
                            setattr(self, prop_name, value)
                        return value
        
        # Maybe templates only give us + values, so we didn't quit, but we already got a
        # self.prop value after all
        template_with_only_plus = hasattr(self, prop_name)
        
        # I do not have ending prop, my templates too... Maybe a plus?
        # warning: if all my templates gave me '+' values, do not forgot to
        # add the already set self.prop value
        if self.has_plus(prop_name):
            if template_with_only_plus:
                value = getattr(self, prop_name) + ',' + self.get_plus_and_delete(prop_name)
                value = _format_inherited_plus_value(value)
            else:
                value = _format_inherited_plus_value(self.get_plus_and_delete(prop_name))
            # Template should keep their '+' chain
            # We must say it's a '+' value, so our son will now that it must
            # still loop
            if self.is_tpl() and handle_additive_inheritance and not value.startswith('+'):
                value = '+' + value
            setattr(self, prop_name, value)
            return value
        
        # SEF-8015 Special case: if we are a cluster template with void bp_rule!, we prefer give nothing in fact
        if self.is_tpl() and prop_name == 'check_command' and getattr(self, 'is_cluster', '0') == '1' and getattr(self, prop_name, None) == 'bp_rule!':
            logger.debug('The cluster %s do not have any bp_rule, so we prefer give nothing instead' % self.get_name())
            return None
        
        # Ok so in the end, we give the value we got if we have one, or None
        # Not even a plus... so None :)
        return getattr(self, prop_name, None)
    
    
    # We fillfull properties with template ones if need
    def get_customs_properties_by_inheritance(self, items, level=0):
        # Protect against too much recursive levels
        if level >= MAX_INHERITANCE_LEVEL:
            return self.customs
        
        for i in self.templates:
            tpl_cv = i.get_customs_properties_by_inheritance(items, level=level + 1)
            if tpl_cv is not {}:
                for prop in tpl_cv:
                    if prop not in self.customs:
                        value = tpl_cv[prop]
                    else:
                        value = self.customs[prop]
                    self.customs[prop] = value
        
        # We can get custom properties in plus, we need to get all
        # entires and put them into customs
        cust_in_plus = self.get_all_plus_and_delete()
        for prop in cust_in_plus:
            self.customs[prop] = cust_in_plus[prop]
        # Clean null values for not template element (null = remove property)
        if not self.is_tpl() or hasattr(self, 'service_description'):
            to_del = []
            for (prop, value) in self.customs.iteritems():
                if value == 'null':
                    to_del.append(prop)
            for prop in to_del:
                del self.customs[prop]
        
        return self.customs
    
    
    def has_plus(self, prop):
        return prop in self.plus
    
    
    def get_all_plus_and_delete(self):
        res = {}
        props = self.plus.keys()  # we delete entries, so no for ... in ...
        for prop in props:
            res[prop] = self.get_plus_and_delete(prop)
        return res
    
    
    def get_plus_and_delete(self, prop):
        return self.plus.pop(prop)
    
    
    def get_templates_as_flatten_list(self, level=0):
        if hasattr(self, '_flatten_templates'):
            return self._flatten_templates
        
        if level > MAX_INHERITANCE_LEVEL:
            return []
        
        _flatten_templates = []
        for template in self.templates:
            if template not in _flatten_templates:
                _flatten_templates.append(template)
                higher_templates = template.get_templates_as_flatten_list(level=level + 1)
                _flatten_templates.extend([h for h in higher_templates if h not in _flatten_templates])
        self._flatten_templates = _flatten_templates
        return self._flatten_templates


class InheritableItems(Items):
    
    # Inheritance for just a property
    def apply_partial_inheritance(self, prop_name):
        inheritance_method = getattr(self.inner_class, 'get_%s_by_inheritance' % prop_name, None)
        
        # First templates
        for inheritable_item in self.templates.values():
            if inheritance_method:
                inheritance_method(inheritable_item, self, only_one_level=False)
            else:
                inheritable_item.get_property_by_inheritance(self, prop_name, in_template_only_loop=True)
        
        for inheritable_item in self:
            if not inheritable_item.is_tpl():
                if inheritance_method:
                    inheritance_method(inheritable_item, self, only_one_level=True)
                else:
                    inheritable_item.get_property_by_inheritance(self, prop_name, in_template_only_loop=False)
            
            if not inheritable_item.is_tpl() or hasattr(inheritable_item, 'service_description'):
                if getattr(inheritable_item, prop_name, None) == 'null':
                    delattr(inheritable_item, prop_name)
    
    
    def apply_inheritance(self, parameter_only=None):
        # We check for all properties if the item has a value if not, it check in all item templates for a value.
        cls = self.inner_class
        
        if parameter_only is None:
            present_properties = set()
            for prop_name, prop_def in cls.properties.iteritems():
                if not prop_def.inherit:
                    continue
                for inheritable_item in self:
                    if hasattr(inheritable_item, prop_name) or prop_name in getattr(inheritable_item, 'plus', {}):
                        present_properties.add(prop_name)
                        break
        
        else:  # ok we filter for just a property
            present_properties = set([parameter_only])
        
        for prop_name in present_properties:
            self.apply_partial_inheritance(prop_name)
        
        # We only want to compute DATA only once (for tunning purpose we get into this method twice)
        if parameter_only is None:
            for inheritable_item in self:
                inheritable_item.get_customs_properties_by_inheritance(self)
    
    
    def create_tpl_list(self):
        for id in self.items:
            i = self.items[id]
            if i.is_tpl():
                self.templates[id] = i
    
    
    def find_tpl_by_name(self, name):
        for i in self.templates.values():
            if hasattr(i, 'name') and i.name == name:
                return i
        return None
    
    
    # We will link all templates, and create the template graph too
    def linkify_templates(self):
        # First we create a list of all templates
        self.create_tpl_list()
        for i in self:
            tpls = i.get_templates()
            new_tpls = []
            for tpl in tpls:
                tpl = tpl.strip()
                # We save this template in the 'tags' set
                if tpl not in i.tags:
                    i.tags.append(tpl)
                # Then we link it
                t = self.find_tpl_by_name(tpl)
                # If it's ok, add the template and update the
                # template graph too
                if t is not None:
                    # add the template object to us
                    new_tpls.append(t)
                    # else:  # not find? not good!
                    #    err = "the template '%s' defined for '%s' is unknown" % (tpl, i.get_name())
                    #    i.configuration_warnings.append(err)
            i.templates = new_tpls
        
        # Now we will create the template graph, so
        # we look only for templates here. First we should declare our nodes
        for tpl in self.templates.values():
            self.templates_graph.add_node(tpl)
        # And then really create our edge
        for tpl in self.templates.values():
            for father in tpl.templates:
                self.templates_graph.add_edge(father, tpl)
        
        # Now get the list of all elements in a loop
        elts_in_loops = self.templates_graph.loop_check()
        
        # and raise errors about it
        for e in elts_in_loops:
            err = "The template '%s' is part of a circular parent/child cyclic endless chain." % e.get_link_tpl_shinken()
            self.configuration_errors.append(err)
    
    
    def remove_templates(self):
        """ Remove useless templates (& properties) of our items ; otherwise we could get errors on config.is_correct() """
        tpls = [i for i in self if i.is_tpl()]
        for i in tpls:
            del self[i.id]
        del self.templates
        del self.templates_graph
