#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2020
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from .item_property.item_property_synchronization import PROPERTY_USER_RIGHTS
from ..component.component_manager import component_manager
from ..dao.checks_inheritance import lookup_items_templates
from ..dao.def_items import ITEM_STATE, get_item_types_for_linked_prop, STOP_INHERITANCE_VALUES, ITEM_TYPE
from ..dao.helpers import is_item_link_in_item_links, split_and_strip_list
from ..dao.helpers import is_user_rights_default_value_everyone, is_stop_inheritance, get_flat_values_to_links

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, List, Str, NoReturn, Optional
    from synchronizer.dao.datamanagerV2 import DataManagerV2


class ItemConsistency(object):
    
    def __init__(self, datamanager_v2):
        # type: (DataManagerV2) -> NoReturn
        self.datamanagerV2 = datamanager_v2
    
    
    def get_missing_value(self, item, item_type, property_name):
        # type: (Dict, Str, Str) -> List
        if property_name in (PROPERTY_USER_RIGHTS.VIEW_CONTACTS, PROPERTY_USER_RIGHTS.VIEW_CONTACT_GROUPS):
            _consistency = ItemConsistencyUserRight(self.datamanagerV2, item, item_type, property_name)
            return _consistency.get_missing_value()
        return []


class ItemConsistencyUserRight(object):
    
    def __init__(self, datamanager_v2, item, item_type, property_name):
        # type: (DataManagerV2, Dict, Str, Str) -> NoReturn
        self.datamanagerV2 = datamanager_v2
        self.translator = None
        self.property_name = property_name
        self.item = item
        self.item_type = item_type
        # get_item_types_for_linked_prop must be return 1 value here.
        # If not we must crash. because we check UserRight here so prop are only view_contacts, edition_contacts, notification_contacts, view_contact_groups,  edition_contact_groups, notification_contact_groups
        self.item_type_property = get_item_types_for_linked_prop(self.item_type, self.property_name)[0]
        self.list_templates = None  # type: Optional[ItemConsistencyListTemplate]
        self.init_templates()
    
    
    def init_templates(self):
        # type: () -> NoReturn
        _templates_to_string = []
        _names_templates = self.get_value_for_property('use')
        lookup_items_templates(self.item_type, _names_templates.split(','), _templates_to_string, self.datamanagerV2)
        self.list_templates = ItemConsistencyListTemplate(_templates_to_string, self.datamanagerV2)
    
    
    def translate(self, key):
        # type: (Str) -> Str
        if not self.translator:
            self.translator = component_manager.get_translate_component().translator()
        return self.translator.translate(key)
    
    
    def get_missing_value(self):
        # type: () -> List
        _to_return = []
        _is_everyone = is_user_rights_default_value_everyone()
        if _is_everyone:
            _has_view_contact = self.has_value_computed_for_property(PROPERTY_USER_RIGHTS.VIEW_CONTACTS)
            _has_view_contact_groups = self.has_value_computed_for_property(PROPERTY_USER_RIGHTS.VIEW_CONTACT_GROUPS)
            if _has_view_contact or _has_view_contact_groups:
                self._add_missing_user_rights(_to_return)
        else:
            self._add_missing_user_rights(_to_return)
        return _to_return
    
    
    def _add_missing_user_rights(self, to_return):
        # type: (List) -> NoReturn
        _properties = PROPERTY_USER_RIGHTS.get_synchronized_property(self.property_name)
        
        _missing = self.get_concatenated_value_for(_properties)
        if _missing:
            _existing = self.get_concatenated_value_for([self.property_name])
            for _current in [_elem for _elem in _missing if not is_item_link_in_item_links(_elem, _existing)]:
                to_return.append(_current)
    
    
    def get_concatenated_value_for(self, item_properties_name):
        # type: (List[Str]) -> List[Dict]
        _to_return = []
        for _prop_name in item_properties_name:
            _to_adds = self.get_values_for(_prop_name)
            for _to_add in [_l for _l in _to_adds if not is_item_link_in_item_links(_l, _to_return)]:
                _to_return.append(_to_add)
        return _to_return
    
    
    def get_values_for(self, property_name):
        # type: (Str) -> List[Dict]
        
        _item_property_value = self.item.get(property_name, '')
        
        if _item_property_value and isinstance(_item_property_value, basestring):
            _item_property_value = get_flat_values_to_links(_item_property_value, self.item_type_property, ITEM_STATE.STAGGING, self.datamanagerV2)
        
        if is_stop_inheritance(_item_property_value):
            return []
        
        if _item_property_value and not _item_property_value.get('has_plus', False):
            return _item_property_value.get('links', [])
        
        _to_return = _item_property_value.get('links', [])[:] if _item_property_value else []
        
        try:
            for _use in split_and_strip_list(self.get_value_for_property(u'use')):
                _template = self.list_templates.templates_mapping[_use]
                _to_add = _template.user_rights[property_name]
                if not _to_add:
                    if _template.user_rights_stop_inheritance[property_name]:
                        break
                    continue
                if _to_add in STOP_INHERITANCE_VALUES:
                    break
                for _to_add_s in [_l for _l in _to_add.get('links', []) if not is_item_link_in_item_links(_l, _to_return)]:
                    _to_return.append(_to_add_s)
                if not _to_add['has_plus']:
                    return _to_return
        except Exception as e:
            logger.error(u'[compute_final LOOP USE] [get_values_for] [%s]' % e)
            logger.print_stack()
        
        return _to_return
    
    
    def get_value_for_property(self, property_name):
        # type: (Str) -> Str
        _to_return = self.datamanagerV2.flatten_prop(self.item, self.item_type, property_name)
        
        if _to_return in STOP_INHERITANCE_VALUES:
            return ''
        
        return _to_return
    
    
    def has_value_computed_for_property(self, property_name):
        # type: (Str) -> bool
        _to_return = self.get_value_for_property(property_name)
        if _to_return:
            if _to_return != '+':
                return True
        
        return bool(self.get_values_for(property_name))


class ItemConsistencyListTemplate(object):
    
    def __init__(self, templates, datamanager_v2):
        # type: (List, DataManagerV2) -> NoReturn
        self.templates = []
        self.templates_mapping = {}
        
        for _tpl in templates:
            _to_add = ItemConsistencyTemplate(_tpl, datamanager_v2)
            self.templates.append(_to_add)
            self.templates_mapping[_to_add.name] = _to_add
        
        self.compute_final()
    
    
    def compute_final(self):
        # type: () -> NoReturn
        self.templates.reverse()
        for _current in self.templates:
            if not _current.use:
                continue
            _to_update_with = []
            try:
                for j in _current.use.split(','):
                    _to_update_with.append(self.templates_mapping[j])
                _current.update_with(_to_update_with)
            except Exception as e:
                logger.error(u'[compute_final LOOP USE] [compute_final] [%s]' % e)
                logger.print_stack()


class ItemConsistencyTemplate(object):
    
    def __init__(self, data, datamanager_v2):
        # type: (Dict, DataManagerV2) -> NoReturn
        self.datamanagerV2 = datamanager_v2
        self.id = data.get('_id', '')
        self.name = data.get('name', '')
        self.use = data.get('use', '')
        self.user_rights = {}
        self.user_rights_stop_inheritance = {}
        for prop_name in PROPERTY_USER_RIGHTS.LIST:
            self.user_rights[prop_name] = self.set_value(prop_name, data.get(prop_name, ''), get_item_types_for_linked_prop(ITEM_TYPE.HOSTS, prop_name)[0])
    
    
    def set_value(self, prop_name, value, item_type):
        # type: (Str, Str, Str) -> NoReturn
        self.user_rights_stop_inheritance[prop_name] = False
        if value == '':
            return value
        if is_stop_inheritance(value):
            self.user_rights_stop_inheritance[prop_name] = True
            return ''
        return get_flat_values_to_links(value, item_type, ITEM_STATE.STAGGING, self.datamanagerV2)
    
    
    def update_with(self, templates):
        # type: (List[ItemConsistencyTemplate]) -> NoReturn
        for prop_name in PROPERTY_USER_RIGHTS.LIST:
            if self.user_rights_stop_inheritance[prop_name]:
                continue
            _stop_plus = False
            for template in templates:
                if _stop_plus:
                    continue
                if self.user_rights[prop_name] and not self.user_rights[prop_name]['has_plus']:
                    continue
                if not template.user_rights[prop_name]:
                    continue
                if is_stop_inheritance(template.user_rights[prop_name]):
                    continue
                if not self.user_rights[prop_name]:
                    self.user_rights[prop_name] = template.user_rights[prop_name]
                else:
                    self.user_rights[prop_name]['links'].extend(template.user_rights[prop_name]['links'])
                    if not template.user_rights[prop_name]['has_plus']:
                        _stop_plus = True
