#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.util import DuplicateForEachParser
from shinkensolutions.toolbox.box_tools_string import ToolsBoxString
from .massive_change_action import MassiveChangeAction
from .massive_change_data import MassiveChangeData
from .massive_change_detail import MassiveChangeDetail
from .massive_change_element import MassiveChangeElement
from .massive_change_helper import add_item_in_list, FORBIDDEN_PROPERTIES, ALLOWED_PROPERTIES, mock_property
from .massive_change_item_with_data import MassiveChangeItemWithData
from .massive_change_operator import MASSIVE_CHANGE_OPERATOR
from .massive_change_override_element import MassiveChangeOverrideElement
from .massive_change_step import MASSIVE_CHANGE_STEP
from .massive_change_type import MASSIVE_CHANGE_TYPE
from .. import exceptions
from ..exceptions import MassiveChangeError, BusinessException, UnauthorizedSaveException
from ...item_controller.edit_item_context import ACTIONS, EditItemContext
from ...sync_ui_common import syncuicommon
from ....dao.def_items import ITEM_STATE, ITEM_TYPE, METADATA, NOT_TO_LOOK, SERVICE_EXCLUDES_BY_NAME, SERVICE_EXCLUDES_BY_ID, SERVICE_OVERRIDE, STOP_INHERITANCE_VALUES, DEF_ITEMS, can_property_contain_links, NO_HERITABLE_ATTR
from ....dao.helpers import update_frontend_links_into_links
from ....dao.item_saving_formatter import _compute_check_index
from ....dao.transactions.transactions import DBTransaction
from ....front_end.helper import natural_keys

if TYPE_CHECKING:
    from shinken.misc.type_hint import List, Optional, Dict, Set, Union, Tuple
    from ....dao.items.contactitem import ContactItem
    from ....dao.items.baseitem import BaseItem
    from ....dao.def_items import ItemType
    from ....synchronizerdaemon import Synchronizer


class MassiveChangeController(object):
    def __init__(self, app):
        # type: (Synchronizer) -> None
        self.app = app
        self.datamanagerV2 = app.datamanagerV2
        exceptions.app = app
    
    
    # ----------- MISC -----------#
    def _get_items_from_item_dict(self, items_list, item_types, allow_duplicate=True):
        # type: (Dict, Optional[Union[Set, List]], bool) -> Tuple[List, List, Dict]
        items = []
        ignored_items = []
        missing_items = {}
        
        for item_uuid, item_details in items_list.iteritems():
            if item_details.get(u'to_ignore', False):
                ignored_items.append(item_details)
                continue
            _where = {u'_id': item_uuid}
            for _type in item_types:
                _states = (ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING) if ITEM_TYPE.has_work_area(_type) else (ITEM_STATE.STAGGING,)
                _items = self.datamanagerV2.find_merge_state_items(_type, _states, where=_where)
                
                if _items:
                    if allow_duplicate or _items[0] not in items:
                        items.append(_items[0])
                    break
            else:
                missing_items[item_uuid] = item_details
        
        return items, ignored_items, missing_items
    
    
    # ----------- VALIDATE -----------#
    def _validate_form(self, form, item_type, user):
        # type: (Dict, ItemType, ContactItem) -> None
        if item_type not in ITEM_TYPE.ALL_TYPES:
            raise MassiveChangeError(text=u'Unknown item_type : [%s].' % item_type)
        
        if not user or not user.is_admin():
            raise UnauthorizedSaveException()
        
        # item_state is mandatory only for step 5 (save)
        if form.get('step', 0) == MASSIVE_CHANGE_STEP._5:
            
            bypass_work_area = form.get(u'bypass_work_area', False)
            
            form[u'item_state'] = ITEM_STATE.WORKING_AREA if ITEM_TYPE.has_work_area(item_type) and not bypass_work_area else ITEM_STATE.STAGGING
        
        type_modification = form.get(u'type_modification', None)
        
        if not type_modification or type_modification not in MASSIVE_CHANGE_TYPE.ALL_CHANGES:
            raise MassiveChangeError(text=u'Received a bad parameter for type_modification : [%s].' % type_modification)
        
        missing_fields = [f for f in MASSIVE_CHANGE_TYPE.MANDATORY_FIELDS[type_modification] if not form.get(f, None)]
        
        if missing_fields:
            raise MassiveChangeError(text=u'To change a %s, you need to set these mandatory fields : %s.' % (type_modification, u', '.join(missing_fields)))
        
        if type_modification == MASSIVE_CHANGE_TYPE.PROPERTIES:
            self._valid_properties_form(form, item_type)
        elif type_modification == MASSIVE_CHANGE_TYPE.OVERRIDES:
            self._valid_override_form(form, item_type)
        elif type_modification == MASSIVE_CHANGE_TYPE.EXCLUDES:
            self._valid_excludes_form(form, item_type)
        elif type_modification in (MASSIVE_CHANGE_TYPE.LOCAL_DATAS, MASSIVE_CHANGE_TYPE.DFE_DATAS):
            self._valid_data_form(form, item_type)
    
    
    def _valid_properties_form(self, form, item_type):
        # type: (Dict, ItemType) -> None
        actions = []
        # Edition_contacts and Notification_contacts must be before views_contacts and view_contacts must be before view_contacts-SYNCHRO-edition_contacts
        
        for names, order in form[u'update'].iteritems():
            props_name = names.split(u'-SYNCHRO-')
            prop_name = props_name[0]
            synchronized_prop = None
            if len(props_name) > 1:
                synchronized_prop = props_name[1]
            
            _action = order.get(u'action', None)
            _extra_action = order.get(u'extra_action', None)
            _is_linked = can_property_contain_links(item_type, prop_name)
            _value = order.get(u'value', None)
            _target = order.get(u'target', None)
            _category = order.get(u'category', None)
            _is_protected = order.get(u'is_protected', False)
            _plus_action = None
            _manage_plus = False
            
            if prop_name.startswith('_') and prop_name not in NOT_TO_LOOK:
                self._valid_one_data(prop_name, _action)
            else:
                _value, _manage_plus, _plus_action = self._valid_one_property(prop_name, item_type, _value, _action, _extra_action, _is_linked)
            
            self._valid_value(_action, _extra_action, _value)
            self._valid_target(_action, _extra_action, _target)
            _action = MassiveChangeAction(
                prop_name,
                item_type,
                _action,
                form[u'type_modification'],
                self.datamanagerV2,
                extra_action=_extra_action,
                value=_value,
                target=_target,
                category=_category,
                app=self.app,
                synchronized_prop=synchronized_prop,
                execution=order.get(u'index', 999),
                manage_plus=_manage_plus,
                is_protected=_is_protected,
                plus_action=_plus_action,
                prop_is_linked=_is_linked
            )
            actions.append(_action)
        
        form[u'actions'] = sorted(actions, key=lambda a: (a.execution, a.prop_name))
    
    
    @staticmethod
    def _valid_one_property(prop_name, item_type, value, action, extra_action, is_linked):
        # type: (unicode, ItemType, Union[Dict, unicode], unicode, unicode, bool) -> (unicode, bool, bool)
        plus_action = None
        properties = DEF_ITEMS[item_type][u'class'].properties
        _prop = properties.get(prop_name, None)
        
        if not _prop and prop_name in ALLOWED_PROPERTIES[item_type]:
            _prop = mock_property(prop_name, False, u'uniq')
        
        if prop_name in FORBIDDEN_PROPERTIES[item_type] or not _prop:
            raise MassiveChangeError(text=u'The property [%s] is not eligible to massive_change_call.' % prop_name)
        
        manage_plus = _prop.handle_additive_inheritance
        if not ITEM_TYPE.has_template(item_type) or prop_name in NO_HERITABLE_ATTR:
            manage_plus = False
        
        # Format link if value is in STOP_INHERITANCE_VALUES
        if is_linked and value and value in STOP_INHERITANCE_VALUES:
            if prop_name == u'bp_rule':
                # CARPET : The bp_rule are not concerned because the STOP_INHERITANCE_VALUES are not manage at expected : it will search a host with the STOP_INHERITANCE_VALUES.
                # The final behaviour is that the cluster did match any element so "it's works" but it's not the real behaviour
                pass
            elif prop_name in DEF_ITEMS[item_type][u'item_links'].get(ITEM_TYPE.COMMANDS, []) and item_type != ITEM_TYPE.NOTIFICATIONWAYS:
                value = {u'node': {u'args': u'', u'link': {u'name': value, u'exists': False}}, u'raw_value': value}
            else:
                value = {u'has_plus': False, u'links': [{u'name': value, u'exists': False}]}
        
        if value and manage_plus:
            plus_action = value.get(u'has_plus', None)
        
        if prop_name in [u'flap_detection_options', u'notification_options']:
            _prop_type = u'uniq'
        elif item_type in [ITEM_TYPE.SERVICESHOSTS, ITEM_TYPE.SERVICESCLUSTERS] and prop_name == u'host_name':
            _prop_type = u'join'
        else:
            _prop_type = _prop.merging
        
        if not action or action not in MASSIVE_CHANGE_OPERATOR.ALL_ACTIONS:
            raise MassiveChangeError(text=u'parameter "action" is incorrect or missing.')
        
        if _prop_type == u'ordered':
            if MASSIVE_CHANGE_OPERATOR.ALL_EXTRA_ACTIONS.get(action, None):
                if not extra_action or extra_action not in MASSIVE_CHANGE_OPERATOR.ALL_EXTRA_ACTIONS[action]:
                    raise MassiveChangeError(text=u'For property "%s" of type %s, you need to set an extra_action.' % (prop_name, _prop_type))
        _action = u'%s_%s' % (action, extra_action) if action and extra_action else action
        
        if _action not in MASSIVE_CHANGE_OPERATOR.RULES[_prop_type]:
            raise MassiveChangeError(text=u'Action [%s%s] not allowed for property type [%s].' % (action, ' %s' % extra_action if extra_action else '', _prop_type))
        
        if prop_name == u'duplicate_foreach' and value and value not in STOP_INHERITANCE_VALUES:
            value = u'_%s' % value.upper().lstrip(u'_')
        
        return value, manage_plus, plus_action
    
    
    @staticmethod
    def _valid_one_data(data_name, action):
        # type: (unicode, unicode) -> None
        
        if not data_name.startswith(u'_'):
            raise MassiveChangeError(text=u'This is not a data : %s.' % data_name)
        
        if data_name in NOT_TO_LOOK:
            raise MassiveChangeError(text=u'Cannot modify this property : %s.' % data_name)
        
        if not action or action not in MASSIVE_CHANGE_OPERATOR.ALL_DATA_ACTIONS:
            raise MassiveChangeError(text=u'parameter "action" is incorrect or missing.')
    
    
    @staticmethod
    def _valid_value(action, extra_action, value):
        # type: (unicode, unicode, Dict) -> None
        if action not in (MASSIVE_CHANGE_OPERATOR.EMPTY, MASSIVE_CHANGE_OPERATOR.ADD_CONSISTENCY):
            if not value:
                raise MassiveChangeError(text=u'Missing value for action : %s%s.' % (action, u' %s' % extra_action if extra_action else u''))
    
    
    @staticmethod
    def _valid_target(action, extra_action, target):
        # type: (unicode, unicode, Dict) -> None
        if action == MASSIVE_CHANGE_OPERATOR.REPLACE or (action in (MASSIVE_CHANGE_OPERATOR.ADD, MASSIVE_CHANGE_OPERATOR.MOVE) and extra_action in (MASSIVE_CHANGE_OPERATOR.AFTER, MASSIVE_CHANGE_OPERATOR.BEFORE)):
            if not target:
                raise MassiveChangeError(text=u'Missing target for action : %s_%s.' % (action, extra_action))
    
    
    def _valid_override_form(self, form, item_type):
        # type: (Dict, ItemType) -> None
        if item_type not in [ITEM_TYPE.HOSTS, ITEM_TYPE.HOSTTPLS]:
            raise MassiveChangeError(text=u'You can not override a service on type %s.' % item_type)
        
        self._valid_properties_form(form, DEF_ITEMS[item_type][u'check_type'])
    
    
    def _valid_excludes_form(self, form, item_type):
        # type: (Dict, unicode) -> None
        if item_type not in [ITEM_TYPE.HOSTS, ITEM_TYPE.HOSTTPLS]:
            raise MassiveChangeError(text=u'You can not exclude a service on type %s.' % item_type)
        
        actions = []
        order = form[u'update'].get(SERVICE_EXCLUDES_BY_NAME, None)
        
        if order:
            
            _action = order.get(u'action', None)
            _value, _manage_plus, _plus_action = self._valid_one_property(SERVICE_EXCLUDES_BY_NAME, item_type, order.get(u'value', None), _action, order.get(u'extra_action', None), True)
            
            _action = MassiveChangeAction(
                SERVICE_EXCLUDES_BY_NAME,
                item_type,
                _action,
                MASSIVE_CHANGE_TYPE.EXCLUDES,
                self.datamanagerV2,
                extra_action=order.get(u'extra_action', None),
                value=_value,
                target=order.get(u'target', None),
                category=order.get(u'category', None),
                app=self.app,
                prop_is_linked=False
            )
            actions.append(_action)
        
        form[u'actions'] = actions
    
    
    def _get_uncrypted_value(self, order):
        # type: (Dict) -> unicode
        value = order.get(u'value')
        if value in STOP_INHERITANCE_VALUES:
            return value
        
        if order.get(u'is_protected', False) and order[u'action'] in (MASSIVE_CHANGE_OPERATOR.SET, MASSIVE_CHANGE_OPERATOR.SET_IF_DATA_EXISTS_ON_ITEM, MASSIVE_CHANGE_OPERATOR.SET_IF_DATA_IS_SET_ON_ITEM):
            return self.app.frontend_cipher._uncipher_value(value)
        else:
            return value
    
    
    def _valid_data_form(self, form, item_type):
        # type: (Dict, ItemType) -> None
        actions = []
        
        for prop_name, order in form[u'update'].iteritems():
            uncrypted_value = self._get_uncrypted_value(order)
            self._valid_one_data(prop_name, order.get(u'action', None))
            _action = MassiveChangeAction(
                prop_name,
                item_type,
                order.get(u'action', None),
                form[u'type_modification'],
                self.datamanagerV2,
                extra_action=order.get(u'extra_action', None),
                value=uncrypted_value,
                target=order.get(u'target', None),
                category=order.get(u'category', None),
                is_protected=order.get(u'is_protected', False),
                app=self.app
            )
            actions.append(_action)
        
        form['actions'] = actions
    
    
    # --------------- APPLY ---------------#
    def _apply_orders_on_one_item(self, form, item_type, item_to_apply_change):
        # type: (Dict, ItemType, MassiveChangeElement) -> Tuple[bool, bool]
        modified = False
        impacted = True
        
        type_modification = form[u'type_modification']
        
        if type_modification in (MASSIVE_CHANGE_TYPE.PROPERTIES, MASSIVE_CHANGE_TYPE.LOCAL_DATAS, MASSIVE_CHANGE_TYPE.DFE_DATAS):
            modified = self._launch_actions(form[u'actions'], item_to_apply_change, type_modification=type_modification)
        elif type_modification == MASSIVE_CHANGE_TYPE.OVERRIDES:
            modified, impacted = self._apply_overrides_modifications(form, item_to_apply_change, item_type)
        elif type_modification == MASSIVE_CHANGE_TYPE.EXCLUDES:
            modified, impacted = self._apply_excludes_modifications(form, item_type, item_to_apply_change)
        
        return modified, impacted
    
    
    @staticmethod
    def _validate_item(item_type, validations, change_element):
        # type: (ItemType, Dict, MassiveChangeElement) -> None
        _id = change_element.get_uuid()
        item_validation = syncuicommon.validator.validate(item_type, change_element.raw_item)
        
        change_element.set_validation(item_validation)
        
        if item_validation[u'has_error'] or item_validation[u'has_warning']:
            if _id not in validations:
                validations[_id] = {}
            validations[_id].update(item_validation)
    
    
    def _launch_actions(self, actions, change_element, type_modification=None):
        # type: (List[MassiveChangeAction], MassiveChangeElement, unicode) -> bool
        modified = False
        _dfe_list = self._get_data_dfe_list(change_element.item) if type_modification in [MASSIVE_CHANGE_TYPE.LOCAL_DATAS, MASSIVE_CHANGE_TYPE.DFE_DATAS] else []
        
        for action in actions:
            if (type_modification == MASSIVE_CHANGE_TYPE.LOCAL_DATAS and action.prop_name in _dfe_list) or (type_modification == MASSIVE_CHANGE_TYPE.DFE_DATAS and action.prop_name not in _dfe_list):
                continue
            _modified = action.apply(change_element)
            modified |= _modified
        return modified
    
    
    # ------------- OVERRIDES -------------#
    def _apply_overrides_modifications(self, form, item_to_apply_change, item_type):
        # type: (Dict, MassiveChangeElement, ItemType) -> (bool, bool)
        modified = False
        impacted = False
        in_creation = False
        raw_item = item_to_apply_change.raw_item
        
        check_ids = [c[u'check'][u'_id'] for c in METADATA.get_metadata(item_to_apply_change.item, METADATA.CHECKS, [])]
        for _host_link, _parent_link, _check_link, dfe_key in form[u'checks']:
            _override = None
            if _host_link[u'_id'] != raw_item[u'_id'] or _check_link[u'_id'] not in check_ids:
                continue
            
            impacted = True
            check_name = _check_link[u'name']
            update_frontend_links_into_links(SERVICE_OVERRIDE, [_host_link, _parent_link, _check_link], item_type, self.datamanagerV2, linkify=False)
            
            service_overrides_to_apply_change = raw_item.get(SERVICE_OVERRIDE, None)
            if not service_overrides_to_apply_change:
                service_overrides_to_apply_change = {
                    u'raw_value'       : u'',
                    u'links'           : [],
                    u'has_errors'      : False,
                    u'has_error'       : False,
                    u'error'           : u'',
                    u'all_linked_types': []
                }
                raw_item[SERVICE_OVERRIDE] = service_overrides_to_apply_change
                in_creation = True
            
            _modified = self._process_modification_in_override(form, check_name, _check_link, _parent_link, dfe_key, service_overrides_to_apply_change, item_to_apply_change, in_creation)
            modified |= _modified
        
        return modified, impacted
    
    
    def _process_modification_in_override(self, form, check_name, _check_link, _host_link, dfe_key, service_overrides, item_to_apply_change, in_creation):
        # type: (Dict, unicode, Dict, Dict, unicode, Dict, MassiveChangeElement, bool) -> bool
        service_override_links = service_overrides.get(u'links', [])  # type: List[Dict]
        keys_in_form = [k for k in form[u'update'].iterkeys()]
        keys_to_add = keys_in_form[:]
        
        _item = {
            u'_id'       : _check_link[u'_id'],
            u'name'      : check_name,
            u'item_type' : _check_link[u'item_type'],
            u'item_state': ITEM_STATE.STAGGING
        }
        
        # First, save _index of matching override_link
        for override_link in service_override_links[:]:
            override_key = override_link[u'key']
            override_value = override_link[u'value']
            override_check_link = override_link[u'check_link']
            
            is_same_when_check_exists = override_check_link[u'exists'] and override_check_link == _check_link and override_link[u'host_link'] == _host_link
            is_same_when_check_not_exists = not override_check_link[u'exists'] and override_check_link.get(u'name', None) == check_name
            is_same_dfe_key = override_key in keys_in_form and override_link.get(u'dfe_key', u'') == dfe_key
            
            if (is_same_when_check_exists or is_same_when_check_not_exists) and is_same_dfe_key:
                
                if can_property_contain_links(_check_link[u'item_type'], override_key) and override_value.get(u'links', False):
                    update_frontend_links_into_links(override_key, override_value[u'links'], _check_link[u'item_type'], self.datamanagerV2, linkify=False)
                
                _item[override_key] = override_value
                service_override_links.remove(override_link)
        
        override_element = MassiveChangeOverrideElement(_item, dfe_key)
        modified = self._launch_actions(form[u'actions'], override_element)
        
        # Then, create the new links
        if keys_to_add:
            self._add_value_in_override(keys_to_add, override_element.raw_item, _check_link, _host_link, dfe_key, service_overrides[u'links'])
        
        if service_override_links:
            # If we have an override, we must add the name into the values links to be consistent with the operation of override
            for _index, override_link in enumerate(service_override_links):
                if override_link[u'check_link'] == _check_link and override_link[u'host_link'] == _host_link and override_link[u'key'] in keys_in_form and override_link.get(u'dfe_key', u'') == dfe_key:
                    override_key = override_link[u'key']
                    
                    if can_property_contain_links(_check_link[u'item_type'], override_key) and override_link[u'value'].get(u'links', False):
                        for linked_values in override_link[u'value'][u'links']:
                            if linked_values[u'exists']:
                                linked_item = self.datamanagerV2.find_item_by_id(linked_values[u'_id'], linked_values[u'item_type'], ITEM_STATE.STAGGING)
                                linked_values[u'name'] = linked_item.get_name()
            
            service_overrides[u'all_linked_types'] = self._compute_all_linked_types_in_override(service_overrides[u'links'], _check_link, _host_link)
        else:
            item_to_apply_change.raw_item.pop(SERVICE_OVERRIDE, None)
            modified = False if in_creation else modified
        
        item_to_apply_change.add_overrides(override_element, modified)
        
        return modified
    
    
    @staticmethod
    def _add_value_in_override(keys_to_add, raw_item, _check_link, _host_link, dfe_key, links):
        # type: (List, Dict, Dict, Dict, ItemType, List) -> None
        for _key in keys_to_add:
            if raw_item.get(_key, None):
                _link = {
                    u'check_link'             : _check_link,
                    u'host_link'              : _host_link,
                    u'key'                    : _key,
                    u'value'                  : raw_item[_key],
                    u'has_already_been_linked': True
                }
                
                if dfe_key:
                    _link[u'dfe_key'] = dfe_key
                
                links.append(_link)
    
    
    @staticmethod
    def _compute_all_linked_types_in_override(links, _check_link, _host_link):
        # type: (List, Dict, Dict) -> Set
        all_linked_types = set()
        for _link in links:
            if _check_link.get(u'item_type', None):
                all_linked_types.add(_check_link[u'item_type'])
            if _host_link.get(u'item_type', None):
                all_linked_types.add(_host_link[u'item_type'])
            if isinstance(_link[u'value'], dict):
                type_in_value = [lk[u'item_type'] for lk in _link[u'value'][u'links'] if lk[u'exists']]
                all_linked_types = all_linked_types.union(set(type_in_value))
        return all_linked_types
    
    
    # ------------ EXCLUDES ------------#
    def _apply_excludes_modifications(self, form, item_type, item_to_apply_change):
        # type: (Dict, ItemType, MassiveChangeElement) -> (bool, bool)
        modified = False
        impacted = False
        actions = []
        
        # SERVICE_EXCLUDES_BY_NAME is already managed by validate_form and it is in form['actions'], we have just to launch_actions
        actions_in_form = form.get(u'actions', [])
        if actions_in_form:
            actions.extend(actions_in_form)
            impacted = True
        
        checks_to_compute = form[u'update'].get(SERVICE_EXCLUDES_BY_ID, {}).get(item_to_apply_change.get_uuid(), [])
        if checks_to_compute:
            impacted = True
            
            _action = MassiveChangeAction(
                SERVICE_EXCLUDES_BY_ID,
                item_type,
                MASSIVE_CHANGE_OPERATOR.EXCLUDE,
                MASSIVE_CHANGE_TYPE.EXCLUDES,
                self.datamanagerV2,
                value=checks_to_compute,
                category=u'checks',
                app=self.app,
                prop_is_linked=True
            )
            actions.append(_action)
        
        if actions:
            modified = self._launch_actions(actions, item_to_apply_change)
        return modified, impacted
    
    
    # ------------ PUBLIC CALLS ------------ #
    def apply_changes(self, form, item_type, user=None, dry_run=True):
        # type: (Dict, ItemType, Optional[ContactItem], bool) -> Dict
        # First valid data from frontend
        self._validate_form(form, item_type, user)
        
        validations = {}
        modifications_number = 0
        previously_deleted_items = []
        
        items, ignored_items, missing_items = self._get_items_from_item_dict(form[u'items'], [item_type])
        
        mass_change_detail = MassiveChangeDetail(item_type, self.app)
        for item in items:
            _id = item.get_uuid()
            item_name = item.get_name()
            change_element = MassiveChangeElement(item)
            
            if dry_run:
                # Only apply change
                modified, impacted = self._apply_orders_on_one_item(form, item_type, change_element)
                if impacted:
                    self._validate_item(item_type, validations, change_element)
            
            else:
                context = EditItemContext(self.app, _id, item_type, item_state=form[u'item_state'], action=ACTIONS.SAVE_OBJECT, bypass_work_area=form.get(u'bypass_work_area', False))
                
                with DBTransaction(user=user):
                    
                    # Apply all massive change
                    modified, impacted = self._apply_orders_on_one_item(form, item_type, change_element)
                    
                    # Validate and Save item
                    if impacted:
                        self._validate_item(item_type, validations, change_element)
                        if change_element.validation_messages.has_critical():
                            # If catch an error, don't save it !
                            mass_change_detail.add_element(change_element)
                            continue
                        if modified:
                            try:
                                change_element.raw_item.pop(u'@metadata', None)
                                result = self.app.state_controller.update(context, change_element.raw_item)
                                if result[u'previously_deleted']:
                                    previously_deleted_items.append((_id, item_name))
                            except BusinessException as e:
                                raise MassiveChangeError(text=u'Error during item saving : %s.' % e.message)
            
            if modified:
                modifications_number += 1
            mass_change_detail.add_element(change_element)
        
        _have_plus_in_list = True if [a for a in form[u'actions'] if a._manage_plus] else False
        
        mass_change_detail.sort_by_name()
        ignored_items.sort(key=lambda k: natural_keys(k[u'name']))
        
        return {
            u'mass_change_detail'      : mass_change_detail,
            u'validations'             : validations,
            u'ignored_items'           : ignored_items,
            u'missing_items'           : missing_items,
            u'nb_elements'             : len(items),
            u'nb_modifications'        : modifications_number,
            u'have_plus_in_list'       : _have_plus_in_list,
            u'previously_deleted_items': previously_deleted_items
        }
    
    
    def get_active_checks_from_item_dict(self, item_ids, item_type):
        # type: (Dict, ItemType) -> (List, List, Dict, Dict)
        checks_list = []
        all_items = {}
        
        items, ignored_items, missing_items = self._get_items_from_item_dict(item_ids, [item_type])
        
        for item in items:
            item_id = item[u'_id']
            item_name = item.get_name()
            
            add_item_in_list(all_items, item_id, item_type, item.get_state(), item_name, item.get(u'enabled', u'1'))
            on_item_checks = item.get_link_checks()
            
            if not on_item_checks:
                continue
            
            _id_excludes = [link[u'_id'] for link in item.get(u'service_excludes_by_id', {}).get(u'links', []) if link[u'exists']]
            
            item_all_templates = item.find_all_templates_by_links()
            on_item_checks = [(_compute_check_index(c[u'check'], c[u'check'].get_type(), item, item_type, item.get_state(), item_all_templates), c[u'from']) for c in on_item_checks]
            
            on_item_checks.sort(key=lambda x: x[0].check_index)
            
            on_item_checks_names = []
            
            for check_index, _parent in on_item_checks:
                _check = check_index.check
                
                if _check.get_name() in on_item_checks_names or not _check.is_enabled():
                    continue
                
                on_item_checks_names.append(_check.get_name())
                
                _check_id = _check['_id']
                _check_name = _check.get_name()
                _check_type = METADATA.get_metadata(_check, METADATA.ITEM_TYPE)
                
                _parent_id = _parent['_id']
                _parent_type = METADATA.get_metadata(_parent, METADATA.ITEM_TYPE)
                
                _check_in_list = {
                    u'item_id'     : item_id,
                    u'item_type'   : item_type,
                    u'item_name'   : item_name,
                    u'item_state'  : _check.get_state(),
                    u'parent_id'   : _parent_id,
                    u'parent_type' : _parent_type,
                    u'service_id'  : _check_id,
                    u'service_type': _check_type,
                    u'service_name': _check_name,
                    u'dfe_key'     : u'',
                    u'is_exclude'  : _check_id in _id_excludes,
                    u'uuid'        : u'%s-%s' % (item_id, _check_id)
                }
                
                if _check_id not in all_items.iterkeys():
                    add_item_in_list(all_items, _check_id, _check_type, _check.get_state(), _check_name, _check.get(u'enabled', u'1'))
                if _parent_id not in all_items.iterkeys():
                    add_item_in_list(all_items, _parent_id, _parent_type, _parent.get_state(), _parent.get_name(), _parent.get(u'enabled', u'1'))
                
                _dfe = _check.get(u'duplicate_foreach', None)
                
                if _dfe:
                    dfe_definition = item.get(_dfe, None)
                    if dfe_definition:
                        check_dfe_list, _ = DuplicateForEachParser.parse(dfe_definition)
                        for check_dfe_definition in check_dfe_list:
                            check_dfe = _check_in_list.copy()
                            dfe_key = check_dfe_definition[u'KEY']
                            check_dfe[u'service_name'] = _check_name.replace(u'$KEY$', dfe_key)
                            check_dfe[u'dfe_key'] = ToolsBoxString.unescape_XSS(dfe_key)
                            checks_list.append(check_dfe)
                
                else:
                    checks_list.append(_check_in_list)
        
        checks_list = sorted(checks_list, key=lambda ck: (ck[u'item_name'].lower(), ck[u'service_name'].lower()))
        return checks_list, ignored_items, missing_items, all_items
    
    
    def get_data_for_check_overrides(self, check_ids):
        # type: (Dict) -> (List, List, Dict, List, List)
        return self.get_data_for_items(check_ids, ITEM_TYPE.ALL_SERVICES, self_source_list=True, allow_duplicate=False)
    
    
    def get_data_for_items(self, item_ids, item_types, self_source_list=False, allow_duplicate=True):
        # type: (Dict, Union[Set, List], bool, bool) -> (List, List, Dict, List, List)
        items_list = []
        sources_list = []
        data_list = []
        tpls_tmp = []
        
        items, ignored_items, missing_items = self._get_items_from_item_dict(item_ids, item_types, allow_duplicate)
        
        sources_list.append(MassiveChangeItemWithData(self.app.t(u'element.all'), uuid=u'___ALL___'))
        if not self_source_list:
            sources_list.append(MassiveChangeItemWithData(self.app.t(u'element.local_data'), uuid=u'___LocalData___'))
        for item in items:
            tmp_item = MassiveChangeItemWithData(item.get_name(), item[u'_id'])
            if not self_source_list:
                tpls_tmp = item.find_all_templates_by_links()
            dfe_list = self._get_data_dfe_list(item)
            for prop_name in item.iterkeys():
                if not prop_name.startswith('_') or prop_name in NOT_TO_LOOK:
                    continue
                
                if prop_name in dfe_list:
                    continue
                
                sources = [item.get_name()]
                if not self_source_list:
                    sources = []
                    for use in tpls_tmp:
                        if prop_name in use:
                            sources.append(use.get_name())
                            for tpl in sources_list:
                                if tpl.get_uuid() == use[u'_id']:
                                    tpl.add_data(prop_name)
                                    break
                            else:
                                tpl = MassiveChangeItemWithData(use.get_name(), use[u'_id'])
                                tpl.add_data(prop_name)
                                sources_list.append(tpl)
                    if not sources:
                        sources_list[1].add_data(prop_name)
                
                sources_list[0].add_data(prop_name)
                data = None
                for tmp_data in data_list:
                    if tmp_data.get_name() == prop_name:
                        data = tmp_data
                        break
                if data:
                    if sources:
                        data.add_sources(sources)
                    data.add_item(item[u'_id'])
                else:
                    data_list.append(MassiveChangeData(prop_name, item[u'_id'], sources))
                tmp_item.add_data(prop_name)
            if self_source_list and tmp_item.get_data_number() != 0:
                sources_list.append(tmp_item)
            items_list.append(tmp_item)
        
        return data_list, ignored_items, missing_items, items_list, sources_list
    
    
    def get_dfe_for_items(self, item_ids, item_types):
        # type: (Dict, Union[Set, List]) -> (List, List, Dict, List, List)
        host_list = []
        template_list = []
        dfe_list = []
        
        items, ignored_items, missing_items = self._get_items_from_item_dict(item_ids, item_types)
        
        template_list.append(MassiveChangeItemWithData(self.app.t(u'element.all'), uuid=u'___ALL___'))
        template_list.append(MassiveChangeItemWithData(self.app.t(u'element.local_data'), uuid=u'___LocalData___'))
        for item in items:
            tmp_host = MassiveChangeItemWithData(item.get_name(), item[u'_id'])
            tpls_tmp = item.find_all_templates_by_links()
            raw_dfe_list = self._get_data_dfe_list(item)
            for raw_dfe in raw_dfe_list:
                if not raw_dfe.startswith('_') or raw_dfe in NOT_TO_LOOK:
                    continue
                
                tpl_source = []
                for use in tpls_tmp:
                    if raw_dfe in use:
                        tpl_source.append(use.get_name())
                        for tpl in template_list:
                            if tpl.get_uuid() == use[u'_id']:
                                tpl.add_data(raw_dfe)
                                break
                        else:
                            tpl = MassiveChangeItemWithData(use.get_name(), use[u'_id'])
                            tpl.add_data(raw_dfe)
                            template_list.append(tpl)
                if not tpl_source:
                    template_list[1].add_data(raw_dfe)
                template_list[0].add_data(raw_dfe)
                dfe = None
                for tmp_dfe in dfe_list:
                    if tmp_dfe.get_name() == raw_dfe:
                        dfe = tmp_dfe
                        break
                if dfe:
                    if tpl_source:
                        dfe.add_sources(tpl_source)
                    dfe.add_item(item['_id'])
                else:
                    dfe_list.append(MassiveChangeData(raw_dfe, item[u'_id'], tpl_source))
                tmp_host.add_data(raw_dfe)
            host_list.append(tmp_host)
        
        return dfe_list, ignored_items, missing_items, host_list, template_list
    
    
    # -------------- DFE --------------#
    @staticmethod
    def _get_data_dfe_list(item):
        # type: (BaseItem) -> List
        dfe_list = set()
        
        checks_on_item = METADATA.get_metadata(item, METADATA.CHECKS)
        
        if not checks_on_item:
            return []
        
        for value in checks_on_item:
            
            _check = value['check']
            if not _check.is_enabled():
                continue
            
            _dfe = _check.get('duplicate_foreach', None)
            
            if _dfe:
                dfe_list.add(_dfe)
        
        return list(dfe_list)
