#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2019:
# This file is part of Shinken Enterprise, all rights reserved.
from shinken.misc.type_hint import TYPE_CHECKING, NoReturn
from shinkensolutions.toolbox.box_tools_string import ToolsBoxString
from .massive_change_helper import parse_form_links_to_html, LINKED_PROPERTIES_FLAT_WHEN_CHANGED, parse_command_link_to_html
from .massive_change_operator import MASSIVE_CHANGE_OPERATOR
from .massive_change_override_element import MassiveChangeOverrideElement
from .massive_change_returns import MASSIVE_CHANGE_RETURNS
from .massive_change_status import MASSIVE_CHANGE_STATUS
from ....dao.def_items import can_property_contain_links, LINKIFY_MANAGE_STATES, STOP_INHERITANCE_VALUES, DEF_ITEMS, ITEM_TYPE
from ....dao.helpers import parse_default_value_to_html, BOOLEAN_PROPERTIES

if TYPE_CHECKING:
    from .massive_change_action import MassiveChangeAction


class MassiveChangeField(object):
    
    def __init__(self, name, item_type, item_id, category, is_protected, action):
        # type: (str, str, str, str, bool, MassiveChangeAction) -> NoReturn
        self.name = name
        self.item_type = item_type
        self.item_id = item_id
        self.category = category
        
        # Previous_value
        self._previous_value = None
        self._previous_filter_value = ''
        self._html_previous_value = ''
        
        # Actual_value
        self._actual_value = None
        self._actual_filter_value = ''
        self._html_actual_value = ''
        
        # Miscellaneous
        self._modified = MASSIVE_CHANGE_STATUS.UNMODIFIED
        self.message = MASSIVE_CHANGE_RETURNS.OK
        self.missing_values = []
        self.is_protected = is_protected
        self.action = action
        if self.name in LINKED_PROPERTIES_FLAT_WHEN_CHANGED[item_type]:
            self.is_linked = False
        else:
            self.is_linked = can_property_contain_links(item_type, name)
    
    
    def is_modified(self):
        return self._modified
    
    
    def set_previous_value(self, massive_change_element, app, datamanagerV2):
        value = massive_change_element.raw_item.get(self.name, None)
        self._previous_value = value
        if value in STOP_INHERITANCE_VALUES:
            self._html_previous_value = parse_default_value_to_html(massive_change_element.item, self.name, app, is_forced=value)
        elif value is None:
            if self.name == 'plus':
                self._html_previous_value = ''
            elif isinstance(massive_change_element, MassiveChangeOverrideElement):
                self._html_previous_value = '<span class="shinken-default-value">%s</span>' % app._('element.in_check')
            else:
                self._html_previous_value = '<span class="shinken-default-value">%s</span>' % parse_default_value_to_html(massive_change_element.item, self.name, app)
        elif self.is_protected:
            self._html_previous_value = app._('element.password_protected')
        elif self.name in LINKED_PROPERTIES_FLAT_WHEN_CHANGED[self.item_type]:
            flattened_value = datamanagerV2.flatten_value(value, self.item_type, self.name, LINKIFY_MANAGE_STATES)
            self._html_previous_value = ToolsBoxString.escape_XSS(flattened_value)
        elif self.is_linked:
            if self.name in DEF_ITEMS[self.item_type]['item_links'].get(ITEM_TYPE.COMMANDS, []) and value and self.item_type != ITEM_TYPE.NOTIFICATIONWAYS:
                self._html_previous_value = parse_command_link_to_html(self._previous_value, self.name, self.item_type, app, datamanagerV2)
            elif value and 'links' in value:
                self._html_previous_value = parse_form_links_to_html(self._previous_value['links'], self.name, massive_change_element.item, app, datamanagerV2)
        elif self.name == 'business_impact' or self.name == 'min_business_impact':
            self._html_previous_value = app._('element.business_impact_%s' % value)
        elif BOOLEAN_PROPERTIES.is_boolean(self.item_type, self.name):
            self._html_previous_value = app._('common.value.boolean.%s' % value)
        elif value:
            self._html_previous_value = ToolsBoxString.escape_XSS(value)
    
    
    def set_actual_value(self, massive_change_element, modified, app, datamanagerV2):
        value = massive_change_element.raw_item.get(self.name, None)
        self._actual_value = value
        self._modified = modified
        if value in STOP_INHERITANCE_VALUES:
            self._html_actual_value = parse_default_value_to_html(massive_change_element.item, self.name, app, is_forced=value)
        elif value is None:
            if self.name == 'plus':
                self._html_actual_value = ''
            elif isinstance(massive_change_element, MassiveChangeOverrideElement):
                self._html_actual_value = '<span class="shinken-default-value">%s</span>' % app._('element.in_check')
            else:
                self._html_actual_value = '<span class="shinken-default-value">%s</span>' % parse_default_value_to_html(massive_change_element.item, self.name, app)
        elif self.is_protected and self.action.action in (MASSIVE_CHANGE_OPERATOR.SET, MASSIVE_CHANGE_OPERATOR.SET_IF_DATA_EXISTS_ON_ITEM, MASSIVE_CHANGE_OPERATOR.SET_IF_DATA_IS_SET_ON_ITEM):
            self._html_actual_value = app._('element.password_protected')
        elif self.is_linked:
            if self.name in DEF_ITEMS[self.item_type]['item_links'].get(ITEM_TYPE.COMMANDS, []) and value and self.item_type != ITEM_TYPE.NOTIFICATIONWAYS:
                self._html_actual_value = parse_command_link_to_html(self._actual_value, self.name, self.item_type, app, datamanagerV2)
            elif value and 'links' in value:
                self._html_actual_value = parse_form_links_to_html(self._actual_value['links'], self.name, massive_change_element.item, app, datamanagerV2)
        elif self.name == 'business_impact' or self.name == 'min_business_impact':
            self._html_actual_value = app._('element.business_impact_%s' % value)
        elif self.name == 'duplicate_foreach' and value and value not in STOP_INHERITANCE_VALUES:
            self._html_actual_value = value.upper().lstrip('_')
        elif BOOLEAN_PROPERTIES.is_boolean(self.item_type, self.name):
            self._html_actual_value = app._('common.value.boolean.%s' % value)
        else:
            self._html_actual_value = ToolsBoxString.escape_XSS(value)
        
        if self.missing_values:
            self.missing_values = parse_form_links_to_html(self.missing_values, self.name, massive_change_element.item, app, datamanagerV2)
    
    
    def set_error(self, error):
        self.message = error
    
    
    def get_previous_filter_value(self):
        return self._previous_filter_value
    
    
    def get_actual_filter_value(self):
        return self._actual_filter_value
    
    
    def get_uuid(self):
        return '%s-%s' % (self.item_id, self.name)
    
    
    def get_previous_value_in_html(self):
        return self._html_previous_value
    
    
    def get_actual_value_in_html(self):
        return self._html_actual_value
    
    
    def get_message_to_html(self, app, step, in_tooltip=False):
        
        if in_tooltip:
            return app._('mass-change.step-%s.messages.tooltip.%s' % (step, self.message))
        
        trad = app._('mass-change.step-%s.messages.%s' % (step, self.message))
        if self.message in (MASSIVE_CHANGE_RETURNS.MISSING_TARGET, MASSIVE_CHANGE_RETURNS.NO_VALUE_ON_ITEM):
            trad = trad.format(self.missing_values)
        
        return '<div class="shinken-message">%s</div>' % trad
    
    
    def get_state_in_html(self):
        icon = 'shinicon-check-circle' if self.message in (MASSIVE_CHANGE_RETURNS.OK, MASSIVE_CHANGE_RETURNS.SYNCHRO_USELESS) else 'shinkon-times-circle'
        return '<div><span class="%s"><span></div>' % icon
