#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import copy
import uuid
from threading import RLock

from shinken.log import logger, LoggerFactory
from shinken.misc.fast_copy import fast_deepcopy
from shinken.misc.type_hint import TYPE_CHECKING
from . import exceptions
from .edit_item_context import ACTIONS
from .exceptions import BusinessException
from .exceptions import UnauthorizedSaveException, ItemNotFoundException, ItemNotValid
from .shinken_return_code import ShinkenReturnCode
from .work_area_helper import (set_proposed_suppressed,
                               set_proposed_modified,
                               set_proposed_created,
                               set_validated,
                               set_validated_modified,
                               set_rejected,
                               set_editing_suppressed,
                               init_work_area,
                               create_comment,
                               get_last_action,
                               clean_diff,
                               clean_work_area_users,
                               clean_last_action,
                               WORK_AREA_INFO_KEY,
                               set_in_working_on_work_area_save, set_editing_modified)
from ..arbiter_controler import ARBITER_AREA
from ..item_controller import work_area_helper
from ..sync_ui_common import syncuicommon
from ...dao.crypto import TAG_FROM_CHANGE
from ...dao.datamanagerV2 import FALLBACK_USER
from ...dao.def_items import ITEM_STATE, ITEM_TYPE, WORKING_AREA_LAST_ACTION, HISTORY_ACTION, WORKING_AREA_STATUS, DEF_ITEMS, METADATA, SERVICE_EXCLUDES_BY_ID
from ...dao.helpers import compute_diff, get_name_from_type
from ...dao.parsers.complex_exp_parser import OPERAND, visit_node
from ...dao.transactions.transactions import DBTransaction
from ...dao.validators.validator import Validator

if TYPE_CHECKING:
    from ...dao.datamanagerV2 import DataManagerV2
    from ...synchronizerdaemon import Synchronizer
    from ...dao.items.contactitem import ContactItem
    from ...dao.items.baseitem import BaseItem
    from .edit_item_context import EditItemContext
    from ...dao.def_items import ItemType
    from shinken.misc.type_hint import Dict, Optional, Callable, Tuple, List, Union
    from shinken_mock.mock_app import MockSynchronizer

PREPROD_LOCK = RLock()
CLONE_LOCK = RLock()
AUTHORIZED_TYPES_FOR_SI_ADMIN = [ITEM_TYPE.HOSTS, ITEM_TYPE.CONTACTS]
USER_PROPERTY_LOCK_FOR_SI_ADMIN = set([
    u'can_submit_commands',
    u'contactgroups',
    u'is_admin',
    u'expert',
    u'enabled',
    u'use',
    u'acl_make_downtime',
    u'acl_make_acknowledge',
    u'acl_force_result_check',
    u'acl_force_retry_check',
    u'acl_in_tab_history',
    u'acl_show_history_range',
    u'acl_show_sla_range',
    u'acl_try_check_on_poller',
    u'acl_try_check_on_synchronizer',
    u'acl_share_everybody',
    u'acl_share_group',
    u'acl_share_private'
])
COMMENT_KEY = '__COMMENT__'


class StateController(object):
    def __init__(self, app):
        # type: (Union[Synchronizer, MockSynchronizer]) -> None
        
        self.app = app
        self.datamanagerV2 = app.datamanagerV2  # type: DataManagerV2
    
    
    def _get_user(self):
        # type: () -> ContactItem
        try:
            user = self.app.get_user_auth()
        except Exception:
            user = FALLBACK_USER
        return user
    
    
    # check acl for item or prev_elt in context with user and to state in context
    def check_acl(self, context, item=None, check_edit_rights=True):
        # type: (EditItemContext, Optional[BaseItem], bool) -> None
        # SEF-9314, if we don't want to check the acl, we can put the "bypass_check_acl" to True. Useful for update work area info in staging
        if context.bypass_check_acl:
            return
        
        item = item if item else context.prev_elt
        
        # check if user can edit the item see : SEF-9314
        if item:
            result = syncuicommon.check_acl_from_context(item, context)
            can_edit = result[u'can_edit']
            can_view = result[u'can_view']
            if not can_edit and check_edit_rights:
                raise UnauthorizedSaveException(code=403, text=self.app.t(u'element.no_edit'))
            elif not check_edit_rights and not can_view:
                raise UnauthorizedSaveException(code=403, text=self.app.t(u'element.no_view'))
    
    
    def check_is_admin(self, context):
        if not context.user.is_admin():
            raise UnauthorizedSaveException(code=403, text=self.app.t(u'element.no_edit'))
    
    
    #   ____ ____  _   _ ____
    #  / ___|  _ \| | | |  _ \
    # | |   | |_) | | | | | | |
    # | |___|  _ <| |_| | |_| |
    #  \____|_| \_\\___/|____/
    
    def update(self, context, item):
        # type: (EditItemContext, Dict) -> Dict
        # A non-admin user cannot create a new user, or change some properties
        previously_deleted = False
        previously_imported = False
        if not context.in_creation and not context.prev_elt and not context.is_new:
            previously_deleted = True
            if context.item_type in ITEM_TYPE.ALL_DEDICATED_SERVICES:
                item_in_new = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, ITEM_STATE.NEW)
            else:
                item_in_new = self.datamanagerV2.find_item_by_name(get_name_from_type(context.item_type, item), context.item_type, ITEM_STATE.NEW)
            if item_in_new:
                _link_on_new = '<a class="shinken-link shinken-space" href="/elements/%s/%s?new=1" target="_blank" ><span class="shinken-tag-label">%s</span></a>' % (context.item_type, context.item_id, item_in_new.get_name())
                raise BusinessException(code=512, text=self.app.t(u'element.already_deleted_and_reimport') % _link_on_new)
        
        if not context.in_creation and context.prev_elt and context.prev_elt.get_state() != ITEM_STATE.NEW and context.is_new:
            previously_imported = True
        
        if context.item_type == ITEM_TYPE.CONTACTS and not context.is_admin:
            if not context.prev_elt:
                raise UnauthorizedSaveException()
            raw_prev_item = context.prev_elt.get_raw_item()
            for prop_name in USER_PROPERTY_LOCK_FOR_SI_ADMIN:
                if prop_name in raw_prev_item:
                    item[prop_name] = raw_prev_item[prop_name]
                elif prop_name in item:
                    del item[prop_name]
        
        self.check_acl(context)
        
        action = HISTORY_ACTION.ELEMENT_MODIFICATION
        if context.with_change:
            action = HISTORY_ACTION.APPLY_CHANGES
        if context.action == HISTORY_ACTION.UPDATE_WORK_AREA_INFO:
            action = HISTORY_ACTION.UPDATE_WORK_AREA_INFO
        
        if not context.prev_elt:
            action = HISTORY_ACTION.ELEMENT_ADD
        elif METADATA.get_metadata(context.prev_elt, METADATA.STATE) == ITEM_STATE.NEW:
            action = HISTORY_ACTION.IMPORT
        elif context.action == ACTIONS.SAVE_IN_WORK_AREA:
            action = HISTORY_ACTION.PUT_IN_WORK_AREA
        
        # SPECIFIC processing for Protected fields
        # - In case of change application, protected fields can have 3 values :
        #    - protected-login-and-password : Standard tag value meaning the value was not modified by
        #                                     the user and no change should be applied Normally already
        #                                     processed by _format_form_to_item()
        #    - TAG_FROM_CHANGE (protected_field_value_from_changes) : Specific tag meaning changes should be applied
        #    - other values : the user updated the value manually, and it should be used
        
        if context.with_change:
            changes = None
            if context.prev_elt:
                changes = METADATA.get_metadata(context.prev_elt, METADATA.CHANGES)
            if not changes:
                # TODO ITEM_STATE.CHANGES not handle by datamanagerV2 so this code is not working
                changes = self.datamanagerV2.find_item_by_id(item['_id'], item_state=ITEM_STATE.CHANGES, item_type=context.item_type)
                
                if not changes and not previously_deleted:
                    raise BusinessException(code=202, text=self.app.t(u'element.exception_changes_already_applied'))
                elif not previously_deleted:
                    changes = changes[u'changes']
            
            if not previously_deleted:
                for prop, value in item.iteritems():
                    match = self.app.frontend_cipher.match_protected_property(prop, context.item_type, context.user) and value == TAG_FROM_CHANGE
                    if match:
                        item[prop] = changes[prop][1]  # 1 => new value ; 0 => old value
            
            # We also update _SYNC_KEYS from source item see SEF-4777
            item_merged = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, ITEM_STATE.MERGE_SOURCES)
            if item_merged:
                item[u'_SYNC_KEYS'] = fast_deepcopy(item_merged[u'_SYNC_KEYS'])
        
        if not context.is_admin and context.is_expert and (context.item_type not in AUTHORIZED_TYPES_FOR_SI_ADMIN):
            raise UnauthorizedSaveException()
        
        # Bypass of working area means removing item from WA and setting its status to validated before saving it in Staging
        if context.bypass_work_area and context.is_admin:
            if WORK_AREA_INFO_KEY not in item:
                item[WORK_AREA_INFO_KEY] = init_work_area(item, context.user)
            set_validated(context.user, item[WORK_AREA_INFO_KEY])
            
            if self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, ITEM_STATE.WORKING_AREA):
                self.datamanagerV2.save_item(item, context.user, context.item_type, item_state=ITEM_STATE.STAGGING, action=action)
                self.datamanagerV2.delete_item(item, context.user, context.item_type, item_state=ITEM_STATE.WORKING_AREA, action=HISTORY_ACTION.VALIDATED_FROM_WORK_AREA)
            else:
                self.datamanagerV2.save_item(item, context.user, context.item_type, item_state=ITEM_STATE.STAGGING, action=action)
        elif action != HISTORY_ACTION.UPDATE_WORK_AREA_INFO and ITEM_TYPE.has_work_area(context.item_type) and context.to_state == ITEM_STATE.WORKING_AREA:
            item_in_staging = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, item_state=ITEM_STATE.STAGGING)
            work_area_info = None
            if item_in_staging:
                work_area_info = item_in_staging.get(WORK_AREA_INFO_KEY, None)
            if not work_area_info:
                work_area_info = item.get(WORK_AREA_INFO_KEY, None)
            if not work_area_info:
                work_area_info = init_work_area(item, context.user)
                
                if item_in_staging:
                    set_in_working_on_work_area_save(item_in_staging, context.item_type, context.prev_elt, item, context.user)
                    work_area_info = item.get(WORK_AREA_INFO_KEY, None)
            
            if get_last_action(work_area_info) == WORKING_AREA_LAST_ACTION.CREATED:
                set_editing_modified({}, context.user, work_area_info, WORKING_AREA_LAST_ACTION.CREATED)
            else:
                set_editing_modified({}, context.user, work_area_info, WORKING_AREA_LAST_ACTION.MODIFIED)
            item[WORK_AREA_INFO_KEY] = work_area_info
            if item_in_staging:
                item_in_staging.set_value(WORK_AREA_INFO_KEY, work_area_info)
                self.datamanagerV2.save_item(item_in_staging, context.user, context.item_type, ITEM_STATE.STAGGING, action=HISTORY_ACTION.UPDATE_WORK_AREA_INFO)
            
            self.datamanagerV2.save_item(item, context.user, context.item_type, context.to_state, action=action)
        else:
            self.datamanagerV2.save_item(item, context.user, context.item_type, context.to_state, action=action)
        
        if context.with_change:
            self.datamanagerV2.delete_item(item, item_state=ITEM_STATE.CHANGES, item_type=context.item_type)
        
        if context.is_new:
            self.datamanagerV2.delete_item(item, context.user, context.item_type, item_state=ITEM_STATE.NEW, old_item=None, action=action)
        
        return {u'previously_deleted': previously_deleted, u'previously_imported': previously_imported}
    
    
    def delete(self, context, item_state, item=None, action=None):
        # type: (EditItemContext, unicode, Optional[BaseItem], Optional[unicode]) -> None
        
        item_in_staging = None
        if context.item_id and not item:
            item = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, item_state)
        if not item:
            raise ItemNotFoundException(code=u'element-unknown', text=self.app.t(u'element.exception_item_to_delete_not_found'), shinken_return_code=ShinkenReturnCode.DELETE_UNKNOWN)
        if item.get(u'presence_protection', u'0') == u'1':  # if the element is protected, do not allow to delete it
            raise UnauthorizedSaveException(code=403, text=self.app.t(u'element.exception_deletion_not_authorized'), shinken_return_code=ShinkenReturnCode.DELETE_PROTECTED)
        # A user cannot delete himself
        if context.item_type == ITEM_TYPE.CONTACTS and context.user[u'_id'] == item[u'_id']:
            raise UnauthorizedSaveException(code=406, text=self.app.t(u'element.exception_suicide_not_authorized'), shinken_return_code=ShinkenReturnCode.DELETE_OWN_USER)
        
        self.check_acl(context, item)
        
        # for item in working area, set the items (work area and staging) as editing suppressed, unless we bypass the working area
        if item_state == ITEM_STATE.WORKING_AREA:
            item_in_staging = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, ITEM_STATE.STAGGING)
            item_in_work_area = item
            if item_in_staging:
                if context.bypass_work_area:
                    self.datamanagerV2.delete_item(item_in_work_area, context.user, context.item_type, item_state=ITEM_STATE.WORKING_AREA, old_item=item, action=action)
                    self.datamanagerV2.delete_item(item_in_staging, context.user, context.item_type, item_state=ITEM_STATE.STAGGING, old_item=item, action=action)
                else:
                    work_area_info = item_in_work_area[WORK_AREA_INFO_KEY]
                    set_editing_suppressed({}, context.user, work_area_info)
                    item_in_staging.set_value(WORK_AREA_INFO_KEY, work_area_info)
                    if hasattr(item_in_work_area, u'set_value'):
                        item_in_work_area.set_value(WORK_AREA_INFO_KEY, work_area_info)
                    else:
                        item_in_work_area[WORK_AREA_INFO_KEY] = work_area_info
                    
                    self.datamanagerV2.save_item(item_in_staging, context.user, context.item_type, ITEM_STATE.STAGGING, action=action)
                    self.datamanagerV2.save_item(item_in_work_area, context.user, context.item_type, ITEM_STATE.WORKING_AREA, action=action)
        
        if not item_in_staging:
            self.datamanagerV2.delete_item(item, context.user, context.item_type, item_state, old_item=item, action=action)
    
    
    # save item in work area and update work area info in staging if exists
    def _put_in_work_area(self, context, item, item_in_staging=None):
        # type: (EditItemContext, BaseItem, Optional[BaseItem]) -> None
        
        self.update(context, item)
        if item_in_staging:
            item_in_staging.set_value(WORK_AREA_INFO_KEY, item[WORK_AREA_INFO_KEY])
            # we need to update the work_area_info of the staging object
            # compute a new context with a new action and to state
            self.update(context.update_work_area(), item_in_staging)
    
    
    # new version of get_in_work_area
    def get_in_work_area(self, context, item=None):
        # type: (EditItemContext, Optional[BaseItem]) -> BaseItem
        
        if not item:
            item = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, ITEM_STATE.STAGGING)
        
        if item:
            return self.make_work_area_item_from_staging_item(context, item)
        else:
            logger.warning(u'[work_area] item [%s-%s] not found' % (context.item_type, context.item_id))
    
    
    # new version of _to_work_area_item
    def make_work_area_item_from_staging_item(self, context, item=None):
        # type: (EditItemContext, Optional[BaseItem]) -> BaseItem
        
        self.check_acl(context, item)
        
        #  extract from _to_work_area_item
        work_area_info = init_work_area(item, context.user)
        work_area_info[u'last_action'] = WORKING_AREA_LAST_ACTION.MODIFIED
        item.set_value(WORK_AREA_INFO_KEY, work_area_info)
        self.datamanagerV2.save_item(item, context.user, context.item_type, item_state=ITEM_STATE.STAGGING, action=HISTORY_ACTION.PUT_IN_WORK_AREA)
        return item
    
    
    def submit_to_staging(self, context, item_in_working_area=None, comment=None):
        # type: (EditItemContext, Optional[BaseItem], Optional[unicode]) -> None
        
        # If item only in working area, it's a creation
        # If item in both areas, it's a modification
        # If parameter item_in_working_area is None, it means this function was called for a mass action
        if not item_in_working_area:
            item_in_working_area = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, ITEM_STATE.WORKING_AREA)
            if not item_in_working_area:
                raise ItemNotFoundException(code=u'element-unknown', text=self.app.t(u'element.outside_removed_from_work_area'))
        item_in_staging = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, ITEM_STATE.STAGGING)
        raw_item_in_staging = self.datamanagerV2.get_raw_item(item_in_staging)
        raw_item_in_working_area = self.datamanagerV2.get_raw_item(item_in_working_area, item_type=context.item_type)
        
        self.check_acl(context)
        
        # if no comment, try to find one
        if not comment:
            forms = self.app.request.forms
            comment_text = forms.get(u'comment', u'').decode(u'utf8', u'ignore').strip()
            comment = {}
            if comment_text:
                comment = create_comment(comment_text, context.user, WORKING_AREA_STATUS.PROPOSED)
        
        work_area_info = raw_item_in_working_area[WORK_AREA_INFO_KEY]
        last_action = get_last_action(work_area_info)
        work_area_info[u'last_action'] = last_action
        
        if last_action == WORKING_AREA_LAST_ACTION.CREATED:
            set_proposed_created(comment, context.user, work_area_info)
            raw_item_in_working_area[WORK_AREA_INFO_KEY] = work_area_info
            self.datamanagerV2.save_item(raw_item_in_working_area, context.user, context.item_type, ITEM_STATE.WORKING_AREA, action=HISTORY_ACTION.SUBMIT_TO_STAGGING)
            logger.info(u'[work_area] submit_to_staging a new item [%s-%s].' % (context.item_type, context.item_id))
        elif last_action == WORKING_AREA_LAST_ACTION.SUPPRESSED:
            set_proposed_suppressed(comment, context.user, work_area_info)
            # We save the updated object in both areas
            if raw_item_in_staging:
                # set the work area info here after modification to be sure that the raw item will be the same
                raw_item_in_staging[WORK_AREA_INFO_KEY] = work_area_info
                raw_item_in_working_area[WORK_AREA_INFO_KEY] = work_area_info
                self.datamanagerV2.save_item(raw_item_in_staging, context.user, context.item_type, ITEM_STATE.STAGGING, action=HISTORY_ACTION.SUBMIT_TO_STAGGING)
                self.datamanagerV2.save_item(raw_item_in_working_area, context.user, context.item_type, ITEM_STATE.WORKING_AREA, action=HISTORY_ACTION.SUBMIT_TO_STAGGING)
            logger.info(u'[work_area] submit_to_staging remove item [%s-%s - %r].' % (context.item_type, context.item_id, work_area_info))
        else:
            diff = []
            for k in compute_diff(raw_item_in_staging, raw_item_in_working_area, context.item_type):
                
                if k == SERVICE_EXCLUDES_BY_ID:
                    work_area_names = []  # type: List[unicode]
                    for link in item_in_working_area.get(k, {}).get(u'links', []):
                        service_type = link.get(u'item_type', u'')
                        service_uuid = link.get(u'_id')
                        service = self.datamanagerV2.find_item_by_id(service_uuid, service_type, ITEM_STATE.STAGGING)
                        if service is None:
                            continue
                        work_area_names.append(service.get_name())
                    
                    workarea_services = u', '.join(work_area_names) if work_area_names else u''
                    staging_services = item_in_staging.get_service_excludes_by_id()
                    
                    diff.append({u'prop': k, u'stagging': staging_services, u'new': workarea_services})
                else:
                    diff.append({u'prop': k, u'stagging': raw_item_in_staging.get(k, u''), u'new': raw_item_in_working_area.get(k, u'')})
            
            work_area_info[u'diff_item'] = diff
            
            if diff:
                set_proposed_modified(comment, context.user, work_area_info)
                # set the work area info here after modification to be sure that the raw item will be the same
                raw_item_in_staging[WORK_AREA_INFO_KEY] = copy.deepcopy(work_area_info)
                raw_item_in_working_area[WORK_AREA_INFO_KEY] = copy.deepcopy(work_area_info)
                self.datamanagerV2.save_item(raw_item_in_working_area, context.user, context.item_type, ITEM_STATE.WORKING_AREA, action=HISTORY_ACTION.SUBMIT_TO_STAGGING)
                self.datamanagerV2.save_item(raw_item_in_staging, context.user, context.item_type, ITEM_STATE.STAGGING, action=HISTORY_ACTION.SUBMIT_TO_STAGGING)
            else:
                logger.info(u'[work_area] submit_to_staging item [%s-%s] with no change it will be validate.' % (context.item_type, context.item_id))
                set_validated(context.user, work_area_info)
                # set the work area info here after modification to be sure that the raw item will be the same
                raw_item_in_staging[WORK_AREA_INFO_KEY] = copy.deepcopy(work_area_info)
                raw_item_in_working_area[WORK_AREA_INFO_KEY] = copy.deepcopy(work_area_info)
                self.datamanagerV2.save_item(raw_item_in_staging, context.user, context.item_type, ITEM_STATE.STAGGING, action=HISTORY_ACTION.DISCARD_MODIFICATION_FROM_WORK_AREA)
                self.datamanagerV2.delete_item(raw_item_in_working_area, context.user, context.item_type, ITEM_STATE.WORKING_AREA, action=HISTORY_ACTION.DISCARD_MODIFICATION_FROM_WORK_AREA)
        
        logger.debug(u'[work_area] submit_to_staging item [%s-%s]' % (context.item_type, context.item_id))
    
    
    def accept_submit(self, context):
        # type: (EditItemContext) -> None
        
        self.check_is_admin(context)
        
        item_in_working_area = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, ITEM_STATE.WORKING_AREA)
        if item_in_working_area is None:
            raise ItemNotFoundException(code=u'element-unknown', text=self.app.t(u'element.outside_removed_from_work_area'))
        
        raw_item_in_working_area = self.datamanagerV2.get_raw_item(item_in_working_area, item_type=context.item_type, flatten_links=None)
        work_area_info = raw_item_in_working_area[WORK_AREA_INFO_KEY]
        last_action = get_last_action(work_area_info)
        
        if last_action == WORKING_AREA_LAST_ACTION.SUPPRESSED:
            item_in_staging = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, ITEM_STATE.STAGGING)
            if item_in_staging:
                work_area_info = item_in_staging[WORK_AREA_INFO_KEY]
                set_validated(context.user, work_area_info)
                clean_diff(work_area_info)
                clean_last_action(work_area_info)
                # set the work area info here after modification to be sure that the raw item will be the same
                item_in_staging.set_value(WORK_AREA_INFO_KEY, work_area_info)
                # delete the item in staging
                self.delete(context, ITEM_STATE.STAGGING, item=item_in_staging, action=HISTORY_ACTION.ELEMENT_DELETE)
            self.delete(context, ITEM_STATE.WORKING_AREA, item=item_in_working_area, action=HISTORY_ACTION.ELEMENT_DELETE)
        else:
            if last_action == WORKING_AREA_LAST_ACTION.CREATED:
                set_validated(context.user, work_area_info)
            else:
                set_validated_modified(context.user, work_area_info)
            clean_diff(work_area_info)
            clean_work_area_users(work_area_info)
            self.datamanagerV2.save_item(raw_item_in_working_area, item_state=ITEM_STATE.STAGGING, item_type=context.item_type, user=context.user, action=HISTORY_ACTION.VALIDATED_FROM_WORK_AREA)
            self.datamanagerV2.delete_item(raw_item_in_working_area, item_state=ITEM_STATE.WORKING_AREA, item_type=context.item_type, user=context.user, action=HISTORY_ACTION.VALIDATED_FROM_WORK_AREA)
    
    
    def reject_submit(self, context, comment):
        # type: (EditItemContext, unicode) -> None
        
        self.check_is_admin(context)
        
        item_in_working_area = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, ITEM_STATE.WORKING_AREA)
        item_in_staging = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, ITEM_STATE.STAGGING)
        
        if item_in_working_area is None:
            raise ItemNotFoundException(code=u'element-unknown', text=self.app.t(u'element.outside_removed_from_work_area'))
        
        work_area_info = item_in_working_area[WORK_AREA_INFO_KEY] if item_in_working_area else item_in_staging[WORK_AREA_INFO_KEY]
        reject_comment = {}
        
        if comment:
            reject_comment = create_comment(comment, context.user, WORKING_AREA_STATUS.REJECTED)
        
        set_rejected(reject_comment, context.user, work_area_info)
        clean_diff(work_area_info)
        
        staging_work_area_info = copy.deepcopy(work_area_info)
        if item_in_working_area:
            item_in_working_area.set_value(WORK_AREA_INFO_KEY, work_area_info)
            self.datamanagerV2.save_item(item_in_working_area, context.user, context.item_type, ITEM_STATE.WORKING_AREA, action=HISTORY_ACTION.REJECT_FROM_WORK_AREA)
        
        if item_in_staging:
            item_in_staging.set_value(WORK_AREA_INFO_KEY, staging_work_area_info)
            self.datamanagerV2.save_item(item_in_staging, context.user, context.item_type, ITEM_STATE.STAGGING, action=HISTORY_ACTION.REJECT_FROM_WORK_AREA)
    
    
    def clone_object(self, context, transform=None):
        # type: (EditItemContext, Optional[Callable]) -> None
        
        with CLONE_LOCK:
            _new_object = self._do_clone_object(context)
            if context.to_state == ITEM_STATE.WORKING_AREA:
                work_area_info = work_area_helper.init_work_area(_new_object, context.user)
                _new_object[WORK_AREA_INFO_KEY] = work_area_info
            # if we cloned an item coming from work_area to staging (bypass), set the item in validates status
            elif context.from_state == ITEM_STATE.WORKING_AREA and context.to_state == ITEM_STATE.STAGGING:
                if WORK_AREA_INFO_KEY not in _new_object:
                    work_area_helper.reset_work_area(_new_object)
                else:
                    work_area_helper.set_validated(context.user, _new_object[WORK_AREA_INFO_KEY])
            if transform:
                transform(_new_object, context)
            self.datamanagerV2.save_item(_new_object, context.user, item_type=context.item_type, item_state=context.to_state, action=HISTORY_ACTION.ELEMENT_CLONED)
    
    
    @staticmethod
    def _update_host_name_node_for_cloned_item(original_item_id, clone_item_name, node, do_update):
        # type: (unicode, unicode, Dict, Dict) -> None
        
        if node[u'operand'] != OPERAND.TEMPLATE_OP:
            return
        
        link = node[u'content']
        if link[u'exists'] and original_item_id == link[u'_id']:
            node[u'operand'] = OPERAND.OR_OP
            node_for_original_link = {
                u'content'  : node[u'content'],
                u'operand'  : OPERAND.TEMPLATE_OP,
                u'not_value': False,
                u'sons'     : []
            }
            node_for_clone_link = {
                u'content'  : {
                    u'name'  : clone_item_name,
                    u'exists': False
                },
                u'operand'  : OPERAND.TEMPLATE_OP,
                u'not_value': False,
                u'sons'     : []
            }
            node[u'sons'] = [node_for_original_link, node_for_clone_link]
            node[u'content'] = None
            do_update[u'done'] = True
    
    
    def _attach_check_on_host_clone(self, context, original_item, new_name):
        # type: (EditItemContext, BaseItem, unicode) -> None
        do_update = {u'done': False}
        
        
        def node_updater_for_cloned_item(node, _level):
            StateController._update_host_name_node_for_cloned_item(original_item[u'_id'], new_name, node, do_update)
        
        
        host_check_type = DEF_ITEMS[context.item_type][u'check_type']
        for check_md in METADATA.get_metadata(original_item, METADATA.CHECKS, []):
            check = check_md[u'check']
            check_state = check.get_state()
            check_type = check.get_type()
            # We update check with datamanagerV2.save_item, so we must update a copy of our check and not our checks directly
            check = check.get_raw_item(flatten_links=False)
            
            do_update[u'done'] = False
            if host_check_type == check_type and context.item_type in (ITEM_TYPE.HOSTS, ITEM_TYPE.CLUSTERS):
                default_host_name = {
                    u'has_plus': False,
                    u'links'   : []
                }
                link = {
                    u'exists': False,
                    u'name'  : new_name,
                }
                host_name = check.get(u'host_name', default_host_name)
                host_name[u'links'].append(link)
                
                do_update[u'done'] = True
            elif host_check_type == check_type and context.item_type in (ITEM_TYPE.HOSTTPLS, ITEM_TYPE.CLUSTERTPLS):
                host_name_root_node = check.get(u'host_name', {}).get(u'node', {})
                if host_name_root_node:
                    visit_node(host_name_root_node, node_updater_for_cloned_item)
            
            if do_update[u'done']:
                self.datamanagerV2.save_item(check, item_type=check_type, item_state=check_state, action=HISTORY_ACTION.AUTO_MODIFICATION)
    
    
    def _do_clone_object(self, context):
        # type: (EditItemContext) -> Dict
        
        item_type = context.item_type
        original_item = self.datamanagerV2.find_item_by_id(context.item_id, item_type, context.from_state)
        if original_item is None:
            raise exceptions.ItemNotFoundException(code=u'element-unknown', text=self.app.t(u'element.outside_deleted'))
        
        self.check_acl(context, original_item, check_edit_rights=False)
        
        clone_item = original_item.get_raw_item(flatten_links=False)
        new_id = uuid.uuid1().hex
        
        # try to find a new name, that is not already used :)
        name_key = DEF_ITEMS[item_type][u'key_name']
        # As usual, services are a bit special, because can be with a description
        if item_type == ITEM_TYPE.SERVICETPLS and clone_item.get(u'service_description', u'') != u'':
            name_key = u'service_description'
        
        # ok now get the current name
        old_name = original_item.get_name()
        # Loop while we can't find a name that is not already used
        new_name = ''
        founded = False
        idx = 0
        while not founded:
            if idx == 0:
                new_name = u'%s [%s]' % (old_name, self.app.t(u'element.copy'))
            else:
                new_name = u'%s [%s %d]' % (old_name, self.app.t(u'element.copy'), idx)
            item = self.datamanagerV2.find_item_by_name(new_name, item_type, ITEM_STATE.STAGGING)
            if item is None and item_type == ITEM_TYPE.HOSTS:
                item = self.datamanagerV2.find_item_by_name(new_name, item_type, ITEM_STATE.WORKING_AREA)
            if item is None:
                founded = True
            idx += 1
        # ok we have our new name, even if it's a really large number :)
        # We need to clean some properties first, like _SYNC_KEYS and the name
        clone_item[u'_id'] = new_id
        clone_item[name_key] = new_name
        
        new_sync_keys = []
        if item_type in ITEM_TYPE.ALL_SERVICES:
            item_sync_keys_uuid = (u'core-%s-%s' % (item_type, clone_item[u'_id'])).lower()
            new_sync_keys.append(item_sync_keys_uuid)
        else:
            if ITEM_TYPE.is_template(item_type):
                item_sync_keys_name = (new_name + '-tpl').lower()
                new_sync_keys.append(item_sync_keys_name)
            else:
                new_sync_keys.append(new_name.lower())
        
        clone_item[u'_SYNC_KEYS'] = new_sync_keys
        
        # Also clean some properties that are specifics to a unique object, not to a clone
        props_to_del = [u'_SE_UUID', u'_SE_UUID_HASH', u'presence_protection', u'editable']
        for prop in props_to_del:
            if prop in clone_item:
                del clone_item[prop]
        
        # A duplicate item come from syncui
        clone_item[u'sources'] = u'syncui'
        
        if item_type in ITEM_TYPE.ALL_HOST_CLASS:
            with DBTransaction():
                self._attach_check_on_host_clone(context, original_item, new_name)
        
        return clone_item
    
    
    def set_enable(self, context, enabled):
        # type: (EditItemContext, bool) -> None
        logger.debug(u'[enable] item [%s-%s] [%s]' % (context.from_state, context.item_type, context.item_id))
        
        initial_item = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, context.from_state)
        if not initial_item:
            logger.debug(u'[enable] item [%s-%s] not found' % (context.item_type, context.item_id))
            raise ItemNotFoundException(code=u'element-unknown', text=self.app.t(u'element.outside_deleted'))
        
        # A user cannot disable himself
        if context.user[u'_id'] == initial_item[u'_id'] and enabled == u'0':
            raise UnauthorizedSaveException(code=406, text=self.app.t(u'element.exception_disable_not_authorized'))
        
        self.check_acl(context, initial_item)
        
        new_item = self.datamanagerV2.get_raw_item(initial_item, flatten_links=False)
        new_item[u'enabled'] = enabled
        
        # Validate to activate doesn't cause a new loop
        validation = syncuicommon.validator.rule_no_loop_in_template(new_item, context.item_type)
        if validation:
            # The validation is not a real validation but only one rule
            has_critical = [v for v in validation if v.get(u'level', u'') == Validator.LEVEL_VALIDATOR_CRITICAL]
            if has_critical:
                message = u' '.join([v[u'message'] for v in validation])
                raise ItemNotValid(text=message)
        
        self.update(context, new_item)
    
    
    def import_one_object(self, context, inheritance_cache=None):
        # type: (EditItemContext, Optional[Dict]) -> Tuple[Dict, Optional[unicode]]
        
        self.check_is_admin(context)
        
        item_from_new = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, ITEM_STATE.NEW)
        no_object_return_value = ({u'has_messages': False}, None)
        if item_from_new is None:  # skip no more valid entries
            return no_object_return_value
        raw_item_from_new = item_from_new.get_raw_item()
        _validation = syncuicommon.validator.validate(context.item_type, raw_item_from_new)
        if _validation[u'has_critical']:
            return _validation, item_from_new.get_name()
        self.datamanagerV2.delete_item(item_from_new, context.user, context.item_type, ITEM_STATE.NEW)
        if context.to_state == ITEM_STATE.WORKING_AREA:
            work_area_info = work_area_helper.init_work_area(raw_item_from_new, context.user)
            raw_item_from_new[WORK_AREA_INFO_KEY] = work_area_info
        self.datamanagerV2.save_item(raw_item_from_new, context.user, context.item_type, context.to_state, action=HISTORY_ACTION.IMPORT, inheritance_cache=inheritance_cache)
        return _validation, item_from_new.get_name()
    
    
    def validate_changes(self, context, inheritance_cache=None):
        # type: (EditItemContext, Optional[Dict]) -> Tuple[Dict, Optional[unicode]]
        
        self.check_is_admin(context)
        
        item_change = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, context.from_state)
        no_object_return_value = ({u'has_messages': False}, None)
        if item_change is None:  # skip no more valid entries
            logger.debug(u'[validate_changes] change not found [%s]' % context.item_id)
            return no_object_return_value
        
        item_name = item_change.get_name()
        raw_item_change = item_change.apply_all_diffs()
        # We also update _SYNC_KEYS from source item see SEF-4777
        item_merged = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, ITEM_STATE.MERGE_SOURCES)
        if item_merged:
            raw_item_change[u'_SYNC_KEYS'] = fast_deepcopy(item_merged[u'_SYNC_KEYS'])
        
        # prepare the staging item to be put in working area
        if context.from_state == ITEM_STATE.STAGGING and context.to_state == ITEM_STATE.WORKING_AREA:
            set_in_working_on_work_area_save(item_change, context.item_type, context.prev_elt, raw_item_change, context.user)
            self._put_in_work_area(context, raw_item_change, item_change)
        
        _validation = syncuicommon.validator.validate(context.item_type, raw_item_change)
        if _validation[u'has_critical']:
            return _validation, item_name
        
        if context.from_state == ITEM_STATE.WORKING_AREA and context.to_state == ITEM_STATE.STAGGING:
            work_area_info = raw_item_change[WORK_AREA_INFO_KEY]
            set_validated(context.user, work_area_info)
            clean_diff(work_area_info)
            raw_item_change[WORK_AREA_INFO_KEY] = work_area_info
            
            self.datamanagerV2.save_item(raw_item_change, user=context.user, item_type=context.item_type, item_state=ITEM_STATE.STAGGING, action=HISTORY_ACTION.APPLY_CHANGES, inheritance_cache=inheritance_cache)
            self.datamanagerV2.delete_item(raw_item_change, user=context.user, item_type=context.item_type, item_state=ITEM_STATE.WORKING_AREA, action=HISTORY_ACTION.VALIDATED_FROM_WORK_AREA)
        else:
            self.datamanagerV2.save_item(raw_item_change, user=context.user, item_type=context.item_type, item_state=context.to_state, action=HISTORY_ACTION.APPLY_CHANGES, inheritance_cache=inheritance_cache)
        return _validation, item_name
    
    
    def save_in_work_area(self, context, item_name, item):
        # type: (EditItemContext, unicode, BaseItem) -> None
        
        item_in_staging = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, ITEM_STATE.STAGGING)
        
        self.check_acl(context, item_in_staging)
        
        # set work area info
        work_area_helper.set_in_working_on_work_area_save(item_in_staging, context.item_type, context.prev_elt, item, context.user)
        if context.must_submit_to_staging:
            # We should push a comment only when we submit to staging, if we just save to the working area, we don't touch the comment
            proposed_comment = {}
            if COMMENT_KEY in item:
                proposed_comment = work_area_helper.create_comment(item[COMMENT_KEY], context.user, WORKING_AREA_STATUS.PROPOSED)
                del item[COMMENT_KEY]
            self.submit_to_staging(context, item_in_working_area=item, comment=proposed_comment)
            if context.is_new:
                logger.info(u'[work_area] Removing [%s-%s] from the new collection' % (context.item_type, item_name))
                self.datamanagerV2.delete_item({u'_id': context.item_id}, context.user, context.item_type, ITEM_STATE.NEW, action=HISTORY_ACTION.IMPORT)
                # logger.debug('[work_area] DONE save item [%s-%s] in work area %s' % (context.item_type, context.item_id, 'and submit to staging' if submit_to_staging else ''))
        else:
            self._put_in_work_area(context, item, item_in_staging=item_in_staging)
    
    
    def unlock_work_area(self, context):
        # type: (EditItemContext) -> bool
        item_in_staging = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, ITEM_STATE.STAGGING)
        item_in_work_area = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, ITEM_STATE.WORKING_AREA)
        
        self.check_acl(context, item_in_staging)
        
        has_item_in_staging = item_in_staging is not None
        if item_in_work_area and has_item_in_staging:
            self.datamanagerV2.delete_item({u'_id': context.item_id}, context.user, context.item_type, ITEM_STATE.WORKING_AREA, action=HISTORY_ACTION.DISCARD_MODIFICATION_FROM_WORK_AREA)
            if item_in_staging:
                work_area_helper.reset_work_area(item_in_staging)
                self.datamanagerV2.save_item(item_in_staging, context.user, context.item_type, ITEM_STATE.STAGGING, action=HISTORY_ACTION.DISCARD_MODIFICATION_FROM_WORK_AREA)
        return has_item_in_staging
    
    
    def delete_item_working_area(self, context):
        # type: (EditItemContext) -> BaseItem
        item_in_work_area = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, ITEM_STATE.WORKING_AREA)
        item_in_staging = self.datamanagerV2.find_item_by_id(context.item_id, context.item_type, ITEM_STATE.STAGGING)
        
        self.check_acl(context, item_in_staging)
        
        if not item_in_work_area and item_in_staging:
            logger.debug(u'[work_area] try to delete item in [%s].' % ITEM_STATE.STAGGING)
            # => Save item in work area (and init work_area_info)
            item_in_work_area = self.get_in_work_area(context, item_in_staging)
        
        self.delete(context, ITEM_STATE.WORKING_AREA, item=item_in_work_area)
        return item_in_staging
    
    
    def put_in_prod(self, _filter, user, item_type):
        # type: (Dict, ContactItem, ItemType) -> Dict
        
        with PREPROD_LOCK:
            return self._put_in_prod(_filter, user, item_type)
    
    
    def _put_in_prod(self, _filter, user, item_type):
        # type: (Dict, ContactItem, ItemType) -> Dict
        
        source_name = _filter[u'sources']
        user_name = user.get_name()
        log = LoggerFactory.get_logger(u'PUT_IN_PRODUCTION')
        log_debug = LoggerFactory.get_logger(u'PUT_IN_PRODUCTION').get_sub_part(u'source:%s' % source_name)
        
        # STEP 01 - Check arbiter availability
        arbiters_aliveness = self.app.arbiter_controller.get_arbiters_aliveness()
        if arbiters_aliveness.get(u'master_reloading', False):  # the arbiter is already reloading, no need for a new configuration
            log.warning(u'The put in production for source: 〖 %s 〗 and user: 〖 %s 〗 was skipped because the arbiter is reloading' % (source_name, user_name))
            return {u'code': 505, u'msg': u'Arbiter is reloading a new configuration'}
        elif not arbiters_aliveness.get(u'master', True):  # the arbiter is dead, not normal, but we won't block for this
            log.warning(u'The put in production for source: 〖 %s 〗 and user: 〖 %s 〗 is launched while the previous arbiter was not reachable:' % (source_name, user_name))
            log.warning(u'  - we are launching a new one with the new configuration')
            log.warning(u'  - please look at the arbiter logs about why it was stopped')
        
        if not user.is_admin():
            return {u'code': 403, u'msg': u'user is not a Shinken admin'}
        
        log_debug.debug(u'STEP 01 - Check arbiter availability : ✔')
        
        # STEP 02 - Prepare preprod (item from production + item from staging which match the filter)
        self.app.datamanagerV2.copy_state(ITEM_STATE.PRODUCTION, ITEM_STATE.PREPROD)
        all_name_in_pre_prod = set([i.get_name() for i in self.app.datamanagerV2.find_items(item_type, ITEM_STATE.PREPROD)])
        with DBTransaction(self.app.datamanagerV2):
            for item in self.app.datamanagerV2.find_items(item_type, ITEM_STATE.STAGGING, where=_filter):
                if item_type not in ITEM_TYPE.ALL_DEDICATED_SERVICES and item.get_name() in all_name_in_pre_prod:
                    item_to_delete = self.app.datamanagerV2.find_item_by_name(item.get_name(), item_type, ITEM_STATE.PREPROD)
                    if item_to_delete[u'_id'] != item[u'_id']:
                        self.app.datamanagerV2.delete_item(item_to_delete, item_type=item_type, item_state=ITEM_STATE.PREPROD, user=user)
                
                self.app.datamanagerV2.save_item(item.get_raw_item(flatten_links=False), user, item_type=item_type, item_state=ITEM_STATE.PREPROD)
        
        log_debug.debug(u'STEP 02 - Prepare preprod (item from production + item from staging which match the filter) : ✔')
        
        # STEP 03 - Validate preprod
        rc, output = self.app.arbiter_controller.launch_arbiter_check(ARBITER_AREA.CHECK_PREPROD)
        if rc != 0:
            error_output = u'\n'.join([i for i in output.split(u'\n') if u' ERROR ' in i])
            log.error(u'The put in production for source: 〖 %s 〗 and user: 〖 %s 〗 failed because the check fail with error: 〖 %s 〗' % (source_name, user_name, error_output))
            return {u'code': 510, u'msg': error_output}
        
        log_debug.debug(u'STEP 03 - Validate preprod: ✔')
        
        # STEP 04 - Put preprod in production
        with DBTransaction(self.app.datamanagerV2):
            for item in self.app.datamanagerV2.find_items(ITEM_TYPE.ELEMENTS, ITEM_STATE.STAGGING, where=_filter):
                last_modification = item.get(u'last_modification', None)
                if last_modification and (last_modification.get(u'show_diffs_state', {}) or last_modification.get(u'show_diffs', True)):
                    item_type = item.get_type()
                    last_modification[u'show_diffs_state'] = {}
                    last_modification[u'show_diffs'] = False
                    self.app.datamanagerV2.save_item(item, user, item_type=item_type, item_state=ITEM_STATE.STAGGING)
        
        self.app.datamanagerV2.copy_state(ITEM_STATE.PREPROD, ITEM_STATE.PRODUCTION)
        syncuicommon.compute_all_diff_staging_production()
        syncuicommon.reset_configuration_stats_cache()
        log.info(u'The put in production for source: 〖 %s 〗 and user: 〖 %s 〗 succeed' % (source_name, user_name))
        
        log_debug.debug(u'STEP 04 - Put preprod in production: ✔')
        
        # STEP 05 - Reboot arbiter
        rc, output = self.app.arbiter_controller.reload_arbiter()
        if rc == 0:
            self.app.arbiter_controller.wait_arbiter_have_reload()
        else:
            error_output = u'\n'.join([i for i in output.split('\n') if u' ERROR: ' in i])
            return {u'code': 503, u'msg': error_output}
        
        log_debug.debug(u'STEP 05 - Reboot arbiter: ✔')
        return {u'code': 200, u'msg': u'arbiter reload OK'}
    
    
    def delete_in_prod(self, uuids, user, item_type):
        # type: (List[unicode], ContactItem, ItemType) -> Dict
        with PREPROD_LOCK:
            return self._delete_in_prod(uuids, user, item_type)
    
    
    def _delete_in_prod(self, uuids, user, item_type):
        # type: (List[unicode], ContactItem, ItemType) -> Dict
        
        result = {u'code': 200, u'data': {}}
        # STEP 01 - Check arbiter availability
        arbiters_aliveness = self.app.arbiter_controller.get_arbiters_aliveness()
        if arbiters_aliveness.get(u'master_reloading', False):
            result[u'code'] = 505
            result[u'data'][u'msg'] = u'Arbiter is reloading a new configuration'
            return result
        elif not arbiters_aliveness.get(u'master', True):
            result[u'code'] = 503
            result[u'data'][u'msg'] = u'Arbiter is not reachable'
            return result
        
        if not user.is_admin():
            result[u'code'] = 403
            result[u'data'][u'msg'] = u'user is not a Shinken admin'
            return result
        
        # STEP 02 - Prepare preprod (item from production - item to delete if exists in staging)
        result[u'data'][u'deleted_items'] = {
            u'list': [],
            u'nb'  : 0,
        }
        result[u'data'][u'not_deleted_items'] = {
            u'list': [],
            u'nb'  : 0,
        }
        self.app.datamanagerV2.copy_state(ITEM_STATE.PRODUCTION, ITEM_STATE.PREPROD)
        with DBTransaction(self.app.datamanagerV2):
            for _uuid in uuids:
                item = self.app.datamanagerV2.find_item_by_id(_uuid, item_type, ITEM_STATE.PREPROD)
                if item:
                    self.app.datamanagerV2.delete_item(item, user, item_state=ITEM_STATE.PREPROD)
                    result[u'data'][u'deleted_items'][u'nb'] += 1
                    result[u'data'][u'deleted_items'][u'list'].append(_uuid)
                else:
                    result[u'data'][u'not_deleted_items'][u'nb'] += 1
                    result[u'data'][u'not_deleted_items'][u'list'].append(_uuid)
        
        if result[u'data'][u'not_deleted_items'][u'nb'] == len(uuids):
            result[u'data'][u'msg'] = u'There is no items to delete. Arbiter was not reload.'
            return result
        
        # STEP 03 - Validate preprod
        rc, output = self.app.arbiter_controller.launch_arbiter_check(ARBITER_AREA.CHECK_PREPROD)
        if rc != 0:
            result[u'code'] = 510
            result[u'data'][u'msg'] = u'\n'.join([i for i in output.split(u'\n') if u' ERROR: ' in i])
            return result
        
        # STEP 04 - Put preprod in production
        self.app.datamanagerV2.copy_state(ITEM_STATE.PREPROD, ITEM_STATE.PRODUCTION)
        syncuicommon.compute_all_diff_staging_production()
        syncuicommon.reset_configuration_stats_cache()
        
        # STEP 05 - Reboot arbiter
        rc, output = self.app.arbiter_controller.reload_arbiter()
        if rc == 0:
            self.app.arbiter_controller.wait_arbiter_have_reload()
        else:
            result[u'code'] = 503
            result[u'data'][u'msg'] = u'\n'.join([i for i in output.split(u'\n') if u' ERROR: ' in i])
            return result
        
        result[u'data'][u'msg'] = u'arbiter reload OK'
        return result
