#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.objects.config import Config
from .abstract_component import AbstractComponent
from .configuration_component import ConfigurationComponent
from .logger_component import LoggerComponent
from .mongo_component import MongoComponent
from .mongo_component_without_auto_reconnect import MongoComponentWithoutAutoReconnect
from .translate_component import TranslateComponent
from ..dao.validators.rules import RulesComponent

try:
    from ordereddict import OrderedDict
except ImportError:
    from collections import OrderedDict

ComponentType = str


class COMPONENT(object):
    CONFIGURATION = 'configuration'
    MONGO = 'mongo'
    MONGO_WITHOUT_AUTO_RECONNECT = 'mongo_without_auto_reconnect'
    TRANSLATE = 'translate'
    LOGGER = 'logger'
    RULE = 'rule'


class ComponentManager(object):
    
    def __init__(self):
        self.components = OrderedDict()  # type: OrderedDict[ComponentType, AbstractComponent]
        self._ready = False
    
    
    def _build(self, conf, host_name, http_start_time):
        # type: (Config , str,  int) -> None
        logger_component = LoggerComponent()
        self.components[COMPONENT.LOGGER] = logger_component
        
        configuration_component = ConfigurationComponent(conf, host_name, http_start_time, logger_component)
        self.components[COMPONENT.CONFIGURATION] = configuration_component
        
        translate_component = TranslateComponent(configuration_component, logger_component)
        self.components[COMPONENT.TRANSLATE] = translate_component
        
        rule_component = RulesComponent()
        self.components[COMPONENT.RULE] = rule_component
        
        mongo_component = MongoComponent(logger_component.get_logger(), configuration_component, 'Synchronizer')
        self.components[COMPONENT.MONGO] = mongo_component
        
        mongo_component_without_auto_reconnect = MongoComponentWithoutAutoReconnect(logger_component.get_logger(), configuration_component, 'Synchronizer')
        self.components[COMPONENT.MONGO_WITHOUT_AUTO_RECONNECT] = mongo_component_without_auto_reconnect
    
    
    def init(self):
        for component_type, component in self.components.iteritems():
            component.init()
        
        self._ready = True
    
    
    def get_component(self, component_type):
        # type: (ComponentType) -> AbstractComponent
        if not self._ready:
            raise Exception('ComponentManager is not ready')
        return self.components[component_type]
    
    
    def is_ready(self):
        # type: () -> bool
        return self._ready
    
    
    def get_translate_component(self):
        # type: () -> TranslateComponent
        return self.get_component(COMPONENT.TRANSLATE)
    
    
    def get_configuration_component(self):
        # type: () -> ConfigurationComponent
        return self.get_component(COMPONENT.CONFIGURATION)
    
    
    def get_mongo_component(self):
        # type: () -> MongoComponent
        return self.get_component(COMPONENT.MONGO)
    
    
    def get_mongo_component_without_auto_reconnect(self):
        # type: () -> MongoComponentWithoutAutoReconnect
        
        # WARNING : this component will skip the AutoReconnect method !! It will be used only in the SynchronizerDaemon IStatsInterface
        # The interface must be fast to reply (timeout is set at 2 seconds only) so we must to skip the AutoReconnect
        
        return self.get_component(COMPONENT.MONGO_WITHOUT_AUTO_RECONNECT)


component_manager = ComponentManager()
