#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.
from shinken.misc.fast_copy import fast_deepcopy
from shinken.misc.type_hint import TYPE_CHECKING
from .base_callback import PreSaver
from ..def_items import ITEM_STATE, HISTORY_ACTION, ITEM_TYPE
from ..helpers import add_source_change
from ..item_comparator import compute_item_diff_existing_vs_from_source
from ...business.item_controller.edit_item_context import ACTIONS
from ...business.sync_ui_common import syncuicommon

if TYPE_CHECKING:
    from shinken.misc.type_hint import Optional, Dict
    from synchronizer.dao.datamanagerV2 import DataManagerV2
    from synchronizer.dao.items import ContactItem

CHANGE_INFO_INDEX = 2


class _CallbackComputeDiffSourceExistingItem(PreSaver):
    def called_before_save(self, item_id=u'', item_type=u'', item_state=u'', item=None, old_item=None, user=None, action=u'', datamanager=None):
        # type: (unicode, unicode, unicode, Optional[Dict], Optional[Dict], Optional[ContactItem], unicode, Optional[DataManagerV2]) -> None
        if item_state not in (ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA) or action in (HISTORY_ACTION.UPDATE_WORK_AREA_INFO, ACTIONS.IMPORT_FROM_SOURCE):
            return
        
        if action == HISTORY_ACTION.APPLY_CHANGES:
            datamanager.delete_item(item, item_type=item_type, item_state=ITEM_STATE.CHANGES)
            return
        
        if item_state == ITEM_STATE.STAGGING and ITEM_TYPE.has_work_area(item_type):
            item_in_work_area = datamanager.find_item_by_id(item_id, item_type, ITEM_STATE.WORKING_AREA)
            if item_in_work_area:
                return
        
        item_source = datamanager.find_item_by_id(item_id, item_type, ITEM_STATE.MERGE_SOURCES)
        
        if item_source and item:
            self.log(item_type, item_state, item_id, action, u'compute diff staging <-> merge item')
            add_source_change(datamanager, item, item_type, action, old_item=old_item)
            changes = compute_item_diff_existing_vs_from_source(item_type, item, item_source, datamanager)
            diff = {u'_id': item[u'_id'], u'changes': changes}
            
            # This will possibly edit the 'changes' parameter
            _remove_changes_from_less_prioritized_sources(changes)
            
            # If the compared items have no common sync keys, they can't be merged together and are considered as separate items
            # So their differences have no meaning and are removed
            if u'_SYNC_KEYS' not in item_source:
                raise Exception(u'_SYNC_KEYS are not found in item MERGE_SOURCE')
            if u'_SYNC_KEYS' not in item:
                raise Exception(u'_SYNC_KEYS are not found in the current item to save')
            if len(changes) == 0 or set(item_source[u'_SYNC_KEYS']).intersection(set(item[u'_SYNC_KEYS'])) == set([]):
                datamanager.delete_item(diff, item_type=item_type, item_state=ITEM_STATE.CHANGES)
            else:
                datamanager.save_item(diff, item_type=item_type, item_state=ITEM_STATE.CHANGES)


def _remove_changes_from_less_prioritized_sources(changes):
    # type: (Dict) -> None
    # We don't want to iterate over an object that we are changing
    changes_copy = fast_deepcopy(changes)
    sources = syncuicommon.app.get_api_sources_from_backend()
    if not sources:
        return
    
    syncui_order = sources[u'syncui'][u'order']
    for property_name, change in changes_copy.iteritems():
        origin_source_name = change[CHANGE_INFO_INDEX][u'property_value'][0][0]
        if origin_source_name not in sources:
            continue
        origin_source_order = sources[origin_source_name][u'order']
        # In this case, the edited object is from a source with a higher priority than the syncui, so we correct the changes object.
        # + We are in the syncui process so we are sure that the change will be between the source gathered just above, and the syncui.
        if origin_source_order > syncui_order:
            changes.pop(property_name)


_callback_compute_diff_source_existing_item = _CallbackComputeDiffSourceExistingItem()
