#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.


import time

from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from .base_callback import PreSaver, PreDeleter
from ..def_items import HISTORY_ACTION, ITEM_STATE, ITEM_TYPE
from ..helpers import compute_diff
from ..items import ContactItem

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Union, Any, Optional
    from ..items.baseitem import BaseItem
    from ..datamanagerV2 import DataManagerV2
    
    HistoryEntry = Dict[unicode, Any]


def add_history_info(datamanager_v2, item, item_type, item_state, user, action, old_item):
    # type: (DataManagerV2, Union[Dict, BaseItem], unicode, unicode, BaseItem, unicode, Optional[Dict]) -> None
    from ..items.baseitem import BaseItem
    
    last_modification = dict()
    if user.get(u'_id', u'-1') == u'-1':
        last_modification[u'contact'] = {
            u'has_plus': False,
            u'links'   : [
                {
                    u'name'  : user.get(u'contact_name'),
                    u'exists': False
                }
            ]
        }
    else:
        last_modification[u'contact'] = {
            u'has_plus': False,
            u'links'   : [
                {
                    u'_id'      : user[u'_id'],
                    u'exists'   : True,
                    u'item_type': ITEM_TYPE.CONTACTS,
                }
            ]
        }
    
    if isinstance(item, BaseItem):
        last_modification[u'contact'][u'links'][0][u'@link'] = {ITEM_STATE.STAGGING: user}
    last_modification[u'action'] = action
    last_modification[u'date'] = time.time()
    last_modification[u'change'] = []
    
    raw_old_item = datamanager_v2.get_raw_item(old_item, item_type=item_type, flatten_links=False)
    raw_item = datamanager_v2.get_raw_item(item, item_type=item_type, flatten_links=False)
    
    show_diffs_state = {}
    if old_item:
        show_diffs_state = old_item.get(u'last_modification', {}).get(u'show_diffs_state', {})
        
        diffs_properties = compute_diff(raw_item, raw_old_item, item_type)
        if len(diffs_properties) > 0:
            for _prop in diffs_properties:
                old_action = show_diffs_state.get(_prop, None)
                if old_action is None or old_action == HISTORY_ACTION.AUTO_MODIFICATION:
                    show_diffs_state[_prop] = action
                else:
                    show_diffs_state[_prop] = old_action
                if _prop != u'last_modification':
                    last_modification[u'change'].append({u'prop': _prop, u'old': raw_old_item.get(_prop, u''), u'new': raw_item.get(_prop, u'')})
    
    show_diffs = bool(next((a for a in show_diffs_state.itervalues() if a != HISTORY_ACTION.AUTO_MODIFICATION), False))
    # If there aren't change when asking a VALIDATED_FROM_WORK_AREA or for AUTO_MODIFICATION we don't change the current last_modification
    if old_item and ((action == HISTORY_ACTION.ELEMENT_MODIFICATION and not last_modification[u'change']) or (action == HISTORY_ACTION.AUTO_MODIFICATION and old_item.get(u'last_modification', {}))):
        last_mod = old_item.get(u'last_modification', {})
        
        if not isinstance(item, BaseItem) and last_mod.get(u'contact', None):
            last_mod[u'contact'][u'links'][0].pop(u'@link', None)

        last_mod[u'show_diffs_state'] = show_diffs_state
        last_mod[u'show_diffs'] = show_diffs
    else:
        last_mod = last_modification
        last_mod[u'show_diffs_state'] = show_diffs_state
        last_mod[u'show_diffs'] = show_diffs
    
    if hasattr(item, u'set_value'):
        item.set_value(u'last_modification', last_mod)
    else:
        item[u'last_modification'] = last_mod
    
    try:
        datamanager_v2.save_history(last_mod, item_type, item_state, raw_item, user)
    except Exception:
        logger.error(u'Fail to save history : %s' % last_mod)
        logger.print_stack()
        
        
def _fix_missing_last_modification_properties(last_modification, action, user):
    # type: (Dict, unicode, BaseItem) -> None
    # updating the last modification of the last object, some old objects does not have actions
    if u'action' not in last_modification:
        last_modification[u'action'] = action

    # The object does not have contact in its last mod. Let's fix this by adding ourselves #SEF-9749
    if u'contact' not in last_modification:
        last_modification[u'contact'] = {
            u'links'   : [{
                u'item_type': ITEM_TYPE.CONTACTS,
                u'_id'      : user[u'_id'],
                u'exists'   : True
            }],
            u'has_plus': False
        }



class _CallbackHistoryInfo(PreSaver, PreDeleter):
    def __init__(self):
        super(_CallbackHistoryInfo, self).__init__()
    
    
    def called_before_save(self, item_id=u'', item_type=u'', item_state=u'', item=None, old_item=None, user=None, action=u'', datamanager_v2=None):
        # type: (unicode, unicode, unicode, Optional[Dict], Optional[Dict], Optional[ContactItem], unicode, Optional[DataManagerV2]) -> None
        if item_state not in (ITEM_STATE.CHANGES, ITEM_STATE.MERGE_SOURCES, ITEM_STATE.PREPROD):
            if action == HISTORY_ACTION.PUT_IN_WORK_AREA:
                old_item = datamanager_v2.find_item_by_id(item_id, item_type, ITEM_STATE.STAGGING)
            self.log(item_type, item_state, item_id, action, u'add history info')
            add_history_info(datamanager_v2, item, item_type, item_state, user, action, old_item)
    
    
    def called_before_delete(self, item_id=u'', item_type=u'', item_state=u'', item=None, old_item=None, user=None, action=u'', datamanager_v2=None):
        # type: (unicode, unicode, unicode, Optional[Dict], Optional[Dict], Optional[ContactItem], unicode, Optional[DataManagerV2]) -> None
        if item_state not in (ITEM_STATE.CHANGES, ITEM_STATE.MERGE_SOURCES, ITEM_STATE.NEW, ITEM_STATE.PREPROD):
            self.log(item_type, item_state, item_id, action, u'add history info')
            add_history_info(datamanager_v2, item, item_type, item_state, user, action, old_item)


_callback_history_info = _CallbackHistoryInfo()
