#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.
import copy
import re

from shinken.log import logger
from shinken.misc.fast_copy import fast_deepcopy
from .def_items import ITEM_TYPE, DEF_ITEMS, ITEM_STATE, NO_HERITABLE_ATTR, METADATA, NOT_TO_LOOK, STOP_INHERITANCE_VALUES, VALUE_FORCE_DEFAULT, SERVICE_EXCLUDES_BY_ID, SERVICE_OVERRIDE, SERVICE_EXCLUDES_BY_NAME, EXCLUDE_FROM_TEMPLATES_RESULTS
from .helpers import split_and_strip_list, get_default_value
from .parsers.complex_exp_parser import do_match


class TEMPLATE_STATUS(object):
    _KEY = 'template-status'
    
    USEFUL = 'useful-template'
    USELESS = 'useless-template'
    BADLOOP = 'badloop-template'
    NEW = 'new-template'
    UNKNOWN = 'unknown-template'
    DISABLED = 'disabled-template'
    
    
    @staticmethod
    def set_status(tpl, status):
        tpl[TEMPLATE_STATUS._KEY] = status
    
    
    @staticmethod
    def get_status(tpl):
        return tpl.get(TEMPLATE_STATUS._KEY, TEMPLATE_STATUS.USELESS)


RESOLVED_PROPS_TO_KEEP = ('low_flap_threshold', 'high_flap_threshold', 'enabled', 'contact_groups', 'check_command')


def is_valid_name(item_type, name):
    illegal_chars = re.compile("""[`~!$%^&*"|'<>?,()=/+]""")
    protected_strings = []
    if item_type in ITEM_TYPE.ALL_SERVICES:
        illegal_chars = re.compile("""[`~!$%^&*"|'<>?,()=+]""")  # same with without / because here it's ok for a service
        protected_strings = ['$KEY$']  # we allow $KEY$ in services
    for token in protected_strings:
        name = name.replace(token, '', 1)
    return not re.search(illegal_chars, name)


def lookup_items_templates(item_type, use_names, linear_result, datamanagerV2, additional_where=None, current_branch=None, add_new=False, service_excludes_by_id=None, service_excludes_by_name=""):
    if not use_names or (len(use_names) == 1 and not use_names[0]):
        return
    if additional_where is None:
        additional_where = {}
    if current_branch is None:
        current_branch = []
    if service_excludes_by_id is None:
        service_excludes_by_id = {'has_plus': False, 'links': []}
    
    template_type = DEF_ITEMS[item_type]['template']
    requested_names = [name for name in use_names if is_valid_name(item_type, name)]
    
    # Now we want also master templates of theses templates
    father_templates_where = {'name': {'$in': requested_names}}
    father_templates_where.update(additional_where)
    father_templates = datamanagerV2.find_items(template_type, ITEM_STATE.STAGGING, where=father_templates_where)
    service_excludes_by_name = split_and_strip_list(service_excludes_by_name)
    templates = []
    for tpl in father_templates:
        raw_tpl = datamanagerV2.get_raw_item(tpl, item_type=item_type, flatten_links=None)
        raw_tpl_flatten = datamanagerV2.get_raw_item(tpl, item_type=item_type)
        if SERVICE_OVERRIDE in raw_tpl:
            raw_tpl_flatten[SERVICE_OVERRIDE] = {}
            raw_tpl_flatten[SERVICE_OVERRIDE]['raw_value'] = tpl[SERVICE_OVERRIDE]['raw_value']
            service_overrides_links = []
            for override in raw_tpl.get(SERVICE_OVERRIDE, {}).get('links', []):
                service_overrides = copy.copy(override)
                service_overrides['key'] = override['key']
                service_overrides['value'] = datamanagerV2.flatten_overridden_property(DEF_ITEMS[item_type]['check_type'], override)
                service_overrides['value_with_link'] = override['value']
                service_overrides_links.append(service_overrides)
            raw_tpl_flatten[SERVICE_OVERRIDE]['links'] = service_overrides_links
        if SERVICE_EXCLUDES_BY_ID in raw_tpl:
            raw_tpl_flatten[SERVICE_EXCLUDES_BY_ID] = fast_deepcopy(raw_tpl[SERVICE_EXCLUDES_BY_ID])
        raw_tpl_flatten[SERVICE_EXCLUDES_BY_ID] = {
            'has_plus': False,
            'links'   : service_excludes_by_id['links'] + raw_tpl_flatten.get(SERVICE_EXCLUDES_BY_ID, {}).get('links', [])
        }
        raw_tpl_flatten[SERVICE_EXCLUDES_BY_NAME] = ",".join(service_excludes_by_name + split_and_strip_list(raw_tpl_flatten.get(SERVICE_EXCLUDES_BY_NAME, "")))
        
        templates.append(raw_tpl_flatten)
    
    father_templates = dict(((tpl['name'], tpl) for tpl in templates))
    
    # we have the fathers templates. Iterate them using names order
    for tpl_name in use_names:
        # Template not found
        if tpl_name not in father_templates:
            dummy_tpl = {'name': tpl_name, 'item_type': template_type}
            if len(current_branch) == 1:
                TEMPLATE_STATUS.set_status(dummy_tpl, TEMPLATE_STATUS.USEFUL)
                dummy_tpl['is_top_level'] = True
            # Try in new ?
            if add_new and is_valid_name(item_type, tpl_name) and datamanagerV2.find_item_by_name_with_case_sensitive(tpl_name, template_type, ITEM_STATE.NEW):
                TEMPLATE_STATUS.set_status(dummy_tpl, TEMPLATE_STATUS.NEW)
                dummy_tpl['is_new'] = True
            if 'is_new' not in dummy_tpl:
                TEMPLATE_STATUS.set_status(dummy_tpl, TEMPLATE_STATUS.UNKNOWN)
                dummy_tpl['unconfigured'] = True
            linear_result.append(dummy_tpl)
            continue
        tpl = father_templates[tpl_name]
        TEMPLATE_STATUS.set_status(tpl, TEMPLATE_STATUS.USELESS)
        tpl['item_type'] = template_type
        tpl['is_template'] = True
        if len(current_branch) == 1 and tpl_name not in [elem['name'] for elem in linear_result if elem['item_type'] == template_type]:
            TEMPLATE_STATUS.set_status(tpl, TEMPLATE_STATUS.USEFUL)
            tpl['is_top_level'] = True
        if (tpl_name, template_type) in [(i[0], i[1]) for i in current_branch]:
            linear_result.append(tpl)
            # Lookup for last top level template
            for prev_tpl in reversed(linear_result):
                if 'is_top_level' in prev_tpl:
                    TEMPLATE_STATUS.set_status(prev_tpl, TEMPLATE_STATUS.BADLOOP)
                    prev_tpl['loop'] = [i[0] for i in current_branch]
                    prev_tpl['loop'].append(tpl_name)
                    break
                else:
                    linear_result.pop()
            continue
        if tpl.get('enabled', '1') == '0' or TEMPLATE_STATUS.DISABLED in [j[2] for i, j in enumerate(current_branch) if i != 0]:
            TEMPLATE_STATUS.set_status(tpl, TEMPLATE_STATUS.DISABLED)
        
        # TODO: Pas propre.. refaire la récupération d'héritage en une seule fois.
        if item_type in (ITEM_TYPE.HOSTS, ITEM_TYPE.CLUSTERS):
            tpl = get_elements_with_inherited(item_type, tpl, set(), datamanagerV2)
        
        # Find if the tpl found is already present in the result.
        # If not, add it,
        # If it is, update the existing tpl with the missing properties
        existing_tpl_list = filter(lambda r: r.get('_id') == tpl.get('_id'), linear_result)
        if existing_tpl_list:
            existing_tpl = existing_tpl_list[0]
            for prop, value in tpl.iteritems():
                if prop not in existing_tpl:
                    existing_tpl[prop] = value
        else:
            linear_result.append(tpl)
        
        tpl_status = TEMPLATE_STATUS.get_status(tpl)
        current_branch.append((tpl_name, template_type, tpl_status))
        use_templates = split_and_strip_list(tpl.get('use', ''))
        current_index = len(linear_result)
        lookup_items_templates(item_type, use_templates, linear_result, datamanagerV2, additional_where, current_branch, service_excludes_by_id=tpl[SERVICE_EXCLUDES_BY_ID], service_excludes_by_name=tpl[SERVICE_EXCLUDES_BY_NAME])
        if TEMPLATE_STATUS.get_status(tpl) == TEMPLATE_STATUS.DISABLED:
            for index, template in enumerate(linear_result):
                if index > current_index:
                    TEMPLATE_STATUS.set_status(template, TEMPLATE_STATUS.DISABLED)
        current_branch.pop()


def _can_attach_check(item):
    return item.get('name', '') and TEMPLATE_STATUS.get_status(item) not in (TEMPLATE_STATUS.UNKNOWN, TEMPLATE_STATUS.BADLOOP)


def _linkify_rm(modulation_period_links):
    modulation_period = None
    for modulation_period_link in modulation_period_links:
        if modulation_period_link['exists']:
            obj = modulation_period_link.get('@link', {}).get(ITEM_STATE.STAGGING, None)
            modulation_period = obj.get_raw_item()
        else:
            modulation_period = {'timeperiod_name': modulation_period_link['name'], 'missing': True}
        return modulation_period


def filter_modulations(modulations_check, modulation_type, datamanagerV2):
    enabled_modulations = []
    if len(modulations_check) and modulations_check[0].get('name', None) in STOP_INHERITANCE_VALUES:
        return VALUE_FORCE_DEFAULT
    
    for check_modulation_link in modulations_check:
        if check_modulation_link['exists']:
            obj = check_modulation_link.get('@link', {}).get(ITEM_STATE.STAGGING, None)
        else:
            obj = datamanagerV2.find_item_by_name(check_modulation_link['name'], item_type=modulation_type, item_state=ITEM_STATE.NEW)
        
        if obj:
            if obj.get_type() == ITEM_TYPE.RESULTMODULATIONS and obj.get('modulation_period', None):
                new_raw_modulation_period = _linkify_rm(obj.get_links('modulation_period'))
                obj = obj.get_raw_item(keep_metadata=(METADATA.STATE,))
                obj['modulation_period'] = new_raw_modulation_period
            else:
                obj = obj.get_raw_item(keep_metadata=(METADATA.STATE,))
        else:
            obj = {DEF_ITEMS[modulation_type]['key_name']: check_modulation_link['name'], 'missing': True}
        if obj:
            enabled_modulations.append(obj)
    return enabled_modulations


def check_and_its_template_are_enabled(check, tpl_on_host):
    return check.get('enabled', '1') == '1' and TEMPLATE_STATUS.get_status(tpl_on_host) != TEMPLATE_STATUS.DISABLED


def inherited_checks_from_hosts_templates(use_names, myname, myuuid, item_type, datamanagerV2, with_superfluous_checks=False):
    father_templates = [{'name': myname, TEMPLATE_STATUS._KEY: TEMPLATE_STATUS.USEFUL, 'is_top_level': True, 'item_type': item_type, 'is_me': True}]
    for _state in (ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA):
        my_self = datamanagerV2.find_item_by_id(myuuid, item_type=item_type, item_state=_state)
        if my_self:
            father_templates[0]['_id'] = myuuid
            break
    
    # TODO find all parent template with link and remove this call
    lookup_items_templates(item_type, use_names, father_templates, datamanagerV2, current_branch=[(myname, item_type, TEMPLATE_STATUS.USEFUL)])
    
    # Disabled templates are pushed back to the end of the list so that their checks don't interfere
    disabled_templates = [tpl for tpl in father_templates if TEMPLATE_STATUS.get_status(tpl) == TEMPLATE_STATUS.DISABLED]
    enabled_templates = [tpl for tpl in father_templates if TEMPLATE_STATUS.get_status(tpl) != TEMPLATE_STATUS.DISABLED]
    father_templates = enabled_templates + disabled_templates
    
    all_templates = []
    for index, tpl in enumerate(father_templates):
        tpl_enabled = TEMPLATE_STATUS.get_status(tpl) != TEMPLATE_STATUS.DISABLED
        if '_id' in tpl:
            all_templates.append({'_id': tpl['_id'], 'exists': True, 'enabled': tpl_enabled})
        elif index == 0:
            # Current item in page can not 'exists' in base for the moment so it check can match by name -> can_match_by_name : True
            # Current item in page are always considered enable otherwise it would not have any checks -> enabled: True}
            all_templates.append({'name': tpl.get('name', ''), 'exists': False, 'can_match_by_name': True, 'enabled': True})
        else:
            all_templates.append({'name': tpl.get('name', ''), 'exists': False, 'enabled': tpl_enabled})
    
    # It contains only template in use_names
    template_in_use_items = [datamanagerV2.find_item_by_name_with_case_sensitive(i, item_type=ITEM_TYPE.get_template_type(item_type), item_state=ITEM_STATE.STAGGING) for i in use_names]
    template_in_use_items = [i for i in template_in_use_items if i and i.get('enabled', '1') == '1']
    
    use_names = [h.get('name', '') for h in father_templates if TEMPLATE_STATUS.get_status(h) in (TEMPLATE_STATUS.USEFUL, TEMPLATE_STATUS.DISABLED)]
    # Add the current template to the list, as a top level one
    used_tpls = dict(((name, []) for name in use_names))
    matched_checks = set()
    
    tpl_on_host = {
        'name': '',
        'type': '',
        'uuid': ''
    }
    
    for host_tpl in father_templates:
        is_template_disable = TEMPLATE_STATUS.get_status(host_tpl) == TEMPLATE_STATUS.DISABLED
        hname = host_tpl.get('name', '')
        
        # If there is a loop, we skip all checks attached here, just go to next top level template
        # if host_tpl.get('is_top_level', False):
        if TEMPLATE_STATUS.get_status(host_tpl) in (TEMPLATE_STATUS.USEFUL, TEMPLATE_STATUS.DISABLED):
            tpl_on_host = {
                'name': hname,
                'type': host_tpl.get('item_type', None),
                'uuid': host_tpl.get('_id', None)
            }
        if not _can_attach_check(host_tpl):
            continue
        type_to_search = DEF_ITEMS[host_tpl['item_type']]['check_type']
        
        if '_id' in host_tpl:
            host_tpl_link = (host_tpl['_id'], True)
        else:
            host_tpl_link = (hname, False)
        
        current_matched_checks = datamanagerV2.find_items(item_type=type_to_search, item_state=ITEM_STATE.STAGGING, where={'host_name_links': host_tpl_link})
        
        check_names = set()
        for check in current_matched_checks:
            check_name = check.get_name()
            if not check_name:
                continue
            
            is_check_already_added_on_host = check_name in (matched_checks | check_names)
            is_check_disabled = not check.is_enabled()
            is_check_inactive = is_check_already_added_on_host or is_template_disable or is_check_disabled
            if is_check_inactive and not with_superfluous_checks:
                continue
            
            host_infos = (type_to_search, tpl_on_host['type'], tpl_on_host['uuid'], not is_check_inactive)
            
            check_must_be_added = False
            if check.get_type() in (ITEM_TYPE.SERVICESHOSTTPLS, ITEM_TYPE.SERVICESCLUSTERTPLS):
                pattern = check.get('host_name', {})
                node = pattern.get('node', {})
                try:
                    if do_match(node, all_templates):
                        check_must_be_added = True
                except SyntaxError as e:
                    logger.warning('Invalid parse [%s]' % e)
            else:
                # check without complex expression match directly
                check_must_be_added = True
            
            # Process check excludes
            if not check_must_be_added:
                continue
            
            used_tpls[tpl_on_host['name']].append((check, host_infos))
            if check_and_its_template_are_enabled(check, tpl_on_host):
                check_names.add(check_name)
        
        # Mark all checks as already added by a previous template.
        matched_checks |= check_names
    
    used_tpls = filter_checks_excluded(template_in_use_items, used_tpls)
    
    # Now we have the final checks list.
    # We flatten the value inherited.
    inherited_checks = dict(((name, []) for name in use_names))
    for hname, checks in used_tpls.iteritems():
        known_checks = set()
        for check, host_infos in checks:
            # Front don't want inactive check on same template even if with 'with_superfluous_checks'
            check_name_and_type = '%s-%s' % (check.get_type(), check.get_name())
            if check_name_and_type in known_checks and not host_infos[3]:
                continue
            known_checks.add(check_name_and_type)
            resultmodulations = None
            if 'resultmodulations' in check:
                enabled_result_modulations = filter_modulations(check.get_links('resultmodulations'), ITEM_TYPE.RESULTMODULATIONS, datamanagerV2)
                resultmodulations = enabled_result_modulations
            
            macromodulations = None
            if 'macromodulations' in check:
                enabled_macro_modulations = filter_modulations(check.get_links('macromodulations'), ITEM_TYPE.MACROMODULATIONS, datamanagerV2)
                macromodulations = enabled_macro_modulations
            
            # Add all Data
            raw_check = check.get_raw_item()
            for check_property, check_value in [(key, value) for (key, value) in check.iteritems() if key.startswith('_') and key not in NOT_TO_LOOK]:
                raw_check[check_property] = get_default_value(check.get_type(), check_property, pythonize=False) if check_value in STOP_INHERITANCE_VALUES else check_value
            
            for check_property in RESOLVED_PROPS_TO_KEEP:
                check_value = datamanagerV2.flatten_prop(check, check.get_type(), check_property)
                raw_check[check_property] = get_default_value(check.get_type(), check_property, pythonize=False) if check_value in STOP_INHERITANCE_VALUES else check_value
            
            METADATA.update_metadata(raw_check, METADATA.ITEM_TYPE, host_infos[0])
            METADATA.update_metadata(raw_check, METADATA.HOST_TYPE, host_infos[1])
            METADATA.update_metadata(raw_check, METADATA.HOST_UUID, host_infos[2])
            
            if resultmodulations:
                raw_check['resultmodulations'] = resultmodulations
            if macromodulations:
                raw_check['macromodulations'] = macromodulations
            inherited_checks[hname].append(raw_check)
    return inherited_checks


def filter_checks_excluded(template_in_use_items, used_tpls):
    for hname, checks in used_tpls.items():
        to_exclude = []
        to_exclude_for_this_template = []
        for check, host_infos in checks:
            
            result = EXCLUDE_FROM_TEMPLATES_RESULTS.ALLOW
            for sub_template in template_in_use_items:
                sub_result = sub_template.is_service_excluded_by_template(check)
                if sub_result == EXCLUDE_FROM_TEMPLATES_RESULTS.DONT_HAVE:
                    continue
                elif sub_result == EXCLUDE_FROM_TEMPLATES_RESULTS.ALLOW:
                    result = EXCLUDE_FROM_TEMPLATES_RESULTS.ALLOW
                    break
                elif sub_result == EXCLUDE_FROM_TEMPLATES_RESULTS.EXCLUDE:
                    if sub_template.get_name() == hname:
                        # For display if a template remove a check we remove it from this template
                        to_exclude_for_this_template.append((check, host_infos))
                    result = EXCLUDE_FROM_TEMPLATES_RESULTS.EXCLUDE
                else:
                    raise Exception('No such sub-result')
            
            if result == EXCLUDE_FROM_TEMPLATES_RESULTS.EXCLUDE:
                logger.debug('the check [%s] has been excluded by all template' % check)
                to_exclude.append((check, host_infos))
        
        used_tpls[hname] = [i for i in checks if i not in to_exclude]
        used_tpls[hname] = [i for i in checks if i not in to_exclude_for_this_template]
    return used_tpls


def get_elements_with_inherited(item_types, final_elem, known_tpls, datamanagerV2):
    higher_use = split_and_strip_list(final_elem.get('use', ''))
    if not higher_use:
        return final_elem
    
    if not isinstance(item_types, tuple):
        item_types = [item_types]
    
    d = {}
    for item_type in item_types:
        key_name = DEF_ITEMS[item_type]['key_name']
        services = datamanagerV2.find_items(item_type=item_type, item_state=ITEM_STATE.STAGGING, where={key_name: {"$in": higher_use}})
        for service in services:
            d[service.get('name', '')] = service.get_raw_item()
            METADATA.update_metadata(service, METADATA.ITEM_TYPE, item_type)
    
    # Loop on each elements
    for tname in higher_use:
        # The template does not exist in staging, cannot inherit
        if tname not in d:
            continue
        # Infinite loop prevention
        if tname in known_tpls:
            continue
        known_tpls.add(tname)
        elem = d[tname]
        use = elem.get('use', '')
        for attr in NO_HERITABLE_ATTR:
            if attr in elem:
                del elem[attr]
        elem.update(final_elem)
        if use:
            elem['use'] = use
            final_elem = get_elements_with_inherited(item_types, elem, known_tpls, datamanagerV2)
        else:
            final_elem = elem
        final_elem['use'] = ','.join(higher_use)
    
    return final_elem
