#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import hashlib
import re

from shinken.log import logger
from shinken.macroresolver import MacroResolver, ARGUMENT_SEPARATOR, COMMAND_LINE_LIMIT
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects import Host, Service
from shinkensolutions.toolbox.box_tools_string import ToolsBoxString
from .def_items import ITEM_TYPE, STOP_INHERITANCE_VALUES
from ..dao.datamanagerV2 import FALLBACK_USER

if TYPE_CHECKING:
    from shinken.misc.type_hint import Union
    from .crypto import FrontendCipher
    from .items.baseitem import BaseItem
    from ..synchronizerdaemon import Synchronizer

# Policy on protected macro:
# Listed macro are macros that match pattern defined by the synchronizer property : protect_fields__substrings_matching_fields
# This property is defined in synchronizer.cfg but can be overload and also manage by the command : shinken-protected-fields-data-manage
# All macro used in definition of protected macro are also protected (hidden)
#   For admin:
#       If frontend_cipher is enabled:
#           * Listed macro are protected
#       Else:
#           * All macro are visible
#   For si admin:
#       * Global marcos are protected
#       If frontend_cipher is enabled:
#           * Listed macro are protected
#       Else:
#           * Listed macros are protected unless they are directly in host

DOUBLEDOLLAR = u'__DOUBLEDOLLAR__'
PROTECTED_EXCLAMATION_MARK = u'__EXCLAMATION_MARK__'
_ARGN_MACROS = set([u'ARG%d' % i for i in xrange(1, 33)])  # 32


class SPECIAL_VALUE(object):
    MISSING_ARGN = u'__MISSING_ARGN__'
    MISSING_DUPLICATE_FOR_EACH = u'__MISSING_DFE__'
    MISSING_HOST = u'__MISSING_HOST__'
    MISSING_CHECK = u'__MISSING_CHECK__'
    MISSING_GLOBAL = u'__MISSING_GLOBAL__'
    PROTECTED_PASSWORD = u'__PROTECTED_PASSWORD__'
    NOT_IMPLEMENTED = u'__NOT_IMPLEMENTED__'


class FROM_INFO(object):
    ARGN = u'argn'
    DUPLICATE_FOR_EACH_KEY = u'duplicate foreach (key)'
    DUPLICATE_FOR_EACH_VALUE = u'duplicate foreach (value)'
    DUPLICATE_FOR_EACH = u'duplicate foreach'
    GLOBAL = u'global'
    HOST = u'host'
    HOST_TPL = u'template'
    CHECK = u'check'


def get_macros(raw_string):
    p = re.compile(r'(\$[^$]*?\$)')
    macros = []
    for m in p.finditer(raw_string):
        macro_name = m.group(1)[1:-1].strip()
        if macro_name:
            item = (macro_name.upper(), macro_name)
            if item not in macros:
                macros.append(item)
    
    return macros


def resolve_macros(app, host, host_templates, check, check_command_arg, to_expand, dfe, item_type, frontend_cipher=None):
    try:
        user = app.get_user_auth()
    except AttributeError:
        # When analyzer ask for a _resolve_macro it not have access to user auth.
        # Other call make user auth check before calling _resolve_macro so
        # We can assume if there are an AttributeError except the user is an admin.
        user = FALLBACK_USER
    expand_macros_infos = {u'!!check_command_arg': check_command_arg}
    macros = get_macros(to_expand)
    logger.debug(u'[macro_resolver] start resolve macros : [%s]' % macros)
    error = None
    for upper_macro_name, original_macro_name in macros:
        is_protected = frontend_cipher and frontend_cipher.match_protected_property(upper_macro_name, item_type, user=user)
        # Show policy on protected macro
        resolution_info = _resolve_macro(app, upper_macro_name, original_macro_name, host, host_templates, check, upper_macro_name, dfe, expand_macros_infos, is_protected, user, frontend_cipher)
        if not resolution_info.get(u'from', None):  # in case of error (recursive limit)
            error = resolution_info.get(u'value', u'')
        
        expand_macros_infos[upper_macro_name] = (upper_macro_name, original_macro_name, resolution_info)
    
    expand_macros_infos.pop(u'!!check_command_arg', None)
    logger.debug(u'[macro_resolver] --- macros resolved ---')
    header = u'[macro_resolver] %30s | %20s | %30s'
    logger.debug(header % (u'macro name', u'from', u'value'))
    for upper_macro_name, macro_info in expand_macros_infos.iteritems():
        logger.debug(header % (upper_macro_name, macro_info[2][u'from'] if len(macro_info) > 2 and macro_info[2] else u'', macro_info[2][u'value'] if len(macro_info) > 2 and macro_info[2] else u''))
    
    return expand_macros_infos, error


def _build_macro_info(expand_macros_infos, macro_name, original_macro_name, _from, value, raw_value, value_unprotected, founded_prop, is_protected, from_info=None, data_modulation=None):
    if not founded_prop:
        is_protected = False
    if expand_macros_infos.get(macro_name, None):
        _, _, macro_info = expand_macros_infos[macro_name]
    else:
        resolve_order = len(expand_macros_infos)
        macro_info = {
            u'founded_prop'     : founded_prop,
            u'from'             : _from,
            u'value'            : SPECIAL_VALUE.PROTECTED_PASSWORD if is_protected else ToolsBoxString.escape_XSS(value),
            u'raw_value'        : SPECIAL_VALUE.PROTECTED_PASSWORD if is_protected else ToolsBoxString.escape_XSS(raw_value),
            u'unprotected_value': value_unprotected,
            u'from_info'        : from_info,
            u'is_protected'     : is_protected,
            u'resolve_order'    : resolve_order,
            u'data_modulation'  : data_modulation
        }
        expand_macros_infos[macro_name] = (macro_name, original_macro_name, macro_info)
    return macro_info


# Try to look at the macro somewhere, manage recursive lookup, split macros and such things
def _resolve_macro(app, macro_name, original_macro_name, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, frontend_cipher, level=32):
    # Ok, it's just too much level here
    if level <= 0:
        value = app._(u'element.resolve_macro_max_recursive') % initial_macro
        return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, u'', value, value, value, initial_macro, is_protected)
    
    logger.debug(u'[macro_resolver] Resolving macro [%s] level [%s]' % (macro_name, level))
    
    # Can be global or on elements
    if not macro_name.startswith(u'HOST') and not macro_name.startswith(u'_HOST') and not macro_name.startswith(u'SERVICE') and not macro_name.startswith(u'_SERVICE'):
        logger.debug(u'[macro_resolver] try to look at global macro [%s]' % macro_name)
        
        # Maybe it's an ARGN that is referenced into a macro.
        if macro_name in _ARGN_MACROS:
            argn_offset = int(macro_name.replace(u'ARG', u''))
            check_command_arg = expand_macros_infos[u'!!check_command_arg']
            logger.debug(u'[macro_resolver] Did detected an ARGN macro: %d => %s' % (argn_offset, macro_name))
            try:
                raw_value = re.split(r'!|\|-\|', check_command_arg)[argn_offset - 1].replace(PROTECTED_EXCLAMATION_MARK, u'!')
                value = raw_value
                value_unprotected = raw_value
                if u'$' in raw_value:
                    value, value_unprotected = _resolve_macro_in_macro_value(app, raw_value, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, frontend_cipher, level=level - 1)
                if ARGUMENT_SEPARATOR in value_unprotected:
                    expand_macros_infos[u'!!check_command_arg'] = check_command_arg.replace(raw_value, value_unprotected.replace(u'!', PROTECTED_EXCLAMATION_MARK))
                    return _resolve_macro(app, macro_name, original_macro_name, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, frontend_cipher, level=level)
            except IndexError:
                value = u''
                value_unprotected = u''
                raw_value = SPECIAL_VALUE.MISSING_ARGN
            
            return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.ARGN, value, raw_value, value_unprotected, u'', is_protected)
        
        # Maybe it's an internal duplicate_foreach value, so show it
        _duplicate_foreach_macros = [u'KEY', u'VALUE']
        for i in xrange(1, 17):  # 16
            _duplicate_foreach_macros.append(u'VALUE%d' % i)
        
        if macro_name in _duplicate_foreach_macros:
            is_protected = is_protected or (frontend_cipher and frontend_cipher.match_protected_property(macro_name, ITEM_TYPE.SERVICESHOSTTPLS, user=user))
            logger.debug(u'[macro_resolver] resolving [%s] type:[duplicate for each]' % macro_name)
            if dfe:
                _dfe_property_name = dfe.get(u'property_name', u'Unknown property name')
                if macro_name == u'KEY':
                    _dfe_key = dfe.get(u'key', u'')
                    logger.debug(u'[macro_resolver] found value [%s]:[%s]' % (macro_name, _dfe_key))
                    
                    return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.DUPLICATE_FOR_EACH_KEY, _dfe_key, _dfe_key, _dfe_key, _dfe_property_name, is_protected)
                else:
                    _dfe_values = dfe.get(u'args', u'')
                    for i in range(len(_dfe_values)):
                        if macro_name == u'VALUE':
                            logger.debug(u'[macro_resolver] found value [%s]:[%s]' % (macro_name, _dfe_values[0]))
                            value = _dfe_values[0]
                            raw_value = value
                            value_unprotected = value
                            if u'$' in _dfe_values[0]:  # if there is another macro, look deeper
                                value, value_unprotected = _resolve_macro_in_macro_value(app, _dfe_values[0], host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, frontend_cipher, level=level - 1)
                            return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.DUPLICATE_FOR_EACH_VALUE, value, raw_value, value_unprotected, _dfe_property_name, is_protected)
                        
                        if macro_name == (u'VALUE%s' % (i + 1)):
                            value = _dfe_values[i]
                            raw_value = value
                            value_unprotected = value
                            logger.debug(u'[macro_resolver] found value [%s]:[%s]' % (macro_name, value))
                            if u'$' in value:  # if there is another macro, look deeper
                                value, value_unprotected = _resolve_macro_in_macro_value(app, value, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, frontend_cipher, level=level - 1)
                            return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.DUPLICATE_FOR_EACH_VALUE, value, raw_value, value_unprotected, _dfe_property_name, is_protected)
            else:
                logger.debug(u'[macro_resolver] fail to resolve [%s] type:[duplicate for each] dfe:[%s]' % (macro_name, dfe))
                value = SPECIAL_VALUE.MISSING_DUPLICATE_FOR_EACH
                return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.DUPLICATE_FOR_EACH, value, value, value, u'', is_protected)
        
        user_is_admin = user.get(u'is_admin', u'0') == u'1'
        is_protected = is_protected or (not user_is_admin and not (frontend_cipher and frontend_cipher.encryption_enable))
        # Ok so not a duplicate_foreach specific macro, try to look in real globals
        if macro_name in MacroResolver.macros.keys():
            value = SPECIAL_VALUE.NOT_IMPLEMENTED
            return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.GLOBAL, value, value, value, u'', is_protected)
        
        value = getattr(app.conf, (u'$%s$' % macro_name).encode(u'utf8', u'ignore'), None)
        if value is None:
            return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.GLOBAL, u'', SPECIAL_VALUE.MISSING_GLOBAL, u'', u'', is_protected)
        raw_value = value
        value_unprotected = value
        if u'$' in value:  # if there is another macro, look deeper and return what ever we did find
            value, value_unprotected = _resolve_macro_in_macro_value(app, value, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, frontend_cipher, level=level - 1)
        
        # return as we did find in global
        return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.GLOBAL, value, raw_value, value_unprotected, macro_name, is_protected)
    
    # Now we can look in the host or the check
    # First host
    if macro_name.startswith(u'HOST') or macro_name.startswith(u'_HOST'):
        logger.debug(u'[macro_resolver] searching [%s] in host and host template' % macro_name)
        is_protected = is_protected or (frontend_cipher and frontend_cipher.match_protected_property(macro_name, ITEM_TYPE.HOSTS, user=user))
        modulation = None
        prop = u''
        value = u''
        
        if macro_name.upper() == u'HOSTUUID':
            value = host[u'_id']
            raw_value = value
        elif macro_name.upper() in Host.macros:
            prop = Host.macros[macro_name.upper()].lower()
            value = host.get(prop, None)
            raw_value = value
        elif macro_name.startswith(u'_HOST'):
            prop = u'_' + macro_name[(len(u'_HOST')):].upper()
            value = host.get(prop, None)
            raw_value = value
            if prop in host.get(u'@modulation', {}):
                value = host[u'@modulation'][prop][0]
                modulation = host[u'@modulation'][prop][1]
        
        if value:
            value_unprotected = ToolsBoxString.unescape_XSS(value)
            if u'$' in value:  # if there is another macro, look deeper
                is_protected = is_protected or (frontend_cipher and frontend_cipher.match_protected_property(macro_name, ITEM_TYPE.HOSTS, user=user))
                value, value_unprotected = _resolve_macro_in_macro_value(app, value, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, frontend_cipher, level=level - 1)
            return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.HOST, value, raw_value, value_unprotected, prop, is_protected, data_modulation=modulation)
        else:  # some values have default, like alias => host_name or address=> host_name
            if prop in (u'alias', u'address', u'display_name'):
                value = host.get(u'host_name', u'')
                raw_value = value
                value_unprotected = ToolsBoxString.unescape_XSS(value)
                is_protected = is_protected or (frontend_cipher and frontend_cipher.match_protected_property(macro_name, ITEM_TYPE.HOSTS, user=user))
                return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.HOST, value, raw_value, value_unprotected, prop, is_protected, data_modulation=modulation)
        
        # Maybe in a template one?
        for tpl in host_templates:
            is_protected = is_protected or (frontend_cipher and frontend_cipher.match_protected_property(macro_name, ITEM_TYPE.HOSTTPLS, user=user))
            logger.debug(u'[macro_resolver] searching in tpl [%s]' % tpl.get(u'name', u'no name'))
            value = tpl.get(prop, None)
            if value is not None:
                raw_value = value
                value_unprotected = ToolsBoxString.unescape_XSS(value)
                if u'$' in value:  # if there is another macro, look deeper
                    value, value_unprotected = _resolve_macro_in_macro_value(app, value, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, frontend_cipher, level=level - 1)
                return _build_macro_info(
                    expand_macros_infos,
                    macro_name,
                    original_macro_name,
                    FROM_INFO.HOST_TPL,
                    value,
                    raw_value,
                    value_unprotected,
                    prop,
                    is_protected,
                    from_info={u'name': tpl.get(u'name', u''), u'uuid': tpl[u'_id'], },
                    data_modulation=modulation
                )
        # Cannot find anything
        return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.HOST, u'', SPECIAL_VALUE.MISSING_HOST, u'', u'', is_protected, data_modulation=modulation)
    
    # Then check
    if macro_name.startswith(u'SERVICE') or macro_name.startswith(u'_SERVICE'):
        is_protected = is_protected or (frontend_cipher and frontend_cipher.match_protected_property(macro_name, ITEM_TYPE.SERVICETPLS, user=user))
        modulation = None
        prop = u''
        value = u''
        if macro_name.upper() == u'SERVICEUUID':
            if dfe:
                # If it's a Duplicate we need to generate the same uuid as it will be generated in Arbiter
                check_name = check[u'service_description'].replace(u'$KEY$', dfe[u'key'])
                check_uuid = hashlib.md5(check_name.encode(u'utf-8')).hexdigest()
            else:
                check_uuid = check[u'_id']
            value = u'%s-%s' % (host[u'_id'], check_uuid)
            raw_value = value
        elif macro_name.upper() in Service.macros:
            prop = Service.macros[macro_name.upper()].lower()
            value = check.get(prop, None)
            raw_value = value
        elif macro_name.startswith(u'_SERVICE'):
            prop = u'_' + macro_name[(len(u'_SERVICE')):].upper()
            value = check.get(prop, None)
            
            raw_value = value
            if prop in check.get(u'@modulation', {}):
                value = check[u'@modulation'][prop][0]
                modulation = check[u'@modulation'][prop][1]
        else:
            raw_value = SPECIAL_VALUE.MISSING_CHECK
        
        logger.debug(u'[macro_resolver] Trying to get the property [%s] from the check [%s]' % (prop, check.get(u'name', check.get(u'service_description', u'name not found'))))
        if value is not None:
            value_unprotected = ToolsBoxString.unescape_XSS(value)
            if u'$' in value:  # if there is another macro, look deeper
                value, value_unprotected = _resolve_macro_in_macro_value(app, value, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, frontend_cipher, level=level - 1)
            return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.CHECK, value, raw_value, value_unprotected, prop, is_protected, data_modulation=modulation)
        
        # cannot find anything
        return _build_macro_info(expand_macros_infos, macro_name, original_macro_name, FROM_INFO.CHECK, u'', SPECIAL_VALUE.MISSING_CHECK, u'', u'', is_protected, data_modulation=modulation)


def _resolve_macro_in_macro_value(app, macro_value, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, frontend_cipher, level=32):
    # type: (Synchronizer, unicode, BaseItem, list, BaseItem, unicode,unicode, dict, bool, bool, FrontendCipher, int) -> (unicode, unicode)
    sub_macros = get_macros(macro_value)
    macro_value_resolved = macro_value
    macro_value_resolved_unprotected = ToolsBoxString.unescape_XSS(macro_value_resolved)
    
    for upper_sub_macro, original_sub_macro in sub_macros:
        logger.debug(u'[macro_resolver] Resolving sub macro [%s] in macro value [%s] u' % (upper_sub_macro, macro_value))
        sub_res = _resolve_macro(app, upper_sub_macro, original_sub_macro, host, host_templates, check, initial_macro, dfe, expand_macros_infos, is_protected, user, frontend_cipher, level=level - 1)
        if sub_res is None or sub_res[u'unprotected_value'] is None or sub_res[u'unprotected_value'] in STOP_INHERITANCE_VALUES:
            logger.debug(u'[macro_resolver] inner macro [%s] in [%s] was not found' % (upper_sub_macro, macro_value))
            macro_value_resolved = macro_value_resolved.replace(u'$%s$' % original_sub_macro, u'')
            macro_value_resolved_unprotected = macro_value_resolved_unprotected.replace(u'$%s$' % original_sub_macro, u'')
        else:
            macro_value_resolved = macro_value_resolved.replace(u'$%s$' % original_sub_macro, sub_res[u'value'])
            macro_value_resolved_unprotected = macro_value_resolved_unprotected.replace(u'$%s$' % original_sub_macro, sub_res[u'unprotected_value'])
    
    return macro_value_resolved, macro_value_resolved_unprotected


def expand_value(to_expand, _resolve_macros):
    for _, original_macro_name, macro_infos in _resolve_macros.itervalues():
        if macro_infos is None:
            continue
        if macro_infos[u'unprotected_value'] in STOP_INHERITANCE_VALUES:
            macro_infos[u'unprotected_value'] = u''
        
        to_expand = to_expand.replace(u'$%s$' % original_macro_name, macro_infos[u'unprotected_value'])
    return to_expand


def truncate_long_command_line(app, command_line):
    # type: (Synchronizer, Union[str,unicode]) -> (Union[str,unicode], bool)
    cmd_len = len(command_line)
    has_truncate = False
    if cmd_len > COMMAND_LINE_LIMIT:
        translate = getattr(app, u'_')
        command_format = u'[ %%.%ds... ]' % COMMAND_LINE_LIMIT
        command_format = u'%s\n\n%s' % (translate(u'element.command_too_long'), command_format)
        command_line = command_format % (cmd_len, COMMAND_LINE_LIMIT, command_line)
        has_truncate = True
    
    return command_line, has_truncate
