#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import copy
import datetime
import sys
import time
import traceback
from hashlib import sha256
from pprint import pformat

from pymongo.errors import BulkWriteError, InvalidOperation

from shinken.log import LoggerFactory
from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING, Optional, NoReturn, Dict
from shinken.objects.host import VIEW_CONTACTS_DEFAULT_VALUE
from .crypto import DatabaseCipher
from .datamanagerV2 import DataManagerV2, get_type_item_from_class
from .dataprovider.dataprovider_mongo import DataProviderMongo
from .def_items import ITEM_STATE, ITEM_TYPE
from .helpers import get_default_value, add_unique_value_and_handle_plus_and_null
from ..component.component_manager import component_manager

if TYPE_CHECKING:
    from ..synchronizerdaemon import Synchronizer

app = None  # type: Optional[Synchronizer]

logger = LoggerFactory.get_logger(name=u'DATA MIGRATION')
logger_protected_fields = logger.get_sub_part(u'PROTECTED FIELDS', part_name_size=16)


def migrate_database(_app):
    global app, migration_datamanager
    app = _app
    logger.info(u'check if your data needs to be migrated')
    t_start = time.time()
    # We use our datamanager with no callback
    mongo_component = component_manager.get_mongo_component()
    migration_datamanager = DataManagerV2(DataProviderMongo(mongo_component, app.database_cipher), synchronizer=app, use_default_callbacks=False)
    mongo_component.clean_source_backup_collection()
    
    _remove_old_tmp_collection()
    _migrate_hosts_contacts_properties_in_db()
    _migrate_contacts_contactgroups_properties_in_db()
    _migrate_database_part_protected_properties()
    _check_old_host_to_cluster_boolean()
    _check_clusters_to_cluster_boolean()
    _add_source_name_in_discovery_confs()
    _migrate_nb_elements_in_last_synchronization()
    _remove_basic_element_from_database()
    logger.info(u'Migration done in %ss' % (time.time() - t_start))


def migrate_hosts_contacts_properties(app, hosts):
    # Host and host templates must be changed from the contacts into:
    # * view_contacts/groups
    # * notification_contacts/groups
    # * edition_contacts/groups
    # This only applies if the host or host template does not have any defined view/notification/edition_contact/groups
    properties = ['contacts', 'contact_groups']
    prefixs = ['view', 'notification', 'edition']
    must_view_prefixs = ['notification', 'edition']
    for item in hosts:
        for property in properties:
            for prefix in prefixs:
                new_property = '%s_%s' % (prefix, property)
                if property in item and new_property in item:
                    # New keys win over old one, del it from object and skip migration
                    del item[property]
            
            if property not in item:
                continue
            
            value = item[property]
            del item[property]
            
            for prefix in prefixs:
                new_property = '%s_%s' % (prefix, property)
                item[new_property] = value
        # Now, duplicate edition/notification contacts in view contacts if not already allowed
        _default_view_contacts = get_default_value(ITEM_TYPE.HOSTS, 'view_contacts')[0] or VIEW_CONTACTS_DEFAULT_VALUE.NOBODY
        
        for property in properties:
            view_key = 'view_%s' % property
            for prefix in must_view_prefixs:
                new_property = '%s_%s' % (prefix, property)
                if item.get(new_property, '') and (_default_view_contacts == VIEW_CONTACTS_DEFAULT_VALUE.NOBODY or item.get(view_key, '')):
                    item[view_key] = add_unique_value_and_handle_plus_and_null(item.get(view_key, ''), item[new_property], handle_plus=True)


def migrate_contact_contactgroups(contacts):
    # Contacts and contact templates must be changed from the contacts into:
    # * contact_groups/contactgroups
    # This only applies if the contact or contact template does not have any defined contactgroups
    for contact in contacts:
        if contact.get('contact_groups', None):
            contact['contactgroups'] = contact.get('contact_groups')
            del contact['contact_groups']


def _migrate_nb_elements_in_last_synchronization():
    mongo_component = component_manager.get_mongo_component()
    for last_sync_col_name in ("last_synchronization", "last_synchronizations"):
        last_synchronization_col = mongo_component.get_collection(last_sync_col_name)
        _last_syncs = list(last_synchronization_col.find())
        for last_synchronization in _last_syncs:
            saved_values = last_synchronization.get('saved', {})
            if saved_values is None:
                continue
            for item_class, value in saved_values.items():
                # if value is an int, we need to set this as dict with an nb_elements keys
                if isinstance(value, int):
                    last_synchronization['saved'][item_class] = {
                        'warning'    : 0,
                        'error'      : 0,
                        'nb_elements': value,
                    }
        try:
            last_synchronization_col.replace_many(_last_syncs)
        except (TypeError, ValueError) as e:
            logger.info("[migrate_nb_elements_in_last_synchronization] %s" % traceback.format_exc())


def _add_source_name_in_discovery_confs():
    mongo_component = component_manager.get_mongo_component()
    discovery_confs_col = mongo_component.col_discovery_confs
    migration_needed_confs = discovery_confs_col.find({'source_name': None})
    for disco_conf in migration_needed_confs:
        disco_conf['source_name'] = 'discovery'
        discovery_confs_col.save(disco_conf)


def _migrate_database_part_protected_properties():
    # type: () -> NoReturn
    logger_protected_fields.info(u'Check that the base encryption is the one expected in the configuration')
    start_time = time.time()
    
    conf_protected_fields_keyfile = app.conf.protect_fields__encryption_keyfile
    mongo_component = component_manager.get_mongo_component()
    col = mongo_component.col_synchronizer_info
    current_db_encryption_configuration = col.find_one({u'_id': u'protected_fields_info'})
    
    if current_db_encryption_configuration is None:
        current_db_encryption_configuration = {}
    
    db_is_currently_encrypted = current_db_encryption_configuration.get(u'protect_fields__activate_database_encryption', False)
    
    if db_is_currently_encrypted is False and not app.conf.protect_fields__activate_encryption:
        logger_protected_fields.info(u'The base is not encrypted and there is no need to encrypt it : nothing to do')
        return
    
    # This case is already managed in SynchronizerDaemon:
    if db_is_currently_encrypted == u'1':
        if not conf_protected_fields_keyfile:
            logger_protected_fields.error(u'The database is currently encrypted and the parameter "protect_fields__encryption_keyfile" is missing in configuration file. The synchronizer cannot start')
            logger_protected_fields.error(u'Please set the parameter "protect_fields__encryption_keyfile" to point to the right key file in /etc/shinken/synchronizer.cfg.')
            logger_protected_fields.error(u'It might be a typing mistake, or the parameter could be overloaded in another configuration file')
            sys.exit(1)
    
    try:
        complete_key = open(app.conf.protect_fields__encryption_keyfile).read().strip()
        key_value = complete_key[complete_key.index(u'|') + 1:]
        key_name = complete_key[:complete_key.index(u'|')].strip().decode(u'utf8', u'ignore')
        key_file_hash = sha256(key_value).hexdigest()
    except (OSError, IOError) as e:
        logger_protected_fields.error(u'Cannot read the protected fields secret file "%s" : %s ' % (app.conf.protect_fields__encryption_keyfile, unicode(e)))
        logger_protected_fields.error(u'Make sure the file exists and that the user "shinken" is allowed to read it')
        logger_protected_fields.error(u'The synchronizer will not start because it will not be able to process encrypted data.')
        sys.exit(1)
    except ValueError:
        logger_protected_fields.error(u'Cannot read the protected fields secret file "%s"' % (app.conf.protect_fields__encryption_keyfile))
        logger_protected_fields.error(u'The keyfile seems to be corrupted. If you have an export available, you can use it to restore the correct key')
        logger_protected_fields.error(u'using the command shinken-protected-fields-keyfile-restore and restart the synchronizer.')
        logger_protected_fields.error(u'The synchronizer will not start in order not to corrupt data.')
        sys.exit(1)
    
    configuration_protected_fields = {
        u'_id'                                         : u'protected_fields_info',
        u'protect_fields__substrings_matching_fields'  : [substring.strip() for substring in app.conf.protect_fields__substrings_matching_fields.split(u',') if substring.strip()],
        u'protect_fields__activate_database_encryption': app.conf.protect_fields__activate_encryption,
        u'protect_fields__encryption_keyfile_hash'     : key_file_hash,
        u'protect_fields__encryption_key_name'         : key_name
    }
    
    if key_file_hash != current_db_encryption_configuration.get(u'protect_fields__encryption_keyfile_hash') or key_name != current_db_encryption_configuration.get(u'protect_fields__encryption_key_name'):
        key_already_extracted = False
    else:
        key_already_extracted = current_db_encryption_configuration.get(u'extracted_key', False)
    
    if _need_protected_fields_migration(current_db_encryption_configuration, configuration_protected_fields):
        _migrate_protected_fields(current_db_encryption_configuration, configuration_protected_fields)
        _migrate_history(current_db_encryption_configuration, configuration_protected_fields)
        
        if db_is_currently_encrypted != configuration_protected_fields[u'protect_fields__activate_database_encryption']:
            _migrate_protected_properties_sources_config(app.conf.protect_fields__activate_encryption)
        
        from_data = {
            u'protect_fields__substrings_matching_fields'  : current_db_encryption_configuration.get(u'protect_fields__substrings_matching_fields', []),
            u'protect_fields__activate_database_encryption': current_db_encryption_configuration.get(u'protect_fields__activate_database_encryption', u''),
            u'protect_fields__encryption_keyfile_hash'     : current_db_encryption_configuration.get(u'protect_fields__encryption_keyfile_hash', u''),
            u'protect_fields__encryption_key_name'         : current_db_encryption_configuration.get(u'protect_fields__encryption_key_name', u''),
            u'extracted_key'                               : current_db_encryption_configuration.get(u'extracted_key', False)
        }
        
        to_data = {
            u'protect_fields__substrings_matching_fields'  : configuration_protected_fields.get(u'protect_fields__substrings_matching_fields', []),
            u'protect_fields__activate_database_encryption': configuration_protected_fields.get(u'protect_fields__activate_database_encryption', u''),
            u'protect_fields__encryption_keyfile_hash'     : configuration_protected_fields.get(u'protect_fields__encryption_keyfile_hash', u''),
            u'protect_fields__encryption_key_name'         : configuration_protected_fields.get(u'protect_fields__encryption_key_name', u''),
            u'extracted_key'                               : key_already_extracted
        }
        this_migration = {u'date': str(datetime.datetime.now()), u'from': from_data, u'to': to_data}
        last_migrations = current_db_encryption_configuration.get(u'last_migrations', [])
        last_migrations.insert(0, this_migration)
        configuration_protected_fields[u'last_migrations'] = last_migrations[:5]
        configuration_protected_fields[u'extracted_key'] = key_already_extracted
        col.save(configuration_protected_fields)
        
        logger_protected_fields.info(u'Done in %0.2fs' % (time.time() - start_time))


def _need_protected_fields_migration(db_protected_fields_info, config_protected_fields_info):
    # type: (Dict, Dict) -> bool
    db_substring = set(db_protected_fields_info.get(u'protect_fields__substrings_matching_fields', []))
    config_substring = set(config_protected_fields_info.get(u'protect_fields__substrings_matching_fields', []))
    
    db_encryption_is_enabled = db_protected_fields_info.get(u'protect_fields__activate_database_encryption', False)
    config_encryption_is_enabled = config_protected_fields_info.get(u'protect_fields__activate_database_encryption', False)
    
    if db_protected_fields_info.get(u'protect_fields__encryption_key_name') != config_protected_fields_info.get(u'protect_fields__encryption_key_name') and db_encryption_is_enabled:
        logger.critical(u'The Protected Field key file was modified.')
        logger.critical(u'The synchronizer will not start in order not to corrupt data.')
        logger.critical(u'The name of the key in the configuration file is : %s' % config_protected_fields_info.get(u'protect_fields__encryption_key_name'))
        logger.critical(u'The name of the key in the database is           : %s' % db_protected_fields_info.get(u'protect_fields__encryption_key_name'))
        logger.critical(u'Please restore the correct key using the command "shinken-protected-fields-keyfile-restore"')
        logger.critical(u'The Protected Field key file was modified.')
        sys.exit(1)
    
    if db_protected_fields_info.get(u'protect_fields__encryption_keyfile_hash', u'') != config_protected_fields_info.get(u'protect_fields__encryption_keyfile_hash', u'') and db_encryption_is_enabled:
        logger.critical(u'The Protected Field key file was modified.')
        logger.critical(u'The synchronizer will not start in order not to corrupt data.')
        logger.critical(u'Please restore the correct key named [%s] using the command "shinken-protected-fields-keyfile-restore"' % db_protected_fields_info.get(u'protect_fields__encryption_key_name', u'UNKNOWN NAME'))
        sys.exit(1)
    
    if (db_encryption_is_enabled == config_encryption_is_enabled) and (db_substring == config_substring):
        logger_protected_fields.info(u'Encryption is already activated with the right parameters : nothing to do')
        return False
    
    elif db_encryption_is_enabled and not config_encryption_is_enabled:
        logger_protected_fields.info(u'The base is encrypted but the configuration indicates that it should not be encrypted. Decryption in progress')
    elif not db_encryption_is_enabled and config_encryption_is_enabled:
        logger_protected_fields.info(u'The base is not encrypted but the configuration indicates that it should be encrypted. Encryption in progress')
    elif (db_encryption_is_enabled == config_encryption_is_enabled) and (db_substring != config_substring):
        logger_protected_fields.info(u'The base is encrypted but the substrings have been updated. Update in progress')
        logger_protected_fields.info(u'Previous substring (in database)  : %s' % u', '.join(sorted(db_substring)))
        logger_protected_fields.info(u'Next substring (in configuration) : %s' % u', '.join(sorted(config_substring)))
    
    return True


def _migrate_protected_properties_sources_config(db_cipher_activated):
    # type: (bool) -> NoReturn
    
    start_time = time.time()
    logger_protected_fields.info(u'Source configuration processing')
    
    db_cipher = DatabaseCipher(True, [], app.conf.protect_fields__encryption_keyfile)
    
    mongo_component = component_manager.get_mongo_component()
    collection = mongo_component.get_collection(u'sources-configuration')
    logger_protected_fields.info(u'%5s sources to proceed' % len(app.sources))
    for source in app.sources:
        name = source.get_name()
        source_conf = collection.find_one({u'_id': name})
        if source_conf:
            for os_type, fields in source.get_configuration_fields().iteritems():
                for field_name, properties in fields.iteritems():
                    value = source_conf[os_type][field_name]
                    if value and properties[u'protected']:
                        if db_cipher_activated:
                            value = db_cipher.cipher_value(value)
                        else:
                            value = db_cipher.decipher_value(value)
                    source_conf[os_type][field_name] = value
            collection.save(source_conf)
    
    logger_protected_fields.info(u'Source configuration processing done in %0.2f' % (time.time() - start_time))


def _migrate_protected_fields(src_config, dst_config):
    # type: (Dict, Dict) -> NoReturn
    
    start_time = time.time()
    logger_protected_fields.info(u'Protected fields processing')
    nb_elements = 0
    
    substring_to_decipher = list(set(src_config.get(u'protect_fields__substrings_matching_fields', [])))
    substring_to_cipher = list(set(dst_config.get(u'protect_fields__substrings_matching_fields', [])))
    
    _db_encryption_is_enabled = src_config.get(u'protect_fields__activate_database_encryption', False)
    _config_encryption_is_enabled = dst_config.get(u'protect_fields__activate_database_encryption', False)
    
    if _db_encryption_is_enabled and not _config_encryption_is_enabled:
        # Decrypt all
        substring_to_cipher = []
    elif not _db_encryption_is_enabled and _config_encryption_is_enabled:
        # Crypt all
        substring_to_decipher = []
    
    logger_protected_fields.debug(u'Substring to decrypt:[%s]' % substring_to_decipher)
    logger_protected_fields.debug(u'Substring to encrypt:[%s]' % substring_to_cipher)
    
    db_decipher = DatabaseCipher(True, substring_to_decipher, app.conf.protect_fields__encryption_keyfile)
    db_cipher = DatabaseCipher(True, substring_to_cipher, app.conf.protect_fields__encryption_keyfile)
    
    # Browse all relevant collection
    filter_col_name = set((u'configuration', u'data', u'merge_from_sources', u'newelements', u'changeelements'))
    mongo_component = component_manager.get_mongo_component()
    all_collection_names = mongo_component.list_name_collections(include_system_collections=False)
    collections_to_process = [{u'collection_name': c, u'class_name': u'%ss' % c.split(u'-')[-1]} for c in all_collection_names if c.split(u'-')[0] in filter_col_name]
    collections_to_process.extend([{u'collection_name': u'%s_confs' % source.source_name, u'class_name': ITEM_TYPE.REMOTE_SYNCHRONIZER} for source in app.sources if source.module_type == u'synchronizer-collector-linker'])
    for collection_to_process in collections_to_process:
        collection_name = collection_to_process[u'collection_name']
        logger_protected_fields.debug(u'Processing collection:[%s]' % collection_name)
        collection = mongo_component.get_collection(collection_name)
        class_name = collection_to_process[u'class_name']
        items = collection.find({})
        items_to_replace = []
        detailled_collection_name = collection_name.split(u'-')
        if detailled_collection_name[0] == u'configuration':
            item_state = detailled_collection_name[1]
            logger_protected_fields.info(u'%5s elements to proceed for state %s and class %s' % (len(items), item_state, class_name))
        elif detailled_collection_name[0] == u'data':
            _source = u'-'.join(detailled_collection_name[1:-1])
            logger_protected_fields.info(u'%5s elements to proceed for source %s and class %s' % (len(items), _source, class_name))
        else:
            logger_protected_fields.info(u'%5s elements to proceed for state %s and class %s' % (len(items), detailled_collection_name[0], class_name))
        
        item_state = None
        if collection_name.startswith(u'changeelements'):
            item_state = ITEM_STATE.CHANGES
        for item in items:
            item_type = get_type_item_from_class(class_name, item)
            working_item = copy.deepcopy(item)
            try:
                if _db_encryption_is_enabled:
                    working_item = db_decipher.uncipher(working_item, item_type=item_type, item_state=item_state)
                if _config_encryption_is_enabled:
                    working_item = db_cipher.cipher(working_item, item_type=item_type, item_state=item_state)
                if cmp(item, working_item):
                    items_to_replace.append(working_item)
            except (TypeError, ValueError) as e:
                logger_protected_fields.error(u'error in before or after save, item skip')
                logger_protected_fields.print_stack()
            nb_elements += 1
        
        try:
            # pymongo splits the operation into multiple bulk executes() according to this link :
            # http://api.mongodb.com/python/2.7rc1/examples/bulk.html#bulk-insert
            collection.replace_many(items_to_replace)
            logger_protected_fields.debug(u'Bulk Done')
        except BulkWriteError as bulk_exception:
            logger_protected_fields.error(u'BulkWriteError in collection [%s]:[%s]' % (collection.get_name(), pformat(bulk_exception.details)))
        except InvalidOperation as mongo_exception:
            # do not log error when mongo tel us that there is nothing to do
            if not u'No operations to execute' in mongo_exception.message:
                logger_protected_fields.error(u'InvalidOperation in collection [%s]:[%s]' % (collection.get_name(), pformat(mongo_exception.message)))
    
    logger_protected_fields.info(u'Protected fields : processed %s elements done in %0.2f' % (nb_elements, time.time() - start_time))


def _migrate_history(db_infos, config_infos):
    # type: (Dict, Dict) -> NoReturn
    
    start_time = time.time()
    logger_protected_fields.info(u'History processing')
    
    substring_to_decipher = set(db_infos.get(u'protect_fields__substrings_matching_fields', []))
    substring_to_cipher = set(config_infos.get(u'protect_fields__substrings_matching_fields', []))
    
    _db_encryption_is_enabled = db_infos.get(u'protect_fields__activate_database_encryption', False)
    _config_encryption_is_enabled = config_infos.get(u'protect_fields__activate_database_encryption', False)
    
    if _db_encryption_is_enabled and not _config_encryption_is_enabled:
        # Decrypt all
        substring_to_cipher = set()
    elif not _db_encryption_is_enabled and _config_encryption_is_enabled:
        # Crypt all
        substring_to_decipher = set()
    
    substring_to_uncipher_str = u','.join(substring_to_decipher)
    substring_to_cipher_str = u','.join(substring_to_cipher)
    
    logger_protected_fields.debug(u'Substring to uncrypt[%s]' % substring_to_uncipher_str)
    logger_protected_fields.debug(u'Substring to encrypt:[%s]' % substring_to_cipher_str)
    
    db_uncipher = DatabaseCipher(True, substring_to_uncipher_str, app.conf.protect_fields__encryption_keyfile)
    db_cipher = DatabaseCipher(True, substring_to_cipher_str, app.conf.protect_fields__encryption_keyfile)
    
    # Browse all relevant collection
    collection_name = u'history'
    logger_protected_fields.debug(u'Processing collection:[%s]' % collection_name)
    mongo_component = component_manager.get_mongo_component()
    collection = mongo_component.get_collection(collection_name)
    items_to_replace = []
    histories = collection.find()
    logger_protected_fields.info(u'%5s history entries to process' % len(histories))
    for history_entry in histories:
        working_history = copy.deepcopy(history_entry)
        try:
            if _db_encryption_is_enabled:
                working_history = db_uncipher.uncipher_history_entry(working_history)
            if _config_encryption_is_enabled:
                working_history = db_cipher.cipher_history_entry(working_history)
            if cmp(history_entry, working_history):
                items_to_replace.append(working_history)
        except (TypeError, ValueError) as e:
            logger_protected_fields.error(u'error in before or after save, history skip')
            logger_protected_fields.print_stack()
    
    try:
        # pymongo splits the operation into multiple bulk executes() according to this link :
        # http://api.mongodb.com/python/2.7rc1/examples/bulk.html#bulk-insert
        collection.replace_many(items_to_replace)
        logger_protected_fields.debug(u'Bulk Done')
    except BulkWriteError as bulk_exception:
        logger_protected_fields.error(u'BulkWriteError in collection [%s]:[%s]' % (collection.get_name(), pformat(bulk_exception.details)))
    except InvalidOperation as mongo_exception:
        # do not log error when mongo tel us that there is nothing to do
        if not u'No operations to execute' in mongo_exception.message:
            logger_protected_fields.error(u'InvalidOperation in collection [%s]:[%s]' % (collection.get_name(), pformat(mongo_exception.message)))
    
    logger_protected_fields.info(u'History processing done in %0.2f' % (time.time() - start_time))


# migration for data pre V02.04.00
# Look for all hosts/hosttemplates/cluster/service/servicetemplate and change the "contacts"
# property into a view/notify/edit lists
def _migrate_hosts_contacts_properties_in_db():
    t0 = time.time()
    migrated = False
    for item_type in (ITEM_TYPE.HOSTS, ITEM_TYPE.HOSTTPLS, ITEM_TYPE.CLUSTERS, ITEM_TYPE.CLUSTERTPLS):
        for item_state in (ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION):
            to_migrate_items = migration_datamanager.find_items(item_type, item_state, where={'$or': [{'contacts': {'$exists': True}}, {'contact_groups': {'$exists': True}}]})
            if len(to_migrate_items) == 0:
                continue
            migrated = True
            logger.info(u'Migrating old contacts to the view/notification/edit format. item_type:[%s] item_state:[%s] number of objects:[%s]' % (item_type, item_state, len(to_migrate_items)))
            migrate_hosts_contacts_properties(app, to_migrate_items)
            for item in to_migrate_items:
                migration_datamanager.save_item(item, item_type=item_type, item_state=item_state)
    if migrated:
        logger.info(u'Migration was done in %.2fs' % (time.time() - t0))


def _migrate_contacts_contactgroups_properties_in_db():
    for item_type in (ITEM_TYPE.CONTACTS, ITEM_TYPE.CONTACTTPLS):
        for item_state in (ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION):
            _change_properties_name(item_type, item_state, 'contact_groups', 'contactgroups')
        
        for source_name in (source.get_name() for source in app.sources):
            _change_properties_name(item_type, ITEM_STATE.RAW_SOURCES, 'contact_groups', 'contactgroups', item_source=source_name)


def _change_properties_name(item_type, item_state, old_property, new_property, item_source=None):
    if item_source:
        to_migrate_items = migration_datamanager.find_items(item_type, item_state, item_source=item_source, where={old_property: {'$exists': True}})
    else:
        to_migrate_items = migration_datamanager.find_items(item_type, item_state, where={old_property: {'$exists': True}})
    if len(to_migrate_items) == 0:
        return
    t0 = time.time()
    
    for item in to_migrate_items:
        value = item.get(old_property)
        del item[old_property]
        item[new_property] = value
        if item_source:
            migration_datamanager.save_item(item, item_type=item_type, item_state=item_state, item_source=item_source)
        else:
            migration_datamanager.save_item(item, item_type=item_type, item_state=item_state)
    
    logger.info('[Synchronizer] DATA MIGRATION   was done in %.2fs' % (time.time() - t0))


# Migrate all host/cluster in V02.04.00 in V02.05.00 version
def _check_old_host_to_cluster_boolean():
    for item_state in (ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION):
        for item_type in (ITEM_TYPE.CLUSTERS, ITEM_TYPE.HOSTS):
            clusters = migration_datamanager.find_items(item_type, item_state, where={'check_command': {'$regex': '^bp_rule'}})
            for cluster in clusters:
                logger.info('[Synchronizer] DATA MIGRATION   we are migrating old cluster element %s with a is_cluster boolean in the area %s' % (cluster.get('host_name', 'UNKNOWN'), item_state))
                migrate_to_clusters(cluster)
                migration_datamanager.save_item(cluster, item_type=ITEM_TYPE.CLUSTERS, item_state=item_state)


# Migrate objects from 2.5 / early 2.6 lacking "is_cluster" property
def _check_clusters_to_cluster_boolean():
    for item_state in (ITEM_STATE.STAGGING, ITEM_STATE.PRODUCTION):
        for item_type in (ITEM_TYPE.ALL_HOST_CLASS):
            clusters = migration_datamanager.find_items(item_type, item_state, where={'bp_rule': {'$exists': True}, "is_cluster": {'$exists': False}})
            for cluster in clusters:
                logger.info('[Synchronizer] DATA MIGRATION   we are migrating cluster element %s with a is_cluster boolean in the area %s' % (cluster.get('host_name', 'UNKNOWN'), item_state))
                migrate_to_clusters(cluster)
                migration_datamanager.save_item(cluster, item_type=ITEM_TYPE.CLUSTERS, item_state=item_state)


def migrate_to_clusters(item):
    cmd = item.get('check_command')
    if cmd and cmd.startswith('bp_rule'):
        if '!' in cmd:
            args = '!'.join(cmd.split('!')[1:])
            if args:
                item['bp_rule'] = args
        del item['check_command']
    if 'bp_rule' in item:
        item['is_cluster'] = '1'


def _remove_basic_element_from_database():
    # We remove the shinken-host and generic-service now we add it in synchronier-import module so UI configuration didn't see it.
    mongo_component = component_manager.get_mongo_component()
    col = mongo_component.get_collection('configuration-production-host')
    col.remove({'_id': 'f7127b3a5ae011e58d51080027f08538'})
    col.remove({'name': 'shinken-host'})
    col = mongo_component.get_collection('configuration-stagging-host')
    col.remove({'_id': 'f7127b3a5ae011e58d51080027f08538'})
    col.remove({'name': 'shinken-host'})
    col = mongo_component.get_collection('configuration-production-service')
    col.remove({'_id': '57ecf196c6a311e5958f08002709eeca'})
    col = mongo_component.get_collection('configuration-stagging-service')
    col.remove({'_id': '57ecf196c6a311e5958f08002709eeca'})


def _remove_old_tmp_collection():
    mongo_component = component_manager.get_mongo_component()
    all_collection_names = mongo_component.list_name_collections(include_system_collections=False)
    for col_name in all_collection_names:
        if col_name.startswith('tmp-'):
            mongo_component.drop_collection(col_name)
