#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

from dataprovider import DataProvider
from shinken.misc.type_hint import TYPE_CHECKING
from shinkensolutions.indexed_collection.collection import IndexedCollection
from ..def_items import DEF_ITEMS, ITEM_TYPE
from ...dao import DataException
from ...dao.def_items import METADATA
from ...dao.helpers import split_list_attr, set_item_property

if TYPE_CHECKING:
    from shinken.misc.type_hint import NoReturn, Dict, Any
    from ..callbacks.callback_history_info import HistoryEntry


# TODO implement lookup parameters in get functions
class DataProviderMemory(DataProvider):
    
    def __init__(self, data, index_double_links=True, save_history=False):
        # type: (Dict[unicode,Dict[unicode, Any]], bool, bool) -> NoReturn
        self.can_save_history = save_history
        self.history = []
        # will contain all objects by types and by state
        self.data = {}
        for item_state in data.iterkeys():
            self.data[item_state] = {}
            for item_type, def_item in DEF_ITEMS.iteritems():
                self.data[item_state][item_type] = DataProviderMemory._collection_factory(item_type, index_double_links)
                self.data[item_state][item_type].extend(data[item_state].get(item_type, []))
    
    
    @staticmethod
    def _collection_factory(item_type, index_double_links=True):
        collection = IndexedCollection(hash_key='_id')
        collection.add_index('_id')
        collection.add_index(DEF_ITEMS[item_type]['key_name'])
        collection.add_index("%s_lower" % DEF_ITEMS[item_type]['key_name'], getter=lambda item: item.get(DEF_ITEMS[item_type]['key_name'], '').lower())
        
        collection.add_index('_SYNC_KEYS', on_list=True)
        collection.add_index('need_recompute_double_link')
        collection.add_index('to_merge')
        if index_double_links:
            for double_link_def in DEF_ITEMS[item_type].get('double_links', {}):
                collection.add_index('double_link_to_%s' % double_link_def['of_type'], on_list=True, getter=lambda item: [x.lower() for x in split_list_attr(item, double_link_def['my_attr'])])
        return collection
    
    
    def find_item_by_name(self, item_name, item_type='', item_state='', item_source='', lookup=None):
        if not item_name:
            raise DataException('[%s] find_item_by_name : Please set item_name' % self.__class__.__name__)
        if not item_type:
            raise DataException('[%s] find_item_by_name : Please set item_type' % self.__class__.__name__)
        if not item_state:
            raise DataException('[%s] find_item_by_name : Please set item_state' % self.__class__.__name__)
        if item_state not in self.data:
            return None
        return self.data[item_state][item_type].find_one({u'%s_lower' % DEF_ITEMS[item_type]['key_name']: item_name.lower()})
    
    
    def find_item_by_name_with_case_sensitive(self, item_name, item_type='', item_state='', item_source='', lookup=None):
        if not item_name:
            raise DataException('[%s] find_item_by_name : Please set item_name' % self.__class__.__name__)
        if not item_type:
            raise DataException('[%s] find_item_by_name : Please set item_type' % self.__class__.__name__)
        if not item_state:
            raise DataException('[%s] find_item_by_name : Please set item_state' % self.__class__.__name__)
        if item_state not in self.data:
            return None
        return self.data[item_state][item_type].find_one({DEF_ITEMS[item_type]['key_name']: item_name})
    
    
    def find_item_by_id(self, item_id, item_type='', item_state='', item_source='', lookup=None):
        if not item_id:
            raise DataException('[%s] find_item_by_id : Please set item_id' % self.__class__.__name__)
        if not item_type:
            raise DataException('[%s] find_item_by_id : Please set item_type' % self.__class__.__name__)
        if not item_state:
            raise DataException('[%s] find_item_by_id : Please set item_state' % self.__class__.__name__)
        if item_state not in self.data:
            return None
        return self.data[item_state][item_type].find_one({'_id': item_id})
    
    
    def find_items(self, item_type, item_state='', item_source='', where=None, lookup=None):
        if not item_type:
            raise DataException('[%s] find_items : Please set item_type' % self.__class__.__name__)
        if not item_state:
            raise DataException('[%s] find_items : Please set item_state' % self.__class__.__name__)
        
        if item_type == ITEM_TYPE.ELEMENTS:
            item_type = DEF_ITEMS.keys()
        if isinstance(item_type, basestring):
            item_type = [item_type]
        if isinstance(item_state, basestring):
            item_state = [item_state]
        
        items = []
        for current_state in item_state:
            if current_state not in self.data:
                continue
            for current_type in item_type:
                items_for_type_and_state = self.data[current_state][current_type].find(where)
                for item in items_for_type_and_state:
                    METADATA.update_metadata(item, METADATA.ITEM_TYPE, current_type)
                    METADATA.update_metadata(item, METADATA.STATE, current_state)
                items.extend(items_for_type_and_state)
        return items
    
    
    def find_merge_state_items(self, item_type, item_states, item_source='', where=None, lookup=None):
        _item_states = list(item_states)
        _item_states.reverse()
        items = {}
        for _item_state in _item_states:
            _items = self.find_items(item_type, _item_state, item_source, where, lookup)
            for _item in _items:
                METADATA.update_metadata(_item, METADATA.STATE, _item_state)
                items[_item['_id']] = _item
        return items.values()
    
    
    def find_double_link_items(self, item_link_name, item_type_link_to, item_type='', item_state='', item_source=''):
        if not item_link_name:
            raise DataException('[%s] find_double_link_items : Please set item_link_name' % self.__class__.__name__)
        if not item_type_link_to:
            raise DataException('[%s] find_double_link_items : Please set item_type_link_to' % self.__class__.__name__)
        if not item_type:
            raise DataException('[%s] find_double_link_items : Please set item_type' % self.__class__.__name__)
        if not item_state:
            raise DataException('[%s] find_double_link_items : Please set item_state' % self.__class__.__name__)
        
        item_link_name = item_link_name.lower().strip()
        item_link_name = item_link_name[1:] if item_link_name.startswith('+') else item_link_name
        return self.find_items(item_type, item_state, where={'double_link_to_%s' % item_type_link_to: item_link_name})
    
    
    def save_item(self, item, item_type='', item_state='', item_source='', **kwargs):
        if not item:
            raise DataException('[%s] save_item : Please set item' % self.__class__.__name__)
        if not item_type:
            raise DataException('[%s] save_item : Please set item_type' % self.__class__.__name__)
        if not item_state:
            raise DataException('[%s] save_item : Please set item_state' % self.__class__.__name__)
        
        item = item.copy()
        item_id = item['_id']
        old_item = self.find_item_by_id(item_id, item_type, item_state)
        if old_item:
            self.data[item_state][item_type].regenerate_index(item)
            
            old_item.clear()
            for param, value in item.iteritems():
                old_item[param] = value
        else:
            self.data[item_state][item_type].append(item)
        
        return item
    
    
    def delete_item(self, item, item_type='', item_state='', item_source=''):
        if not item:
            raise DataException('[%s] delete_item : Please set item' % self.__class__.__name__)
        if not item_type:
            raise DataException('[%s] delete_item : Please set item_type' % self.__class__.__name__)
        if not item_state:
            raise DataException('[%s] delete_item : Please set item_state' % self.__class__.__name__)
        # dataprovider don't raise exception when remove an inexisting item
        try:
            self.data[item_state][item_type].remove({'_id': item['_id']})
        except:
            # raise exception only if error is not that the item is already deleted
            collection = self.data.get(item_state, {}).get(item_type, [])
            if collection and collection.find_one({'_id': item['_id']}):
                raise
    
    
    def count_items(self, item_type, item_state='', item_source='', where=None):
        if not item_type:
            raise DataException('[%s] count_items : Please set item_type' % self.__class__.__name__)
        if not item_state:
            raise DataException('[%s] count_items : Please set item_state' % self.__class__.__name__)
        
        return self.data[item_state][item_type].count(where)
    
    
    def update_all_item(self, item_state, update):
        for item_type in DEF_ITEMS.iterkeys():
            for item in self.find_items(item_type, item_state):
                for update_type, update_value in update.iteritems():
                    if update_type == '$set':
                        for item_prop, item_value in update_value.iteritems():
                            set_item_property(item, item_prop, item_value)
                    else:
                        raise DataException('[%s] update_all_item : unsupported update_type [%s]' % (self.__class__.__name__, update_type))
    
    
    def copy_state(self, from_state, to_state):
        for item_type in DEF_ITEMS.iterkeys():
            for item in self.find_items(item_type, from_state):
                self.save_item(item.copy(), item_type, to_state)
    
    
    def find_in_transaction(self, item_id, item_type, item_state):
        return self.find_item_by_id(item_id, item_type, item_state)
    
    
    def start_transaction(self, transaction_uuid):
        pass
    
    
    def commit_transaction(self, transaction_uuid):
        pass
    
    
    def save_history(self, history_entry):
        # type: (HistoryEntry) -> HistoryEntry
        if self.can_save_history:
            self.history.append(history_entry)
        return history_entry
    
    
    def find_history_for_item(self, item_id, item_type):
        # type: (unicode, unicode) -> list
        if self.can_save_history:
            return [h for h in self.history if h[u'item_uuid'] == item_id and h[u'item_type'] == item_type]
        return []
