#!/usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.host import VIEW_CONTACTS_DEFAULT_VALUE
from shinken.objects.item import PERIOD_PROPERTIES
from shinken.objects.service import PROPERTIES_RETRIEVABLE_FROM_HOST
from shinken.property import BoolProp
from shinken.property import none_object as property_no_default_object
from .def_items import DEF_ITEMS, METADATA, NO_HERITABLE_ATTR, NOT_TO_LOOK, STOP_INHERITANCE_VALUES, LINKIFY_MANAGE_STATES, INTERNAL_ATTR, ITEM_TYPE, PROP_DEFAULT_VALUE, VALUE_FORCE_DEFAULT, ITEM_STATE, HISTORY_ACTION

if TYPE_CHECKING:
    from ..synchronizerdaemon import Synchronizer
    from .datamanagerV2 import DataManagerV2
    from .items.baseitem import BaseItem
    from shinken.misc.type_hint import NoReturn, Dict, List, Str, Union, Any, Optional


class ShinkenDatabaseConsistencyError(Exception):
    def __init__(self, link='', item_id='', item_type='', item_state='', asking_state=''):
        error_message = 'the item [%s:%s:%s] have a inconsistency link for asking state [%s] : [%s]' % (item_id, item_type, item_state, asking_state, link)
        super(ShinkenDatabaseConsistencyError, self).__init__(error_message)


def compute_diff(item1, item2, item_type=None):
    changed = []
    renamed_properties = {}
    if item_type:
        renamed_properties = getattr(DEF_ITEMS[item_type]['class'], 'old_properties', {})
    
    # new set with elements from both item1 properties and item2 properties and not NOT_TO_LOOK properties
    item1_and_item2_properties = (set(item1.keys()) | set(item2.keys())) - NOT_TO_LOOK
    for item_property in item1_and_item2_properties:
        item_property = renamed_properties.get(item_property, item_property)
        item1_value = item1.get(item_property, '')
        item2_value = item2.get(item_property, '')
        
        # catch the same alias for STOP_INHERITANCE_VALUES
        if item1_value in STOP_INHERITANCE_VALUES and item2_value in STOP_INHERITANCE_VALUES:
            continue
        
        property_def = get_property_def(item_type, item_property)
        if property_def and property_def.merging == 'join' and isinstance(item1_value, basestring) and isinstance(item2_value, basestring):
            if set(split_and_strip_list(item1_value)) != set(split_and_strip_list(item2_value)):
                changed.append(item_property)
        elif item1_value != item2_value:
            changed.append(item_property)
    
    return changed


def add_unique_value_and_handle_plus_and_null(initial_values, to_add_values, handle_plus, separator=','):
    initial_values = initial_values.strip()
    to_add_values = to_add_values.strip()
    has_plus = False
    if initial_values.startswith('+'):
        has_plus = True
        initial_values = initial_values[1:]
    if to_add_values.startswith('+'):
        has_plus = True
        to_add_values = to_add_values[1:]
    has_plus = handle_plus and has_plus
    
    initial_values = split_and_strip_list(initial_values, separator)
    to_add_values = split_and_strip_list(to_add_values, separator)
    
    result_values = initial_values
    for to_add_value in to_add_values:
        to_add_value = to_add_value[1:].strip() if to_add_value.startswith('+') else to_add_value
        if to_add_value not in result_values:
            result_values.append(to_add_value)
    
    if 'null' in result_values:
        return 'null'
    
    result_values = separator.join(result_values)
    return '+' + result_values if has_plus else result_values


def safe_add_to_dict(dictionary, key, value):
    # type: (Dict, unicode, Any) -> None
    if key not in dictionary:
        dictionary[key] = [value]
    else:
        dictionary[key].append(value)


def safe_add_to_dict_but_uniq(dictionary, key, value):
    # type: (Dict, unicode, Any) -> None
    if key not in dictionary:
        dictionary[key] = [value]
    elif value not in dictionary[key]:
        dictionary[key].append(value)


def safe_add_to_sub_dict_but_uniq(dictionary, key, sub_dict_key, value):
    # type: (Dict[Any, Dict[Any, List[Any]]], Any, Any, Any) -> None
    if key not in dictionary:
        dictionary[key] = {}
    if sub_dict_key not in dictionary[key]:
        dictionary[key][sub_dict_key] = []
    safe_add_to_dict_but_uniq(dictionary[key], sub_dict_key, value)


def safe_extend_to_dict(dictionary, key, value):
    # type: (Dict, unicode, List) -> None
    if key not in dictionary:
        dictionary[key] = value
    else:
        dictionary[key].extend(value)


def split_and_strip_list(str_list, separator=','):
    if isinstance(str_list, list):
        return str_list
    if str_list:
        return [i.strip() for i in str_list.split(separator) if i.strip()]
    else:
        return []


def split_list_attr(item, item_attr):
    # Remove void items and be sure to have strip elements as the data can come from the import and so have space and such void values
    target_attr_value = item.get(item_attr, None)
    if target_attr_value:
        target_attr_value = target_attr_value[1:] if target_attr_value.startswith('+') else target_attr_value
        return [i.strip() for i in target_attr_value.split(',') if i.strip()]
    return []


def get_name_from_type(item_type, item):
    if item is None:
        return ''
    name = ''
    try:
        name = item.get(DEF_ITEMS[item_type]['key_name'], '')
    except TypeError as e:
        logger.error("[get_name_from_type] item not defined (None) [%s]" % e)
        logger.print_stack()
    except KeyError as e:  # Should never occur
        logger.error("[get_name_from_type] key not defined for item [%s] [%s]" % (item, e))
        logger.print_stack()
    except Exception as e:
        logger.error("[get_name_from_type] item_type:[%s] item:[%s] [%s]" % (item_type, item, e))
        logger.print_stack()
        raise
    
    return name


class BOOLEAN_PROPERTIES(object):
    
    @staticmethod
    def is_boolean(item_type, property_name):
        _prop_def = get_property_def(item_type, property_name)
        return isinstance(_prop_def, BoolProp)
    
    
    @staticmethod
    def get_label_for(value, app):
        return app._('common.value.boolean.%s' % value)


def get_property_def(item_type, prop):
    _prop = None
    try:
        _prop = DEF_ITEMS[item_type]['class'].properties[prop]
    except KeyError:
        # logger.warning('The definition of [%s] for type [%s] was not found' % (item_type, prop))
        pass
    if _prop is None:
        try:
            _prop = DEF_ITEMS[item_type]['class'].not_inherited_passthrough[prop]
        except KeyError:
            # logger.warning('The definition of [%s] for type [%s] was not found' % (item_type, prop))
            pass
    return _prop


def is_property_heritable(item_type, prop):
    # DATA are automatically inherited
    if prop.startswith('_'):
        return True
    _def = get_property_def(item_type, prop)
    return _def.inherit


def get_property_separator(item_type, property_name):
    prop_def = get_property_def(item_type, property_name)
    return prop_def.separator if prop_def else ','


def get_default_label(item_type, property_name, value, app):
    if property_name == 'view_contacts':
        _default = get_default_value(item_type, property_name)
        return app._('element.%s' % _default[0])
    if property_name == 'check_command':
        if item_type == ITEM_TYPE.HOSTS or item_type == ITEM_TYPE.HOSTTPLS:
            if value == PROP_DEFAULT_VALUE:
                return 'check-host-alive (ping)'
    if property_name == 'poller_tag' or property_name == 'reactionner_tag':
        return app._('element.untagged')
    if property_name == 'modulation_period' \
            or property_name == 'notification_period' \
            or property_name == 'check_period' \
            or property_name == 'host_notification_period' \
            or property_name == 'service_notification_period':
        if item_type == ITEM_TYPE.SERVICESCLUSTERS or item_type == ITEM_TYPE.SERVICESCLUSTERTPLS:
            return app._('element.same_as_cluster')
        if item_type == ITEM_TYPE.SERVICESHOSTTPLS or item_type == ITEM_TYPE.SERVICESHOSTS:
            return app._('element.same_as_host')
        if item_type == ITEM_TYPE.SERVICETPLS:
            return app._('element.same_as_host')
        return app._('element.always')
    return app._('element.none')


def get_default_value(item_type, prop, pythonize=True, app=None, same_as_host=False):
    # type: (unicode, unicode, bool, Synchronizer, bool) -> Optional[unicode]
    _default = None
    
    if app and prop == 'realm':
        default_realm = app.conf.realms.get_default()
        realm_name = default_realm.get_name()
        return realm_name
    
    if prop.startswith('_'):
        return _default
    try:
        _prop = get_property_def(item_type, prop)
        if same_as_host and app and item_type in (ITEM_TYPE.SERVICESHOSTS, ITEM_TYPE.SERVICESHOSTTPLS) and prop in PROPERTIES_RETRIEVABLE_FROM_HOST:
            _default = app._('element.same_as_host')
        elif same_as_host and app and item_type in (ITEM_TYPE.SERVICESCLUSTERS, ITEM_TYPE.SERVICESCLUSTERTPLS) and prop in PROPERTIES_RETRIEVABLE_FROM_HOST:
            _default = app._('element.same_as_cluster')
        elif same_as_host and app and item_type == ITEM_TYPE.SERVICETPLS and prop in PROPERTIES_RETRIEVABLE_FROM_HOST:
            _default = app._('element.same_as_host_parent')
        elif pythonize:
            _default = None if _prop.default == property_no_default_object else _prop.pythonize(_prop.default)
        else:
            _default = '' if _prop.default == property_no_default_object else _prop.default
    except Exception as e:
        logger.debug('[default] default value not find for property [%s]-[%s] [%s]' % (item_type, prop, e))
    
    return _default


def get_default_value_to_html(item_type_or_item, property_name, app, is_forced=False):
    _item_type = get_item_type(item_type_or_item)
    _to_return = get_default_value(_item_type, property_name, app=app, same_as_host=True, pythonize=False)
    _extra_label = ''
    if not _to_return:
        _to_return = app._('element.none')
        if _item_type in (ITEM_TYPE.HOSTS, ITEM_TYPE.HOSTTPLS):
            if property_name == 'address' or property_name == 'display_name':
                _to_return = '' if isinstance(item_type_or_item, basestring) else get_name_from_type(_item_type, item_type_or_item)
                _extra_label = app._('element.from_name_short')
    
    if BOOLEAN_PROPERTIES.is_boolean(_item_type, property_name):
        _to_return = BOOLEAN_PROPERTIES.get_label_for(_to_return, app)
    
    if property_name in ('business_impact', 'min_business_impact') and len(_to_return) == 1:
        _to_return = app._('element.business_impact_%s' % _to_return)
    
    elif property_name == 'notification_interval' and _item_type == ITEM_TYPE.ESCALATIONS:
        _to_return = app._('escalations.place_holder_default')
    
    elif property_name == 'view_contacts' and VIEW_CONTACTS_DEFAULT_VALUE.is_valid_default_value(_to_return):
        _to_return = app._('element.%s' % _to_return)
    
    elif PERIOD_PROPERTIES.is_period(property_name) and _to_return == PERIOD_PROPERTIES.DEFAULT_VALUE:
        _to_return = app._('element.always')
    
    elif PERIOD_PROPERTIES.ESCALATION_PERIOD == property_name:
        _to_return = app._('element.always')
    
    elif property_name in (PERIOD_PROPERTIES.HOST_NOTIFICATION_PERIOD, PERIOD_PROPERTIES.SERVICE_NOTIFICATION_PERIOD):
        _to_return = app._('element.always')
    
    elif PERIOD_PROPERTIES.MODULATION_PERIOD == property_name:
        _to_return = app._('element.always')
    
    elif property_name == 'check_command' and _to_return == 'check-host-alive':
        _to_return = 'check-host-alive (ping)'
    
    elif property_name == 'check_running_timeout':
        _to_return = app._('element.same_as_shinken__cfg_check_running_timeout') % _to_return
    
    elif property_name == 'notification_interval' and _to_return == '1440':
        _to_return = '%s (%s)' % (_to_return, app._('element.day'))
    
    elif property_name == 'timeout' and _item_type == ITEM_TYPE.COMMANDS:
        _to_return = app._('element.same_as_service')
    
    elif property_name == 'warning_threshold_cpu_usage':
        if _item_type == ITEM_TYPE.COMMANDS:
            _to_return = app._('element.same_as_service')
        else:
            _to_return = app._('element.same_as_shinken__cfg_warning_threshold_cpu_usage') % _to_return
    
    elif property_name in ('poller_tag', 'reactionner_tag'):
        _to_return = app._('element.untagged')
    
    if _to_return and is_forced != VALUE_FORCE_DEFAULT:
        _to_return = '[%s]' % _to_return
    if _extra_label:
        _to_return = '%s %s' % (_extra_label, _to_return)
    return _to_return


def parse_default_value_to_html(item_type_or_item, property_name, app, is_forced=False):
    _label_value = get_default_value_to_html(item_type_or_item, property_name, app)
    _label_default = app._('element.default')
    if is_forced == VALUE_FORCE_DEFAULT:
        return _label_value
    elif is_forced == PROP_DEFAULT_VALUE:
        _label_default = app._('element.forced_default')
    if is_forced is False and ITEM_TYPE.has_template(get_item_type(item_type_or_item)) and is_property_heritable(get_item_type(item_type_or_item), property_name):
        return '%s <span>%s</span>' % (app._('apply.value_not_set'), _label_value)
    
    return '%s <span class="shinken-default-value">%s</span>' % (_label_default, _label_value)


def get_item_type(item_type_or_item):
    if isinstance(item_type_or_item, basestring):
        return item_type_or_item
    else:
        return item_type_or_item.get('item_type', '') or item_type_or_item.get_type()


DEFAULT = object()


def get_item_property(item, item_property, default=DEFAULT):
    if '.' in item_property:
        param_to_link_attr_name_split = item_property.split('.', 1)
        item = item.get(param_to_link_attr_name_split[0], {})
        item_property = param_to_link_attr_name_split[1]
        if item:
            return get_item_property(item, item_property)
        else:
            return [] if default is DEFAULT else default
    else:
        return item.get(item_property, [] if default is DEFAULT else default)


def set_item_property(item, item_property, value):
    if '.' in item_property:
        item_property_split = item_property.split('.', 1)
        sub_prop = item.get(item_property_split[0], {})
        item_property = item_property_split[1]
        if not sub_prop:
            sub_prop = {}
            item[item_property_split[0]] = sub_prop
        set_item_property(sub_prop, item_property, value)
        value = sub_prop
        item_property = item_property_split[0]
    if hasattr(item, 'set_value'):
        item.set_value(item_property, value)
    else:
        item[item_property] = value


def get_dict_for_key(item, item_property):
    # manage when property is in format 'last_modification.contact'
    # we need to find the last dict (working_dict) and use the last value as property
    working_dict = item
    if '.' in item_property:
        property_split = item_property.split('.', 1)
        item_property = property_split[-1]
        # remove the last value of the split, so we can have the last dict and not the last value
        for new_key in property_split[:-1]:
            if new_key in working_dict:
                working_dict = working_dict[new_key]
    return working_dict, item_property


# https://stackoverflow.com/questions/59825/how-to-retrieve-an-element-from-a-set-without-removing-it
def get_first(iterator):
    e = None
    for e in iterator:
        break
    return e


def is_inherited(item_property, item):
    if item_property in NO_HERITABLE_ATTR:
        return False
    inherited = False
    from_infos = item['@metadata']['from'].get(item_property, None)
    if isinstance(from_infos, set):
        # fast case, if we have more than one FromInfo in the set we contain inherited values
        if len(from_infos) > 1:
            inherited = True
        else:
            # else check if the only one FromInfo.origin is the item
            from_info = get_first(from_infos)
            if not from_info:
                inherited = False
            elif from_info.origin != item['@metadata']['name']:
                inherited = True
            # manage case where tpl has the same name of tpl, in this case the type will tell us if the value come from us
            elif from_info.origin == item['@metadata']['name'] and from_info.type != item['@metadata']['type']:
                inherited = True
    return inherited


def get_original_not_inherited_value(item_property, item):
    item_value = item.get(item_property, None)
    if item_property in NO_HERITABLE_ATTR:
        return item_value
    
    from_infos = item['@metadata']['from'].get(item_property, None)
    if from_infos and isinstance(from_infos, set):
        item_name = item['@metadata']['name']
        item_type = item['@metadata']['type']
        
        item_value = set()
        for from_info in from_infos:
            if from_info.origin == item_name and from_info.type == item_type:
                item_value.add(from_info.value)
        item_value = ','.join(item_value)
        has_plus = item['@metadata'].get('has_plus', {}).get(item_property, False)
    else:
        has_plus = isinstance(item_value, basestring) and item_value.startswith('+')
        item_value = item_value[1:] if has_plus else item_value
    
    return item_value, has_plus


def remove_plus(value):
    return value[1:] if value.startswith('+') else value


def get_inherited_without_default(datamanagerV2, item, flatten_states=LINKIFY_MANAGE_STATES, data_on_item=None):
    if data_on_item is None:
        data_on_item = set()
    inherited_without_defaults = {'_id': item['_id']}
    for prop_name, from_info in item.get_from().iteritems():
        if prop_name not in INTERNAL_ATTR and (not from_info or from_info[0].origin != '__DEFAULT__'):
            inherited_without_defaults[prop_name] = datamanagerV2.flatten_value(item[prop_name], item.get_type(), prop_name, flatten_states)
            # If you want to get the list of data on item, we compute it here. Needed in SEF-4417
            if prop_name.startswith('_') and prop_name not in NOT_TO_LOOK:
                data_on_item.add(prop_name)
    return inherited_without_defaults


def item_for_link(item_link, states=LINKIFY_MANAGE_STATES, only_exist=False):
    return next(iter(items_for_link(item_link, states, only_exist)), None)


def items_for_link(item_link, states=LINKIFY_MANAGE_STATES, only_exist=False):
    if item_link['exists']:
        return [item_link.get('@link', {}).get(item_state, {}) for item_state in states if item_link.get('@link', {}).get(item_state, {})]
    elif not only_exist:
        from .items import BaseItem
        
        mock_item = BaseItem({'exists': False})
        METADATA.update_metadata(mock_item, METADATA.NAME, item_link['name'])
        return [mock_item]
    return []


# The front-end sends incomplete links based on what it knows about the objects.
# This function updates these links to put them in the right format for the back to use
# and sends back the names of the found linked objects

# For all links defined for the .property, we try to find objects for all possible types of
# linkable objects based on _id, and then on name if we can't find objects with _id.

# Based on these results, the links' data are updated and formatted properly:
# - exists
# - item_type
# - @link if requested

def update_frontend_links_into_links(property_name, frontend_links, item_type, datamanagerV2, linkify=True):
    # type: (unicode, List, unicode, DataManagerV2, bool) -> list
    link_item_names = []
    for link in frontend_links:
        update_frontend_link_into_link(property_name, link, item_type, datamanagerV2, linkify=linkify, link_item_names=link_item_names)
    
    return link_item_names


def update_frontend_link_into_link(property_name, link, item_type, datamanagerV2, linkify=True, link_item_names=None):
    # type: (unicode, Dict, unicode, DataManagerV2, bool, List) -> NoReturn
    if link_item_names is None:
        link_item_names = []
    if 'item_type' in link:
        potential_link_types = [link['item_type']]
    else:
        potential_link_types = DEF_ITEMS[item_type]['props_links'][property_name]
    
    linked_items = []
    if '_id' in link:
        where = {'_id': link['_id']}
        for potential_link_type in potential_link_types:
            linked_items.extend(datamanagerV2.find_merge_state_items(potential_link_type, item_states=LINKIFY_MANAGE_STATES, where=where))
    
    if not linked_items:
        for potential_link_type in potential_link_types:
            where = {DEF_ITEMS[potential_link_type]['key_name']: link['name']}
            linked_items.extend(datamanagerV2.find_merge_state_items(potential_link_type, item_states=LINKIFY_MANAGE_STATES, where=where))
    
    if linked_items:
        link['exists'] = True
        link.pop('name', None)
        link['item_type'] = linked_items[0].get_type()
        link['_id'] = linked_items[0]['_id']
        if linkify:
            link['@link'] = {}
            for item in linked_items:
                link['@link'][item.get_state()] = item
                link_item_names.append(item.get_name())
    else:
        link['exists'] = False
        link.pop('item_type', None)
        link.pop('_id', None)
        link_item_names.append(link['name'])


def read_frontend_links(property_name, frontend_links, item_type, datamanagerV2, states=LINKIFY_MANAGE_STATES, only_exist=False):
    link_items = []
    for link in frontend_links.get('links', []):
        if 'item_type' in link:
            potential_link_types = [link['item_type']]
        else:
            potential_link_types = DEF_ITEMS[item_type]['props_links'][property_name]
        
        linked_items = []
        if '_id' in link:
            where = {'_id': link['_id']}
            for potential_link_type in potential_link_types:
                linked_items.extend(datamanagerV2.find_merge_state_items(potential_link_type, item_states=states, where=where))
        
        if not linked_items:
            for potential_link_type in potential_link_types:
                where = {DEF_ITEMS[potential_link_type]['key_name']: link['name']}
                linked_items.extend(datamanagerV2.find_merge_state_items(potential_link_type, item_states=states, where=where))
        
        if linked_items:
            link_items.extend(linked_items)
        elif not only_exist:
            from .items import BaseItem
            
            mock_item = BaseItem({'exists': False})
            METADATA.update_metadata(mock_item, METADATA.NAME, link['name'])
            link_items.append(mock_item)
    
    return link_items


def get_origin(item, item_property):
    from_info = METADATA.get_metadata(item, METADATA.FROM, {}).get(item_property, [])
    origin = from_info[0].origin if from_info else None
    return origin


def is_stop_inheritance(value_to_test):
    # type: (Union[Dict,Str]) -> bool
    if not value_to_test:
        return False
    
    if value_to_test in STOP_INHERITANCE_VALUES:
        return True
    try:
        for link in value_to_test.get('links', []):
            if link['exists']:
                return False
            if link['name'] in STOP_INHERITANCE_VALUES:
                return True
    except Exception:
        return False
    
    return False


def get_flat_values_to_links(values, item_type, item_state, datamanager_v2):
    # type: ( Str, Str, Str, DataManagerV2) -> Dict
    _to_return = {'has_plus': False, 'links': []}
    if values.startswith('+'):
        _to_return['has_plus'] = True
        values = values[1:]
    if values:
        for _value in values.split(','):
            _to_return['links'].append(get_item_to_link_find_by_name(_value, item_type, item_state, datamanager_v2))
    return _to_return


def get_item_to_link_find_by_name(name, item_type, item_state, datamanager_v2):
    # type: ( Str, Str, Str, DataManagerV2) -> Dict
    item = datamanager_v2.find_item_by_name(name, item_type, item_state)
    _to_return = {'item_type': item_type}
    if item:
        _to_return['exists'] = True
        _to_return['_id'] = item.get_uuid()
    else:
        _to_return['exists'] = False
        _to_return['name'] = name
    
    return _to_return


def is_item_link_in_item_links(link_item, links):
    # type: ( Dict, List[Dict]) -> bool
    _exists = link_item['exists']
    for _link in links:
        if _exists != _link['exists']:
            continue
        if _exists and link_item['_id'] == _link['_id']:
            return True
        if not _exists and link_item['name'] == _link['name']:
            return True
    
    return False


def get_item_to_link_for_stop_inheritance(stop_inheritance_value):
    # type: ( Str) -> Dict
    return {
        'has_plus': False,
        'links'   : [
            {
                'name'  : stop_inheritance_value,
                'exists': False
            }
        ]
    }


def is_user_rights_default_value_everyone():
    # type: () -> bool
    return get_default_value(ITEM_TYPE.HOSTS, 'view_contacts', pythonize=False) == 'everyone'


def add_source_change(datamanager, item, item_type, action, old_item=None, item_property=None):
    if not old_item and not item_property:
        return
    
    merge_from_source_item = datamanager.find_item_by_id(item[u'_id'], item_type, ITEM_STATE.MERGE_SOURCES)
    if not merge_from_source_item:
        return
    
    effective = False
    source_info = METADATA.get_metadata(merge_from_source_item, METADATA.SOURCE_INFO)[u'_info']
    if item_property:
        if source_info.get(item_property, None):
            old_action = source_info[item_property].get(u'property_modif_auto', None)
            if not old_action or old_action == HISTORY_ACTION.AUTO_MODIFICATION:
                source_info[item_property][u'property_modif_auto'] = action
                effective = True
    else:
        old_item = datamanager.get_raw_item(old_item, item_type=item_type, flatten_links=False)
        item = datamanager.get_raw_item(item, item_type=item_type, flatten_links=False)
        diffs_properties = compute_diff(item, old_item, item_type)
        for item_property in diffs_properties:
            if source_info.get(item_property, None):
                old_action = source_info[item_property].get(u'property_modif_auto', None)
                if not old_action or old_action == HISTORY_ACTION.AUTO_MODIFICATION:
                    source_info[item_property][u'property_modif_auto'] = action
                    effective = True
    
    if effective:
        datamanager.save_item(merge_from_source_item, None, item_type, item_state=ITEM_STATE.MERGE_SOURCES)
