#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

import copy
import re
import time

from shinken.log import logger
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.service import PREFIX_LINK_UUID, PREFIX_LINK_DFE_KEY
from ..def_items import METADATA, ITEM_TYPE, DEF_ITEMS, SERVICE_OVERRIDE, ITEM_STATE, MAX_TEMPLATE_LOOKUP_LEVEL, SERVICE_OVERRIDE_UUID_SEP, LINKIFY_MANAGE_STATES, SERVICE_EXCLUDES_BY_ID, SERVICE_EXCLUDES_BY_NAME, EXCLUDE_FROM_TEMPLATES_RESULTS, \
    can_property_contain_links
from ..helpers import get_name_from_type, ShinkenDatabaseConsistencyError
from ..parsers.bp_rule_parser import BpRuleParser
from ..parsers.complex_exp_parser import do_match, visit_node

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Union, Tuple, Set, List, Optional
    from . import BaseItem, ServiceItem
    from .misc_items import InstanceItem

AT_LINK_PROP_NAME = u'@link'


def _get_item_from_link(link, prop_item_states, base_item):
    # We search first item with state in prop_item_states in self value the item link
    item = next((link[AT_LINK_PROP_NAME][item_state] for item_state in prop_item_states if link[AT_LINK_PROP_NAME].get(item_state, None)), None)
    if item:
        return item
    else:
        raise ShinkenDatabaseConsistencyError(link, base_item['_id'], base_item.get_type(), base_item.get_state(), prop_item_states)


def _get_item_name_from_link(link, prop_item_states, base_item):
    if link['exists']:
        
        item_name = _get_item_from_link(link, prop_item_states, base_item).get_name()
    else:
        item_name = link['name']
    return item_name


def _get_item_name_from_datamanager(link, prop_item_states, datamanagerV2):
    if link['exists']:
        item = datamanagerV2.find_merge_state_items(link['item_type'], prop_item_states, where={'_id': link['_id']})
        if not item:
            # In frontend link we have _id and name
            if 'name' in link:
                return link['name']
            else:
                raise ShinkenDatabaseConsistencyError(link, asking_state=prop_item_states)
        item_name = get_name_from_type(link['item_type'], item[0])
    else:
        item_name = link['name']
    return item_name


class ModulateDataMixin(object):
    def prepare_modulate_data(self, timeperiods):
        self.set_value('@modulation', {})
        
        now = time.time()
        for modulation in reversed(self.get_link_items('macromodulations', only_exist=True)):
            if not modulation.is_enabled():
                continue
            mod_name = modulation.get_name()
            
            modulation_period = self.get_link_item('modulation_period', only_exist=True)
            if modulation_period:
                timeperiod = timeperiods.find_by_name(modulation_period.get_name())
                if not timeperiod or not timeperiod.is_time_valid(now):
                    continue
            
            for data_name, data_value in modulation.iteritems():
                if not data_name.startswith('_'):
                    continue
                self['@modulation'][data_name] = (data_value, mod_name)


class CommandMixin(object):
    
    @staticmethod
    def flatten_not_linked_prop(property_value, _property_name, prop_item_states, datamanager_v2):
        if not property_value:
            return ''
        if not prop_item_states:
            raise Exception('missing prop_item_states param in _common_get_command_flatten')
        if not isinstance(property_value, dict):
            raise Exception('raw_item_value param in _common_get_command_flatten is not a dict type it is a [%s] type value [%s]' % (type(property_value), property_value))
        if 'node' not in property_value:
            return property_value['raw_value']
        
        command_name = _get_item_name_from_datamanager(property_value['node']['link'], prop_item_states, datamanager_v2)
        args = property_value['node']['args']
        if args:
            return '%s!%s' % (command_name, args)
        return command_name
    
    
    #################
    # Common        #
    #################
    def _common_get_command_link(self, property_name, _linked_items_type=None):
        link_info = self.get(property_name, None)
        if link_info and isinstance(link_info, dict):
            link = link_info.get('node', {}).get('link', None)
            if link:
                return [link]
        return []
    
    
    def _common_get_command_flatten(self, raw_item_value, property_name, prop_item_states):
        if not raw_item_value:
            return ''
        if not prop_item_states:
            raise Exception('missing prop_item_states param in _common_get_command_flatten')
        if not isinstance(raw_item_value, dict):
            raise Exception('raw_item_value param in _common_get_command_flatten is not a dict type it is a [%s] type value [%s]' % (type(raw_item_value), raw_item_value))
        if 'node' not in raw_item_value:
            return raw_item_value['raw_value']
        
        # We can get the value from base item because it is the same as raw_item because our raw_item_value is not empty
        item_value = self.get(property_name, None)
        command_name = _get_item_name_from_link(item_value['node']['link'], prop_item_states, self)
        args = raw_item_value['node']['args']
        if args:
            return "%s!%s" % (command_name, args)
        return command_name
    
    
    def _common_update_command_links(self, property_name, item_to_link):
        if not self.get(property_name, {}) or isinstance(self.get(property_name, {}), basestring):
            return False
        
        effective = False
        property_value = self[property_name]
        command_link = property_value['node']['link']
        
        if command_link['exists'] and command_link['_id'] == item_to_link['_id']:
            command_link[AT_LINK_PROP_NAME][item_to_link.get_state()] = item_to_link
        
        elif not command_link['exists'] and command_link['name'].lower() == item_to_link.get_name().lower() and item_to_link.get_state() in LINKIFY_MANAGE_STATES:
            command_link = property_value['node']['link'] = {
                '_id'      : item_to_link.get('_id', None),
                'item_type': ITEM_TYPE.COMMANDS,
                'exists'   : True
            }
            raw_property_value = METADATA.get_metadata(self, METADATA.RAW_ITEM)[property_name]
            raw_property_value['node']['link'] = command_link.copy()
            command_link[AT_LINK_PROP_NAME] = {item_to_link.get_state(): item_to_link}
            effective = True
        
        return effective
    
    
    def _common_remove_item_link(self, property_name, _item_to_unlink):
        if not self.get(property_name, {}) or isinstance(self.get(property_name, {}), basestring):
            return False
        
        raw_item = METADATA.get_metadata(self, METADATA.RAW_ITEM, {})
        edited = raw_item.pop(property_name, None)
        del self[property_name]
        
        if edited:
            return 'must_save_with_callback'
        return False
    
    
    def _common_set_inexisting_link(self, property_name, item_to_unlink):
        if not self.get(property_name, {}) or isinstance(self.get(property_name, {}), basestring):
            return False
        
        effective = False
        property_value = self[property_name]
        command_link = property_value['node']['link']
        
        if command_link['exists'] and command_link['_id'] == item_to_unlink['_id']:
            command_link = {
                'exists': False,
                'name'  : item_to_unlink.get_name()
            }
            property_value['node']['link'] = command_link
            raw_property_value = METADATA.get_metadata(self, METADATA.RAW_ITEM)[property_name]
            raw_property_value['node']['link'] = command_link.copy()
            effective = True
        
        return effective
    
    
    ##################
    # check_command  #
    ##################
    def _get_links_check_command(self, property_name, linked_items_type=None):
        return self._common_get_command_link(property_name, linked_items_type)
    
    
    def _flatten_prop_check_command(self, raw_item_value, property_name, prop_item_states, _for_arbiter=False):
        return self._common_get_command_flatten(raw_item_value, property_name, prop_item_states)
    
    
    def _update_link_check_command(self, property_name, item):
        return self._common_update_command_links(property_name, item)
    
    
    def _remove_item_link_check_command(self, property_name, item_to_unlink):
        return self._common_remove_item_link(property_name, item_to_unlink)
    
    
    def _set_inexisting_link_check_command(self, property_name, item_to_unlink):
        self._common_set_inexisting_link(property_name, item_to_unlink)
    
    
    #################
    # event_handler #
    #################
    def _get_links_event_handler(self, property_name, linked_items_type=None):
        return self._common_get_command_link(property_name, linked_items_type)
    
    
    def _flatten_prop_event_handler(self, raw_item_value, property_name, prop_item_states, _for_arbiter=False):
        return self._common_get_command_flatten(raw_item_value, property_name, prop_item_states)
    
    
    def _update_link_event_handler(self, property_name, item):
        return self._common_update_command_links(property_name, item)
    
    
    def _remove_item_link_event_handler(self, property_name, item_to_unlink):
        return self._common_remove_item_link(property_name, item_to_unlink)
    
    
    def _set_inexisting_link_event_handler(self, property_name, item_to_unlink):
        self._common_set_inexisting_link(property_name, item_to_unlink)
    
    
    #################################
    # service_notification_commands #
    #################################
    def _get_links_service_notification_commands(self, property_name, linked_items_type=None):
        return self._common_get_command_link(property_name, linked_items_type)
    
    
    def _flatten_prop_service_notification_commands(self, raw_item_value, property_name, prop_item_states, _for_arbiter=False):
        return self._common_get_command_flatten(raw_item_value, property_name, prop_item_states)
    
    
    def _update_link_service_notification_commands(self, property_name, item):
        return self._common_update_command_links(property_name, item)
    
    
    def _remove_item_link_service_notification_commands(self, property_name, item_to_unlink):
        return self._common_remove_item_link(property_name, item_to_unlink)
    
    
    def _set_inexisting_link_service_notification_commands(self, property_name, item_to_unlink):
        self._common_set_inexisting_link(property_name, item_to_unlink)
    
    
    ################################
    # host_notification_commands   #
    ################################
    def _get_links_service_host_notification_commands(self, property_name, linked_items_type=None):
        return self._common_get_command_link(property_name, linked_items_type)
    
    
    def _flatten_prop_host_notification_commands(self, raw_item_value, property_name, prop_item_states, _for_arbiter=False):
        return self._common_get_command_flatten(raw_item_value, property_name, prop_item_states)
    
    
    def _update_link_host_notification_commands(self, property_name, item):
        return self._common_update_command_links(property_name, item)
    
    
    def _remove_item_link_host_notification_commands(self, property_name, item_to_unlink):
        return self._common_remove_item_link(property_name, item_to_unlink)
    
    
    def _set_inexisting_link_host_notification_commands(self, property_name, item_to_unlink):
        self._common_set_inexisting_link(property_name, item_to_unlink)


class BPRuleMixin(object):
    
    @staticmethod
    def flatten_not_linked_prop(property_value, _property_name, prop_item_states, datamanager_v2):
        if not property_value:
            return ''
        if not prop_item_states:
            raise Exception('missing prop_item_states param in flatten_not_linked_prop')
        if not isinstance(property_value, dict):
            raise Exception('raw_item_value param in flatten_not_linked_prop is not a dict type it is a [%s] type value [%s]' % (type(property_value), property_value))
        if not property_value['node']:
            return property_value['raw_value']
        
        raw_bp_rule = []
        for token in property_value['node']:
            if BpRuleParser._is_linkable_token(token):
                _to_string = token['to_string']
                name = _get_item_name_from_datamanager(token['content'], prop_item_states, datamanager_v2)
                value = _to_string % name
            else:
                value = token['to_string'] % token['content']
            
            raw_bp_rule.append(value)
        return ''.join(raw_bp_rule)
    
    
    def _flatten_prop_bp_rule(self, raw_item_value, _property_name, prop_item_states, _for_arbiter=False):
        if not raw_item_value:
            return ''
        if not prop_item_states:
            raise Exception('missing prop_item_states param in _flatten_prop_bp_rule')
        if not isinstance(raw_item_value, dict):
            raise Exception('raw_item_value param in _flatten_prop_bp_rule is not a dict type it is a [%s] type value [%s]' % (type(raw_item_value), raw_item_value))
        if not raw_item_value['node']:
            return raw_item_value['raw_value']
        
        item_value = self.get('bp_rule', None)
        raw_bp_rule = []
        for index, token in enumerate(raw_item_value['node']):
            if BpRuleParser._is_linkable_token(token):
                _to_string = token['to_string']
                name = _get_item_name_from_link(item_value['node'][index]['content'], prop_item_states, self)
                value = _to_string % name
            else:
                value = token['to_string'] % token['content']
            
            raw_bp_rule.append(value)
        return ''.join(raw_bp_rule)
    
    
    def _get_links_bp_rule(self, property_name, linked_items_type=None):
        links = []
        if not self.get(property_name, {}):
            return links
        if linked_items_type is None or linked_items_type in self[property_name]['all_linked_types']:
            for token in self.get(property_name, {}).get('node'):
                if BpRuleParser._is_linkable_token(token):
                    if linked_items_type is None or (token['content'].get('item_type', None) == linked_items_type or token['content']['exists'] is False):
                        links.append(token['content'])
        return links
    
    
    def _remove_item_link_bp_rule(self, property_name, item_to_unlink):
        if not self.get(property_name, {}) or isinstance(self.get(property_name, {}), basestring):
            return False
        
        raw_item_bp_rule = METADATA.get_metadata(self, METADATA.RAW_ITEM, {}).get(property_name, {})
        found_type = set()
        effective = False
        for index, token in enumerate(self[property_name]['node']):
            if BpRuleParser._is_linkable_token(token):
                link = token['content']
                if link['exists'] and link.get('_id', None) == item_to_unlink['_id']:
                    link.get(AT_LINK_PROP_NAME, {}).pop(item_to_unlink.get_state(), None)
                    if not link.get(AT_LINK_PROP_NAME, {}):
                        link.pop('_id', None)
                        link.pop(AT_LINK_PROP_NAME, None)
                        old_link_type = link.pop('item_type', None)
                        link['exists'] = False
                        link['name'] = item_to_unlink.get_name()
                        
                        token['item_type'] = [BpRuleParser.TYPE_BY_FLAGS[key] for key, value in BpRuleParser.TYPE_BY_FLAGS.iteritems() if old_link_type in value][0]
                        raw_item_bp_rule['node'][index]['content'] = link.copy()
                        raw_item_bp_rule['node'][index]['item_type'] = token['item_type']
                        effective = True
                
                if link['exists']:
                    found_type.add(token['item_type'])
                else:
                    found_type.update(token['item_type'])
        
        self[property_name]['all_linked_types'] = list(found_type)
        raw_item_bp_rule['all_linked_types'] = list(found_type)
        
        return effective
    
    
    def _update_link_bp_rule(self, property_name, item_to_link):
        if not self.get(property_name, {}) or isinstance(self.get(property_name, {}), basestring):
            return False
        
        raw_item_bp_rule = METADATA.get_metadata(self, METADATA.RAW_ITEM, {}).get(property_name, {})
        found_type = set()
        effective = False
        
        item_to_link_name = item_to_link.get_name()
        item_to_link_type = item_to_link.get_type()
        for index, token in enumerate(self[property_name]['node']):
            if BpRuleParser._is_linkable_token(token):
                link = token['content']
                if not link['exists'] and link['name'].lower() == item_to_link_name.lower() and item_to_link.get_state() in LINKIFY_MANAGE_STATES:
                    link = {
                        'exists'   : True,
                        '_id'      : item_to_link['_id'],
                        'item_type': item_to_link_type
                    }
                    token['content'] = link
                    token['item_type'] = item_to_link_type
                    effective = True
                    raw_item_bp_rule['node'][index] = copy.deepcopy(token)
                    link[AT_LINK_PROP_NAME] = {item_to_link.get_state(): item_to_link}
                
                if link['exists'] and link['_id'] == item_to_link['_id']:
                    link[AT_LINK_PROP_NAME][item_to_link.get_state()] = item_to_link
                
                if link['exists']:
                    found_type.add(token['item_type'])
                else:
                    found_type.update(token['item_type'])
        
        self[property_name]['all_linked_types'] = list(found_type)
        raw_item_bp_rule['all_linked_types'] = list(found_type)
        return effective
    
    
    def _set_inexisting_link_bp_rule(self, property_name, item_to_unlink):
        if not self.get(property_name, {}) or isinstance(self.get(property_name, {}), basestring):
            return False
        
        raw_item_bp_rule = METADATA.get_metadata(self, METADATA.RAW_ITEM, {}).get(property_name, {})
        found_type = set()
        effective = False
        for index, token in enumerate(self[property_name]['node']):
            if BpRuleParser._is_linkable_token(token):
                link = token['content']
                if link['exists'] and link.get('_id', None) == item_to_unlink['_id']:
                    link.pop(AT_LINK_PROP_NAME, None)
                    link.pop('_id', None)
                    link['exists'] = False
                    link['name'] = item_to_unlink.get_name()
                    
                    old_link_type = link.pop('item_type', None)
                    token['item_type'] = [BpRuleParser.TYPE_BY_FLAGS[key] for key, value in BpRuleParser.TYPE_BY_FLAGS.iteritems() if old_link_type in value][0]
                    raw_item_bp_rule['node'][index]['content'] = link.copy()
                    raw_item_bp_rule['node'][index]['item_type'] = token['item_type']
                    effective = True
                
                if link['exists']:
                    found_type.add(token['item_type'])
                else:
                    found_type.update(token['item_type'])
        
        self[property_name]['all_linked_types'] = list(found_type)
        raw_item_bp_rule['all_linked_types'] = list(found_type)
        
        return effective


# Abstract class : use sub class HostNameMixin or HostgroupName
class _ComplexExpMixin(object):
    
    @staticmethod
    def _return_parent_node(n, ns, l):
        if n['not_value']:
            return '!(%s)' % n['operand'].join(ns)
        else:
            return '(%s)' % n['operand'].join(ns) if l else n['operand'].join(ns)
    
    
    @staticmethod
    def flatten_not_linked_prop(property_value, property_name, prop_item_states, datamanagerV2):
        if not property_value:
            return ''
        if not prop_item_states:
            raise Exception('missing prop_item_states param in flatten_not_linked_prop')
        if not isinstance(property_value, dict):
            raise Exception('raw_item_value param in flatten_not_linked_prop is not a dict type it is a [%s] type value [%s]' % (type(property_value), property_value))
        if not property_value['node']:
            return property_value['raw_value']
        
        
        def flatten(node, level):
            template_name = _get_item_name_from_datamanager(node['content'], prop_item_states, datamanagerV2)
            return '%s%s' % ('!' if node['not_value'] else '', template_name)
        
        
        return visit_node(property_value['node'], flatten, _ComplexExpMixin._return_parent_node)
    
    
    def _generic_get_links(self, property_name, linked_items_type=None):
        links = []
        if self.get(property_name, None) and self[property_name].get('node', None):
            node = self[property_name]['node']
            return_leaf_node = lambda n, l: links.append(n['content']) if not linked_items_type or linked_items_type == n['content'].get('item_type', linked_items_type) else None
            visit_node(node, return_leaf_node)
        return links
    
    
    def _generic_flatten_prop(self, raw_item_value, property_name, prop_item_states):
        if not raw_item_value:
            return ''
        if not prop_item_states:
            raise Exception('missing prop_item_states param in _generic_flatten_prop')
        if not isinstance(raw_item_value, dict):
            raise Exception('raw_item_value param in _generic_flatten_prop is not a dict type it is a [%s] type value [%s]' % (type(raw_item_value), raw_item_value))
        if not raw_item_value['node']:
            return raw_item_value['raw_value']
        
        # IMPORTANT NOTE : Field with ComplexExp are not inheritable and they have not default value so we can flatten item_value because in this case item_value == raw_item_value
        item_value = self.get(property_name, None)
        
        
        def flatten(node, level):
            template_name = _get_item_name_from_link(node['content'], prop_item_states, self)
            return '%s%s' % ('!' if node['not_value'] else '', template_name)
        
        
        return visit_node(item_value['node'], flatten, _ComplexExpMixin._return_parent_node)
    
    
    def _generic_remove_item_link(self, property_name, item_to_unlink):
        if not self.get(property_name, {}) or isinstance(self.get(property_name, {}), basestring):
            return False
        do_update = {'done': False}
        node = self[property_name]['node']
        raw_item_value = METADATA.get_metadata(self, METADATA.RAW_ITEM)[property_name]
        
        
        def update_link(node, _level):
            link = node['content']
            if link['exists'] and item_to_unlink['_id'] == link['_id']:
                link.get(AT_LINK_PROP_NAME, {}).pop(item_to_unlink.get_state(), None)
                if not link.get(AT_LINK_PROP_NAME, {}):
                    link.pop('_id', None)
                    link.pop(AT_LINK_PROP_NAME, None)
                    link.pop('item_type', None)
                    link['exists'] = False
                    link['name'] = item_to_unlink.get_name()
                    do_update['done'] = True
        
        
        visit_node(node, update_link)
        visit_node(raw_item_value['node'], update_link)
        return do_update['done']
    
    
    def _generic_update_link(self, property_name, to_update_item):
        if not self.get(property_name, {}) or isinstance(self.get(property_name, {}), basestring):
            return False
        
        do_update = {'done': False}
        node = self[property_name]['node']
        raw_item_value = METADATA.get_metadata(self, METADATA.RAW_ITEM)[property_name]
        
        
        def update_link(node, level):
            link = node['content']
            link_name = link[u'name'].lower() if link.get('name', None) else None
            if to_update_item.get_name().lower() == link_name and to_update_item.get_state() in LINKIFY_MANAGE_STATES:
                link.pop('name', None)
                link['exists'] = True
                link['item_type'] = to_update_item.get_type()
                link['_id'] = to_update_item['_id']
                link[AT_LINK_PROP_NAME] = {to_update_item.get_state(): to_update_item}
                do_update['done'] = True
            elif to_update_item['_id'] == link.get('_id', None):
                link[AT_LINK_PROP_NAME][to_update_item.get_state()] = to_update_item
        
        
        def update_link_no_at_link(node, level):
            link = node['content']
            if to_update_item.get_name() == link.get('name', None) and to_update_item.get_state() in LINKIFY_MANAGE_STATES:
                link.pop('name', None)
                link['exists'] = True
                link['item_type'] = to_update_item.get_type()
                link['_id'] = to_update_item['_id']
                do_update['done'] = True
        
        
        visit_node(node, update_link)
        visit_node(raw_item_value['node'], update_link_no_at_link)
        return do_update['done']
    
    
    def _generic_set_inexisting_link(self, property_name, item_to_unlink):
        if not self.get(property_name, {}) or isinstance(self.get(property_name, {}), basestring):
            return False
        do_update = {'done': False}
        node = self[property_name]['node']
        raw_item_value = METADATA.get_metadata(self, METADATA.RAW_ITEM)[property_name]
        
        
        def update_link(node, level):
            link = node['content']
            if link['exists'] and item_to_unlink['_id'] == link['_id']:
                link.pop('_id', None)
                link.pop(AT_LINK_PROP_NAME, None)
                link.pop('item_type', None)
                link['exists'] = False
                link['name'] = item_to_unlink.get_name()
                do_update['done'] = True
        
        
        visit_node(node, update_link)
        visit_node(raw_item_value['node'], update_link)
        return do_update['done']


# usable for property : host_name in service host/cluster templates
class HostNameMixin(_ComplexExpMixin):
    
    def _get_links_host_name(self, property_name, linked_items_type=None):
        return self._generic_get_links(property_name, linked_items_type)
    
    
    def _flatten_prop_host_name(self, raw_item_value, property_name, prop_item_states, for_arbiter=False):
        return self._generic_flatten_prop(raw_item_value, property_name, prop_item_states)
    
    
    def _remove_item_link_host_name(self, property_name, item_to_unlink):
        return self._generic_remove_item_link(property_name, item_to_unlink)
    
    
    def _update_link_host_name(self, property_name, to_update_item):
        return self._generic_update_link(property_name, to_update_item)
    
    
    def _set_inexisting_link_host_name(self, property_name, to_update_item):
        return self._generic_set_inexisting_link(property_name, to_update_item)


# usable for property : hostgroup_name in service host/cluster templates
class HostgroupNameMixin(_ComplexExpMixin):
    
    def _get_links_hostgroup_name(self, property_name, linked_items_type=None):
        return self._generic_get_links(property_name, linked_items_type)
    
    
    def _flatten_prop_hostgroup_name(self, raw_item_value, property_name, prop_item_states, for_arbiter=False):
        return self._generic_flatten_prop(raw_item_value, property_name, prop_item_states)
    
    
    def _remove_item_link_hostgroup_name(self, property_name, item_to_unlink):
        return self._generic_remove_item_link(property_name, item_to_unlink)
    
    
    def _update_link_hostgroup_name(self, property_name, to_update_item):
        return self._generic_update_link(property_name, to_update_item)
    
    
    def _set_inexisting_link_hostgroup_name(self, property_name, to_update_item):
        return self._generic_set_inexisting_link(property_name, to_update_item)


class UseMixin(object):
    
    def get_parents(self):
        # type: (Union[InstanceItem, UseMixin]) -> List[Union[InstanceItem, UseMixin]]
        return self.get_all_state_link_items('use', only_exist=True)
    
    
    def get_sons(self):
        # type: (Union[InstanceItem, UseMixin]) -> List[Union[InstanceItem, UseMixin]]
        sons = METADATA.get_metadata(self, METADATA.SONS, {})
        # We remove old son that are not anymore son.
        sons = [i for i in sons.itervalues() if self in i.get_parents()]
        return sons
    
    
    def add_son(self, son):
        # type: (Union[InstanceItem, UseMixin]) -> None
        sons = METADATA.get_metadata(self, METADATA.SONS, {})
        if son.get_key() not in sons:
            sons[son.get_key()] = son
        METADATA.update_metadata(self, METADATA.SONS, sons)
    
    
    def remove_son(self, son):
        # type: (Union[InstanceItem, UseMixin], Union[InstanceItem, UseMixin]) -> None
        sons = METADATA.get_metadata(self, METADATA.SONS, {})
        # Here we set the pop with None because of SEF-5395 : We cannot reproduce the bug :
        # A tpl haven't the son we want to remove. The result of this function is ok because the sons is already removed, but this indicates that there is a problem upstream
        sons.pop(son.get_key(), None)
    
    
    def _update_link_use(self, property_name, to_update_item):
        # type: (Union[InstanceItem, UseMixin], unicode, Union[InstanceItem, UseMixin]) -> bool
        effective, link_was_found = self._update_link(property_name, to_update_item)
        if link_was_found:
            to_update_item.add_son(self)
        return effective
    
    
    def _remove_item_link_use(self, property_name, item_to_unlink):
        # type: (Union[InstanceItem, UseMixin], unicode, Union[InstanceItem, UseMixin]) -> bool
        if not self.get(property_name, None):
            return False
        
        effective = False
        item_to_unlink_id = item_to_unlink['_id']
        my_links = self[property_name]['links']
        link_item_index, link_item = next(((_index, link_item) for (_index, link_item) in enumerate(my_links) if link_item.get('_id', None) == item_to_unlink_id), (0, {}))
        
        if not link_item:
            return False
        
        link_item.get(AT_LINK_PROP_NAME, {}).pop(item_to_unlink.get_state(), None)
        if not link_item.get(AT_LINK_PROP_NAME, {}):
            link_item.pop(AT_LINK_PROP_NAME, None)
            link_item.pop('_id', None)
            link_item.pop('item_type', None)
            link_item['exists'] = False
            link_item['name'] = item_to_unlink.get_name()
            
            raw_item_links = METADATA.get_metadata(self, METADATA.RAW_ITEM, {}).get(property_name, {}).get('links', [])
            raw_item_links[link_item_index] = link_item.copy()
            effective = True
        
        if effective:
            item_to_unlink.remove_son(self)
        return effective
    
    
    def get_use_only_exist(self):
        return ','.join([i.get_name() for i in self.get_parents()])


class ServiceOverridesMixinException(Exception):
    pass


class ServiceOverridesMixin(object):
    def _get_links_service_overrides(self, property_name, linked_items_type=None):
        property_value = self.get(property_name, '')
        if not property_value or isinstance(property_value, basestring):
            return []
        
        # if there are a linked_items_type ask we check if we have this type in our all_linked_types cache
        # Note : all_linked_types contain special type __ALL_SERVICES__ and __ALL_HOSTS__ for match all services and all hosts types
        if linked_items_type is not None and \
                not (
                        linked_items_type in set(property_value['all_linked_types']) or
                        '__ALL_SERVICES__' in property_value['all_linked_types'] and linked_items_type in ITEM_TYPE.ALL_SERVICES or
                        '__ALL_HOSTS__' in property_value['all_linked_types'] and linked_items_type in ITEM_TYPE.ALL_HOST_CLASS):
            return []
        
        links = []
        for service_override in property_value.get('links', []):
            check_link = service_override['check_link']
            host_link = service_override.get('host_link', None)
            property_link = service_override.get('value', None)
            if linked_items_type is None:
                links.append(check_link)
                if host_link:
                    links.append(host_link)
            else:
                if linked_items_type in ITEM_TYPE.ALL_SERVICES:
                    if check_link['exists']:
                        if check_link['item_type'] == linked_items_type:
                            links.append(check_link)
                    else:
                        links.append(check_link)
                elif host_link and linked_items_type in ITEM_TYPE.ALL_HOST_CLASS:
                    if host_link['exists']:
                        if host_link['item_type'] == linked_items_type:
                            links.append(host_link)
                    else:
                        links.append(host_link)
            
            property_type = None
            check_type = check_link['item_type'] if check_link['exists'] else DEF_ITEMS[self.get_type()]['check_type']
            
            if isinstance(property_link, dict):
                property_types = DEF_ITEMS[check_type]['props_links'].get(service_override['key'], [])
                if len(property_types) == 1:
                    property_type = property_types[0]
                elif len(property_types) > 1:
                    # This only manage simple standard links; should be updated for all links
                    # but for now there aren't possible value in service override which is not a simple link
                    raise NotImplementedError
                
                for prop_link in property_link['links']:
                    if linked_items_type is None:
                        links.append(prop_link)
                    elif prop_link['exists'] and prop_link['item_type'] == linked_items_type:
                        links.append(prop_link)
                    elif property_type == linked_items_type:
                        links.append(prop_link)
        
        return links
    
    
    def _update_link_service_overrides(self, property_name, item):
        property_value = self.get(property_name, '')
        if not property_value or isinstance(property_value, basestring):
            return False
        
        all_types_cache = set(property_value['all_linked_types'])
        effective = False
        raw_item_value = METADATA.get_metadata(self, METADATA.RAW_ITEM, {})[property_name]
        
        for link_index, service_override in enumerate(property_value.get('links', [])):
            _effective = self._update_service_override_value_link(all_types_cache, item, link_index, service_override, raw_item_value)
            effective = effective or _effective
        
        property_value['all_linked_types'] = list(all_types_cache)
        if effective:
            raw_item_value['all_linked_types'] = list(all_types_cache)
        return effective
    
    
    def _update_service_override_value_link(self, all_types_cache, item, link_index, service_override, raw_item_value):
        effective = False
        # Check if the item to which we must update the link is the value of an overridden property
        property_links = service_override[u'value']
        property_name = service_override[u'key']
        # build link fail if DEF_ITEMS[self.get_type()]['props_links'] return more than one value
        check_type = DEF_ITEMS[self.get_type()][u'check_type']
        property_type = DEF_ITEMS[check_type][u'props_links'].get(property_name, [u'no link property'])[0]
        if isinstance(property_links, dict) and item.get_type() == property_type:
            for property_link_index, link in enumerate(property_links[u'links']):
                if link[u'exists'] and link[u'_id'] == item[u'_id']:
                    link[AT_LINK_PROP_NAME][item.get_state()] = item
                elif not link[u'exists'] and link[u'name'].lower() == item.get_name().lower() and item.get_state() in LINKIFY_MANAGE_STATES:
                    link.pop(u'name', None)
                    link[u'_id'] = item[u'_id']
                    link[u'item_type'] = property_type
                    link[u'exists'] = True
                    if link_index < len(raw_item_value[u'links']):
                        raw_item_value[u'links'][link_index][u'value'][u'links'][property_link_index] = link.copy()
                        effective = True
                    link[AT_LINK_PROP_NAME] = link.get(AT_LINK_PROP_NAME, {})
                    link[AT_LINK_PROP_NAME][item.get_state()] = item
                    all_types_cache.add(property_type)
        
        return effective
    
    
    @staticmethod
    def flatten_not_linked_prop(property_value, property_name, prop_item_states, datamanagerV2):
        if not property_value:
            return ''
        if not prop_item_states:
            raise Exception('missing prop_item_states param in _flatten_prop_service_overrides')
        if not isinstance(property_value, dict):
            raise Exception('raw_item_value param in _flatten_prop_service_overrides is not a dict type it is a [%s] type value [%s]' % (type(property_value), property_value))
        if property_value.get('has_error', False):
            return property_value.get('raw_value', '')
        
        flattened_infos = {}
        for link in property_value['links']:
            check_name = ServiceOverridesMixin.get_check_name_with_datamanager(link, datamanagerV2, prop_item_states)
            check_id = link['check_link'].get('_id', 'unexisting')
            check_type = ITEM_TYPE.SERVICESHOSTS
            link_key = link['key']
            if can_property_contain_links(check_type, link_key):
                link_value = datamanagerV2.flatten_value(link['value'], check_type, link_key, prop_item_states)
            else:
                link_value = link['value']
            
            key_ = (check_name, link_key)
            flattened_info = flattened_infos.get(key_, [])
            flattened_info.append((check_name, check_id, link_key, link_value))
            flattened_infos[key_] = flattened_info
        
        clean_flattened_infos = []
        for flattened_info_key, flattened_info in flattened_infos.items():
            if len(flattened_info) > 1:
                unknown_counter = 0
                for (check_name, check_id, link_key, link_value) in flattened_info:
                    if check_id == 'unexisting':
                        unknown_counter += 1
                        if unknown_counter > 1:
                            check_id = 'unexisting-%s' % (unknown_counter - 1)
                    clean_flattened_infos.append((check_name + SERVICE_OVERRIDE_UUID_SEP + check_id, check_id, link_key, link_value))
            
            else:
                clean_flattened_infos.append(flattened_info[0])
        
        return "-=#=-".join(["%s, %s %s" % (check_name, link_key, link_value) for check_name, check_id, link_key, link_value in clean_flattened_infos])
    
    
    @staticmethod
    def get_check_name_with_datamanager(link, datamanagerV2, prop_item_states):
        check_name = _get_item_name_from_datamanager(link['check_link'], prop_item_states, datamanagerV2)
        dfe_key = link.get('dfe_key', '')
        if dfe_key:
            check_name = check_name.replace('$KEY$', dfe_key)
        return check_name
    
    
    def _flatten_prop_service_overrides(self, raw_item_value, property_name, prop_item_states, for_arbiter=False):
        if not raw_item_value:
            return ''
        if not prop_item_states:
            raise Exception('missing prop_item_states param in _flatten_prop_service_overrides')
        if not isinstance(raw_item_value, dict):
            raise Exception('raw_item_value param in _flatten_prop_service_overrides is not a dict type it is a [%s] type value [%s]' % (type(raw_item_value), raw_item_value))
        if raw_item_value.get('has_error', False):
            return raw_item_value.get('raw_value', '')
        
        flattened_infos = {}
        clean_flattened_infos = []
        for link_index in xrange(len(raw_item_value['links'])):
            link = self.get(SERVICE_OVERRIDE, None)['links'][link_index]
            link_key = link['key']
            check_link = link['check_link']
            check_id = link['check_link'].get('_id', 'unexisting')
            if for_arbiter:
                if check_link['exists']:
                    check_name = '%s%s%s%s' % (PREFIX_LINK_UUID, check_link['_id'], PREFIX_LINK_DFE_KEY, link.get('dfe_key', '-'))
                else:
                    check_name = check_link['name']
            else:
                check_name = _get_item_name_from_link(check_link, prop_item_states, self)
            
            dfe_key = link.get('dfe_key', '')
            if dfe_key:
                check_name = check_name.replace('$KEY$', dfe_key)
            link_value = self._flatten_service_override_value(link['value'], prop_item_states=prop_item_states)
            
            key_ = (check_name, link_key)
            flattened_info = flattened_infos.get(key_, [])
            flattened_info.append((check_name, check_id, link_key, link_value))
            clean_flattened_infos.append((check_name, check_id, link_key, link_value))
            flattened_infos[key_] = flattened_info
        
        if not for_arbiter:
            clean_flattened_infos = []
            for flattened_info_key, flattened_info in flattened_infos.items():
                if len(flattened_info) > 1:
                    unknown_counter = 0
                    for (check_name, check_id, link_key, link_value) in flattened_info:
                        if check_id == 'unexisting':
                            unknown_counter += 1
                            if unknown_counter > 1:
                                check_id = 'unexisting-%s' % (unknown_counter - 1)
                        clean_flattened_infos.append((check_name + SERVICE_OVERRIDE_UUID_SEP + check_id, check_id, link_key, link_value))
                
                else:
                    clean_flattened_infos.append(flattened_info[0])
        
        return "-=#=-".join(["%s, %s %s" % (check_name, link_key, link_value) for check_name, check_id, link_key, link_value in clean_flattened_infos])
    
    
    def _flatten_service_override_value(self, service_override_value, prop_item_states):
        if isinstance(service_override_value, basestring):
            return service_override_value
        
        item_name_links = []
        for link_info in service_override_value.get('links', []):
            link_item_name = _get_item_name_from_link(link_info, prop_item_states, self)
            item_name_links.append(link_item_name)
        
        if len(item_name_links) > 1:
            item_name_links = (n for n in item_name_links if n != 'null')
        
        plus_as_string = '+' if service_override_value.get('has_plus', False) else ''
        return '%s%s' % (plus_as_string, ','.join(item_name_links))
    
    
    def _remove_item_link_service_overrides(self, _property_name, item_to_unlink):
        service_overrides = self.get(SERVICE_OVERRIDE, None)
        if not (service_overrides and isinstance(service_overrides, dict)):
            return False
        
        effective = False
        item_state = item_to_unlink.get_state()
        raw_item_service_overrides = METADATA.get_metadata(self, METADATA.RAW_ITEM, {}).get(SERVICE_OVERRIDE, {}).get('links', [])
        override_properties_indexes_to_remove = []
        
        for override_property_index, override_property in enumerate(service_overrides['links']):
            if not isinstance(override_property['value'], dict):
                continue
            
            raw_item_override_property = self._get_raw_service_override(override_property, override_property_index)
            values_links_to_remove = []
            for value_link_index, value_link in enumerate(override_property['value']['links']):
                if value_link.get('_id', None) == item_to_unlink['_id'] and item_state in value_link.get(AT_LINK_PROP_NAME, {}):
                    del value_link[AT_LINK_PROP_NAME][item_state]
                    if not value_link.get(AT_LINK_PROP_NAME, {}):
                        del value_link[AT_LINK_PROP_NAME]
                        values_links_to_remove.append(value_link)
            
            if values_links_to_remove:
                for value_link in values_links_to_remove:
                    override_property['value']['links'].remove(value_link)
                    if raw_item_override_property:
                        effective = True
                        raw_item_override_property['value']['links'].remove(value_link)
                
                if not override_property['value']['links']:
                    override_properties_indexes_to_remove.append(override_property_index)
        
        if override_properties_indexes_to_remove:
            override_properties_indexes_to_remove.reverse()
            for override_property_index in override_properties_indexes_to_remove:
                service_overrides['links'].pop(override_property_index)
                if override_property_index < len(raw_item_service_overrides):
                    # The firsts service_override are the raw_item ones. If the index is more than the len, it's inherited service_overrides
                    del raw_item_service_overrides[override_property_index]
        if not service_overrides['links']:
            # The del remove also the raw_item key, so don't need to do it
            del self[SERVICE_OVERRIDE]
        return effective
    
    
    def _set_inexisting_link_service_overrides(self, property_name, item_to_unlink):
        property_value = self.get(SERVICE_OVERRIDE, None)
        if not (property_value and isinstance(property_value, dict)):
            return False
        
        effective = False
        raw_item_links = METADATA.get_metadata(self, METADATA.RAW_ITEM, {}).get(SERVICE_OVERRIDE, {}).get('links', [])
        for link_index, service_override in enumerate(property_value['links']):
            value = service_override['value']
            if isinstance(value, dict) and value.get('_id', None) == item_to_unlink['_id']:
                service_override['value'] = {
                    'exists': False,
                    'name'  : item_to_unlink.get_name()
                }
                if link_index < len(raw_item_links):
                    raw_item_links[link_index]['value'] = {
                        'exists': False,
                        'name'  : item_to_unlink.get_name()
                    }
                    effective = True
        return effective
    
    
    def _on_remove_check_service_override(self, check):
        # type: (Union[BaseItem, ServiceOverridesMixin], ServiceItem) -> Tuple[bool, Optional[Dict]]
        effective = False
        new_link = None
        property_value = self.get(SERVICE_OVERRIDE, None)
        if not property_value:
            return effective, new_link
        
        raw_item_value = METADATA.get_metadata(self, METADATA.RAW_ITEM, {}).get(SERVICE_OVERRIDE, {})
        raw_item_links = raw_item_value.get(u'links', [])
        all_types_cache = set(property_value[u'all_linked_types'])
        all_types_cache.discard(check.get_type())
        all_types_cache.discard(DEF_ITEMS[check.get_type()][u'apply_on_type'])
        for link_index, service_override in enumerate(property_value.get(u'links', [])):
            check_link = service_override[u'check_link']
            if not check_link[u'exists']:
                continue
            elif check_link[u'_id'] != check[u'_id']:
                all_types_cache.add(check_link[u'item_type'])
                host_link = service_override.get(u'host_link', None)
                if host_link and len(host_link) > 0 and host_link[u'exists']:
                    all_types_cache.add(host_link[u'item_type'])
                continue
            
            # Try to find a check on which the override can link and link it
            new_link = self._find_matching_check_in_check_links(check)
            if new_link:
                service_override[u'host_link'] = {
                    u'item_type': new_link[u'from'].get_type(),
                    u'_id'      : new_link[u'from'][u'_id'],
                    u'exists'   : True
                }
                service_override[u'check_link'] = {
                    u'item_type': new_link[u'check'].get_type(),
                    u'_id'      : new_link[u'check'][u'_id'],
                    u'exists'   : True
                }
                if link_index < len(raw_item_links):
                    raw_item_links[link_index][u'host_link'] = service_override[u'host_link'].copy()
                    raw_item_links[link_index][u'check_link'] = service_override[u'check_link'].copy()
                else:
                    raw_item_links.append(service_override.copy())
                
                service_override[u'host_link'][u'@link'] = {
                    new_link[u'from'].get_state(): new_link[u'from']
                }
                service_override[u'check_link'][u'@link'] = {
                    new_link[u'check'].get_state(): new_link[u'check']
                }
                all_types_cache.add(new_link[u'check'].get_type())
                all_types_cache.add(new_link[u'from'].get_type())
            else:
                service_override[u'has_already_been_linked'] = False
                service_override[u'host_link'] = None
                service_override[u'check_link'] = {
                    u'exists': False,
                    u'name'  : check.get_name()
                }
                
                if link_index < len(raw_item_links):
                    raw_item_links[link_index][u'has_already_been_linked'] = False
                    raw_item_links[link_index][u'host_link'] = None
                    raw_item_links[link_index][u'check_link'] = {
                        u'exists': False,
                        u'name'  : check.get_name()
                    }
            if not service_override[u'check_link'][u'exists']:
                all_types_cache.add(u'__ALL_SERVICES__')
            if not service_override[u'host_link'] or not service_override[u'host_link'][u'exists']:
                all_types_cache.add(u'__ALL_HOSTS__')
            effective = True
        
        property_value[u'all_linked_types'] = list(all_types_cache)
        raw_item_value[u'all_linked_types'] = list(all_types_cache)
        
        return effective, new_link
    
    
    def _find_matching_check_in_check_links(self, check):
        # type: (Union[BaseItem, ServiceOverridesMixin], ServiceItem) -> Optional[Dict]
        my_check_links = self.get_link_checks()
        
        for template in [self] + self.get_link_items(u'use', only_exist=True):
            for link_to_check in my_check_links:
                if link_to_check[u'check'] == check:
                    continue
                if link_to_check[u'from'] == template and link_to_check[u'check'].get_name() == check.get_name():
                    return link_to_check
        return None
    
    
    def _on_add_check_service_override(self, check, host):
        # type: (Union[Dict, ServiceOverridesMixin], BaseItem, BaseItem) -> bool
        effective = False
        property_value = self.get(SERVICE_OVERRIDE, None)
        if not property_value:
            return effective
        
        all_types_cache = self._get_all_types_cache_from_property_value(property_value)
        raw_item_value = METADATA.get_metadata(self, METADATA.RAW_ITEM, {}).get(SERVICE_OVERRIDE, None)
        
        for link_index, service_override_entry in enumerate(property_value.get(u'links', [])):
            try:
                self._update_all_types_cache_from_service_override_entry(service_override_entry, all_types_cache)
                check_link, dfe_key = self._create_updated_check_link_from_service_override_entry(check, service_override_entry)
            except ServiceOverridesMixinException:
                continue
            
            # We do not want to override effective once it's True
            if self._update_items_with_new_check_link(check, all_types_cache, raw_item_value, link_index, check_link, dfe_key):
                effective = True
            
            self._update_host_link(host, service_override_entry, all_types_cache, raw_item_value, link_index)
        
        property_value[u'all_linked_types'] = list(all_types_cache)
        if effective:
            raw_item_value[u'all_linked_types'] = list(all_types_cache)
        
        return effective
    
    
    @staticmethod
    def _get_all_types_cache_from_property_value(property_value):
        # type: (Dict) -> Set
        all_types_cache = set(property_value[u'all_linked_types'])
        if u'__ALL_SERVICES__' in all_types_cache:
            all_types_cache.remove(u'__ALL_SERVICES__')
        if u'__ALL_HOSTS__' in all_types_cache:
            all_types_cache.remove(u'__ALL_HOSTS__')
        return all_types_cache
    
    
    @staticmethod
    def _update_all_types_cache_from_service_override_entry(service_override_entry, all_types_cache):
        # type: (Dict, Set) -> None
        if service_override_entry.get(u'has_already_been_linked', False):
            if not service_override_entry[u'check_link'][u'exists']:
                all_types_cache.add(u'__ALL_SERVICES__')
            if not service_override_entry[u'host_link'] or not service_override_entry[u'host_link'][u'exists']:
                all_types_cache.add(u'__ALL_HOSTS__')
            raise ServiceOverridesMixinException(u'Service override is already linked.')
    
    
    def _create_updated_check_link_from_service_override_entry(self, check, service_override_entry):
        # type: (BaseItem, Dict) -> Tuple[Dict, unicode]
        check_link = service_override_entry[u'check_link']
        if check_link[u'exists']:
            if check_link[u'_id'] == check[u'_id']:
                check_link[AT_LINK_PROP_NAME][check.get_state()] = check
            raise ServiceOverridesMixinException(u'Check link already exists.')
        
        check_link_name = check_link[u'name']
        dfe_key = None
        if check.is_dfe():
            generated_check_names = {}
            dfe_property_name = check[u'duplicate_foreach']
            
            dfe_values = self.get(dfe_property_name, u'')
            if not dfe_values:
                raise ServiceOverridesMixinException(u'Check is not duplicate for each.')
            
            for dfe_value in dfe_values.split(u','):
                check_name = check.get_name().replace(u'$KEY$', dfe_value)
                generated_check_names[check_name] = dfe_value
            
            if check_link_name not in generated_check_names:
                raise ServiceOverridesMixinException(u'Check is not related to this host.')
            
            check_link = {
                u'_id'      : check[u'_id'],
                u'item_type': check.get_type(),
                u'exists'   : True
            }
            dfe_key = generated_check_names[check_link_name]
            service_override_entry[u'dfe_key'] = dfe_key
        else:
            if check_link_name != check.get_name():
                raise ServiceOverridesMixinException(u'Check is not related to this host.')
            
            check_link = {
                u'_id'      : check[u'_id'],
                u'item_type': check.get_type(),
                u'exists'   : True
            }
        
        service_override_entry[u'check_link'] = check_link
        return check_link, dfe_key
    
    
    @staticmethod
    def _update_host_link(host, service_override_entry, all_types_cache, raw_item_value, link_index):
        # type: (BaseItem, Dict, Set, Dict, int) -> None
        if host.get_state() in LINKIFY_MANAGE_STATES:
            host_link = {
                u'exists'   : True,
                u'item_type': host.get_type(),
                u'_id'      : host[u'_id']
            }
            service_override_entry[u'host_link'] = host_link
            
            all_types_cache.add(host.get_type())
            if raw_item_value and link_index < len(raw_item_value[u'links']):
                raw_item_value[u'links'][link_index][u'host_link'] = host_link.copy()
            
            host_link[AT_LINK_PROP_NAME] = host_link.get(AT_LINK_PROP_NAME, {})
            host_link[AT_LINK_PROP_NAME][host.get_state()] = host
        
        if not service_override_entry[u'check_link'][u'exists']:
            all_types_cache.add(u'__ALL_SERVICES__')
        if not service_override_entry[u'host_link'] or not service_override_entry[u'host_link'][u'exists']:
            all_types_cache.add(u'__ALL_HOSTS__')
    
    
    @staticmethod
    def _update_items_with_new_check_link(check, all_types_cache, raw_item_value, link_index, check_link, dfe_key):
        # type: (BaseItem, Set, Dict, int, Dict, unicode) -> bool
        effective = False
        all_types_cache.add(check.get_type())
        if raw_item_value and link_index < len(raw_item_value[u'links']):
            raw_item_value[u'links'][link_index][u'check_link'] = check_link.copy()
            if dfe_key:
                raw_item_value[u'links'][link_index][u'dfe_key'] = dfe_key
            effective = True
        
        check_link[AT_LINK_PROP_NAME] = check_link.get(AT_LINK_PROP_NAME, {})
        check_link[AT_LINK_PROP_NAME][check.get_state()] = check
        return effective
    
    
    def _get_raw_service_override(self, value, index):
        # type: (Dict, int) -> Union[None, Dict]
        
        raw_value = None
        
        raw_item_service_override = METADATA.get_metadata(self, METADATA.RAW_ITEM, {}).get(SERVICE_OVERRIDE, {})
        
        # the raw value are set firsts on the item. So if the index is more than the len of the raw_value, it's an inherited value
        if raw_item_service_override and index < len(raw_item_service_override[u'links']):
            raw_value = raw_item_service_override['links'][index]
        
        if not self.is_same_override(raw_value, value):
            for raw_value_entry in raw_item_service_override.get('links', []):
                if self.is_same_override(raw_value_entry, value):
                    logger.warning(u'The item %s has a defined service_override on itself but the raw_value is not ordered like it could be' % self)
                    return raw_value_entry
            
            # Oh ! No match, log it and go out
            logger.error(u'The item %s has a defined service_override on itself but this service_override does not exist in the raw_value' % self)
            return None
        return raw_value
    
    
    @staticmethod
    def is_same_override(raw_value_entry, value):
        # type: (Union[Dict, None], Dict) -> bool
        if not raw_value_entry:
            return False
        
        if raw_value_entry.get(u'key', u'') != value.get(u'key', u'') or \
                raw_value_entry.get(u'dfe_key', u'') != value.get(u'dfe_key', u''):
            return False
        
        for key, raw_value in raw_value_entry[u'check_link'].iteritems():
            if raw_value != value[u'check_link'][key]:
                return False
        
        if not raw_value_entry[u'host_link'] and value[u'host_link']:
            return False
        
        if raw_value_entry[u'host_link'] and not value[u'host_link']:
            return False
        
        if raw_value_entry[u'host_link'] and value[u'host_link']:
            for key, raw_value in raw_value_entry[u'host_link'].iteritems():
                if raw_value != value[u'host_link'].get(key, None):
                    return False
        
        # Same key, same dfe_key, same check_link, same host_link, okay it's you !
        return True


# Abstract class : use sub class LinkHostCheckForHostMixin or LinkHostCheckForCheckMixin
class _LinkHostCheckMixin(object):
    def _link_check_template_to_host(self, check, host, reverse_links_checks, host_to_save, level=0, _from_host=None, add_on_sons=True, to_keep_checks=None, to_keep_hosts=None, already_visited=None, data_has_changed=False):
        if already_visited is None:
            already_visited = set()
        checks = host.get_link_checks()
        # if check already in host
        check_link = next((i for i in checks if i['check'] == check), None)
        
        if check_link:
            # Note : 'if check_to_keep is not None' cannot be 'if check_to_keep' for this one because check_to_keep can be a empty array
            if to_keep_checks is not None:
                to_keep_checks.append(check_link)
            if to_keep_hosts is not None:
                _LinkHostCheckMixin._get_all_sons(host, to_keep_hosts)
            return
        
        # we test if the host match the check host_name
        all_template = host.find_all_templates_by_links()
        if ITEM_TYPE.is_template(host.get_type()):
            all_template.append(host)
        check_on_me = do_match(check['host_name']['node'], all_template)
        
        if check_on_me:
            check_link = host.add_check(check, _from_host, reverse_links_checks, host_to_save, data_has_changed)
            if to_keep_checks is not None:
                to_keep_checks.append(check_link)
            if to_keep_hosts is not None:
                to_keep_hosts.append(host)
        
        # we add the check on all host sons
        if add_on_sons and ITEM_TYPE.is_template(host.get_type()):
            for son in host.get_sons():
                if son.get_key() in already_visited:
                    continue
                already_visited.add(son.get_key())
                self._link_check_template_to_host(
                    check,
                    son,
                    reverse_links_checks,
                    host_to_save,
                    level=level + 1,
                    _from_host=host,
                    to_keep_checks=to_keep_checks,
                    to_keep_hosts=to_keep_hosts,
                    already_visited=already_visited,
                    data_has_changed=data_has_changed
                )
    
    
    # TODO : this method is not currently use and the from in METADATA check is not accurate.
    # see test_framework.test_synchronizer.test_dao.dataprovider.test_dataprovider_metadata_memory.TestDataProviderMetadataMemory#__test_update_from_for_check_is_host_template_order_change
    # For a use case of inaccurate from
    # It will be fix in next release
    @staticmethod
    def _compute_from_template_where_the_check_are_coming(check, host):
        return _LinkHostCheckMixin._compute_from_template_the_check_are_coming(check.get_all_state_link_items('host_name'), [host])
    
    
    @staticmethod
    def _compute_from_template_the_check_are_coming(in_check_tpls, hosts):
        for host_tpl in hosts:
            if host_tpl in in_check_tpls:
                return host_tpl
            return _LinkHostCheckMixin._compute_from_template_the_check_are_coming(in_check_tpls, host_tpl.get_all_state_link_items('use', only_exist=True))
        # We ask from for a check we didn't have
        return None
    
    
    @staticmethod
    def _get_all_sons(host, to_keep_hosts, deep=0):
        if to_keep_hosts is None:
            to_keep_hosts = []
        to_keep_hosts.append(host)
        if deep > MAX_TEMPLATE_LOOKUP_LEVEL:
            logger.error('[_link_check_template_to_host] There are too many templates used for [%s]' % host)
            return
        host_sons = METADATA.get_metadata(host, METADATA.SONS)
        if host_sons:
            for son in host_sons.itervalues():
                _LinkHostCheckMixin._get_all_sons(son, to_keep_hosts, deep=deep + 1)


class LinkHostCheckForHostMixin(_LinkHostCheckMixin):
    
    def get_link_checks(self):
        # type: (Union[Dict, LinkHostCheckForHostMixin]) -> Dict
        return METADATA.get_metadata(self, METADATA.CHECKS, [])
    
    
    def add_check(self, check, host_from, reverse_links_checks, host_to_save, data_has_changed=False):
        # type: (Union[ServiceOverridesMixin ,LinkHostCheckForHostMixin], BaseItem, BaseItem, List, List, bool) -> Dict
        checks = self.get_link_checks()
        host_from = host_from if host_from else self
        check_link = {'check': check, 'from': host_from}
        checks.append(check_link)
        METADATA.update_metadata(self, METADATA.CHECKS, checks)
        if self not in reverse_links_checks:
            reverse_links_checks.append(self)
        
        # we need the 2 call here
        must_save = self._on_add_check_service_override(check, host_from)
        must_save = self._on_add_check_service_excludes_by_id(check) or must_save
        if must_save:
            host_to_save.append(self)
        return check_link
    
    
    def remove_check(self, check, host_to_save):
        # type: (Union[BaseItem,LinkHostCheckForHostMixin], BaseItem, List[BaseItem]) -> None
        new_link_checks = []
        for check_link in self.get_link_checks():
            if check_link['check'] == check:
                # we need the 2 call here
                effective, new_link = self._on_remove_check_service_override(check)
                effective = self._on_remove_check_service_excludes_by_id(check) or effective
                if effective:
                    if new_link:
                        new_link_checks.append(new_link)
                    host_to_save.append(self)
            else:
                new_link_checks.append(check_link)
        METADATA.update_metadata(self, METADATA.CHECKS, new_link_checks)
    
    
    def _unlinkify_host_check(self, host_to_save):
        # type: (Union[BaseItem,LinkHostCheckForHostMixin], BaseItem) -> None
        for check_link in self.get_link_checks():
            reverse_links_checks = [i for i in METADATA.get_metadata(check_link['check'], METADATA.REVERSE_LINKS_CHECKS, []) if i.get_key() != self.get_key()]
            METADATA.update_metadata(check_link['check'], METADATA.REVERSE_LINKS_CHECKS, reverse_links_checks)
    
    
    def _linkify_host_check(self, old_item_in_rg_use_only_exist, _find_by_name, host_to_save, modified_data_keys=None):
        # type: (Union[UseMixin, LinkHostCheckForHostMixin] ,BaseItem, callable, BaseItem, Set) -> None
        if modified_data_keys is None:
            modified_data_keys = set()
        rg_use_only_exist = self.get_use_only_exist() or u'new_use'
        if not old_item_in_rg_use_only_exist or old_item_in_rg_use_only_exist != rg_use_only_exist or modified_data_keys:
            self._link_check_to_host_on_update(host_to_save, modified_data_keys=modified_data_keys)
    
    
    def _link_check_to_host_on_update(self, host_to_save, hosts_visited=None, level=32, modified_data_keys=None):
        # type: (Union[BaseItem, LinkHostCheckForHostMixin], List, List, int, Set) -> None
        if level <= 0 or (hosts_visited and self.get_key() in hosts_visited):
            return
        
        if modified_data_keys is None:
            modified_data_keys = set()
        
        if hosts_visited:
            hosts_visited.append(self.get_key())
        else:
            hosts_visited = [self.get_key()]
        
        if modified_data_keys:
            check_links = self.get_link_checks()
            for check_link in check_links:
                check = check_link[u'check']
                if not check.is_dfe():
                    continue
                if not check.get(u'duplicate_foreach', u'') in modified_data_keys:
                    continue
                
                reverse_links_checks = METADATA.get_metadata(check, METADATA.REVERSE_LINKS_CHECKS, [])
                if self._on_add_check_service_override(check, self):
                    host_to_save.append(self)
                METADATA.update_metadata(check, METADATA.REVERSE_LINKS_CHECKS, reverse_links_checks)
        
        parents = self.get_all_state_link_items(u'use', only_exist=True)
        if parents:
            to_keep_checks = [check_link for check_link in self.get_link_checks() if check_link[u'from'] == self]
            
            for parent in parents:
                for check_link in parent.get_link_checks():
                    check = check_link[u'check']
                    reverse_links_checks = METADATA.get_metadata(check, METADATA.REVERSE_LINKS_CHECKS, [])
                    self._link_check_template_to_host(check, self, reverse_links_checks, host_to_save, add_on_sons=False, _from_host=parent, to_keep_checks=to_keep_checks)
                    METADATA.update_metadata(check, METADATA.REVERSE_LINKS_CHECKS, reverse_links_checks)
            
            check_to_remove = (check_link for check_link in self.get_link_checks() if check_link not in to_keep_checks)
            for check in check_to_remove:
                self.remove_check(check[u'check'], host_to_save)
        else:
            for check in [check_link for check_link in self.get_link_checks() if check_link[u'from'] != self]:
                self.remove_check(check[u'check'], host_to_save)
        
        if ITEM_TYPE.is_template(self.get_type()):
            for son in self.get_sons():  # type: Union[BaseItem, UseMixin, LinkHostCheckForHostMixin]
                son._link_check_to_host_on_update(host_to_save, hosts_visited=hosts_visited, level=level - 1, modified_data_keys=modified_data_keys)


class LinkHostCheckForCheckMixin(_LinkHostCheckMixin):
    
    def _unlinkify_host_check(self, host_to_save):
        reverse_links_checks = METADATA.get_metadata(self, METADATA.REVERSE_LINKS_CHECKS, [])
        for host in reverse_links_checks:
            host.remove_check(self, host_to_save)
    
    
    def _linkify_host_check(self, old_item_in_rg_use_only_exist, find_by_name, host_to_save, has_dfe_key_changed):
        # type: (bool, callable, List[BaseItem], bool) -> None
        # Special link check <-> host
        if self.get_type() in (ITEM_TYPE.SERVICESHOSTS, ITEM_TYPE.SERVICESCLUSTERS):
            self._link_host_check_on_check_change(find_by_name, host_to_save)
        
        if self.get_type() in (ITEM_TYPE.SERVICESHOSTTPLS, ITEM_TYPE.SERVICESCLUSTERTPLS):
            self._link_host_template_check_template_on_check_change(find_by_name, host_to_save)
    
    
    def _link_host_template_check_template_on_check_change(self, find_by_name, host_to_save):
        link_hosts = self.get_all_state_link_items('host_name')
        old_link_hosts = METADATA.get_metadata(self, METADATA.REVERSE_LINKS_CHECKS, [])
        
        to_keep_hosts = []
        for link_host in link_hosts:
            if not link_host.get('exists', True):
                link_host = find_by_name(link_host.get_name(), ITEM_TYPE.HOSTS, ITEM_STATE.NEW)
            if not link_host:
                continue
            
            self._link_check_template_to_host(self, link_host, old_link_hosts, host_to_save, to_keep_hosts=to_keep_hosts)
        
        to_remove_hosts = [h for h in old_link_hosts if h not in to_keep_hosts]
        for to_remove_host in to_remove_hosts:
            to_remove_host.remove_check(self, host_to_save)
            old_link_hosts.remove(to_remove_host)
        
        METADATA.update_metadata(self, METADATA.REVERSE_LINKS_CHECKS, old_link_hosts)
    
    
    def _link_host_check_on_check_change(self, find_by_name, host_to_save):
        link_hosts = self.get_all_state_link_items('host_name')
        old_link_hosts = METADATA.get_metadata(self, METADATA.REVERSE_LINKS_CHECKS, [])
        
        to_remove_hosts = [old_link_host for old_link_host in old_link_hosts if old_link_host not in link_hosts]
        to_add_hosts = [link_host for link_host in link_hosts if link_host not in old_link_hosts]
        
        for to_remove_host in to_remove_hosts:
            to_remove_host.remove_check(self, host_to_save)
            old_link_hosts.remove(to_remove_host)
        
        for to_add_host in to_add_hosts:
            if not to_add_host.get('exists', True):
                to_add_host = find_by_name(to_add_host.get_name(), ITEM_TYPE.HOSTS, ITEM_STATE.NEW)
            if not to_add_host:
                continue
            
            to_add_host.add_check(self, to_add_host, old_link_hosts, host_to_save)
        METADATA.update_metadata(self, METADATA.REVERSE_LINKS_CHECKS, old_link_hosts)


class ServiceExcludesMixin(object):
    
    def get_service_excludes_by_id(self):
        # type: () -> unicode
        service_links = self.get(SERVICE_EXCLUDES_BY_ID, {}).get(u'links', [])
        if not service_links:
            return u''
        
        services_names = []
        for link in service_links:
            service = link.get(u'@link', {}).get(ITEM_STATE.STAGGING, None)  # type: Optional[BaseItem]
            if service is None:
                continue
            
            services_names.append(service.get_name())
        
        return u', '.join(services_names) if services_names else u''
    
    
    def _flatten_prop_service_excludes_by_id(self, raw_item_value, property_name, prop_item_states, for_arbiter=False):
        flattened_value = []
        for link in raw_item_value['links']:
            if link['exists']:
                check_value = "%s%s" % (PREFIX_LINK_UUID, link['_id'])
                if AT_LINK_PROP_NAME in link:
                    for state in prop_item_states:
                        if link[AT_LINK_PROP_NAME][state]:
                            flattened_value.append(check_value)
                            break
                else:
                    flattened_value.append(check_value)
        
        return ",".join(flattened_value)
    
    
    @staticmethod
    def flatten_not_linked_prop(property_value, prop_to_flatten, flatten_states, self):
        item_name_links = []
        for link in property_value['links']:
            if link['exists']:
                link_item_type = link['item_type']
                for link_item_state in flatten_states:
                    item = self.find_item_by_id(link['_id'], link_item_type, link_item_state)
                    if item:
                        item_name_links.append("%s%s" % (PREFIX_LINK_UUID, link['_id']))
                        break
        
        if len(item_name_links) > 1:
            item_name_links = [n for n in item_name_links if n != 'null']
        
        plus_as_string = '+' if property_value.get('has_plus', False) else ''
        return '%s%s' % (plus_as_string, ','.join(item_name_links))
    
    
    def _on_add_check_service_excludes_by_id(self, check):
        effective = False
        property_value = self.get(SERVICE_EXCLUDES_BY_ID)
        if not property_value:
            return effective
        raw_item_value = METADATA.get_metadata(self, METADATA.RAW_ITEM, {}).get(SERVICE_EXCLUDES_BY_ID)
        for (link_index, service_exclude) in enumerate(property_value['links']):
            if service_exclude['exists']:
                if service_exclude['_id'] == check['_id']:
                    service_exclude[AT_LINK_PROP_NAME][check.get_state()] = check
                continue
            
            check_link = {
                '_id'      : check['_id'],
                'item_type': check.get_type(),
                'exists'   : True
            }
            property_value['links'][link_index] = check_link
            
            if raw_item_value and link_index < len(raw_item_value['links']):
                raw_item_value['links'][link_index]['check_link'] = check_link.copy()
                effective = True
            
            check_link[AT_LINK_PROP_NAME] = check_link.get(AT_LINK_PROP_NAME, {})
            check_link[AT_LINK_PROP_NAME][check.get_state()] = check
        
        return effective
    
    
    def _on_remove_check_service_excludes_by_id(self, check):
        # type: (Union[InstanceItem, ServiceExcludesMixin], InstanceItem) -> bool
        
        effective = False
        property_value = self.get(SERVICE_EXCLUDES_BY_ID, None)
        if not property_value:
            return effective
        
        raw_item = METADATA.get_metadata(self, METADATA.RAW_ITEM, {})
        raw_item_links = raw_item.get(SERVICE_EXCLUDES_BY_ID, {}).get(u'links', [])
        
        index_to_del = [link_index for link_index, _link in enumerate(property_value.get(u'links', [])) if _link[u'exists'] and _link[u'_id'] == check[u'_id']]
        
        # this prop is not HERITABLE so the raw_item value is the same as the item value. see def_items.NO_HERITABLE_ATTR
        for index in index_to_del:
            del property_value[u'links'][index]
            del raw_item_links[index]
            effective = True
        
        if not property_value[u'links']:
            self.pop(SERVICE_EXCLUDES_BY_ID)
            raw_item.pop(SERVICE_EXCLUDES_BY_ID)
        return effective
    
    
    def is_service_excluded_by_template(self, check, already_checked=None):
        if already_checked is None:
            already_checked = set()
        if check not in [i['check'] for i in self.get_link_checks()]:
            return EXCLUDE_FROM_TEMPLATES_RESULTS.DONT_HAVE
        
        if self._is_check_excluded(check):
            return EXCLUDE_FROM_TEMPLATES_RESULTS.EXCLUDE
        else:
            result = EXCLUDE_FROM_TEMPLATES_RESULTS.ALLOW
            for sub_template in self.get_parents():
                if sub_template.get_key() in already_checked:
                    continue
                already_checked.add(sub_template.get_key())
                sub_result = sub_template.is_service_excluded_by_template(check, already_checked=already_checked)
                if sub_result == EXCLUDE_FROM_TEMPLATES_RESULTS.DONT_HAVE:
                    continue
                elif sub_result == EXCLUDE_FROM_TEMPLATES_RESULTS.ALLOW:
                    result = EXCLUDE_FROM_TEMPLATES_RESULTS.ALLOW
                elif sub_result == EXCLUDE_FROM_TEMPLATES_RESULTS.EXCLUDE:
                    return EXCLUDE_FROM_TEMPLATES_RESULTS.EXCLUDE
                else:
                    raise Exception('No such sub-result')
            
            return result
    
    
    def _is_check_excluded(self, check):
        by_id_excluded_checks = self.get_ids_in_link(SERVICE_EXCLUDES_BY_ID)
        by_name_excluded_checks = [exclude for exclude in self.get(SERVICE_EXCLUDES_BY_NAME, "").split(",") if exclude.strip()]
        
        check_id = check['_id']
        check_name = check.get_name()
        check_name_stripped_from_dfe = check_name.replace('$KEY$', '')
        
        if check_id in by_id_excluded_checks and not check.get('duplicate_foreach'):
            return True
        
        for exclude_description in by_name_excluded_checks:
            if ServiceExcludesMixin._is_matching_name_regex_star_only(check_name_stripped_from_dfe, exclude_description):
                return True
        return False
    
    
    @staticmethod
    def _is_matching_name_regex_star_only(name_service, exclude_by_name_label):
        exclude_by_name_label_escape = re.escape(exclude_by_name_label)
        exclude_by_name_label_escape = exclude_by_name_label_escape.replace('\\*', ".*")
        return re.match(r'^%s$' % exclude_by_name_label_escape, name_service)
