#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2018:
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.daterange import Daterange
from shinken.objects.service import PREFIX_LINK_UUID, PREFIX_LINK_DFE_KEY
from shinken.util import WeakMethod
from shinkensolutions.time_period_parser import TimePeriodParser
from shinkensolutions.toolbox.box_tools_string import ToolsBoxString
from validator import Validator, ILLEGAL_CHARS, ILLEGAL_CHARS_SERVICE
from ...dao.def_items import ITEM_TYPE, DEF_ITEMS, PROP_DEFAULT_VALUE, SERVICE_OVERRIDE, SERVICE_EXCLUDES_BY_ID, METADATA, NOT_TO_LOOK
from ...dao.helpers import get_name_from_type, split_and_strip_list
from ...front_end.service.collapse_content import CollapsedContent

PROPERTIES_TO_CHECK_VALUE_SYNTAX = (u'view_contacts', u'view_contact_groups', u'notification_contacts', u'notification_contact_groups', u'edition_contacts',
                                    u'edition_contact_groups', u'resultmodulations', u'use', u'parents', u'notification_period', u'notification_options',
                                    u'escalations', u'maintenance_period', u'check_period', u'macromodulations',
                                    u'hostgroups', u'business_impact_modulations',
                                    u'apply_on_type', u'contacts', u'contact_groups', u'template_members', u'members', u'notificationways',
                                    u'contactgroups', u'host_notification_commands', u'service_notification_commands', u'service_notification_period',
                                    u'host_notification_period', u'host_notification_options', u'service_notification_options', u'escalation_period',
                                    u'modulation_period', u'check_command', u'event_handler')


class ImportValidator(Validator):
    
    def __init__(self, *args, **kwargs):
        super(ImportValidator, self).__init__(*args, **kwargs)
        # add the _remove_incorrect_properties into the commons rule
        # each object have to be validate with this specific validation during the import
        self._COMMON_RULES.insert(0, WeakMethod(self._remove_incorrect_properties))
        self._COMMON_RULES.insert(1, WeakMethod(self._remove_whitespace_in_string_link))
        self._COMMON_RULES.append(WeakMethod(self._rule_sync_keys_must_be_common_in_source))
        self._COMMON_RULES.append(WeakMethod(self._rule_valide_SE_UUID))
        self._COMMON_RULES.append(WeakMethod(self._validate_properties_syntax))
    
    
    def _validate_contacts(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_acl_in_tab_history(item))
        return validation
    
    
    def _validate_contacttpls(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_acl_in_tab_history(item))
        return validation
    
    
    def _validate_hosts(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_no_localhost_item_name(item, item_type))
        validation.extend(self._rule_valide_check_interval_gte_1(item))
        validation.extend(self._rule_valide_service_overrides(item, item_type))
        validation.extend(self._rule_valide_check_running_timeout_property(item))
        validation.extend(self._rule_valide_warning_threshold_cpu_usage_property(item))
        validation.extend(self._rule_valide_notification_interval_property(item))
        validation.extend(self._validate_service_excludes(item))
        validation.extend(self._rule_valide_service_excludes_by_id_format(item, ITEM_TYPE.HOSTS))
        validation.extend(self._rule_valide_external_url(item))
        validation.extend(self._rule_valid_realm(item))
        return validation
    
    
    def _validate_hosttpls(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_no_localhost_item_name(item, item_type))
        validation.extend(self._rule_valide_check_interval_gte_1(item))
        validation.extend(self._rule_valide_service_overrides(item, item_type))
        validation.extend(self._rule_valide_check_running_timeout_property(item))
        validation.extend(self._rule_valide_warning_threshold_cpu_usage_property(item))
        validation.extend(self._rule_valide_notification_interval_property(item))
        validation.extend(self._validate_service_excludes(item))
        validation.extend(self._rule_valide_service_excludes_by_id_format(item, ITEM_TYPE.HOSTTPLS))
        validation.extend(self._rule_valide_external_url(item))
        validation.extend(self._rule_valid_realm(item))
        return validation
    
    
    def _validate_hostgroups(self, item, item_type):
        validation = []
        return validation
    
    
    def _validate_clusters(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_no_localhost_item_name(item, item_type))
        validation.extend(self._rule_valide_check_interval_gte_1(item))
        validation.extend(self._rule_valide_service_overrides(item, item_type))
        validation.extend(self._validate_service_excludes(item))
        validation.extend(self._rule_valide_service_excludes_by_id_format(item, ITEM_TYPE.CLUSTERS))
        validation.extend(self._rule_valide_external_url(item))
        validation.extend(self._validate_bp_rule(item))
        validation.extend(self._rule_valid_realm(item))
        return validation
    
    
    def _validate_clustertpls(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_no_localhost_item_name(item, item_type))
        validation.extend(self._rule_valide_check_interval_gte_1(item))
        validation.extend(self._rule_valide_service_overrides(item, item_type))
        validation.extend(self._validate_service_excludes(item))
        validation.extend(self._rule_valide_service_excludes_by_id_format(item, ITEM_TYPE.CLUSTERTPLS))
        validation.extend(self._rule_valide_external_url(item))
        validation.extend(self._validate_bp_rule(item))
        validation.extend(self._rule_valid_realm(item))
        return validation
    
    
    def _validate_services(self, item, item_type):
        if item_type.startswith('override_'):
            item_type = item_type.replace('override_', '')
        
        validation = []
        validation.extend(self._rule_valide_data_name(item))
        validation.extend(self._rule_valide_dollar_in_data(item))
        validation.extend(self._rule_valide_check_interval_gte_1(item))
        validation.extend(self._rule_valide_host_name(item, item_type))
        validation.extend(self._rule_valide_hostgroup_names(item, item_type))
        validation.extend(self._rule_valide_check_running_timeout_property(item))
        validation.extend(self._rule_valide_data_duplicate_foreach(item))
        validation.extend(self._rule_valide_warning_threshold_cpu_usage_property(item))
        validation.extend(self._rule_valide_notification_interval_property(item))
        validation.extend(self._rule_valide_external_url(item))
        if item_type in (ITEM_TYPE.SERVICESHOSTTPLS, ITEM_TYPE.SERVICESCLUSTERTPLS):
            validation.extend(self._rule_valide_complex_expression(item, item_type))
        return validation
    
    
    def _validate_servicetpls(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_no_host_name_in_servicetpls(item, item_type))
        validation.extend(self._rule_valide_host_name(item, item_type))
        validation.extend(self._rule_valide_hostgroup_names(item, item_type))
        validation.extend(self._rule_valide_check_interval_gte_1(item))
        validation.extend(self._rule_valide_check_running_timeout_property(item))
        validation.extend(self._rule_valide_warning_threshold_cpu_usage_property(item))
        validation.extend(self._rule_valide_notification_interval_property(item))
        validation.extend(self._rule_valide_external_url(item))
        return validation
    
    
    def _validate_macromodulations(self, item, item_type):
        validation = []
        return validation
    
    
    def _validate_timeperiods(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_periods(item))
        validation.extend(self._rule_valide_timeperiod_options(item))
        return validation
    
    
    def _validate_properties_syntax(self, item, item_type):
        validation = []
        for property_name in PROPERTIES_TO_CHECK_VALUE_SYNTAX:
            if property_name not in item:
                continue
            _illegal_chars = ILLEGAL_CHARS
            if item_type in ITEM_TYPE.ALL_SERVICES and property_name in (u'use', u'name', u'service_description'):
                _illegal_chars = ILLEGAL_CHARS_SERVICE
            _to_test = item[property_name]
            if property_name in DEF_ITEMS[item_type].get(u'fields_with_plus', []) and _to_test.startswith(u'+'):
                _to_test = _to_test[1:]
            if (ITEM_TYPE.COMMANDS in DEF_ITEMS[ITEM_TYPE.HOSTS]['props_links'].get(property_name, []) and '!' in item[property_name]):
                _to_test = item[property_name].split('!', 1)
                _to_test = _to_test[0]
            self._rule_is_valid_name_deprecated_after_02_08_01(_to_test, _illegal_chars, property_name, validation)
        return validation
    
    
    def _validate_resultmodulations(self, item, item_type):
        return []
    
    
    def _validate_businessimpactmodulations(self, item, item_type):
        validation = []
        return validation
    
    
    def _validate_commands(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_macro_in_command_line(item))
        validation.extend(self._rule_valide_timeout_property(item))
        validation.extend(self._rule_valide_warning_threshold_cpu_usage_property(item))
        return validation
    
    
    def _validate_notificationways(self, item, item_type):
        validation = []
        return validation
    
    
    def _validate_escalations(self, item, item_type):
        validation = []
        validation.extend(self._rule_valide_notification_interval_property(item))
        return validation
    
    
    def _validate_contactgroups(self, item, item_type):
        return []
    
    
    def _rule_valide_timeperiod_options(self, item):
        ALLOWED_WEEK_OFFSETS = ("-5", "-4", "-3", "-2", "-1", "1", "2", "3", "4", "5")
        validation = []
        for weekday in Daterange.weekdays:
            week_day_has_error = False
            value = item.get(weekday, "")
            if not value:
                continue
            try:
                tokens = TimePeriodParser.tokenize_day(value)
                tp_day = TimePeriodParser.parse_day(tokens)
            except IndexError:
                validation.append(self._create_message(self.app._('validator.timeperiod_wrong_syntax') % (weekday, value), level=Validator.LEVEL_VALIDATOR_WARNING))
                continue
            
            for timeperiod_for_day in tp_day['defs']:
                if timeperiod_for_day.week != "None" and timeperiod_for_day.week == 0 or timeperiod_for_day.week not in ALLOWED_WEEK_OFFSETS:
                    week_day_has_error = True
                
                if timeperiod_for_day.month != "None" and timeperiod_for_day.month not in Daterange.months:
                    week_day_has_error = True
                    validation.append(self._create_message(self.app._('validator.timeperiod_month_out_of_range') % (weekday, timeperiod_for_day.flat_value), level=Validator.LEVEL_VALIDATOR_WARNING))
            
            if week_day_has_error:
                item.pop(weekday)
        
        return validation
    
    
    def _rule_valide_no_localhost_item_name(self, item, item_type):
        validation = []
        # Must have a name (or service description if service)
        _name = get_name_from_type(item_type, item)
        _name = _name.strip()
        if _name == "localhost":
            validation.append(self._create_message(self.app._('validator.forbiden_name_localhost'), level=Validator.LEVEL_VALIDATOR_CRITICAL))
        return validation
    
    
    def _rule_valide_no_host_name_in_servicetpls(self, item, item_type):
        validation = []
        _name = get_name_from_type(item_type, item)
        if item.get('host_name', ''):
            validation.append(self._create_message(self.app._('validator.servicetpls_with_host_name') % (_name), level=Validator.LEVEL_VALIDATOR_WARNING))
        return validation
    
    
    def _remove_whitespace_in_string_link(self, item, item_type):
        # the goal is to remove the " " in links ex
        # "contact_groups test1, test2, test3" will be "contact_groups test1,test2,test3"
        
        for prop_link in DEF_ITEMS[item_type]['props_links']:
            link_value = item.get(prop_link, None)
            if link_value and " " in link_value and prop_link != SERVICE_OVERRIDE:
                item[prop_link] = ",".join(split_and_strip_list(link_value))
        
        return []
    
    
    def _service_override_validate_properties(self, check_type, property_name):
        validation = []
        if property_name in self.OVERRIDE_FORBIDDEN_PROPERTIES:
            _msg = self._create_message(self.app._('validator.service_overrides_forbidden') % property_name, level=Validator.LEVEL_VALIDATOR_WARNING)
            validation.append(_msg)
        if property_name in self.OVERRIDE_REPLACED_KEYWORDS:
            _msg = self._create_message(self.app._('validator.service_overrides_specific_%s' % property_name), level=Validator.LEVEL_VALIDATOR_WARNING)
            validation.append(_msg)
        return validation
    
    
    def _remove_incorrect_properties(self, item, item_type):
        # Remove incorrect properties, incorrect properties are :
        #  - properties starting with $
        #  - properties in blacklist
        #  - properties with +null
        
        validation = []
        keys_to_del = []
        data_to_uppercase = []
        
        for prop, value in item.iteritems():
            if not prop.startswith('_') and not item[prop]:
                keys_to_del.append(prop)
                validation.append(self._create_message(self.app.t(u'validator.skipped_properties_without_value') % prop, level=Validator.LEVEL_VALIDATOR_WARNING))
                continue
            elif prop.startswith('_') and prop not in NOT_TO_LOOK and prop != prop.upper():
                data_to_uppercase.append(prop)
                keys_to_del.append(prop)
                validation.append(self._create_message(self.app.t(u'validator.macro_name_not_in_upper') % (ToolsBoxString.escape_XSS(prop), ToolsBoxString.escape_XSS(prop).upper()), level=Validator.LEVEL_VALIDATOR_WARNING))
            
            if isinstance(value, basestring):
                if u'+null' in value:
                    keys_to_del.append(prop)
                    validation.append(self._create_message(self.app.t(u'validator.skipped_properties_with_plus_and_null') % prop, level=Validator.LEVEL_VALIDATOR_WARNING))
                    continue
            
            if u'[FORCE]' in prop:
                prop = prop.replace(u'[FORCE]', u'').strip()
            
            if prop in self.KNOWN_PROPERTIES[item_type][u'blacklist']:
                keys_to_del.append(prop)
                validation.append(self._create_message(self.app.t(u'validator.skipped_prohibited_properties') % prop, level=Validator.LEVEL_VALIDATOR_WARNING))
            
            elif (item_type != ITEM_TYPE.TIMEPERIODS) and (prop not in self.KNOWN_PROPERTIES[item_type][u'all']) and (not prop.startswith(u'_')):
                keys_to_del.append(prop)
                validation.append(self._create_message(self.app.t(u'validator.skipped_unknown_properties') % ToolsBoxString.escape_XSS(prop), level=Validator.LEVEL_VALIDATOR_WARNING))
        
        for k in data_to_uppercase:
            item[k.upper()] = item[k]
        for k in keys_to_del:
            del item[k]
        return validation
    
    
    def _rule_valide_acl_in_tab_history(self, item):
        
        item_value = item.get('acl_in_tab_history', None)
        if item_value == 'null':
            item['acl_in_tab_history'] = PROP_DEFAULT_VALUE
        
        return super(ImportValidator, self)._rule_valide_acl_in_tab_history(item)
    
    
    def _rule_valide_service_excludes_by_id_format(self, item, item_type):
        validation = []
        value = item.get(SERVICE_EXCLUDES_BY_ID)
        
        if not value:
            return validation
        
        if item_type not in (ITEM_TYPE.HOSTS, ITEM_TYPE.HOSTTPLS, ITEM_TYPE.CLUSTERS, ITEM_TYPE.CLUSTERTPLS):
            validation.append(self._create_message(self.app._('validator.service_excludes_unmanaged_type') % self.app._("type.%s" % item_type), level=Validator.LEVEL_VALIDATOR_CRITICAL))
            return validation
        
        formatted_excludes = []
        for exclude in split_and_strip_list(value):
            if PREFIX_LINK_DFE_KEY in exclude:
                validation.append(self._create_message(self.app._('validator.service_excludes_dfe_check') % exclude, level=Validator.LEVEL_VALIDATOR_WARNING))
            elif PREFIX_LINK_UUID in exclude:
                (check_name, check_id) = exclude.split(PREFIX_LINK_UUID, 1)
                
                formatted_exclude = "%s%s" % (PREFIX_LINK_UUID, check_id)
                formatted_excludes.append(formatted_exclude)
            else:
                validation.append(self._create_message(self.app._('validator.service_excludes_only_by_id') % exclude, level=Validator.LEVEL_VALIDATOR_WARNING))
        
        item[SERVICE_EXCLUDES_BY_ID] = ",".join(formatted_excludes)
        return validation
    
    
    def _rule_sync_keys_must_be_common_in_source(self, item, item_type):
        validation = []
        my_sync_keys = item.get('_SYNC_KEYS', None)
        if not my_sync_keys:
            my_sync_keys = []
        sync_keys_index = METADATA.get_metadata(item, METADATA.SYNC_KEYS_INDEX)
        bad_items = []
        common_items = []
        
        for sync_key in my_sync_keys:
            lst = sync_keys_index[sync_key]
            for i in lst:
                if i not in common_items:
                    common_items.append(i)
            bad_items.extend([i for i in lst if i['_SYNC_KEYS'] != my_sync_keys])
        
        if bad_items:
            sync_keys_list = []
            for c_item in common_items:
                _sync_keys = []
                for sync_key in c_item['_SYNC_KEYS']:
                    if sync_key not in my_sync_keys:
                        _sync_keys.append('''<span class='shinken-error'>%s</span>''' % sync_key)
                    else:
                        _sync_keys.append('''<span class="shinken-sync-key">%s</span>''' % sync_key)
                sync_keys_list.append([c_item['imported_from'], ', '.join(_sync_keys)])
            
            html_msg = CollapsedContent(sync_keys_list, header=[self.app._('validator.element_define_from'), self.app._('validator.sync_keys')], label_more=self.app._('common.see_all'), label_less=self.app._('common.see_only_first'))
            validation.append(self._create_message(self.app._('validator.error_differents_SK_in_same_source') % {
                'item_def'          : item['imported_from'],
                'possible_sync_keys': '''</span>, <span class='shinken-data-user'>'''.join(METADATA.get_metadata(item, METADATA.POSSIBLE_SYNC_KEYS, [DEF_ITEMS[item_type]['key_name']])),
                'sync_keys_list'    : html_msg.get_html()
            }, level=Validator.LEVEL_VALIDATOR_CRITICAL))
        
        return validation
    
    
    def _rule_is_valid_name_deprecated_after_02_08_01(self, _to_test, invalids_chars, property, validation, level=None):
        if not level:
            level = Validator.LEVEL_VALIDATOR_ERROR
        
        array_to_test = _to_test.split(',')
        for value_to_test in array_to_test:
            if invalids_chars[u'regex'].search(value_to_test):
                validation.append(
                    self._create_message(self.app._('validator.invalid_setting') % (ToolsBoxString.escape_XSS(value_to_test), property, ToolsBoxString.format_for_message(invalids_chars[u'value'])), level=level)
                )
                break
