#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import copy
import threading
import time
import uuid
from threading import RLock

import item_lists
import massive_change
import shinkensolutions.shinkenjson as json
import try_check
from item_lists import elements_generic
from massive_change import massive_change_call
from shinken.log import logger, LoggerFactory
from shinken.misc.fast_copy import fast_deepcopy
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.objects.host import VIEW_CONTACTS_DEFAULT_VALUE
from shinken.objects.resultmodulation import ModulationRule
from shinken.property import AclShareProp
from shinken.util import make_unicode, blank_context
from shinkensolutions.toolbox.box_tools_string import ToolsBoxString
from try_check import run_check
from .unauthorized_save_exception_handler import return_error_if_unauthorized_save_exception
from ...business.item_controller import exceptions
from ...business.item_controller.edit_item_context import EditItemContext, ACTIONS
from ...business.item_controller.exceptions import UnauthorizedSaveException, SaveException
from ...business.item_controller.shinken_return_code import ShinkenReturnCode
from ...business.source.sourceinfoproperty import SourceInfoProperty
from ...business.sync_ui_common import syncuicommon
from ...dao.checks_inheritance import lookup_items_templates, inherited_checks_from_hosts_templates, TEMPLATE_STATUS
from ...dao.crypto import match_protected_property, PROTECTED_DEFAULT_TAG, PROTECTED_TAG_FROM_CHANGE, get_frontend_cipher_from_request
from ...dao.datamanagerV2 import DataManagerV2
from ...dao.def_items import ITEM_STATE, ITEM_TYPE, DEF_ITEMS, METADATA, LINKIFY_MANAGE_STATES, SERVICE_OVERRIDE, NOT_TO_LOOK, UNFLATTENNED_PROPERTIES_IN_FRONT
from ...dao.helpers import get_name_from_type, get_default_value, get_inherited_without_default, update_frontend_links_into_links
from ...dao.item_saving_formatter import update_frontend_service_override_links_into_links
from ...dao.items import BaseItem, get_item_instance
from ...dao.transactions.transactions import DBTransaction
from ...front_end import helper
from ...front_end.helper_frontend_response import FrontEndSaveResponse
from ...front_end.object.messages import Messages, Message, MESSAGE, ValidatorMessages

if TYPE_CHECKING:
    from ...synchronizerdaemon import Synchronizer
    from ...dao.def_items import ItemType
    from shinken.misc.type_hint import Optional, List, Dict
    from ...dao.items.misc_items import InstanceItem

MULTI_MASS_ACTION_UPDATE = {}

# Identify the dummy property for submit /reject comments
object_edition_lock = RLock()
logger = LoggerFactory.get_logger(u'ELEMENT EDITION')
logger_save = logger.get_sub_part(u'SAVE')

# These vars are set by SynchronizerDaemon at runtime through the set_app() function
app = None  # type: Optional[Synchronizer]
datamanagerV2 = None  # type: Optional[DataManagerV2]


def set_app(_app):
    # type: (Synchronizer) -> None
    global app
    global datamanagerV2
    
    app = _app
    datamanagerV2 = _app.datamanagerV2
    
    item_lists.app = _app
    item_lists.load_column_configuration()
    
    try_check.app = _app
    try_check.datamanagerV2 = datamanagerV2
    
    massive_change.app = _app
    massive_change.datamanagerV2 = datamanagerV2
    
    helper.app = _app
    exceptions.app = _app


def elements_checksbytemplates():
    app.response.content_type = 'application/json'
    user = app.get_user_auth()
    
    use_links = [item_link for item_link in json.loads(app.request.POST.get('links', '{}').decode('utf-8', 'ignore'))['links'] if item_link]
    my_name = app.request.POST.get('my_name', '').decode('utf-8', 'ignore')
    item_type = app.request.POST.get('item_type', '')
    my_uuid = app.request.POST.get('uuid', '')
    
    use_names = update_frontend_links_into_links('use', use_links, item_type, datamanagerV2)
    
    inherited_checks = inherited_checks_from_hosts_templates(use_names, my_name, my_uuid, item_type, datamanagerV2, with_superfluous_checks=True)
    
    for host_name, checks in inherited_checks.iteritems():
        inherited_checks[host_name] = [app.frontend_cipher.cipher(e, item_type=item_type, user=user) for e in checks]
    
    return json.dumps(inherited_checks, ensure_ascii=False).encode('utf8')


def _get_specific_item(item_id, item_type, current_page=None):
    # type: (unicode, ItemType, unicode) -> Dict
    user = app.get_user_auth()
    is_admin = user.is_admin()
    item_is_new = (app.request.GET.get(u'new', u'0') == u'1')
    _has_staging_instance = True
    
    if current_page == u'working_area':
        item_state = ITEM_STATE.NEW if item_is_new else ITEM_STATE.WORKING_AREA
        backup_state = ITEM_STATE.STAGGING
        _working_area = True
        flatten_links_states = (ITEM_STATE.WORKING_AREA, ITEM_STATE.STAGGING)
    else:
        current_page = u'stagging'
        item_state = ITEM_STATE.NEW if item_is_new else ITEM_STATE.STAGGING
        backup_state = ITEM_STATE.WORKING_AREA
        _working_area = False
        flatten_links_states = (ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA)
    
    item = datamanagerV2.find_item_by_id(item_id, item_type, item_state)
    # If a user ask from staging an item create in working area it isn't in ITEM_STATE.STAGGING but in ITEM_STATE.WORKING_AREA base
    
    if not item:
        if not _working_area:
            _has_staging_instance = False
        item = datamanagerV2.find_item_by_id(item_id, item_type, backup_state)
    
    if not item:
        logger.info(u'[%s-%s] not found' % (item_type, item_id))
        return app.abort(404, app.t(u'element.no_item_with_this_id') % (app.t(u'type.' + item_type[:-1]), item_id))
    
    result = syncuicommon.check_acl(item, user, current_page)
    can_edit = result[u'can_edit']
    can_view = result[u'can_view']
    block_acl_cause = result[u'cause']
    
    if not can_view:
        return app.abort(404, app.t(u'element.no_item_with_this_id') % (app.t(u'type.' + item_type[:-1]), item_id))
    
    changes = {}
    if is_admin and not (current_page == u'stagging' and item_type == ITEM_TYPE.HOSTS):
        changes = fast_deepcopy(item.get_changes(), additional_dispatcher={SourceInfoProperty: SourceInfoProperty.fast_copy_source_info_property})
        for _property_name in changes:
            changes[_property_name][0] = ToolsBoxString.escape_XSS(changes[_property_name][0])
            changes[_property_name][1] = ToolsBoxString.escape_XSS(changes[_property_name][1])
        app.frontend_cipher.cipher({u'changes': changes}, item_type=item_type, item_state=ITEM_STATE.CHANGES, user=user)
    
    unflattened_item = item.get_raw_item(flatten_links=False)
    service_overrides_links = []
    for override in unflattened_item.get(SERVICE_OVERRIDE, {}).get(u'links', []):
        service_overrides = copy.copy(override)
        service_overrides[u'key'] = override[u'key']
        service_overrides[u'value'] = datamanagerV2.flatten_overridden_property(DEF_ITEMS[item_type][u'check_type'], override)
        service_overrides[u'value_with_link'] = override[u'value']
        service_overrides_links.append(service_overrides)
    
    services_in_service_overrides = []
    if item_type in (ITEM_TYPE.HOSTS, ITEM_TYPE.HOSTTPLS, ITEM_TYPE.CLUSTERS, ITEM_TYPE.CLUSTERTPLS):
        check_types = set([DEF_ITEMS[item_type][u'check_type'], DEF_ITEMS[DEF_ITEMS[item_type][u'template']][u'check_type']])
        for check_type in check_types:
            services_in_service_overrides.extend(item.get_link_items(SERVICE_OVERRIDE, only_exist=True, item_type=check_type))
    
    item = item.get_raw_item(flatten_links=flatten_links_states, keep_metadata=[METADATA.ITEM_TYPE])
    
    item[u'editable'] = u'1' if can_edit else u'0'
    
    for item_property, item_value in item.iteritems():
        if isinstance(item_value, bool):
            item[item_property] = '1' if item_value else u'0'
        # Trail leading _ in duplicate foreach field
        if item_property == u'duplicate_foreach':
            item[item_property] = item_value[1:]
        elif item_property in UNFLATTENNED_PROPERTIES_IN_FRONT:
            item[item_property] = copy.copy(unflattened_item[item_property])
            if item_property == SERVICE_OVERRIDE:
                item[item_property][u'links'] = service_overrides_links
        elif isinstance(item_value, list) and item_property != u'_SYNC_KEYS' and item_property != SERVICE_OVERRIDE:
            item[item_property] = u','.join(item_value)
    
    METADATA.update_metadata(item, METADATA.NEW, item_is_new)
    METADATA.update_metadata(item, METADATA.HAS_DATA, DEF_ITEMS[item_type][u'has_data'])
    METADATA.update_metadata(item, 'has_staging_instance', _has_staging_instance)
    
    if item_type in (ITEM_TYPE.HOSTS, ITEM_TYPE.HOSTTPLS, ITEM_TYPE.CLUSTERS, ITEM_TYPE.CLUSTERTPLS):
        shinken_elements_lists = _build_shinken_elements_lists([
            ITEM_TYPE.TIMEPERIODS,
            ITEM_TYPE.CONTACTS,
            ITEM_TYPE.CONTACTGROUPS,
            ITEM_TYPE.BUSINESSIMPACTMODULATIONS,
            ITEM_TYPE.MACROMODULATIONS,
            ITEM_TYPE.RESULTMODULATIONS,
            ITEM_TYPE.COMMANDS,
            ITEM_TYPE.ESCALATIONS,
        ], services_in_service_overrides)
    else:
        shinken_elements_lists = {}
    
    return {
        u'app'                   : app,
        u'user'                  : user,
        u'item'                  : app.frontend_cipher.cipher(item, item_type=item_type, user=user),
        u'shinken_elements_lists': shinken_elements_lists,
        u'helper'                : app.helper,
        u'is_tpl'                : ITEM_TYPE.is_template(item_type),
        u'protected_fields'      : DEF_ITEMS[item_type].get(u'protected_fields', []),
        u'item_type'             : item_type,
        u'name'                  : get_name_from_type(item_type, item).replace(u' ', u'\u00a0'),  # Here the name is only use for to show and need to be unbreakable
        u'changes'               : changes,
        u'item_is_new'           : item_is_new,
        u'working_area'          : _working_area,
        u'can_view'              : can_view,
        u'can_edit'              : can_edit,
        u'block_acl_cause'       : block_acl_cause
    }


def _build_shinken_elements_lists(types_to_list, services_in_service_overrides=None):
    # type: (List[ItemType], List) -> Dict[ItemType, List]
    services_in_service_overrides = services_in_service_overrides if services_in_service_overrides else []
    shinken_elements_lists = {}
    
    for linked_item_type in types_to_list:
        name_field = DEF_ITEMS[linked_item_type][u'key_name']
        shinken_elements_lists[linked_item_type] = []
        for linked_item in datamanagerV2.find_merge_state_items(linked_item_type, (ITEM_STATE.NEW, ITEM_STATE.STAGGING, ITEM_STATE.WORKING_AREA)):
            linked_item_for_list = _format_item_for_shinken_elements_lists(linked_item, name_field, linked_item_type)
            shinken_elements_lists[linked_item_type].append(linked_item_for_list)
    
    for service in services_in_service_overrides:
        linked_item_type = METADATA.get_metadata(service, METADATA.ITEM_TYPE)
        name_field = DEF_ITEMS[linked_item_type][u'key_name']
        linked_item_for_list = _format_item_for_shinken_elements_lists(service, name_field, linked_item_type)
        
        if not shinken_elements_lists.get(linked_item_type, None):
            shinken_elements_lists[linked_item_type] = []
        shinken_elements_lists[linked_item_type].append(linked_item_for_list)
    
    return shinken_elements_lists


def _format_item_for_shinken_elements_lists(linked_item, name_field, linked_item_type):
    # type: (InstanceItem, unicode, ItemType) -> Dict[unicode, unicode]
    linked_item_for_list = {
        name_field: linked_item[name_field],
        u'_id'    : linked_item[u'_id'],
        u'enabled': linked_item.get(u'enabled', u'1')
    }
    METADATA.update_metadata(linked_item_for_list, METADATA.ITEM_TYPE, linked_item_type)
    
    linked_item_state = METADATA.get_metadata(linked_item, METADATA.STATE)
    METADATA.update_metadata(linked_item_for_list, METADATA.STATE, linked_item_state)
    
    if linked_item_type in (ITEM_TYPE.RESULTMODULATIONS, ITEM_TYPE.BUSINESSIMPACTMODULATIONS, ITEM_TYPE.MACROMODULATIONS):
        linked_item_for_list[u'modulation_period'] = linked_item.flatten_prop(u'modulation_period', LINKIFY_MANAGE_STATES)
    
    if linked_item_type == ITEM_TYPE.CONTACTS:
        if linked_item.is_admin():
            linked_item_for_list[u'is_admin'] = u'1'
        if linked_item.is_expert():
            linked_item_for_list[u'is_expert'] = u'1'
    return linked_item_for_list


def list_elements(item_type):
    return elements_generic(item_type)


def new_host():
    return new_object(ITEM_TYPE.HOSTS)


def new_host_tpl():
    return new_object(ITEM_TYPE.HOSTTPLS)


def new_hostgroup():
    return new_object(ITEM_TYPE.HOSTGROUPS)


def new_cluster():
    return new_object(ITEM_TYPE.CLUSTERS)


def new_cluster_tpl():
    return new_object(ITEM_TYPE.CLUSTERTPLS)


def new_services_host():
    return new_object(ITEM_TYPE.SERVICESHOSTS)


def new_services_host_tpl():
    return new_object(ITEM_TYPE.SERVICESHOSTTPLS)


def new_services_cluster():
    return new_object(ITEM_TYPE.SERVICESCLUSTERS)


def new_services_cluster_tpl():
    return new_object(ITEM_TYPE.SERVICESCLUSTERTPLS)


def new_service_tpl():
    return new_object(ITEM_TYPE.SERVICETPLS)


def new_contact():
    return new_object(ITEM_TYPE.CONTACTS)


def new_contact_tpl():
    return new_object(ITEM_TYPE.CONTACTTPLS)


def new_contactgroup():
    return new_object(ITEM_TYPE.CONTACTGROUPS)


def new_escalation():
    return new_object(ITEM_TYPE.ESCALATIONS)


def new_notificationway():
    return new_object(ITEM_TYPE.NOTIFICATIONWAYS)


def new_businessimpactmodulation():
    return new_object(ITEM_TYPE.BUSINESSIMPACTMODULATIONS)


def new_macromodulation():
    return new_object(ITEM_TYPE.MACROMODULATIONS)


def new_resultmodulation():
    item = new_object(ITEM_TYPE.RESULTMODULATIONS)
    item['modulation_rules'] = [ModulationRule()]
    return item


def new_command():
    return new_object(ITEM_TYPE.COMMANDS)


def new_timeperiod():
    return new_object(ITEM_TYPE.TIMEPERIODS)


def new_object(item_type):
    user = app.get_user_auth()
    is_tpl = ITEM_TYPE.is_template(item_type)
    item = {'_id': uuid.uuid1().hex}
    METADATA.update_metadata(item, METADATA.ITEM_TYPE, item_type)
    METADATA.update_metadata(item, METADATA.IN_CREATION, True)
    METADATA.update_metadata(item, METADATA.HAS_DATA, DEF_ITEMS[item_type]['has_data'])
    
    if ITEM_TYPE.has_user_right(item_type) and not ITEM_TYPE.is_template(item_type):
        value_to_set = '+%s' % user['contact_name']
        
        # passer par un composant (02.08.02 min)
        _default_view_contacts = get_default_value(ITEM_TYPE.HOSTS, 'view_contacts')[0] or VIEW_CONTACTS_DEFAULT_VALUE.NOBODY
        if _default_view_contacts != VIEW_CONTACTS_DEFAULT_VALUE.EVERYONE:
            item['view_contacts'] = value_to_set
        item['notification_contacts'] = value_to_set
        item['edition_contacts'] = value_to_set
    
    if item_type in (ITEM_TYPE.HOSTS, ITEM_TYPE.HOSTTPLS, ITEM_TYPE.CLUSTERS, ITEM_TYPE.CLUSTERTPLS):
        shinken_elements_lists = _build_shinken_elements_lists([
            ITEM_TYPE.TIMEPERIODS,
            ITEM_TYPE.CONTACTS,
            ITEM_TYPE.CONTACTGROUPS,
            ITEM_TYPE.BUSINESSIMPACTMODULATIONS,
            ITEM_TYPE.MACROMODULATIONS,
            ITEM_TYPE.RESULTMODULATIONS,
            ITEM_TYPE.COMMANDS,
            ITEM_TYPE.ESCALATIONS
        ])
    else:
        shinken_elements_lists = {}
    
    return {
        u'app'                   : app,
        u'user'                  : user,
        u'item'                  : app.frontend_cipher.cipher(item, item_type, user=user),
        u'shinken_elements_lists': shinken_elements_lists,
        u'protected_fields'      : DEF_ITEMS[item_type].get(u'protected_fields', []),
        u'name'                  : u'',
        u'helper'                : app.helper,
        u'is_tpl'                : is_tpl,
        u'is_in_creation'        : True,
        u'item_type'             : item_type,
        u'changes'               : {},
        u'item_is_new'           : False,
        u'can_edit'              : True,
        u'can_view'              : True,
        u'block_acl_cause'       : u'',
    }


def edit_host(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.HOSTS)


def edit_hosttpl(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.HOSTTPLS)


def edit_hostgroup(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.HOSTGROUPS)


def edit_cluster(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.CLUSTERS)


def edit_clustertpl(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.CLUSTERTPLS)


def edit_services_clusters(item_id):
    return _get_specific_item(item_id, item_type=ITEM_TYPE.SERVICESCLUSTERS)


def edit_services_cluster_tpls(item_id):
    return _get_specific_item(item_id, item_type=ITEM_TYPE.SERVICESCLUSTERTPLS)


def edit_services_hosts(item_id):
    return _get_specific_item(item_id, item_type=ITEM_TYPE.SERVICESHOSTS)


def edit_services_hosts_override(item_id):
    return _get_specific_item(item_id, item_type=ITEM_TYPE.SERVICESHOSTS)


def edit_services_host_tpls(item_id):
    return _get_specific_item(item_id, item_type=ITEM_TYPE.SERVICESHOSTTPLS)


def edit_service_tpls(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.SERVICETPLS)


def edit_contact(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.CONTACTS)


def edit_contacttpl(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.CONTACTTPLS)


def edit_contactgroup(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.CONTACTGROUPS)


def edit_command(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.COMMANDS)


def edit_timeperiod(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.TIMEPERIODS)


def edit_businessimpactmodulation(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.BUSINESSIMPACTMODULATIONS)


def edit_macromodulation(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.MACROMODULATIONS)


def edit_resultmodulation(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.RESULTMODULATIONS)


def edit_notificationway(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.NOTIFICATIONWAYS)


def edit_escalation(item_id):
    return _get_specific_item(item_id, ITEM_TYPE.ESCALATIONS)


def get_service_override(host_type, host_id, service_type, service_id, apply_on_type):
    user = app.get_user_auth()
    host_cluster_templates = [
        datamanagerV2.find_item_by_name(tpl_name, ITEM_TYPE.get_template_type(apply_on_type), ITEM_STATE.STAGGING) or
        datamanagerV2.find_item_by_name(tpl_name, ITEM_TYPE.get_template_type(apply_on_type), ITEM_STATE.NEW) for tpl_name in
        app.request.GET.get(u'tpl', u'').split(u',') if tpl_name]
    check_name = make_unicode(app.request.GET.get(u'name', u''))
    dfe_name = make_unicode(app.request.GET.get(u'dfe_name', u''))
    force_clean = make_unicode(app.request.GET.get(u'force_clean', u''))
    force_disable = make_unicode(app.request.GET.get(u'force_disable', u''))
    dom_uuid = make_unicode(app.request.GET.get(u'dom-uuid', u''))
    is_host_editable = app.request.GET.get(u'editable', u'1')
    check_context = EditItemContext(app, service_id, service_type)
    host_context = EditItemContext(app, host_id, host_type)
    _item_state = make_unicode(app.request.GET.get(u'item-state', host_context.from_state))
    checks = datamanagerV2.find_merge_state_items(item_type=service_type, item_states=LINKIFY_MANAGE_STATES, where={u'_id': service_id})  # type: List[BaseItem]
    if not checks:
        return app.abort(404, u'There is no [%s] with this id [%s] in [%s]' % (app.t(u'type.' + service_type[:-1]), service_id, check_context.item_state))
    
    host = datamanagerV2.find_item_by_id(host_id, item_type=host_type, item_state=_item_state)  # type: BaseItem
    if not host and not host_context.in_creation:
        return app.abort(404, u'There is no [%s] with this id [%s] in [%s]' % (app.t(u'type.' + host_type[:-1]), host_id, _item_state))
    
    check = checks[0]
    data_on_check = set()
    inherited_without_defaults = get_inherited_without_default(datamanagerV2, check, data_on_item=data_on_check)
    from_info = {}
    data_not_on_check = set()
    for tpl in host_cluster_templates:
        if not tpl:
            continue
        resolve_template_value = tpl.get(SERVICE_OVERRIDE, {})
        if not resolve_template_value:
            continue
        for override in resolve_template_value['links']:
            link_to_service = override['check_link']
            if (link_to_service.get('_id', '') == service_id and dfe_name == override.get('dfe_key', '')) or link_to_service.get('name', '') == check_name:
                if override['key'] == 'check_command_args' and inherited_without_defaults.get('check_command', None):
                    cmd = inherited_without_defaults['check_command'].split('!')[0]
                    inherited_without_defaults['check_command'] = '%s!%s' % (cmd, override['value'])
                    from_info['check_command'] = {'item_name': tpl.get_name(), 'item_id': tpl['_id'], 'item_type': tpl.get_type()}
                else:
                    flattened_value = datamanagerV2.flatten_overridden_property(check.get_type(), override)
                    inherited_without_defaults[override['key']] = flattened_value
                    from_info[override['key']] = {'item_name': tpl.get_name(), 'item_id': tpl['_id'], 'item_type': tpl.get_type()}
                    if override['key'] not in data_on_check and override['key'].startswith('_') and override['key'] not in NOT_TO_LOOK:
                        data_not_on_check.add(override['key'])
    
    METADATA.update_metadata(inherited_without_defaults, METADATA.WINDOW_UUID, dom_uuid)
    METADATA.update_metadata(inherited_without_defaults, METADATA.NAME, check_name)
    METADATA.update_metadata(inherited_without_defaults, METADATA.DFE_NAME, dfe_name)
    METADATA.update_metadata(inherited_without_defaults, METADATA.FROM, from_info)
    
    if force_disable:
        can_edit = False
    elif is_host_editable == '0':
        can_edit = False
    elif host_context.in_creation:
        can_edit = True
    else:
        result = syncuicommon.check_acl(host, host_context.user, _item_state)
        can_edit = result['can_edit']
    
    service_overrides = {
        '_id': inherited_without_defaults['_id']
    }
    if not force_clean:
        if host:
            host = host.get_raw_item(keep_metadata=True, flatten_links=False)
        if host and host.get(SERVICE_OVERRIDE):
            for override in host.get(SERVICE_OVERRIDE, {}).get('links', []):
                if not (override['check_link'].get('name', None) == check_name or override['check_link'].get('_id', None) == service_id):
                    continue
                _dfe_key_override = override.get('dfe_key', '')
                if dfe_name and _dfe_key_override and dfe_name != _dfe_key_override:
                    continue
                
                flattened_value = datamanagerV2.flatten_overridden_property(check.get_type(), override)
                service_overrides[override['key']] = flattened_value
    
    service_overrides['editable'] = '1' if can_edit else '0'
    data_not_on_check.update([data for data in service_overrides if data.startswith('_') and data not in NOT_TO_LOOK and data not in data_on_check])
    
    return {
        'app'                       : app,
        'user'                      : host_context.user,
        'shinken_element_overridden': app.frontend_cipher.cipher(service_overrides, item_type=ITEM_TYPE.HOSTS, user=user),
        'shinken_element_original'  : app.frontend_cipher.cipher(inherited_without_defaults, item_type=ITEM_TYPE.HOSTS, user=user),
        'helper'                    : app.helper,
        'dfe_name'                  : dfe_name,
        'is_tpl'                    : ITEM_TYPE.is_template(service_type),
        'protected_fields'          : DEF_ITEMS[service_type].get('protected_fields', []),
        'item_type'                 : service_type,
        'data_not_on_check'         : data_not_on_check,
        'data_on_check'             : data_on_check,
    }


def elements_item_type_tpls_by_names(tpl_item_type, item_type):
    app.response.content_type = 'application/json'
    
    links = [link for link in json.loads(app.request.POST.get('links', '{}').decode('utf-8', 'ignore'))['links']]
    
    my_name = app.request.POST.get('my_name', '').decode('utf-8', 'ignore')
    linear_result = []
    
    names = update_frontend_links_into_links('use', links, item_type, datamanagerV2)
    
    additional_where = {}
    if app.request.POST.get('is_cluster', '0') == '1':
        additional_where = {'is_cluster': '1'}
    
    lookup_items_templates(item_type, names, linear_result, datamanagerV2, current_branch=[(my_name, item_type, TEMPLATE_STATUS.USEFUL)], add_new=True, additional_where=additional_where)
    
    # Encrypt each item in linear_result
    return json.dumps([app.frontend_cipher.cipher(e, item_type=tpl_item_type, user=app.get_user_auth()) for e in linear_result])


def elements_hosttpls_by_names():
    item_type = app.request.POST.get('item_type', '').decode('utf-8', 'ignore')
    return elements_item_type_tpls_by_names(ITEM_TYPE.HOSTTPLS, item_type)


def elements_clustertpls_by_names():
    item_type = app.request.POST.get('item_type', '').decode('utf-8', 'ignore')
    return elements_item_type_tpls_by_names(ITEM_TYPE.CLUSTERTPLS, item_type)


def elements_servicepls_by_names():
    item_type = app.request.POST.get('item_type', '').decode('utf-8', 'ignore')
    return elements_item_type_tpls_by_names(ITEM_TYPE.SERVICETPLS, item_type)


def elements_contacttpls_by_names():
    item_type = app.request.POST.get('item_type', '').decode('utf-8', 'ignore')
    return elements_item_type_tpls_by_names(ITEM_TYPE.CONTACTTPLS, item_type)


def disable_object(item_type, item_id):
    context = EditItemContext(app, item_id, item_type=item_type, action=ACTIONS.DISABLE_OBJECT)
    app.state_controller.check_acl(context)
    return _set_enabled(item_type, context, '0')


def enable_object(item_type, item_id):
    context = EditItemContext(app, item_id, item_type=item_type, action=ACTIONS.ENABLE_OBJECT)
    app.state_controller.check_acl(context)
    return _set_enabled(item_type, context, '1')


def _set_enabled(_item_type, context, enabled):
    validator_message = ValidatorMessages()
    app.state_controller.set_enable(context, enabled)
    response = FrontEndSaveResponse(validator_message)
    return response.get_response()


# Cloning object means you clone ALL properties of these objects,
# so ACL too (users permissions)
def clone_object(item_type, item_id, transform=None):
    context = EditItemContext(app, item_id, item_type, action=ACTIONS.CLONE_OBJECT)
    app.state_controller.check_acl(context)
    app.state_controller.clone_object(context, transform)


# Create from: like duplication, but without ACL/permissions
def create_from_object(item_type, item_id):
    context = EditItemContext(app, item_id, item_type=item_type, action=ACTIONS.CREATE_FROM_OBJECT)
    app.state_controller.check_acl(context, check_edit_rights=False)
    
    
    def transform_acl(item, _context):
        # Overwrite users parameters and set current user as view & edition
        # clean groups
        if ITEM_TYPE.has_user_right(_context.item_type) and not ITEM_TYPE.is_template(_context.item_type):
            props_to_relink = [u'view_contacts', u'edition_contacts', u'notification_contacts']
            props_to_clean = [u'view_contact_groups', u'notification_contact_groups', u'edition_contact_groups']
            
            new_link = {u'has_plus': True, u'links': [{u'_id': _context.user[u'_id'], u'item_type': ITEM_TYPE.CONTACTS, u'exists': True}]}
            for prop in props_to_relink:
                item[prop] = new_link
            
            for prop in props_to_clean:
                item.pop(prop, None)
        elif ITEM_TYPE.has_user_right(_context.item_type):
            props_to_clean = [
                'view_contact_groups',
                'notification_contact_groups',
                'edition_contact_groups',
                'view_contacts',
                'edition_contacts',
                'notification_contacts'
            ]
            
            for prop in props_to_clean:
                item.pop(prop, None)
    
    
    app.state_controller.clone_object(context, transform=transform_acl)


def save_object(item_type, item_id=u''):
    app.response.content_type = u'application/json'
    
    if not item_id:
        error_message = ValidatorMessages()
        error_message.add_message(Message(MESSAGE.STATUS_CRITICAL, app.t(u'element.missing_id')))
        response = FrontEndSaveResponse(error_message)
        return response.get_response()
    context = EditItemContext(app, item_id, item_type, action=ACTIONS.SAVE_OBJECT)
    logger_save.info(u'Start saving [%s] with uuid:[%s] in:[%s] from:[%s]' % (context.item_type, context.item_id, context.to_state, context.from_state))
    
    app.state_controller.check_acl(context)
    
    item_name, sent_object, validation_messages = _format_form_to_item(app.request.forms, context)
    result = app.state_controller.update(context, sent_object)
    
    if result[u'previously_deleted']:
        validation_messages.add_message(Message(MESSAGE.STATUS_WARNING, app.t(u'element.previously_deleted_item'), weight=MESSAGE.WEIGHT_HIGH))
    if result[u'previously_imported']:
        validation_messages.add_message(Message(MESSAGE.STATUS_WARNING, app.t(u'element.previously_imported_item'), weight=MESSAGE.WEIGHT_HIGH))
    
    response = FrontEndSaveResponse(validation_messages)
    
    return response.get_response()


def _format_form_to_item(request_forms, context):
    # type: (Dict, EditItemContext) -> (unicode, Dict, Messages)
    warning_messages = []
    deleted_properties = []
    item_type = context.item_type
    prev_elt = context.prev_elt
    
    # Use a FrontendCipher from the protected_fields__substrings_list sent by Front to decipher data
    encryption_status = {
        u'protect_fields__substrings_matching_fields'   : u','.join(json.loads(request_forms[u'protect_fields__substrings_matching_fields'])),
        u'protect_fields__activate_interface_encryption': json.loads(request_forms[u'protect_fields__activate_interface_encryption']),
        u'protect_fields__are_viewable_by_admin_si'     : json.loads(request_forms[u'protect_fields__are_viewable_by_admin_si']),
    }
    forms_item = json.loads(request_forms[u'item'])
    
    # If the item is not in creation but, we haven't the previous item, somebody deletes it before we save it. In this case, we can't find the protected data if the user doesn't modify it because front send TAGs
    if not context.in_creation and not context.prev_elt:
        for item_property, value in forms_item.items():
            if match_protected_property(item_property, encryption_status[u'protect_fields__substrings_matching_fields'], item_type) and value in (PROTECTED_TAG_FROM_CHANGE, PROTECTED_DEFAULT_TAG):
                deleted_properties.append(item_property)
                del forms_item[item_property]
    
    if deleted_properties:
        _deleted_html = u'<ul><li><span class="shinken-data-user">%s</span></li></ul>' % u'</span></li><li><span class="shinken-data-user">'.join(deleted_properties)
        warning_messages.append(app.t(u'element.data_loose') % _deleted_html)
    
    format_item_frontend_cipher = get_frontend_cipher_from_request(encryption_status)
    forms_item = format_item_frontend_cipher.uncipher(forms_item, item_type=item_type, old_item=prev_elt, user=context.user)
    
    item_name = get_name_from_type(item_type, forms_item)
    
    # The dfe value must be uppercase and prefixed by _
    if u'duplicate_foreach' in forms_item:
        forms_item[u'duplicate_foreach'] = u'_%s' % forms_item[u'duplicate_foreach'].upper().lstrip(u'_')
    
    # frontend links are not the same as backend links because
    # in link of item to save id can not exist if item was deleted or an item can be created before the save
    # As there is no static get_links() method, cast forms_item into a BaseItem
    # DO NOT RE-USE THIS base_item
    base_item = get_item_instance(item_type, raw_item=forms_item)
    METADATA.update_metadata(base_item, METADATA.ITEM_TYPE, item_type)
    for linking_property in DEF_ITEMS[item_type][u'props_links']:
        if linking_property != SERVICE_OVERRIDE:
            update_frontend_links_into_links(linking_property, base_item.get_links(linking_property), item_type, datamanagerV2, linkify=False)
    
    if SERVICE_OVERRIDE in forms_item:
        update_frontend_service_override_links_into_links(forms_item, item_type, context.to_state, datamanagerV2)
    
    # pass the validator on object and raise error if not validate
    _validation = syncuicommon.validator.validate(item_type, forms_item)
    _validation_messages = ValidatorMessages(_validation)
    if _validation_messages.has_critical():
        
        raise SaveException(400, validator_messages=_validation_messages)
    
    # check ACL
    if item_type in [ITEM_TYPE.CONTACTS, ITEM_TYPE.CONTACTTPLS]:
        acl_share_prop = AclShareProp()
        for acl_share_prop_name in [u'acl_share_private', u'acl_share_group', u'acl_share_everybody']:
            if acl_share_prop_name in forms_item:
                acl_share = forms_item[acl_share_prop_name]
                acl_share = acl_share_prop.unpythonize(acl_share)
                forms_item[acl_share_prop_name] = acl_share
    
    _update_from_previous(item_type, prev_elt, forms_item)
    
    # make sure syncui is in the source list
    if u'syncui' not in forms_item.get(u'sources', u'syncui'):
        forms_item[u'sources'] += u',syncui'
    elif forms_item.get(u'sources', '') == '':
        forms_item[u'sources'] = u'syncui'
    
    for mandatory_field, value in DEF_ITEMS[item_type].get(u'mandatory_fields', {}).iteritems():
        forms_item[mandatory_field] = value
    _validation_messages.add_messages([Message(MESSAGE.STATUS_WARNING, w) for w in warning_messages])
    
    if u'_SYNC_KEYS' not in forms_item:
        # by default use _SYNC_KEYS of syncui
        forms_item[u'_SYNC_KEYS'] = get_name_from_type(item_type, forms_item).lower()
    
    return item_name, forms_item, _validation_messages


def _update_from_previous(item_type, prev_elt, sent_object):
    if not prev_elt:
        return
    if 'work_area_info' in prev_elt:
        sent_object['work_area_info'] = prev_elt['work_area_info']
    previous_raw_elt = prev_elt.get_raw_item()
    cls = DEF_ITEMS[item_type]['class']
    prop_dict = getattr(cls, 'passthrough', {})
    for prop in prop_dict.iterkeys():
        if (prop in previous_raw_elt) and (prop not in sent_object):
            sent_object[prop] = previous_raw_elt[prop]
    
    sync_keys = prev_elt.get('_SYNC_KEYS', [])
    sent_object['_SYNC_KEYS'] = map(lambda key: key.lower(), sync_keys)
    if '_SE_UUID' in prev_elt:
        sent_object['_SE_UUID'] = prev_elt['_SE_UUID']


def delete_ui_entry(item_type, item_id=''):
    logger.debug(u'Delete UI entry %s:%s' % (item_type, item_id))
    app.response.content_type = 'application/json'
    if not item_id:
        return app.abort(400, 'Missing the element _id')
    
    context = EditItemContext(app, item_id, item_type, action=ACTIONS.DELETE_UI_ENTRY)
    validator_message = ValidatorMessages()
    shinken_return_code = ShinkenReturnCode.DELETE_OK
    
    # if item in work area, try to delete it from work area
    if context.from_state == ITEM_STATE.WORKING_AREA and not context.bypass_work_area:
        return delete_item_working_area(context.item_type, context.item_id)
    
    app.state_controller.delete(context, context.from_state)
    
    response = FrontEndSaveResponse(validator_message, shinken_return_code=shinken_return_code, default_text=app.t(u'element.object_deleted'))
    return response.get_response()


# DEPRECATED : use import_many_from_source instead
def import_from_source(item_type):
    with object_edition_lock:
        _ids = []
        for k in app.request.forms:
            if app.request.forms.get(k):
                _ids.append(k)
    
    logger_save.info(u'Start importing %s %s' % (len(_ids), item_type))
    
    for _id in _ids:
        context = EditItemContext(app, _id, item_type, action=ACTIONS.IMPORT_FROM_SOURCE)
        logger_save.info(u'Start importing in [%s] [%s] with uuid:[%s]' % (context.to_state, context.item_type, context.item_id))
        app.state_controller.import_one_object(context)


def _do_async_mass_action(action):
    user = app.get_user_auth()
    raw_data = app.request.body.read()
    data = json.loads(raw_data)
    object_count = data.get(u'object_count', 0)
    mass_action_uuid = uuid.uuid1().hex
    MULTI_MASS_ACTION_UPDATE[mass_action_uuid] = {u'total': object_count, u'current': 0}
    contexts = {}
    elements = data.get(u'elements', {})
    
    logger_save.info(u'Start %s %s elements' % (action, object_count))
    
    for item_type, item_ids in elements.iteritems():
        for item_id in item_ids:
            context = EditItemContext(app, item_id, item_type, action=action)
            if item_type not in contexts:
                contexts[item_type] = {}
            contexts[item_type][item_id] = context
    launch_many_mass_action_thread(action, mass_action_uuid, elements, contexts, user)
    return {u'uuid': mass_action_uuid}


def import_many_from_source():
    user = app.get_user_auth()
    if not user.is_admin():
        raise UnauthorizedSaveException
    return _do_async_mass_action(ACTIONS.IMPORT_FROM_SOURCE)


def validate_many_changes():
    user = app.get_user_auth()
    if not user.is_admin():
        raise UnauthorizedSaveException
    return _do_async_mass_action(ACTIONS.VALIDATE_CHANGES)


def launch_many_mass_action_thread(action, action_uuid, elements, contexts, user):
    thread_name = 'many-mass-action-thread-%s-%s' % (action, action_uuid)  # this can't be unicode because threading.Thread name is not unicode
    t = threading.Thread(None, target=_launch_many_mass_action_thread, name=thread_name, args=(action, action_uuid, elements, contexts, user))
    t.daemon = True
    t.start()


def _launch_many_mass_action_thread(action, action_uuid, elements, contexts, user):
    try:
        _real_launch_many_mass_action_thread(action, action_uuid, elements, contexts, user)
    except Exception:
        logger.print_stack()


def _real_launch_many_mass_action_thread(action, action_uuid, elements, contexts, user):
    mass_actions_functions = {
        ACTIONS.IMPORT_FROM_SOURCE: (app.state_controller.import_one_object, blank_context),
        ACTIONS.VALIDATE_CHANGES  : (app.state_controller.validate_changes, app.MASS_IMPORT_LOCK),
    }
    mass_action_function, _lock = mass_actions_functions[action]
    with _lock:
        with DBTransaction(user=user):
            inheritance_cache = set()
            MULTI_MASS_ACTION_UPDATE[action_uuid][u'counter_error'] = 0
            MULTI_MASS_ACTION_UPDATE[action_uuid][u'counter_warning'] = 0
            for item_type, item_ids in elements.iteritems():
                for item_id in item_ids:
                    context = contexts[item_type][item_id]
                    return_values = mass_action_function(context, inheritance_cache=inheritance_cache)
                    validation = return_values[0]
                    item_name = return_values[1]
                    if validation[u'has_messages']:
                        validation_message = u''.join([u'<li>%s</li>' % _validation_msg for _validation_msg in validation[u'messages']])
                        element_type = app.t(u'type.%s' % context.item_type[:-1])
                        param_message = {}
                        if validation[u'has_critical']:
                            param_message = {
                                u'type_message'      : u'critical-error-message',
                                u'type_tag'          : u'error',
                                u'type_tag_text'     : app.t(u'element.critical-error'),
                                u'element_type'      : element_type,
                                u'item_name'         : item_name,
                                u'validation_message': validation_message,
                            }
                            MULTI_MASS_ACTION_UPDATE[action_uuid][u'counter_error'] = MULTI_MASS_ACTION_UPDATE[action_uuid][u'counter_error'] + 1
                        elif validation[u'has_error']:
                            param_message = {
                                u'type_message'      : u'error-message',
                                u'type_tag'          : u'error',
                                u'type_tag_text'     : app.t(u'element.error'),
                                u'element_type'      : element_type,
                                u'item_name'         : item_name,
                                u'validation_message': validation_message,
                            }
                            MULTI_MASS_ACTION_UPDATE[action_uuid][u'counter_error'] = MULTI_MASS_ACTION_UPDATE[action_uuid][u'counter_error'] + 1
                        elif validation[u'has_warning']:
                            param_message = {
                                u'type_message'      : u'warning-message',
                                u'type_tag'          : u'warning',
                                u'type_tag_text'     : app.t(u'element.warning_saving_log'),
                                u'element_type'      : element_type,
                                u'item_name'         : item_name,
                                u'validation_message': validation_message,
                            }
                            MULTI_MASS_ACTION_UPDATE[action_uuid][u'counter_warning'] = MULTI_MASS_ACTION_UPDATE[action_uuid][u'counter_warning'] + 1
                        
                        message = u'''
                        <div class="shinken-saving-log-message %(type_message)s">
                            <div class="shinken-saving-log-message-header">
                                <span class="tag %(type_tag)s">%(type_tag_text)s</span>
                                <span class="tag element-type">%(element_type)s</span>
                                <span class="element-name">%(item_name)s</span> :
                            </div><ul>%(validation_message)s</ul>
                        </div>''' % param_message
                        
                        if u'validation_messages' not in MULTI_MASS_ACTION_UPDATE[action_uuid]:
                            MULTI_MASS_ACTION_UPDATE[action_uuid][u'validation_messages'] = []
                        MULTI_MASS_ACTION_UPDATE[action_uuid][u'validation_messages'].append(message)
                    
                    MULTI_MASS_ACTION_UPDATE[action_uuid][u'current'] = MULTI_MASS_ACTION_UPDATE[action_uuid][u'current'] + 1
            # logger.debug('Mass action %s done progress %s' % (action_uuid, MULTI_MASS_ACTION_UPDATE[action_uuid]['current']))
        logger.debug(u'Mass action %s done' % action_uuid)
        # if no validation error are presents we can delete the update now. If validation the frontend need to get it when it will receive the last update
        _time = time.time()
        time.sleep(1)
        if u'validation_messages' in MULTI_MASS_ACTION_UPDATE:
            while action_uuid in MULTI_MASS_ACTION_UPDATE and time.time() - _time < 10:
                time.sleep(1)
        MULTI_MASS_ACTION_UPDATE.pop(action_uuid, None)


def get_many_mass_action_progress():
    ###################################################################################
    #
    #  NO LOCK FOR THIS ROUTE === DO NOT CHANGE DATA
    #
    ###################################################################################
    
    mass_action_uuid = app.request.GET.get('uuid', None)
    if not mass_action_uuid:
        return app.abort(404, 'The mass action \'uuid\' was not given')
    mass_action_progress = MULTI_MASS_ACTION_UPDATE.get(mass_action_uuid, None)
    if mass_action_progress is None:
        # the mass action is probably done, return a mock info to simulate the end of the mass action
        return {'total': 1, 'current': 1}
    if mass_action_progress['current'] >= mass_action_progress['total']:
        MULTI_MASS_ACTION_UPDATE.pop(mass_action_uuid, None)
    return mass_action_progress


@return_error_if_unauthorized_save_exception
def save_in_work_area(item_type, item_id):
    app.response.content_type = 'application/json'
    if not item_id:
        error_message = ValidatorMessages()
        error_message.add_message(Message(MESSAGE.STATUS_CRITICAL, app.t('element.missing_id')))
        response = FrontEndSaveResponse(error_message)
        return response.get_response()
    context = EditItemContext(app, item_id, item_type=item_type, action=ACTIONS.SAVE_IN_WORK_AREA)
    
    app.state_controller.check_acl(context)
    
    item_name, sent_object, validation_messages = _format_form_to_item(app.request.forms, context)
    app.state_controller.save_in_work_area(context, item_name, sent_object)
    
    response = FrontEndSaveResponse(validation_messages)
    
    return response.get_response()


def submit_to_staging(item_type, item_id):
    context = EditItemContext(app, item_id, item_type, action=ACTIONS.SUBMIT_TO_STAGING)
    app.state_controller.submit_to_staging(context)


def accept_submit(item_type, item_id):
    app.response.content_type = 'application/json'
    logger.debug('[work_area] accept submission for item [%s-%s]' % (item_type, item_id))
    
    context = EditItemContext(app, item_id, item_type, action=ACTIONS.ACCEPT_SUBMIT)
    
    app.state_controller.accept_submit(context)
    
    response = FrontEndSaveResponse(ValidatorMessages(), default_text=app.t('element.apply_work_area_OK'))
    return response.get_response()


def reject_submit(item_type, item_id):
    app.response.content_type = 'application/json'
    
    forms = app.request.forms
    reject_comment_text = forms.get(u'comment', u'').decode(u'utf8', u'ignore').strip()
    context = EditItemContext(app, item_id, item_type, action=ACTIONS.REJECT_SUBMIT)
    
    app.state_controller.reject_submit(context, reject_comment_text)
    
    logger.debug(u'[work_area] reject_submit item [%s-%s] it will be resend in work area' % (item_type, item_id))
    response = FrontEndSaveResponse(ValidatorMessages(), default_text=app.t(u'element.reject_submit_OK'), was_reject=True)
    return response.get_response()


# unlock_work_area AKA Cancel modifications
def unlock_work_area(item_type, item_id):
    app.response.content_type = 'application/json'
    context = EditItemContext(app, item_id, item_type, action=ACTIONS.UNLOCK_WORK_AREA)
    
    has_item_in_staging = app.state_controller.unlock_work_area(context)
    
    response = FrontEndSaveResponse(ValidatorMessages(), has_item_in_staging=has_item_in_staging, shinken_return_code=ShinkenReturnCode.DELETE_OK, default_text=app.t(u'element.unlock_item_OK'))
    return response.get_response()


def working_area(item_type):
    user = app.get_user_auth()
    logger.debug(u'[work_area] call working_area for user [%s]' % (user[u'contact_name']))
    return elements_generic(item_type, items_state=ITEM_STATE.WORKING_AREA)


def my_working_area(item_type):
    user = app.get_user_auth()
    logger.debug(u'[work_area] call my_working_area for user [%s]' % (user[u'contact_name']))
    
    return elements_generic(item_type, items_state=ITEM_STATE.WORKING_AREA, only_my_item=True)


def get_item_working_area(item_type, item_id):
    return_value = _get_specific_item(item_id, item_type, current_page='working_area')
    return return_value


def delete_item_working_area(item_type, item_id):
    app.response.content_type = 'application/json'
    logger.debug('[work_area] delete entry [%s-%s]' % (item_type, item_id))
    if not item_id:
        return app.abort(400, 'Missing the element _id')
    context = EditItemContext(app, item_id, item_type, action=ACTIONS.DELETE_ITEM_WORKING_AREA)
    validator_message = ValidatorMessages()
    shinken_return_code = ShinkenReturnCode.DELETE_OK
    
    item_in_staging = app.state_controller.delete_item_working_area(context)
    
    validator_message.add_message(Message(MESSAGE.STATUS_INFO, app.t('element.object_deleted')))
    response = FrontEndSaveResponse(validator_message, shinken_return_code=shinken_return_code, has_item_in_staging=(item_in_staging is not None))
    return response.get_response()


def add_item_working_area(item_type):
    return_value = new_object(item_type)
    return_value['working_area'] = True
    return return_value


pages = {
    # HOSTS
    new_host                     : {'routes': ['/elements/add/host'], 'view': 'elements_host', 'static': True, 'wrappers': ['auth']},
    edit_host                    : {'routes': ['/elements/hosts/:item_id'], 'view': 'elements_host', 'static': True, 'wrappers': ['auth']},
    # HOSTTPL
    new_host_tpl                 : {'routes': ['/elements/add/hosttpl'], 'view': 'elements_host', 'static': True},
    edit_hosttpl                 : {'routes': ['/elements/hosttpls/:item_id'], 'view': 'elements_host', 'static': True},
    # HOST GROUPS
    new_hostgroup                : {'routes': ['/elements/add/hostgroup'], 'view': 'elements_hostgroup', 'static': True},
    edit_hostgroup               : {'routes': ['/elements/hostgroups/:item_id'], 'view': 'elements_hostgroup', 'static': True},
    # CLUSTERS
    new_cluster                  : {'routes': ['/elements/add/cluster'], 'view': 'elements_cluster', 'static': True},
    edit_cluster                 : {'routes': ['/elements/clusters/:item_id'], 'view': 'elements_cluster', 'static': True},
    # CLUSTERTPL
    new_cluster_tpl              : {'routes': ['/elements/add/clustertpl'], 'view': 'elements_cluster', 'static': True},
    edit_clustertpl              : {'routes': ['/elements/clustertpls/:item_id'], 'view': 'elements_cluster', 'static': True},
    # SERVICES
    new_services_host            : {'routes': ['/elements/add/serviceshost'], 'view': 'elements_service', 'static': True},
    edit_services_hosts          : {'routes': ['/elements/serviceshosts/:item_id'], 'view': 'elements_service', 'static': True},
    edit_services_hosts_override : {'routes': ['/elements/override2/serviceshosts/:item_id'], 'view': 'elements_service_override', 'static': True},
    # SERVICES HOSTTPL
    new_services_host_tpl        : {'routes': ['/elements/add/serviceshosttpl'], 'view': 'elements_service', 'static': True},
    edit_services_host_tpls      : {'routes': ['/elements/serviceshosttpls/:item_id'], 'view': 'elements_service', 'static': True},
    # SERVICES CLUSTER
    new_services_cluster         : {'routes': ['/elements/add/servicescluster'], 'view': 'elements_service', 'static': True},
    edit_services_clusters       : {'routes': ['/elements/servicesclusters/:item_id'], 'view': 'elements_service', 'static': True},
    # SERVICES CLUSTERTPL
    new_services_cluster_tpl     : {'routes': ['/elements/add/servicesclustertpl'], 'view': 'elements_service', 'static': True},
    edit_services_cluster_tpls   : {'routes': ['/elements/servicesclustertpls/:item_id'], 'view': 'elements_service', 'static': True},
    # SERVICE TPL
    new_service_tpl              : {'routes': ['/elements/add/servicetpl'], 'view': 'elements_service', 'static': True},
    edit_service_tpls            : {'routes': ['/elements/servicetpls/:item_id'], 'view': 'elements_service', 'static': True},
    # CONTACTS
    new_contact                  : {'routes': ['/elements/add/contact'], 'view': 'elements_contact', 'static': True},
    edit_contact                 : {'routes': ['/elements/contacts/:item_id'], 'view': 'elements_contact', 'static': True, 'wrappers': ['auth']},
    # CONTACTTPL
    new_contact_tpl              : {'routes': ['/elements/add/contacttpl'], 'view': 'elements_contact', 'static': True},
    edit_contacttpl              : {'routes': ['/elements/contacttpls/:item_id'], 'view': 'elements_contact', 'static': True},
    # CONTACT GROUPS
    new_contactgroup             : {'routes': ['/elements/add/contactgroup'], 'view': 'elements_contactgroup', 'static': True},
    edit_contactgroup            : {'routes': ['/elements/contactgroups/:item_id'], 'view': 'elements_contactgroup', 'static': True},
    # TIME PERIODS
    new_timeperiod               : {'routes': ['/elements/add/timeperiod'], 'view': 'elements_timeperiod', 'static': True},
    edit_timeperiod              : {'routes': ['/elements/timeperiods/:item_id'], 'view': 'elements_timeperiod', 'static': True},
    # COMMANDS
    new_command                  : {'routes': ['/elements/add/command'], 'view': 'elements_command', 'static': True},
    edit_command                 : {'routes': ['/elements/commands/:item_id'], 'view': 'elements_command', 'static': True},
    # BUSINESS IMPACT MODULATION
    new_businessimpactmodulation : {'routes': ['/elements/add/businessimpactmodulation'], 'view': 'elements_businessimpactmodulation', 'static': True},
    edit_businessimpactmodulation: {'routes': ['/elements/businessimpactmodulations/:item_id'], 'view': 'elements_businessimpactmodulation', 'static': True},
    # MACRO MODULATION
    new_macromodulation          : {'routes': ['/elements/add/macromodulation'], 'view': 'elements_macromodulation', 'static': True},
    edit_macromodulation         : {'routes': ['/elements/macromodulations/:item_id'], 'view': 'elements_macromodulation', 'static': True},
    # RESULT MODULATION
    new_resultmodulation         : {'routes': ['/elements/add/resultmodulation'], 'view': 'elements_resultmodulation', 'static': True},
    edit_resultmodulation        : {'routes': ['/elements/resultmodulations/:item_id'], 'view': 'elements_resultmodulation', 'static': True},
    # NOTIFICATION WAY
    new_notificationway          : {'routes': ['/elements/add/notificationway'], 'view': 'elements_notificationway', 'static': True},
    edit_notificationway         : {'routes': ['/elements/notificationways/:item_id'], 'view': 'elements_notificationway', 'static': True},
    # ESCALATIONS
    new_escalation               : {'routes': ['/elements/add/escalation'], 'view': 'elements_escalation', 'static': True},
    edit_escalation              : {'routes': ['/elements/escalations/:item_id'], 'view': 'elements_escalation', 'static': True},
    
    # TEMPLATING INFORMATION
    # Get checks for templates
    elements_checksbytemplates   : {'routes': ['/elements/checksbytemplates/'], 'method': 'POST', 'static': True, 'wrappers': ['auth']},
    # Get template information
    elements_servicepls_by_names : {'routes': ['/elements/byname/servicetpls/'], 'method': 'POST', 'static': True, 'wrappers': ['auth']},
    elements_contacttpls_by_names: {'routes': ['/elements/byname/contacttpls/'], 'method': 'POST', 'static': True, 'wrappers': ['auth']},
    elements_hosttpls_by_names   : {'routes': ['/elements/byname/hosttpls/'], 'method': 'POST', 'static': True, 'wrappers': ['auth']},
    elements_clustertpls_by_names: {'routes': ['/elements/byname/clustertpls/'], 'method': 'POST', 'static': True, 'wrappers': ['auth']},
    # To run things like checks
    run_check                    : {'routes': ['/element/run'], 'method': 'POST', 'wrappers': ['auth']},
    
    # GENERIC MASS ACTIONS
    disable_object               : {'routes': ['/element/q/:item_type/disable/:item_id'], 'wrappers': ['auth', 'transaction', 'db_transaction', 'format_validator_message_if_base_exception']},
    enable_object                : {'routes': ['/element/q/:item_type/enable/:item_id'], 'wrappers': ['auth', 'transaction', 'db_transaction', 'format_validator_message_if_base_exception']},
    clone_object                 : {'routes': ['/element/q/:item_type/clone/:item_id'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'db_transaction', 'format_validator_message_if_base_exception']},
    create_from_object           : {'routes': ['/element/q/:item_type/create-from/:item_id'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'db_transaction', 'format_validator_message_if_base_exception']},
    import_from_source           : {'routes': ['/element/q/:item_type/validate/'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'db_transaction']},
    # GENERIC ACTIONS
    delete_ui_entry              : {'routes': ['/element/q/:item_type/delete/:item_id'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'db_transaction', 'format_validator_message_if_base_exception']},
    
    # STAGING STUFF
    # List Elements
    list_elements                : {'routes': ['/elements/<item_type>'], 'view': '_list_elements', 'static': True, 'wrappers': ['auth']},
    # specific actions
    save_object                  : {'routes': ['/element/q/:item_type/save/:item_id'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'db_transaction', 'format_validator_message_if_base_exception']},
    accept_submit                : {'routes': ['/element/q/:item_type/apply_work_area/:item_id'], 'method': 'POST', 'wrappers': ['format_validator_message_if_not_admin', 'transaction', 'db_transaction']},
    reject_submit                : {'routes': ['/element/q/:item_type/reject_submit/:item_id'], 'method': 'POST', 'wrappers': ['format_validator_message_if_not_admin', 'transaction', 'db_transaction']},
    
    # WORKING AREA STUFF
    # List Elements
    working_area                 : {'routes': ['/elements/working_area/<item_type>'], 'view': 'working_area', 'static': True, 'wrappers': ['auth']},
    my_working_area              : {'routes': ['/elements/my_working_area/<item_type>'], 'view': 'my_working_area', 'static': True, 'wrappers': ['auth']},
    # get a specific element in working area
    get_item_working_area        : {'routes': ['/elements/working_area/<item_type>/<item_id>'], 'view': 'elements_host', 'static': True, 'wrappers': ['auth', 'transaction']},
    # mass action work area
    delete_item_working_area     : {'routes': ['/elements/delete/working_area/<item_type>/<item_id>'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'db_transaction', 'format_validator_message_if_base_exception']},
    add_item_working_area        : {'routes': ['/elements/add/working_area/<item_type>'], 'view': 'elements_host', 'static': True, 'wrappers': ['auth', 'transaction', 'db_transaction']},
    # specific actions
    submit_to_staging            : {'routes': ['/element/q/:item_type/submit_to_staging/:item_id'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'db_transaction', 'format_validator_message_if_base_exception']},
    save_in_work_area            : {'routes': ['/element/q/:item_type/save_in_work_area/:item_id'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'db_transaction', 'format_validator_message_if_base_exception']},
    unlock_work_area             : {'routes': ['/element/q/:item_type/unlock_work_area/:item_id'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'format_validator_message_if_base_exception']},
    
    get_service_override         : {'routes': ['/elements/service_overrides/:host_type/:host_id/:service_type/:service_id/:apply_on_type'], 'view': 'elements_service_override', 'static': True, 'wrappers': ['auth']},
    import_many_from_source      : {'routes': ['/elements/q/validate/'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'json', 'format_validator_message_if_not_admin']},
    validate_many_changes        : {'routes': ['/elements/q/validate-changes/'], 'method': 'POST', 'wrappers': ['auth', 'transaction', 'json', 'format_validator_message_if_not_admin']},
    
    get_many_mass_action_progress: {'routes': ['/elements/mass_action/progress/'], 'method': 'GET', 'wrappers': ['json']},
    
    # Massive changes
    massive_change_call          : {'routes': ['/elements/massive_change/<item_type>'], 'method': 'POST', 'view': 'massive_change/massive_change', 'wrappers': ['auth']},
}
